﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/pinpoint-sms-voice-v2/PinpointSMSVoiceV2Request.h>
#include <aws/pinpoint-sms-voice-v2/PinpointSMSVoiceV2_EXPORTS.h>
#include <aws/pinpoint-sms-voice-v2/model/CloudWatchLogsDestination.h>
#include <aws/pinpoint-sms-voice-v2/model/EventType.h>
#include <aws/pinpoint-sms-voice-v2/model/KinesisFirehoseDestination.h>
#include <aws/pinpoint-sms-voice-v2/model/SnsDestination.h>

#include <utility>

namespace Aws {
namespace PinpointSMSVoiceV2 {
namespace Model {

/**
 */
class UpdateEventDestinationRequest : public PinpointSMSVoiceV2Request {
 public:
  AWS_PINPOINTSMSVOICEV2_API UpdateEventDestinationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateEventDestination"; }

  AWS_PINPOINTSMSVOICEV2_API Aws::String SerializePayload() const override;

  AWS_PINPOINTSMSVOICEV2_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The configuration set to update with the new event destination. Valid values
   * for this can be the ConfigurationSetName or ConfigurationSetArn.</p>
   */
  inline const Aws::String& GetConfigurationSetName() const { return m_configurationSetName; }
  inline bool ConfigurationSetNameHasBeenSet() const { return m_configurationSetNameHasBeenSet; }
  template <typename ConfigurationSetNameT = Aws::String>
  void SetConfigurationSetName(ConfigurationSetNameT&& value) {
    m_configurationSetNameHasBeenSet = true;
    m_configurationSetName = std::forward<ConfigurationSetNameT>(value);
  }
  template <typename ConfigurationSetNameT = Aws::String>
  UpdateEventDestinationRequest& WithConfigurationSetName(ConfigurationSetNameT&& value) {
    SetConfigurationSetName(std::forward<ConfigurationSetNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name to use for the event destination.</p>
   */
  inline const Aws::String& GetEventDestinationName() const { return m_eventDestinationName; }
  inline bool EventDestinationNameHasBeenSet() const { return m_eventDestinationNameHasBeenSet; }
  template <typename EventDestinationNameT = Aws::String>
  void SetEventDestinationName(EventDestinationNameT&& value) {
    m_eventDestinationNameHasBeenSet = true;
    m_eventDestinationName = std::forward<EventDestinationNameT>(value);
  }
  template <typename EventDestinationNameT = Aws::String>
  UpdateEventDestinationRequest& WithEventDestinationName(EventDestinationNameT&& value) {
    SetEventDestinationName(std::forward<EventDestinationNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>When set to true logging is enabled.</p>
   */
  inline bool GetEnabled() const { return m_enabled; }
  inline bool EnabledHasBeenSet() const { return m_enabledHasBeenSet; }
  inline void SetEnabled(bool value) {
    m_enabledHasBeenSet = true;
    m_enabled = value;
  }
  inline UpdateEventDestinationRequest& WithEnabled(bool value) {
    SetEnabled(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An array of event types that determine which events to log.</p>  <p>The
   * <code>TEXT_SENT</code> event type is not supported.</p>
   */
  inline const Aws::Vector<EventType>& GetMatchingEventTypes() const { return m_matchingEventTypes; }
  inline bool MatchingEventTypesHasBeenSet() const { return m_matchingEventTypesHasBeenSet; }
  template <typename MatchingEventTypesT = Aws::Vector<EventType>>
  void SetMatchingEventTypes(MatchingEventTypesT&& value) {
    m_matchingEventTypesHasBeenSet = true;
    m_matchingEventTypes = std::forward<MatchingEventTypesT>(value);
  }
  template <typename MatchingEventTypesT = Aws::Vector<EventType>>
  UpdateEventDestinationRequest& WithMatchingEventTypes(MatchingEventTypesT&& value) {
    SetMatchingEventTypes(std::forward<MatchingEventTypesT>(value));
    return *this;
  }
  inline UpdateEventDestinationRequest& AddMatchingEventTypes(EventType value) {
    m_matchingEventTypesHasBeenSet = true;
    m_matchingEventTypes.push_back(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An object that contains information about an event destination that sends
   * data to CloudWatch Logs.</p>
   */
  inline const CloudWatchLogsDestination& GetCloudWatchLogsDestination() const { return m_cloudWatchLogsDestination; }
  inline bool CloudWatchLogsDestinationHasBeenSet() const { return m_cloudWatchLogsDestinationHasBeenSet; }
  template <typename CloudWatchLogsDestinationT = CloudWatchLogsDestination>
  void SetCloudWatchLogsDestination(CloudWatchLogsDestinationT&& value) {
    m_cloudWatchLogsDestinationHasBeenSet = true;
    m_cloudWatchLogsDestination = std::forward<CloudWatchLogsDestinationT>(value);
  }
  template <typename CloudWatchLogsDestinationT = CloudWatchLogsDestination>
  UpdateEventDestinationRequest& WithCloudWatchLogsDestination(CloudWatchLogsDestinationT&& value) {
    SetCloudWatchLogsDestination(std::forward<CloudWatchLogsDestinationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An object that contains information about an event destination for logging to
   * Firehose.</p>
   */
  inline const KinesisFirehoseDestination& GetKinesisFirehoseDestination() const { return m_kinesisFirehoseDestination; }
  inline bool KinesisFirehoseDestinationHasBeenSet() const { return m_kinesisFirehoseDestinationHasBeenSet; }
  template <typename KinesisFirehoseDestinationT = KinesisFirehoseDestination>
  void SetKinesisFirehoseDestination(KinesisFirehoseDestinationT&& value) {
    m_kinesisFirehoseDestinationHasBeenSet = true;
    m_kinesisFirehoseDestination = std::forward<KinesisFirehoseDestinationT>(value);
  }
  template <typename KinesisFirehoseDestinationT = KinesisFirehoseDestination>
  UpdateEventDestinationRequest& WithKinesisFirehoseDestination(KinesisFirehoseDestinationT&& value) {
    SetKinesisFirehoseDestination(std::forward<KinesisFirehoseDestinationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An object that contains information about an event destination that sends
   * data to Amazon SNS.</p>
   */
  inline const SnsDestination& GetSnsDestination() const { return m_snsDestination; }
  inline bool SnsDestinationHasBeenSet() const { return m_snsDestinationHasBeenSet; }
  template <typename SnsDestinationT = SnsDestination>
  void SetSnsDestination(SnsDestinationT&& value) {
    m_snsDestinationHasBeenSet = true;
    m_snsDestination = std::forward<SnsDestinationT>(value);
  }
  template <typename SnsDestinationT = SnsDestination>
  UpdateEventDestinationRequest& WithSnsDestination(SnsDestinationT&& value) {
    SetSnsDestination(std::forward<SnsDestinationT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_configurationSetName;

  Aws::String m_eventDestinationName;

  bool m_enabled{false};

  Aws::Vector<EventType> m_matchingEventTypes;

  CloudWatchLogsDestination m_cloudWatchLogsDestination;

  KinesisFirehoseDestination m_kinesisFirehoseDestination;

  SnsDestination m_snsDestination;
  bool m_configurationSetNameHasBeenSet = false;
  bool m_eventDestinationNameHasBeenSet = false;
  bool m_enabledHasBeenSet = false;
  bool m_matchingEventTypesHasBeenSet = false;
  bool m_cloudWatchLogsDestinationHasBeenSet = false;
  bool m_kinesisFirehoseDestinationHasBeenSet = false;
  bool m_snsDestinationHasBeenSet = false;
};

}  // namespace Model
}  // namespace PinpointSMSVoiceV2
}  // namespace Aws
