<?php
/**
 * The Horde_Widgets class provides various functions used to generate
 * and handle form input.
 *
 * $Horde: horde/lib/Widgets.php,v 1.1 2003/08/07 22:27:33 slusarz Exp $
 *
 * Copyright 2002-2003 Jon Parise <jon@horde.org>
 * Copyright 2003 Michael Slusarz <slusarz@bigworm.colorado.edu>
 *
 * See the enclosed file LICENSE for license information (LGPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/lgpl.html.
 *
 * @author  Jon Parise <jon@horde.org>
 * @author  Michael Slusarz <slusarz@bigworm.colorado.edu>
 * @version $Revision: 1.1 $
 * @since   Horde 3.0
 * @package horde
 */
class Horde_Widgets {

    /**
     * Generates the HTML for a day selection widget.
     *
     * @access public
     *
     * @param string $name               The name of the widget.
     * @param optional integer $default  The value to select by default.
     *                                   Range: 1-31
     * @param optional string $params    Any additional parameters to
     *                                   include in the <a> tag.
     *
     * @return string  The HTML <select> widget.
     */
    function buildDayWidget($name, $default = null, $params = null)
    {
        $html = '<select id="' . $name . '" name="' . $name. '"';
        if (!is_null($params)) {
            $html .= ' ' . $params;
        }
        $html .= '>';

        for ($day = 1; $day <= 31; $day++) {
            $html .= '<option value="' . $day . '"';
            $html .= ($day == $default) ? ' selected="selected">' : '>';
            $html .= $day . '</option>';
        }

        return $html . "</select>\n";
    }

    /**
     * Generates the HTML for a month selection widget.
     *
     * @access public
     *
     * @param string $name               The name of the widget.
     * @param optional integer $default  The value to select by default.
     * @param optional string $params    Any additional parameters to
     *                                   include in the <a> tag.
     *
     * @return string  The HTML <select> widget.
     */
    function buildMonthWidget($name, $default = null, $params = null)
    {
        $html = '<select id="' . $name . '" name="' . $name. '"';
        if (!is_null($params)) {
            $html .= ' ' . $params;
        }
        $html .= '>';

        for ($month = 1; $month <= 12; $month++) {
            $html .= '<option value="' . $month . '"';
            $html .= ($month == $default) ? ' selected="selected">' : '>';
            $html .= strftime('%B', mktime(0, 0, 0, $month, 1)) . '</option>';
        }

        return $html . "</select>\n";
    }

    /**
     * Generates the HTML for a year selection widget.
     *
     * @access public
     *
     * @param integer $name             The name of the widget.
     * @param integer $years            The number of years to include.
     *                                  If (+): future years
     *                                  If (-): past years
     * @param optional string $default  The timestamp to select by default.
     * @param optional string $params   Any additional parameters to
     *                                  include in the <a> tag.
     *
     * @return string  The HTML <select> widget.
     */
    function buildYearWidget($name, $years, $default = null, $params = null)
    {
        $curr_year = date('Y');
        $yearlist = array();

        $startyear = (!is_null($default) && ($default < $curr_year) && ($years > 0)) ? $default : $curr_year;
        $startyear = min($startyear, $startyear + $years);
        for ($i = 0; $i <= abs($years); $i++) {
            $yearlist[] = $startyear++;
        }
        if ($years < 0) {
            $yearlist = array_reverse($yearlist);
        }

        $html = '<select id="' . $name . '" name="' . $name. '"';
        if (!is_null($params)) {
            $html .= ' ' . $params;
        }
        $html .= '>';

        foreach ($yearlist as $year) {
            $html .= '<option value="' . $year . '"';
            $html .= ($year == $default) ? ' selected="selected">' : '>';
            $html .= $year . '</option>';
        }

        return $html . "</select>\n";
    }

    /**
     * Generates the HTML for an hour selection widget.
     *
     * @access public
     *
     * @param string $name               The name of the widget.
     * @param optional integer $default  The timestamp to select by default.
     * @param optional string $params    Any additional parameters to
     *                                   include in the <a> tag.
     *
     * @return string  The HTML <select> widget.
     */
    function buildHourWidget($name, $default = null, $params = null)
    {
        $html = '<select name="' . $name. '"';
        if (!is_null($params)) {
            $html .= ' ' . $params;
        }
        $html .= '>';

        for ($hour = 0; $hour <= 23; $hour++) {
            $html .= '<option value="' . $hour . '"';
            $html .= ($hour == $default) ? ' selected="selected">' : '>';
            $html .= sprintf("%02d", $hour) . '</option>';
        }

        return $html . '</select>';
    }

    /**
     * Generates the HTML for a minute selection widget.
     *
     * @access public
     *
     * @param string $name                 The name of the widget.
     * @param optional integer $increment  The increment between minutes.
     * @param optional integer $default    The timestamp to select by default.
     * @param optional string $params      Any additional parameters to
     *                                     include in the <a> tag.
     *
     * @return string  The HTML <select> widget.
     */
    function buildMinuteWidget($name, $increment = 1, $default = null,
                               $params = null)
    {
        $html = '<select name="' . $name. '"';
        if (!is_null($params)) {
            $html .= ' ' . $params;
        }
        $html .= '>';

        for ($minute = 0; $minute <= 60; $minute += $increment) {
            $html .= '<option value="' . $minute . '"';
            $html .= ($minute == $default) ? ' selected="selected">' : '>';
            $html .= sprintf("%02d", $minute) . '</option>';
        }

        return $html . "</select>\n";
    }

}
