/* vim: set sw=8: -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/**
 * lotus-formula.c: Lotus 123 formula support for Gnumeric
 *
 * Author:
 *    See: README
 *    Michael Meeks <michael@imagiantor.com>
 * Revamped in Aug 2002
 *    Jody Goldberg <jody@gnome.org>
 * New 123 formats done September 2005
 *    Morten Welinder (terra@gnome.org)
 **/
#include <gnumeric-config.h>
#include <gnumeric.h>
#include <string.h>
#include "lotus.h"
#include "lotus-types.h"
#include "lotus-formula.h"

#include <expr.h>
#include <parse-util.h>
#include <value.h>
#include <gutils.h>
#include <func.h>
#include <gsf/gsf-utils.h>

#define FORMULA_DEBUG 0

typedef struct _Wk1Func Wk1Func;
struct _Wk1Func {
	int		 args; /* -1 for multiple arguments */
	unsigned	 idx;
	char const	*name;
	int (*handler) (GnmExprList **stack, Wk1Func const *func, guint8 const *data, const GnmParsePos *orig);
	guint32  data;
};

static int wk1_unary_func  (GnmExprList **stack, Wk1Func const *func, guint8 const *data, const GnmParsePos *orig);
static int wk1_binary_func (GnmExprList **stack, Wk1Func const *func, guint8 const *data, const GnmParsePos *orig);
static int wk1_std_func	   (GnmExprList **stack, Wk1Func const *func, guint8 const *data, const GnmParsePos *orig);
/* a,b,c -> a,,-c,b */
static int wk1_nper_func   (GnmExprList **stack, Wk1Func const *func, guint8 const *data, const GnmParsePos *orig);
/* year - 1900 */
static int wk1_year_func   (GnmExprList **stack, Wk1Func const *func, guint8 const *data, const GnmParsePos *orig);
/* find - 1 */
static int wk1_find_func   (GnmExprList **stack, Wk1Func const *func, guint8 const *data, const GnmParsePos *orig);
/* a,b,c -> b,c,-a */
static int wk1_fv_pv_pmt_func (GnmExprList **stack, Wk1Func const *func, guint8 const *data, const GnmParsePos *orig);
/* a,b -> b,a */
static int wk1_irr_func    (GnmExprList **stack, Wk1Func const *func, guint8 const *data, const GnmParsePos *orig);
static int wk1_rate_func   (GnmExprList **stack, Wk1Func const *func, guint8 const *data, const GnmParsePos *orig);

static const Wk1Func functions[] = {
	{  1, 0x08, "-",	wk1_unary_func,  GNM_EXPR_OP_UNARY_NEG },
	{  2, 0x09, "+",	wk1_binary_func, GNM_EXPR_OP_ADD },
	{  2, 0x0A, "-",	wk1_binary_func, GNM_EXPR_OP_SUB },
	{  2, 0x0B, "*",	wk1_binary_func, GNM_EXPR_OP_MULT },
	{  2, 0x0C, "/",	wk1_binary_func, GNM_EXPR_OP_DIV },
	{  2, 0x0D, "pow",	wk1_binary_func, GNM_EXPR_OP_EXP },
	{  2, 0x0E, "EQ",	wk1_binary_func, GNM_EXPR_OP_EQUAL },
	{  2, 0x0F, "NE",	wk1_binary_func, GNM_EXPR_OP_NOT_EQUAL },
	{  2, 0x10, "LE",	wk1_binary_func, GNM_EXPR_OP_LTE },
	{  2, 0x11, "GE",	wk1_binary_func, GNM_EXPR_OP_GTE },
	{  2, 0x12, "LT",	wk1_binary_func, GNM_EXPR_OP_LT },
	{  2, 0x13, "GT",	wk1_binary_func, GNM_EXPR_OP_GT },
	{  2, 0x14, "BITAND",	wk1_std_func, 0 },	/* from number theory */
	{  2, 0x15, "BITOR",	wk1_std_func, 0 },	/* from number theory */
	{  1, 0x16, "NOT",	wk1_std_func, 0 },
	{  1, 0x17, "+",	wk1_unary_func, GNM_EXPR_OP_UNARY_PLUS },
	{  0, 0x1F, "NA",	wk1_std_func, 0 },
	{  1, 0x20, "ERR",	wk1_std_func, 0 },
	{  1, 0x21, "ABS",	wk1_std_func, 0 },
	{  1, 0x22, "FLOOR",	wk1_std_func, 0 },
	{  1, 0x23, "SQRT",	wk1_std_func, 0 },
	{  1, 0x24, "LOG",	wk1_std_func, 0 },
	{  1, 0x25, "LN",	wk1_std_func, 0 },
	{  0, 0x26, "PI",	wk1_std_func, 0 },
	{  1, 0x27, "SIN",	wk1_std_func, 0 },
	{  1, 0x28, "COS",	wk1_std_func, 0 },
	{  1, 0x29, "TAN",	wk1_std_func, 0 },
	{  2, 0x2A, "ATAN2",	wk1_std_func, 0 },
	{  1, 0x2B, "ATAN",	wk1_std_func, 0 },
	{  1, 0x2C, "ASIN",	wk1_std_func, 0 },
	{  1, 0x2D, "ACOS",	wk1_std_func, 0 },
	{  1, 0x2E, "EXP",	wk1_std_func, 0 },
	{  2, 0x2F, "MOD",	wk1_std_func, 0 },
	{ -1, 0x30, "CHOOSE",	wk1_std_func, 0 },
	{  1, 0x31, "ISNA",	wk1_std_func, 0 },
	{  1, 0x32, "ISERR",	wk1_std_func, 0 },
	{  0, 0x33, "FALSE",	wk1_std_func, 0 },
	{  0, 0x34, "TRUE",	wk1_std_func, 0 },
	{  0, 0x35, "RAND",	wk1_std_func, 0 },
	{  3, 0x36, "DATE",	wk1_std_func, 0 },
	{  0, 0x37, "TODAY",	wk1_std_func, 0 },
	{  3, 0x38, "PMT",	wk1_fv_pv_pmt_func, 0 },
	{  3, 0x39, "PV",	wk1_fv_pv_pmt_func, 0 },
	{  3, 0x3A, "FV",	wk1_fv_pv_pmt_func, 0 },
	{  3, 0x3B, "IF",	wk1_std_func, 0 },
	{  1, 0x3C, "DAY",	wk1_std_func, 0 },
	{  1, 0x3D, "MONTH",	wk1_std_func, 0 },
	{  1, 0x3E, "YEAR",	wk1_year_func, 0 },
	{  2, 0x3F, "ROUND",	wk1_std_func, 0 },
	{  3, 0x40, "TIME",	wk1_std_func, 0 },
	{  1, 0x41, "HOUR",	wk1_std_func, 0 },
	{  1, 0x42, "MINUTE",	wk1_std_func, 0 },
	{  1, 0x43, "SECOND",	wk1_std_func, 0 },
	{  1, 0x44, "ISNUMBER",	wk1_std_func, 0 },
	{  1, 0x45, "ISTEXT",	wk1_std_func, 0 },
	{  1, 0x46, "LEN",	wk1_std_func, 0 },
	{  1, 0x47, "VALUE",	wk1_std_func, 0 },
	{  2, 0x48, "FIXED",	wk1_std_func, 0 },
	{  3, 0x49, "MID",	wk1_std_func, 0 },
	{  1, 0x4A, "CHAR",	wk1_std_func, 0 },
	{  1, 0x4B, "CODE",	wk1_std_func, 0 },
	{  3, 0x4C, "FIND",	wk1_find_func, 0 },
	{  1, 0x4D, "DATEVALUE",wk1_std_func, 0 },
	{  1, 0x4E, "TIMEVALUE",wk1_std_func, 0 },
	{  1, 0x4F, "CELLPOINTER", wk1_std_func, 0 },
	{ -1, 0x50, "SUM",	wk1_std_func, 0 },
	{ -1, 0x51, "AVERAGE",	wk1_std_func, 0 },
	{ -1, 0x52, "COUNT",	wk1_std_func, 0 },
	{ -1, 0x53, "MIN",	wk1_std_func, 0 },
	{ -1, 0x54, "MAX",	wk1_std_func, 0 },
	{  3, 0x55, "VLOOKUP",	wk1_std_func, 0 },
	{  2, 0x56, "NPV",	wk1_std_func, 0 },
	{ -1, 0x57, "VAR",	wk1_std_func, 0 },
	{ -1, 0x58, "STDEVP",	wk1_std_func, 0 },
	{  2, 0x59, "IRR",	wk1_irr_func, 0 },
	{  3, 0x5A, "HLOOKUP",	wk1_std_func, 0 },
	{ -2, 0x5B, "DSUM",	wk1_std_func, 0 },
	{ -2, 0x5C, "DAVERAGE",	wk1_std_func, 0 },
	{ -2, 0x5D, "DCOUNT",	wk1_std_func, 0 },
	{ -2, 0x5E, "DMIN",	wk1_std_func, 0 },
	{ -2, 0x5F, "DMAX",	wk1_std_func, 0 },
	{ -2, 0x60, "DVAR",	wk1_std_func, 0 },
	{ -2, 0x61, "DSTD",	wk1_std_func, 0 },
	{ -3, 0x62, "INDEX",	wk1_std_func, 0 },
	{  1, 0x63, "COLUMNS",	wk1_std_func, 0 },
	{  1, 0x64, "ROWS",	wk1_std_func, 0 },
	{  2, 0x65, "REPT",	wk1_std_func, 0 },
	{  1, 0x66, "UPPER",	wk1_std_func, 0 },
	{  1, 0x67, "LOWER",	wk1_std_func, 0 },
	{  2, 0x68, "LEFT",	wk1_std_func, 0 },
	{  2, 0x69, "RIGHT",	wk1_std_func, 0 },
	{  4, 0x6A, "REPLACE",	wk1_std_func, 0 },
	{  1, 0x6B, "PROPER",	wk1_std_func, 0 },
	{  2, 0x6C, "CELL",	wk1_std_func, 0 },
	{  1, 0x6D, "TRIM",	wk1_std_func, 0 },
	{  1, 0x6E, "CLEAN",	wk1_std_func, 0 },
	{  1, 0x6F, "S",        wk1_std_func, 0 },
	{  1, 0x70, "N",        wk1_std_func, 0 },
	{  2, 0x71, "EXACT",	wk1_std_func, 0 },
	{  1, 0x72, "CALL",	wk1_std_func, 0 },
	{  1, 0x73, "INDIRECT",	wk1_std_func, 0 },
	{  3, 0x74, "RATE",	wk1_rate_func, 0 },
	{  3, 0x75, "TERM",	wk1_std_func, 0 },
	{  3, 0x76, "NPER",	wk1_nper_func, 0 },
	{  3, 0x77, "SLN",	wk1_std_func, 0 },
	{  4, 0x78, "SYD",	wk1_std_func, 0 },
	{  4, 0x79, "DDB",	wk1_std_func, 0 },
	/* 0x7A is SPLFUNC which needs special handling */
	/* WK4 and up.  This list form wkrel9.txt */
	{  1, 0x7B, "SHEETS",	wk1_std_func, 0 },
	{  1, 0x7C, "INFO",	wk1_std_func, 0 },
	{ -1, 0x7D, "SUMPRODUCT",wk1_std_func, 0 },
	{  1, 0x7E, "ISRANGE",	wk1_std_func, 0 },
	{ -1, 0x7F, "DGET",	wk1_std_func, 0 },
	{ -1, 0x80, "DQUERY",	wk1_std_func, 0 },
	{  4, 0x81, "COORD",	wk1_std_func, 0 },
	{  0, 0x83, "TODAY",	wk1_std_func, 0 },
	{ -1, 0x84, "VDB",	wk1_std_func, 0 },
	{ -1, 0x85, "DVARS",	wk1_std_func, 0 },
	{ -1, 0x86, "DSTDS",	wk1_std_func, 0 },
	{ -1, 0x87, "VARS",	wk1_std_func, 0 },
	{ -1, 0x88, "STDS",	wk1_std_func, 0 },
	{  2, 0x89, "D360",	wk1_std_func, 0 },
	{  1, 0x8B, "ISAPP",	wk1_std_func, 0 },
	{  1, 0x8C, "ISAAF",	wk1_std_func, 0 },
	{  1, 0x8D, "WEEKDAY",	wk1_std_func, 0 },
	{  3, 0x8E, "DATEDIF",	wk1_std_func, 0 },
	{ -1, 0x8F, "RANK",	wk1_std_func, 0 },
	{  2, 0x90, "NUMBERSTRING",wk1_std_func, 0 },
	{  1, 0x91, "DATESTRING",wk1_std_func, 0 },
	{  1, 0x92, "DECIMAL",	wk1_std_func, 0 },
	{  1, 0x93, "HEX",	wk1_std_func, 0 },
	{  4, 0x94, "DB",	wk1_std_func, 0 },
	{  4, 0x95, "PMTI",	wk1_std_func, 0 },
	{  4, 0x96, "SPI",	wk1_std_func, 0 },
	{  1, 0x97, "FULLP",	wk1_std_func, 0 },
	{  1, 0x98, "HALFP",	wk1_std_func, 0 },
	{ -1, 0x99, "PUREAVG",	wk1_std_func, 0 },
	{ -1, 0x9A, "PURECOUNT",wk1_std_func, 0 },
	{ -1, 0x9B, "PUREMAX",	wk1_std_func, 0 },
	{ -1, 0x9C, "PUREMIN",	wk1_std_func, 0 },
	{ -1, 0x9D, "PURESTD",	wk1_std_func, 0 },
	{ -1, 0x9E, "PUREVAR",	wk1_std_func, 0 },
	{ -1, 0x9F, "PURESTDS",	wk1_std_func, 0 },
	{ -1, 0xA0, "PUREVARS",	wk1_std_func, 0 },
	{  3, 0xA1, "PMT2",	wk1_std_func, 0 },
	{  3, 0xA2, "PV2",	wk1_std_func, 0 },
	{  3, 0xA3, "FV2",	wk1_std_func, 0 },
	{  3, 0xA4, "TERM2",	wk1_std_func, 0 }
};

static void
parse_list_push_expr (GnmExprList **list, GnmExpr const *pd)
{
	g_return_if_fail (pd != NULL);
	*list = gnm_expr_list_prepend (*list, pd) ;
}

static void
parse_list_push_value (GnmExprList **list, GnmValue *v)
{
	parse_list_push_expr (list, gnm_expr_new_constant (v));
}

static GnmExpr const *
parse_list_pop (GnmExprList **list, const GnmParsePos *orig)
{
	GnmExprList *tmp = *list;
	if (tmp != NULL) {
		GnmExpr const *ans = tmp->data ;
		*list = g_slist_remove (*list, ans) ;
		return ans ;
	}

	g_warning ("%s: Incorrect number of parsed formula arguments",
		   cell_coord_name (orig->eval.col, orig->eval.row));
	return gnm_expr_new_constant (value_new_error_REF (NULL));
}

/**
 * Returns a new list composed of the last n items pop'd off the list.
 **/
static GnmExprList *
parse_list_last_n (GnmExprList **list, gint n, const GnmParsePos *orig)
{
	GnmExprList *l = NULL;
	while (n-- > 0)
		l = gnm_expr_list_prepend (l, parse_list_pop (list, orig));
	return l;
}


static int
wk1_unary_func  (GnmExprList **stack, Wk1Func const *f,
		 guint8 const *data, const GnmParsePos *orig)
{
	GnmExpr const *r = parse_list_pop (stack, orig);
	parse_list_push_expr (stack, gnm_expr_new_unary (f->data, r));
	return 1;
}
static int
wk1_binary_func (GnmExprList **stack, Wk1Func const *f,
		 guint8 const *data, const GnmParsePos *orig)
{
	GnmExpr const *r = parse_list_pop (stack, orig);
	GnmExpr const *l = parse_list_pop (stack, orig);
	parse_list_push_expr (stack, gnm_expr_new_binary (l, f->data, r));
	return 1;
}

static int
wk1_std_func (GnmExprList **stack, Wk1Func const *f,
	      guint8 const *data, const GnmParsePos *orig)
{
	GnmFunc *func = gnm_func_lookup (f->name, NULL);
	int numargs, size;

	if (f->args < 0) {
		numargs = data[1];
		size = 2;
	} else {
		numargs = f->args;
		size = 1;
	}

	if (func == NULL) {
		func = gnm_func_add_placeholder (NULL, f->name, "Lotus ", TRUE);
		puts (cell_coord_name (orig->eval.col, orig->eval.row));
	}
	parse_list_push_expr (stack, gnm_expr_new_funcall (func,
		parse_list_last_n (stack, numargs, orig)));

	return size;
}
static int
wk1_nper_func (GnmExprList **stack, Wk1Func const *func,
	       guint8 const *data, const GnmParsePos *orig)
{
	/* a,b,c -> a,,-c,b */
	return wk1_std_func (stack, func, data, orig);
}

static int
wk1_year_func (GnmExprList **stack, Wk1Func const *func,
	       guint8 const *data, const GnmParsePos *orig)
{
	/* year - 1900 */
	return wk1_std_func (stack, func, data, orig);
}

static int
wk1_find_func (GnmExprList **stack, Wk1Func const *func,
	       guint8 const *data, const GnmParsePos *orig)
{
	/* find - 1 */
	return wk1_std_func (stack, func, data, orig);
}

static int
wk1_fv_pv_pmt_func (GnmExprList **stack, Wk1Func const *func,
		     guint8 const *data, const GnmParsePos *orig)
{
	/* a,b,c -> b,c,-a */
	return wk1_std_func (stack, func, data, orig);
}

static int
wk1_irr_func (GnmExprList **stack, Wk1Func const *func,
		 guint8 const *data, const GnmParsePos *orig)
{
	/* a,b -> b,a */
	return wk1_std_func (stack, func, data, orig);
}

static int
wk1_rate_func (GnmExprList **stack, Wk1Func const *func,
	       guint8 const *data, const GnmParsePos *orig)
{
	return wk1_std_func (stack, func, data, orig);
}

static gint32
make_function (GnmExprList **stack, guint8 const *data, const GnmParsePos *orig)
{
	Wk1Func const *f = NULL;
	unsigned i;

	for (i = 0; i < G_N_ELEMENTS (functions); i++)
		if (*data == functions [i].idx) {
			f = functions + i;
			break;
		}

	if (f == NULL) {
		g_warning ("%s: unknown PTG 0x%x",
			   cell_coord_name (orig->eval.col, orig->eval.row),
			   *data);
		return 1;
	}

	return (f->handler) (stack, f, data, orig);
}

static gint16
sign_extend (guint16 num)
{
	gint16 i = (num << 3);
	return (i / 8);
}

/* FIXME: dodgy stuff, hacked for now */
static void
get_cellref (GnmCellRef *ref, guint8 const *dataa, guint8 const *datab,
	     const GnmParsePos *orig)
{
	guint16 i;

	ref->sheet = NULL;
	i = GSF_LE_GET_GUINT16 (dataa);
	if (i & 0x8000) {
		ref->col_relative = TRUE;
		ref->col = sign_extend (i & 0x3fff);
	} else {
		ref->col_relative = FALSE;
		ref->col = i & 0x3fff;
	}

	i = GSF_LE_GET_GUINT16 (datab);
	if (i & 0x8000) {
		ref->row_relative = TRUE;
		ref->row = sign_extend (i & 0x3fff);
	} else {
		ref->row_relative = FALSE;
		ref->row = i & 0x3fff;
	}

#if FORMULA_DEBUG > 0
	printf ("0x%x 0x%x -> (%d, %d)\n", *(guint16 *)dataa, *(guint16 *)datab,
		ref->col, ref->row);
#endif
}

static GnmExpr const *
lotus_parse_formula_old (LotusState *state, GnmParsePos *orig,
			 guint8 const *data, guint32 len)
{
	GnmExprList *stack = NULL;
	guint     i;
	GnmCellRef   a, b;
	gboolean done = FALSE;

	for (i = 0; (i < len) && !done;) {
		switch (data[i]) {
		case LOTUS_FORMULA_CONSTANT:
			parse_list_push_value (&stack,
				value_new_float (gsf_le_get_double (data + i + 1)));
			i += 9;
			break;

		case LOTUS_FORMULA_VARIABLE:
			get_cellref (&a, data + i + 1, data + i + 3, orig);
			parse_list_push_expr (&stack, gnm_expr_new_cellref (&a));
			i += 5;
			break;

		case LOTUS_FORMULA_RANGE:
			get_cellref (&a, data + i + 1, data + i + 3, orig);
			get_cellref (&b, data + i + 5, data + i + 7, orig);
			parse_list_push_value (&stack,
				value_new_cellrange (&a, &b, orig->eval.col, orig->eval.row));
			i += 9;
			break;

		case LOTUS_FORMULA_RETURN:
			done = TRUE;
			break;

		case LOTUS_FORMULA_BRACKET:
			i += 1; /* Ignore */
			break;

		case LOTUS_FORMULA_INTEGER:
			parse_list_push_value (&stack,
					       value_new_int (GSF_LE_GET_GINT16 (data + i + 1)));
			i += 3;
			break;

		case LOTUS_FORMULA_STRING:
			parse_list_push_value (&stack,
				lotus_new_string (data + i + 1));
			i += 2 + strlen (data + i + 1);
			break;

		case LOTUS_FORMULA_UNARY_PLUS:
			i++;
			break;

		default:
			i += make_function (&stack, data + i, orig);
		}
	}

	if (gnm_expr_list_length (stack) != 1)
		g_warning ("%s: args remain on stack",
			   cell_coord_name (orig->eval.col, orig->eval.row));
	return parse_list_pop (&stack, orig);
}

static void
get_new_cellref (GnmCellRef *dst, int relbits, const guint8 *data,
		 const GnmParsePos *orig)
{
	dst->row = GSF_LE_GET_GUINT16 (data);
	dst->sheet = lotus_get_sheet (orig->sheet->workbook, data[2]);
	dst->col = data[3];

	dst->row_relative = (relbits & 1) != 0;
	if (dst->row_relative)
		dst->row -= orig->eval.row;

	dst->col_relative = (relbits & 2) != 0;
	if (dst->col_relative)
		dst->col -= orig->eval.col;
}


#define HANDLE_BINARY(op)						\
  {									\
	GnmExpr const *r = parse_list_pop (&stack, orig);		\
	GnmExpr const *l = parse_list_pop (&stack, orig);		\
	parse_list_push_expr (&stack, gnm_expr_new_binary (l, op, r));	\
        i++;								\
        break;								\
  }

#define HANDLE_UNARY(op)						\
  {									\
	GnmExpr const *a = parse_list_pop (&stack, orig);		\
	parse_list_push_expr (&stack, gnm_expr_new_unary (op, a));	\
        i++;								\
        break;								\
  }

static void
handle_named_func (GnmExprList **stack, const GnmParsePos *orig,
		   const char *name, int args)
{
	GnmFunc *func = gnm_func_lookup (name, NULL);
	if (!func) {
		/* A hack, for now.  */
		if (strcmp (name, "ISEMPTY") == 0)
			name = "ISBLANK";
		func = gnm_func_lookup (name, NULL);
	}

	if (func == NULL)
		func = gnm_func_add_placeholder (NULL, name, "Lotus ", TRUE);
	parse_list_push_expr (stack, gnm_expr_new_funcall (func,
		parse_list_last_n (stack, args, orig)));
}


static GnmExpr const *
lotus_parse_formula_new (LotusState *state, GnmParsePos *orig,
			 guint8 const *data, guint32 len)
{
	GnmExprList *stack = NULL;
	guint i;
	gboolean done = FALSE;

	for (i = 0; i < len && !done;) {
		switch (data[i]) {
		case LOTUS_FORMULA_CONSTANT:
			parse_list_push_value (&stack,
				value_new_float (gsf_le_get_double (data + i + 1)));
			i += 9;
			break;

		case LOTUS_FORMULA_VARIABLE: {
			GnmCellRef a;
			get_new_cellref (&a, data[1] & 7, data + i + 2, orig);
			if (a.sheet == orig->sheet)
				a.sheet = NULL;
			parse_list_push_expr (&stack, gnm_expr_new_cellref (&a));
			i += 6;
			break;
		}

		case LOTUS_FORMULA_RANGE: {
			GnmCellRef a, b;
			get_new_cellref (&a, data[1] & 7, data + i + 2, orig);
			get_new_cellref (&b, (data[1] >> 3) & 7, data + i + 6, orig);
			if (b.sheet == a.sheet)
				b.sheet = NULL;
			if (a.sheet == orig->sheet && b.sheet == NULL)
				a.sheet = NULL;
			parse_list_push_value
				(&stack,
				 value_new_cellrange (&a, &b,
						      orig->eval.col,
						      orig->eval.row));
			i += 10;
			break;
		}

		case LOTUS_FORMULA_RETURN:
			done = TRUE;
			break;

		case LOTUS_FORMULA_BRACKET:
			i += 1; /* Ignore */
			break;

		case LOTUS_FORMULA_PACKED_NUMBER: {
			double v = lotus_unpack_number (GSF_LE_GET_GUINT32 (data + i + 1));
			GnmValue *val;

			if (v == gnm_floor (v) && v >= G_MININT && v <= G_MAXINT)
				val = value_new_int ((int)v);
			else
				val = value_new_float (v);
			parse_list_push_value (&stack, val);
			i += 5;
			break;
		}

		case LOTUS_FORMULA_STRING:
			parse_list_push_value (&stack,
				lotus_new_string (data + i + 1));
			i += 2 + strlen (data + i + 1);
			break;

		case LOTUS_FORMULA_NAMED:
		case LOTUS_FORMULA_ABS_NAMED:
			g_warning ("Named ranges not implemented.");
			i += 1;  /* Guess */
			break;

		case LOTUS_FORMULA_ERR_RREF:
			parse_list_push_value (&stack,
					       value_new_error_REF (NULL));
			i += 5;
			break;

		case LOTUS_FORMULA_ERR_CREF:
			parse_list_push_value (&stack,
					       value_new_error_REF (NULL));
			i += 6;
			break;


		case LOTUS_FORMULA_ERR_CONSTANT:
			parse_list_push_value (&stack,
					       value_new_error_VALUE (NULL));
			i += 12;
			break;

		case LOTUS_FORMULA_OP_NEG: HANDLE_UNARY (GNM_EXPR_OP_UNARY_PLUS);
		case LOTUS_FORMULA_OP_PLU: HANDLE_BINARY (GNM_EXPR_OP_ADD);
		case LOTUS_FORMULA_OP_MNS: HANDLE_BINARY (GNM_EXPR_OP_SUB);
		case LOTUS_FORMULA_OP_MUL: HANDLE_BINARY (GNM_EXPR_OP_MULT);
		case LOTUS_FORMULA_OP_DIV: HANDLE_BINARY (GNM_EXPR_OP_DIV);
		case LOTUS_FORMULA_OP_POW: HANDLE_BINARY (GNM_EXPR_OP_EXP);
		case LOTUS_FORMULA_OP_EQ: HANDLE_BINARY (GNM_EXPR_OP_EQUAL);
		case LOTUS_FORMULA_OP_NE: HANDLE_BINARY (GNM_EXPR_OP_NOT_EQUAL);
		case LOTUS_FORMULA_OP_LE: HANDLE_BINARY (GNM_EXPR_OP_LTE);
		case LOTUS_FORMULA_OP_GE: HANDLE_BINARY (GNM_EXPR_OP_GTE);
		case LOTUS_FORMULA_OP_LT: HANDLE_BINARY (GNM_EXPR_OP_LT);
		case LOTUS_FORMULA_OP_GT: HANDLE_BINARY (GNM_EXPR_OP_GT);
		case LOTUS_FORMULA_OP_UPLU: HANDLE_UNARY (GNM_EXPR_OP_UNARY_PLUS);
		case LOTUS_FORMULA_OP_CAT: HANDLE_BINARY (GNM_EXPR_OP_CAT);

		case LOTUS_FORMULA_OP_AND:
			/* FIXME: Check if we need bit versions.  */
			handle_named_func (&stack, orig, "AND", 2);
			break;
		case LOTUS_FORMULA_OP_OR:
			handle_named_func (&stack, orig, "OR", 2);
			break;
		case LOTUS_FORMULA_OP_NOT:
			handle_named_func (&stack, orig, "NOT", 1);
			break;

		case LOTUS_FORMULA_SPLFUNC: {
			int args = data[i + 1];
			int fnamelen = GSF_LE_GET_GUINT16 (data + i + 2);
			char *name = lotus_get_lmbcs (data + (i + 4), len - (i + 4));
			size_t namelen;
			char *p;

			if (name == NULL)
				name = g_strdup ("bogus");

			/* Get rid of the '(' in the name.  */
			namelen = strlen (name);
			if (namelen && name[namelen - 1] == '(')
				name[--namelen] = 0;
			/* There is a weird prefix -- ignore it.  */
			for (p = name + namelen; p > name; p--)
				if (!g_ascii_isalnum (p[-1]))
					break;
			handle_named_func (&stack, orig, p, args);
			g_free (name);
			i += 4 + fnamelen;
			break;
		}

		default:
			i += make_function (&stack, data + i, orig);
		}
	}

	if (gnm_expr_list_length (stack) != 1)
		g_warning ("%s: args remain on stack",
			   cell_coord_name (orig->eval.col, orig->eval.row));
	return parse_list_pop (&stack, orig);
}


GnmExpr const *
lotus_parse_formula (LotusState *state, GnmParsePos *pos,
		     guint8 const *data, guint32 len)
{
	const GnmExpr *result = (state->version >= LOTUS_VERSION_123V6)
		? lotus_parse_formula_new (state, pos, data, len)
		: lotus_parse_formula_old (state, pos, data, len);

#if FORMULA_DEBUG > 0
	{
		char *txt = gnm_expr_as_string (result, pos, gnm_expr_conventions_default);
		g_print ("Lotus: %s!%s: %s\n",
			 pos->sheet->name_unquoted,
			 cell_coord_name (pos->eval.col, pos->eval.row),
			 txt);
		g_free (txt);
	}
#endif

	return result;
}
