"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TimeGraphStateController = void 0;
var TimeGraphStateController = /** @class */ (function () {
    function TimeGraphStateController(canvas, unitController) {
        var _this = this;
        this.canvas = canvas;
        this.unitController = unitController;
        this._worldRenderedHandlers = [];
        this.zoomChangedHandlers = [];
        this.canvasWidthChangedHandlers = [];
        this.positionChangedHandlers = [];
        this.scaleFactorChangedHandlers = [];
        this.updateZoomFactor = function () {
            var newZoom = _this.canvasDisplayWidth / Number(_this.unitController.viewRangeLength);
            if (_this._zoomFactor !== newZoom) {
                _this.handleZoomChange(_this._zoomFactor = newZoom);
            }
        };
        // Adjust the scale factor if the view range changes
        this.updateScaleFactor = function (oldViewRange, newViewRange) {
            var oldViewRangeLength = oldViewRange.end - oldViewRange.start;
            var newViewRangeLength = newViewRange.end - newViewRange.start;
            var newScaleFactor = Number(oldViewRangeLength) / Number(newViewRangeLength) * _this._scaleFactor;
            _this.scaleFactor = newScaleFactor;
        };
        this.onWorldRender = function (handler) {
            _this._worldRenderedHandlers.push(handler);
        };
        this.handleOnWorldRender = function () {
            _this._worldRenderedHandlers.forEach(function (handler) { return handler(_this.unitController.worldRange); });
        };
        this.removeWorldRenderHandler = function (handler) {
            var index = _this._worldRenderedHandlers.indexOf(handler);
            if (index > -1) {
                _this._worldRenderedHandlers.splice(index, 1);
            }
        };
        this.ratio = window.devicePixelRatio;
        this._canvasDisplayWidth = this.canvas.width / this.ratio;
        this._canvasDisplayHeight = this.canvas.height / this.ratio;
        this._initialZoomFactor = this.zoomFactor;
        this._positionOffset = { x: 0, y: 0 };
        this.oldPositionOffset = { x: 0, y: 0 };
        this.snapped = false;
        this._unscaledCanvasWidth = this._canvasDisplayWidth;
        this._scaleFactor = 1;
        this.unitController.onViewRangeChanged(this.updateZoomFactor);
        this.unitController.onViewRangeChanged(this.updateScaleFactor);
    }
    TimeGraphStateController.prototype.handleZoomChange = function (zoomFactor) {
        this.zoomChangedHandlers.forEach(function (handler) { return handler(zoomFactor); });
    };
    TimeGraphStateController.prototype.handlePositionChange = function () {
        this.positionChangedHandlers.forEach(function (handler) { return handler(); });
    };
    TimeGraphStateController.prototype.handleScaleFactorChange = function () {
        var _this = this;
        this.scaleFactorChangedHandlers.forEach(function (handler) { return handler(_this._scaleFactor); });
    };
    TimeGraphStateController.prototype.onZoomChanged = function (handler) {
        this.zoomChangedHandlers.push(handler);
    };
    TimeGraphStateController.prototype.onPositionChanged = function (handler) {
        this.positionChangedHandlers.push(handler);
    };
    TimeGraphStateController.prototype.onScaleFactorChange = function (handler) {
        this.scaleFactorChangedHandlers.push(handler);
    };
    TimeGraphStateController.prototype.removeOnZoomChanged = function (handler) {
        var index = this.zoomChangedHandlers.indexOf(handler);
        if (index > -1) {
            this.zoomChangedHandlers.splice(index, 1);
        }
    };
    TimeGraphStateController.prototype.removeOnScaleFactorChanged = function (handler) {
        var index = this.scaleFactorChangedHandlers.indexOf(handler);
        if (index > -1) {
            this.scaleFactorChangedHandlers.splice(index, 1);
        }
    };
    Object.defineProperty(TimeGraphStateController.prototype, "canvasDisplayWidth", {
        /**
            It is not the width of the canvas display buffer but of the canvas element in browser. Can be different depending on the display pixel ratio.
        */
        get: function () {
            return this._canvasDisplayWidth;
        },
        enumerable: false,
        configurable: true
    });
    TimeGraphStateController.prototype.updateDisplayWidth = function () {
        this._canvasDisplayWidth = this.canvas.width / this.ratio;
        // Adjust the scale factor if the display canvas width changes
        this.scaleFactor = this._canvasDisplayWidth / this._unscaledCanvasWidth;
    };
    Object.defineProperty(TimeGraphStateController.prototype, "canvasDisplayHeight", {
        /**
            It is not the heigth of the canvas display buffer but of the canvas element in browser. Can be different depending on the display pixel ratio.
        */
        get: function () {
            return this._canvasDisplayHeight;
        },
        enumerable: false,
        configurable: true
    });
    TimeGraphStateController.prototype.updateDisplayHeight = function () {
        this._canvasDisplayHeight = this.canvas.height / this.ratio;
    };
    Object.defineProperty(TimeGraphStateController.prototype, "initialZoomFactor", {
        get: function () {
            return this._initialZoomFactor;
        },
        enumerable: false,
        configurable: true
    });
    TimeGraphStateController.prototype.resetScale = function () {
        this._unscaledCanvasWidth = this._canvasDisplayWidth;
        this.scaleFactor = 1;
    };
    Object.defineProperty(TimeGraphStateController.prototype, "scaleFactor", {
        get: function () {
            return this._scaleFactor;
        },
        set: function (newScaleFactor) {
            this._scaleFactor = newScaleFactor;
            this.handleScaleFactorChange();
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(TimeGraphStateController.prototype, "zoomFactor", {
        get: function () {
            this.updateZoomFactor();
            return this._zoomFactor;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(TimeGraphStateController.prototype, "absoluteResolution", {
        get: function () {
            return this.canvasDisplayWidth / Number(this.unitController.absoluteRange);
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(TimeGraphStateController.prototype, "positionOffset", {
        get: function () {
            return this._positionOffset;
        },
        set: function (value) {
            this._positionOffset = value;
            this.handlePositionChange();
        },
        enumerable: false,
        configurable: true
    });
    TimeGraphStateController.prototype.removeHandlers = function () {
        this.unitController.removeViewRangeChangedHandler(this.updateZoomFactor);
        this.unitController.removeViewRangeChangedHandler(this.updateScaleFactor);
    };
    return TimeGraphStateController;
}());
exports.TimeGraphStateController = TimeGraphStateController;
//# sourceMappingURL=time-graph-state-controller.js.map