"use strict";
// *****************************************************************************
// Copyright (C) 2022 Red Hat, Inc. and others.
//
// This program and the accompanying materials are made available under the
// terms of the Eclipse Public License v. 2.0 which is available at
// http://www.eclipse.org/legal/epl-2.0.
//
// This Source Code may also be made available under the following Secondary
// Licenses when the conditions for such availability set forth in the Eclipse
// Public License v. 2.0 are satisfied: GNU General Public License, version 2
// with the GNU Classpath Exception which is available at
// https://www.gnu.org/software/classpath/license.html.
//
// SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
// *****************************************************************************
/* eslint-disable @typescript-eslint/no-explicit-any */
Object.defineProperty(exports, "__esModule", { value: true });
exports.registerMsgPackExtensions = exports.MsgPackMessageDecoder = exports.MsgPackMessageEncoder = exports.defaultMsgPack = exports.EncodingError = exports.ResponseError = void 0;
const msgpackr_1 = require("msgpackr");
const msg_pack_extension_manager_1 = require("./msg-pack-extension-manager");
/**
 * A special error that can be returned in case a request
 * has failed. Provides additional information i.e. an error code
 * and additional error data.
 */
class ResponseError extends Error {
    constructor(code, message, data) {
        super(message);
        this.code = code;
        this.data = data;
    }
}
exports.ResponseError = ResponseError;
/**
 * Custom error thrown by the {@link RpcMessageEncoder} if an error occurred during the encoding and the
 * object could not be written to the given {@link WriteBuffer}
 */
class EncodingError extends Error {
    constructor(msg, cause) {
        super(msg);
        this.cause = cause;
    }
}
exports.EncodingError = EncodingError;
exports.defaultMsgPack = new msgpackr_1.Packr({ moreTypes: true, encodeUndefinedAsNil: false, bundleStrings: false });
class MsgPackMessageEncoder {
    constructor(msgPack = exports.defaultMsgPack) {
        this.msgPack = msgPack;
    }
    cancel(buf, requestId) {
        this.encode(buf, { type: 5 /* Cancel */, id: requestId });
    }
    notification(buf, requestId, method, args) {
        this.encode(buf, { type: 2 /* Notification */, id: requestId, method, args });
    }
    request(buf, requestId, method, args) {
        this.encode(buf, { type: 1 /* Request */, id: requestId, method, args });
    }
    replyOK(buf, requestId, res) {
        this.encode(buf, { type: 3 /* Reply */, id: requestId, res });
    }
    replyErr(buf, requestId, err) {
        this.encode(buf, { type: 4 /* ReplyErr */, id: requestId, err });
    }
    encode(buf, value) {
        try {
            buf.writeBytes(this.msgPack.encode(value));
        }
        catch (err) {
            if (err instanceof Error) {
                throw new EncodingError(`Error during encoding: '${err.message}'`, err);
            }
            throw err;
        }
    }
}
exports.MsgPackMessageEncoder = MsgPackMessageEncoder;
class MsgPackMessageDecoder {
    constructor(msgPack = exports.defaultMsgPack) {
        this.msgPack = msgPack;
    }
    decode(buf) {
        const bytes = buf.readBytes();
        return this.msgPack.decode(bytes);
    }
    parse(buffer) {
        return this.decode(buffer);
    }
}
exports.MsgPackMessageDecoder = MsgPackMessageDecoder;
function registerMsgPackExtensions() {
    // Register custom msgPack extension for Errors.
    msg_pack_extension_manager_1.MsgPackExtensionManager.getInstance().registerExtensions({
        class: Error,
        tag: 1,
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        serialize: (error) => {
            var _a;
            const { code, data, message, name } = error;
            const stack = (_a = error.stacktrace) !== null && _a !== void 0 ? _a : error.stack;
            const isResponseError = error instanceof ResponseError;
            return { code, data, message, name, stack, isResponseError };
        },
        deserialize: data => {
            const error = data.isResponseError ? new ResponseError(data.code, data.message, data.data) : new Error(data.message);
            error.name = data.name;
            error.stack = data.stack;
            return error;
        }
    });
}
exports.registerMsgPackExtensions = registerMsgPackExtensions;
//# sourceMappingURL=rpc-message-encoder.js.map