/*******************************************************************************
 * Copyright (c) 2011, 2025 Google, Inc. and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * https://www.eclipse.org/legal/epl-2.0.
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *    Google, Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.wb.internal.core.eval.evaluators;

import org.eclipse.wb.core.eval.AstEvaluationEngine;
import org.eclipse.wb.core.eval.EvaluationContext;
import org.eclipse.wb.core.eval.IExpressionEvaluator;
import org.eclipse.wb.internal.core.utils.ast.AstNodeUtils;
import org.eclipse.wb.internal.core.utils.check.Assert;
import org.eclipse.wb.internal.core.utils.reflect.ReflectionUtils;

import org.eclipse.jdt.core.dom.Expression;
import org.eclipse.jdt.core.dom.FieldAccess;
import org.eclipse.jdt.core.dom.FieldDeclaration;
import org.eclipse.jdt.core.dom.ITypeBinding;
import org.eclipse.jdt.core.dom.ThisExpression;
import org.eclipse.jdt.core.dom.TypeDeclaration;
import org.eclipse.jdt.core.dom.VariableDeclarationFragment;

import org.apache.commons.lang3.reflect.FieldUtils;

import java.lang.reflect.Field;

/**
 * Implementation of {@link IExpressionEvaluator} for {@link FieldAccess}.
 *
 * @author scheglov_ke
 * @coverage core.evaluation
 */
public final class FieldAccessEvaluator implements IExpressionEvaluator {
	////////////////////////////////////////////////////////////////////////////
	//
	// IExpressionEvaluator
	//
	////////////////////////////////////////////////////////////////////////////
	@Override
	public Object evaluate(EvaluationContext context,
			Expression expression,
			ITypeBinding typeBinding,
			String typeQualifiedName) throws Exception {
		if (expression instanceof FieldAccess fieldAccess) {
			String fieldName = fieldAccess.getName().getIdentifier();
			// check "this" expression
			Expression fieldAccessExpression = fieldAccess.getExpression();
			if (fieldAccessExpression instanceof ThisExpression) {
				// prepare field declaration
				TypeDeclaration typeDeclaration = AstNodeUtils.getEnclosingType(expression);
				VariableDeclarationFragment fragment =
						AstNodeUtils.getFieldFragmentByName(typeDeclaration, fieldName);
				Assert.isNotNull(fragment);
				// calculate field value
				Expression fieldInitializer = fragment.getInitializer();
				if (fieldInitializer == null) {
					FieldDeclaration fieldDeclaration =
							AstNodeUtils.getEnclosingNode(fragment, FieldDeclaration.class);
					String className = AstNodeUtils.getFullyQualifiedName(fieldDeclaration.getType(), true);
					return ReflectionUtils.getDefaultValue(className);
				}
				return AstEvaluationEngine.evaluate(context, fieldInitializer);
			}
			// prepare expression
			String expressionClassName = AstNodeUtils.getFullyQualifiedName(fieldAccessExpression, true);
			Class<?> expressionClass = context.getClassLoader().loadClass(expressionClassName);
			Object expressionValue = AstEvaluationEngine.evaluate(context, fieldAccessExpression);
			// prepare field
			Field field = ReflectionUtils.getFieldByName(expressionClass, fieldName);
			Assert.isNotNull(field);
			// return static value
			return FieldUtils.readField(field, expressionValue, true);
		}
		// we don't understand given expression
		return AstEvaluationEngine.UNKNOWN;
	}
}
