"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Red Hat. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const node_1 = require("vscode-languageserver/node");
const yamlSettings_1 = require("../src/yamlSettings");
const serviceSetup_1 = require("./utils/serviceSetup");
const testHelper_1 = require("./utils/testHelper");
const chai_1 = require("chai");
const verifyError_1 = require("./utils/verifyError");
describe('Auto Completion Fix Tests', () => {
    let languageSettingsSetup;
    let languageService;
    let languageHandler;
    let yamlSettings;
    before(() => {
        languageSettingsSetup = new serviceSetup_1.ServiceSetup().withCompletion().withSchemaFileMatch({
            uri: 'https://raw.githubusercontent.com/yannh/kubernetes-json-schema/master/v1.20.5-standalone-strict/all.json',
            fileMatch: [testHelper_1.SCHEMA_ID],
        });
        const { languageService: langService, languageHandler: langHandler, yamlSettings: settings } = testHelper_1.setupLanguageService(languageSettingsSetup.languageSettings);
        languageService = langService;
        languageHandler = langHandler;
        yamlSettings = settings;
    });
    function parseSetup(content, line, character) {
        const testTextDocument = testHelper_1.setupSchemaIDTextDocument(content);
        yamlSettings.documents = new yamlSettings_1.TextDocumentTestManager();
        yamlSettings.documents.set(testTextDocument);
        return languageHandler.completionHandler({
            position: node_1.Position.create(line, character),
            textDocument: testTextDocument,
        });
    }
    afterEach(() => {
        languageService.deleteSchema(testHelper_1.SCHEMA_ID);
        languageService.configure(languageSettingsSetup.languageSettings);
    });
    it('should show completion on map under array', () => __awaiter(void 0, void 0, void 0, function* () {
        languageService.addSchema(testHelper_1.SCHEMA_ID, {
            type: 'array',
            items: {
                type: 'object',
                properties: {
                    from: {
                        type: 'object',
                        properties: {
                            foo: {
                                type: 'boolean',
                            },
                        },
                    },
                },
            },
        });
        const content = '- from:\n    ';
        const completion = yield parseSetup(content, 1, 3);
        chai_1.expect(completion.items).lengthOf(1);
        chai_1.expect(completion.items[0]).eql(verifyError_1.createExpectedCompletion('foo', 'foo: $1', 1, 3, 1, 3, 10, 2, {
            documentation: '',
        }));
    }));
    it('should show completion on array empty array item', () => __awaiter(void 0, void 0, void 0, function* () {
        languageService.addSchema(testHelper_1.SCHEMA_ID, {
            type: 'array',
            items: {
                type: 'object',
                properties: {
                    from: {
                        type: 'object',
                        properties: {
                            foo: {
                                type: 'boolean',
                            },
                        },
                    },
                },
            },
        });
        const content = '- ';
        const completion = yield parseSetup(content, 0, 2);
        chai_1.expect(completion.items).lengthOf(1);
        chai_1.expect(completion.items[0]).eql(verifyError_1.createExpectedCompletion('from', 'from:\n    $1', 0, 2, 0, 2, 10, 2, {
            documentation: '',
        }));
    }));
    it('should show completion items in the middle of map in array', () => __awaiter(void 0, void 0, void 0, function* () {
        const content = `apiVersion: v1
kind: Pod
metadata:
  name: foo
spec:
  containers:
    - name: test
      
      image: alpine
    `;
        const completion = yield parseSetup(content, 7, 6);
        chai_1.expect(completion.items).length.greaterThan(1);
    }));
    it('should show completion on array item on first line', () => __awaiter(void 0, void 0, void 0, function* () {
        const content = '-d';
        const completion = yield parseSetup(content, 0, 1);
        chai_1.expect(completion.items).is.empty;
    }));
    it('should complete without error on map inside array', () => __awaiter(void 0, void 0, void 0, function* () {
        const content = '- foo\n- bar:\n    so';
        const completion = yield parseSetup(content, 2, 6);
        chai_1.expect(completion.items).is.empty;
    }));
});
//# sourceMappingURL=autoCompletionFix.test.js.map