/**
 * Copyright (c) 2010-2016, Abel Hegedus, IncQuery Labs Ltd.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * Contributors:
 * Abel Hegedus - initial API and implementation
 */
package org.eclipse.viatra.query.tooling.ui.queryregistry;

import com.google.common.collect.Maps;
import java.util.Arrays;
import java.util.Map;
import java.util.function.Consumer;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.viatra.query.runtime.registry.IQuerySpecificationRegistry;
import org.eclipse.viatra.query.runtime.registry.IQuerySpecificationRegistryChangeListener;
import org.eclipse.viatra.query.runtime.registry.IQuerySpecificationRegistryEntry;
import org.eclipse.viatra.query.runtime.registry.IRegistryView;
import org.eclipse.viatra.query.runtime.registry.IRegistryViewFactory;
import org.eclipse.viatra.query.runtime.registry.view.AbstractRegistryView;
import org.eclipse.viatra.query.tooling.ui.queryregistry.QueryRegistryTreeEntry;
import org.eclipse.viatra.query.tooling.ui.queryregistry.QueryRegistryTreeInputChange;
import org.eclipse.viatra.query.tooling.ui.queryregistry.QueryRegistryTreePackage;
import org.eclipse.viatra.query.tooling.ui.queryregistry.QueryRegistryTreeSource;
import org.eclipse.xtend.lib.annotations.AccessorType;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtext.xbase.lib.Pure;

/**
 * @author Abel Hegedus
 */
@SuppressWarnings("all")
public class QueryRegistryTreeInput {
  @Accessors(AccessorType.PUBLIC_GETTER)
  private Map<String, QueryRegistryTreeSource> sources = Maps.<String, QueryRegistryTreeSource>newTreeMap();
  
  @Accessors(AccessorType.PUBLIC_GETTER)
  private IQuerySpecificationRegistry registry;
  
  @Accessors(AccessorType.PUBLIC_GETTER)
  private IRegistryView view;
  
  @Accessors(AccessorType.PUBLIC_GETTER)
  private IQuerySpecificationRegistryChangeListener listener;
  
  public QueryRegistryTreeInput(final IQuerySpecificationRegistry registry) {
    this.registry = registry;
    final IRegistryViewFactory _function = (IQuerySpecificationRegistry it) -> {
      return new AbstractRegistryView(registry, true) {
        @Override
        protected boolean isEntryRelevant(final IQuerySpecificationRegistryEntry entry) {
          return true;
        }
      };
    };
    this.view = registry.createView(_function);
    final Consumer<IQuerySpecificationRegistryEntry> _function_1 = (IQuerySpecificationRegistryEntry it) -> {
      final QueryRegistryTreeInputChange source = this.getOrCreateSource(it.getSourceIdentifier());
      final QueryRegistryTreeInputChange treePackage = this.getOrCreatePackage(source.getSource(), this.getPackageName(it.getFullyQualifiedName()));
      QueryRegistryTreePackage _pckg = treePackage.getPckg();
      final QueryRegistryTreeEntry entry = new QueryRegistryTreeEntry(_pckg, it);
      treePackage.getPckg().getEntries().put(it.getFullyQualifiedName(), entry);
    };
    this.view.getEntries().forEach(_function_1);
  }
  
  public void setListener(final IQuerySpecificationRegistryChangeListener listener) {
    if ((this.listener != null)) {
      this.view.removeViewListener(this.listener);
    }
    this.listener = listener;
    if ((listener != null)) {
      this.view.addViewListener(listener);
    }
  }
  
  public QueryRegistryTreeInputChange addEntryToInput(final IQuerySpecificationRegistryEntry entry) {
    final QueryRegistryTreeInputChange source = this.getOrCreateSource(entry.getSourceIdentifier());
    final QueryRegistryTreeInputChange treePackage = this.getOrCreatePackage(source.getSource(), this.getPackageName(entry.getFullyQualifiedName()));
    final QueryRegistryTreePackage treePckg = treePackage.getPckg();
    final QueryRegistryTreeEntry treeEntry = new QueryRegistryTreeEntry(treePckg, entry);
    final boolean emptyPckg = treePckg.getEntries().isEmpty();
    treePackage.getPckg().getEntries().put(entry.getFullyQualifiedName(), treeEntry);
    boolean _isSourceAffected = source.isSourceAffected();
    QueryRegistryTreeSource _source = source.getSource();
    return new QueryRegistryTreeInputChange(true, treeEntry, (treePackage.isPckgAffected() || emptyPckg), treePckg, _isSourceAffected, _source);
  }
  
  public QueryRegistryTreeInputChange removeEntry(final IQuerySpecificationRegistryEntry entry) {
    final QueryRegistryTreeInputChange sourceDTO = this.getOrCreateSource(entry.getSourceIdentifier());
    final QueryRegistryTreeSource source = sourceDTO.getSource();
    final QueryRegistryTreeInputChange treePackageDTO = this.getOrCreatePackage(sourceDTO.getSource(), this.getPackageName(entry.getFullyQualifiedName()));
    final QueryRegistryTreePackage treePckg = treePackageDTO.getPckg();
    final QueryRegistryTreeEntry treeEntry = treePckg.getEntries().remove(entry.getFullyQualifiedName());
    final boolean emptyPckg = treePckg.getEntries().isEmpty();
    if (emptyPckg) {
      source.getPackages().remove(treePckg.getPackageName());
    }
    final boolean emptySource = source.getPackages().isEmpty();
    if (emptySource) {
      this.sources.remove(source.getSourceIdentifier());
    }
    return new QueryRegistryTreeInputChange(true, treeEntry, (treePackageDTO.isPckgAffected() || emptyPckg), treePckg, (sourceDTO.isSourceAffected() || emptySource), source);
  }
  
  public QueryRegistryTreeInputChange getOrCreateSource(final String sourceIdentifier) {
    final QueryRegistryTreeSource existingSource = this.sources.get(sourceIdentifier);
    if ((existingSource == null)) {
      final QueryRegistryTreeSource newSource = new QueryRegistryTreeSource(this, sourceIdentifier);
      this.sources.put(sourceIdentifier, newSource);
      return new QueryRegistryTreeInputChange(false, null, false, null, true, newSource);
    } else {
      return new QueryRegistryTreeInputChange(false, null, false, null, false, existingSource);
    }
  }
  
  public QueryRegistryTreeInputChange getOrCreatePackage(final QueryRegistryTreeSource source, final String packageName) {
    final QueryRegistryTreePackage existingPackage = source.getPackages().get(packageName);
    if ((existingPackage == null)) {
      final QueryRegistryTreePackage newPackage = new QueryRegistryTreePackage(source, packageName);
      source.getPackages().put(packageName, newPackage);
      return new QueryRegistryTreeInputChange(false, null, true, newPackage, false, null);
    } else {
      return new QueryRegistryTreeInputChange(false, null, false, existingPackage, false, null);
    }
  }
  
  public String getPackageName(final String fullyQualifiedName) {
    return fullyQualifiedName.substring(0, Math.max(0, fullyQualifiedName.lastIndexOf(".")));
  }
  
  protected TreePath _getTreePath(final QueryRegistryTreePackage pckg) {
    QueryRegistryTreeSource _parent = pckg.getParent();
    return new TreePath(new Object[] { _parent, pckg });
  }
  
  protected TreePath _getTreePath(final QueryRegistryTreeEntry entry) {
    return this.getTreePath(entry.getParent()).createChildPath(entry);
  }
  
  public TreePath getTreePath(final Object entry) {
    if (entry instanceof QueryRegistryTreeEntry) {
      return _getTreePath((QueryRegistryTreeEntry)entry);
    } else if (entry instanceof QueryRegistryTreePackage) {
      return _getTreePath((QueryRegistryTreePackage)entry);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(entry).toString());
    }
  }
  
  @Pure
  public Map<String, QueryRegistryTreeSource> getSources() {
    return this.sources;
  }
  
  @Pure
  public IQuerySpecificationRegistry getRegistry() {
    return this.registry;
  }
  
  @Pure
  public IRegistryView getView() {
    return this.view;
  }
  
  @Pure
  public IQuerySpecificationRegistryChangeListener getListener() {
    return this.listener;
  }
}
