/*******************************************************************************
 * Copyright (c) 2007, 2008 Ecliptical Software Inc. and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Ecliptical Software Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.mint.internal.ui.search;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceVisitor;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.edit.provider.ComposedAdapterFactory;
import org.eclipse.emf.mint.IItemJavaElementDescriptor;
import org.eclipse.emf.mint.IItemJavaElementSource;
import org.eclipse.emf.mint.IJavaTypeReference;
import org.eclipse.emf.mint.internal.ui.MintUI;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.search.IJavaSearchConstants;
import org.eclipse.jdt.core.search.IJavaSearchScope;
import org.eclipse.jdt.core.search.SearchPattern;
import org.eclipse.jdt.ui.JavaElementLabels;
import org.eclipse.jdt.ui.search.ElementQuerySpecification;
import org.eclipse.jdt.ui.search.IMatchPresentation;
import org.eclipse.jdt.ui.search.IQueryParticipant;
import org.eclipse.jdt.ui.search.ISearchRequestor;
import org.eclipse.jdt.ui.search.PatternQuerySpecification;
import org.eclipse.jdt.ui.search.QuerySpecification;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.activities.IActivityManager;

public class GenModelQueryParticipant implements IQueryParticipant {

	private static final String SEARCH_ACTIVITY_ID = "org.eclipse.emf.mint.ui.search"; //$NON-NLS-1$

	private static final String GENMODEL_EXT = "genmodel"; //$NON-NLS-1$

	private IMatchPresentation uiParticipant;

	private Boolean enabled;

	public int estimateTicks(QuerySpecification query) {
		if (!isEnabled(true))
			return 1;

		if (query.getLimitTo() != IJavaSearchConstants.REFERENCES
				&& query.getLimitTo() != IJavaSearchConstants.ALL_OCCURRENCES)
			return 1;

		return 100;
	}

	public synchronized IMatchPresentation getUIParticipant() {
		if (uiParticipant == null)
			uiParticipant = new GenModelMatchPresentation();

		return uiParticipant;
	}

	public void search(ISearchRequestor requestor, QuerySpecification query,
			IProgressMonitor monitor) throws CoreException {
		if (!isEnabled(false))
			return;

		if (query.getLimitTo() != IJavaSearchConstants.REFERENCES
				&& query.getLimitTo() != IJavaSearchConstants.ALL_OCCURRENCES)
			return;

		SearchHelper helper = null;
		monitor.beginTask(Messages.GenModelQueryParticipant_SearchTask, 2);
		try {
			SearchPattern pattern;
			String stringPattern;
			if (query instanceof ElementQuerySpecification) {
				IJavaElement element = ((ElementQuerySpecification) query)
						.getElement();
				stringPattern = JavaElementLabels.getElementLabel(element,
						JavaElementLabels.ALL_DEFAULT);
				pattern = SearchPattern.createPattern(element, query
						.getLimitTo(), SearchPattern.R_EXACT_MATCH
						| SearchPattern.R_CASE_SENSITIVE
						| SearchPattern.R_ERASURE_MATCH);
			} else {
				PatternQuerySpecification patternSpec = (PatternQuerySpecification) query;
				stringPattern = patternSpec.getPattern();
				int matchMode = getMatchMode(stringPattern)
						| SearchPattern.R_ERASURE_MATCH;
				if (patternSpec.isCaseSensitive())
					matchMode |= SearchPattern.R_CASE_SENSITIVE;

				pattern = SearchPattern.createPattern(patternSpec.getPattern(),
						patternSpec.getSearchFor(), patternSpec.getLimitTo(),
						matchMode);
			}

			IJavaSearchScope scope = query.getScope();
			IPath[] roots = scope.enclosingProjectsAndJars();
			ResourceSet resourceSet = new ResourceSetImpl();
			Collection<FileURI> fileURIs = findGenModels(roots,
					new SubProgressMonitor(monitor, 1));

			if (fileURIs.isEmpty())
				return;

			helper = new SearchHelper(requestor, query, pattern, stringPattern
					.toCharArray());
			search(resourceSet, fileURIs, helper, new SubProgressMonitor(
					monitor, 1));
		} finally {
			if (helper != null)
				helper.dispose();

			monitor.done();
		}
	}

	private Collection<FileURI> findGenModels(IPath[] roots,
			IProgressMonitor monitor) throws CoreException {
		monitor.beginTask(Messages.GenModelQueryParticipant_CollectTask,
				roots.length);
		try {
			ArrayList<FileURI> result = new ArrayList<FileURI>();
			IWorkspaceRoot workspaceRoot = ResourcesPlugin.getWorkspace()
					.getRoot();
			for (int i = 0; i < roots.length; ++i) {
				collectGenModels(workspaceRoot, roots[i], result);

				if (monitor.isCanceled())
					return Collections.emptySet();

				monitor.worked(1);
			}

			return result;
		} finally {
			monitor.done();
		}
	}

	private void collectGenModels(IWorkspaceRoot root, IPath path,
			final Collection<FileURI> collector) throws CoreException {
		IResource resource = root.findMember(path);
		if (resource != null) {
			switch (resource.getType()) {
			case IResource.ROOT:
			case IResource.PROJECT:
			case IResource.FOLDER:
				resource.accept(new IResourceVisitor() {
					public boolean visit(IResource resource)
							throws CoreException {
						if (resource.getType() == IResource.FILE) {
							visitFile((IFile) resource, collector);
							return false;
						}

						return true;
					}
				});

				break;
			case IResource.FILE:
				visitFile((IFile) resource, collector);
				break;
			}
		}
	}

	private void visitFile(IFile file, Collection<FileURI> collector) {
		if (GENMODEL_EXT.equals(file.getFileExtension())) {
			URI uri = URI.createPlatformResourceURI(file.getFullPath()
					.toString(), false);
			collector.add(new FileURI(file, uri));
		}
	}

	private void search(ResourceSet resourceSet, Collection<FileURI> fileURIs,
			SearchHelper helper, IProgressMonitor monitor) throws CoreException {
		monitor.beginTask("", fileURIs.size()); //$NON-NLS-1$
		try {
			for (FileURI fileURI : fileURIs) {
				IFile file = fileURI.getFile();
				monitor.subTask(file.getName());
				URI uri = fileURI.getURI();
				try {
					Resource resource = resourceSet.getResource(uri, true);
					for (Iterator<EObject> i = resource.getAllContents(); i
							.hasNext();)
						search(i.next(), file, helper);
				} catch (RuntimeException e) {
					String msg = NLS.bind(
							Messages.GenModelQueryParticipant_SearchError,
							uri);
					MintUI.getDefault().logError(msg, e);
				}

				if (monitor.isCanceled())
					return;

				monitor.worked(1);
			}
		} finally {
			monitor.done();
		}
	}

	private void search(EObject object, IFile file, SearchHelper helper) {
		AdapterFactory adapterFactory = helper.getAdapterFactory();
		Object adapter = adapterFactory.adapt(object,
				IItemJavaElementSource.class);
		if (adapter instanceof IItemJavaElementSource) {
			IItemJavaElementSource provider = (IItemJavaElementSource) adapter;
			List<IItemJavaElementDescriptor> descriptors = provider
					.getJavaElementDescriptors(object);
			for (IItemJavaElementDescriptor descriptor : descriptors)
				search(descriptor, object, file, helper);
		}
	}

	private void search(IItemJavaElementDescriptor descriptor, EObject target,
			IFile file, SearchHelper helper) {
		QuerySpecification query = helper.getQuerySpecification();
		if (query instanceof ElementQuerySpecification) {
			IJavaElement queryElement = ((ElementQuerySpecification) query)
					.getElement();
			IJavaElement element = descriptor.getJavaElement(target);
			if (element == null) {
				IJavaTypeReference ref = descriptor
						.getJavaTypeReference(target);
				String name = ref.getTypeName();
				if (helper.matches(name.toCharArray()))
					helper.reportMatch(target, file);
			} else {
				if (element.equals(queryElement))
					helper.reportMatch(target, file);
			}
		} else {
			IJavaElement element = descriptor.getJavaElement(target);
			if (element == null) {
				if (((PatternQuerySpecification) query).getSearchFor() == IJavaElement.TYPE) {
					IJavaTypeReference ref = descriptor
							.getJavaTypeReference(target);
					String name = ref.getTypeName();
					if (helper.matches(name.toCharArray()))
						helper.reportMatch(target, file);
				}
			} else {
				String name = element.getElementName();
				if (helper.matches(name.toCharArray()))
					helper.reportMatch(target, file);
			}
		}
	}

	private int getMatchMode(String pattern) {
		if (pattern.indexOf('*') != -1 || pattern.indexOf('?') != -1)
			return SearchPattern.R_PATTERN_MATCH;

		if (isCamelCasePattern(pattern))
			return SearchPattern.R_CAMELCASE_MATCH;

		return SearchPattern.R_EXACT_MATCH;
	}

	private boolean isCamelCasePattern(String pattern) {
		return SearchPattern.validateMatchRule(pattern,
				SearchPattern.R_CAMELCASE_MATCH) == SearchPattern.R_CAMELCASE_MATCH;
	}

	private synchronized boolean isEnabled(boolean update) {
		if (enabled != null && !update)
			return enabled;

		final IWorkbench wb = PlatformUI.getWorkbench();
		Display display = wb.getDisplay();
		display.syncExec(new Runnable() {
			public void run() {
				IActivityManager mgr = wb.getActivitySupport()
						.getActivityManager();
				enabled = mgr.getActivity(SEARCH_ACTIVITY_ID).isEnabled();
			}
		});

		return enabled;
	}

	private static class FileURI {

		private final IFile file;

		private final URI uri;

		public FileURI(IFile file, URI uri) {
			this.file = file;
			this.uri = uri;
		}

		public IFile getFile() {
			return file;
		}

		public URI getURI() {
			return uri;
		}
	}

	private static class SearchHelper {

		private final ISearchRequestor requestor;

		private final QuerySpecification querySpecification;

		private final SearchPattern searchPattern;

		private final char[] stringPatternChars;

		private ComposedAdapterFactory adapterFactory;

		public SearchHelper(ISearchRequestor requestor,
				QuerySpecification querySpecification,
				SearchPattern searchPattern, char[] stringPatternChars) {
			this.requestor = requestor;
			this.querySpecification = querySpecification;
			this.searchPattern = searchPattern;
			this.stringPatternChars = stringPatternChars;
		}

		public void reportMatch(EObject target, IFile file) {
			JavaSearchMonitor.ensureStarted();
			GenModelMatch match = new GenModelMatch(target, file);
			requestor.reportMatch(match);
		}

		public boolean matches(char[] pattern) {
			return searchPattern.matchesName(stringPatternChars, pattern);
		}

		public QuerySpecification getQuerySpecification() {
			return querySpecification;
		}

		public AdapterFactory getAdapterFactory() {
			if (adapterFactory == null)
				adapterFactory = new ComposedAdapterFactory(
						ComposedAdapterFactory.Descriptor.Registry.INSTANCE);

			return adapterFactory;
		}

		public void dispose() {
			if (adapterFactory != null)
				adapterFactory.dispose();
		}
	}
}
