/*
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.lsat.common.scheduler.algorithm;

import java.math.BigDecimal;

import org.eclipse.lsat.common.scheduler.graph.Constraint;
import org.eclipse.lsat.common.scheduler.graph.JitConstraint;
import org.eclipse.lsat.common.scheduler.graph.Task;
import org.eclipse.lsat.common.scheduler.graph.TimeConstraint;

import dispatching.DispatchGroup;
import lsat_graph.ActionTask;
import lsat_graph.DispatchGraph;

public class ConstraintsUtil {
    private static final BigDecimal INFINITE_TIME_CONSTRAINT_UPPER_BOUND = BigDecimal.valueOf(1.0E99);

    private ConstraintsUtil() {
    }

    public static String constraintToString(Constraint constraint) {
        var source = getNodeName(constraint.getSource());
        var target = getNodeName(constraint.getTarget());
        if (constraint instanceof JitConstraint) {
            return source + " -jit-> " + target;
        } else if (constraint instanceof TimeConstraint timeConstraint) {
            if (timeConstraint.getUpperBound().compareTo(INFINITE_TIME_CONSTRAINT_UPPER_BOUND) == 0) {
                return source + " -> " + target;
            } else {
                return source + " -[" + timeConstraint.getLowerBound().toString() + ","
                        + timeConstraint.getUpperBound().toString() + "]-> " + target;
            }
        } else {
            return source + " -> " + target;
        }
    }

    private static <T extends Task> String getNodeName(T node) {
        if (node instanceof ActionTask task) {
            var action = task.getAction();
            var dispatchGraph = (DispatchGraph)task.eContainer();
            var dispatch = dispatchGraph.getDispatch();
            var dispatchName = dispatch.getName();
            var dpGroup = (DispatchGroup)dispatch.eContainer();
            var dpGroupName = dpGroup.getName();
            var activity = dispatch.getActivity();
            var result = "";
            if (dpGroupName != null && !dpGroupName.equals("default")) {
                result = dpGroupName + ".";
            }
            if (dispatchName != null && !dispatchName.startsWith("null")) {
                var lastUnderscore = dispatchName.lastIndexOf('_');
                dispatchName = (lastUnderscore != -1) ? dispatchName.substring(0, lastUnderscore) : dispatchName;
                result += dispatchName + ".";
            }
            return result + activity.getName() + "." + action.getName();
        } else {
            return node.getName();
        }
    }
}
