/*
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package activity.util;

import static com.google.common.collect.Sets.cartesianProduct;
import static org.eclipse.lsat.common.emf.ecore.util.EcoreUtility.containsCrossReference;
import static org.eclipse.lsat.common.queries.QueryableIterable.from;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.util.EcoreUtil;

import activity.Activity;
import activity.ActivitySet;
import activity.Claim;
import activity.Event;
import activity.HasResourceAndItem;
import activity.ModelTypeDefinitionRef;
import activity.Move;
import activity.SimpleAction;
import common.HasName;
import common.ParameterDeclaration;
import common.TypeDefinition;
import common.util.CommonUtil;
import machine.ActionType;
import machine.HasResourcePeripheral;
import machine.IResource;
import machine.Machine;
import machine.Peripheral;
import machine.Profile;
import machine.Resource;
import machine.ResourceItem;
import machine.SymbolicPosition;
import product.ProductDefinition;

public final class ActivityParametersUtil {
    private ActivityParametersUtil() {
        // no instantiation allowed
    }

    private static final String ITEM_DELIMITER = "_";

    private static final String REF_DELIMITER = "__";

    private static final String ESCAPE = "_us_";

    /**
     * Creates an activity name with resource names appended as postfix
     */
    public static String expandName(Activity activity, List<? extends Collection<? extends HasName>> parameters) {
        var name = activity.getName();

        if (!activity.getParameterDeclarations().isEmpty()) {
            var parNames = parameters.stream().map(refs -> String.join(REF_DELIMITER,
                    refs.stream().map(HasName::getName).map(ActivityParametersUtil::esc).toList())).toList();
            return Stream.concat(Stream.of(name), parNames.stream()).collect(Collectors.joining(ITEM_DELIMITER));
        }
        if (!activity.getResourcesNeedingItem().isEmpty()) {
            var parametersAsResource = convertToIResources(parameters);
            // make sure name is sorted in resources order.
            return activity.getResourcesNeedingItem().stream()
                    .map(r -> parametersAsResource.stream().filter(i -> i.getResource() == r).map(IResource::getName)
                            .findFirst().orElse(null))
                    .filter(Objects::nonNull).collect(Collectors.joining(ITEM_DELIMITER, name + ITEM_DELIMITER, ""));
        }
        return name;
    }

    /**
     * Generates all possible expanded names for an activity based on its parameters
     */
    public static Collection<String> getAllExpandedNames(Activity activity) {
        var result = new LinkedHashSet<String>();
        result.add(activity.getName());
        var candidates = findCandidates(activity);
        cartesianProduct(candidates).stream().map(pars -> expandName(activity, pars)).forEach(result::add);
        return result;
    }

    /**
     * Checks if an activity uses a specific type definition
     */
    public static boolean uses(Activity activity, TypeDefinition dec) {
        return Stream.concat(activity.getNodes().stream(), activity.getPrerequisites().stream()) //
                .anyMatch(o -> containsCrossReference(o, dec));
    }

    /**
     * Creates a human-readable display name for an activity with parameters in brackets
     */
    public static String displayName(Activity activity) {
        if (activity == null) {
            return null;
        }
        if (activity.getOriginalName() == null) {
            return activity.getName();
        }
        return getParameterNames(activity.getOriginalName(), activity.getName()).stream() //
                .map(i -> String.join(".", i)).collect(Collectors.joining(",", activity.getOriginalName() + "[", "]"));
    }

    /**
     * Extracts parameter names from an expanded activity name
     */
    public static Collection<List<String>> getParameterNames(String orgActivityName, String expandedActivityName) {
        if (expandedActivityName.startsWith(orgActivityName + ActivityUtil.EXPAND_DELIMITER)) {
            // make if easy to split "_" by replacing "__" to "|":
            var expandActivityName = expandedActivityName.replaceAll(REF_DELIMITER, "|");
            var parsStrings = expandActivityName
                    .substring(orgActivityName.length() + ActivityUtil.EXPAND_DELIMITER.length());
            return Arrays.stream(parsStrings.split(ITEM_DELIMITER))
                    .map(par -> Arrays.stream(par.split("|")).map(ActivityParametersUtil::unesc).toList()).toList();
        }
        return Collections.emptyList();
    }

    /**
     * Retrieves parameter declarations for an activity (explicit or implicit)
     */
    public static List<TypeDefinition> getDeclarations(Activity activity) {
        if (!activity.getParameterDeclarations().isEmpty()) {
            return activity.getParameterDeclarations().stream() //
                    .flatMap(pd -> pd.getDeclarations().stream()).toList();
        }
        return activity.getResourcesNeedingItem().stream() //
                .map(TypeDefinition.class::cast).distinct().toList();
    }

    /**
     * Creates or retrieves an expanded activity with concrete parameters
     */
    public static Activity queryCreateExpandedActivity(final Activity orgActivity,
            final List<? extends Collection<? extends HasName>> parameters)
    {
        var eContainer = orgActivity.eContainer();
        if (eContainer == null) {
            throw new RuntimeException("Activity is not part of a container");
        }
        if (!(eContainer instanceof ActivitySet)) {
            throw new RuntimeException("Activity must be part of an ActivitySet");
        }
        var set = ((ActivitySet)eContainer);
        var activities = set.getActivities();
        var activityName = expandName(orgActivity, parameters);
        var result = from(set.getActivities()).select(a -> a.getName().equals(activityName)).first();
        if (result == null) {
            result = expand(orgActivity, parameters);
            activities.add(activities.indexOf(orgActivity), result);
        }
        return result;
    }

    /**
     * Finds all possible parameter candidates for an activity
     */
    public static List<Set<List<? extends HasName>>> findCandidates(Activity activity) {
        var result = new ArrayList<Set<List<? extends HasName>>>();
        var resources = collectResourceCandidates(activity);
        for (var dec: getDeclarations(activity)) {
            // for declared IResources and Actions we need to do have the full path
            if (dec instanceof Resource && !(dec instanceof Event) && isDeclaration(dec)) {
                var candidates = findAllResources(EcoreUtil.getRootContainer(dec));
                result.add(candidates);
            } else if (dec instanceof ActionType) {
                // an action must specify resource, item?, peripheral and action
                // hence to list of lists.
                var candidates = findAllActionCandidates(resources);
                result.add(candidates);
            } else {
                var candidates = findCandidate(activity, resources, dec, List.of());
                var format = new LinkedHashSet<List<? extends HasName>>();
                candidates.stream().map(List::of).forEach(format::add);
                result.add(format);
            }
        }
        return result;
    }

    /**
     * Finds candidates for a specific parameter index
     */
    public static boolean matchesType(TypeDefinition declaration, EObject parameter) {
        var type = declaration instanceof ModelTypeDefinitionRef tdRef ? tdRef.getTypeDefinition() : declaration;
        if (type instanceof Resource && parameter instanceof IResource iResource) {
            // a resource is valid it is a ResourceItem or a machine.Resource without items.
            return iResource instanceof ResourceItem || iResource.getResource().getItems().size() == 0;
        }
        return type.eClass().isInstance(parameter);
    }

    /**
     * Finds candidates for a specific parameter index
     */
    public static EClass getExpectedType(EObject declaration, EObject parameter) {
        var type = declaration instanceof ModelTypeDefinitionRef tdRef ? tdRef.getTypeDefinition() : declaration;
        if (type instanceof Resource && parameter instanceof IResource iResource) {
            if (!iResource.getResource().getItems().isEmpty()) {
                return machine.MachinePackage.Literals.RESOURCE_ITEM;
            }
            return machine.MachinePackage.Literals.RESOURCE;
        }
        return type.eClass();
    }

    /**
     * Finds appropriate candidates for a specific type definition
     */
    public static Collection<? extends HasName> findCandidate(Activity activity, List<? extends IResource> resources,
            TypeDefinition dec, List<? extends HasName> current)
    {
        if (dec == null) {
            return Set.of();
        }
        // action types and resource can be a multiple of references
        if (!(dec instanceof ActionType || dec instanceof IResource)) {
            assert current.isEmpty();
        }
        if (dec instanceof Event) { // event before Resource!! (because Event is a Resource)
            var parent = activity.eContainer();
            if (parent instanceof ActivitySet set) {
                return set.getEvents().stream() //
                        .flatMap(e -> e.getItems().isEmpty() ? Stream.of(e) : e.getItems().stream()) //
                        .map(TypeDefinition.class::cast).toList();
            }
        }
        if (dec instanceof Resource r) {
            // a Resource can be resolved resolved using Resource and (possibly) ResourceItem
            // in case of a declaration (f.i. r:Resource) all global resources are used
            // else in case of a declared resource (r:Robot1) only the declared one
            var candidates = (isDeclaration(dec)) ? //
                    findAllResources(EcoreUtil.getRootContainer(dec)) : //
                    createResourcePossibilities(List.of(r));

            return returnCandidateColumn(current, candidates);
        }
        if (dec instanceof ModelTypeDefinitionRef mtdr && mtdr.getResource() != null) {
            if (mtdr.getResource() instanceof Resource r) {
                return r.getItems().isEmpty() ? List.of(r) : r.getItems();
            }
        }
        if (dec instanceof ModelTypeDefinitionRef mtdr && mtdr.getTypeDefinition() instanceof ProductDefinition pd) {
            // later product instances
            return List.of(pd);
        }
        if (dec instanceof SymbolicPosition) {
            return activity.eContents().stream() //
                    .filter(HasResourcePeripheral.class::isInstance).map(HasResourcePeripheral.class::cast) //
                    .filter(n -> containsCrossReference(n, dec)) //
                    .map(HasResourcePeripheral::getPeripheral).filter(Objects::nonNull)
                    .flatMap(pe -> pe.getPositions().stream()).distinct().toList();
        }
        if (dec instanceof Profile) {
            return activity.getNodes().stream() //
                    .filter(Move.class::isInstance).map(Move.class::cast) //
                    .filter(n -> containsCrossReference(n, dec)).map(Move::getPeripheral).filter(Objects::nonNull)
                    .flatMap(pe -> pe.getProfiles().stream()).distinct().toList();
        }
        if (dec instanceof ActionType) {
            // get all potential action types derived from peripheral that considering the availability of slots.
            var candidates = findAllActionCandidates(resources);
            return returnCandidateColumn(current, candidates);
        }
        return CommonUtil.collect(EcoreUtil.getRootContainer(dec), dec.getClass()).stream()
                .map(TypeDefinition.class::cast).distinct().toList();
    }

    /** returns the items in the resource contains items else the resource */
    public static List<IResource> getResourceOrItems(IResource resource) {
        if (resource instanceof ResourceItem || resource.getResource().getItems().isEmpty()) {
            return List.of(resource);
        }
        return resource.getResource().getItems().stream().map(IResource.class::cast).toList();
    }

    public static boolean isDeclaration(EObject object) {
        return EcoreUtil.getRootContainer(object) instanceof ActivitySet;
    }

    /**
     * From the candidate matrix the column is return where the preceding columns match exactly with current
     *
     */
    private static Collection<? extends HasName> returnCandidateColumn(List<? extends HasName> current,
            Set<List<? extends HasName>> candidates)
    {
        var result = new LinkedHashSet<HasName>();
        if (current.isEmpty()) {
            candidates.stream().map(List::getFirst).distinct().forEach(result::add);
            // to figure out whether we can only use the last column we can reduce to the first items with a distinct
            // last column
            var finalCandidates = candidates.stream().map(List::getLast).toList();

            // add all candidates that can be uniquely resolved using the final candidate only
            // this can be determined by checking if it is unique in the finalCandidates list
            finalCandidates.stream().collect(Collectors.groupingBy(e -> e)) //
                    .entrySet().stream().filter(e -> e.getValue().size() == 1).map(Map.Entry::getKey)
                    .forEach(result::add);
        } else {
            candidates.stream() //
                    .filter(candidate -> candidate.subList(0, current.size()).equals(current)) //
                    .map(candidate -> candidate.get(current.size())).distinct().forEach(result::add);
        }
        return result;
    }

    /**
     * Finds all resource and resource items that have not been used yet
     */
    private static Set<List<? extends HasName>> findAllResources(EObject root) {
        // collect Resource on machine to avoid events.
        var allResources = CommonUtil.collect(root, machine.Machine.class).stream().map(Machine::getResources)
                .flatMap(Collection::stream).toList();
        var result = createResourcePossibilities(allResources);
        return result;
    }

    private static LinkedHashSet<List<? extends HasName>> createResourcePossibilities(List<Resource> allResources) {
        var result = new LinkedHashSet<List<? extends HasName>>();
        for (var resource: allResources) {
            for (var item: resource.getItems()) {
                result.add(List.of(item.getResource(), item));
            }
            // if there are no items candidates are resource, peripheral and actions
            if (resource.getItems().isEmpty()) {
                result.add(List.of(resource));
            }
        }
        return result;
    }

    /**
     * Finds all possible action candidates for given resources
     */
    private static Set<List<? extends HasName>> findAllActionCandidates(List<? extends IResource> resources) {
        var result = new LinkedHashSet<List<? extends HasName>>();
        for (var resource: resources) {
            for (var per: resource.getResource().getPeripherals()) {
                // if there are items candidates are resource, item, peripheral and actions
                if (resource instanceof ResourceItem item) {
                    for (var action: per.getType().getActions()) {
                        result.add(List.of(item.getResource(), item, per, action));
                    }
                }
                // if there are no items candidates are resource, peripheral and actions
                if (resource instanceof Resource mResource && mResource.getItems().isEmpty()) {
                    for (var action: per.getType().getActions()) {
                        result.add(List.of(mResource, per, action));
                    }
                }
            }
        }
        return result;
    }

    /**
     * Creates an expanded copy of an activity with parameters replaced
     */
    private static Activity expand(final Activity activity,
            final List<? extends Collection<? extends HasName>> parameters)
    {
        var result = EcoreUtil.copy(activity);
        result.setOriginalName(activity.getName());
        result.setName(expandName(activity, parameters));
        var parIter = parameters.iterator();
        var oldValues = getDeclarations(result).subList(0, parameters.size());
        for (var oldValue: oldValues) {
            var newValue = parIter.next();
            result.getNodes().forEach(n -> replace(n, oldValue, newValue));
            result.getPrerequisites().forEach(n -> replace(n, oldValue, newValue));
            // remove the resolved value
            if (oldValue.eContainer() instanceof ParameterDeclaration pd) {
                pd.getDeclarations().remove(oldValue);
                if (pd.getDeclarations().isEmpty()) {
                    result.getParameterDeclarations().remove(pd);
                }
            }
        }
        return result;
    }

    /**
     * Replaces references to source element with new values
     */
    private static void replace(EObject parent, EObject sourceElement, Collection<? extends HasName> newValue) {
        // Use a set to track visited objects to prevent infinite recursion
        var visited = new HashSet<EObject>();
        replaceRecursive(parent, sourceElement, newValue, visited);
    }

    /**
     * Recursively replaces references in the model structure
     */
    private static void replaceRecursive(Object parent, EObject sourceElement, Collection<? extends HasName> newValues,
            Set<EObject> visited)
    {
        var newValue = newValues.stream().reduce((a, b) -> b).get(); // at least one value must exist take the last
        if (parent instanceof List<?> list) {
            list.forEach(o -> replaceRecursive(o, sourceElement, newValues, visited));
        } else if (parent instanceof EObject eParent) {
            // Skip if we've already processed this object
            if (!visited.add(eParent)) {
                return;
            }

            // Process all references
            for (EReference reference: eParent.eClass().getEAllReferences()) {
                if (reference.isChangeable() && !reference.isDerived()) {
                    // Replace references
                    var object = eParent.eGet(reference);
                    if (reference.isMany()) {
                        @SuppressWarnings("unchecked")
                        var values = (List<EObject>)object;
                        // Replace all occurrences of sourceElement with targetElement
                        for (int i = 0; i < values.size(); i++) {
                            if (values.get(i) == sourceElement) {
                                values.set(i, newValue);
                            }
                        }
                    } else if (object == sourceElement) {
                        // special treatment for IResource
                        if (eParent instanceof HasResourceAndItem hasResourceAndItem
                                && newValue instanceof IResource resource)
                        {
                            hasResourceAndItem.setResource(resource);
                        } else if (eParent instanceof SimpleAction simpleAction) {
                            // set peripheral and resource if not set yet.
                            setSimpleAction(simpleAction, newValues);
                        } else {
                            eParent.eSet(reference, newValue);
                        }
                    }
                    // Continue recursion for containment references
                    if (reference.isContainment()) {
                        replaceRecursive(object, sourceElement, newValues, visited);
                    }
                }
            }
        }
    }

    /**
     * Maps action type to peripheral and resource, then sets them on the SimpleAction. Extracts resources from claims,
     * finds matching peripherals for the action type, and ensures a unique peripheral-resource mapping.
     *
     * @param simpleAction Target to update with the found peripheral and resource
     * @param newValues Action type to match against peripherals
     *
     * @throws IllegalArgumentException If multiple peripherals match the same action type
     */
    private static void setSimpleAction(SimpleAction simpleAction, Collection<? extends HasName> newValues) {
        for (var value: newValues) {
            if (value instanceof IResource res) {
                // this might happen twice (for resource and item) but item wins which is what is needed.
                simpleAction.setResource(res);
            } else if (value instanceof Peripheral per) {
                simpleAction.setPeripheral(per);
            } else if (value instanceof ActionType at) {
                simpleAction.setType(at);
            } else {
                throw new IllegalArgumentException(
                        "Don't know how to handle value of type: " + value.getClass().getName());
            }
            // infer resource and peripheral if only action type has been supplied
            if (simpleAction.getResource() == null && simpleAction.getGraph() instanceof Activity activity) {
                var resources = activity.getNodes().stream().filter(Claim.class::isInstance).map(Claim.class::cast)
                        .map(Claim::getResource).toList();
                for (var resource: resources) {
                    for (var peripheral: resource.getResource().getPeripherals()) {
                        if (peripheral.getType().getActions().contains(simpleAction.getType())) {
                            if (simpleAction.getPeripheral() != null) {
                                throw new IllegalArgumentException("Supplied action " + simpleAction.getType().getName()
                                        + " refers to multiple peripherals: "
                                        + List.of(simpleAction.getPeripheral().fqn(), peripheral.fqn()));
                            }
                            simpleAction.setResource(resource);
                            simpleAction.setPeripheral(peripheral);
                        }
                    }
                }
            }
        }
    }

    private static List<IResource> convertToIResources(List<? extends Collection<? extends HasName>> parameters) {
        var parametersAsResource = new ArrayList<IResource>();
        for (var refs: parameters) {
            var par = refs.stream().reduce((a, b) -> b).orElse(null); // get the last (resource of resource item) in the
                                                                      // collection
            if (par instanceof IResource resource) {
                parametersAsResource.add(resource);
            } else {
                throw new IllegalArgumentException("Expecting an IResource not a " + par.getClass().getSimpleName());
            }
        }
        return parametersAsResource;
    }

    private static String esc(String name) {
        return name.replace(ITEM_DELIMITER, ESCAPE);
    }

    private static String unesc(String name) {
        return name.replace(ESCAPE, ITEM_DELIMITER);
    }

    /**
     * collect resource or resource items if a resource has items.
     */
    private static List<? extends IResource> collectResourceCandidates(Activity activity) {
        var resources = activity.getNodes().stream().filter(Claim.class::isInstance).map(Claim.class::cast)
                .map(Claim::getResource).map(ActivityParametersUtil::getResourceOrItems).flatMap(Collection::stream)
                .toList();
        return resources;
    }
}
