/*
 * Copyright (c) 2021 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
/**
 */

package activity.util;

import static activity.util.ActivityUtil.addClaim;
import static activity.util.ActivityUtil.addEdge;
import static activity.util.ActivityUtil.addRelease;
import static activity.util.ActivityUtil.delete;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.stream.Collectors;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.lsat.common.graph.directed.editable.Edge;

import activity.Activity;
import activity.ActivitySet;
import activity.Claim;
import activity.Event;
import activity.RaiseEvent;
import activity.Release;
import activity.RequireEvent;
import activity.ResourceAction;
import machine.IResource;
import machine.Machine;
import machine.MachineFactory;
import machine.Resource;
import machine.ResourceType;

public class Event2Resource {
    private Event2Resource() {
        // Empty
    }

    /**
     * Converts events in {@link Activity}s to resources. The resources are added to the first machine that is found.
     * Around each {@link RequireEvent} or {@link RaiseEvent} a claim and release are inserted. The event itself is
     * removed.
     */
    public static void replaceEventsWithClaimRelease(ActivitySet activitySet) {
        insertClaimReleaseForEvents(activitySet, false);
    }

    /**
     * Convert event in {@link Activity}s to resources. The resources are added to the first machine that is found.
     * Arround each {@link RequireEvent} or {@link RaiseEvent} a claim and release are inserted. The event itself is
     * kept.
     */
    public static void surroundEventsWithClaimRelease(ActivitySet activitySet) {
        insertClaimReleaseForEvents(activitySet, true);
    }

    /**
     * Convert event in {@link Activity}s to resources. The resources are added to the first machine that is found. On
     * each {@link RequireEvent} or {@link RaiseEvent} a claim and release are inserted.
     */
    private static void insertClaimReleaseForEvents(ActivitySet activitySet, boolean keepEvents) {
        Machine machine = getMachine(activitySet);
        addEventsAsResource(machine, activitySet);
        getEvents(activitySet).forEach(event -> {
            Resource resource = machine.getResources().stream().filter(r -> r.getName().equals(event.getEventName()))
                    .findFirst().get();
            insertClaimReleaseForEvents(event, resource, keepEvents);
        });
    }

    private static void addEventsAsResource(Machine machine, ActivitySet activitySet) {
        getEvents(activitySet).stream().map(Event::getEventName).distinct().forEach(event -> {
            Resource r = MachineFactory.eINSTANCE.createResource();
            r.setResourceType(ResourceType.EVENT);
            r.setName(event);
            machine.getResources().add(r);
        });
    }

    private static Machine getMachine(ActivitySet activitySet) {
        return activitySet.getActivities().stream().flatMap(a -> a.getNodes().stream())
                .filter(ResourceAction.class::isInstance).map(ResourceAction.class::cast)
                .map(ResourceAction::getResource).map(IResource::getResource).map(EObject::eContainer)
                .filter(Machine.class::isInstance).map(Machine.class::cast).findFirst().orElse(null);
    }

    private static Collection<Event> getEvents(ActivitySet activitySet) {
        return activitySet.getActivities().stream().flatMap(a -> a.getNodes().stream()).filter(Event.class::isInstance)
                .map(Event.class::cast).collect(Collectors.toCollection(LinkedHashSet::new));
    }

    private static void insertClaimReleaseForEvents(Event node, Resource resource, boolean keepEvent) {
        Activity activity = (Activity)node.eContainer();
        // single events must insert claim and release to conform to max plus logic.
        boolean containsSingleEvent = activity.getNodes().stream().filter(Event.class::isInstance).count() == 1;
        if (containsSingleEvent || node instanceof RequireEvent) {
            insertClaim(node, resource);
        }
        if (containsSingleEvent || node instanceof RaiseEvent) {
            insertRelease(node, resource);
        }
        if (!keepEvent) {
            shortcutAndRemoveNode(node);
        }
    }

    private static void insertClaim(Event node, Resource resource) {
        Activity activity = (Activity)node.eContainer();
        Claim cl = addClaim(activity, resource);
        node.getIncomingEdges().forEach(e -> addEdge(activity, e.getSourceNode(), cl));
        // delete the original edges
        new ArrayList<>(node.getIncomingEdges()).forEach(e -> delete(e));
        addEdge(activity, cl, node);
    }

    private static void insertRelease(Event node, Resource resource) {
        Activity activity = (Activity)node.eContainer();
        Release rl = addRelease(activity, resource);
        node.getOutgoingEdges().forEach(e -> addEdge(activity, rl, e.getTargetNode()));
        // delete the original edges
        new ArrayList<>(node.getOutgoingEdges()).forEach(e -> delete(e));
        addEdge(activity, node, rl);
    }

    private static void shortcutAndRemoveNode(Event node) {
        Activity activity = (Activity)node.eContainer();
        for (Edge in: node.getIncomingEdges()) {
            for (Edge out: node.getOutgoingEdges()) {
                addEdge(activity, in.getSourceNode(), out.getTargetNode());
            }
        }
        new ArrayList<>(node.getIncomingEdges()).forEach(e -> delete(e));
        new ArrayList<>(node.getOutgoingEdges()).forEach(e -> delete(e));
        delete(node);
    }
}
