/**
 ********************************************************************************
 * Copyright (c) 2020-2021 Eclipse APP4MC contributors.
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 ********************************************************************************
 */

package org.eclipse.app4mc.atdb._import.amalthea;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.net.URL;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Arrays;
import java.util.List;

import org.eclipse.app4mc.atdb.ATDBConnection;
import org.eclipse.app4mc.atdb.MetricAggregation;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.osgi.framework.Bundle;
import org.osgi.framework.FrameworkUtil;

public class EventChainMetricCalculator implements IRunnableWithProgress {
	
	private final ATDBConnection con;
	
	public EventChainMetricCalculator(final ATDBConnection con) {
		this.con = con;
	}

	@Override
	public void run(IProgressMonitor progressMonitor) throws InvocationTargetException, InterruptedException {
		final SubMonitor subMon = SubMonitor.convert(progressMonitor, "Calculating event chain metrics...", 3);
		try (final Statement mStmt = this.con.createStatement()) {
			// create event chain instances with separate sql program (a sequence of queries)
			String ecInstCalculationSQL = "";
			final Bundle bundle = FrameworkUtil.getBundle(EventChainMetricCalculator.class);
			final String pathInBundle = EventChainMetricCalculator.class.getPackage().getName().replace('.', '/') + "/ecinststatements.sql";
			final URL fileURL = bundle.getResource(pathInBundle);
			ecInstCalculationSQL = loadStringFromFile(fileURL);
			if (ecInstCalculationSQL.length() == 0) return;
			this.con.executeUpdate(ecInstCalculationSQL);
			subMon.worked(1);
			
			this.con.executeBatchUpdate(atdbCon -> {
				// calculate event chain metrics min max avg for age and reaction latency values
				final List<String> latencyTypes = Arrays.asList("Age", "Reaction");
				for (final String latencyType : latencyTypes) {
					atdbCon.insertMetric(latencyType.toLowerCase() + "Latency", "time");
					// add latencies to entity instance metrics
					mStmt.addBatch("INSERT OR IGNORE INTO entityInstanceMetricValue SELECT entityId, entityInstance, (SELECT id FROM metric WHERE "//
							+ "name = '" + latencyType.toLowerCase() + "Latency'), responseTimestamp - stimulusTimestamp FROM eventChainInstanceInfo "//
							+ "WHERE is" + latencyType + ";");
					for (final MetricAggregation kind : MetricAggregation.values()) {
						final String metricName = latencyType.toLowerCase() + "Latency" + "_" + kind;
						final String aggregateFunction = kind.getSQLStr("value");
						atdbCon.insertMetric(metricName, "time");
						mStmt.addBatch("INSERT OR IGNORE INTO entityMetricValue SELECT entityId, (SELECT id FROM metric WHERE name = '" + metricName//
								+ "'), " + aggregateFunction + " FROM entityInstanceMetricValue WHERE (SELECT is" + latencyType + " FROM "//
								+ "eventChainInstanceInfo WHERE eventChainInstanceInfo.entityId = entityInstanceMetricValue.entityId AND "//
								+ "eventChainInstanceInfo.entityInstance = entityInstanceMetricValue.entityInstance) "//
								+ "GROUP BY entityId;");
					}
				}
			});
			subMon.worked(1);
			
			this.con.executeBatchStatements(mStmt);
			subMon.worked(1);
		} catch (SQLException e) {
			throw new InvocationTargetException(e);
		}
	}
	
	private static String loadStringFromFile(final URL fileURL) {
		try (final InputStream fileIS = fileURL.openStream();
			 final ByteArrayOutputStream bAOS = new ByteArrayOutputStream();) {
			byte[] buffer = new byte[1024];
			int length;
			while ((length = fileIS.read(buffer)) != -1) {
			    bAOS.write(buffer, 0, length);
			}
			return bAOS.toString();
		} catch (IOException e) {
			// fail silently
		}
		return "";
	}

}
