/**
 ********************************************************************************
 * Copyright (c) 2019-2020 Robert Bosch GmbH and others.
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 ********************************************************************************
 */

package org.eclipse.app4mc.validation.util;

import java.io.PrintStream;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.app4mc.validation.core.IProfile;
import org.eclipse.app4mc.validation.core.IProfileConfiguration;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ReferencePolicy;
import org.osgi.service.component.annotations.ReferencePolicyOption;

@Component(service=ProfileManager.class)
public class ProfileManager {
	
	private final HashMap<Class<? extends IProfile>, CachedProfile> profileMap = new HashMap<>();

	@Reference(
			cardinality = ReferenceCardinality.MULTIPLE, 
			policy = ReferencePolicy.DYNAMIC, 
			policyOption = ReferencePolicyOption.GREEDY)
	public void bindProfileConfiguration(IProfileConfiguration config) {
		synchronized (this.profileMap) {
			Class<? extends IProfile> profileClass = config.getClass();
			this.profileMap.put(profileClass, getCachedProfile(profileClass));
		}
	}
	
	public void unbindProfileConfiguration(IProfileConfiguration config) {
		synchronized (this.profileMap) {
			this.profileMap.remove(config.getClass());
		}
	}
	
	public Map<Class<? extends IProfile>, CachedProfile> getRegisteredValidationProfiles() {
		return Collections.unmodifiableMap(this.profileMap);
	}

	protected void loadProfile(Class<? extends IProfile> profileClass) {
		if (profileClass == null) {
			return;
		}

		this.profileMap.put(profileClass, new CachedProfile(profileClass));
	}

	public CachedProfile getCachedProfile(Class<? extends IProfile> profileClass) {
		if (profileClass == null) {
			return null;
		}

		if (!this.profileMap.containsKey(profileClass)) {
			loadProfile(profileClass);
		}

		return this.profileMap.get(profileClass);
	}

	// *** Some helper methods

	public void dumpProfile(Class<? extends IProfile> profileClass, PrintStream out) {
		if (out == null) return;

		CachedProfile profile = getCachedProfile(profileClass);
		dumpProfile(profile, 0, out);
	}

	private void dumpProfile(CachedProfile profile, int indent, PrintStream stream) {
		if (profile == null) return;

		String in = "";
		for (int i = 0; i < indent; i++) {
			in = in + "    ";
		}

		// Profile data
		stream.println(in + "PROFILE " + profile.getProfileClass().getSimpleName());
		stream.println(in + " - name: " + profile.getName());
		if (!profile.getDescription().isEmpty()) {
			stream.println(in + " - description: " + profile.getDescription());
		}

		// Validations
		if (!profile.getCachedValidations().isEmpty()) {
			stream.println(in + " - validations:");
		}
		for (CachedValidator conf : profile.getCachedValidations().values()) {
			stream.println(in + "    " + conf.getValidationID() + "(" + conf.getSeverity() + " - "
					+ conf.getTargetClass().getSimpleName() + ")");

			for (String check : conf.getValidationChecks()) {
				stream.println(in + "        * " + check);
			}
		}

		// Sub-Profiles
		if (!profile.getCachedProfiles().isEmpty()) {
			stream.println(in + " - profiles:");
		}
		for (CachedProfile conf : profile.getCachedProfiles().values()) {
			dumpProfile(conf, 1, stream);
		}
	}

}
