
/*--------------------------------------------------------------------*/
/*--- begin                                       guest_ppc_toIR.c ---*/
/*--------------------------------------------------------------------*/

/*
   This file is part of Valgrind, a dynamic binary instrumentation
   framework.

   Copyright (C) 2004-2017 OpenWorks LLP
      info@open-works.net

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, see <http://www.gnu.org/licenses/>.

   The GNU General Public License is contained in the file COPYING.

   Neither the names of the U.S. Department of Energy nor the
   University of California nor the names of its contributors may be
   used to endorse or promote products derived from this software
   without prior written permission.
*/

/* TODO 18/Nov/05:

   Spot rld... cases which are simply left/right shifts and emit
   Shl64/Shr64 accordingly.

   Altivec
   - datastream insns
   - lvxl,stvxl: load/store with 'least recently used' hint
   - vexptefp, vlogefp

   LIMITATIONS:

   Various, including:

   - Some invalid forms of lswi and lswx are accepted when they should
     not be.

   - Floating Point:
     - All exceptions disabled in FPSCR
     - condition codes not set in FPSCR

   - Altivec floating point:
     - vmaddfp, vnmsubfp
       Because we're using Java/IEEE mode (FPSCR[NJ]), rather than the
       system default of Non-Java mode, we get some small errors
       (lowest bit only).
       This is because Non-Java mode brutally hacks denormalised results
       to zero, whereas we keep maximum accuracy.  However, using
       Non-Java mode would give us more inaccuracy, as our intermediate
       results would then be zeroed, too.

   - AbiHints for the stack red zone are only emitted for
       unconditional calls and returns (bl, blr).  They should also be
       emitted for conditional calls and returns, but we don't have a 
       way to express that right now.  Ah well.

   - Uses of Iop_{Add,Sub,Mul}32Fx4: the backend (host_ppc_isel.c)
       ignores the rounding mode, and generates code that assumes
       round-to-nearest.  This means V will compute incorrect results
       for uses of these IROps when the rounding mode (first) arg is
       not mkU32(Irrm_NEAREST).
*/

/* "Special" instructions.

   This instruction decoder can decode four special instructions
   which mean nothing natively (are no-ops as far as regs/mem are
   concerned) but have meaning for supporting Valgrind.  A special
   instruction is flagged by a 16-byte preamble:

      32-bit mode: 5400183E 5400683E 5400E83E 5400983E
                   (rlwinm 0,0,3,0,31; rlwinm 0,0,13,0,31; 
                    rlwinm 0,0,29,0,31; rlwinm 0,0,19,0,31)

      64-bit mode: 78001800 78006800 7800E802 78009802
                   (rotldi 0,0,3; rotldi 0,0,13;
                    rotldi 0,0,61; rotldi 0,0,51)

   Following that, one of the following 3 are allowed
   (standard interpretation in parentheses):

      7C210B78 (or 1,1,1)   %R3 = client_request ( %R4 )
      7C421378 (or 2,2,2)   %R3 = guest_NRADDR
      7C631B78 (or 3,3,3)   branch-and-link-to-noredir %R11  Big endian
      7C631B78 (or 3,3,3)   branch-and-link-to-noredir %R12  Little endian
      7C842378 (or 4,4,4)   %R3 = guest_NRADDR_GPR2
      7CA52B78 (or 5,5,5)   IR injection

   Any other bytes following the 16-byte preamble are illegal and
   constitute a failure in instruction decoding.  This all assumes
   that the preamble will never occur except in specific code
   fragments designed for Valgrind to catch.
*/

/*  Little Endian notes  */
/*
 * Vector operations in little Endian mode behave in non-obvious ways at times.
 * Below is an attempt at explaining this.
 *
 * LE/BE vector example
 *   With a vector of unsigned ints declared as follows:
 *     vector unsigned int vec_inA =
                            { 0x11111111, 0x22222222, 0x33333333, 0x44444444 };
 *   The '0x11111111' word is word zero in both LE and BE format.  But the
 *   loaded vector register will have word zero on the far left in BE mode and
 *   on the far right in LE mode. The lvx and stvx instructions work naturally
 *   for whatever endianness is in effect.  For example, in LE mode, the stvx
 *   stores word zero (far right word) of the vector at the lowest memory
 *   address of the EA; in BE mode, stvx still stores word zero at the lowest
 *   memory address, but with word zero interpreted as the one at the far left
 *   of the register.
 *
 *   The lxvd2x and stxvd2x instructions are not so well suited for LE mode.
 *   When the compiler generates an lxvd2x instruction to load the
 *   above-declared vector of unsigned integers, it loads the vector as two
 *   double words, but they are in BE word-wise format.  To put the vector in
 *   the right order for LE, the compiler also generates an xxswapd after the
 *   load, which puts it in proper LE format.  Similarly, the stxvd2x
 *   instruction has a BE bias, storing the vector in BE word-wise format. But
 *   the compiler also generates an xxswapd prior to the store, thus ensuring
 *   the vector is stored in memory in the correct LE order.
 *
 *   Vector-flavored Iops, such Iop_V128Hito64, reference the hi and lo parts
 *   of a double words and words within a vector.  Because of the reverse order
 *   of numbering for LE as described above, the high part refers to word 1 in
 *   LE format. When input data is saved to a guest state vector register
 *   (e.g., via Iop_64HLtoV128), it is first saved to memory and then the
 *   register is loaded via PPCInstr_AvLdSt, which does an lvx instruction.
 *   The saving of the data to memory must be done in proper LE order.  For the
 *   inverse operation of extracting data from a vector register (e.g.,
 *   Iop_V128Hito64), the register is first saved (by PPCInstr_AvLdSt resulting
 *   in stvx), and then integer registers are loaded from the memory location
 *   from where the vector register was saved.  Again, this must be done in
 *   proper LE order.  So for these various vector Iops, we have LE-specific
 *   code in host_ppc_isel.c
 *
 *   Another unique behavior of vectors in LE mode is with the vector scalar
 *   (VSX) operations that operate on "double word 0" of the source register,
 *   storing the result in "double word 0" of the output vector register.  For
 *   these operations, "double word 0" is interpreted as "high half of the
 *   register" (i.e, the part on the left side).
 *
 */

/* Notes on handling subnormal results:
 *
 * The various vector floating point instructions:
 *    vmaddfp, vaddfp, vsubfp, vmaxfp, vminfp, vrefp, vexptefp,
 *    vlogefp, vcmpeqfp, vcmpgefp, vcmpgtfp, vcmpbfp, vrfin, vrfiz,
 *     vrfip, vrfim
 * generate subnormal results that are controled by the VSCR[NJ] bit setting.
 *
 * The following describes how the host and guest is setup so that the function
 * dnorm_adj_Vector() can properly handle the results of the Iops in the guest
 * state.
 *
 *   At startup, on all host variants, we set VSCR[NJ].host = 0 (don't flush to
 *   zero).  It stays at 0 permanently.
 *
 *   At startup, we set VSCR[NJ].guest = (if BE then 1 else 0)
 *
 *   When running, guest insns can set/clear/query VSCR[NJ].guest as they
 *   like.
 *
 *   When running, any (guest) insn whose result depends on VSCR[NJ] will query
 *   VSCR[NJ].guest and the results will be truncated accordingly, by
 *   dnorm_adj_Vector().  Because VSCR[NJ].host is always 0, we will always
 *   be able to provide correct guest results for either value of
 *   VSCR[NJ].guest.
 */


/* Translates PPC32/64 code to IR. */

/* References

#define PPC32
   "PowerPC Microprocessor Family:
    The Programming Environments Manual for 32-Bit Microprocessors"
    02/21/2000
    http://www-3.ibm.com/chips/techlib/techlib.nsf/techdocs/852569B20050FF778525699600719DF2

#define PPC64
   "PowerPC Microprocessor Family:
    Programming Environments Manual for 64-Bit Microprocessors"
    06/10/2003
   http://www-3.ibm.com/chips/techlib/techlib.nsf/techdocs/F7E732FF811F783187256FDD004D3797

#define AV
   "PowerPC Microprocessor Family:
    AltiVec(TM) Technology Programming Environments Manual"
    07/10/2003
   http://www-3.ibm.com/chips/techlib/techlib.nsf/techdocs/FBFA164F824370F987256D6A006F424D
*/

#include "libvex_basictypes.h"
#include "libvex_ir.h"
#include "libvex.h"
#include "libvex_emnote.h"
#include "libvex_guest_ppc32.h"
#include "libvex_guest_ppc64.h"

#include "main_util.h"
#include "main_globals.h"
#include "guest_generic_bb_to_IR.h"
#include "guest_ppc_defs.h"

/*------------------------------------------------------------*/
/*--- Globals                                              ---*/
/*------------------------------------------------------------*/

/* These are set at the start of the translation of an insn, right
   down in disInstr_PPC, so that we don't have to pass them around
   endlessly.  They are all constant during the translation of any
   given insn. */

/* We need to know this to do sub-register accesses correctly. */
static VexEndness host_endness;

/* Pointer to the guest code area. */
static const UChar* guest_code;

/* The guest address corresponding to guest_code[0]. */
static Addr64 guest_CIA_bbstart;

/* The guest address for the instruction currently being
   translated. */
static Addr64 guest_CIA_curr_instr;

/* The IRSB* into which we're generating code. */
static IRSB* irsb;

/* Is our guest binary 32 or 64bit?  Set at each call to
   disInstr_PPC below. */
static Bool mode64 = False;

// Given a pointer to a function as obtained by "& functionname" in C,
// produce a pointer to the actual entry point for the function.  For
// most platforms it's the identity function.  Unfortunately, on
// ppc64-linux it isn't (sigh)
static void* fnptr_to_fnentry( const VexAbiInfo* vbi, void* f )
{
   if (vbi->host_ppc_calls_use_fndescrs) {
      /* f is a pointer to a 3-word function descriptor, of which the
         first word is the entry address. */
      /* note, this is correct even with cross-jitting, since this is
         purely a host issue, not a guest one. */
      HWord* fdescr = (HWord*)f;
      return (void*)(fdescr[0]);
   } else {
      /* Simple; "& f" points directly at the code for f. */
      return f;
   }
}

/* The OV32 and CA32 bits were added with ISA3.0 */
static Bool OV32_CA32_supported = False;

#define SIGN_BIT  0x8000000000000000ULL
#define SIGN_MASK 0x7fffffffffffffffULL
#define SIGN_BIT32  0x80000000
#define SIGN_MASK32 0x7fffffff

/* The instruction size can be either 4 byte (word instruction) or 8 bytes
   (prefix instruction) starting with ISA 3.1 */
#define WORD_INST_SIZE    4
#define PREFIX_INST_SIZE  8

/*------------------------------------------------------------*/
/*--- Debugging output                                     ---*/
/*------------------------------------------------------------*/

/* Pre DIP macro for prefix instruction printing.  */
#define pDIP(flag,format, args...)     \
   if (vex_traceflags & VEX_TRACE_FE){				\
     if (flag) {vex_printf("p"); vex_printf(format, ## args);}	\
     else {vex_printf(format, ## args); vex_printf("\n");}}

/* Post DIP macro to print additional args for prefix instruction printing.  */
#define DIPp(flag,format, args...)     	\
   if (vex_traceflags & VEX_TRACE_FE) {				\
     if (flag) {vex_printf(format, ## args); vex_printf("\n");}}

/* Post DIP macro with no additional args for prefix instruction printing.  */
#define DIPn(flag)                     \
   if (vex_traceflags & VEX_TRACE_FE) {if (flag) vex_printf("\n");}

#define DIP(format, args...)           \
   if (vex_traceflags & VEX_TRACE_FE)  \
      vex_printf(format, ## args)

#define DIS(buf, format, args...)      \
   if (vex_traceflags & VEX_TRACE_FE)  \
      vex_sprintf(buf, format, ## args)


/*------------------------------------------------------------*/
/*--- Offsets of various parts of the ppc32/64 guest state ---*/
/*------------------------------------------------------------*/

#define offsetofPPCGuestState(_x) \
   (mode64 ? offsetof(VexGuestPPC64State, _x) : \
             offsetof(VexGuestPPC32State, _x))

#define OFFB_CIA         offsetofPPCGuestState(guest_CIA)
#define OFFB_IP_AT_SYSCALL offsetofPPCGuestState(guest_IP_AT_SYSCALL)
#define OFFB_SPRG3_RO    offsetofPPCGuestState(guest_SPRG3_RO)
#define OFFB_LR          offsetofPPCGuestState(guest_LR)
#define OFFB_CTR         offsetofPPCGuestState(guest_CTR)
#define OFFB_XER_SO      offsetofPPCGuestState(guest_XER_SO)
#define OFFB_XER_OV      offsetofPPCGuestState(guest_XER_OV)
#define OFFB_XER_OV32    offsetofPPCGuestState(guest_XER_OV32)
#define OFFB_XER_CA      offsetofPPCGuestState(guest_XER_CA)
#define OFFB_XER_CA32    offsetofPPCGuestState(guest_XER_CA32)
#define OFFB_XER_BC      offsetofPPCGuestState(guest_XER_BC)
#define OFFB_FPROUND     offsetofPPCGuestState(guest_FPROUND)
#define OFFB_DFPROUND    offsetofPPCGuestState(guest_DFPROUND)
#define OFFB_C_FPCC      offsetofPPCGuestState(guest_C_FPCC)
#define OFFB_VRSAVE      offsetofPPCGuestState(guest_VRSAVE)
#define OFFB_VSCR        offsetofPPCGuestState(guest_VSCR)
#define OFFB_EMNOTE      offsetofPPCGuestState(guest_EMNOTE)
#define OFFB_CMSTART     offsetofPPCGuestState(guest_CMSTART)
#define OFFB_CMLEN       offsetofPPCGuestState(guest_CMLEN)
#define OFFB_NRADDR      offsetofPPCGuestState(guest_NRADDR)
#define OFFB_NRADDR_GPR2 offsetofPPCGuestState(guest_NRADDR_GPR2)
#define OFFB_TFHAR       offsetofPPCGuestState(guest_TFHAR)
#define OFFB_TEXASR      offsetofPPCGuestState(guest_TEXASR)
#define OFFB_TEXASRU     offsetofPPCGuestState(guest_TEXASRU)
#define OFFB_TFIAR       offsetofPPCGuestState(guest_TFIAR)
#define OFFB_PPR         offsetofPPCGuestState(guest_PPR)
#define OFFB_PSPB        offsetofPPCGuestState(guest_PSPB)
#define OFFB_DSCR        offsetofPPCGuestState(guest_DSCR)
/* Note the offset for the various ACC entries are calculated based on
   the OFFB_ACC_0_r0 value.  */
#define OFFB_ACC_0_r0    offsetofPPCGuestState(guest_ACC_0_r0)
#define OFFB_syscall_flag  offsetofPPCGuestState(guest_syscall_flag)


/*------------------------------------------------------------*/
/*--- Extract instruction fields                          --- */
/*------------------------------------------------------------*/

/* Extract field from insn, given idx (zero = lsb) and field length */
#define IFIELD( insn, idx, len ) ((insn >> idx) & ((1<<len)-1))

/* Extract primary opcode, instr[31:26] */
static UChar ifieldOPC( UInt instr ) {
   return toUChar( IFIELD( instr, 26, 6 ) );
}

/* Extract 11-bit secondary opcode, instr[10:0] */
static UInt ifieldOPClo11 ( UInt instr) {
   return IFIELD( instr, 0, 11 );
}

/* Extract 10-bit secondary opcode, instr[10:1] */
static UInt ifieldOPClo10 ( UInt instr) {
   return IFIELD( instr, 1, 10 );
}

/* Extract 9-bit secondary opcode, instr[9:1] */
static UInt ifieldOPClo9 ( UInt instr) {
   return IFIELD( instr, 1, 9 );
}

/* Extract 8-bit secondary opcode, instr[8:1] */
static UInt ifieldOPClo8 ( UInt instr) {
   return IFIELD( instr, 1, 8 );
}

/* Extract 4-bit secondary opcode, instr[5:1] */
static UInt ifieldOPClo4 ( UInt instr) {
   return IFIELD( instr, 0, 4 );
}

/* Extract 5-bit secondary opcode, instr[5:1] */
static UInt ifieldOPClo5 ( UInt instr) {
   return IFIELD( instr, 1, 5 );
}

/* Extract 2-bit secondary opcode, instr[1:0] */
static UInt ifieldOPC0o2 ( UInt instr) {
   return IFIELD( instr, 0, 2 );
}

/* Extract RD (destination register) field, instr[25:21] */
static UChar ifieldRegDS( UInt instr ) {
   return toUChar( IFIELD( instr, 21, 5 ) );
}

/* Extract XTp (destination register) field, instr[25:22, 21] */
static UChar ifieldRegXTp ( UInt instr )
{
   UChar TX = toUChar (IFIELD (instr, 21, 1));
   UChar Tp = toUChar (IFIELD (instr, 22, 4));
   /* XTp = 32 * TX + 2* Tp;  Only even values of XTp can be encoded.  */
   return (TX << 5) | (Tp << 1);
}

/* Extract XT (destination register) field, instr[0,25:21] */
static UChar ifieldRegXT ( UInt instr )
{
  UChar upper_bit = toUChar (IFIELD (instr, 0, 1));
  UChar lower_bits = toUChar (IFIELD (instr, 21, 5));
  return (upper_bit << 5) | lower_bits;
}

/* Extract XS (store source register) field, instr[0,25:21] */
static inline UChar ifieldRegXS ( UInt instr )
{
  return ifieldRegXT ( instr );
}

/* Extract RA (1st source register) field, instr[20:16] */
static UChar ifieldRegA ( UInt instr ) {
   return toUChar( IFIELD( instr, 16, 5 ) );
}

/* Extract XA (1st source register) field, instr[2,20:16] */
static UChar ifieldRegXA ( UInt instr )
{
  UChar upper_bit = toUChar (IFIELD (instr, 2, 1));
  UChar lower_bits = toUChar (IFIELD (instr, 16, 5));
  return (upper_bit << 5) | lower_bits;
}

/* Extract RB (2nd source register) field, instr[15:11] */
static UChar ifieldRegB ( UInt instr ) {
   return toUChar( IFIELD( instr, 11, 5 ) );
}

/* Extract XB (2nd source register) field, instr[1,15:11] */
static UChar ifieldRegXB ( UInt instr )
{
  UChar upper_bit = toUChar (IFIELD (instr, 1, 1));
  UChar lower_bits = toUChar (IFIELD (instr, 11, 5));
  return (upper_bit << 5) | lower_bits;
}

/* Extract RC (3rd source register) field, instr[10:6] */
static UChar ifieldRegC ( UInt instr ) {
   return toUChar( IFIELD( instr, 6, 5 ) );
}

/* Extract XC (3rd source register) field, instr[3,10:6] */
static UChar ifieldRegXC ( UInt instr )
{
  UChar upper_bit = toUChar (IFIELD (instr, 3, 1));
  UChar lower_bits = toUChar (IFIELD (instr, 6, 5));
  return (upper_bit << 5) | lower_bits;
}

/* Extract bit 10, instr[10] */
static UChar ifieldBIT10 ( UInt instr ) {
   return toUChar( IFIELD( instr, 10, 1 ) );
}

/* Extract 2nd lowest bit, instr[1] */
static UChar ifieldBIT1 ( UInt instr ) {
   return toUChar( IFIELD( instr, 1, 1 ) );
}

/* Extract lowest bit, instr[0] */
static UChar ifieldBIT0 ( UInt instr ) {
   return toUChar( instr & 0x1 );
}

/* Extract unsigned bottom half, instr[15:0] */
static UInt ifieldUIMM16 ( UInt instr ) {
   return instr & 0xFFFF;
}

/* Extract unsigned bottom 26 bits, instr[25:0] */
static UInt ifieldUIMM26 ( UInt instr ) {
   return instr & 0x3FFFFFF;
}

/* Extract DM field, instr[9:8] */
static UChar ifieldDM ( UInt instr ) {
   return toUChar( IFIELD( instr, 8, 2 ) );
}

/* Extract SHW field, instr[9:8] */
static inline UChar ifieldSHW ( UInt instr )
{
  return ifieldDM ( instr );
}

/* Extract AT field from theInstr  8LS:D form */
static UChar ifieldAT ( UInt instr ) {
   return toUChar( IFIELD( instr, 23, 3 ) );
}

/*------------------------------------------------------------*/
/*--- Guest-state identifiers                              ---*/
/*------------------------------------------------------------*/

typedef enum {
    PPC_GST_CIA,    // Current Instruction Address
    PPC_GST_LR,     // Link Register
    PPC_GST_CTR,    // Count Register
    PPC_GST_XER,    // Overflow, carry flags, byte count
    PPC_GST_CR,     // Condition Register
    PPC_GST_FPSCR,  // Floating Point Status/Control Register
    PPC_GST_VRSAVE, // Vector Save/Restore Register
    PPC_GST_VSCR,   // Vector Status and Control Register
    PPC_GST_EMWARN, // Emulation warnings
    PPC_GST_CMSTART,// For icbi: start of area to invalidate
    PPC_GST_CMLEN,  // For icbi: length of area to invalidate
    PPC_GST_IP_AT_SYSCALL, // the CIA of the most recently executed SC insn
    PPC_GST_SPRG3_RO, // SPRG3
    PPC_GST_TFHAR,  // Transactional Failure Handler Address Register
    PPC_GST_TFIAR,  // Transactional Failure Instruction Address Register
    PPC_GST_TEXASR, // Transactional EXception And Summary Register
    PPC_GST_TEXASRU, // Transactional EXception And Summary Register Upper
    PPC_GST_PPR,     // Program Priority register
    PPC_GST_PPR32,   // Upper 32-bits of Program Priority register
    PPC_GST_PSPB,    /* Problem State Priority Boost register, Note, the
                      * register is initialized to a non-zero value.  Currently
                      * Valgrind is not supporting the register value to
                      * automatically decrement. Could be added later if
                      * needed.
                      */
    PPC_GST_DSCR,     // Data Stream Control Register
    PPC_GST_ACC_0_r0, /* Accumulator register file.  Eight accumulators each
                       * with four 128-bit registers.
                       */
    PPC_GST_ACC_0_r1,
    PPC_GST_ACC_0_r2,
    PPC_GST_ACC_0_r3,
    PPC_GST_ACC_1_r0,
    PPC_GST_ACC_1_r1,
    PPC_GST_ACC_1_r2,
    PPC_GST_ACC_1_r3,
    PPC_GST_ACC_2_r0,
    PPC_GST_ACC_2_r1,
    PPC_GST_ACC_2_r2,
    PPC_GST_ACC_2_r3,
    PPC_GST_ACC_3_r0,
    PPC_GST_ACC_3_r1,
    PPC_GST_ACC_3_r2,
    PPC_GST_ACC_3_r3,
    PPC_GST_ACC_4_r0,
    PPC_GST_ACC_4_r1,
    PPC_GST_ACC_4_r2,
    PPC_GST_ACC_4_r3,
    PPC_GST_ACC_5_r0,
    PPC_GST_ACC_5_r1,
    PPC_GST_ACC_5_r2,
    PPC_GST_ACC_5_r3,
    PPC_GST_ACC_6_r0,
    PPC_GST_ACC_6_r1,
    PPC_GST_ACC_6_r2,
    PPC_GST_ACC_6_r3,
    PPC_GST_ACC_7_r0,
    PPC_GST_ACC_7_r1,
    PPC_GST_ACC_7_r2,
    PPC_GST_ACC_7_r3,
    PPC_GST_MAX
} PPC_GST;

#define MASK_FPSCR_RN     0x3ULL         // Binary floating point rounding mode
#define MASK_FPSCR_DRN    0x700000000ULL // Decimal floating point rounding mode
#define MASK_FPSCR_C_FPCC 0x1F000ULL     // Floating-Point Condition code FPCC

#define MASK_VSCR_VALID 0x00010001


/*------------------------------------------------------------*/
/*--- Misc Helpers                                         ---*/
/*------------------------------------------------------------*/

static void Get_lmd( IRTemp * lmd, IRExpr * gfield_0_4 );

/* Generate mask with 1's from 'begin' through 'end',
   wrapping if begin > end.
   begin->end works from right to left, 0=lsb
*/
static UInt MASK32( UInt begin, UInt end )
{
   UInt m1, m2, mask;
   vassert(begin < 32);
   vassert(end < 32);
   m1   = ((UInt)(-1)) << begin;
   m2   = ((UInt)(-1)) << end << 1;
   mask = m1 ^ m2;
   if (begin > end) mask = ~mask;  // wrap mask
   return mask;
}

static ULong MASK64( UInt begin, UInt end )
{
   ULong m1, m2, mask;
   vassert(begin < 64);
   vassert(end < 64);
   m1   = ((ULong)(-1)) << begin;
   m2   = ((ULong)(-1)) << end << 1;
   mask = m1 ^ m2;
   if (begin > end) mask = ~mask;  // wrap mask
   return mask;
}

static Addr64 nextInsnAddr( void )
{
   /* Note in the case of a prefix instruction, delta has already been
      incremented by WORD_INST_SIZE to move past the prefix part of the
      instruction.  So only need to increment by WORD_INST_SIZE to get to
      the start of the next instruction.  */
   return guest_CIA_curr_instr + WORD_INST_SIZE;
}


/*------------------------------------------------------------*/
/*--- Helper bits and pieces for deconstructing the        ---*/
/*--- ppc32/64 insn stream.                                ---*/
/*------------------------------------------------------------*/

/* Add a statement to the list held by "irsb". */
static void stmt ( IRStmt* st )
{
   addStmtToIRSB( irsb, st );
}

/* Generate a new temporary of the given type. */
static IRTemp newTemp ( IRType ty )
{
   vassert(isPlausibleIRType(ty));
   return newIRTemp( irsb->tyenv, ty );
}

/* Various simple conversions */

static UChar extend_s_5to8 ( UChar x )
{
   return toUChar((((Int)x) << 27) >> 27);
}

static UInt extend_s_8to32( UChar x )
{
   return (UInt)((((Int)x) << 24) >> 24);
}

static UInt extend_s_16to32 ( UInt x )
{
   return (UInt)((((Int)x) << 16) >> 16);
}

static ULong extend_s_16to64 ( UInt x )
{
   return (ULong)((((Long)x) << 48) >> 48);
}

static ULong extend_s_26to64 ( UInt x )
{
   return (ULong)((((Long)x) << 38) >> 38);
}

static ULong extend_s_32to64 ( UInt x )
{
   return (ULong)((((Long)x) << 32) >> 32);
}

/* Do a proper-endian load of a 32-bit word, regardless of the endianness
   of the underlying host. */
static UInt getUIntPPCendianly ( const UChar* p )
{
   UInt w = 0;
   if (host_endness == VexEndnessBE) {
       w = (w << 8) | p[0];
       w = (w << 8) | p[1];
       w = (w << 8) | p[2];
       w = (w << 8) | p[3];
   } else {
       w = (w << 8) | p[3];
       w = (w << 8) | p[2];
       w = (w << 8) | p[1];
       w = (w << 8) | p[0];
   }
   return w;
}


/*------------------------------------------------------------*/
/*--- Helpers for constructing IR.                         ---*/
/*------------------------------------------------------------*/

static void assign ( IRTemp dst, IRExpr* e )
{
   stmt( IRStmt_WrTmp(dst, e) );
}

/* This generates a normal (non store-conditional) store. */
static void store ( IRExpr* addr, IRExpr* data )
{
   IRType tyA = typeOfIRExpr(irsb->tyenv, addr);
   vassert(tyA == Ity_I32 || tyA == Ity_I64);

   if (host_endness == VexEndnessBE)
      stmt( IRStmt_Store(Iend_BE, addr, data) );
   else
      stmt( IRStmt_Store(Iend_LE, addr, data) );
}

static IRExpr* unop ( IROp op, IRExpr* a )
{
   return IRExpr_Unop(op, a);
}

static IRExpr* binop ( IROp op, IRExpr* a1, IRExpr* a2 )
{
   return IRExpr_Binop(op, a1, a2);
}

static IRExpr* triop ( IROp op, IRExpr* a1, IRExpr* a2, IRExpr* a3 )
{
   return IRExpr_Triop(op, a1, a2, a3);
}

static IRExpr* qop ( IROp op, IRExpr* a1, IRExpr* a2, 
                              IRExpr* a3, IRExpr* a4 )
{
   return IRExpr_Qop(op, a1, a2, a3, a4);
}

static IRExpr* mkexpr ( IRTemp tmp )
{
   return IRExpr_RdTmp(tmp);
}

#define mkU1(_n)  IRExpr_Const(IRConst_U1(_n))

static IRExpr* mkU8 ( UChar i )
{
   return IRExpr_Const(IRConst_U8(i));
}

static IRExpr* mkU16 ( UInt i )
{
   return IRExpr_Const(IRConst_U16(i));
}

static IRExpr* mkU32 ( UInt i )
{
   return IRExpr_Const(IRConst_U32(i));
}

static IRExpr* mkU64 ( ULong i )
{
   return IRExpr_Const(IRConst_U64(i));
}

static IRExpr* mkV128 ( UShort i )
{
   vassert(i == 0 || i == 0xffff);
   return IRExpr_Const(IRConst_V128(i));
}

/* This generates a normal (non load-linked) load. */
static IRExpr* load ( IRType ty, IRExpr* addr )
{
   if (host_endness == VexEndnessBE)
      return IRExpr_Load(Iend_BE, ty, addr);
   else
      return IRExpr_Load(Iend_LE, ty, addr);
}

static IRStmt* stmt_load ( IRTemp result,
                           IRExpr* addr, IRExpr* storedata )
{
   if (host_endness == VexEndnessBE)
      return IRStmt_LLSC(Iend_BE, result, addr, storedata);
   else
      return IRStmt_LLSC(Iend_LE, result, addr, storedata);
}

static IRExpr* mkOR1 ( IRExpr* arg1, IRExpr* arg2 )
{
   vassert(typeOfIRExpr(irsb->tyenv, arg1) == Ity_I1);
   vassert(typeOfIRExpr(irsb->tyenv, arg2) == Ity_I1);
   return unop(Iop_32to1, binop(Iop_Or32, unop(Iop_1Uto32, arg1), 
                                          unop(Iop_1Uto32, arg2)));
}

static IRExpr* mkAND1 ( IRExpr* arg1, IRExpr* arg2 )
{
   vassert(typeOfIRExpr(irsb->tyenv, arg1) == Ity_I1);
   vassert(typeOfIRExpr(irsb->tyenv, arg2) == Ity_I1);
   return unop(Iop_32to1, binop(Iop_And32, unop(Iop_1Uto32, arg1), 
                                           unop(Iop_1Uto32, arg2)));
}

static inline IRExpr* mkXOr4_32( IRTemp t0, IRTemp t1, IRTemp t2,
                                 IRTemp t3 )
{
   return binop( Iop_Xor32,
                 binop( Iop_Xor32, mkexpr( t0 ), mkexpr( t1 ) ),
                 binop( Iop_Xor32, mkexpr( t2 ), mkexpr( t3 ) ) );
}

static inline IRExpr* mkOr3_V128( IRTemp t0, IRTemp t1, IRTemp t2 )
{
   return binop( Iop_OrV128,
                 mkexpr( t0 ),
                 binop( Iop_OrV128, mkexpr( t1 ), mkexpr( t2 ) ) );
}

static inline IRExpr* mkOr4_V128( IRTemp t0, IRTemp t1, IRTemp t2,
                                  IRTemp t3 )
{
   return binop( Iop_OrV128,
                 binop( Iop_OrV128, mkexpr( t0 ), mkexpr( t1 ) ),
                 binop( Iop_OrV128, mkexpr( t2 ), mkexpr( t3 ) ) );
}

static inline IRExpr* mkOr4_V128_expr( IRExpr* t0, IRExpr* t1, IRExpr* t2,
                                       IRExpr* t3 )
{
   /* arguments are already expressions */
   return binop( Iop_OrV128,
                 binop( Iop_OrV128, ( t0 ), ( t1 ) ),
                 binop( Iop_OrV128, ( t2 ), ( t3 ) ) );
}

static IRExpr* mkNOT1 ( IRExpr* arg1 )
{
   vassert(typeOfIRExpr(irsb->tyenv, arg1) == Ity_I1);
   return unop(Iop_32to1, unop(Iop_Not32, unop(Iop_1Uto32, arg1) ) );
}

/* expand V128_8Ux16 to 2x V128_16Ux8's */
static void expand8Ux16( IRExpr* vIn,
                         /*OUTs*/ IRTemp* vEvn, IRTemp* vOdd )
{
   IRTemp ones8x16 = newTemp(Ity_V128);

   vassert(typeOfIRExpr(irsb->tyenv, vIn) == Ity_V128);
   vassert(vEvn && *vEvn == IRTemp_INVALID);
   vassert(vOdd && *vOdd == IRTemp_INVALID);
   *vEvn = newTemp(Ity_V128);
   *vOdd = newTemp(Ity_V128);

   assign( ones8x16, unop(Iop_Dup8x16, mkU8(0x1)) );
   assign( *vOdd, binop(Iop_MullEven8Ux16, mkexpr(ones8x16), vIn) );
   assign( *vEvn, binop(Iop_MullEven8Ux16, mkexpr(ones8x16), 
                        binop(Iop_ShrV128, vIn, mkU8(8))) );
}

/* expand V128_8Sx16 to 2x V128_16Sx8's */
static void expand8Sx16( IRExpr* vIn,
                         /*OUTs*/ IRTemp* vEvn, IRTemp* vOdd )
{
   IRTemp ones8x16 = newTemp(Ity_V128);

   vassert(typeOfIRExpr(irsb->tyenv, vIn) == Ity_V128);
   vassert(vEvn && *vEvn == IRTemp_INVALID);
   vassert(vOdd && *vOdd == IRTemp_INVALID);
   *vEvn = newTemp(Ity_V128);
   *vOdd = newTemp(Ity_V128);

   assign( ones8x16, unop(Iop_Dup8x16, mkU8(0x1)) );
   assign( *vOdd, binop(Iop_MullEven8Sx16, mkexpr(ones8x16), vIn) );
   assign( *vEvn, binop(Iop_MullEven8Sx16, mkexpr(ones8x16), 
                        binop(Iop_ShrV128, vIn, mkU8(8))) );
}

/* expand V128_16Uto8 to 2x V128_32Ux4's */
static void expand16Ux8( IRExpr* vIn,
                         /*OUTs*/ IRTemp* vEvn, IRTemp* vOdd )
{
   IRTemp ones16x8 = newTemp(Ity_V128);

   vassert(typeOfIRExpr(irsb->tyenv, vIn) == Ity_V128);
   vassert(vEvn && *vEvn == IRTemp_INVALID);
   vassert(vOdd && *vOdd == IRTemp_INVALID);
   *vEvn = newTemp(Ity_V128);
   *vOdd = newTemp(Ity_V128);

   assign( ones16x8, unop(Iop_Dup16x8, mkU16(0x1)) );
   assign( *vOdd, binop(Iop_MullEven16Ux8, mkexpr(ones16x8), vIn) );
   assign( *vEvn, binop(Iop_MullEven16Ux8, mkexpr(ones16x8), 
                        binop(Iop_ShrV128, vIn, mkU8(16))) );
}

/* expand V128_16Sto8 to 2x V128_32Sx4's */
static void expand16Sx8( IRExpr* vIn,
                         /*OUTs*/ IRTemp* vEvn, IRTemp* vOdd )
{
   IRTemp ones16x8 = newTemp(Ity_V128);

   vassert(typeOfIRExpr(irsb->tyenv, vIn) == Ity_V128);
   vassert(vEvn && *vEvn == IRTemp_INVALID);
   vassert(vOdd && *vOdd == IRTemp_INVALID);
   *vEvn = newTemp(Ity_V128);
   *vOdd = newTemp(Ity_V128);

   assign( ones16x8, unop(Iop_Dup16x8, mkU16(0x1)) );
   assign( *vOdd, binop(Iop_MullEven16Sx8, mkexpr(ones16x8), vIn) );
   assign( *vEvn, binop(Iop_MullEven16Sx8, mkexpr(ones16x8), 
                       binop(Iop_ShrV128, vIn, mkU8(16))) );
}

/* break V128 to 4xF64's*/
static void breakV128to4xF64( IRExpr* t128,
                              /*OUTs*/
                              IRTemp* t3, IRTemp* t2,
                              IRTemp* t1, IRTemp* t0 )
{
   IRTemp hi64 = newTemp(Ity_I64);
   IRTemp lo64 = newTemp(Ity_I64);

   vassert(typeOfIRExpr(irsb->tyenv, t128) == Ity_V128);
   vassert(t0 && *t0 == IRTemp_INVALID);
   vassert(t1 && *t1 == IRTemp_INVALID);
   vassert(t2 && *t2 == IRTemp_INVALID);
   vassert(t3 && *t3 == IRTemp_INVALID);
   *t0 = newTemp(Ity_F64);
   *t1 = newTemp(Ity_F64);
   *t2 = newTemp(Ity_F64);
   *t3 = newTemp(Ity_F64);

   assign( hi64, unop(Iop_V128HIto64, t128) );
   assign( lo64, unop(Iop_V128to64,   t128) );
   assign( *t3,
           unop( Iop_F32toF64,
                 unop( Iop_ReinterpI32asF32,
                       unop( Iop_64HIto32, mkexpr( hi64 ) ) ) ) );
   assign( *t2,
           unop( Iop_F32toF64,
                 unop( Iop_ReinterpI32asF32, unop( Iop_64to32, mkexpr( hi64 ) ) ) ) );
   assign( *t1,
           unop( Iop_F32toF64,
                 unop( Iop_ReinterpI32asF32,
                       unop( Iop_64HIto32, mkexpr( lo64 ) ) ) ) );
   assign( *t0,
           unop( Iop_F32toF64,
                 unop( Iop_ReinterpI32asF32, unop( Iop_64to32, mkexpr( lo64 ) ) ) ) );
}


/* break V128 to 4xI32's, then sign-extend to I64's */
static void breakV128to4x64S( IRExpr* t128,
                              /*OUTs*/
                              IRTemp* t3, IRTemp* t2,
                              IRTemp* t1, IRTemp* t0 )
{
   IRTemp hi64 = newTemp(Ity_I64);
   IRTemp lo64 = newTemp(Ity_I64);

   vassert(typeOfIRExpr(irsb->tyenv, t128) == Ity_V128);
   vassert(t0 && *t0 == IRTemp_INVALID);
   vassert(t1 && *t1 == IRTemp_INVALID);
   vassert(t2 && *t2 == IRTemp_INVALID);
   vassert(t3 && *t3 == IRTemp_INVALID);
   *t0 = newTemp(Ity_I64);
   *t1 = newTemp(Ity_I64);
   *t2 = newTemp(Ity_I64);
   *t3 = newTemp(Ity_I64);

   assign( hi64, unop(Iop_V128HIto64, t128) );
   assign( lo64, unop(Iop_V128to64,   t128) );
   assign( *t3, unop(Iop_32Sto64, unop(Iop_64HIto32, mkexpr(hi64))) );
   assign( *t2, unop(Iop_32Sto64, unop(Iop_64to32,   mkexpr(hi64))) );
   assign( *t1, unop(Iop_32Sto64, unop(Iop_64HIto32, mkexpr(lo64))) );
   assign( *t0, unop(Iop_32Sto64, unop(Iop_64to32,   mkexpr(lo64))) );
}

/* break V128 to 4xI32's, then zero-extend to I64's */
static void breakV128to4x64U ( IRExpr* t128,
                               /*OUTs*/
                               IRTemp* t3, IRTemp* t2,
                               IRTemp* t1, IRTemp* t0 )
{
   IRTemp hi64 = newTemp(Ity_I64);
   IRTemp lo64 = newTemp(Ity_I64);

   vassert(typeOfIRExpr(irsb->tyenv, t128) == Ity_V128);
   vassert(t0 && *t0 == IRTemp_INVALID);
   vassert(t1 && *t1 == IRTemp_INVALID);
   vassert(t2 && *t2 == IRTemp_INVALID);
   vassert(t3 && *t3 == IRTemp_INVALID);
   *t0 = newTemp(Ity_I64);
   *t1 = newTemp(Ity_I64);
   *t2 = newTemp(Ity_I64);
   *t3 = newTemp(Ity_I64);

   assign( hi64, unop(Iop_V128HIto64, t128) );
   assign( lo64, unop(Iop_V128to64,   t128) );
   assign( *t3, unop(Iop_32Uto64, unop(Iop_64HIto32, mkexpr(hi64))) );
   assign( *t2, unop(Iop_32Uto64, unop(Iop_64to32,   mkexpr(hi64))) );
   assign( *t1, unop(Iop_32Uto64, unop(Iop_64HIto32, mkexpr(lo64))) );
   assign( *t0, unop(Iop_32Uto64, unop(Iop_64to32,   mkexpr(lo64))) );
}

static void breakV128to4x32( IRExpr* t128,
                              /*OUTs*/
                              IRTemp* t3, IRTemp* t2,
                              IRTemp* t1, IRTemp* t0 )
{
   IRTemp hi64 = newTemp(Ity_I64);
   IRTemp lo64 = newTemp(Ity_I64);

   vassert(typeOfIRExpr(irsb->tyenv, t128) == Ity_V128);
   vassert(t0 && *t0 == IRTemp_INVALID);
   vassert(t1 && *t1 == IRTemp_INVALID);
   vassert(t2 && *t2 == IRTemp_INVALID);
   vassert(t3 && *t3 == IRTemp_INVALID);
   *t0 = newTemp(Ity_I32);
   *t1 = newTemp(Ity_I32);
   *t2 = newTemp(Ity_I32);
   *t3 = newTemp(Ity_I32);

   assign( hi64, unop(Iop_V128HIto64, t128) );
   assign( lo64, unop(Iop_V128to64,   t128) );
   assign( *t3, unop(Iop_64HIto32, mkexpr(hi64)) );
   assign( *t2, unop(Iop_64to32,   mkexpr(hi64)) );
   assign( *t1, unop(Iop_64HIto32, mkexpr(lo64)) );
   assign( *t0, unop(Iop_64to32,   mkexpr(lo64)) );
}

static IRExpr* mkV128from32( IRTemp t3, IRTemp t2,
                               IRTemp t1, IRTemp t0 )
{
   return
      binop( Iop_64HLtoV128,
             binop(Iop_32HLto64, mkexpr(t3), mkexpr(t2)),
             binop(Iop_32HLto64, mkexpr(t1), mkexpr(t0))
   );
}

static IRExpr* extract_field_from_vector( IRTemp vB, IRExpr* index, UInt mask)
{
   /* vB is a vector, extract bits starting at index to size of mask */
   return unop( Iop_V128to64,
                binop( Iop_AndV128,
                       binop( Iop_ShrV128,
                              mkexpr( vB ),
                              unop( Iop_64to8,
                                    binop( Iop_Mul64, index,
                                           mkU64( 8 ) ) ) ),
                       binop( Iop_64HLtoV128,
                              mkU64( 0x0 ),
                              mkU64( mask ) ) ) );
}

static IRExpr* insert_field_into_vector( IRTemp vSrc, IRExpr* index,
                                         IRExpr* bits, IRExpr* mask)
{
   /* vSrc is a vector v128, index is I64 between 0 and 15 bytes, bits is I64,
      mask is I64.  Indexing is based on the least significant byte being
      index 0. Insert bits starting at index to size of mask */
   IRTemp shift = newTemp(Ity_I8);
   IRTemp tmp_mask  = newTemp(Ity_V128);
   IRTemp tmp_not_mask  = newTemp(Ity_V128);
   UInt index_mask = 0xF;  //Index is only 4-bits wide

   assign( shift, unop( Iop_64to8,
                        binop( Iop_Mul64,
                               binop( Iop_And64,
                                      index,
                                      mkU64( index_mask ) ),
                               mkU64( 8 ) ) ) );
   assign( tmp_mask, binop( Iop_ShlV128,
                            binop( Iop_64HLtoV128,
                                   mkU64( 0x0 ),
                                   mask ),
                            mkexpr( shift) ) );
   assign( tmp_not_mask, unop( Iop_NotV128, mkexpr( tmp_mask ) ) );
   return binop( Iop_OrV128,
                 binop( Iop_AndV128,
                        mkexpr( vSrc ),
                        mkexpr( tmp_not_mask ) ),
                 binop( Iop_AndV128,
                        binop( Iop_ShlV128,
                               binop( Iop_64HLtoV128,
                                      mkU64( 0x0 ),
                                      bits ),
                               mkexpr( shift) ),
                        mkexpr( tmp_mask ) ) );
}

static IRExpr* extractBytefromV256( IRTemp vA, IRTemp vB, UInt byte_index)
{
   UInt byte_mask  = 0xFF;
   UInt byte_size  = 8;   // size in bits
   IRTemp shift = newTemp(Ity_I8);
   IRTemp select_tmp = newTemp(Ity_I64);
   IRTemp reg_select = newTemp(Ity_V128);
   IRTemp src_tmp = newTemp(Ity_V128);

   /* The byte numbering is right to left:  byte_n-1, byte_n-2, ...., byte0.
      The byte-index is between 0 and 31.  */
   assign( shift, unop( Iop_64to8,
                        binop( Iop_Mul64,
                               binop( Iop_And64,
                                      mkU64( 0xF ),
                                      mkexpr( byte_index ) ),
                               mkU64( byte_size ) ) ) );

   /* Create mask to select byte from srcA if byte_index >  16 or
      from srcB.  Use byte_index[4] to select srcA or srcB.   */
   assign( select_tmp, unop( Iop_1Sto64,
                             unop( Iop_64to1,
                                   binop( Iop_Shr64,
                                          mkexpr( byte_index ),
                                          mkU8( 4 ) ) ) ) );

   assign( reg_select, binop( Iop_64HLtoV128,
                              mkexpr( select_tmp ),
                              mkexpr( select_tmp ) ) );

   assign( src_tmp,
           binop( Iop_OrV128,
                  binop( Iop_AndV128,
                         mkexpr( reg_select ),
                         binop( Iop_ShrV128,
                                mkexpr( vA ),
                                mkexpr( shift ) ) ),
                  binop( Iop_AndV128,
                         unop( Iop_NotV128, mkexpr( reg_select ) ),
                         binop( Iop_ShrV128,
                                mkexpr( vB ),
                                mkexpr( shift ) ) ) ) );

   /* Mask off element */
   return binop( Iop_And64,
                 unop( Iop_V128to64, mkexpr( src_tmp ) ),
                 mkU64( byte_mask ) );
}

/* Signed saturating narrow 64S to 32 */
static IRExpr* mkQNarrow64Sto32 ( IRExpr* t64 )
{
   IRTemp hi32 = newTemp(Ity_I32);
   IRTemp lo32 = newTemp(Ity_I32);

   vassert(typeOfIRExpr(irsb->tyenv, t64) == Ity_I64);

   assign( hi32, unop(Iop_64HIto32, t64));
   assign( lo32, unop(Iop_64to32,   t64));

   return IRExpr_ITE(
             /* if (hi32 == (lo32 >>s 31)) */
             binop(Iop_CmpEQ32, mkexpr(hi32),
                   binop( Iop_Sar32, mkexpr(lo32), mkU8(31))),
             /* then: within signed-32 range: lo half good enough */
             mkexpr(lo32),
             /* else: sign dep saturate: 1->0x80000000, 0->0x7FFFFFFF */
             binop(Iop_Add32, mkU32(0x7FFFFFFF),
                   binop(Iop_Shr32, mkexpr(hi32), mkU8(31))));
}

/* Unsigned saturating narrow 64S to 32 */
static IRExpr* mkQNarrow64Uto32 ( IRExpr* t64 )
{
   IRTemp hi32 = newTemp(Ity_I32);
   IRTemp lo32 = newTemp(Ity_I32);

   vassert(typeOfIRExpr(irsb->tyenv, t64) == Ity_I64);

   assign( hi32, unop(Iop_64HIto32, t64));
   assign( lo32, unop(Iop_64to32,   t64));

   return IRExpr_ITE(
            /* if (top 32 bits of t64 are 0) */
            binop(Iop_CmpEQ32, mkexpr(hi32), mkU32(0)),
            /* then: within unsigned-32 range: lo half good enough */
            mkexpr(lo32),
            /* else: positive saturate -> 0xFFFFFFFF */
            mkU32(0xFFFFFFFF));
}

/* Signed saturate narrow 64->32, combining to V128 */
static IRExpr* mkV128from4x64S ( IRExpr* t3, IRExpr* t2,
                                 IRExpr* t1, IRExpr* t0 )
{
   vassert(typeOfIRExpr(irsb->tyenv, t3) == Ity_I64);
   vassert(typeOfIRExpr(irsb->tyenv, t2) == Ity_I64);
   vassert(typeOfIRExpr(irsb->tyenv, t1) == Ity_I64);
   vassert(typeOfIRExpr(irsb->tyenv, t0) == Ity_I64);
   return binop(Iop_64HLtoV128,
                binop(Iop_32HLto64,
                      mkQNarrow64Sto32( t3 ),
                      mkQNarrow64Sto32( t2 )),
                binop(Iop_32HLto64,
                      mkQNarrow64Sto32( t1 ),
                      mkQNarrow64Sto32( t0 )));
}

/* Unsigned saturate narrow 64->32, combining to V128 */
static IRExpr* mkV128from4x64U ( IRExpr* t3, IRExpr* t2,
                                 IRExpr* t1, IRExpr* t0 )
{
   vassert(typeOfIRExpr(irsb->tyenv, t3) == Ity_I64);
   vassert(typeOfIRExpr(irsb->tyenv, t2) == Ity_I64);
   vassert(typeOfIRExpr(irsb->tyenv, t1) == Ity_I64);
   vassert(typeOfIRExpr(irsb->tyenv, t0) == Ity_I64);
   return binop(Iop_64HLtoV128,
                binop(Iop_32HLto64,
                      mkQNarrow64Uto32( t3 ),
                      mkQNarrow64Uto32( t2 )),
                binop(Iop_32HLto64,
                      mkQNarrow64Uto32( t1 ),
                      mkQNarrow64Uto32( t0 )));
}

/* Simulate irops Iop_MullOdd*, since we don't have them  */
#define MK_Iop_MullOdd8Ux16( expr_vA, expr_vB ) \
      binop(Iop_MullEven8Ux16, \
            binop(Iop_ShrV128, expr_vA, mkU8(8)), \
            binop(Iop_ShrV128, expr_vB, mkU8(8)))

#define MK_Iop_MullOdd8Sx16( expr_vA, expr_vB ) \
      binop(Iop_MullEven8Sx16, \
            binop(Iop_ShrV128, expr_vA, mkU8(8)), \
            binop(Iop_ShrV128, expr_vB, mkU8(8)))

#define MK_Iop_MullOdd16Ux8( expr_vA, expr_vB ) \
      binop(Iop_MullEven16Ux8, \
            binop(Iop_ShrV128, expr_vA, mkU8(16)), \
            binop(Iop_ShrV128, expr_vB, mkU8(16)))

#define MK_Iop_MullOdd32Ux4( expr_vA, expr_vB ) \
      binop(Iop_MullEven32Ux4, \
            binop(Iop_ShrV128, expr_vA, mkU8(32)), \
            binop(Iop_ShrV128, expr_vB, mkU8(32)))

#define MK_Iop_MullOdd16Sx8( expr_vA, expr_vB ) \
      binop(Iop_MullEven16Sx8, \
            binop(Iop_ShrV128, expr_vA, mkU8(16)), \
            binop(Iop_ShrV128, expr_vB, mkU8(16)))

#define MK_Iop_MullOdd32Sx4( expr_vA, expr_vB ) \
      binop(Iop_MullEven32Sx4, \
            binop(Iop_ShrV128, expr_vA, mkU8(32)), \
            binop(Iop_ShrV128, expr_vB, mkU8(32)))


static IRExpr* /* :: Ity_I64 */ mk64lo32Sto64 ( IRExpr* src )
{
   vassert(typeOfIRExpr(irsb->tyenv, src) == Ity_I64);
   return unop(Iop_32Sto64, unop(Iop_64to32, src));
}

static IRExpr* /* :: Ity_I64 */ mk64lo32Uto64 ( IRExpr* src )
{
   vassert(typeOfIRExpr(irsb->tyenv, src) == Ity_I64);
   return unop(Iop_32Uto64, unop(Iop_64to32, src));
}

static IROp mkSzOp ( IRType ty, IROp op8 )
{
   Int adj;
   vassert(ty == Ity_I8  || ty == Ity_I16 ||
           ty == Ity_I32 || ty == Ity_I64);
   vassert(op8 == Iop_Add8   || op8 == Iop_Sub8   || op8 == Iop_Mul8 ||
           op8 == Iop_Or8    || op8 == Iop_And8   || op8 == Iop_Xor8 ||
           op8 == Iop_Shl8   || op8 == Iop_Shr8   || op8 == Iop_Sar8 ||
           op8 == Iop_CmpEQ8 || op8 == Iop_CmpNE8 ||
           op8 == Iop_Not8 );
   adj = ty==Ity_I8 ? 0 : (ty==Ity_I16 ? 1 : (ty==Ity_I32 ? 2 : 3));
   return adj + op8;
}

/* Make sure we get valid 32 and 64bit addresses */
static Addr64 mkSzAddr ( IRType ty, Addr64 addr )
{
   vassert(ty == Ity_I32 || ty == Ity_I64);
   return ( ty == Ity_I64 ?
            (Addr64)addr :
            (Addr64)extend_s_32to64( toUInt(addr) ) );
}

/* sz, ULong -> IRExpr */
static IRExpr* mkSzImm ( IRType ty, ULong imm64 )
{
   vassert(ty == Ity_I32 || ty == Ity_I64);
   return ty == Ity_I64 ? mkU64(imm64) : mkU32((UInt)imm64);
}

/* sz, ULong -> IRConst */
static IRConst* mkSzConst ( IRType ty, ULong imm64 )
{
   vassert(ty == Ity_I32 || ty == Ity_I64);
   return ( ty == Ity_I64 ?
            IRConst_U64(imm64) :
            IRConst_U32((UInt)imm64) );
}

/* Sign extend imm16 -> IRExpr* */
static IRExpr* mkSzExtendS16 ( IRType ty, UInt imm16 )
{
   vassert(ty == Ity_I32 || ty == Ity_I64);
   return ( ty == Ity_I64 ?
            mkU64(extend_s_16to64(imm16)) :
            mkU32(extend_s_16to32(imm16)) );
}

/* Sign extend imm32 -> IRExpr* */
static IRExpr* mkSzExtendS32 ( IRType ty, UInt imm32 )
{
   vassert(ty == Ity_I32 || ty == Ity_I64);
   return ( ty == Ity_I64 ?
            mkU64(extend_s_32to64(imm32)) :
            mkU32(imm32) );
}

/* IR narrows I32/I64 -> I8/I16/I32 */
static IRExpr* mkNarrowTo8 ( IRType ty, IRExpr* src )
{
   vassert(ty == Ity_I32 || ty == Ity_I64);
   return ty == Ity_I64 ? unop(Iop_64to8, src) : unop(Iop_32to8, src);
}

static IRExpr* mkNarrowTo16 ( IRType ty, IRExpr* src )
{
   vassert(ty == Ity_I32 || ty == Ity_I64);
   return ty == Ity_I64 ? unop(Iop_64to16, src) : unop(Iop_32to16, src);
}

static IRExpr* mkNarrowTo32 ( IRType ty, IRExpr* src )
{
   vassert(ty == Ity_I32 || ty == Ity_I64);
   return ty == Ity_I64 ? unop(Iop_64to32, src) : src;
}

/* Signed/Unsigned IR widens I8/I16/I32 -> I32/I64 */
static IRExpr* mkWidenFrom8 ( IRType ty, IRExpr* src, Bool sined )
{
   IROp op;
   vassert(ty == Ity_I32 || ty == Ity_I64);
   if (sined) op = (ty==Ity_I32) ? Iop_8Sto32 : Iop_8Sto64;
   else       op = (ty==Ity_I32) ? Iop_8Uto32 : Iop_8Uto64;
   return unop(op, src);
}

static IRExpr* mkWidenFrom16 ( IRType ty, IRExpr* src, Bool sined )
{
   IROp op;
   vassert(ty == Ity_I32 || ty == Ity_I64);
   if (sined) op = (ty==Ity_I32) ? Iop_16Sto32 : Iop_16Sto64;
   else       op = (ty==Ity_I32) ? Iop_16Uto32 : Iop_16Uto64;
   return unop(op, src);
}

static IRExpr* mkWidenFrom32 ( IRType ty, IRExpr* src, Bool sined )
{
   vassert(ty == Ity_I32 || ty == Ity_I64);
   if (ty == Ity_I32)
      return src;
   return (sined) ? unop(Iop_32Sto64, src) : unop(Iop_32Uto64, src);
}


static Int integerGuestRegOffset ( UInt archreg )
{
   vassert(archreg < 32);
   
   // jrs: probably not necessary; only matters if we reference sub-parts
   // of the ppc registers, but that isn't the case
   // later: this might affect Altivec though?

   switch (archreg) {
   case  0: return offsetofPPCGuestState(guest_GPR0);
   case  1: return offsetofPPCGuestState(guest_GPR1);
   case  2: return offsetofPPCGuestState(guest_GPR2);
   case  3: return offsetofPPCGuestState(guest_GPR3);
   case  4: return offsetofPPCGuestState(guest_GPR4);
   case  5: return offsetofPPCGuestState(guest_GPR5);
   case  6: return offsetofPPCGuestState(guest_GPR6);
   case  7: return offsetofPPCGuestState(guest_GPR7);
   case  8: return offsetofPPCGuestState(guest_GPR8);
   case  9: return offsetofPPCGuestState(guest_GPR9);
   case 10: return offsetofPPCGuestState(guest_GPR10);
   case 11: return offsetofPPCGuestState(guest_GPR11);
   case 12: return offsetofPPCGuestState(guest_GPR12);
   case 13: return offsetofPPCGuestState(guest_GPR13);
   case 14: return offsetofPPCGuestState(guest_GPR14);
   case 15: return offsetofPPCGuestState(guest_GPR15);
   case 16: return offsetofPPCGuestState(guest_GPR16);
   case 17: return offsetofPPCGuestState(guest_GPR17);
   case 18: return offsetofPPCGuestState(guest_GPR18);
   case 19: return offsetofPPCGuestState(guest_GPR19);
   case 20: return offsetofPPCGuestState(guest_GPR20);
   case 21: return offsetofPPCGuestState(guest_GPR21);
   case 22: return offsetofPPCGuestState(guest_GPR22);
   case 23: return offsetofPPCGuestState(guest_GPR23);
   case 24: return offsetofPPCGuestState(guest_GPR24);
   case 25: return offsetofPPCGuestState(guest_GPR25);
   case 26: return offsetofPPCGuestState(guest_GPR26);
   case 27: return offsetofPPCGuestState(guest_GPR27);
   case 28: return offsetofPPCGuestState(guest_GPR28);
   case 29: return offsetofPPCGuestState(guest_GPR29);
   case 30: return offsetofPPCGuestState(guest_GPR30);
   case 31: return offsetofPPCGuestState(guest_GPR31);
   default: break;
   }
   vpanic("integerGuestRegOffset(ppc,be)"); /*notreached*/
}

static IRExpr* getIReg ( UInt archreg )
{
   IRType ty = mode64 ? Ity_I64 : Ity_I32;
   vassert(archreg < 32);
   return IRExpr_Get( integerGuestRegOffset(archreg), ty );
}

/* Ditto, but write to a reg instead. */
static void putIReg ( UInt archreg, IRExpr* e )
{
   IRType ty = mode64 ? Ity_I64 : Ity_I32;
   vassert(archreg < 32);
   vassert(typeOfIRExpr(irsb->tyenv, e) == ty );
   stmt( IRStmt_Put(integerGuestRegOffset(archreg), e) );
}


/* Floating point egisters are mapped to VSX registers[0..31]. */
static Int floatGuestRegOffset ( UInt archreg )
{
   vassert(archreg < 32);
   
   if (host_endness == VexEndnessLE) {
      switch (archreg) {
         case  0: return offsetofPPCGuestState(guest_VSR0) + 8;
         case  1: return offsetofPPCGuestState(guest_VSR1) + 8;
         case  2: return offsetofPPCGuestState(guest_VSR2) + 8;
         case  3: return offsetofPPCGuestState(guest_VSR3) + 8;
         case  4: return offsetofPPCGuestState(guest_VSR4) + 8;
         case  5: return offsetofPPCGuestState(guest_VSR5) + 8;
         case  6: return offsetofPPCGuestState(guest_VSR6) + 8;
         case  7: return offsetofPPCGuestState(guest_VSR7) + 8;
         case  8: return offsetofPPCGuestState(guest_VSR8) + 8;
         case  9: return offsetofPPCGuestState(guest_VSR9) + 8;
         case 10: return offsetofPPCGuestState(guest_VSR10) + 8;
         case 11: return offsetofPPCGuestState(guest_VSR11) + 8;
         case 12: return offsetofPPCGuestState(guest_VSR12) + 8;
         case 13: return offsetofPPCGuestState(guest_VSR13) + 8;
         case 14: return offsetofPPCGuestState(guest_VSR14) + 8;
         case 15: return offsetofPPCGuestState(guest_VSR15) + 8;
         case 16: return offsetofPPCGuestState(guest_VSR16) + 8;
         case 17: return offsetofPPCGuestState(guest_VSR17) + 8;
         case 18: return offsetofPPCGuestState(guest_VSR18) + 8;
         case 19: return offsetofPPCGuestState(guest_VSR19) + 8;
         case 20: return offsetofPPCGuestState(guest_VSR20) + 8;
         case 21: return offsetofPPCGuestState(guest_VSR21) + 8;
         case 22: return offsetofPPCGuestState(guest_VSR22) + 8;
         case 23: return offsetofPPCGuestState(guest_VSR23) + 8;
         case 24: return offsetofPPCGuestState(guest_VSR24) + 8;
         case 25: return offsetofPPCGuestState(guest_VSR25) + 8;
         case 26: return offsetofPPCGuestState(guest_VSR26) + 8;
         case 27: return offsetofPPCGuestState(guest_VSR27) + 8;
         case 28: return offsetofPPCGuestState(guest_VSR28) + 8;
         case 29: return offsetofPPCGuestState(guest_VSR29) + 8;
         case 30: return offsetofPPCGuestState(guest_VSR30) + 8;
         case 31: return offsetofPPCGuestState(guest_VSR31) + 8;
         default: break;
      }
   } else {
      switch (archreg) {
         case  0: return offsetofPPCGuestState(guest_VSR0);
         case  1: return offsetofPPCGuestState(guest_VSR1);
         case  2: return offsetofPPCGuestState(guest_VSR2);
         case  3: return offsetofPPCGuestState(guest_VSR3);
         case  4: return offsetofPPCGuestState(guest_VSR4);
         case  5: return offsetofPPCGuestState(guest_VSR5);
         case  6: return offsetofPPCGuestState(guest_VSR6);
         case  7: return offsetofPPCGuestState(guest_VSR7);
         case  8: return offsetofPPCGuestState(guest_VSR8);
         case  9: return offsetofPPCGuestState(guest_VSR9);
         case 10: return offsetofPPCGuestState(guest_VSR10);
         case 11: return offsetofPPCGuestState(guest_VSR11);
         case 12: return offsetofPPCGuestState(guest_VSR12);
         case 13: return offsetofPPCGuestState(guest_VSR13);
         case 14: return offsetofPPCGuestState(guest_VSR14);
         case 15: return offsetofPPCGuestState(guest_VSR15);
         case 16: return offsetofPPCGuestState(guest_VSR16);
         case 17: return offsetofPPCGuestState(guest_VSR17);
         case 18: return offsetofPPCGuestState(guest_VSR18);
         case 19: return offsetofPPCGuestState(guest_VSR19);
         case 20: return offsetofPPCGuestState(guest_VSR20);
         case 21: return offsetofPPCGuestState(guest_VSR21);
         case 22: return offsetofPPCGuestState(guest_VSR22);
         case 23: return offsetofPPCGuestState(guest_VSR23);
         case 24: return offsetofPPCGuestState(guest_VSR24);
         case 25: return offsetofPPCGuestState(guest_VSR25);
         case 26: return offsetofPPCGuestState(guest_VSR26);
         case 27: return offsetofPPCGuestState(guest_VSR27);
         case 28: return offsetofPPCGuestState(guest_VSR28);
         case 29: return offsetofPPCGuestState(guest_VSR29);
         case 30: return offsetofPPCGuestState(guest_VSR30);
         case 31: return offsetofPPCGuestState(guest_VSR31);
         default: break;
      }
   }
   vpanic("floatGuestRegOffset(ppc)"); /*notreached*/
}

static IRExpr* getFReg ( UInt archreg )
{
   vassert(archreg < 32);
   return IRExpr_Get( floatGuestRegOffset(archreg), Ity_F64 );
}

/* Ditto, but write to a reg instead. */
static void putFReg ( UInt archreg, IRExpr* e )
{
   vassert(archreg < 32);
   vassert(typeOfIRExpr(irsb->tyenv, e) == Ity_F64);
   stmt( IRStmt_Put(floatGuestRegOffset(archreg), e) );
}

/* get Decimal float value.  Note, they share floating point register file. */
static IRExpr* getDReg(UInt archreg) {
   IRExpr *e;
   vassert( archreg < 32 );
   e = IRExpr_Get( floatGuestRegOffset( archreg ), Ity_D64 );
   return e;
}
static IRExpr* getDReg32(UInt archreg) {
   IRExpr *e;
   vassert( archreg < 32 );
   e = IRExpr_Get( floatGuestRegOffset( archreg ), Ity_D32 );
   return e;
}

/* Read a floating point register pair and combine their contents into a
 128-bit value */
static IRExpr *getDReg_pair(UInt archreg) {
   IRExpr *high = getDReg( archreg );
   IRExpr *low = getDReg( archreg + 1 );

   return binop( Iop_D64HLtoD128, high, low );
}

/* Ditto, but write to a reg instead. */
static void putDReg32(UInt archreg, IRExpr* e) {
   vassert( archreg < 32 );
   vassert( typeOfIRExpr(irsb->tyenv, e) == Ity_D32 );
   stmt( IRStmt_Put( floatGuestRegOffset( archreg ), e ) );
}

static void putDReg(UInt archreg, IRExpr* e) {
   vassert( archreg < 32 );
   vassert( typeOfIRExpr(irsb->tyenv, e) == Ity_D64 );
   stmt( IRStmt_Put( floatGuestRegOffset( archreg ), e ) );
}

/* Write a 128-bit floating point value into a register pair. */
static void putDReg_pair(UInt archreg, IRExpr *e) {
   IRTemp low = newTemp( Ity_D64 );
   IRTemp high = newTemp( Ity_D64 );

   vassert( archreg < 32 );
   vassert( typeOfIRExpr(irsb->tyenv, e) == Ity_D128 );

   assign( low, unop( Iop_D128LOtoD64, e ) );
   assign( high, unop( Iop_D128HItoD64, e ) );

   stmt( IRStmt_Put( floatGuestRegOffset( archreg ), mkexpr( high ) ) );
   stmt( IRStmt_Put( floatGuestRegOffset( archreg + 1 ), mkexpr( low ) ) );
}

static Int vsxGuestRegOffset ( UInt archreg )
{
   vassert(archreg < 64);
   switch (archreg) {
   case  0: return offsetofPPCGuestState(guest_VSR0);
   case  1: return offsetofPPCGuestState(guest_VSR1);
   case  2: return offsetofPPCGuestState(guest_VSR2);
   case  3: return offsetofPPCGuestState(guest_VSR3);
   case  4: return offsetofPPCGuestState(guest_VSR4);
   case  5: return offsetofPPCGuestState(guest_VSR5);
   case  6: return offsetofPPCGuestState(guest_VSR6);
   case  7: return offsetofPPCGuestState(guest_VSR7);
   case  8: return offsetofPPCGuestState(guest_VSR8);
   case  9: return offsetofPPCGuestState(guest_VSR9);
   case 10: return offsetofPPCGuestState(guest_VSR10);
   case 11: return offsetofPPCGuestState(guest_VSR11);
   case 12: return offsetofPPCGuestState(guest_VSR12);
   case 13: return offsetofPPCGuestState(guest_VSR13);
   case 14: return offsetofPPCGuestState(guest_VSR14);
   case 15: return offsetofPPCGuestState(guest_VSR15);
   case 16: return offsetofPPCGuestState(guest_VSR16);
   case 17: return offsetofPPCGuestState(guest_VSR17);
   case 18: return offsetofPPCGuestState(guest_VSR18);
   case 19: return offsetofPPCGuestState(guest_VSR19);
   case 20: return offsetofPPCGuestState(guest_VSR20);
   case 21: return offsetofPPCGuestState(guest_VSR21);
   case 22: return offsetofPPCGuestState(guest_VSR22);
   case 23: return offsetofPPCGuestState(guest_VSR23);
   case 24: return offsetofPPCGuestState(guest_VSR24);
   case 25: return offsetofPPCGuestState(guest_VSR25);
   case 26: return offsetofPPCGuestState(guest_VSR26);
   case 27: return offsetofPPCGuestState(guest_VSR27);
   case 28: return offsetofPPCGuestState(guest_VSR28);
   case 29: return offsetofPPCGuestState(guest_VSR29);
   case 30: return offsetofPPCGuestState(guest_VSR30);
   case 31: return offsetofPPCGuestState(guest_VSR31);
   case 32: return offsetofPPCGuestState(guest_VSR32);
   case 33: return offsetofPPCGuestState(guest_VSR33);
   case 34: return offsetofPPCGuestState(guest_VSR34);
   case 35: return offsetofPPCGuestState(guest_VSR35);
   case 36: return offsetofPPCGuestState(guest_VSR36);
   case 37: return offsetofPPCGuestState(guest_VSR37);
   case 38: return offsetofPPCGuestState(guest_VSR38);
   case 39: return offsetofPPCGuestState(guest_VSR39);
   case 40: return offsetofPPCGuestState(guest_VSR40);
   case 41: return offsetofPPCGuestState(guest_VSR41);
   case 42: return offsetofPPCGuestState(guest_VSR42);
   case 43: return offsetofPPCGuestState(guest_VSR43);
   case 44: return offsetofPPCGuestState(guest_VSR44);
   case 45: return offsetofPPCGuestState(guest_VSR45);
   case 46: return offsetofPPCGuestState(guest_VSR46);
   case 47: return offsetofPPCGuestState(guest_VSR47);
   case 48: return offsetofPPCGuestState(guest_VSR48);
   case 49: return offsetofPPCGuestState(guest_VSR49);
   case 50: return offsetofPPCGuestState(guest_VSR50);
   case 51: return offsetofPPCGuestState(guest_VSR51);
   case 52: return offsetofPPCGuestState(guest_VSR52);
   case 53: return offsetofPPCGuestState(guest_VSR53);
   case 54: return offsetofPPCGuestState(guest_VSR54);
   case 55: return offsetofPPCGuestState(guest_VSR55);
   case 56: return offsetofPPCGuestState(guest_VSR56);
   case 57: return offsetofPPCGuestState(guest_VSR57);
   case 58: return offsetofPPCGuestState(guest_VSR58);
   case 59: return offsetofPPCGuestState(guest_VSR59);
   case 60: return offsetofPPCGuestState(guest_VSR60);
   case 61: return offsetofPPCGuestState(guest_VSR61);
   case 62: return offsetofPPCGuestState(guest_VSR62);
   case 63: return offsetofPPCGuestState(guest_VSR63);
   default: break;
   }
   vpanic("vsxGuestRegOffset(ppc)"); /*notreached*/
}

/* Vector registers are mapped to VSX registers[32..63]. */
static Int vectorGuestRegOffset ( UInt archreg )
{
   vassert(archreg < 32);
   
   switch (archreg) {
   case  0: return offsetofPPCGuestState(guest_VSR32);
   case  1: return offsetofPPCGuestState(guest_VSR33);
   case  2: return offsetofPPCGuestState(guest_VSR34);
   case  3: return offsetofPPCGuestState(guest_VSR35);
   case  4: return offsetofPPCGuestState(guest_VSR36);
   case  5: return offsetofPPCGuestState(guest_VSR37);
   case  6: return offsetofPPCGuestState(guest_VSR38);
   case  7: return offsetofPPCGuestState(guest_VSR39);
   case  8: return offsetofPPCGuestState(guest_VSR40);
   case  9: return offsetofPPCGuestState(guest_VSR41);
   case 10: return offsetofPPCGuestState(guest_VSR42);
   case 11: return offsetofPPCGuestState(guest_VSR43);
   case 12: return offsetofPPCGuestState(guest_VSR44);
   case 13: return offsetofPPCGuestState(guest_VSR45);
   case 14: return offsetofPPCGuestState(guest_VSR46);
   case 15: return offsetofPPCGuestState(guest_VSR47);
   case 16: return offsetofPPCGuestState(guest_VSR48);
   case 17: return offsetofPPCGuestState(guest_VSR49);
   case 18: return offsetofPPCGuestState(guest_VSR50);
   case 19: return offsetofPPCGuestState(guest_VSR51);
   case 20: return offsetofPPCGuestState(guest_VSR52);
   case 21: return offsetofPPCGuestState(guest_VSR53);
   case 22: return offsetofPPCGuestState(guest_VSR54);
   case 23: return offsetofPPCGuestState(guest_VSR55);
   case 24: return offsetofPPCGuestState(guest_VSR56);
   case 25: return offsetofPPCGuestState(guest_VSR57);
   case 26: return offsetofPPCGuestState(guest_VSR58);
   case 27: return offsetofPPCGuestState(guest_VSR59);
   case 28: return offsetofPPCGuestState(guest_VSR60);
   case 29: return offsetofPPCGuestState(guest_VSR61);
   case 30: return offsetofPPCGuestState(guest_VSR62);
   case 31: return offsetofPPCGuestState(guest_VSR63);
   default: break;
   }
   vpanic("vextorGuestRegOffset(ppc)"); /*notreached*/
}

static IRExpr* getVReg ( UInt archreg )
{
   vassert(archreg < 32);
   return IRExpr_Get( vectorGuestRegOffset(archreg), Ity_V128 );
}

/* Get contents of 128-bit reg guest register */
static IRExpr* getF128Reg ( UInt archreg )
{
   vassert(archreg < 64);
   return IRExpr_Get( vectorGuestRegOffset(archreg), Ity_F128 );
}

/* Ditto, but write to a reg instead. */
static void putF128Reg ( UInt archreg, IRExpr* e )
{
   vassert(archreg < 64);
   vassert(typeOfIRExpr(irsb->tyenv, e) == Ity_F128);
   stmt( IRStmt_Put(vectorGuestRegOffset(archreg), e) );
}

/* Ditto, but write to a reg instead. */
static void putVReg ( UInt archreg, IRExpr* e )
{
   vassert(archreg < 32);
   vassert(typeOfIRExpr(irsb->tyenv, e) == Ity_V128);
   stmt( IRStmt_Put(vectorGuestRegOffset(archreg), e) );
}

/* Get contents of VSX guest register */
static IRExpr* getVSReg ( UInt archreg )
{
   vassert(archreg < 64);
   return IRExpr_Get( vsxGuestRegOffset(archreg), Ity_V128 );
}

/* Ditto, but write to a VSX reg instead. */
static void putVSReg ( UInt archreg, IRExpr* e )
{
   vassert(archreg < 64);
   vassert(typeOfIRExpr(irsb->tyenv, e) == Ity_V128);
   stmt( IRStmt_Put(vsxGuestRegOffset(archreg), e) );
}


static Int guestCR321offset ( UInt cr )
{
   switch (cr) {
   case 0: return offsetofPPCGuestState(guest_CR0_321 );
   case 1: return offsetofPPCGuestState(guest_CR1_321 );
   case 2: return offsetofPPCGuestState(guest_CR2_321 );
   case 3: return offsetofPPCGuestState(guest_CR3_321 );
   case 4: return offsetofPPCGuestState(guest_CR4_321 );
   case 5: return offsetofPPCGuestState(guest_CR5_321 );
   case 6: return offsetofPPCGuestState(guest_CR6_321 );
   case 7: return offsetofPPCGuestState(guest_CR7_321 );
   default: vpanic("guestCR321offset(ppc)");
   }
} 

static Int guestCR0offset ( UInt cr )
{
   switch (cr) {
   case 0: return offsetofPPCGuestState(guest_CR0_0 );
   case 1: return offsetofPPCGuestState(guest_CR1_0 );
   case 2: return offsetofPPCGuestState(guest_CR2_0 );
   case 3: return offsetofPPCGuestState(guest_CR3_0 );
   case 4: return offsetofPPCGuestState(guest_CR4_0 );
   case 5: return offsetofPPCGuestState(guest_CR5_0 );
   case 6: return offsetofPPCGuestState(guest_CR6_0 );
   case 7: return offsetofPPCGuestState(guest_CR7_0 );
   default: vpanic("guestCR3offset(ppc)");
   }
}

typedef enum {
   _placeholder0,
   _placeholder1,
   _placeholder2,
   BYTE,
   HWORD,
   WORD,
   DWORD
} _popcount_data_type;

/*-----------------------------------------------------------*/
/*---  IR popcount helpers                                ---*/
/*-----------------------------------------------------------*/
/* Generate an IR sequence to do a popcount operation on the supplied
   IRTemp, and return a new IRTemp holding the result.  'ty' may be
   Ity_I32 or Ity_I64 only. */
static IRTemp gen_POPCOUNT ( IRType ty, IRTemp src,
                             _popcount_data_type data_type )
{
  /* Do count across 2^data_type bits,
     byte:        data_type = 3
     half word:   data_type = 4
     word:        data_type = 5
     double word: data_type = 6  (not supported for 32-bit type)
    */
   Int shift[6];
   _popcount_data_type idx, i;
   IRTemp mask[6];
   IRTemp old = IRTemp_INVALID;
   IRTemp nyu = IRTemp_INVALID;

   vassert(ty == Ity_I64 || ty == Ity_I32);

   // Use a single IROp in cases where we can.

   if (ty == Ity_I64 && data_type == DWORD) {
      IRTemp res = newTemp(Ity_I64);
      assign(res, unop(Iop_PopCount64, mkexpr(src)));
      return res;
   }

   if (ty == Ity_I32 && data_type == WORD) {
      IRTemp res = newTemp(Ity_I32);
      assign(res, unop(Iop_PopCount32, mkexpr(src)));
      return res;
   }

   // For the rest, we have to do it the slow way.

   if (ty == Ity_I32) {

      for (idx = 0; idx < WORD; idx++) {
         mask[idx]  = newTemp(ty);
         shift[idx] = 1 << idx;
      }
      assign(mask[0], mkU32(0x55555555));
      assign(mask[1], mkU32(0x33333333));
      assign(mask[2], mkU32(0x0F0F0F0F));
      assign(mask[3], mkU32(0x00FF00FF));
      assign(mask[4], mkU32(0x0000FFFF));
      old = src;
      for (i = 0; i < data_type; i++) {
         nyu = newTemp(ty);
         assign(nyu,
                binop(Iop_Add32,
                      binop(Iop_And32,
                            mkexpr(old),
                            mkexpr(mask[i])),
                      binop(Iop_And32,
                            binop(Iop_Shr32, mkexpr(old), mkU8(shift[i])),
                            mkexpr(mask[i]))));
         old = nyu;
      }
      return nyu;
   }

   // else, ty == Ity_I64
   vassert(mode64);

   for (i = 0; i < DWORD; i++) {
      mask[i] = newTemp( Ity_I64 );
      shift[i] = 1 << i;
   }
   assign( mask[0], mkU64( 0x5555555555555555ULL ) );
   assign( mask[1], mkU64( 0x3333333333333333ULL ) );
   assign( mask[2], mkU64( 0x0F0F0F0F0F0F0F0FULL ) );
   assign( mask[3], mkU64( 0x00FF00FF00FF00FFULL ) );
   assign( mask[4], mkU64( 0x0000FFFF0000FFFFULL ) );
   assign( mask[5], mkU64( 0x00000000FFFFFFFFULL ) );
   old = src;
   for (i = 0; i < data_type; i++) {
      nyu = newTemp( Ity_I64 );
      assign( nyu,
              binop( Iop_Add64,
                     binop( Iop_And64, mkexpr( old ), mkexpr( mask[i] ) ),
                     binop( Iop_And64,
                            binop( Iop_Shr64, mkexpr( old ), mkU8( shift[i] ) ),
                            mkexpr( mask[i] ) ) ) );
      old = nyu;
   }
   return nyu;
}

/* Special purpose population count function for
 * vpopcntd in 32-bit mode.
 */
static IRTemp gen_vpopcntd_mode32 ( IRTemp src1, IRTemp src2 )
{
   IRTemp retval = newTemp(Ity_I64);

   vassert(!mode64);

   assign(retval,
          unop(Iop_32Uto64,
               binop(Iop_Add32,
                     unop(Iop_PopCount32, mkexpr(src1)),
                     unop(Iop_PopCount32, mkexpr(src2)))));
   return retval;
}


// ROTL(src32/64, rot_amt5/6)
static IRExpr* /* :: Ity_I32/64 */ ROTL ( IRExpr* src,
                                          IRExpr* rot_amt )
{
   IRExpr *mask, *rot;
   vassert(typeOfIRExpr(irsb->tyenv,rot_amt) == Ity_I8);

   if (typeOfIRExpr(irsb->tyenv,src) == Ity_I64) {
      // rot = (src << rot_amt) | (src >> (64-rot_amt))
      mask = binop(Iop_And8, rot_amt, mkU8(63));
      rot  = binop(Iop_Or64,
                binop(Iop_Shl64, src, mask),
                binop(Iop_Shr64, src, binop(Iop_Sub8, mkU8(64), mask)));
   } else {
      // rot = (src << rot_amt) | (src >> (32-rot_amt))
      mask = binop(Iop_And8, rot_amt, mkU8(31));
      rot  = binop(Iop_Or32,
                binop(Iop_Shl32, src, mask),
                binop(Iop_Shr32, src, binop(Iop_Sub8, mkU8(32), mask)));
   }
   /* Note: the ITE not merely an optimisation; it's needed
      because otherwise the Shr is a shift by the word size when
      mask denotes zero.  For rotates by immediates, a lot of
      this junk gets folded out. */
   return IRExpr_ITE( binop(Iop_CmpNE8, mask, mkU8(0)),
                      /* non-zero rotate */ rot,
                      /*     zero rotate */ src);
}

/* Standard effective address calc: (rA + rB) */
static IRExpr* ea_rA_idxd ( UInt rA, UInt rB )
{
   IRType ty = mode64 ? Ity_I64 : Ity_I32;
   vassert(rA < 32);
   vassert(rB < 32);
   return binop(mkSzOp(ty, Iop_Add8), getIReg(rA), getIReg(rB));
}

/* Standard effective address calc: (rA + simm) */
static IRExpr* ea_rA_simm ( UInt rA, UInt simm16 )
{
   IRType ty = mode64 ? Ity_I64 : Ity_I32;
   vassert(rA < 32);
   return binop(mkSzOp(ty, Iop_Add8), getIReg(rA),
                mkSzExtendS16(ty, simm16));
}

/* Standard effective address calc: (rA|0) */
static IRExpr* ea_rAor0 ( UInt rA )
{
   IRType ty = mode64 ? Ity_I64 : Ity_I32;
   vassert(rA < 32);
   if (rA == 0) {
      return mkSzImm(ty, 0);
   } else {
      return getIReg(rA);
   }
}

/* Standard effective address calc: (rA|0) + rB */
static IRExpr* ea_rAor0_idxd ( UInt rA, UInt rB )
{
   vassert(rA < 32);
   vassert(rB < 32);
   return (rA == 0) ? getIReg(rB) : ea_rA_idxd( rA, rB );
}

/* Standard effective address calc: (rA|0) + simm16 */
static IRExpr* ea_rAor0_simm ( UInt rA, UInt simm16 )
{
   IRType ty = mode64 ? Ity_I64 : Ity_I32;
   vassert(rA < 32);
   if (rA == 0) {
      return mkSzExtendS16(ty, simm16);
   } else {
      return ea_rA_simm( rA, simm16 );
   }
}


/* Align effective address */
static IRExpr* addr_align( IRExpr* addr, UChar align )
{
   IRType ty = mode64 ? Ity_I64 : Ity_I32;
   ULong mask;
   switch (align) {
   case 1:  return addr;                    // byte aligned
   case 2:  mask = ~0ULL << 1; break;       // half-word aligned
   case 4:  mask = ~0ULL << 2; break;       // word aligned
   case 16: mask = ~0ULL << 4; break;       // quad-word aligned
   default:
      vex_printf("addr_align: align = %u\n", align);
      vpanic("addr_align(ppc)");
   }

   vassert(typeOfIRExpr(irsb->tyenv,addr) == ty);
   return binop( mkSzOp(ty, Iop_And8), addr, mkSzImm(ty, mask) );
}


/* Exit the trace if ADDR (intended to be a guest memory address) is
   not ALIGN-aligned, generating a request for a SIGBUS followed by a
   restart of the current insn. */
static void gen_SIGBUS_if_misaligned ( IRTemp addr, UChar align )
{
   vassert(align == 2 || align == 4 || align == 8 || align == 16);
   if (mode64) {
      vassert(typeOfIRTemp(irsb->tyenv, addr) == Ity_I64);
      stmt(
         IRStmt_Exit(
            binop(Iop_CmpNE64,
                  binop(Iop_And64, mkexpr(addr), mkU64(align-1)),
                  mkU64(0)),
            Ijk_SigBUS,
            IRConst_U64( guest_CIA_curr_instr ), OFFB_CIA
         )
      );
   } else {
      vassert(typeOfIRTemp(irsb->tyenv, addr) == Ity_I32);
      stmt(
         IRStmt_Exit(
            binop(Iop_CmpNE32,
                  binop(Iop_And32, mkexpr(addr), mkU32(align-1)),
                  mkU32(0)),
            Ijk_SigBUS,
            IRConst_U32( guest_CIA_curr_instr ), OFFB_CIA
         )
      );
   }
}


/* Generate AbiHints which mark points at which the ELF or PowerOpen
   ABIs say that the stack red zone (viz, -N(r1) .. -1(r1), for some
   N) becomes undefined.  That is at function calls and returns.  ELF
   ppc32 doesn't have this "feature" (how fortunate for it).  nia is
   the address of the next instruction to be executed.
*/
static void make_redzone_AbiHint ( const VexAbiInfo* vbi, 
                                   IRTemp nia, const HChar* who )
{
   Int szB = vbi->guest_stack_redzone_size;
   if (0) vex_printf("AbiHint: %s\n", who);
   vassert(szB >= 0);
   if (szB > 0) {
      if (mode64) {
         vassert(typeOfIRTemp(irsb->tyenv, nia) == Ity_I64);
         stmt( IRStmt_AbiHint( 
                  binop(Iop_Sub64, getIReg(1), mkU64(szB)), 
                  szB,
                  mkexpr(nia)
         ));
      } else {
         vassert(typeOfIRTemp(irsb->tyenv, nia) == Ity_I32);
         stmt( IRStmt_AbiHint( 
                  binop(Iop_Sub32, getIReg(1), mkU32(szB)), 
                  szB,
                  mkexpr(nia)
         ));
      }
   }
}


/*------------------------------------------------------------*/
/*--- Helpers for condition codes.                         ---*/
/*------------------------------------------------------------*/

/* Condition register layout. 

   In the hardware, CR is laid out like this.  The leftmost end is the
   most significant bit in the register; however the IBM documentation
   numbers the bits backwards for some reason.

   CR0      CR1    ..........   CR6       CR7
   0 .. 3   .......................  28 .. 31    (IBM bit numbering)
   31  28                             3    0     (normal bit numbering)

   Each CR field is 4 bits:  [<,>,==,SO]

   Hence in IBM's notation, BI=0 is CR7[SO], BI=1 is CR7[==], etc.

   Indexing from BI to guest state:

     let    n = BI / 4
          off = BI % 4
     this references CR n:

        off==0   ->  guest_CRn_321 >> 3
        off==1   ->  guest_CRn_321 >> 2
        off==2   ->  guest_CRn_321 >> 1
        off==3   ->  guest_CRn_SO

   Bear in mind the only significant bit in guest_CRn_SO is bit 0
   (normal notation) and in guest_CRn_321 the significant bits are
   3, 2 and 1 (normal notation).
*/

static void putCR321 ( UInt cr, IRExpr* e )
{
   vassert(cr < 8);
   vassert(typeOfIRExpr(irsb->tyenv, e) == Ity_I8);
   stmt( IRStmt_Put(guestCR321offset(cr), e) );
}

static void putCR0 ( UInt cr, IRExpr* e )
{
   vassert(cr < 8);
   vassert(typeOfIRExpr(irsb->tyenv, e) == Ity_I8);
   stmt( IRStmt_Put(guestCR0offset(cr), e) );
}

static void putC ( IRExpr* e )
{
   /* The assumption is that the value of the Floating-Point Result
    * Class Descriptor bit (C) is passed in the lower four bits of a
    * 32 bit value.
    *
    * Note, the C and FPCC bits which are fields in the FPSCR
    * register are stored in their own memory location of
    * memory.  The FPCC bits are in the lower 4 bits.  The C bit needs
    * to be shifted to bit 4 in the memory location that holds C and FPCC.
    * Note not all of the FPSCR register bits are supported.  We are
    * only writing C bit.
    */
   IRExpr* tmp;

   vassert(typeOfIRExpr(irsb->tyenv, e) == Ity_I32);

   /* Get the FPCC bit field */
   tmp = binop( Iop_And32,
                mkU32( 0xF ),
                unop( Iop_8Uto32, IRExpr_Get( OFFB_C_FPCC, Ity_I8 ) ) );

   stmt( IRStmt_Put( OFFB_C_FPCC,
                     unop( Iop_32to8,
                           binop( Iop_Or32, tmp,
                                  binop( Iop_Shl32,
                                         binop( Iop_And32, mkU32( 0x1 ), e ),
                                         mkU8( 4 ) ) ) ) ) );
}

static IRExpr* /* :: Ity_I8 */ getCR0 ( UInt cr )
{
   vassert(cr < 8);
   return IRExpr_Get(guestCR0offset(cr), Ity_I8);
}

static IRExpr* /* :: Ity_I8 */ getCR321 ( UInt cr )
{
   vassert(cr < 8);
   return IRExpr_Get(guestCR321offset(cr), Ity_I8);
}

/* Fetch the specified CR bit (as per IBM/hardware notation) and
   return it at the bottom of an I32; the top 31 bits are guaranteed
   to be zero. */
static IRExpr* /* :: Ity_I32 */ getCRbit ( UInt bi )
{
   UInt n   = bi / 4;
   UInt off = bi % 4;
   vassert(bi < 32);
   if (off == 3) {
      /* Fetch the SO bit for this CR field */
      /* Note: And32 is redundant paranoia iff guest state only has 0
         or 1 in that slot. */
      return binop(Iop_And32, unop(Iop_8Uto32, getCR0(n)), mkU32(1));
   } else {
      /* Fetch the <, > or == bit for this CR field */
      return binop( Iop_And32, 
                    binop( Iop_Shr32, 
                           unop(Iop_8Uto32, getCR321(n)),
                           mkU8(toUChar(3-off)) ),
                    mkU32(1) );
   }
}

/* Dually, write the least significant bit of BIT to the specified CR
   bit.  Indexing as per getCRbit. */
static void putCRbit ( UInt bi, IRExpr* bit )
{
   UInt    n, off;
   IRExpr* safe;
   vassert(typeOfIRExpr(irsb->tyenv,bit) == Ity_I32);
   safe = binop(Iop_And32, bit, mkU32(1));
   n   = bi / 4;
   off = bi % 4;
   vassert(bi < 32);
   if (off == 3) {
      /* This is the SO bit for this CR field */
      putCR0(n, unop(Iop_32to8, safe));
   } else {
      off = 3 - off;
      vassert(off == 1 || off == 2 || off == 3);
      putCR321(
         n,
         unop( Iop_32to8,
               binop( Iop_Or32,
                      /* old value with field masked out */
                      binop(Iop_And32, unop(Iop_8Uto32, getCR321(n)),
                                       mkU32(~(1 << off))),
                      /* new value in the right place */
                      binop(Iop_Shl32, safe, mkU8(toUChar(off)))
               )
         )
      );
   }
}

/* Fetch the specified CR bit (as per IBM/hardware notation) and
   return it somewhere in an I32; it does not matter where, but
   whichever bit it is, all other bits are guaranteed to be zero.  In
   other words, the I32-typed expression will be zero if the bit is
   zero and nonzero if the bit is 1.  Write into *where the index
   of where the bit will be. */

static
IRExpr* /* :: Ity_I32 */ getCRbit_anywhere ( UInt bi, Int* where )
{
   UInt n   = bi / 4;
   UInt off = bi % 4;
   vassert(bi < 32);
   if (off == 3) {
      /* Fetch the SO bit for this CR field */
      /* Note: And32 is redundant paranoia iff guest state only has 0
         or 1 in that slot. */
      *where = 0;
      return binop(Iop_And32, unop(Iop_8Uto32, getCR0(n)), mkU32(1));
   } else {
      /* Fetch the <, > or == bit for this CR field */
      *where = 3-off;
      return binop( Iop_And32, 
                    unop(Iop_8Uto32, getCR321(n)),
                    mkU32(1 << (3-off)) );
   }
}

/* Set the CR0 flags following an arithmetic operation.
   (Condition Register CR0 Field Definition, PPC32 p60)
*/
static IRExpr* getXER_SO ( void );
static void set_CR0 ( IRExpr* result )
{
   vassert(typeOfIRExpr(irsb->tyenv,result) == Ity_I32 ||
           typeOfIRExpr(irsb->tyenv,result) == Ity_I64);
   if (mode64) {
      putCR321( 0, unop(Iop_64to8,
                        binop(Iop_CmpORD64S, result, mkU64(0))) );
   } else {
      putCR321( 0, unop(Iop_32to8,
                        binop(Iop_CmpORD32S, result, mkU32(0))) );
   }
   putCR0( 0, getXER_SO() );
}


/* Set the CR6 flags following an AltiVec compare operation.
 * NOTE: This also works for VSX single-precision compares.
 * */
static void set_AV_CR6 ( IRExpr* result, Bool test_all_ones )
{
   /* CR6[0:3] = {all_ones, 0, all_zeros, 0}
      32 bit: all_zeros =  (v[0] || v[1] || v[2] || v[3]) == 0x0000'0000
              all_ones  = ~(v[0] && v[1] && v[2] && v[3]) == 0x0000'0000
              where v[] denotes 32-bit lanes
      or
      64 bit: all_zeros =  (v[0] || v[1]) == 0x0000'0000'0000'0000
              all_ones  = ~(v[0] && v[1]) == 0x0000'0000'0000'0000
              where v[] denotes 64-bit lanes

      The 32- and 64-bit versions compute the same thing, but the 64-bit one
      tries to be a bit more efficient.
   */
   vassert(typeOfIRExpr(irsb->tyenv,result) == Ity_V128);

   IRTemp overlappedOred  = newTemp(Ity_V128);
   IRTemp overlappedAnded = newTemp(Ity_V128);

   if (mode64) {
      IRTemp v0 = newTemp(Ity_V128);
      IRTemp v1 = newTemp(Ity_V128);
      assign( v0, result );
      assign( v1, binop(Iop_ShrV128, result, mkU8(64)) );
      assign(overlappedOred,
             binop(Iop_OrV128, mkexpr(v0), mkexpr(v1)));
      assign(overlappedAnded,
             binop(Iop_AndV128, mkexpr(v0), mkexpr(v1)));
   } else {
      IRTemp v0 = newTemp(Ity_V128);
      IRTemp v1 = newTemp(Ity_V128);
      IRTemp v2 = newTemp(Ity_V128);
      IRTemp v3 = newTemp(Ity_V128);
      assign( v0, result );
      assign( v1, binop(Iop_ShrV128, result, mkU8(32)) );
      assign( v2, binop(Iop_ShrV128, result, mkU8(64)) );
      assign( v3, binop(Iop_ShrV128, result, mkU8(96)) );
      assign(overlappedOred,
             binop(Iop_OrV128,
                   binop(Iop_OrV128, mkexpr(v0), mkexpr(v1)),
                   binop(Iop_OrV128, mkexpr(v2), mkexpr(v3))));
      assign(overlappedAnded,
             binop(Iop_AndV128,
                   binop(Iop_AndV128, mkexpr(v0), mkexpr(v1)),
                   binop(Iop_AndV128, mkexpr(v2), mkexpr(v3))));
   }

   IRTemp rOnes   = newTemp(Ity_I8);
   IRTemp rZeroes = newTemp(Ity_I8);

   if (mode64) {
      assign(rZeroes,
             unop(Iop_1Uto8,
                  binop(Iop_CmpEQ64,
                        mkU64(0),
                        unop(Iop_V128to64, mkexpr(overlappedOred)))));
      assign(rOnes,
             unop(Iop_1Uto8,
                  binop(Iop_CmpEQ64,
                        mkU64(0),
                        unop(Iop_Not64,
                             unop(Iop_V128to64, mkexpr(overlappedAnded))))));
   } else {
      assign(rZeroes,
             unop(Iop_1Uto8,
                  binop(Iop_CmpEQ32,
                        mkU32(0),
                        unop(Iop_V128to32, mkexpr(overlappedOred)))));
      assign(rOnes,
             unop(Iop_1Uto8,
                  binop(Iop_CmpEQ32,
                        mkU32(0),
                        unop(Iop_Not32,
                             unop(Iop_V128to32, mkexpr(overlappedAnded))))));
   }

   // rOnes might not be used below.  But iropt will remove it, so there's no
   // inefficiency as a result.

   if (test_all_ones) {
      putCR321( 6, binop(Iop_Or8,
                         binop(Iop_Shl8, mkexpr(rOnes),  mkU8(3)),
                         binop(Iop_Shl8, mkexpr(rZeroes), mkU8(1))) );
   } else {
      putCR321( 6, binop(Iop_Shl8, mkexpr(rZeroes), mkU8(1)) );
   }
   putCR0( 6, mkU8(0) );
} 


static IRExpr * create_DCM ( IRType size, IRTemp NaN, IRTemp inf, IRTemp zero,
                             IRTemp dnorm, IRTemp pos)
{
   /* This is a general function for creating the DCM for a 32-bit or
      64-bit expression based on the passes size.
   */
   IRTemp neg;
   IROp opAND, opOR, opSHL, opXto1, op1UtoX;

   vassert( ( size == Ity_I32 ) || ( size == Ity_I64 ) );

   if ( size == Ity_I32 ) {
      opSHL = Iop_Shl32;
      opAND = Iop_And32;
      opOR  = Iop_Or32;
      opXto1 = Iop_32to1;
      op1UtoX = Iop_1Uto32;
      neg = newTemp( Ity_I32 );

   } else {
      opSHL = Iop_Shl64;
      opAND = Iop_And64;
      opOR  = Iop_Or64;
      opXto1 = Iop_64to1;
      op1UtoX = Iop_1Uto64;
      neg = newTemp( Ity_I64 );
   }

   assign( neg, unop( op1UtoX, mkNOT1( unop( opXto1,
                                             mkexpr ( pos ) ) ) ) );

   return binop( opOR,
                 binop( opSHL, mkexpr( NaN ), mkU8( 6 ) ),
                 binop( opOR,
                        binop( opOR,
                               binop( opOR,
                                      binop( opSHL,
                                             binop( opAND,
                                                    mkexpr( pos ),
                                                    mkexpr( inf ) ),
                                             mkU8( 5 ) ),
                                      binop( opSHL,
                                             binop( opAND,
                                                    mkexpr( neg ),
                                                    mkexpr( inf ) ),
                                             mkU8( 4 ) ) ),
                               binop( opOR,
                                      binop( opSHL,
                                             binop( opAND,
                                                    mkexpr( pos ),
                                                    mkexpr( zero ) ),
                                             mkU8( 3 ) ),
                                      binop( opSHL,
                                             binop( opAND,
                                                    mkexpr( neg ),
                                                    mkexpr( zero ) ),
                                             mkU8( 2 ) ) ) ),
                        binop( opOR,
                               binop( opSHL,
                                      binop( opAND,
                                             mkexpr( pos ),
                                             mkexpr( dnorm ) ),
                                      mkU8( 1 ) ),
                               binop( opAND,
                                      mkexpr( neg ),
                                      mkexpr( dnorm ) ) ) ) );
}

/*------------------------------------------------------------*/
/*--- Helpers for XER flags.                               ---*/
/*------------------------------------------------------------*/

static void putXER_SO ( IRExpr* e )
{
   IRExpr* so;
   vassert(typeOfIRExpr(irsb->tyenv, e) == Ity_I8);
   so = binop(Iop_And8, e, mkU8(1));
   stmt( IRStmt_Put( OFFB_XER_SO, so ) );
}

static void putXER_OV ( IRExpr* e )
{
   /* Interface to write XER[OV] */
   IRExpr* ov;
   vassert(typeOfIRExpr(irsb->tyenv, e) == Ity_I8);
   ov = binop(Iop_And8, e, mkU8(1));
   stmt( IRStmt_Put( OFFB_XER_OV, ov ) );
}

static void putXER_OV32 ( IRExpr* e )
{
   /*Interface to write XER[OV32] */
   IRExpr* ov;
   vassert(typeOfIRExpr(irsb->tyenv, e) == Ity_I8);
   ov = binop(Iop_And8, e, mkU8(1));

   /* The OV32 bit was added to XER in ISA 3.0.  Do not write unless we
    * ISA 3.0 or beyond is supported. */
   if( OV32_CA32_supported )
      stmt( IRStmt_Put( OFFB_XER_OV32, ov ) );
}

static void putXER_CA ( IRExpr* e )
{
   /* Interface to write XER[CA] */
   IRExpr* ca;
   vassert(typeOfIRExpr(irsb->tyenv, e) == Ity_I8);
   ca = binop(Iop_And8, e, mkU8(1));
   stmt( IRStmt_Put( OFFB_XER_CA, ca ) );
}

static void putXER_CA32 ( IRExpr* e )
{
   /* Interface to write XER[CA32] */
   IRExpr* ca;
   vassert(typeOfIRExpr(irsb->tyenv, e) == Ity_I8);
   ca = binop(Iop_And8, e, mkU8(1));

   /* The CA32 bit was added to XER in ISA 3.0.  Do not write unless we
    * ISA 3.0 or beyond is supported. */
   if( OV32_CA32_supported )
      stmt( IRStmt_Put( OFFB_XER_CA32, ca ) );
}

static void putXER_BC ( IRExpr* e )
{
   IRExpr* bc;
   vassert(typeOfIRExpr(irsb->tyenv, e) == Ity_I8);
   bc = binop(Iop_And8, e, mkU8(0x7F));
   stmt( IRStmt_Put( OFFB_XER_BC, bc ) );
}

static IRExpr* /* :: Ity_I8 */ getXER_SO ( void )
{
   return IRExpr_Get( OFFB_XER_SO, Ity_I8 );
}

static IRExpr* /* :: Ity_I32 */ getXER_SO_32 ( void )
{
   return binop( Iop_And32, unop(Iop_8Uto32, getXER_SO()), mkU32(1) );
}

static IRExpr* /* :: Ity_I8 */ getXER_OV ( void )
{
   return IRExpr_Get( OFFB_XER_OV, Ity_I8 );
}

static IRExpr* /* :: Ity_I8 */ getXER_OV32 ( void )
{
   return IRExpr_Get( OFFB_XER_OV32, Ity_I8 );
}

static IRExpr* /* :: Ity_I32 */ getXER_OV_32 ( void )
{
   /* get XER[OV], 32-bit interface */
   return binop( Iop_And32, unop(Iop_8Uto32, getXER_OV()), mkU32(1) );
}

static IRExpr* /* :: Ity_I32 */ getXER_OV32_32 ( void )
{
   /* get XER[OV32], 32-bit interface */
   return binop( Iop_And32, unop(Iop_8Uto32, getXER_OV32()), mkU32(1) );
}

static IRExpr* /* :: Ity_I32 */ getXER_CA_32 ( void )
{
   /* get XER[CA], 32-bit interface */
   IRExpr* ca = IRExpr_Get( OFFB_XER_CA, Ity_I8 );
   return binop( Iop_And32, unop(Iop_8Uto32, ca ), mkU32(1) );
}

static IRExpr* /* :: Ity_I32 */ getXER_CA32_32 ( void )
{
   /* get XER[CA32], 32-bit interface */
   IRExpr* ca = IRExpr_Get( OFFB_XER_CA32, Ity_I8 );
   return binop( Iop_And32, unop(Iop_8Uto32, ca ), mkU32(1) );
}

static IRExpr* /* :: Ity_I8 */ getXER_BC ( void )
{
   return IRExpr_Get( OFFB_XER_BC, Ity_I8 );
}

static IRExpr* /* :: Ity_I32 */ getXER_BC_32 ( void )
{
   IRExpr* bc = IRExpr_Get( OFFB_XER_BC, Ity_I8 );
   return binop( Iop_And32, unop(Iop_8Uto32, bc), mkU32(0x7F) );
}


/* RES is the result of doing OP on ARGL and ARGR.  Set %XER.OV and
   %XER.SO accordingly. */

static IRExpr* calculate_XER_OV_32( UInt op, IRExpr* res,
                                    IRExpr* argL, IRExpr* argR )
{
   IRTemp  t64;
   IRExpr* xer_ov;

#  define INT32_MIN 0x80000000

#  define XOR2(_aa,_bb) \
      binop(Iop_Xor32,(_aa),(_bb))

#  define XOR3(_cc,_dd,_ee) \
      binop(Iop_Xor32,binop(Iop_Xor32,(_cc),(_dd)),(_ee))

#  define AND3(_ff,_gg,_hh) \
      binop(Iop_And32,binop(Iop_And32,(_ff),(_gg)),(_hh))

#define NOT(_jj) \
      unop(Iop_Not32, (_jj))

   switch (op) {
   case /* 0  */ PPCG_FLAG_OP_ADD:
   case /* 1  */ PPCG_FLAG_OP_ADDE:
      /* (argL^argR^-1) & (argL^res) & (1<<31)  ?1:0 */
      // i.e. ((both_same_sign) & (sign_changed) & (sign_mask))
      xer_ov 
         = AND3( XOR3(argL,argR,mkU32(-1)),
                 XOR2(argL,res),
                 mkU32(INT32_MIN) );
      /* xer_ov can only be 0 or 1<<31 */
      xer_ov 
         = binop(Iop_Shr32, xer_ov, mkU8(31) );
      break;
      
   case /* 2  */ PPCG_FLAG_OP_DIVW:
      /* (argL == INT32_MIN && argR == -1) || argR == 0 */
      xer_ov
         = mkOR1(
              mkAND1( 
                 binop(Iop_CmpEQ32, argL, mkU32(INT32_MIN)),
                 binop(Iop_CmpEQ32, argR, mkU32(-1)) 
              ),
              binop(Iop_CmpEQ32, argR, mkU32(0) ) 
           );
      xer_ov 
         = unop(Iop_1Uto32, xer_ov);
      break;
      
   case /* 3  */ PPCG_FLAG_OP_DIVWU:
      /* argR == 0 */
      xer_ov 
         = unop(Iop_1Uto32, binop(Iop_CmpEQ32, argR, mkU32(0)));
      break;
      
   case /* 4  */ PPCG_FLAG_OP_MULLW:
      /* OV true if result can't be represented in 32 bits
         i.e sHi != sign extension of sLo */
      t64 = newTemp(Ity_I64);
      assign( t64, binop(Iop_MullS32, argL, argR) );
      xer_ov 
         = binop( Iop_CmpNE32,
                  unop(Iop_64HIto32, mkexpr(t64)),
                  binop( Iop_Sar32, 
                         unop(Iop_64to32, mkexpr(t64)), 
                         mkU8(31))
                  );
      xer_ov
         = unop(Iop_1Uto32, xer_ov);
      break;
      
   case /* 5  */ PPCG_FLAG_OP_NEG:
      /* argL == INT32_MIN */
      xer_ov
         = unop( Iop_1Uto32, 
                 binop(Iop_CmpEQ32, argL, mkU32(INT32_MIN)) );
      break;
      
   case /* 6  */ PPCG_FLAG_OP_SUBF:
   case /* 7  */ PPCG_FLAG_OP_SUBFC:
   case /* 8  */ PPCG_FLAG_OP_SUBFE:
      /* ((~argL)^argR^-1) & ((~argL)^res) & (1<<31) ?1:0; */
      xer_ov 
         = AND3( XOR3(NOT(argL),argR,mkU32(-1)),
                 XOR2(NOT(argL),res),
                 mkU32(INT32_MIN) );
      /* xer_ov can only be 0 or 1<<31 */
      xer_ov 
         = binop(Iop_Shr32, xer_ov, mkU8(31) );
      break;
      
   case PPCG_FLAG_OP_DIVWEU:
      xer_ov
               = binop( Iop_Or32,
                        unop( Iop_1Uto32, binop( Iop_CmpEQ32, argR, mkU32( 0 ) ) ),
                        unop( Iop_1Uto32, binop( Iop_CmpLT32U, argR, argL ) ) );
      break;

   case PPCG_FLAG_OP_DIVWE:

      /* If argR == 0 of if the result cannot fit in the 32-bit destination register,
       * then OV <- 1.   If dest reg is 0 AND both dividend and divisor are non-zero,
       * an overflow is implied.
       */
      xer_ov = binop( Iop_Or32,
                      unop( Iop_1Uto32, binop( Iop_CmpEQ32, argR, mkU32( 0 ) ) ),
                      unop( Iop_1Uto32, mkAND1( binop( Iop_CmpEQ32, res, mkU32( 0 ) ),
                              mkAND1( binop( Iop_CmpNE32, argL, mkU32( 0 ) ),
                                      binop( Iop_CmpNE32, argR, mkU32( 0 ) ) ) ) ) );
      break;



   default: 
      vex_printf("calculate_XER_OV_32: op = %u\n", op);
      vpanic("calculate_XER_OV_32(ppc)");
   }

   return xer_ov;

#  undef INT32_MIN
#  undef AND3
#  undef XOR3
#  undef XOR2
#  undef NOT
}

static void set_XER_OV_OV32_32( UInt op, IRExpr* res,
                                IRExpr* argL, IRExpr* argR )
{
   IRExpr* xer_ov;

   vassert(op < PPCG_FLAG_OP_NUMBER);
   vassert(typeOfIRExpr(irsb->tyenv,res)  == Ity_I32);
   vassert(typeOfIRExpr(irsb->tyenv,argL) == Ity_I32);
   vassert(typeOfIRExpr(irsb->tyenv,argR) == Ity_I32);

   xer_ov = calculate_XER_OV_32( op, res, argL, argR );

   /* xer_ov MUST denote either 0 or 1, no other value allowed */
   putXER_OV( unop(Iop_32to8, xer_ov) );
   putXER_OV32( unop(Iop_32to8, xer_ov) );
}

static IRExpr* calculate_XER_OV_64( UInt op, IRExpr* res,
                                 IRExpr* argL, IRExpr* argR )
{
   IRExpr* xer_ov;

#  define INT64_MIN 0x8000000000000000ULL

#  define XOR2(_aa,_bb) \
      binop(Iop_Xor64,(_aa),(_bb))

#  define XOR3(_cc,_dd,_ee) \
      binop(Iop_Xor64,binop(Iop_Xor64,(_cc),(_dd)),(_ee))

#  define AND3(_ff,_gg,_hh) \
      binop(Iop_And64,binop(Iop_And64,(_ff),(_gg)),(_hh))

#define NOT(_jj) \
      unop(Iop_Not64, (_jj))

   switch (op) {
   case /* 0  */ PPCG_FLAG_OP_ADD:
   case /* 1  */ PPCG_FLAG_OP_ADDE:
      /* (argL^argR^-1) & (argL^res) & (1<<63)  ? 1:0 */
      // i.e. ((both_same_sign) & (sign_changed) & (sign_mask))
      xer_ov 
         = AND3( XOR3(argL,argR,mkU64(-1)),
                 XOR2(argL,res),
                 mkU64(INT64_MIN) );
      /* xer_ov can only be 0 or 1<<63 */
      xer_ov 
         = unop(Iop_64to1, binop(Iop_Shr64, xer_ov, mkU8(63)));
      break;
      
   case /* 2  */ PPCG_FLAG_OP_DIVW:
      /* (argL == INT64_MIN && argR == -1) || argR == 0 */
      xer_ov
         = mkOR1(
              mkAND1( 
                 binop(Iop_CmpEQ64, argL, mkU64(INT64_MIN)),
                 binop(Iop_CmpEQ64, argR, mkU64(-1)) 
              ),
              binop(Iop_CmpEQ64, argR, mkU64(0) ) 
           );
      break;

   case /* 3  */ PPCG_FLAG_OP_DIVWU:
      /* argR == 0 */
      xer_ov 
         = binop(Iop_CmpEQ64, argR, mkU64(0));
      break;
      
   case /* 4  */ PPCG_FLAG_OP_MULLW: {
      /* OV true if result can't be represented in 64 bits
         i.e sHi != sign extension of sLo */
      xer_ov 
         = binop( Iop_CmpNE32,
                  unop(Iop_64HIto32, res),
                  binop( Iop_Sar32, 
                         unop(Iop_64to32, res), 
                         mkU8(31))
                  );
      break;
   }
      
   case /* 5  */ PPCG_FLAG_OP_NEG:
      /* argL == INT64_MIN */
      xer_ov
         = binop(Iop_CmpEQ64, argL, mkU64(INT64_MIN));
      break;
      
   case /* 6  */ PPCG_FLAG_OP_SUBF:
   case /* 7  */ PPCG_FLAG_OP_SUBFC:
   case /* 8  */ PPCG_FLAG_OP_SUBFE:
      /* ((~argL)^argR^-1) & ((~argL)^res) & (1<<63) ?1:0; */
      xer_ov 
         = AND3( XOR3(NOT(argL),argR,mkU64(-1)),
                 XOR2(NOT(argL),res),
                 mkU64(INT64_MIN) );
      /* xer_ov can only be 0 or 1<<63 */
      xer_ov 
         = unop(Iop_64to1, binop(Iop_Shr64, xer_ov, mkU8(63)));
      break;
      
   case /* 14 */ PPCG_FLAG_OP_DIVDE:

      /* If argR == 0, we must set the OV bit.  But there's another condition
       * where we can get overflow set for divde . . . when the
       * result cannot fit in the 64-bit destination register.  If dest reg is 0 AND
       * both dividend and divisor are non-zero, it implies an overflow.
       */
      xer_ov
                  = mkOR1( binop( Iop_CmpEQ64, argR, mkU64( 0 ) ),
                           mkAND1( binop( Iop_CmpEQ64, res, mkU64( 0 ) ),
                                   mkAND1( binop( Iop_CmpNE64, argL, mkU64( 0 ) ),
                                           binop( Iop_CmpNE64, argR, mkU64( 0 ) ) ) ) );
      break;

   case /* 17 */ PPCG_FLAG_OP_DIVDEU:
     /* If argR == 0 or if argL >= argR, set OV. */
     xer_ov = mkOR1( binop( Iop_CmpEQ64, argR, mkU64( 0 ) ),
                         binop( Iop_CmpLE64U, argR, argL ) );
     break;

   case /* 18 */ PPCG_FLAG_OP_MULLD: {
      IRTemp  t128;
      /* OV true if result can't be represented in 64 bits
         i.e sHi != sign extension of sLo */
      t128 = newTemp(Ity_I128);
      assign( t128, binop(Iop_MullS64, argL, argR) );
      xer_ov 
         = binop( Iop_CmpNE64,
                  unop(Iop_128HIto64, mkexpr(t128)),
                  binop( Iop_Sar64,
                         unop(Iop_128to64, mkexpr(t128)),
                         mkU8(63))
                  );
      break;
   }
      
   default: 
      vex_printf("calculate_XER_OV_64: op = %u\n", op);
      vpanic("calculate_XER_OV_64(ppc64)");
   }

   return xer_ov;

#  undef INT64_MIN
#  undef AND3
#  undef XOR3
#  undef XOR2
#  undef NOT
}

static void set_XER_OV_64( UInt op, IRExpr* res,
                           IRExpr* argL, IRExpr* argR )
{
   IRExpr* xer_ov;
   vassert(op < PPCG_FLAG_OP_NUMBER);
   vassert(typeOfIRExpr(irsb->tyenv,res)  == Ity_I64);
   vassert(typeOfIRExpr(irsb->tyenv,argL) == Ity_I64);
   vassert(typeOfIRExpr(irsb->tyenv,argR) == Ity_I64);

   /* xer_ov MUST denote either 0 or 1, no other value allowed */
   xer_ov = calculate_XER_OV_64( op, res, argL, argR);
   putXER_OV( unop(Iop_1Uto8, xer_ov) );

   /* Update the summary overflow */
   putXER_SO( binop(Iop_Or8, getXER_SO(), getXER_OV()) );
}

static void update_SO( void ) {
   /* Update the summary overflow bit */
   putXER_SO( binop(Iop_Or8, getXER_SO(), getXER_OV()) );
}

static void copy_OV_to_OV32( void ) {
   /* Update the OV32 to match OV */
   putXER_OV32( getXER_OV() );
}

static void set_XER_OV_OV32_SO ( IRType ty, UInt op, IRExpr* res,
                                 IRExpr* argL, IRExpr* argR )
{
   if (ty == Ity_I32) {
      set_XER_OV_OV32_32( op, res, argL, argR );
   } else {
      IRExpr* xer_ov_32;
      set_XER_OV_64( op, res, argL, argR );
      xer_ov_32 = calculate_XER_OV_32( op, unop(Iop_64to32, res),
                                       unop(Iop_64to32, argL),
                                       unop(Iop_64to32, argR));
      putXER_OV32( unop(Iop_32to8, xer_ov_32) );
   }
   update_SO();
}



/* RES is the result of doing OP on ARGL and ARGR with the old %XER.CA
   value being OLDCA.  Set %XER.CA accordingly. */

static IRExpr* calculate_XER_CA_32 ( UInt op, IRExpr* res,
                                     IRExpr* argL, IRExpr* argR, IRExpr* oldca )
{
   IRExpr* xer_ca;

   switch (op) {
   case /* 0 */ PPCG_FLAG_OP_ADD:
      /* res <u argL */
      xer_ca
         = unop(Iop_1Uto32, binop(Iop_CmpLT32U, res, argL));
      break;
      
   case /* 1 */ PPCG_FLAG_OP_ADDE:
      /* res <u argL || (old_ca==1 && res==argL) */
      xer_ca 
         = mkOR1( 
              binop(Iop_CmpLT32U, res, argL),
              mkAND1( 
                 binop(Iop_CmpEQ32, oldca, mkU32(1)),
                 binop(Iop_CmpEQ32, res, argL) 
              ) 
           );
      xer_ca 
         = unop(Iop_1Uto32, xer_ca);
      break;
      
   case /* 8 */ PPCG_FLAG_OP_SUBFE:
      /* res <u argR || (old_ca==1 && res==argR) */
      xer_ca 
         = mkOR1( 
              binop(Iop_CmpLT32U, res, argR),
              mkAND1( 
                 binop(Iop_CmpEQ32, oldca, mkU32(1)),
                 binop(Iop_CmpEQ32, res, argR) 
              ) 
           );
      xer_ca 
         = unop(Iop_1Uto32, xer_ca);
      break;
      
   case /* 7 */ PPCG_FLAG_OP_SUBFC:
   case /* 9 */ PPCG_FLAG_OP_SUBFI:
      /* res <=u argR */
      xer_ca
         = unop(Iop_1Uto32, binop(Iop_CmpLE32U, res, argR));
      break;
      
   case /* 10 */ PPCG_FLAG_OP_SRAW:
      /* The shift amount is guaranteed to be in 0 .. 63 inclusive.
         If it is <= 31, behave like SRAWI; else XER.CA is the sign
         bit of argL. */
      /* This term valid for shift amount < 32 only */
      xer_ca
         = binop(
              Iop_And32,
              binop(Iop_Sar32, argL, mkU8(31)),
              binop( Iop_And32,
                     argL,
                     binop( Iop_Sub32,
                            binop(Iop_Shl32, mkU32(1),
                                             unop(Iop_32to8,argR)),
                            mkU32(1) )
                     )
              );
      xer_ca 
         = IRExpr_ITE(
              /* shift amt > 31 ? */
              binop(Iop_CmpLT32U, mkU32(31), argR),
              /* yes -- get sign bit of argL */
              binop(Iop_Shr32, argL, mkU8(31)),
              /* no -- be like srawi */
              unop(Iop_1Uto32, binop(Iop_CmpNE32, xer_ca, mkU32(0)))
           );
      break;

   case /* 11 */ PPCG_FLAG_OP_SRAWI:
      /* xer_ca is 1 iff src was negative and bits_shifted_out != 
         0.  Since the shift amount is known to be in the range
         0 .. 31 inclusive the following seems viable:
         xer.ca == 1 iff the following is nonzero:
         (argL >>s 31)           -- either all 0s or all 1s
         & (argL & (1<<argR)-1)  -- the stuff shifted out */
      xer_ca
         = binop(
              Iop_And32,
              binop(Iop_Sar32, argL, mkU8(31)),
              binop( Iop_And32,
                     argL,
                     binop( Iop_Sub32,
                            binop(Iop_Shl32, mkU32(1),
                                             unop(Iop_32to8,argR)),
                            mkU32(1) )
                     )
              );
      xer_ca 
         = unop(Iop_1Uto32, binop(Iop_CmpNE32, xer_ca, mkU32(0)));
      break;
      
   default: 
      vex_printf("set_XER_CA: op = %u\n", op);
      vpanic("set_XER_CA(ppc)");
   }

   return xer_ca;
}

static void set_XER_CA_32 ( UInt op, IRExpr* res,
                            IRExpr* argL, IRExpr* argR, IRExpr* oldca )
{
   IRExpr* xer_ca;
   vassert(op < PPCG_FLAG_OP_NUMBER);
   vassert(typeOfIRExpr(irsb->tyenv,res)   == Ity_I32);
   vassert(typeOfIRExpr(irsb->tyenv,argL)  == Ity_I32);
   vassert(typeOfIRExpr(irsb->tyenv,argR)  == Ity_I32);
   vassert(typeOfIRExpr(irsb->tyenv,oldca) == Ity_I32);

   /* Incoming oldca is assumed to hold the values 0 or 1 only.  This
      seems reasonable given that it's always generated by
      getXER_CA_32(), which masks it accordingly.  In any case it being
      0 or 1 is an invariant of the ppc guest state representation;
      if it has any other value, that invariant has been violated. */

   xer_ca = calculate_XER_CA_32( op, res, argL, argR, oldca);

   /* xer_ca MUST denote either 0 or 1, no other value allowed */
   putXER_CA( unop(Iop_32to8, xer_ca) );
}

static IRExpr* calculate_XER_CA_64 ( UInt op, IRExpr* res,
                                     IRExpr* argL, IRExpr* argR, IRExpr* oldca )
{
   IRExpr* xer_ca;

   switch (op) {
   case /* 0 */ PPCG_FLAG_OP_ADD:
      /* res <u argL */
      xer_ca
         = unop(Iop_1Uto32, binop(Iop_CmpLT64U, res, argL));
      break;
      
   case /* 1 */ PPCG_FLAG_OP_ADDE:
      /* res <u argL || (old_ca==1 && res==argL) */
      xer_ca 
         = mkOR1( 
              binop(Iop_CmpLT64U, res, argL),
              mkAND1( 
                 binop(Iop_CmpEQ64, oldca, mkU64(1)),
                 binop(Iop_CmpEQ64, res, argL) 
                 ) 
              );
      xer_ca 
         = unop(Iop_1Uto32, xer_ca);
      break;
      
   case /* 8 */ PPCG_FLAG_OP_SUBFE:
      /* res <u argR || (old_ca==1 && res==argR) */
      xer_ca 
         = mkOR1( 
              binop(Iop_CmpLT64U, res, argR),
              mkAND1( 
                 binop(Iop_CmpEQ64, oldca, mkU64(1)),
                 binop(Iop_CmpEQ64, res, argR) 
              ) 
           );
      xer_ca 
         = unop(Iop_1Uto32, xer_ca);
      break;
      
   case /* 7 */ PPCG_FLAG_OP_SUBFC:
   case /* 9 */ PPCG_FLAG_OP_SUBFI:
      /* res <=u argR */
      xer_ca
         = unop(Iop_1Uto32, binop(Iop_CmpLE64U, res, argR));
      break;
      
      
   case /* 10 */ PPCG_FLAG_OP_SRAW:
      /* The shift amount is guaranteed to be in 0 .. 31 inclusive.
         If it is <= 31, behave like SRAWI; else XER.CA is the sign
         bit of argL. */
         /* This term valid for shift amount < 31 only */

      xer_ca
         = binop(
              Iop_And64,
              binop(Iop_Sar64, argL, mkU8(31)),
              binop( Iop_And64,
                     argL,
                     binop( Iop_Sub64,
                            binop(Iop_Shl64, mkU64(1),
                                             unop(Iop_64to8,argR)),
                            mkU64(1) )
              )
           );
      xer_ca 
         = IRExpr_ITE(
              /* shift amt > 31 ? */
              binop(Iop_CmpLT64U, mkU64(31), argR),
              /* yes -- get sign bit of argL */
              unop(Iop_64to32, binop(Iop_Shr64, argL, mkU8(63))),
              /* no -- be like srawi */
              unop(Iop_1Uto32, binop(Iop_CmpNE64, xer_ca, mkU64(0)))
          );
      break;
      
   case /* 11 */ PPCG_FLAG_OP_SRAWI:
      /* xer_ca is 1 iff src was negative and bits_shifted_out != 0.
         Since the shift amount is known to be in the range 0 .. 31
         inclusive the following seems viable:
         xer.ca == 1 iff the following is nonzero:
         (argL >>s 31)           -- either all 0s or all 1s
         & (argL & (1<<argR)-1)  -- the stuff shifted out */

      xer_ca
         = binop(
              Iop_And64,
              binop(Iop_Sar64, argL, mkU8(31)),
              binop( Iop_And64,
                     argL,
                     binop( Iop_Sub64,
                            binop(Iop_Shl64, mkU64(1),
                                             unop(Iop_64to8,argR)),
                            mkU64(1) )
              )
           );
      xer_ca 
         = unop(Iop_1Uto32, binop(Iop_CmpNE64, xer_ca, mkU64(0)));
      break;
      

   case /* 12 */ PPCG_FLAG_OP_SRAD:
      /* The shift amount is guaranteed to be in 0 .. 63 inclusive.
         If it is <= 63, behave like SRADI; else XER.CA is the sign
         bit of argL. */
         /* This term valid for shift amount < 63 only */

      xer_ca
         = binop(
              Iop_And64,
              binop(Iop_Sar64, argL, mkU8(63)),
              binop( Iop_And64,
                     argL,
                     binop( Iop_Sub64,
                            binop(Iop_Shl64, mkU64(1),
                                             unop(Iop_64to8,argR)),
                            mkU64(1) )
              )
           );
      xer_ca 
         = IRExpr_ITE(
              /* shift amt > 63 ? */
              binop(Iop_CmpLT64U, mkU64(63), argR),
              /* yes -- get sign bit of argL */
              unop(Iop_64to32, binop(Iop_Shr64, argL, mkU8(63))),
              /* no -- be like sradi */
              unop(Iop_1Uto32, binop(Iop_CmpNE64, xer_ca, mkU64(0)))
           );
      break;


   case /* 13 */ PPCG_FLAG_OP_SRADI:
      /* xer_ca is 1 iff src was negative and bits_shifted_out != 0.
         Since the shift amount is known to be in the range 0 .. 63
         inclusive, the following seems viable:
         xer.ca == 1 iff the following is nonzero:
         (argL >>s 63)           -- either all 0s or all 1s
         & (argL & (1<<argR)-1)  -- the stuff shifted out */

      xer_ca
         = binop(
              Iop_And64,
              binop(Iop_Sar64, argL, mkU8(63)),
              binop( Iop_And64,
                     argL,
                     binop( Iop_Sub64,
                            binop(Iop_Shl64, mkU64(1),
                                             unop(Iop_64to8,argR)),
                            mkU64(1) )
              )
           );
      xer_ca 
         = unop(Iop_1Uto32, binop(Iop_CmpNE64, xer_ca, mkU64(0)));
      break;

   default: 
      vex_printf("set_XER_CA: op = %u\n", op);
      vpanic("set_XER_CA(ppc64)");
   }

   return xer_ca;
}

static void set_XER_CA_64 ( UInt op, IRExpr* res,
                            IRExpr* argL, IRExpr* argR, IRExpr* oldca )
{
   IRExpr* xer_ca;
   vassert(op < PPCG_FLAG_OP_NUMBER);
   vassert(typeOfIRExpr(irsb->tyenv,res)   == Ity_I64);
   vassert(typeOfIRExpr(irsb->tyenv,argL)  == Ity_I64);
   vassert(typeOfIRExpr(irsb->tyenv,argR)  == Ity_I64);
   vassert(typeOfIRExpr(irsb->tyenv,oldca) == Ity_I64);

   /* Incoming oldca is assumed to hold the values 0 or 1 only.  This
      seems reasonable given that it's always generated by
      getXER_CA_32(), which masks it accordingly.  In any case it being
      0 or 1 is an invariant of the ppc guest state representation;
      if it has any other value, that invariant has been violated. */

   xer_ca = calculate_XER_CA_64( op, res, argL, argR, oldca );

   /* xer_ca MUST denote either 0 or 1, no other value allowed */
   putXER_CA( unop(Iop_32to8, xer_ca) );
}

static void set_XER_CA_CA32 ( IRType ty, UInt op, IRExpr* res,
                              IRExpr* argL, IRExpr* argR, IRExpr* oldca )
{
   if (ty == Ity_I32) {
      set_XER_CA_32( op, res, argL, argR, oldca );
   } else {
      set_XER_CA_64( op, res, argL, argR, oldca );
   }
}

/* Used only by addex instruction, which uses and sets OV as carry.  */
static void set_XER_OV_OV32_ADDEX ( IRType ty, IRExpr* res,
                                    IRExpr* argL, IRExpr* argR,
                                    IRExpr* old_ov )
{
   if (ty == Ity_I32) {
      IRTemp xer_ov = newTemp(Ity_I32);
      assign ( xer_ov, unop(Iop_32to8,
                            calculate_XER_CA_32( PPCG_FLAG_OP_ADDE,
                                                 res, argL, argR, old_ov ) ) );
      putXER_OV( mkexpr (xer_ov) );
      putXER_OV32( mkexpr (xer_ov) );
   } else {
      IRExpr *xer_ov;
      IRExpr* xer_ov_32;
      xer_ov = calculate_XER_CA_64( PPCG_FLAG_OP_ADDE,
                                    res, argL, argR, old_ov );
      putXER_OV( unop(Iop_32to8, xer_ov) );
      xer_ov_32 = calculate_XER_CA_32( PPCG_FLAG_OP_ADDE,
                                       unop(Iop_64to32, res),
                                       unop(Iop_64to32, argL),
                                       unop(Iop_64to32, argR),
                                       unop(Iop_64to32, old_ov) );
      putXER_OV32( unop(Iop_32to8, xer_ov_32) );
   }
}

static IRExpr * absI64( IRTemp src )
{
   IRTemp sign_mask;
   IRTemp twos_comp;
   sign_mask = newTemp( Ity_I64 );
   twos_comp = newTemp( Ity_I64 );

   assign( sign_mask, unop( Iop_1Sto64,  unop( Iop_64to1, binop( Iop_Shr64,
                                                                 mkexpr( src ), mkU8( 63 ) ) ) ) );
   assign( twos_comp, binop( Iop_Add64, unop( Iop_Not64, mkexpr( src ) ), mkU64( 1 ) ) );

   return binop( Iop_Or64,
                 binop( Iop_And64, mkexpr ( src ), unop( Iop_Not64, mkexpr( sign_mask ) ) ),
                 binop( Iop_And64, mkexpr( twos_comp ),  mkexpr( sign_mask ) ) );
}

static IRExpr * locate_vector_ele_eq ( IRTemp src, IRExpr *value,
                                       UInt dir, IRType size )
{
#define MAX_ELE 16
   /* Find the index, 0 to max-1, of the element in 128-bit vector that matches
      value.  The returned value will be index+1. Return the index as an
      Ity_I8.  If no match is found, the returned value is equal to the number
      of elements in the vector plus one. The argument dir specifies match from
      left (dir = 0) or from the right (dir != 0).  */
   UInt i, num_bytes;
   UInt max  = 0;          /* number of vector elements */
   UInt mask = 0;
   IRTemp cnt[MAX_ELE+1];
   IRTemp flag[MAX_ELE+1];
   IRTemp cmp_result[MAX_ELE];
   UInt byte_index;

   vassert(size == Ity_I8  || size == Ity_I16);

   if (size == Ity_I8) {
      mask = 0xFF;
      max = 128/8;
      num_bytes = 1;
   } else {
      mask = 0xFFFF;
      max = 128/16;
      num_bytes = 2;       // num bytes in half word
   }

   cnt[0] = newTemp(Ity_I8);
   assign( cnt[0], mkU8( 1 ) );
   flag[0] = newTemp(Ity_I8);
   assign( flag[0], mkU8( 1 ) );

   for (i = 0; i < max; i++) {
      if (dir == 0) {
         byte_index = (max - 1 - i)*num_bytes;
      } else {
         byte_index = i*num_bytes;
      }

      cnt[i+1] = newTemp(Ity_I8);
      cmp_result[i] = newTemp(Ity_I8);
      flag[i+1] = newTemp(Ity_I8);

      assign( cmp_result[i],
              unop( Iop_1Uto8,
                    binop( Iop_CmpEQ64,
                           binop( Iop_And64,
                                  mkU64( mask ),
                                  value ),
                           extract_field_from_vector( src,
                                                      mkU64( byte_index ),
                                                      mask ) ) ) );

      assign( flag[i+1], binop( Iop_And8,
                                mkexpr( flag[i] ),
                                unop( Iop_Not8,
                                      mkexpr( cmp_result[i] ) ) ) );

      // Once flag[i] becomes zero, it forces the increment to zero
      assign( cnt[i+1],
              binop( Iop_Add8,
                     binop( Iop_And8, mkexpr( flag[i+1] ), mkU8( 1 ) ),
                     mkexpr( cnt[i] ) ) );
   }
   return mkexpr( cnt[max] );
#undef MAX_ELE
}

/*-----------------------------------------------------------*/
/*---  Prefix instruction helpers                         ---*/
/*-----------------------------------------------------------*/
#define DFORM_IMMASK  0xffffffff
#define DSFORM_IMMASK 0xfffffffc
#define DQFORM_IMMASK 0xfffffff0
#define DA8LSFORM_IMMASK 0x3fffffff   // Algebraic 8LS Dform

#define ISA_3_1_PREFIX_CHECK if (prefix) {if (!allow_isa_3_1) goto decode_noIsa3_1;}

/* ENABLE_PREFIX_CHECK is for development purposes.  Turn off for production
   releases to improve performance.  */
#define ENABLE_PREFIX_CHECK  0

#if ENABLE_PREFIX_CHECK
#define PREFIX_CHECK { vassert( !prefix_instruction( prefix ) ); }
#else
#define PREFIX_CHECK { }
#endif

/* Bits 0:5 of all prefix instructions are assigned the primary opcode
   value 0b000001. 0b000001 is not available for use as a primary opcode for
   either word instructions or suffixes of prefixed instructions.  */

#define PREFIX_INST 0x1
#define PREFIX_NOP_INVALID  -1

#define CONCAT(_aa,_bb,_cc) ((_aa) << (_cc) | (_bb))

/* The codes for the prefix types */
#define pType0  0  /* Eight-Byte Load/Store Instructions */
#define pType1  1  /* Eight-Byte Register-to-Register Instructions */
#define pType2  2  /* Modified Load/Store Instructions */
#define pType3  3  /* Modified Register-to-Register Instructions */

/* Extract unsigned from prefix instr[17:0] */
static UInt ifieldUIMM18 ( UInt instr ) {
   return instr & 0x3FFFF;
}

static ULong extend_s_34to64 ( ULong x )
{
   return (ULong)((((Long)x) << 30) >> 30);
}

static UChar PrefixType( UInt instr ) {
   return toUChar( IFIELD( instr, 24, 2 ) );
}

/* Extract XT 8RR D-form (destination register) field, instr[38:42] | [47] */
static UChar ifieldRegXT_8RR_D ( UInt instr )
{
   UChar TX = toUChar (IFIELD (instr, (63 - 47), 1));
   UChar T  = toUChar (IFIELD (instr, (63 - 42), 4));
   return (TX << 5) | T;
}

/* Extract immediate 8RR D-form prefix[16:31] or inst[48:63] */
static UInt ifield_imm_8RR_D ( UInt instr )
{
   return IFIELD( instr, 0, 16 );
}

static UChar ifieldR( UInt instr ) {
   return toUChar( IFIELD( instr, 20, 1 ) );
}

/* Sign extend imm34 -> IRExpr* */
static IRExpr* mkSzExtendS34 ( ULong imm64 )
{
   return ( mkU64(extend_s_34to64(imm64)));
}

/* Prefix instruction effective address calc: (rA + simm) */
static IRExpr* ea_rA_simm34 ( UInt rA, ULong simm34 )
{
   vassert(rA < 32);
   vassert(mode64);
   return binop(Iop_Add64, getIReg(rA), mkSzExtendS34(simm34));
}

/* Standard prefix instruction effective address calc: (rA|0) + simm16 */
static IRExpr* ea_rAor0_simm34 ( UInt rA, ULong simm34 )
{
   vassert(rA < 32);
   vassert(mode64);
   if (rA == 0) {
      return mkSzExtendS34(simm34);
   } else {
      return ea_rA_simm34( rA, simm34 );
   }
}

static int prefix_instruction ( UInt instr )
{
  /* Format of first 4 bytes of prefix instruction
     bits [0:5]  -  must be 0x1 identifying this as a prefix inst
     bits [6:7]  -  prefix instruction type.  */
  UChar opcode = IFIELD( instr, 26, 6);

  if (opcode == PREFIX_INST) return True;
  return False;
}

/* standard offset calculation, check prefix type */
static IRExpr* calculate_prefix_EA ( UInt prefix, UInt suffixInstr,
                                     UChar rA_addr, UInt ptype,
                                     UInt immediate_mask,
                                     ULong *immediate_val,
                                     UInt *R )
{
   IRType  ty     = Ity_I64;
   ULong   d0     = ifieldUIMM18(prefix);  // Will be zero for word inst
   ULong   d1     = ifieldUIMM16(suffixInstr) & immediate_mask;
   ULong   D      = CONCAT( d0, d1, 16 );  // result is 34 bits wide
   Bool    is_prefix = prefix_instruction( prefix );
   IRTemp  tmp    = newTemp(ty);

   if ( !is_prefix ) {
      *immediate_val = extend_s_16to32( d1 );
      assign( tmp, ea_rAor0_simm( rA_addr, d1 ) );
      *R = 0;

   } else {
      vassert( ty == Ity_I64 );    // prefix instructions must be 64-bit
      vassert( (ptype == pType0) || (ptype == pType2) );
      *R = ifieldR( prefix );
      *immediate_val = extend_s_34to64( D );  // D is 34 bits wide
      assign( tmp, ea_rAor0_simm34( rA_addr, D ) );
   }

   /* Get the EA */
   if ( *R == 0 )
      return mkexpr ( tmp );

   /* Add immediate value from instruction to the current instruction
      address. guest_CIA_curr_instr is pointing at the prefix, use address
      of the instruction prefix. */
   return binop( Iop_Add64,
                 mkexpr ( tmp ),
                 mkU64( guest_CIA_curr_instr ) );
}

/* Extract prefix intruction register fields 8RR:XX4-form */
static UChar ifieldRegXT_8RR_XX4 ( UInt instr ) {
   return toUChar( ( IFIELD( instr, (63-63), 1) << 5)
                   | ( IFIELD( instr, (63-42), 5 ) ) );  // instr[38:42] | [63]
}

static UChar ifieldRegXA_8RR_XX4 ( UInt instr ) {
   return toUChar( ( IFIELD( instr, (63-61), 1) << 5)
                   | ( IFIELD( instr, (63-47), 5 ) ) );  // instr[43:47] | [61]
}

static UChar ifieldRegXB_8RR_XX4 ( UInt instr ) {
   return toUChar( ( IFIELD( instr, (63-62), 1) << 5)
                   | ( IFIELD( instr, (63-52), 5 ) ) );  // instr[48:52] | [62]
}

static UChar ifieldRegXC_8RR_XX4 ( UInt instr ) {
   return toUChar( ( IFIELD( instr, (63-60), 1) << 5)
                   | ( IFIELD( instr, (63-57), 5 ) ) );  // instr[53:57] | [60]
}

/*------------------------------------------------------------*/
/*--- Read/write to guest-state                           --- */
/*------------------------------------------------------------*/

static IRExpr* /* :: Ity_I32/64 */ getGST ( PPC_GST reg )
{
   IRType ty = mode64 ? Ity_I64 : Ity_I32;
   switch (reg) {
   case PPC_GST_SPRG3_RO:
      return IRExpr_Get( OFFB_SPRG3_RO, ty );

   case PPC_GST_CIA: 
      return IRExpr_Get( OFFB_CIA, ty );

   case PPC_GST_LR: 
      return IRExpr_Get( OFFB_LR, ty );

   case PPC_GST_CTR: 
      return IRExpr_Get( OFFB_CTR, ty );

   case PPC_GST_VRSAVE: 
      return IRExpr_Get( OFFB_VRSAVE, Ity_I32 );

   case PPC_GST_VSCR:
      return binop(Iop_And32, IRExpr_Get( OFFB_VSCR,Ity_I32 ),
                              mkU32(MASK_VSCR_VALID));

   case PPC_GST_CR: {
      /* Synthesise the entire CR into a single word.  Expensive. */
#     define FIELD(_n)                                               \
         binop(Iop_Shl32,                                            \
               unop(Iop_8Uto32,                                      \
                    binop(Iop_Or8,                                   \
                          binop(Iop_And8, getCR321(_n), mkU8(7<<1)), \
                          binop(Iop_And8, getCR0(_n), mkU8(1))       \
                    )                                                \
               ),                                                    \
               mkU8(4 * (7-(_n)))                                    \
         )
      return binop(Iop_Or32,
                   binop(Iop_Or32,
                         binop(Iop_Or32, FIELD(0), FIELD(1)),
                         binop(Iop_Or32, FIELD(2), FIELD(3))
                         ),
                   binop(Iop_Or32,
                         binop(Iop_Or32, FIELD(4), FIELD(5)),
                         binop(Iop_Or32, FIELD(6), FIELD(7))
                         )
                   );
#     undef FIELD
   }

   case PPC_GST_XER:
      return binop(Iop_Or32,
                   binop(Iop_Or32,
                         binop(Iop_Or32,
                               binop( Iop_Shl32, getXER_SO_32(), mkU8(31)),
                               binop( Iop_Shl32, getXER_OV_32(), mkU8(30))),
                         binop(Iop_Or32,
                               binop( Iop_Shl32, getXER_CA_32(), mkU8(29)),
                               getXER_BC_32())),
                   binop(Iop_Or32,
                         binop( Iop_Shl32, getXER_OV32_32(), mkU8(19)),
                         binop( Iop_Shl32, getXER_CA32_32(), mkU8(18))));

   case PPC_GST_TFHAR:
      return IRExpr_Get( OFFB_TFHAR, ty );

   case PPC_GST_TEXASR:
      return IRExpr_Get( OFFB_TEXASR, ty );

   case PPC_GST_TEXASRU:
      return IRExpr_Get( OFFB_TEXASRU, ty );

   case PPC_GST_TFIAR:
      return IRExpr_Get( OFFB_TFIAR, ty );

   case PPC_GST_PPR:
      return IRExpr_Get( OFFB_PPR, ty );

   case PPC_GST_PPR32:
      return unop( Iop_64HIto32, IRExpr_Get( OFFB_PPR, ty ) );

   case PPC_GST_PSPB:
      return IRExpr_Get( OFFB_PSPB, ty );

   case PPC_GST_DSCR:
      return IRExpr_Get( OFFB_DSCR, ty );

   default:
      vex_printf("getGST(ppc): reg = %u", reg);
      vpanic("getGST(ppc)");
   }
}

/* Get a masked word from the given reg */
static IRExpr* /* ::Ity_I32 */ getGST_masked ( PPC_GST reg, ULong mask )
{
   IRTemp val = newTemp(Ity_I32);
   vassert( reg < PPC_GST_MAX );
    
   switch (reg) {

   case PPC_GST_FPSCR: {
      /* Vex-generated code expects the FPSCR to be set as follows:
         all exceptions masked, round-to-nearest.
         This corresponds to a FPSCR value of 0x0. */

      /* In the lower 32 bits of FPSCR, we're keeping track of the binary
       * floating point rounding mode and Floating-point Condition code, so
       * if the mask isn't asking for either of these, just return 0x0.
       */
      if ( mask & ( MASK_FPSCR_C_FPCC | MASK_FPSCR_RN ) ) {
         assign( val, binop( Iop_Or32,
                             unop( Iop_8Uto32, IRExpr_Get( OFFB_FPROUND, Ity_I8 ) ),
                             binop( Iop_Shl32,
                                    unop( Iop_8Uto32,
                                          IRExpr_Get( OFFB_C_FPCC, Ity_I8 ) ),
                                    mkU8( 12 ) ) ) );
      } else {
         assign( val, mkU32(0x0) );
      }

      break;
   }

   default:
      vex_printf("getGST_masked(ppc): reg = %u", reg);
      vpanic("getGST_masked(ppc)");
   }

   if ( mask != 0xFFFFFFFF ) {
      return binop(Iop_And32, mkexpr(val), mkU32(mask));
   } else {
      return mkexpr(val);
   }
}

/* Get a masked word from the given reg */
static IRExpr* /* ::Ity_I32 */getGST_masked_upper(PPC_GST reg, ULong mask) {
   IRExpr * val;
   vassert( reg < PPC_GST_MAX );

   switch (reg) {

   case PPC_GST_FPSCR: {
      /* In the upper 32 bits of FPSCR, we're only keeping track
       * of the decimal floating point rounding mode, so if the mask
       * isn't asking for this, just return 0x0.
       */
      if (mask & MASK_FPSCR_DRN) {
         val = binop( Iop_And32,
                      unop( Iop_8Uto32, IRExpr_Get( OFFB_DFPROUND, Ity_I8 ) ),
                      unop( Iop_64HIto32, mkU64( mask ) ) );
      } else {
         val = mkU32( 0x0ULL );
      }
      break;
   }

   default:
      vex_printf( "getGST_masked_upper(ppc): reg = %u", reg );
      vpanic( "getGST_masked_upper(ppc)" );
   }
   return val;
}


/* Fetch the specified REG[FLD] nibble (as per IBM/hardware notation)
   and return it at the bottom of an I32; the top 27 bits are
   guaranteed to be zero. */
static IRExpr* /* ::Ity_I32 */ getGST_field ( PPC_GST reg, UInt fld )
{
   UInt shft, mask;

   vassert( fld < 8 );
   vassert( reg < PPC_GST_MAX );
   
   shft = 4*(7-fld);
   mask = 0xF<<shft;

   switch (reg) {
   case PPC_GST_XER:
      vassert(fld ==7);
      return binop(Iop_Or32,
                   binop(Iop_Or32,
                         binop(Iop_Shl32, getXER_SO_32(), mkU8(3)),
                         binop(Iop_Shl32, getXER_OV_32(), mkU8(2))),
                   binop(      Iop_Shl32, getXER_CA_32(), mkU8(1)));
      break;

   default:
      if (shft == 0)
         return getGST_masked( reg, mask );
      else
         return binop(Iop_Shr32,
                      getGST_masked( reg, mask ),
                      mkU8(toUChar( shft )));
   }
}

static void putGST ( PPC_GST reg, IRExpr* src )
{
   IRType ty     = mode64 ? Ity_I64 : Ity_I32;
   IRType ty_src = typeOfIRExpr(irsb->tyenv,src );
   vassert( reg < PPC_GST_MAX );
   switch (reg) {
   case PPC_GST_IP_AT_SYSCALL: 
      vassert( ty_src == ty );
      stmt( IRStmt_Put( OFFB_IP_AT_SYSCALL, src ) );
      break;
   case PPC_GST_CIA: 
      vassert( ty_src == ty );
      stmt( IRStmt_Put( OFFB_CIA, src ) );
      break;
   case PPC_GST_LR: 
      vassert( ty_src == ty );
      stmt( IRStmt_Put( OFFB_LR, src ) );
      break;
   case PPC_GST_CTR: 
      vassert( ty_src == ty );
      stmt( IRStmt_Put( OFFB_CTR, src ) );
      break;
   case PPC_GST_VRSAVE: 
      vassert( ty_src == Ity_I32 );
      stmt( IRStmt_Put( OFFB_VRSAVE,src));
      break;
   case PPC_GST_VSCR:
      vassert( ty_src == Ity_I32 );
      stmt( IRStmt_Put( OFFB_VSCR,
                        binop(Iop_And32, src,
                              mkU32(MASK_VSCR_VALID)) ) );
      break;
   case PPC_GST_XER:
      vassert( ty_src == Ity_I32 );
      putXER_SO( unop(Iop_32to8, binop(Iop_Shr32, src, mkU8(31))) );
      putXER_OV( unop(Iop_32to8, binop(Iop_Shr32, src, mkU8(30))) );
      putXER_CA( unop(Iop_32to8, binop(Iop_Shr32, src, mkU8(29))) );
      putXER_OV32( unop(Iop_32to8, binop(Iop_Shr32, src, mkU8(19))) );
      putXER_CA32( unop(Iop_32to8, binop(Iop_Shr32, src, mkU8(18))) );
      putXER_BC( unop(Iop_32to8, src) );
      break;
      
   case PPC_GST_EMWARN:
      vassert( ty_src == Ity_I32 );
      stmt( IRStmt_Put( OFFB_EMNOTE,src) );
      break;
      
   case PPC_GST_CMSTART: 
      vassert( ty_src == ty );
      stmt( IRStmt_Put( OFFB_CMSTART, src) );
      break;
      
   case PPC_GST_CMLEN: 
      vassert( ty_src == ty );
      stmt( IRStmt_Put( OFFB_CMLEN, src) );
      break;
      
   case PPC_GST_TEXASR:
      vassert( ty_src == Ity_I64 );
      stmt( IRStmt_Put( OFFB_TEXASR, src ) );
      break;

   case PPC_GST_TEXASRU:
      vassert( ty_src == Ity_I32 );
      stmt( IRStmt_Put( OFFB_TEXASRU, src ) );
      break;

   case PPC_GST_TFIAR:
      vassert( ty_src == Ity_I64 );
      stmt( IRStmt_Put( OFFB_TFIAR, src ) );
      break;
   case PPC_GST_TFHAR:
      vassert( ty_src == Ity_I64 );
      stmt( IRStmt_Put( OFFB_TFHAR, src ) );
      break;

   case PPC_GST_PPR32:
   case PPC_GST_PPR:
      {
         /* The Program Priority Register (PPR) stores the priority in
          * bits [52:50].  The user setable priorities are:
          *
          *    001  very low
          *    010  low
          *    011  medium low
          *    100  medium
          *    101  medium high
          *
          * If the argument is not between 0b001 and 0b100 the priority is set
          * to 0b100.  The priority can only be set to 0b101 if the the Problem
          * State Boost Register is non-zero.  The value of the PPR is not
          * changed if the input is not valid.
          */

         IRTemp not_valid = newTemp(Ity_I64);
         IRTemp has_perm = newTemp(Ity_I64);
         IRTemp new_src  = newTemp(Ity_I64);
         IRTemp PSPB_val = newTemp(Ity_I64);
         IRTemp value    = newTemp(Ity_I64);

         vassert(( ty_src == Ity_I64 ) || ( ty_src == Ity_I32 ));
         assign( PSPB_val, binop( Iop_32HLto64,
                                  mkU32( 0 ),
                                  IRExpr_Get( OFFB_PSPB, Ity_I32 ) ) );
         if( reg == PPC_GST_PPR32 ) {
            vassert( ty_src == Ity_I32 );
            assign( value, binop( Iop_32HLto64,
                                  mkU32(0),
                                  binop( Iop_And32,
                                         binop( Iop_Shr32, src,  mkU8( 18 ) ),
                                         mkU32( 0x7 ) ) ) );
         } else {
            vassert( ty_src == Ity_I64 );
            assign( value, binop( Iop_And64,
                                  binop( Iop_Shr64, src,  mkU8( 50 ) ),
                                  mkU64( 0x7 ) ) );
         }
         assign( has_perm,
                 binop( Iop_And64,
                        unop( Iop_1Sto64,
                              binop( Iop_CmpEQ64,
                                     mkexpr( PSPB_val ),
                                     mkU64( 0 ) ) ),
                        unop( Iop_1Sto64,
                              binop( Iop_CmpEQ64,
                                     mkU64( 0x5 ),
                                     mkexpr( value ) ) ) ) );
         assign( not_valid,
                 binop( Iop_Or64,
                        unop( Iop_1Sto64,
                              binop( Iop_CmpEQ64,
                                     mkexpr( value ),
                                     mkU64( 0 ) ) ),
                        unop( Iop_1Sto64,
                              binop( Iop_CmpLT64U,
                                     mkU64( 0x5 ),
                                     mkexpr( value ) ) ) ) );
         assign( new_src,
                 binop( Iop_Or64,
                        binop( Iop_And64,
                               unop( Iop_Not64,
                                     mkexpr( not_valid ) ),
                               src ),
                        binop( Iop_And64,
                               mkexpr( not_valid ),
                               binop( Iop_Or64,
                                      binop( Iop_And64,
                                             mkexpr( has_perm),
                                             binop( Iop_Shl64,
                                                    mkexpr( value ),
                                                    mkU8( 50 ) ) ),
                                      binop( Iop_And64,
                                             IRExpr_Get( OFFB_PPR, ty ),
                                             unop( Iop_Not64,
                                                   mkexpr( has_perm )
                                                   ) ) ) ) ) );

                 /* make sure we only set the valid bit field [52:50] */
                 stmt( IRStmt_Put( OFFB_PPR,
                                   binop( Iop_And64,
                                          mkexpr( new_src ),
                                          mkU64( 0x1C000000000000) ) ) );
      break;
      }
   case PPC_GST_DSCR:
      vassert( ty_src == Ity_I64 );
      stmt( IRStmt_Put( OFFB_DSCR, src ) );
      break;

   default:
      vex_printf("putGST(ppc): reg = %u", reg);
      vpanic("putGST(ppc)");
   }
}

/* Write masked src to the given reg */
static void putGST_masked ( PPC_GST reg, IRExpr* src, ULong mask )
{
   IRType ty = mode64 ? Ity_I64 : Ity_I32;
   vassert( reg < PPC_GST_MAX );
   vassert( typeOfIRExpr( irsb->tyenv,src ) == Ity_I64 );

   switch (reg) {
   case PPC_GST_FPSCR: {
      /* Allow writes to either binary or decimal floating point
         Rounding Mode.
      */
      /* If any part of |mask| covers FPSCR.RN, update the bits of
         FPSCR.RN by copying in |src| for locations where the
         corresponding bit in |mask| is 1, and leaving it unchanged
         for corresponding |mask| zero bits. */
      if (mask & MASK_FPSCR_RN) {
         stmt( 
            IRStmt_Put(
               OFFB_FPROUND,
               unop(
                  Iop_32to8,
                  binop(
                     Iop_Or32, 
                     binop(
                        Iop_And32,
                        unop(Iop_64to32, src),
                        mkU32(MASK_FPSCR_RN & mask)
                     ),
                     binop(
                        Iop_And32, 
                        unop(Iop_8Uto32, IRExpr_Get(OFFB_FPROUND,Ity_I8)),
                        mkU32(MASK_FPSCR_RN & ~mask)
                     )
                  )
               )
            )
         );
      }

      if (mask & MASK_FPSCR_C_FPCC) {
         /* FPCC bits are in [47:51] */
         stmt(
            IRStmt_Put(
               OFFB_C_FPCC,
               unop(
                    Iop_32to8,
                    binop(Iop_Shr32,
                          binop(
                                Iop_Or32,
                                binop(
                                      Iop_And32,
                                      unop(Iop_64to32, src),
                                      mkU32(MASK_FPSCR_C_FPCC & mask) ),
                                binop(
                                      Iop_And32,
                                      unop(Iop_8Uto32,
                                           IRExpr_Get(OFFB_C_FPCC,Ity_I8)),
                                      mkU32(MASK_FPSCR_C_FPCC & ~mask)
                                      ) ),
                          mkU8( 12 ) )
                    ) ) );
      }

      /* Similarly, update FPSCR.DRN if any bits of |mask|
         corresponding to FPSCR.DRN are set. */
      if (mask & MASK_FPSCR_DRN) {
         stmt( 
            IRStmt_Put(
               OFFB_DFPROUND,
               unop(
                  Iop_32to8,
                  binop(
                     Iop_Or32, 
                     binop(
                        Iop_And32,
                        unop(Iop_64HIto32, src),
                        mkU32((MASK_FPSCR_DRN & mask) >> 32)
                     ),
                     binop(
                        Iop_And32, 
                        unop(Iop_8Uto32, IRExpr_Get(OFFB_DFPROUND,Ity_I8)),
                        mkU32((MASK_FPSCR_DRN & ~mask) >> 32)
                     )
                  )
               )
            )
         );
      }

      /* Give EmNote for attempted writes to:
         - Exception Controls
         - Non-IEEE Mode
      */
      if (mask & 0xFC) {  // Exception Control, Non-IEE mode
         VexEmNote ew = EmWarn_PPCexns;

         /* If any of the src::exception_control bits are actually set,
            side-exit to the next insn, reporting the warning,
            so that Valgrind's dispatcher sees the warning. */
         putGST( PPC_GST_EMWARN, mkU32(ew) );
         stmt( 
            IRStmt_Exit(
               binop(Iop_CmpNE32, mkU32(ew), mkU32(EmNote_NONE)),
               Ijk_EmWarn,
               mkSzConst( ty, nextInsnAddr()), OFFB_CIA ));
      }

      /* Ignore all other writes */
      break;
   }

   default:
      vex_printf("putGST_masked(ppc): reg = %u", reg);
      vpanic("putGST_masked(ppc)");
   }
}

/* Write the least significant nibble of src to the specified
   REG[FLD] (as per IBM/hardware notation). */
static void putGST_field ( PPC_GST reg, IRExpr* src, UInt fld )
{
   UInt shft;
   ULong mask;

   vassert( typeOfIRExpr(irsb->tyenv,src ) == Ity_I32 );
   vassert( fld < 16 );
   vassert( reg < PPC_GST_MAX );
   
   if (fld < 8)
      shft = 4*(7-fld);
   else
      shft = 4*(15-fld);
   mask = 0xF;
   mask = mask << shft;

   switch (reg) {
   case PPC_GST_CR:
      putCR0  (fld, binop(Iop_And8, mkU8(1   ), unop(Iop_32to8, src)));
      putCR321(fld, binop(Iop_And8, mkU8(7<<1), unop(Iop_32to8, src)));
      break;

   default:
      {
         IRExpr * src64 = unop( Iop_32Uto64, src );

         if (shft == 0) {
            putGST_masked( reg, src64, mask );
         } else {
            putGST_masked( reg,
                           binop( Iop_Shl64, src64, mkU8( toUChar( shft ) ) ),
                           mask );
         }
      }
   }
}

static void putFPCC ( IRExpr* e )
{
   /* The assumption is that the value of the FPCC are passed in the lower
    * four bits of a 32 bit value.
    *
    * Note, the C and FPCC bits which are a field of the FPSCR
    * register are stored in their own "register" in
    * memory.  The FPCC bits are in the lower 4 bits.  We don't need to
    * shift it to the bits to their location in the FPSCR register.  Note,
    * not all of the FPSCR register bits are supported.  We are writing all
    * of the bits in the FPCC field but not the C field.
    */
   IRExpr* tmp;

   vassert( typeOfIRExpr( irsb->tyenv, e ) == Ity_I32 );
   /* Get the C bit field */
   tmp = binop( Iop_And32,
                mkU32( 0x10 ),
                unop( Iop_8Uto32, IRExpr_Get( OFFB_C_FPCC, Ity_I8 ) ) );

   stmt( IRStmt_Put( OFFB_C_FPCC,
                     unop( Iop_32to8,
                           binop( Iop_Or32, tmp,
                                  binop( Iop_And32, mkU32( 0xF ), e ) ) ) ) );

}

static IRExpr* /* ::Ity_I32 */  getC ( void )
{
   /* Note, the Floating-Point Result Class Descriptor (C) bit is a field of
    * the FPSCR registered are stored in its own "register" in guest state
    * with the FPCC bit field.   C | FPCC
    */
   IRTemp val = newTemp(Ity_I32);

   assign( val, binop( Iop_Shr32,
                       unop( Iop_8Uto32, IRExpr_Get( OFFB_C_FPCC, Ity_I8 ) ),
                       mkU8( 4 ) ) );
   return mkexpr(val);
}

static IRExpr* /* ::Ity_I32 */  getFPCC ( void )
{
   /* Note, the FPCC bits are a field of the FPSCR
    * register are stored in their own "register" in
    * guest state with the C bit field.   C | FPCC
    */
   IRTemp val = newTemp( Ity_I32 );

   assign( val, binop( Iop_And32, unop( Iop_8Uto32,
                                        IRExpr_Get( OFFB_C_FPCC, Ity_I8 ) ),
                       mkU32( 0xF ) ));
   return mkexpr(val);
}

static void put_syscall_flag( IRExpr* src )
{
   /* Need to pass a flag indicating if the system call is using the sc or
      scv instructions.  Because Valgrind does an end-of-block after the
      system call, the contents of a gpr can not be saved and restored after
      the system call.  A custom guest state register guest_syscall_flag is
      used to pass the flag so the guest state is not disturbed.  */

   stmt( IRStmt_Put( offsetofPPCGuestState(guest_syscall_flag), src ) );
}


/*-----------------------------------------------------------*/
/* Helpers to access VSX Accumulator register file
 *-----------------------------------------------------------*/
static UInt ACC_offset( UInt index, UInt reg )
{
#define SizeofACC_row       16     /* size of ACC row in bytes */
#define ACC_row_per_entry   4
#define ACC_num_entries     8

   vassert(index < ACC_num_entries);
   vassert(reg < ACC_row_per_entry);
   return index * ACC_row_per_entry * SizeofACC_row  + reg * SizeofACC_row;
}

static UInt base_acc_addr( Bool ACC_mapped_on_VSR )
{
   /* Return base ACC address if ACC mapped over vsrs or as a separate
      register file.  */
   if ( ACC_mapped_on_VSR )   /* ISA 3.1 implementation */
      return offsetofPPCGuestState( guest_VSR0 );
   else
      return offsetofPPCGuestState( guest_ACC_0_r0 );
}

static void putACC( UInt index, UInt reg, IRExpr* src, Bool ACC_mapped_on_VSR)

{
   stmt( IRStmt_Put( base_acc_addr( ACC_mapped_on_VSR )
                     + ACC_offset( index, reg), src ) );
}

static IRExpr* /* :: Ity_V128 */ getACC ( UInt index, UInt reg,
                                          Bool ACC_mapped_on_VSR)
{
   vassert( (index >= 0) && (index < 8) );
   vassert( (reg >= 0) && (reg < 4) );

   return IRExpr_Get( base_acc_addr( ACC_mapped_on_VSR )
                      + ACC_offset( index, reg), Ity_V128 );
}


/*------------------------------------------------------------*/
/* Helpers for VSX instructions that do floating point
 * operations and need to determine if a src contains a
 * special FP value.
 *
 *------------------------------------------------------------*/

#define NONZERO_FRAC_MASK 0x000fffffffffffffULL
#define FP_FRAC_PART(x) binop( Iop_And64, \
                               mkexpr( x ), \
                               mkU64( NONZERO_FRAC_MASK ) )

#define NONZERO_FRAC_MASK32 0x007fffffULL
#define FP_FRAC_PART32(x) binop( Iop_And32,   \
                                 mkexpr( x ), \
                                 mkU32( NONZERO_FRAC_MASK32 ) )

// Returns exponent part of floating point src as I32
static IRExpr * fp_exp_part( IRType size, IRTemp src )
{
   IRExpr *shift_by, *mask, *tsrc;

   vassert( ( size == Ity_I16 ) || ( size == Ity_I32 )
            || ( size == Ity_I64 ) );

   if( size == Ity_I16 ) {
      /* The 16-bit floating point value is in the lower 16-bits
       * of the 32-bit input value.
       */
      tsrc  = mkexpr( src );
      mask  = mkU32( 0x1F );
      shift_by = mkU8( 10 );

   } else if( size == Ity_I32 ) {
      tsrc  = mkexpr( src );
      mask  = mkU32( 0xFF );
      shift_by = mkU8( 23 );

   } else if( size == Ity_I64 ) {
      tsrc  = unop( Iop_64HIto32, mkexpr( src ) );
      mask  = mkU32( 0x7FF );
      shift_by = mkU8( 52 - 32 );

   } else {
      /*NOTREACHED*/
      vassert(0); // Stops gcc complaining at "-Og"
   }

   return binop( Iop_And32, binop( Iop_Shr32, tsrc, shift_by ), mask );
}

/* The following functions check the floating point value to see if it
   is zero, infinity, NaN, Normalized, Denormalized.
*/
/* 16-bit floating point number is stored in the lower 16-bits of 32-bit value */
#define I16_EXP_MASK       0x7C00
#define I16_FRACTION_MASK  0x03FF
#define I16_MSB_FRACTION_MASK  0x0200
#define I32_EXP_MASK       0x7F800000
#define I32_FRACTION_MASK  0x007FFFFF
#define I32_MSB_FRACTION_MASK  0x00400000
#define I32_SIGN_MASK      0x80000000
#define I64_EXP_MASK       0x7FF0000000000000ULL
#define I64_FRACTION_MASK  0x000FFFFFFFFFFFFFULL
#define I64_MSB_FRACTION_MASK  0x0008000000000000ULL
#define V128_EXP_MASK      0x7FFF000000000000ULL
#define V128_FRACTION_MASK 0x0000FFFFFFFFFFFFULL  /* upper 64-bit fractional mask */
#define V128_MSB_FRACTION_MASK 0x0000800000000000ULL  /* upper 64-bit fractional mask */

void setup_value_check_args( IRType size, IRTemp *exp_mask, IRTemp *frac_mask,
                             IRTemp *msb_frac_mask, IRTemp *zero );

void setup_value_check_args( IRType size, IRTemp *exp_mask, IRTemp *frac_mask,
                             IRTemp *msb_frac_mask, IRTemp *zero ) {

   vassert( ( size == Ity_I16 ) || ( size == Ity_I32 )
            || ( size == Ity_I64 ) || ( size == Ity_V128 ) );

   if( size == Ity_I16 ) {
      /* The 16-bit floating point value is in the lower 16-bits of
         the 32-bit input value */
      *frac_mask = newTemp( Ity_I32 );
      *msb_frac_mask = newTemp( Ity_I32 );
      *exp_mask  = newTemp( Ity_I32 );
      *zero  = newTemp( Ity_I32 );
      assign( *exp_mask, mkU32( I16_EXP_MASK ) );
      assign( *frac_mask, mkU32( I16_FRACTION_MASK ) );
      assign( *msb_frac_mask, mkU32( I16_MSB_FRACTION_MASK ) );
      assign( *zero, mkU32( 0 ) );

   } else if( size == Ity_I32 ) {
      *frac_mask = newTemp( Ity_I32 );
      *msb_frac_mask = newTemp( Ity_I32 );
      *exp_mask  = newTemp( Ity_I32 );
      *zero  = newTemp( Ity_I32 );
      assign( *exp_mask, mkU32( I32_EXP_MASK ) );
      assign( *frac_mask, mkU32( I32_FRACTION_MASK ) );
      assign( *msb_frac_mask, mkU32( I32_MSB_FRACTION_MASK ) );
      assign( *zero, mkU32( 0 ) );

   } else if( size == Ity_I64 ) {
      *frac_mask = newTemp( Ity_I64 );
      *msb_frac_mask = newTemp( Ity_I64 );
      *exp_mask  = newTemp( Ity_I64 );
      *zero  = newTemp( Ity_I64 );
      assign( *exp_mask, mkU64( I64_EXP_MASK ) );
      assign( *frac_mask, mkU64( I64_FRACTION_MASK ) );
      assign( *msb_frac_mask, mkU64( I64_MSB_FRACTION_MASK ) );
      assign( *zero, mkU64( 0 ) );

   } else {
      /* V128 is converted to upper and lower 64 bit values, */
      /* uses 64-bit operators and temps */
      *frac_mask = newTemp( Ity_I64 );
      *msb_frac_mask = newTemp( Ity_I64 );
      *exp_mask  = newTemp( Ity_I64 );
      *zero  = newTemp( Ity_I64 );
      assign( *exp_mask, mkU64( V128_EXP_MASK ) );
      /* upper 64-bit fractional mask */
      assign( *frac_mask, mkU64( V128_FRACTION_MASK ) );
      assign( *msb_frac_mask, mkU64( V128_MSB_FRACTION_MASK ) );
      assign( *zero, mkU64( 0 ) );
   }
}

/* Helper function for the various function which check the value of
   the floating point value.
*/
static IRExpr * exponent_compare( IRType size, IRTemp src,
                                  IRTemp exp_mask, IRExpr *exp_val )
{
   IROp opAND, opCmpEQ;

   if( ( size == Ity_I16 ) || ( size == Ity_I32 ) )  {
      /* The 16-bit floating point value is in the lower 16-bits of
         the 32-bit input value */
      opAND = Iop_And32;
      opCmpEQ = Iop_CmpEQ32;

   } else {
      opAND = Iop_And64;
      opCmpEQ = Iop_CmpEQ64;
   }

   if( size == Ity_V128 ) {
      return binop( opCmpEQ,
                    binop ( opAND,
                            unop( Iop_V128HIto64, mkexpr( src ) ),
                            mkexpr( exp_mask ) ),
                    exp_val );

   } else if( ( size == Ity_I16 ) || ( size == Ity_I32 ) )  {
      return binop( opCmpEQ,
                    binop ( opAND, mkexpr( src ), mkexpr( exp_mask ) ),
                    exp_val );
   } else {
      /* 64-bit operands */

      if (mode64) {
         return binop( opCmpEQ,
                       binop ( opAND, mkexpr( src ), mkexpr( exp_mask ) ),
                       exp_val );
      } else {
         /* No support for 64-bit compares in 32-bit mode, need to do upper
          * and lower parts using 32-bit compare operators.
          */
         return 
            mkAND1( binop( Iop_CmpEQ32,
                           binop ( Iop_And32,
                                   unop(Iop_64HIto32, mkexpr( src ) ),
                                   unop(Iop_64HIto32, mkexpr( exp_mask ) ) ),
                           unop(Iop_64HIto32, exp_val ) ),
                    binop( Iop_CmpEQ32,
                           binop ( Iop_And32,
                                   unop(Iop_64to32, mkexpr( src ) ),
                                   unop(Iop_64to32, mkexpr( exp_mask ) ) ),
                           unop(Iop_64to32, exp_val ) ) );
      }
   }
}

static IRExpr *fractional_part_compare( IRType size, IRTemp src,
                                        IRTemp frac_mask, IRExpr *zero )
{
   IROp opAND, opCmpEQ;

   if( ( size == Ity_I16 ) || ( size == Ity_I32 ) )  {
      /*The 16-bit floating point value is in the lower 16-bits of
        the 32-bit input value */
      opAND = Iop_And32;
      opCmpEQ = Iop_CmpEQ32;

   } else {
      opAND = Iop_And64;
      opCmpEQ = Iop_CmpEQ64;
   }

   if( size == Ity_V128 ) {
      /* 128-bit, note we only care if the fractional part is zero so take upper
         52-bits of fractional part and lower 64-bits and OR them together and test
         for zero.  This keeps the temp variables and operators all 64-bit.
      */
      return binop( opCmpEQ,
                    binop( Iop_Or64,
                           binop( opAND,
                                  unop( Iop_V128HIto64, mkexpr( src ) ),
                                  mkexpr( frac_mask ) ),
                           unop( Iop_V128to64, mkexpr( src ) ) ),
                    zero );

   } else if( ( size == Ity_I16 ) || ( size == Ity_I32 ) )  {
         return binop( opCmpEQ,
                       binop( opAND,  mkexpr( src ), mkexpr( frac_mask ) ),
                       zero );
   } else {
      if (mode64) {
         return binop( opCmpEQ,
                       binop( opAND,  mkexpr( src ), mkexpr( frac_mask ) ),
                       zero );
      } else {
         /* No support for 64-bit compares in 32-bit mode, need to do upper
          * and lower parts using 32-bit compare operators.
          */
         return 
            mkAND1( binop( Iop_CmpEQ32,
                           binop ( Iop_And32,
                                   unop(Iop_64HIto32, mkexpr( src ) ),
                                   unop(Iop_64HIto32, mkexpr( frac_mask ) ) ),
                           mkU32 ( 0 ) ),
                    binop( Iop_CmpEQ32,
                           binop ( Iop_And32,
                                   unop(Iop_64to32, mkexpr( src ) ),
                                   unop(Iop_64to32, mkexpr( frac_mask ) ) ),
                           mkU32 ( 0 ) ) );
      }
   }
}

// Infinity: exp has all bits set, and fraction is zero; s = 0/1
static IRExpr * is_Inf( IRType size, IRTemp src )
{
   IRExpr *max_exp, *zero_frac;
   IRTemp exp_mask, frac_mask, msb_frac_mask, zero;

   setup_value_check_args( size, &exp_mask, &frac_mask, &msb_frac_mask,
                           &zero );

   /* check exponent is all ones, i.e. (exp AND exp_mask) = exp_mask */
   max_exp = exponent_compare( size, src, exp_mask, mkexpr( exp_mask ) );

   /* check fractional part is all zeros */
   zero_frac = fractional_part_compare( size, src, frac_mask, mkexpr( zero ) );

   return  mkAND1( max_exp, zero_frac );
}

// Zero: exp is zero and fraction is zero; s = 0/1
static IRExpr * is_Zero( IRType size, IRTemp src )
{
   IRExpr *zero_exp, *zero_frac;
   IRTemp exp_mask, frac_mask, msb_frac_mask, zero;

   setup_value_check_args( size, &exp_mask, &frac_mask, &msb_frac_mask,
                           &zero );

   /* check the exponent is all zeros, i.e. (exp AND exp_mask) = zero */
   zero_exp = exponent_compare( size, src, exp_mask, mkexpr( zero ) );

   /* check fractional part is all zeros */
   zero_frac = fractional_part_compare( size, src, frac_mask, mkexpr( zero ) );

   return  mkAND1( zero_exp, zero_frac );
}

/*  SNAN: s = 1/0; exp all 1's; fraction is nonzero, with highest bit '1'
 *  QNAN: s = 1/0; exp all 1's; fraction is nonzero, with highest bit '0'
 */
static IRExpr * is_NaN( IRType size, IRTemp src )
{
   IRExpr *max_exp, *not_zero_frac;
   IRTemp exp_mask, frac_mask, msb_frac_mask, zero;

   setup_value_check_args( size, &exp_mask, &frac_mask, &msb_frac_mask,
                           &zero );

   /* check exponent is all ones, i.e. (exp AND exp_mask) = exp_mask */
   max_exp = exponent_compare( size, src, exp_mask, mkexpr( exp_mask ) );

   /* check fractional part is not zero */
   not_zero_frac = unop( Iop_Not1,
                         fractional_part_compare( size, src, frac_mask,
                                                  mkexpr( zero ) ) );

   return  mkAND1( max_exp, not_zero_frac );
}

static IRExpr * is_sNaN( IRType size, IRTemp src )
{
   IRExpr *max_exp, *not_zero_frac, *msb_zero;
   IRTemp exp_mask, frac_mask, msb_frac_mask, zero;

   setup_value_check_args( size, &exp_mask, &frac_mask, &msb_frac_mask,
                           &zero );

   /* check exponent is all ones, i.e. (exp AND exp_mask) = exp_mask */
   max_exp = exponent_compare( size, src, exp_mask, mkexpr( exp_mask ) );

   /* Most significant fractional bit is zero for sNaN */
   msb_zero = fractional_part_compare ( size, src, msb_frac_mask,
                                        mkexpr( zero ) );

   /* check fractional part is not zero */
   not_zero_frac = unop( Iop_Not1,
                         fractional_part_compare( size, src, frac_mask,
                                                  mkexpr( zero ) ) );

   return  mkAND1( msb_zero, mkAND1( max_exp, not_zero_frac ) );
}

/* Denormalized number has a zero exponent and non zero fraction. */
static IRExpr * is_Denorm( IRType size, IRTemp src )
{
   IRExpr *zero_exp, *not_zero_frac;
   IRTemp exp_mask, frac_mask, msb_frac_mask, zero;

   setup_value_check_args( size, &exp_mask, &frac_mask, &msb_frac_mask,
                           &zero );

   /* check exponent is all zeros */
   zero_exp = exponent_compare( size, src, exp_mask, mkexpr( zero ) );

   /* check fractional part is not zero */
   not_zero_frac = unop( Iop_Not1,
                         fractional_part_compare( size, src, frac_mask,
                                                  mkexpr( zero ) ) );

   return  mkAND1( zero_exp, not_zero_frac );
}

static IRExpr * is_Zero_Vector( IRType element_size, IRExpr *src )
{
/* Check elements of a 128-bit floating point vector, with element size are
   zero.  Return 1's in the elements of the vector which are values. */
   IRTemp exp_maskV128 = newTemp( Ity_V128 );
   IRTemp exp_zeroV128 = newTemp( Ity_V128 );
   IRTemp frac_maskV128 = newTemp( Ity_V128 );
   IRTemp frac_zeroV128 = newTemp( Ity_V128 );
   IRTemp zeroV128 = newTemp( Ity_V128 );

   assign( zeroV128, mkV128( 0 ) );

   if ( element_size == Ity_I32 ) {
      assign( exp_maskV128, unop( Iop_Dup32x4, mkU32( I32_EXP_MASK ) ) );
      assign( frac_maskV128, unop( Iop_Dup32x4, mkU32( I32_FRACTION_MASK ) ) );

   } else
      vpanic("ERROR, is_Zero_Vector:  Unknown input size");

   /* CmpEQ32x4 returns all 1's in elements where comparison is true */
   assign( exp_zeroV128,
           binop( Iop_CmpEQ32x4,
                  binop( Iop_AndV128,
                         mkexpr( exp_maskV128 ), src ),
                  mkexpr( zeroV128 ) ) );

   assign( frac_zeroV128,
           binop( Iop_CmpEQ32x4,
                  binop( Iop_AndV128,
                         mkexpr( frac_maskV128 ), src ),
                  mkexpr( zeroV128 ) ) );

   return binop( Iop_AndV128, mkexpr( exp_zeroV128 ),
                 mkexpr( frac_zeroV128 ) );
}

static IRExpr * Abs_Zero_Vector( IRType element_size, IRExpr *src )
/* Vector of four 32-bit elements, convert any negative zeros to
   positive zeros.  */
{
   IRTemp result = newTemp( Ity_V128 );

   if ( element_size == Ity_I32 ) {
      assign( result, binop( Iop_AndV128,
                             src,
                             unop( Iop_NotV128,
                                   is_Zero_Vector( element_size, src) ) ) );
   } else
      vex_printf("ERROR, Abs_Zero_Vector:  Unknown input size\n");

   return mkexpr( result );
}

static IRExpr * is_Denorm_Vector( IRType element_size, IRExpr *src )
{
/* Check elements of a 128-bit floating point vector, with element size
   element_size, are Denorm.  Return 1's in the elements of the vector
   which are denormalized values. */
   IRTemp exp_maskV128 = newTemp( Ity_V128 );
   IRTemp exp_zeroV128 = newTemp( Ity_V128 );
   IRTemp frac_maskV128 = newTemp( Ity_V128 );
   IRTemp frac_nonzeroV128 = newTemp( Ity_V128 );
   IRTemp zeroV128 = newTemp( Ity_V128 );

   assign( zeroV128, mkV128(0 ) );

   if ( element_size == Ity_I32 ) {
      assign( exp_maskV128, unop( Iop_Dup32x4, mkU32( I32_EXP_MASK ) ) );
      assign( frac_maskV128, unop( Iop_Dup32x4, mkU32( I32_FRACTION_MASK ) ) );

   } else
      vpanic("ERROR, is_Denorm_Vector:  Unknown input size");

   /* CmpEQ32x4 returns all 1's in elements where comparison is true */
   assign( exp_zeroV128,
           binop( Iop_CmpEQ32x4,
                  binop( Iop_AndV128,
                         mkexpr( exp_maskV128 ), src ),
                  mkexpr( zeroV128 ) ) );

   assign( frac_nonzeroV128,
           unop( Iop_NotV128,
                 binop( Iop_CmpEQ32x4,
                        binop( Iop_AndV128,
                               mkexpr( frac_maskV128 ), src ),
                        mkexpr( zeroV128 ) ) ) );

   return binop( Iop_AndV128, mkexpr( exp_zeroV128 ),
                 mkexpr( frac_nonzeroV128 ) );
}

static IRExpr * is_NaN_Vector( IRType element_size, IRExpr *src )
{
   IRTemp max_expV128 = newTemp( Ity_V128 );
   IRTemp not_zero_fracV128 = newTemp( Ity_V128 );
   IRTemp zeroV128  = newTemp( Ity_V128 );
   IRTemp exp_maskV128 = newTemp( Ity_V128 );
   IRTemp frac_maskV128 = newTemp( Ity_V128 );
   IROp   opCmpEQ =  Iop_INVALID;

   assign( zeroV128, mkV128( 0 ) );

   if ( element_size == Ity_I32 ) {
      assign( exp_maskV128, unop( Iop_Dup32x4, mkU32( I32_EXP_MASK ) ) );
      assign( frac_maskV128, unop( Iop_Dup32x4, mkU32( I32_FRACTION_MASK ) ) );
      opCmpEQ = Iop_CmpEQ32x4;

   } else
      vpanic("ERROR, is_NaN_Vector:  Unknown input size");

   /* check exponent is all ones, i.e. (exp AND exp_mask) = exp_mask */
   assign( max_expV128,
           binop( opCmpEQ,
                  binop( Iop_AndV128, src, mkexpr( exp_maskV128 ) ),
                  mkexpr( exp_maskV128 ) ) );

   /* check fractional part is not zero */
   assign( not_zero_fracV128,
           unop( Iop_NotV128,
           binop( opCmpEQ,
                        binop( Iop_AndV128, src, mkexpr( frac_maskV128 ) ),
                        mkexpr( zeroV128 ) ) ) );

   return  binop( Iop_AndV128, mkexpr( max_expV128 ),
                  mkexpr( not_zero_fracV128 ) );
}

#if 0
/* Normalized number has exponent between 1 and max_exp -1, or in other words
   the exponent is not zero and not equal to the max exponent value. */
 Currently not needed since generate_C_FPCC is now done with a C helper.
 Keep it around, might be useful in the future.
static IRExpr * is_Norm( IRType size, IRTemp src )
{
   IRExpr *not_zero_exp, *not_max_exp;
   IRTemp exp_mask, zero;

   vassert( ( size == Ity_I16 ) || ( size == Ity_I32 )
            || ( size == Ity_I64 ) || ( size == Ity_V128 ) );

   if( size == Ity_I16 ) {
      /* The 16-bit floating point value is in the lower 16-bits of
         the 32-bit input value */
      exp_mask = newTemp( Ity_I32 );
      zero = newTemp( Ity_I32 );
      assign( exp_mask, mkU32( I16_EXP_MASK ) );
      assign( zero,  mkU32( 0 ) );

   } else if( size == Ity_I32 ) {
      exp_mask = newTemp( Ity_I32 );
      zero = newTemp( Ity_I32 );
      assign( exp_mask, mkU32( I32_EXP_MASK ) );
      assign( zero, mkU32( 0 ) );

   } else if( size == Ity_I64 ) {
      exp_mask = newTemp( Ity_I64 );
      zero = newTemp( Ity_I64 );
      assign( exp_mask, mkU64( I64_EXP_MASK ) );
      assign( zero, mkU64( 0 ) );

   } else {
      /* V128 is converted to upper and lower 64 bit values, */
      /* uses 64-bit operators and temps */
      exp_mask = newTemp( Ity_I64 );
      zero = newTemp( Ity_I64 );
      assign( exp_mask, mkU64( V128_EXP_MASK ) );
      assign( zero, mkU64( 0 ) );
   }

   not_zero_exp = unop( Iop_Not1,
                        exponent_compare( size, src,
                                          exp_mask, mkexpr( zero ) ) );
   not_max_exp = unop( Iop_Not1,
                       exponent_compare( size, src,
                                         exp_mask, mkexpr( exp_mask ) ) );

   return  mkAND1( not_zero_exp, not_max_exp );
}
#endif

static void generate_store_FPRF( IRType size, IRTemp src,
				 const VexAbiInfo* vbi )
{

  /* This function was originally written using IR code.  It has been
   * replaced with a clean helper due to the large amount of IR code
   * needed by this function.
   */

   IRTemp tmp = newTemp( Ity_I64 );
   vassert( ( size == Ity_I16 ) || ( size == Ity_I32 )
            || ( size == Ity_I64 ) || ( size == Ity_F128 ) );

   vassert( ( typeOfIRExpr(irsb->tyenv, mkexpr( src ) )   == Ity_I32 )
            || ( typeOfIRExpr(irsb->tyenv, mkexpr( src ) ) == Ity_I64 )
            || ( typeOfIRExpr(irsb->tyenv, mkexpr( src ) ) == Ity_F128 ) );

   if( size == Ity_I16 ) {
      assign( tmp,
              mkIRExprCCall( Ity_I64, 0 /*regparms*/,
                             "generate_store_C_FPCC_helper",
                             fnptr_to_fnentry( vbi, &generate_C_FPCC_helper ),
                             mkIRExprVec_3( mkU64( size ), mkU64( 0 ),
                                            mkexpr( src ) ) ) );
   } else if( size == Ity_I32 ) {
      assign( tmp,
              mkIRExprCCall( Ity_I64, 0 /*regparms*/,
                             "generate_store_C_FPCC_helper",
                             fnptr_to_fnentry( vbi, &generate_C_FPCC_helper ),
                             mkIRExprVec_3( mkU64( size ), mkU64( 0 ),
                                            mkexpr( src ) ) ) );
   } else if( size == Ity_I64 ) {
      assign( tmp,
              mkIRExprCCall( Ity_I64, 0 /*regparms*/,
                             "generate_store_C_FPCC_helper",
                             fnptr_to_fnentry( vbi, &generate_C_FPCC_helper ),
                             mkIRExprVec_3( mkU64( size ), mkU64( 0 ),
                                            mkexpr( src ) ) ) );
   } else if( size == Ity_F128 ) {
      assign( tmp,
              mkIRExprCCall( Ity_I64, 0 /*regparms*/,
                             "generate_store_C_FPCC_helper",
                             fnptr_to_fnentry( vbi, &generate_C_FPCC_helper ),
                             mkIRExprVec_3( mkU64( size ),
                                            unop( Iop_ReinterpF64asI64,
                                                  unop( Iop_F128HItoF64,
                                                        mkexpr( src ) ) ),
                                            unop( Iop_ReinterpF64asI64,
                                                  unop( Iop_F128LOtoF64,
                                                        mkexpr( src ) ) ) ) ) );
   }

   /* C is in the upper 32-bits, FPCC is in the lower 32-bits of the
    * value returned by the helper function
    */
   putC( unop(  Iop_64HIto32, mkexpr( tmp) ) );
   putFPCC( unop(  Iop_64to32, mkexpr( tmp) ) );
}

/* This function takes an Ity_I32 input argument interpreted
   as a single-precision floating point value. If src is a
   SNaN, it is changed to a QNaN and returned; otherwise,
   the original value is returned. */
static IRExpr * handle_SNaN_to_QNaN_32(IRExpr * src)
{
#define SNAN_MASK32 0x00400000
   IRTemp tmp = newTemp(Ity_I32);
   IRTemp mask = newTemp(Ity_I32);
   IRTemp is_SNAN = newTemp(Ity_I1);

   vassert( typeOfIRExpr(irsb->tyenv, src ) == Ity_I32 );
   assign(tmp, src);

   /* check if input is SNaN, if it is convert to QNaN */
   assign( is_SNAN,
           mkAND1( is_NaN( Ity_I32, tmp ),
                   binop( Iop_CmpEQ32,
                          binop( Iop_And32, mkexpr( tmp ),
                                 mkU32( SNAN_MASK32 ) ),
                          mkU32( 0 ) ) ) );
   /* create mask with QNaN bit set to make it a QNaN if tmp is SNaN */
   assign ( mask, binop( Iop_And32,
                         unop( Iop_1Sto32, mkexpr( is_SNAN ) ),
                         mkU32( SNAN_MASK32 ) ) );
   return binop( Iop_Or32, mkexpr( mask ), mkexpr( tmp) );
}


/* This helper function performs the negation part of operations of the form:
 *    "Negate Multiply-<op>"
 *  where "<op>" is either "Add" or "Sub".
 *
 * This function takes one argument -- the floating point intermediate result (converted to
 * Ity_I64 via Iop_ReinterpF64asI64) that was obtained from the "Multip-<op>" part of
 * the operation described above.
 */
static IRTemp getNegatedResult(IRTemp intermediateResult)
{
   ULong signbit_mask = 0x8000000000000000ULL;
   IRTemp signbit_32 = newTemp(Ity_I32);
   IRTemp resultantSignbit = newTemp(Ity_I1);
   IRTemp negatedResult = newTemp(Ity_I64);
   assign( signbit_32, binop( Iop_Shr32,
                          unop( Iop_64HIto32,
                                 binop( Iop_And64, mkexpr( intermediateResult ),
                                        mkU64( signbit_mask ) ) ),
                                 mkU8( 31 ) ) );
   /* We negate the signbit if and only if the intermediate result from the
    * multiply-<op> was NOT a NaN.  This is an XNOR predicate.
    */
   assign( resultantSignbit,
        unop( Iop_Not1,
              binop( Iop_CmpEQ32,
                     binop( Iop_Xor32,
                            mkexpr( signbit_32 ),
                            unop( Iop_1Uto32, is_NaN( Ity_I64,
                                                      intermediateResult ) ) ),
                     mkU32( 1 ) ) ) );

   assign( negatedResult,
        binop( Iop_Or64,
               binop( Iop_And64,
                      mkexpr( intermediateResult ),
                      mkU64( ~signbit_mask ) ),
               binop( Iop_32HLto64,
                      binop( Iop_Shl32,
                             unop( Iop_1Uto32, mkexpr( resultantSignbit ) ),
                             mkU8( 31 ) ),
                      mkU32( 0 ) ) ) );

   return negatedResult;
}

/* This helper function performs the negation part of operations of the form:
 *    "Negate Multiply-<op>"
 *  where "<op>" is either "Add" or "Sub".
 *
 * This function takes one argument -- the floating point intermediate result (converted to
 * Ity_I32 via Iop_ReinterpF32asI32) that was obtained from the "Multip-<op>" part of
 * the operation described above.
 */
static IRTemp getNegatedResult_32(IRTemp intermediateResult)
{
   UInt signbit_mask = 0x80000000;
   IRTemp signbit_32 = newTemp(Ity_I32);
   IRTemp resultantSignbit = newTemp(Ity_I1);
   IRTemp negatedResult = newTemp(Ity_I32);
   assign( signbit_32, binop( Iop_Shr32,
                                 binop( Iop_And32, mkexpr( intermediateResult ),
                                        mkU32( signbit_mask ) ),
                                 mkU8( 31 ) ) );
   /* We negate the signbit if and only if the intermediate result from the
    * multiply-<op> was NOT a NaN.  This is an XNOR predicate.
    */
   assign( resultantSignbit,
        unop( Iop_Not1,
              binop( Iop_CmpEQ32,
                     binop( Iop_Xor32,
                            mkexpr( signbit_32 ),
                            unop( Iop_1Uto32, is_NaN( Ity_I32,
                                                      intermediateResult ) ) ),
                     mkU32( 1 ) ) ) );

   assign( negatedResult,
           binop( Iop_Or32,
                  binop( Iop_And32,
                         mkexpr( intermediateResult ),
                         mkU32( ~signbit_mask ) ),
                  binop( Iop_Shl32,
                         unop( Iop_1Uto32, mkexpr( resultantSignbit ) ),
                         mkU8( 31 ) ) ) );

   return negatedResult;
}

static IRExpr* negate_Vector ( IRType element_size, IRExpr* value )
{
   /* This function takes a vector of floats.  If the value is
      not a NaN, the value is negated.  */

   IRTemp not_nan_mask = newTemp( Ity_V128 );
   IRTemp sign_maskV128 = newTemp( Ity_V128 );

   if ( element_size == Ity_I32 ) {
      assign( sign_maskV128, unop( Iop_Dup32x4, mkU32( I32_SIGN_MASK ) ) );

   } else
      vpanic("ERROR, negate_Vector:  Unknown input size");

   /* Determine if vector elementes are not a NaN, negate sign bit
      for non NaN elements */
   assign ( not_nan_mask,
            unop( Iop_NotV128, is_NaN_Vector( element_size, value ) ) );

   return binop( Iop_XorV128,
                 binop( Iop_AndV128,
                        mkexpr( sign_maskV128 ), mkexpr( not_nan_mask ) ),
                 value );
}

/* This function takes two quad_precision unsigned/signed integers of type
   V128 and return 1 (Ity_Bit) if src_A = src_B, 0 otherwise. */
static IRExpr * Quad_precision_int_eq ( IRTemp src_A, IRTemp  src_B )
{
   return  mkAND1( binop( Iop_CmpEQ64,
                          unop( Iop_V128HIto64, mkexpr( src_A ) ),
                          unop( Iop_V128HIto64, mkexpr( src_B ) ) ),
                   binop( Iop_CmpEQ64,
                          unop( Iop_V128to64, mkexpr( src_A ) ),
                          unop( Iop_V128to64, mkexpr( src_B ) ) ) );
}

/* This function takes two quad_precision unsigned integers of type
   V128 and return 1 if src_A > src_B, 0 otherwise. */
static IRExpr * Quad_precision_uint_gt ( IRTemp src_A, IRTemp  src_B )
{
   IRExpr * hi_eq = binop( Iop_CmpEQ64,
                           unop( Iop_V128HIto64, mkexpr( src_A ) ),
                           unop( Iop_V128HIto64, mkexpr( src_B ) ) );

   IRExpr * hi_gt = binop( Iop_CmpLT64U,
                           unop( Iop_V128HIto64, mkexpr( src_B ) ),
                           unop( Iop_V128HIto64, mkexpr( src_A ) ) );

   IRExpr * lo_gt = binop( Iop_CmpLT64U,
                           unop( Iop_V128to64, mkexpr( src_B ) ),
                           unop( Iop_V128to64, mkexpr( src_A ) ) );

   return mkOR1( hi_gt, mkAND1( hi_eq, lo_gt ) );
}

/* This function takes two quad_precision signed integers of type
   V128 and return 1 if src_A > src_B, 0 otherwise. */
static IRExpr * Quad_precision_sint_gt ( IRTemp src_A, IRTemp  src_B )
{

   IRExpr * hi_eq = binop( Iop_CmpEQ64,
                           unop( Iop_V128HIto64, mkexpr( src_A ) ),
                           unop( Iop_V128HIto64, mkexpr( src_B ) ) );

   IRExpr * lo_eq = binop( Iop_CmpEQ64,
                           unop( Iop_V128to64, mkexpr( src_A ) ),
                           unop( Iop_V128to64, mkexpr( src_B ) ) );

   IRExpr * hi_gt = binop( Iop_CmpLT64S,
                           unop( Iop_V128HIto64, mkexpr( src_B ) ),
                           unop( Iop_V128HIto64, mkexpr( src_A ) ) );

/* If srcA and srcB are positive and srcA > srcB then lo_gteq = 1.
   If srcA and srcB are negative and srcA > srcB, then the unsigned value
   of the lower 64-bits are 2's complemented values means lower bits of srcB
   must be less then the lower bits of srcA.

     srcA = 8000012380000123 7000000080000000 =>     (smaller/less negative)
                           - 7FFFFEDC7FFFFEDD 8FFFFFFF7FFFFFFF
     srcB = 8000012380000123 8000012380000123 =>
                           - 7FFFFEDC7FFFFEDD 7FFFFEDC7FFFFEDD
*/
   IRExpr * lo_gteq =  binop( Iop_CmpLT64U,
                            unop( Iop_V128to64, mkexpr( src_B ) ),
                            unop( Iop_V128to64, mkexpr( src_A ) ) );

   /* If hi is eq, then lower must be GT and not equal.  */
   return mkOR1( hi_gt, mkAND1( hi_eq, mkAND1(  lo_gteq, mkNOT1 ( lo_eq ) ) ) );
}

/* This function takes two quad_precision floating point numbers of type
   V128 and return 1 if src_A > src_B, 0 otherwise. */
static IRExpr * Quad_precision_gt ( IRTemp src_A, IRTemp  src_B )
{
#define FRAC_MASK64Hi 0x0000ffffffffffffULL
#define MASK 0x7FFFFFFFFFFFFFFFULL    /* exclude sign bit in upper 64 bits */
#define EXP_MASK 0x7fff

   IRType ty = Ity_I64;
   IRTemp sign_A = newTemp( ty );
   IRTemp sign_B = newTemp( ty );
   IRTemp exp_A = newTemp( ty );
   IRTemp exp_B = newTemp( ty );
   IRTemp frac_A_hi = newTemp( ty );
   IRTemp frac_B_hi = newTemp( ty );
   IRTemp frac_A_lo = newTemp( ty );
   IRTemp frac_B_lo = newTemp( ty );


   /* extract exponents, and fractional parts so they can be compared */
   assign( sign_A, binop( Iop_Shr64,
                          unop( Iop_V128HIto64, mkexpr( src_A ) ),
                          mkU8( 63 ) ) );
   assign( sign_B, binop( Iop_Shr64,
                          unop( Iop_V128HIto64, mkexpr( src_B ) ),
                          mkU8( 63 ) ) );
   assign( exp_A, binop( Iop_And64,
                         binop( Iop_Shr64,
                                unop( Iop_V128HIto64, mkexpr( src_A ) ),
                                mkU8( 48 ) ),
                         mkU64( EXP_MASK ) ) );
   assign( exp_B, binop( Iop_And64,
                         binop( Iop_Shr64,
                                unop( Iop_V128HIto64, mkexpr( src_B ) ),
                                mkU8( 48 ) ),
                         mkU64( EXP_MASK ) ) );
   assign( frac_A_hi, binop( Iop_And64,
                             unop( Iop_V128HIto64, mkexpr( src_A ) ),
                             mkU64( FRAC_MASK64Hi ) ) );
   assign( frac_B_hi, binop( Iop_And64,
                             unop( Iop_V128HIto64, mkexpr( src_B ) ),
                             mkU64( FRAC_MASK64Hi ) ) );
   assign( frac_A_lo, unop( Iop_V128to64, mkexpr( src_A ) ) );
   assign( frac_B_lo, unop( Iop_V128to64, mkexpr( src_B ) ) );

   IRExpr * A_zero =  mkAND1( binop( Iop_CmpEQ64,
                                     binop( Iop_And64,
                                            unop( Iop_V128HIto64,
                                                  mkexpr( src_A ) ),
                                            mkU64( MASK ) ),
                                     mkU64( 0 ) ),
                              binop( Iop_CmpEQ64,
                                     unop( Iop_V128to64, mkexpr( src_A ) ),
                                     mkU64( 0 ) ) );
   IRExpr * B_zero =  mkAND1( binop( Iop_CmpEQ64,
                                     binop( Iop_And64,
                                            unop( Iop_V128HIto64,
                                                  mkexpr( src_B ) ),
                                            mkU64( MASK ) ),
                                     mkU64( 0 ) ),
                              binop( Iop_CmpEQ64,
                                     unop( Iop_V128to64, mkexpr( src_B ) ),
                                     mkU64( 0 ) ) );
   IRExpr * A_B_zero = mkAND1( A_zero, B_zero );

   /* Compare numbers */
   IRExpr * both_pos = mkAND1( binop( Iop_CmpEQ64, mkexpr( sign_A ),
                                      mkU64( 0 ) ),
                               binop( Iop_CmpEQ64, mkexpr( sign_B ),
                                      mkU64( 0 ) ) );
   IRExpr * both_neg = mkAND1( binop( Iop_CmpEQ64, mkexpr( sign_A ),
                                      mkU64( 1 ) ),
                               binop( Iop_CmpEQ64, mkexpr( sign_B ),
                                      mkU64( 1 ) ) );
   IRExpr * sign_eq = binop( Iop_CmpEQ64, mkexpr( sign_A ), mkexpr( sign_B ) );
   IRExpr * sign_gt = binop( Iop_CmpLT64U, mkexpr( sign_A ),
                             mkexpr( sign_B ) ); /* A pos, B neg */

   IRExpr * exp_eq = binop( Iop_CmpEQ64, mkexpr( exp_A ), mkexpr( exp_B ) );
   IRExpr * exp_gt = binop( Iop_CmpLT64U, mkexpr( exp_B ), mkexpr( exp_A ) );
   IRExpr * exp_lt = binop( Iop_CmpLT64U, mkexpr( exp_A ), mkexpr( exp_B ) );

   IRExpr * frac_hi_eq = binop( Iop_CmpEQ64, mkexpr( frac_A_hi),
                                mkexpr( frac_B_hi ) );
   IRExpr * frac_hi_gt = binop( Iop_CmpLT64U, mkexpr( frac_B_hi ),
                                mkexpr( frac_A_hi ) );
   IRExpr * frac_hi_lt = binop( Iop_CmpLT64U, mkexpr( frac_A_hi ),
                                mkexpr( frac_B_hi ) );

   IRExpr * frac_lo_gt = binop( Iop_CmpLT64U, mkexpr( frac_B_lo ),
                                mkexpr( frac_A_lo ) );
   IRExpr * frac_lo_lt = binop( Iop_CmpLT64U, mkexpr( frac_A_lo ),
                                mkexpr( frac_B_lo ) );

   /* src_A and src_B both positive */
   IRExpr *pos_cmp = mkOR1( exp_gt,
                            mkAND1( exp_eq,
                                    mkOR1( frac_hi_gt,
                                           mkAND1( frac_hi_eq, frac_lo_gt ) )
                                    ) );

   /* src_A and src_B both negative */
   IRExpr *neg_cmp = mkOR1( exp_lt,
                            mkAND1( exp_eq,
                                    mkOR1( frac_hi_lt,
                                           mkAND1( frac_hi_eq, frac_lo_lt ) )
                                    ) );

   /* Need to check the case where one value is a positive
    * zero and the other value is a negative zero
    */
   return mkAND1( mkNOT1( A_B_zero ),
                  mkOR1( sign_gt,
                         mkAND1( sign_eq,
                                 mkOR1( mkAND1( both_pos, pos_cmp ),
                                        mkAND1( both_neg, neg_cmp ) ) ) ) );
}

/*-----------------------------------------------------------
 * Helpers for VX instructions that work on National decimal values,
 * Zoned decimal values and BCD values.
 *
 *------------------------------------------------------------*/
static IRExpr * is_National_decimal (IRTemp src)
{
   /* The src is a 128-bit value containing a sign code in half word 7
    * and seven digits in halfwords 0 to 6 (IBM numbering).  A valid
    * national decimal value has the following:
    *   - the sign code must be 0x002B (positive) or 0x002D (negative)
    *   - the digits must be in the range 0x0030 to 0x0039
    */
   Int i;
   IRExpr * valid_pos_sign;
   IRExpr * valid_neg_sign;
   IRTemp valid_num[8];
   IRTemp digit[7];

   valid_pos_sign = binop( Iop_CmpEQ64,
                           binop( Iop_And64,
                                  mkU64( 0xFFFF ),
                                  unop( Iop_V128to64, mkexpr( src ) ) ),
                           mkU64( 0x002B ) );

   valid_neg_sign = binop( Iop_CmpEQ64,
                           binop( Iop_And64,
                                  mkU64( 0xFFFF ),
                                  unop( Iop_V128to64, mkexpr( src ) ) ),
                           mkU64( 0x002D ) );

   valid_num[0] = newTemp( Ity_I1 );
   digit[0] = newTemp( Ity_I64 );
   assign( valid_num[0], mkU1( 1 ) );   // Assume true to start

   for(i = 0; i < 7; i++) {
      valid_num[i+1] = newTemp( Ity_I1 );
      digit[i] = newTemp( Ity_I64 );
      assign( digit[i], binop( Iop_And64,
                               unop( Iop_V128to64,
                                     binop( Iop_ShrV128,
                                            mkexpr( src ),
                                            mkU8( (7-i)*16 ) ) ),
                               mkU64( 0xFFFF ) ) );

      assign( valid_num[i+1],
              mkAND1( mkexpr( valid_num[i] ),
                      mkAND1( binop( Iop_CmpLE64U,
                                     mkexpr( digit[i] ),
                                     mkU64( 0x39 ) ),
                              binop( Iop_CmpLE64U,
                                     mkU64( 0x30 ),
                                     mkexpr( digit[i] ) ) ) ) );
   }

   return mkAND1( mkOR1( valid_pos_sign, valid_neg_sign),
                  mkexpr( valid_num[7] ) );
}

static IRExpr * is_Zoned_decimal (IRTemp src, UChar ps)
{
   /* The src is a 128-bit value containing a sign code the least significant
    * two bytes. The upper pairs of bytes contain digits.  A valid Zoned
    * decimal value has the following:
    *   - the sign code must be between 0x0X to 0xFX inclusive (X - don't care)
    *   - bits [0:3] of each digit must be equal to 0x3
    *   - bits [4:7] of each digit must be between 0x0 and 0x9
    *
    *  If ps = 0
    *     Positive sign codes are: 0x0, 0x1, 0x2, 0x3, 0x8, 0x9, 0xA, 0xB
    *       (note 0bX0XX XXXX  is positive)
    *
    *     Negative sign codes are 0x4, 0x5, 0x6, 0x7, 0xC, 0xD, 0xE, 0xF
    *       (note 0bX1XX XXXX  is negative)
    *
    *  If ps = 1, then the sign code must be in the range 0xA to 0xF
    *     Positive sign codes are: 0xA, 0xC, 0xE, 0xF
    *
    *     Negative sign codes are 0xB, 0xD
    */
   Int i, mask_hi, mask_lo;
   IRExpr *valid_range;
   IRTemp valid_num[16];
   IRTemp digit[15];

   /* check the range of the sign value based on the value of ps */
   valid_range = mkOR1(
                       mkAND1( binop( Iop_CmpEQ64,
                                      mkU64( 1 ),
                                      mkU64( ps ) ),
                               mkAND1( binop( Iop_CmpLE64U,
                                              binop( Iop_And64,
                                                     mkU64( 0xF0 ),
                                                     unop( Iop_V128to64,
                                                           mkexpr( src ) ) ),

                                              mkU64( 0xF0 ) ),
                                       binop( Iop_CmpLE64U,
                                              mkU64( 0xA0 ),
                                              binop( Iop_And64,
                                                     mkU64( 0xF0 ),
                                                     unop( Iop_V128to64,
                                                           mkexpr( src ) ))))),
                       binop( Iop_CmpEQ64,
                              mkU64( 0 ),
                              mkU64( ps ) ) );

   valid_num[0] = newTemp( Ity_I1 );
   assign( valid_num[0], mkU1( 1) );   // Assume true to start

   if (ps == 0) {
      mask_hi = 0x39;
      mask_lo = 0x30;
   } else {
      mask_hi = 0xF9;
      mask_lo = 0xF0;
   }

   for(i = 0; i < 15; i++) {
      valid_num[i+1] = newTemp( Ity_I1 );
      digit[i] = newTemp( Ity_I64 );
      assign( digit[i], binop( Iop_And64,
                               unop( Iop_V128to64,
                                     binop( Iop_ShrV128,
                                            mkexpr( src ),
                                            mkU8( (15-i)*8 ) ) ),
                               mkU64( 0xFF ) ) );

      assign( valid_num[i+1],
              mkAND1( mkexpr( valid_num[i] ),
                      mkAND1( binop( Iop_CmpLE64U,
                                     mkexpr( digit[i] ),
                                     mkU64( mask_hi ) ),
                              binop( Iop_CmpLE64U,
                                     mkU64( mask_lo ),
                                     mkexpr( digit[i] ) ) ) ) );
   }

   return mkAND1( valid_range, mkexpr( valid_num[15] ) );
}

static IRExpr * CmpGT128U ( IRExpr *src1, IRExpr *src2 )
{
   /* Unsigend compare of two 128-bit values */
   IRExpr *pos_upper_gt, *pos_upper_eq, *pos_lower_gt;

   pos_upper_gt = binop( Iop_CmpLT64U,
                         unop( Iop_V128HIto64, src2 ),
                         unop( Iop_V128HIto64, src1 ) );
   pos_upper_eq = binop( Iop_CmpEQ64,
                         unop( Iop_V128HIto64, src1 ),
                         unop( Iop_V128HIto64, src2 ) );
   pos_lower_gt = binop( Iop_CmpLT64U,
                         unop( Iop_V128to64, src2),
                         unop( Iop_V128to64, src1) );
   return mkOR1( pos_upper_gt,
                 mkAND1( pos_upper_eq,
                         pos_lower_gt ) );
}


static IRExpr * is_BCDstring128 ( const VexAbiInfo* vbi,
                                  UInt Signed, IRExpr *src )
{

   IRTemp valid = newTemp( Ity_I64 );

   /* The src is a 128-bit value containing a MAX_DIGITS BCD digits and
    * a sign. The upper bytes are BCD values between 0x0 and 0x9.  The sign
    * byte is the least significant byte. This function returns 64-bit 1
    * value if sign and digits are valid, 0 otherwise.
    *
    * This function was originally written using IR code.  It has been
    * replaced with a clean helper due to the large amount of IR code
    * needed by this function.
    */
   assign( valid,
           mkIRExprCCall( Ity_I64, 0 /*regparms*/,
                          "is_BCDstring128_helper",
                          fnptr_to_fnentry( vbi, &is_BCDstring128_helper ),
                          mkIRExprVec_3( mkU64( Signed ),
                                         unop( Iop_V128HIto64, src ),
                                         unop( Iop_V128to64, src ) ) ) );
   return mkexpr( valid );
}

static IRExpr * BCDstring_zero (IRExpr *src)
{
   /* The src is a 128-bit value containing a BCD string.  The function
    * returns a 1 if the BCD string values are all zero, 0 otherwise.
    */
   IRTemp tsrc = newTemp( Ity_V128 );
   assign( tsrc, src);

   if ( mode64 ) {
      return mkAND1( binop( Iop_CmpEQ64,
                            mkU64( 0 ),
                            unop( Iop_V128HIto64,
                                  mkexpr( tsrc ) ) ),
                     binop( Iop_CmpEQ64,
                            mkU64( 0 ),
                            unop( Iop_V128to64,
                                  mkexpr( tsrc ) ) ) );
   } else {
      /* make this work in 32-bit mode */
      return mkAND1(
                    mkAND1( binop( Iop_CmpEQ32,
                                   mkU32( 0 ),
                                   unop( Iop_64HIto32,
                                         unop( Iop_V128HIto64,
                                               mkexpr( tsrc ) ) ) ),
                            binop( Iop_CmpEQ32,
                                   mkU32( 0 ),
                                   unop( Iop_64to32,
                                         unop( Iop_V128HIto64,
                                               mkexpr( tsrc ) ) ) ) ),
                    mkAND1( binop( Iop_CmpEQ32,
                                   mkU32( 0 ),
                                   unop( Iop_64HIto32,
                                         unop( Iop_V128to64,
                                               mkexpr( tsrc ) ) ) ),
                            binop( Iop_CmpEQ32,
                                   mkU32( 0 ),
                                   unop( Iop_64to32,
                                         unop( Iop_V128to64,
                                               mkexpr( tsrc ) ) ) ) ) );
   }
}

static IRExpr * check_BCD_round (IRExpr *src, IRTemp shift)
{
   /* The src is a 128-bit value containing 31 BCD digits with the sign in
    * the least significant byte. The bytes are BCD values between 0x0 and 0x9.
    * This routine checks the BCD digit in position shift (counting from
    * the least significant digit). If the digit is greater then five,
    * a 1 is returned indicating the string needs to be rounded up,
    * otherwise, 0 is returned.  The value of shift (I64) is the index of
    * the BCD digit times four bits.
    */
   return  binop( Iop_CmpLE64U,
                  mkU64( 6 ),
                  binop( Iop_And64,
                         unop( Iop_V128to64,
                               binop( Iop_ShrV128,
                                      src,
                                      unop( Iop_64to8, mkexpr( shift ) ) ) ),
                         mkU64( 0xF ) ) );
}

static IRTemp increment_BCDstring ( const VexAbiInfo* vbi,
                                    IRExpr *src, IRExpr *carry_in )
{
   /* The src is a 128-bit value containing 31 BCD digits with the sign in
    * the least significant byte. The bytes are BCD values between 0x0 and 0x9.
    * This function returns the BCD string incremented by 1.
    *
    * Call a clean helper to do the computation as it requires a lot of
    * IR code to do this.
    *
    * The helper function takes a 32-bit BCD string, in a 64-bit value, and
    * increments the string by the 32-bi carry in value.
    *
    * The incremented value is returned in the lower 32-bits of the result.
    * The carry out is returned in bits [35:32] of the result.  The
    * helper function will be called for each of the four 32-bit strings
    * that make up the src string passing the returned carry out to the
    * next call.
    */
   IRTemp bcd_result  = newTemp( Ity_V128 );
   IRTemp bcd_result0 = newTemp( Ity_I64 );
   IRTemp bcd_result1 = newTemp( Ity_I64 );
   IRTemp bcd_result2 = newTemp( Ity_I64 );
   IRTemp bcd_result3 = newTemp( Ity_I64 );
   IRExpr *bcd_string0, *bcd_string1, *bcd_string2, *bcd_string3;

   bcd_string0 = binop( Iop_And64,
                        mkU64( 0xFFFFFFFF ), unop( Iop_V128to64, src ) );
   bcd_string1 = binop( Iop_Shr64, unop( Iop_V128to64, src ), mkU8( 32 ) );
   bcd_string2 = binop( Iop_And64,
                        mkU64( 0xFFFFFFFF ), unop( Iop_V128HIto64, src ) );
   bcd_string3 = binop( Iop_Shr64, unop( Iop_V128HIto64, src ), mkU8( 32 ) );

   assign( bcd_result0,
           mkIRExprCCall( Ity_I64, 0 /*regparms*/,
                          "increment_BCDstring32_helper",
                          fnptr_to_fnentry( vbi,
                                            &increment_BCDstring32_helper ),
                          mkIRExprVec_3( mkU64( True /*Signed*/ ),
                                         bcd_string0,
                                         binop( Iop_32HLto64, mkU32( 0 ),
                                                carry_in ) ) ) );

   assign( bcd_result1,
           mkIRExprCCall( Ity_I64, 0 /*regparms*/,
                          "increment_BCDstring32_helper",
                          fnptr_to_fnentry( vbi,
                                            &increment_BCDstring32_helper ),
                          mkIRExprVec_3( mkU64( False /*Unsigned*/ ),
                                         bcd_string1,
                                         binop( Iop_Shr64,
                                                mkexpr( bcd_result0 ),
                                                mkU8( 32 ) ) ) ) );
   assign( bcd_result2,
           mkIRExprCCall( Ity_I64, 0 /*regparms*/,
                          "increment_BCDstring32_helper",
                          fnptr_to_fnentry( vbi,
                                            &increment_BCDstring32_helper ),
                          mkIRExprVec_3( mkU64( False /*Unsigned*/ ),
                                         bcd_string2,
                                         binop( Iop_Shr64,
                                                mkexpr( bcd_result1 ),
                                                mkU8( 32 ) ) ) ) );
   assign( bcd_result3,
           mkIRExprCCall( Ity_I64, 0 /*regparms*/,
                          "increment_BCDstring32_helper",
                          fnptr_to_fnentry( vbi,
                                            &increment_BCDstring32_helper ),
                          mkIRExprVec_3( mkU64( False /*Unsigned*/ ),
                                         bcd_string3,
                                         binop( Iop_Shr64,
                                                mkexpr( bcd_result2 ),
                                                mkU8( 32 ) ) ) ) );

   /* Put the 128-bit result together from the intermediate results. Remember
    * to mask out the carry out from the upper 32 bits of the results.
    */
   assign( bcd_result,
           binop( Iop_64HLtoV128,
                  binop( Iop_Or64,
                         binop( Iop_And64,
                                mkU64( 0xFFFFFFFF ), mkexpr (bcd_result2 ) ),
                         binop( Iop_Shl64,
                                mkexpr (bcd_result3 ), mkU8( 32 ) ) ),
                  binop( Iop_Or64,
                         binop( Iop_And64,
                                mkU64( 0xFFFFFFFF ), mkexpr (bcd_result0 ) ),
                         binop( Iop_Shl64,
                                mkexpr (bcd_result1 ), mkU8( 32 ) ) ) ) );
   return bcd_result;
}

static IRExpr * convert_to_zoned ( const VexAbiInfo* vbi,
                                   IRExpr *src, IRExpr *upper_byte )
{
   /* The function takes a V128 packed decimal value and returns
    * the value in zoned format.  Note, the sign of the value is ignored.
    */
   IRTemp result_low = newTemp( Ity_I64 );
   IRTemp result_hi  = newTemp( Ity_I64 );
   IRTemp result     = newTemp( Ity_V128 );

   /* Since we can only return 64-bits from a clean helper, we will
    * have to get the lower and upper 64-bits separately.
    */

   assign( result_low,
           mkIRExprCCall( Ity_I64, 0 /*regparms*/,
                          "convert_to_zoned_helper",
                          fnptr_to_fnentry( vbi, &convert_to_zoned_helper ),
                          mkIRExprVec_4( unop( Iop_V128HIto64, src ),
                                         unop( Iop_V128to64, src ),
                                         upper_byte,
                                         mkU64( 0 ) ) ) );

   assign( result_hi,
           mkIRExprCCall( Ity_I64, 0 /*regparms*/,
                          "convert_to_zoned_helper",
                          fnptr_to_fnentry( vbi, &convert_to_zoned_helper ),
                          mkIRExprVec_4( unop( Iop_V128HIto64, src ),
                                         unop( Iop_V128to64, src ),
                                         upper_byte,
                                         mkU64( 1 ) ) ) );


   assign( result,
           binop( Iop_64HLtoV128, mkexpr( result_hi ), mkexpr( result_low ) ) );

   return mkexpr( result );
}

static IRExpr * convert_to_national ( const VexAbiInfo* vbi,  IRExpr *src ) {
   /* The function takes 128-bit value which has a 64-bit packed decimal
    * value in the lower 64-bits of the source.  The packed decimal is
    * converted to the national format via a clean helper.  The clean
    * helper is used to to the large amount of IR code needed to do the
    * conversion.  The helper returns the upper 64-bits of the 128-bit
    * result if return_upper != 0.  Otherwise, the lower 64-bits of the
    * result is returned.
    */
   IRTemp result_low = newTemp( Ity_I64 );
   IRTemp result_hi  = newTemp( Ity_I64 );
   IRTemp result     = newTemp( Ity_V128 );

   /* Since we can only return 64-bits from a clean helper, we will
    * have to get the lower and upper 64-bits separately.
    */

   assign( result_low,
           mkIRExprCCall( Ity_I64, 0 /*regparms*/,
                          "convert_to_national_helper",
                          fnptr_to_fnentry( vbi, &convert_to_national_helper ),
                          mkIRExprVec_2( unop( Iop_V128to64, src ),
                                         mkU64( 0 ) ) ) );

   assign( result_hi,
           mkIRExprCCall( Ity_I64, 0 /*regparms*/,
                          "convert_to_national_helper",
                          fnptr_to_fnentry( vbi, &convert_to_national_helper ),
                          mkIRExprVec_2( unop( Iop_V128to64, src ),
                                         mkU64( 1 ) ) ) );

   assign( result,
           binop( Iop_64HLtoV128, mkexpr( result_hi ), mkexpr( result_low ) ) );

   return mkexpr( result );
}

static IRExpr * convert_from_zoned ( const VexAbiInfo* vbi, IRExpr *src ) {
   /* The function takes 128-bit zoned value and returns a signless 64-bit
    * packed decimal value in the lower 64-bits of the 128-bit result.
    */
   IRTemp result = newTemp( Ity_V128 );

   assign( result,
           binop( Iop_ShlV128,
                  binop( Iop_64HLtoV128,
                         mkU64( 0 ),
                         mkIRExprCCall( Ity_I64, 0 /*regparms*/,
                                        "convert_from_zoned_helper",
                                        fnptr_to_fnentry( vbi,
                                                          &convert_from_zoned_helper ),
                                        mkIRExprVec_2( unop( Iop_V128HIto64,
                                                             src ),
                                                       unop( Iop_V128to64,
                                                             src ) ) ) ),
                  mkU8( 4 ) ) );

   return mkexpr( result );
}

static IRExpr * convert_from_national ( const VexAbiInfo* vbi, IRExpr *src ) {
   /* The function takes 128-bit national value and returns a 64-bit
    * packed decimal value.
    */
   IRTemp result = newTemp( Ity_I64);

   assign( result,
           mkIRExprCCall( Ity_I64, 0 /*regparms*/,
                          "convert_from_national_helper",
                          fnptr_to_fnentry( vbi,
                                            &convert_from_national_helper ),
                          mkIRExprVec_2( unop( Iop_V128HIto64,
                                               src ),
                                         unop( Iop_V128to64,
                                               src ) ) ) );

   return mkexpr( result );
}

static IRExpr * vector_convert_floattobf16 ( const VexAbiInfo* vbi,
                                             IRExpr *src ) {
   /* The function takes 128-bit value containing four 32-bit floats and
      returns a 128-bit value containint four 16-bit bfloats in the lower
      halfwords. */

   IRTemp resultHi = newTemp( Ity_I64);
   IRTemp resultLo = newTemp( Ity_I64);

   assign( resultHi,
           mkIRExprCCall( Ity_I64, 0 /*regparms*/,
                          "vector_convert_floattobf16_helper",
                          fnptr_to_fnentry( vbi,
                                            &convert_from_floattobf16_helper ),
                          mkIRExprVec_1( unop( Iop_V128HIto64, src ) ) ) );

   assign( resultLo,
           mkIRExprCCall( Ity_I64, 0 /*regparms*/,
                          "vector_convert_floattobf16_helper",
                          fnptr_to_fnentry( vbi,
                                            &convert_from_floattobf16_helper ),
                          mkIRExprVec_1( unop( Iop_V128to64, src ) ) ) );

   return binop( Iop_64HLtoV128, mkexpr( resultHi ), mkexpr( resultLo ) );
}

static IRExpr * vector_convert_bf16tofloat ( const VexAbiInfo* vbi,
                                             IRExpr *src ) {
   /* The function takes 128-bit value containing four 16-bit bfloats in
      the lower halfwords and returns a 128-bit value containint four
      32-bit floats. */
   IRTemp resultHi = newTemp( Ity_I64);
   IRTemp resultLo = newTemp( Ity_I64);

   assign( resultHi,
           mkIRExprCCall( Ity_I64, 0 /*regparms*/,
                          "vector_convert_bf16tofloat_helper",
                          fnptr_to_fnentry( vbi,
                                            &convert_from_bf16tofloat_helper ),
                          mkIRExprVec_1( unop( Iop_V128HIto64, src ) ) ) );

   assign( resultLo,
           mkIRExprCCall( Ity_I64, 0 /*regparms*/,
                          "vector_convert_bf16tofloat_helper",
                          fnptr_to_fnentry( vbi,
                                            &convert_from_bf16tofloat_helper ),
                          mkIRExprVec_1( unop( Iop_V128to64, src ) ) ) );

   return binop( Iop_64HLtoV128, mkexpr( resultHi ), mkexpr( resultLo ) );
}

static IRExpr * popcnt64 ( const VexAbiInfo* vbi,
                           IRExpr *src ){
   /* The function takes a 64-bit source and counts the number of bits in the
      source that are 1's.  */
   IRTemp result = newTemp( Ity_I64);

   assign( result,
           mkIRExprCCall( Ity_I64, 0 /*regparms*/,
                          "population_count64_helper",
                          fnptr_to_fnentry( vbi,
                                            &population_count64_helper ),
                          mkIRExprVec_1( src ) ) );

   return mkexpr( result );
}

static IRExpr * extract_bits_under_mask ( const VexAbiInfo* vbi,
                                          IRExpr *src, IRExpr *mask,
                                          IRExpr *flag ) {

   /* The function takes a 64-bit value and a 64-bit mask.  It will extract the
    * bits from the source that align with 1's in the mask or it will extract
    * the bits from the source that align with 0's in the mask.
    */
   IRTemp result = newTemp( Ity_I64);

   assign( result,
           mkIRExprCCall( Ity_I64, 0 /*regparms*/,
                          "extract_bits_under_mask_helper",
                          fnptr_to_fnentry( vbi,
                                            &extract_bits_under_mask_helper ),
                          mkIRExprVec_3( src, mask, flag ) ) );

   return mkexpr( result );
}

static IRExpr * count_bits_under_mask ( const VexAbiInfo* vbi,
                                        IRExpr *src , IRExpr *mask,
                                        IRExpr *flag ) {

   /* The function takes a 64-bit value and a 64-bit mask.  It will count the
    * bits from the source that align with 1's in the mask or it will count
    * the bits from the source that align with 0's in the mask.
    */
   IRTemp result = newTemp( Ity_I32);

   assign( result,
           mkIRExprCCall( Ity_I32, 0 /*regparms*/,
                          "count_bits_under_mask_helper",
                          fnptr_to_fnentry( vbi,
                                            &count_bits_under_mask_helper ),
                          mkIRExprVec_3( src, mask, flag ) ) );

   return mkexpr( result );
}

static IRExpr * deposit_bits_under_mask ( const VexAbiInfo* vbi,
                                          IRExpr *src , IRExpr *mask ) {

   /* The contents of the rightmost n bits of src are placed into bits_rtn
    * under the control of the mask.  The LSB (bit 63) of src is placed into
    * the bit of bits_rtn corresponding to the right most bit of mask that is
    * a 1.  The LSB+1 (bit 62) of src is placed into the bit of bits_rtn
    * corresponding to the second right most bit of mask that is a 1, etc.
    */

   IRTemp result = newTemp( Ity_I64);

   assign( result,
           mkIRExprCCall( Ity_I64, 0 /*regparms*/,
                          "deposit_bits_under_mask_helper",
                          fnptr_to_fnentry( vbi,
                                            &deposit_bits_under_mask_helper ),
                          mkIRExprVec_2( src, mask) ) );

   return mkexpr( result );
}

static IRExpr * vector_evaluate_inst ( const VexAbiInfo* vbi,
                                       IRExpr *srcA, IRExpr *srcB,
                                       IRExpr *srcC, IRExpr *IMM ){
   /* This function implements the ISA 3.1 instruction xxeval.  The
      instruction is too complex to do with Iops.  */

   /* The instruction description, note the IBM bit numbering is left to right:

        For each integer value i, 0 to 127, do the following.

        Let j be the value of the concatenation of the contents of bit i of
        srcA, bit i of srcB, bit i of srcC. (j = srcA[i] | srcB[i] | srcC[i])

        The value of bit IMM[j] is placed into bit result[i].

      Basically the instruction lets you set each of the 128 bits in the result
      by selecting one of the eight bits in the IMM value.  */

   /* Calling clean helpers with 128-bit args is currently not supported. We
      will simply call a 64-bit clean helper to do the upper 64-bits of the
      result and then call it do do the lower 64-bits of the result.  */

   IRTemp result_hi = newTemp( Ity_I64 );
   IRTemp result_lo = newTemp( Ity_I64 );
   IRExpr *srcA_hi;
   IRExpr *srcB_hi;
   IRExpr *srcC_hi;
   IRExpr *srcA_lo;
   IRExpr *srcB_lo;
   IRExpr *srcC_lo;

   srcA_hi = unop( Iop_V128HIto64, srcA );
   srcA_lo = unop( Iop_V128to64, srcA );
   srcB_hi = unop( Iop_V128HIto64, srcB );
   srcB_lo = unop( Iop_V128to64, srcB );
   srcC_hi = unop( Iop_V128HIto64, srcC );
   srcC_lo = unop( Iop_V128to64, srcC );

   assign( result_hi,
           mkIRExprCCall( Ity_I64, 0 /*regparms*/,
                          "vector_evaluate64_helper",
                          fnptr_to_fnentry( vbi,
                                            &vector_evaluate64_helper ),
                          mkIRExprVec_4( srcA_hi, srcB_hi, srcC_hi, IMM ) ) );

   assign( result_lo,
           mkIRExprCCall( Ity_I64, 0 /*regparms*/,
                          "vector_evaluate64_helper",
                          fnptr_to_fnentry( vbi,
                                            &vector_evaluate64_helper ),
                          mkIRExprVec_4( srcA_lo, srcB_lo, srcC_lo, IMM ) ) );

   return binop( Iop_64HLtoV128, mkexpr( result_hi ), mkexpr( result_lo ) );
}

static void setup_fxstate_struct( IRDirty* d, UInt AT, IREffect AT_fx,
                                  Bool ACC_mapped_on_VSR ) {
   UInt acc_base_address;

   /* declare guest state effects, writing to four ACC 128-bit regs. */
   d->nFxState = 4;
   vex_bzero(&d->fxState, sizeof(d->fxState));
   d->fxState[0].fx     = AT_fx;
   d->fxState[0].size   = sizeof(U128);
   d->fxState[1].fx     = AT_fx;
   d->fxState[1].size   = sizeof(U128);
   d->fxState[2].fx     = AT_fx;
   d->fxState[2].size   = sizeof(U128);
   d->fxState[3].fx     = AT_fx;
   d->fxState[3].size   = sizeof(U128);

   vassert( (AT >= 0) && (AT < 8));

   acc_base_address = base_acc_addr( ACC_mapped_on_VSR );

   d->fxState[0].offset = acc_base_address + ACC_offset( AT, 0);
   d->fxState[1].offset = acc_base_address + ACC_offset( AT, 1);
   d->fxState[2].offset = acc_base_address + ACC_offset( AT, 2);
   d->fxState[3].offset = acc_base_address + ACC_offset( AT, 3);
   return;
}
#define MATRIX_4BIT_INT_GER     1
#define MATRIX_8BIT_INT_GER     2
#define MATRIX_16BIT_INT_GER    3
#define MATRIX_16BIT_FLOAT_GER  4
#define MATRIX_32BIT_FLOAT_GER  5
/* Note, the 64-bit float instructions have their caller.  */

static void vsx_matrix_ger ( const VexAbiInfo* vbi,
                             UInt inst_class,
                             IRExpr *srcA, IRExpr *srcB,
                             UInt AT, UInt mask_inst,
                             Bool ACC_mapped_on_VSR) {

   UInt acc_base_addr = base_acc_addr( ACC_mapped_on_VSR );

   /* This helper function does the VSX Matrix 4-bit Signed Integer GER
      (Rank-8 Update) instructions xvi4ger8, xvi4ger8pp, pmxvi4ger8,
      pmxvi4ger8pp.  The instructions work on four V128 values, and three
      8-bit masks.  */

   IRTemp srcA_hi = newTemp( Ity_I64);
   IRTemp srcA_lo = newTemp( Ity_I64);
   IRTemp srcB_hi = newTemp( Ity_I64);
   IRTemp srcB_lo = newTemp( Ity_I64);
   IRDirty* d;
   UInt instruction = mask_inst & 0xFF; /* Instruction is lower 8-bits.  */
   IREffect AT_fx;

   assign( srcA_hi, unop( Iop_V128HIto64, srcA ) );
   assign( srcA_lo, unop( Iop_V128to64, srcA ) );
   assign( srcB_hi, unop( Iop_V128HIto64, srcB ) );
   assign( srcB_lo, unop( Iop_V128to64, srcB ) );

   /* Using a dirty helper so we can access the contents of the ACC for use in
      by the instruction and then write the result directly back to the ACC.
      The dirty helper does not return data.  */
   IRExpr** args = mkIRExprVec_7(
      IRExpr_GSPTR(),
      mkU32( acc_base_addr ),
      mkexpr(srcA_hi), mkexpr(srcA_lo),
      mkexpr(srcB_hi), mkexpr(srcB_lo),
      mkU32( (mask_inst << 5) | AT ));

   /* Set AT_fx to Write if the instruction only writes the ACC.  Set
      AT_fx to modify if the instruction uses the AT entry and writes
      to the ACC entry.  */
   switch (instruction) {
   case XVI4GER8:
   case XVI8GER4:
   case XVI16GER2:
   case XVI16GER2S:
   case XVF16GER2:
   case XVBF16GER2:
   case XVF32GER:
         AT_fx = Ifx_Write;
         break;
   case XVI4GER8PP:
   case XVI8GER4PP:
   case XVI8GER4SPP:
   case XVI16GER2PP:
   case XVI16GER2SPP:
   case XVBF16GER2PP:
   case XVBF16GER2PN:
   case XVBF16GER2NP:
   case XVBF16GER2NN:
   case XVF16GER2PP:
   case XVF16GER2PN:
   case XVF16GER2NP:
   case XVF16GER2NN:
   case XVF32GERPP:
   case XVF32GERPN:
   case XVF32GERNP:
   case XVF32GERNN:
         AT_fx = Ifx_Modify;
         break;
   default:
      vassert(0);  /* Unknown instruction  */
   }

   switch(inst_class) {
   case MATRIX_4BIT_INT_GER:

      d = unsafeIRDirty_0_N (
         0/*regparms*/,
         "vsx_matrix_4bit_ger_dirty_helper",
         fnptr_to_fnentry( vbi, &vsx_matrix_4bit_ger_dirty_helper ),
         args );
      break;

   case MATRIX_8BIT_INT_GER:

      d = unsafeIRDirty_0_N (
         0/*regparms*/,
         "vsx_matrix_8bit_ger_dirty_helper",
         fnptr_to_fnentry( vbi, &vsx_matrix_8bit_ger_dirty_helper ),
         args );
      break;

   case MATRIX_16BIT_INT_GER:

      d = unsafeIRDirty_0_N (
         0/*regparms*/,
         "vsx_matrix_16bit_ger_dirty_helper",
         fnptr_to_fnentry( vbi, &vsx_matrix_16bit_ger_dirty_helper ),
         args );
      break;

   case MATRIX_16BIT_FLOAT_GER:

      d = unsafeIRDirty_0_N (
         0/*regparms*/,
         "vsx_matrix_16bit_float_ger_dirty_helper",
         fnptr_to_fnentry( vbi, &vsx_matrix_16bit_float_ger_dirty_helper ),
         args );
      break;

   case MATRIX_32BIT_FLOAT_GER:

      d = unsafeIRDirty_0_N (
         0/*regparms*/,
         "vsx_matrix_32bit_float_ger_dirty_helper",
         fnptr_to_fnentry( vbi, &vsx_matrix_32bit_float_ger_dirty_helper ),
         args );
      break;

   default:
      vex_printf("ERROR: Unkown inst_class = %u in vsx_matrix_ger()\n",
                 inst_class);
      return;
   }

   setup_fxstate_struct( d, AT, AT_fx, ACC_mapped_on_VSR );

   /* execute the dirty call, side-effecting guest state */
   stmt( IRStmt_Dirty(d) );
}

static void vsx_matrix_64bit_float_ger ( const VexAbiInfo* vbi,
                                         IRExpr *srcA, IRExpr *srcA1,
                                         IRExpr *srcB,
                                         UInt AT, UInt mask_inst,
                                         Bool ACC_mapped_on_VSR ) {

   UInt acc_base_addr = base_acc_addr( ACC_mapped_on_VSR );

   /* This helper function does the VSX Matrix 64-bit floating-point GER
      (Rank-1 Update) instructions xvf64ger, xvf64gerpp, xvf64gerpn,
      xvf64gernp, xvf64gernn, pmxvf64ger, pmxvf64gerpp, pmxvf64gerpn,
      pmxvf64gernp, pmxvf64gernn.  */
   IRTemp srcX_hi = newTemp( Ity_I64);
   IRTemp srcX_lo = newTemp( Ity_I64);
   IRTemp srcX1_hi = newTemp( Ity_I64);
   IRTemp srcX1_lo = newTemp( Ity_I64);
   IRTemp srcY_hi = newTemp( Ity_I64);
   IRTemp srcY_lo = newTemp( Ity_I64);
   UInt start_i;
   IRDirty* d;
   ULong combined_args;
   UInt instruction = mask_inst & 0xFF; /* Instruction is lower 8-bits.  */
   IREffect AT_fx;

   assign( srcX_lo, unop( Iop_V128HIto64, srcA ) );
   assign( srcX_hi, unop( Iop_V128to64, srcA ) );
   assign( srcX1_lo, unop( Iop_V128HIto64, srcA1 ) );
   assign( srcX1_hi, unop( Iop_V128to64, srcA1 ) );
   assign( srcY_lo, unop( Iop_V128HIto64, srcB ) );
   assign( srcY_hi, unop( Iop_V128to64, srcB ) );

   /* Using a dirty helper so we can access the contents of the ACC for use in
      by the instruction and then write the result directly back to the ACC.
      The dirty helper does not return data.

      There is a restriction of 8 args in a dirty helper.  Can't pass the four
      srcX values.  So, just do two calls calculating the first two ACC
      results then the second two ACC results.  */

   start_i = 0;
   combined_args =  (mask_inst << 8) | (start_i << 4) |  AT;

   IRExpr** args1 = mkIRExprVec_7(
      IRExpr_GSPTR(),
      mkU32( acc_base_addr ),
      mkexpr(srcX1_hi), mkexpr(srcX1_lo),
      mkexpr(srcY_hi), mkexpr(srcY_lo),
      mkU32( combined_args ));

   /* Set AT_fx to Write if the instruction only writes the ACC.  Set
      AT_fx to modify if the instruction uses the AT entry and writes
      to the ACC entry.  */
   switch (instruction) {
   case XVF64GER:
         AT_fx = Ifx_Write;
         break;
   case XVF64GERPP:
   case XVF64GERPN:
   case XVF64GERNP:
   case XVF64GERNN:
         AT_fx = Ifx_Modify;
         break;
   default:
      vassert(0);  /* Unknown instruction  */
   }

   d = unsafeIRDirty_0_N (
      0/*regparms*/,
      "vsx_matrix_64bit_float_ger_dirty_helper",
      fnptr_to_fnentry( vbi, &vsx_matrix_64bit_float_ger_dirty_helper ),
      args1 );

   setup_fxstate_struct( d, AT, AT_fx, ACC_mapped_on_VSR );

   /* execute the dirty call, side-effecting guest state */
   stmt( IRStmt_Dirty(d) );

   start_i = 2;
   combined_args = (mask_inst << 8) | (start_i << 4) |  AT;

   IRExpr** args2 = mkIRExprVec_7(
      IRExpr_GSPTR(),
      mkU32( acc_base_addr ),
      mkexpr(srcX_hi), mkexpr(srcX_lo),
      mkexpr(srcY_hi), mkexpr(srcY_lo),
      mkU32( combined_args ));

   d = unsafeIRDirty_0_N (
      0/*regparms*/,
      "vsx_matrix_64bit_float_ger_dirty_helper",
      fnptr_to_fnentry( vbi, &vsx_matrix_64bit_float_ger_dirty_helper ),
      args2 );

   setup_fxstate_struct( d, AT, AT_fx, ACC_mapped_on_VSR );

   /* execute the dirty call, side-effecting guest state */
   stmt( IRStmt_Dirty(d) );
}

static void vector_gen_pvc_mask ( const VexAbiInfo* vbi,
                                   IRExpr *src, UInt IMM,
                                   UInt opc2, UInt VSX_addr ) {
   /* The function takes a 64-bit source and an immediate value.  The function
      calls a helper to execute the xxgenpcvbm, xxgenpcvhm, xxgenpcvwm,
      xxgenpcvdm instruction.  The instructions are not practical to do with
      Iops.  The instruction is implemented with a dirty helper that
      calculates the 128-bit result and writes it directly into the guest
      state VSX register.
  */
   IRTemp src_hi = newTemp( Ity_I64);
   IRTemp src_lo = newTemp( Ity_I64);

   IRDirty* d;

   vassert( (VSX_addr >= 0) && (VSX_addr < 64) );
   UInt reg_offset = offsetofPPCGuestState( guest_VSR0 )
      + sizeof(U128) * VSX_addr;

   assign( src_hi, unop( Iop_V128HIto64, src ) );
   assign( src_lo, unop( Iop_V128to64, src ) );

   IRExpr** args = mkIRExprVec_5(
      IRExpr_GSPTR(),
      mkexpr( src_hi ),
      mkexpr( src_lo ),
      mkU32( reg_offset ),
      mkU64( IMM ) );

   switch( opc2 ) {
   case 0x394: // xxgenpcvbm
      d = unsafeIRDirty_0_N (
         0 /*regparms*/,
         "vector_gen_pvc_byte_mask_dirty_helper",
         fnptr_to_fnentry( vbi,
                           &vector_gen_pvc_byte_mask_dirty_helper ),
         args);
      break;

   case 0x395: // xxgenpcvhm
      d = unsafeIRDirty_0_N (
         0 /*regparms*/,
         "vector_gen_pvc_hword_mask_dirty_helper",
         fnptr_to_fnentry( vbi,
                           &vector_gen_pvc_hword_mask_dirty_helper ),
         args);
      break;

   case 0x3B4: // xxgenpcvwm
      d = unsafeIRDirty_0_N (
         0 /*regparms*/,
         "vector_gen_pvc_word_mask_dirty_helper",
         fnptr_to_fnentry( vbi,
                           &vector_gen_pvc_word_mask_dirty_helper ),
         args);
      break;

   case 0x3B5: // xxgenpcvdm
      d = unsafeIRDirty_0_N (
         0 /*regparms*/,
         "vector_gen_pvc_dword_mask_dirty_helper",
         fnptr_to_fnentry( vbi,
                           &vector_gen_pvc_dword_mask_dirty_helper ),
         args);
      break;
   default:
      vex_printf("ERROR: Unkown instruction = %u in vector_gen_pvc_mask()\n",
                 opc2);
      return;
   }

   d->nFxState = 1;
   vex_bzero(&d->fxState, sizeof(d->fxState));
   d->fxState[0].fx     = Ifx_Modify;
   d->fxState[0].size   = sizeof(U128);
   d->fxState[0].offset = reg_offset;

   /* execute the dirty call, side-effecting guest state */
   stmt( IRStmt_Dirty(d) );
}

static IRExpr * UNSIGNED_CMP_GT_V128 ( IRExpr *vA, IRExpr *vB ) {
   /* This function does an unsigned compare of two V128 values. The
    * function is for use in 32-bit mode only as it is expensive.  The
    * issue is that compares (GT, LT, EQ) are not supported for operands
    * larger then 32-bits when running in 32-bit mode.  The function returns
    * a 1-bit expression, 1 for TRUE and 0 for FALSE.
    */
   IRTemp vA_word0 = newTemp( Ity_I32);
   IRTemp vA_word1 = newTemp( Ity_I32);
   IRTemp vA_word2 = newTemp( Ity_I32);
   IRTemp vA_word3 = newTemp( Ity_I32);
   IRTemp vB_word0 = newTemp( Ity_I32);
   IRTemp vB_word1 = newTemp( Ity_I32);
   IRTemp vB_word2 = newTemp( Ity_I32);
   IRTemp vB_word3 = newTemp( Ity_I32);

   IRTemp eq_word1 = newTemp( Ity_I1);
   IRTemp eq_word2 = newTemp( Ity_I1);
   IRTemp eq_word3 = newTemp( Ity_I1);


   IRExpr *gt_word0, *gt_word1, *gt_word2, *gt_word3;
   IRExpr *eq_word3_2, *eq_word3_2_1;
   IRTemp result = newTemp( Ity_I1 );

   assign( vA_word0, unop( Iop_64to32, unop( Iop_V128to64, vA ) ) );
   assign( vA_word1, unop( Iop_64HIto32, unop( Iop_V128to64, vA ) ) );
   assign( vA_word2, unop( Iop_64to32, unop( Iop_V128HIto64, vA ) ) );
   assign( vA_word3, unop( Iop_64HIto32, unop( Iop_V128HIto64, vA ) ) );

   assign( vB_word0, unop( Iop_64to32, unop( Iop_V128to64, vB ) ) );
   assign( vB_word1, unop( Iop_64HIto32, unop( Iop_V128to64, vB ) ) );
   assign( vB_word2, unop( Iop_64to32, unop( Iop_V128HIto64, vB ) ) );
   assign( vB_word3, unop( Iop_64HIto32, unop( Iop_V128HIto64, vB ) ) );

   assign( eq_word3, binop( Iop_CmpEQ32, mkexpr( vA_word3 ),
                            mkexpr( vB_word3 ) ) );
   assign( eq_word2, binop( Iop_CmpEQ32, mkexpr( vA_word2 ),
                            mkexpr( vB_word2 ) ) );
   assign( eq_word1, binop( Iop_CmpEQ32, mkexpr( vA_word1 ),
                            mkexpr( vB_word1 ) ) );

   gt_word3 = binop( Iop_CmpLT32U, mkexpr( vB_word3 ), mkexpr( vA_word3 ) );
   gt_word2 = binop( Iop_CmpLT32U, mkexpr( vB_word2 ), mkexpr( vA_word2 ) );
   gt_word1 = binop( Iop_CmpLT32U, mkexpr( vB_word1 ), mkexpr( vA_word1 ) );
   gt_word0 = binop( Iop_CmpLT32U, mkexpr( vB_word0 ), mkexpr( vA_word0 ) );

   eq_word3_2   = mkAND1( mkexpr( eq_word3 ), mkexpr( eq_word2 ) );
   eq_word3_2_1 = mkAND1( mkexpr( eq_word1 ), eq_word3_2 );

   assign( result, mkOR1(
                         mkOR1( gt_word3,
                                mkAND1( mkexpr( eq_word3 ), gt_word2 ) ),
                         mkOR1( mkAND1( eq_word3_2, gt_word1 ),
                                mkAND1( eq_word3_2_1, gt_word0 ) ) ) );
   return mkexpr( result );
}

/*------------------------------------------------------------*/
/*---  FP Helpers                                          ---*/
/*------------------------------------------------------------*/

static IRExpr* /* :: Ity_I32 */ get_IR_roundingmode ( void ); // prototype

/* Produce the 32-bit pattern corresponding to the supplied
   float. */
static UInt float_to_bits ( Float f )
{
   union { UInt i; Float f; } u;
   vassert(4 == sizeof(UInt));
   vassert(4 == sizeof(Float));
   vassert(4 == sizeof(u));
   u.f = f;
   return u.i;
}

static IRExpr* dnorm_adj_Vector ( IRExpr* src )
{
   /* This function takes a vector of 32-bit floats.  It does the required
      adjustment on denormalized values based on the setting of the
      VSCR[NJ] bit.

      The VSCR[NJ] bit controlls how subnormal (denormalized) results for
      vector floating point operations are handled. VSCR[NJ] is bit 17
      (bit 111 IBM numbering).

      VSCR[NJ] = 0  Denormalized values are handled as
                    specified by Java and the IEEE standard.

      VSCR[NJ] = 1  If an element in a source VR contains a denormalized
                    value, the value 0 is used instead. If an instruction
                    causes an Underflow Exception, the corresponding element
                    in the target VR is set to 0.  In both cases the 0 has
                    the same sign as the denormalized or underflowing value.
                    Convert negative zero to positive zero.

      The ABI for LE requires VSCR[NJ] = 0.  For BE mode, VSCR[NJ] = 1 by
      default.  The PPC guest state is initialized to match the HW setup.
   */
   IRTemp sign_bit_maskV128 = newTemp( Ity_V128 );
   IRTemp ones_maskV128 = newTemp( Ity_V128 );
   IRTemp clear_dnorm_maskV128 = newTemp( Ity_V128 );
   IRTemp adj_valueV128 = newTemp( Ity_V128 );
   IRTemp dnormV128 = newTemp( Ity_V128 );
   IRTemp zeroV128  = newTemp( Ity_V128 );
   IRTemp VSCR_NJ = newTemp( Ity_I64 );
   IRTemp VSCR_NJ_mask = newTemp( Ity_V128 );
   IRTemp resultV128 = newTemp( Ity_V128 );

   /* get the VSCR[NJ] bit */
   assign( VSCR_NJ,
           unop( Iop_1Sto64,
                 unop( Iop_32to1,
                       binop( Iop_Shr32,
                              getGST( PPC_GST_VSCR ),
                              mkU8( 16 ) ) ) ) );

   assign ( VSCR_NJ_mask, binop( Iop_64HLtoV128,
                                 mkexpr( VSCR_NJ ) ,
                                 mkexpr( VSCR_NJ ) ) );

   /* Create the masks to do the rounding of dnorm values and absolute
      value of zero. */
   assign( dnormV128, is_Denorm_Vector( Ity_I32, src ) );
   assign( zeroV128, is_Zero_Vector( Ity_I32, src ) );

   /* If the value is dnorm, then we need to clear the significand and
      exponent but leave the sign bit. Put 1'x in elements that are not
      denormalized values.  */
   assign( sign_bit_maskV128, unop( Iop_Dup32x4, mkU32( 0x80000000 ) ) );

   assign( clear_dnorm_maskV128,
           binop( Iop_OrV128,
                  binop( Iop_AndV128,
                         mkexpr( dnormV128 ),
                         mkexpr( sign_bit_maskV128 ) ),
                  unop( Iop_NotV128, mkexpr( dnormV128 ) ) ) );

   assign( ones_maskV128, mkV128( 0xFFFF ) );

   assign( adj_valueV128, binop( Iop_AndV128,
                                 mkexpr( clear_dnorm_maskV128 ),
                                 binop( Iop_AndV128,
                                        src,
                                        mkexpr( ones_maskV128 ) ) ) );

   /* If the VSCR[NJ] bit is 1, then clear the denormalized values,
      otherwise just return the input unchanged.  */
   assign( resultV128,
           binop( Iop_OrV128,
                  binop( Iop_AndV128,
                         mkexpr( VSCR_NJ_mask ),
                         mkexpr( adj_valueV128 ) ),
                  binop( Iop_AndV128,
                         unop( Iop_NotV128, mkexpr( VSCR_NJ_mask ) ),
                         src ) ) );

   return mkexpr(resultV128);
}

/*------------------------------------------------------------*/
/* Transactional memory helpers
 *
 *------------------------------------------------------------*/

static ULong generate_TMreason( UInt failure_code,
                                             UInt persistant,
                                             UInt nest_overflow,
                                             UInt tm_exact )
{
   ULong tm_err_code =
     ( (ULong) 0) << (63-6)   /* Failure code */
     | ( (ULong) persistant) << (63-7)     /* Failure persistant */
     | ( (ULong) 0) << (63-8)   /* Disallowed */
     | ( (ULong) nest_overflow) << (63-9)   /* Nesting Overflow */
     | ( (ULong) 0) << (63-10)  /* Footprint Overflow */
     | ( (ULong) 0) << (63-11)  /* Self-Induced Conflict */
     | ( (ULong) 0) << (63-12)  /* Non-Transactional Conflict */
     | ( (ULong) 0) << (63-13)  /* Transactional Conflict */
     | ( (ULong) 0) << (63-14)  /* Translation Invalidation Conflict */
     | ( (ULong) 0) << (63-15)  /* Implementation-specific */
     | ( (ULong) 0) << (63-16)  /* Instruction Fetch Conflict */
     | ( (ULong) 0) << (63-30)  /* Reserved */
     | ( (ULong) 0) << (63-31)  /* Abort */
     | ( (ULong) 0) << (63-32)  /* Suspend */
     | ( (ULong) 0) << (63-33)  /* Reserved */
     | ( (ULong) 0) << (63-35)  /* Privilege */
     | ( (ULong) 0) << (63-36)  /* Failure Summary */
     | ( (ULong) tm_exact) << (63-37)  /* TFIAR Exact */
     | ( (ULong) 0) << (63-38)  /* ROT */
     | ( (ULong) 0) << (63-51)  /* Reserved */
     | ( (ULong) 0) << (63-63);  /* Transaction Level */

     return tm_err_code;
}

static void storeTMfailure( Addr64 err_address, ULong tm_reason,
                            Addr64 handler_address )
{
   putGST( PPC_GST_TFIAR,   mkU64( err_address ) );
   putGST( PPC_GST_TEXASR,  mkU64( tm_reason ) );
   putGST( PPC_GST_TEXASRU, mkU32( 0 ) );
   putGST( PPC_GST_TFHAR,   mkU64( handler_address ) );
}

/*------------------------------------------------------------*/
/*--- Integer Instruction Translation                     --- */
/*------------------------------------------------------------*/

/*
  Byte reverse instructions
*/
static Bool dis_byte_reverse ( UInt prefixInstr, UInt theInstr )
{
   UChar rS_addr = ifieldRegDS( theInstr );
   UChar rA_addr = ifieldRegA( theInstr );
   UInt  opc2    = IFIELD( theInstr, 1, 10 );
   IRTemp rS = newTemp( Ity_I64 );
   IRTemp tmp_0 = newTemp( Ity_I64 );
   IRTemp tmp_1 = newTemp( Ity_I64 );
   IRTemp result = newTemp( Ity_I64 );

   assign( rS, getIReg( rS_addr ) );

   /* NOTE: rA is the destination and rS is the source.  Reverse of the normal usage. */
   switch (opc2) {
   case 0xDB:  // brh  Byte-Reverse half word X-form
      DIP("brh r%u,r%u\n", rA_addr, rS_addr);
      assign( tmp_0,
              binop( Iop_And64, mkexpr( rS ), mkU64( 0xFF00FF00FF00FF00 ) ) );
      assign( tmp_1,
              binop( Iop_And64, mkexpr( rS ), mkU64( 0x00FF00FF00FF00FF ) ) );
      assign( result,
              binop( Iop_Or64,
                     binop( Iop_Shr64, mkexpr( tmp_0 ), mkU8( 8 ) ),
                     binop( Iop_Shl64, mkexpr( tmp_1 ), mkU8( 8 ) ) ) );
      break;

   case 0x9B: {  // brw  Byte-Reverse word X-form
      IRTemp tmp_2 = newTemp( Ity_I64 );
      IRTemp tmp_3 = newTemp( Ity_I64 );

      DIP("brw r%u,r%u\n", rA_addr, rS_addr);
      assign( tmp_0,
              binop( Iop_And64, mkexpr( rS ), mkU64( 0xFF000000FF000000 ) ) );
      assign( tmp_1,
              binop( Iop_And64, mkexpr( rS ), mkU64( 0x00FF000000FF0000 ) ) );
      assign( tmp_2,
              binop( Iop_And64, mkexpr( rS ), mkU64( 0x0000FF000000FF00 ) ) );
      assign( tmp_3,
              binop( Iop_And64, mkexpr( rS ), mkU64( 0x000000FF000000FF ) ) );
      assign( result,
              binop( Iop_Or64,
                     binop( Iop_Or64,
                            binop( Iop_Shl64, mkexpr( tmp_3 ), mkU8( 24 ) ),
                            binop( Iop_Shl64, mkexpr( tmp_2 ), mkU8( 8 ) ) ),
                     binop( Iop_Or64,
                            binop( Iop_Shr64, mkexpr( tmp_1 ), mkU8( 8 ) ),
                            binop( Iop_Shr64, mkexpr( tmp_0 ), mkU8( 24 ) ) )
                 ) );
      break;
   }

   case 0xBB: {      // brd  Byte-Reverse double word X-form
      IRTemp tmp_2 = newTemp( Ity_I64 );
      IRTemp tmp_3 = newTemp( Ity_I64 );
      IRTemp tmp_4 = newTemp( Ity_I64 );
      IRTemp tmp_5 = newTemp( Ity_I64 );
      IRTemp tmp_6 = newTemp( Ity_I64 );
      IRTemp tmp_7 = newTemp( Ity_I64 );

      DIP("brd r%u,r%u\n", rA_addr, rS_addr);
      assign( tmp_0,
              binop( Iop_And64, mkexpr( rS ), mkU64( 0xFF00000000000000 ) ) );
      assign( tmp_1,
               binop( Iop_And64, mkexpr( rS ), mkU64( 0x00FF000000000000 ) ) );
      assign( tmp_2,
              binop( Iop_And64, mkexpr( rS ), mkU64( 0x0000FF0000000000 ) ) );
      assign( tmp_3,
              binop( Iop_And64, mkexpr( rS ), mkU64( 0x000000FF00000000 ) ) );
      assign( tmp_4,
              binop( Iop_And64, mkexpr( rS ), mkU64( 0x00000000FF000000 ) ) );
      assign( tmp_5,
              binop( Iop_And64, mkexpr( rS ), mkU64( 0x0000000000FF0000 ) ) );
      assign( tmp_6,
              binop( Iop_And64, mkexpr( rS ), mkU64( 0x000000000000FF00 ) ) );
      assign( tmp_7,
              binop( Iop_And64, mkexpr( rS ), mkU64( 0x00000000000000FF ) ) );
      assign( result,
              binop( Iop_Or64,
                     binop( Iop_Or64,
                            binop( Iop_Or64,
                                   binop( Iop_Shl64, mkexpr( tmp_7 ), mkU8( 56 ) ),
                                   binop( Iop_Shl64, mkexpr( tmp_6 ), mkU8( 40 ) ) ),
                            binop( Iop_Or64,
                                   binop( Iop_Shl64, mkexpr( tmp_5 ), mkU8( 24 ) ),
                                   binop( Iop_Shl64, mkexpr( tmp_4 ), mkU8( 8 ) ) ) ),
                     binop( Iop_Or64,
                            binop( Iop_Or64,
                                   binop( Iop_Shr64, mkexpr( tmp_3 ), mkU8( 8 ) ),
                                   binop( Iop_Shr64, mkexpr( tmp_2 ), mkU8( 24 ) ) ),
                            binop( Iop_Or64,
                                   binop( Iop_Shr64, mkexpr( tmp_1 ), mkU8( 40 ) ),
                                   binop( Iop_Shr64, mkexpr( tmp_0 ), mkU8( 56 ) ) ) )
                 ) );
      break;
   }

   default:
      vex_printf("dis_byte_reverse(ppc): unrecognized instruction\n");
      return False;
   }

   putIReg( rA_addr, mkexpr( result ) );
   return True;
}

/*
  Integer Arithmetic Instructions
*/
static Bool dis_int_mult_add ( UInt prefix, UInt theInstr )
{
   /* VA-Form */
   UChar rD_addr = ifieldRegDS( theInstr );
   UChar rA_addr = ifieldRegA( theInstr );
   UChar rB_addr = ifieldRegB( theInstr );
   UChar rC_addr = ifieldRegC( theInstr );
   UInt  opc2    = IFIELD( theInstr, 0, 6 );
   IRType ty     = Ity_I64;
   IRTemp rA     = newTemp( ty );
   IRTemp rB     = newTemp( ty );
   IRTemp rC     = newTemp( ty );
   IRTemp rD     = newTemp( ty );
   IRTemp tmpLo  = newTemp( Ity_I64 );
   IRTemp tmpHi  = newTemp( Ity_I64 );
   IRTemp tmp2Hi = newTemp( Ity_I64 );
   IRTemp result = newTemp( Ity_I128 );
   IRTemp resultLo = newTemp( Ity_I64 );
   IRExpr* carryout;

   assign( rA, getIReg( rA_addr ) );
   assign( rB, getIReg( rB_addr ) );
   assign( rC, getIReg( rC_addr ) );

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   switch (opc2) {
   case 0x30:  // maddhd  multiply-add High doubleword signed
      DIP("maddhd r%u,r%u,r%u,r%u\n", rD_addr, rA_addr, rB_addr, rC_addr);

      assign( result, binop( Iop_MullS64, mkexpr( rA ), mkexpr( rB ) ) );
      assign( tmpLo, unop( Iop_128to64, mkexpr( result ) ) );
      assign( tmpHi, unop( Iop_128HIto64, mkexpr( result ) ) );

      /* Multiply rA and rB then add rC.  If the lower 32-bits of the result
       * is less then rC and the result rA * rB, a carry out of the lower 32
       * bits occurred and the upper 32 bits must be incremented by 1. Sign
       * extend rC and do the add to the upper 64 bits to handle the
       * negative case for rC.
       */
      assign( resultLo, binop( Iop_Add64, mkexpr( tmpLo ), mkexpr( rC ) ) );
      assign( tmp2Hi, binop( Iop_Add64,
                             mkexpr( tmpHi ),
                             unop( Iop_1Sto64,
                                   unop( Iop_64to1,
                                         binop( Iop_Shr64,
                                                mkexpr( rC ),
                                                mkU8( 63 ) ) ) ) ) );

      /* need to do calculation for the upper 32 bit result */
      carryout = mkAND1( binop( Iop_CmpLT64U,
                                mkexpr( resultLo ), mkexpr( rC ) ),
                         binop( Iop_CmpLT64U,
                                mkexpr( resultLo ), mkexpr( tmpLo ) ) );
      assign( rD, binop( Iop_Add64,
                         mkexpr( tmp2Hi ),
                         unop( Iop_1Uto64, carryout ) ) );
      break;

   case 0x31:  // maddhdu   multiply-add High doubleword unsigned
      DIP("maddhdu r%u,r%u,r%u,r%u\n", rD_addr, rA_addr, rB_addr, rC_addr);

      assign( result, binop( Iop_MullU64, mkexpr( rA ), mkexpr( rB ) ) );
      assign( tmpLo, unop( Iop_128to64, mkexpr( result ) ) );
      assign( tmpHi, unop( Iop_128HIto64, mkexpr( result ) ) );

      /* Add rC, if the lower 32-bits of the result is less then rC and
       * tmpLo, a carry out of the lower 32 bits occurred. Upper 32 bits
       * must be incremented by 1.
       */
      assign( resultLo, binop( Iop_Add64, mkexpr( tmpLo ), mkexpr( rC ) ) );

      /* need to do calculation for the upper 32 bit result */
      carryout = mkAND1( binop( Iop_CmpLT64U,
                                mkexpr(resultLo), mkexpr( rC ) ),
                         binop( Iop_CmpLT64U,
                                mkexpr(resultLo), mkexpr( tmpLo ) ) );
      assign( rD, binop( Iop_Add64,
                         mkexpr( tmpHi ),
                         unop( Iop_1Uto64, carryout ) ) );
      break;

   case 0x33:  // maddld   multiply-add Low doubleword
      DIP("maddld r%u,r%u,r%u,r%u\n", rD_addr, rA_addr, rB_addr, rC_addr);

      assign( result, binop( Iop_MullS64, mkexpr( rA ), mkexpr( rB ) ) );
      assign( tmpLo, unop( Iop_128to64, mkexpr( result ) ) );
      assign( tmpHi, unop( Iop_128HIto64, mkexpr( result ) ) );

      assign( rD, binop( Iop_Add64, mkexpr( tmpLo ), mkexpr( rC ) ) );
      break;

   default:
      vex_printf("dis_int_mult(ppc): unrecognized instruction\n");
      return False;
   }

   putIReg( rD_addr, mkexpr(rD) );

   return True;
}

static Bool dis_int_arith_prefix ( UInt prefix, UInt theInstr )
{

   UChar opc1    = ifieldOPC(theInstr);
   UChar rT_addr = ifieldRegDS(theInstr);
   UChar rA_addr = ifieldRegA(theInstr);
   IRType ty     = mode64 ? Ity_I64 : Ity_I32;
   IRTemp rA     = newTemp(ty);
   IRTemp rT     = newTemp(ty);
   IRTemp tmp    = newTemp(ty);
   IRTemp value  = newTemp(ty);
   ULong  si0    = ifieldUIMM18(prefix);
   ULong  si1    = ifieldUIMM16(theInstr);   // AKA, SI
   UInt   ptype  = PrefixType(prefix);
   Long   simm16 = extend_s_16to64(si1);
   Bool   is_prefix = prefix_instruction( prefix );
   UInt   R      = 0;                   // must be zero for word instruction

   if ( !is_prefix ) {
     assign( value, mkSzExtendS16( ty, si1 ));

   } else {
     vassert( ty == Ity_I64 );    // prefix instructions must be 64-bit
     vassert( ptype == pType2 );

     R = ifieldR(prefix);
     assign( value, mkSzExtendS34( CONCAT( si0, si1, 16 )));
   }

   assign( rA, getIReg(rA_addr) );

   switch (opc1) {
   /* D-Form */

   case 0x0E: // addi   (Add Immediate, PPC32 p350)
     // li rD,val   == addi rD,0,val
     // la disp(rA) == addi rD,rA,disp

     if ( rA_addr == 0 ) {
       pDIP(is_prefix, "li r%u,%d", rT_addr, (Int)simm16);
       DIPn(is_prefix);
       assign( tmp, mkexpr( value ) );

     } else {
       pDIP(is_prefix, "addi r%u,r%u,%d", rT_addr, rA_addr, (Int)simm16);
       DIPp(is_prefix, ",%u", R);
       assign( tmp, binop( mkSzOp(ty, Iop_Add8), mkexpr( rA ), mkexpr( value ) ) );
     }

     if ( R == 0 )
       assign( rT, mkexpr( tmp ) );
     else
        /* Add immediate value from instruction to the current instruction addr.
           guest_CIA_curr_instr is pointing at the prefix, use address of the
           instruction prefix.  */
        assign( rT, binop( Iop_Add64,
                           mkU64( mkSzAddr( Ity_I64, guest_CIA_curr_instr ) ),
                           mkexpr( tmp ) ) );

     break;

   default:
      vex_printf("dis_int_arith_prefix(ppc)(opc1)\n");
      return False;
   }

   putIReg( rT_addr, mkexpr(rT) );

   return True;
}

static Bool dis_int_arith ( UInt prefix, UInt theInstr )
{
   /* D-Form, XO-Form */
   UChar opc1    = ifieldOPC(theInstr);
   UChar rD_addr = ifieldRegDS(theInstr);
   UChar rA_addr = ifieldRegA(theInstr);
   UInt  uimm16  = ifieldUIMM16(theInstr);
   UChar rB_addr = ifieldRegB(theInstr);
   UChar flag_OE = ifieldBIT10(theInstr);
   UInt  opc2    = ifieldOPClo9(theInstr);
   UChar flag_rC = ifieldBIT0(theInstr);

   Long   simm16 = extend_s_16to64(uimm16);
   IRType ty     = mode64 ? Ity_I64 : Ity_I32;
   IRTemp rA     = newTemp(ty);
   IRTemp rB     = newTemp(ty);
   IRTemp rD     = newTemp(ty);

   Bool do_rc = False;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   assign( rA, getIReg(rA_addr) );
   assign( rB, getIReg(rB_addr) );         // XO-Form: rD, rA, rB

   switch (opc1) {
   /* D-Form */
   case 0x0C: // addic  (Add Immediate Carrying, PPC32 p351
      DIP("addic r%u,r%u,%d\n", rD_addr, rA_addr, (Int)simm16);
      assign( rD, binop( mkSzOp(ty, Iop_Add8), mkexpr(rA),
                         mkSzExtendS16(ty, uimm16) ) );
      set_XER_CA_CA32( ty, PPCG_FLAG_OP_ADD,
                       mkexpr(rD), mkexpr(rA), mkSzExtendS16(ty, uimm16),
                       mkSzImm(ty, 0)/*old xer.ca, which is ignored*/ );
      break;
    
   case 0x0D: // addic. (Add Immediate Carrying and Record, PPC32 p352)
      DIP("addic. r%u,r%u,%d\n", rD_addr, rA_addr, (Int)simm16);
      assign( rD, binop( mkSzOp(ty, Iop_Add8), mkexpr(rA),
                         mkSzExtendS16(ty, uimm16) ) );
      set_XER_CA_CA32( ty, PPCG_FLAG_OP_ADD,
                       mkexpr(rD), mkexpr(rA), mkSzExtendS16(ty, uimm16),
                       mkSzImm(ty, 0)/*old xer.ca, which is ignored*/ );
      do_rc = True;  // Always record to CR
      flag_rC = 1;
      break;

   case 0x0E: // addi   (Add Immediate, PPC32 p350)
      // li rD,val   == addi rD,0,val
      // la disp(rA) == addi rD,rA,disp
      if ( rA_addr == 0 ) {
         DIP("li r%u,%d\n", rD_addr, (Int)simm16);
         assign( rD, mkSzExtendS16(ty, uimm16) );
      } else {
         DIP("addi r%u,r%u,%d\n", rD_addr, rA_addr, (Int)simm16);
         assign( rD, binop( mkSzOp(ty, Iop_Add8), mkexpr(rA),
                            mkSzExtendS16(ty, uimm16) ) );
      }
      break;

   case 0x0F: // addis  (Add Immediate Shifted, PPC32 p353)
      // lis rD,val == addis rD,0,val
      if ( rA_addr == 0 ) {
         DIP("lis r%u,%d\n", rD_addr, (Int)simm16);
         assign( rD, mkSzExtendS32(ty, uimm16 << 16) );
      } else {
         DIP("addis r%u,r%u,0x%x\n", rD_addr, rA_addr, (UInt)simm16);
         assign( rD, binop( mkSzOp(ty, Iop_Add8), mkexpr(rA),
                            mkSzExtendS32(ty, uimm16 << 16) ) );
      }
      break;

   case 0x07: // mulli    (Multiply Low Immediate, PPC32 p490)
      DIP("mulli r%u,r%u,%d\n", rD_addr, rA_addr, (Int)simm16);
      if (mode64)
         assign( rD, unop(Iop_128to64,
                          binop(Iop_MullS64, mkexpr(rA),
                                mkSzExtendS16(ty, uimm16))) );
      else
         assign( rD, unop(Iop_64to32,
                          binop(Iop_MullS32, mkexpr(rA),
                                mkSzExtendS16(ty, uimm16))) );
      break;

   case 0x08: // subfic   (Subtract from Immediate Carrying, PPC32 p540)
      DIP("subfic r%u,r%u,%d\n", rD_addr, rA_addr, (Int)simm16);
      // rD = simm16 - rA
      assign( rD, binop( mkSzOp(ty, Iop_Sub8),
                         mkSzExtendS16(ty, uimm16),
                         mkexpr(rA)) );
      set_XER_CA_CA32( ty, PPCG_FLAG_OP_SUBFI,
                       mkexpr(rD), mkexpr(rA), mkSzExtendS16(ty, uimm16),
                       mkSzImm(ty, 0)/*old xer.ca, which is ignored*/ );
      break;

   /* XO-Form */
   case 0x1F:
      do_rc = True;    // All below record to CR
      
      switch (opc2) {
      case 0x10A: // add  (Add, PPC32 p347)
         DIP("add%s%s r%u,r%u,r%u\n",
             flag_OE ? "o" : "", flag_rC ? ".":"",
             rD_addr, rA_addr, rB_addr);
         assign( rD, binop( mkSzOp(ty, Iop_Add8),
                            mkexpr(rA), mkexpr(rB) ) );
         if (flag_OE) {
            set_XER_OV_OV32_SO( ty, PPCG_FLAG_OP_ADD,
                                mkexpr(rD), mkexpr(rA), mkexpr(rB) );
         }
         break;

      case 0x00A: // addc      (Add Carrying, PPC32 p348)
         DIP("addc%s%s r%u,r%u,r%u\n",
             flag_OE ? "o" : "", flag_rC ? ".":"",
             rD_addr, rA_addr, rB_addr);
         assign( rD, binop( mkSzOp(ty, Iop_Add8),
                            mkexpr(rA), mkexpr(rB)) );
         set_XER_CA_CA32( ty, PPCG_FLAG_OP_ADD,
                          mkexpr(rD), mkexpr(rA), mkexpr(rB),
                          mkSzImm(ty, 0)/*old xer.ca, which is ignored*/ );
         if (flag_OE) {
            set_XER_OV_OV32_SO( ty, PPCG_FLAG_OP_ADD,
                                mkexpr(rD), mkexpr(rA), mkexpr(rB) );
         }
         break;
         
      case 0x08A: { // adde      (Add Extended, PPC32 p349)
         IRTemp old_xer_ca = newTemp(ty);
         DIP("adde%s%s r%u,r%u,r%u\n",
             flag_OE ? "o" : "", flag_rC ? ".":"",
             rD_addr, rA_addr, rB_addr);
         // rD = rA + rB + XER[CA]
         assign( old_xer_ca, mkWidenFrom32(ty, getXER_CA_32(), False) );
         assign( rD, binop( mkSzOp(ty, Iop_Add8), mkexpr(rA),
                            binop( mkSzOp(ty, Iop_Add8),
                                   mkexpr(rB), mkexpr(old_xer_ca))) );
         set_XER_CA_CA32( ty, PPCG_FLAG_OP_ADDE,
                          mkexpr(rD), mkexpr(rA), mkexpr(rB),
                          mkexpr(old_xer_ca) );
         if (flag_OE) {
            set_XER_OV_OV32_SO( ty, PPCG_FLAG_OP_ADDE,
                                mkexpr(rD), mkexpr(rA), mkexpr(rB) );
         }
         break;
      }

      case 0xAA: {// addex (Add Extended alternate carry bit Z23-form)
         IRTemp old_xer_ov = newTemp(ty);
         DIP("addex r%u,r%u,r%u,%d\n", rD_addr, rA_addr, rB_addr, (Int)flag_OE);
         assign( old_xer_ov, mkWidenFrom32(ty, getXER_OV_32(), False) );
         assign( rD, binop( mkSzOp(ty, Iop_Add8), mkexpr(rA),
                            binop( mkSzOp(ty, Iop_Add8), mkexpr(rB),
                                   mkexpr(old_xer_ov) ) ) );

         /* CY bit is same as OE bit */
         if (flag_OE == 0) {
            /* Exception, do not set SO bit and set OV from carry. */
            set_XER_OV_OV32_ADDEX( ty, mkexpr(rD), mkexpr(rA), mkexpr(rB),
                                   mkexpr(old_xer_ov) );
         } else {
            /* CY=1, 2 and 3 (AKA flag_OE) are reserved */
            vex_printf("addex instruction, CY = %d is reserved.\n", flag_OE);
            vpanic("addex instruction\n");
         }
         break;
      }

      case 0x0EA: { // addme     (Add to Minus One Extended, PPC32 p354)
         IRTemp old_xer_ca = newTemp(ty);
         IRExpr *min_one;
         if (rB_addr != 0) {
            vex_printf("dis_int_arith(ppc)(addme,rB_addr)\n");
            return False;
         }
         DIP("addme%s%s r%u,r%u,r%u\n",
             flag_OE ? "o" : "", flag_rC ? ".":"",
             rD_addr, rA_addr, rB_addr);
         // rD = rA + (-1) + XER[CA]
         // => Just another form of adde
         assign( old_xer_ca, mkWidenFrom32(ty, getXER_CA_32(), False) );
         min_one = mkSzImm(ty, (Long)-1);
         assign( rD, binop( mkSzOp(ty, Iop_Add8), mkexpr(rA),
                            binop( mkSzOp(ty, Iop_Add8),
                                   min_one, mkexpr(old_xer_ca)) ));
         set_XER_CA_CA32( ty, PPCG_FLAG_OP_ADDE,
                          mkexpr(rD), mkexpr(rA), min_one,
                          mkexpr(old_xer_ca) );
         if (flag_OE) {
            set_XER_OV_OV32_SO( ty, PPCG_FLAG_OP_ADDE,
                                mkexpr(rD), mkexpr(rA), min_one );
         }
         break;
      }

      case 0x0CA: { // addze      (Add to Zero Extended, PPC32 p355)
         IRTemp old_xer_ca = newTemp(ty);
         if (rB_addr != 0) {
            vex_printf("dis_int_arith(ppc)(addze,rB_addr)\n");
            return False;
         }
         DIP("addze%s%s r%u,r%u,r%u\n",
             flag_OE ? "o" : "", flag_rC ? ".":"",
             rD_addr, rA_addr, rB_addr);
         // rD = rA + (0) + XER[CA]
         // => Just another form of adde
         assign( old_xer_ca, mkWidenFrom32(ty, getXER_CA_32(), False) );
         assign( rD, binop( mkSzOp(ty, Iop_Add8),
                            mkexpr(rA), mkexpr(old_xer_ca)) );
         set_XER_CA_CA32( ty, PPCG_FLAG_OP_ADDE,
                          mkexpr(rD), mkexpr(rA), mkSzImm(ty, 0),
                          mkexpr(old_xer_ca) );
         if (flag_OE) {
            set_XER_OV_OV32_SO( ty, PPCG_FLAG_OP_ADDE,
                                mkexpr(rD), mkexpr(rA), mkSzImm(ty, 0) );
         }
         break;
      }

      case 0x1EB: // divw       (Divide Word, PPC32 p388)
         DIP("divw%s%s r%u,r%u,r%u\n",
             flag_OE ? "o" : "", flag_rC ? ".":"",
             rD_addr, rA_addr, rB_addr);
         if (mode64) {
            /* Note:
               XER settings are mode independent, and reflect the 
               overflow of the low-order 32bit result
               CR0[LT|GT|EQ] are undefined if flag_rC && mode64
            */
            /* rD[hi32] are undefined: setting them to sign of lo32
                - makes set_CR0 happy */
            IRExpr* dividend = mk64lo32Sto64( mkexpr(rA) );
            IRExpr* divisor  = mk64lo32Sto64( mkexpr(rB) );
            assign( rD, mk64lo32Uto64( binop(Iop_DivS64, dividend,
                                                         divisor) ) );
            if (flag_OE) {
               set_XER_OV_OV32_SO( ty, PPCG_FLAG_OP_DIVW,
                                   mkexpr(rD), dividend, divisor );
            }
         } else {
            assign( rD, binop(Iop_DivS32, mkexpr(rA), mkexpr(rB)) );
            if (flag_OE) {
               set_XER_OV_OV32_SO( ty, PPCG_FLAG_OP_DIVW,
                                   mkexpr(rD), mkexpr(rA), mkexpr(rB) );
            }
         }
         /* Note:
            if (0x8000_0000 / -1) or (x / 0)
            => rD=undef, if(flag_rC) CR7=undef, if(flag_OE) XER_OV=1
            => But _no_ exception raised. */
         break;

      case 0x1CB: // divwu      (Divide Word Unsigned, PPC32 p389)
         DIP("divwu%s%s r%u,r%u,r%u\n",
             flag_OE ? "o" : "", flag_rC ? ".":"",
             rD_addr, rA_addr, rB_addr);
         if (mode64) {
            /* Note:
               XER settings are mode independent, and reflect the 
               overflow of the low-order 32bit result
               CR0[LT|GT|EQ] are undefined if flag_rC && mode64
            */
            IRExpr* dividend = mk64lo32Uto64( mkexpr(rA) );
            IRExpr* divisor  = mk64lo32Uto64( mkexpr(rB) );
            assign( rD, mk64lo32Uto64( binop(Iop_DivU64, dividend,
                                                         divisor) ) );
            if (flag_OE) {
               set_XER_OV_OV32_SO( ty, PPCG_FLAG_OP_DIVWU,
                                   mkexpr(rD), dividend, divisor );
            }
         } else {
            assign( rD, binop(Iop_DivU32, mkexpr(rA), mkexpr(rB)) );
            if (flag_OE) {
               set_XER_OV_OV32_SO( ty, PPCG_FLAG_OP_DIVWU,
                                   mkexpr(rD), mkexpr(rA), mkexpr(rB) );
            }
         }
         /* Note: ditto comment divw, for (x / 0) */
         break;

      case 0x04B: // mulhw      (Multiply High Word, PPC32 p488)
         if (flag_OE != 0) {
            vex_printf("dis_int_arith(ppc)(mulhw,flag_OE)\n");
            return False;
         }
         DIP("mulhw%s r%u,r%u,r%u\n", flag_rC ? ".":"",
             rD_addr, rA_addr, rB_addr);
         if (mode64) {
            /* rD[hi32] are undefined: setting them to sign of lo32
                - makes set_CR0 happy */
            assign( rD, binop(Iop_Sar64,
                           binop(Iop_Mul64,
                                 mk64lo32Sto64( mkexpr(rA) ),
                                 mk64lo32Sto64( mkexpr(rB) )),
                              mkU8(32)) );
         } else {
            assign( rD, unop(Iop_64HIto32,
                             binop(Iop_MullS32,
                                   mkexpr(rA), mkexpr(rB))) );
         }
         break;

      case 0x00B: // mulhwu    (Multiply High Word Unsigned, PPC32 p489)
         if (flag_OE != 0) {
            vex_printf("dis_int_arith(ppc)(mulhwu,flag_OE)\n");
            return False;
         }
         DIP("mulhwu%s r%u,r%u,r%u\n", flag_rC ? ".":"",
             rD_addr, rA_addr, rB_addr);
         if (mode64) {
            /* rD[hi32] are undefined: setting them to sign of lo32
                - makes set_CR0 happy */
            assign( rD, binop(Iop_Sar64,
                           binop(Iop_Mul64,
                                 mk64lo32Uto64( mkexpr(rA) ),
                                 mk64lo32Uto64( mkexpr(rB) ) ),
                              mkU8(32)) );
         } else {
            assign( rD, unop(Iop_64HIto32, 
                             binop(Iop_MullU32,
                                   mkexpr(rA), mkexpr(rB))) );
         }
         break;
         
      case 0x0EB: // mullw      (Multiply Low Word, PPC32 p491)
         DIP("mullw%s%s r%u,r%u,r%u\n",
             flag_OE ? "o" : "", flag_rC ? ".":"",
             rD_addr, rA_addr, rB_addr);
         if (mode64) {
            /* rD[hi32] are undefined: setting them to sign of lo32
                - set_XER_OV() and set_CR0() depend on this */
            IRExpr *a = unop(Iop_64to32, mkexpr(rA) );
            IRExpr *b = unop(Iop_64to32, mkexpr(rB) );
            assign( rD, binop(Iop_MullS32, a, b) );
            if (flag_OE) {
               set_XER_OV_OV32_SO( ty, PPCG_FLAG_OP_MULLW,
                                mkexpr(rD),
                                unop(Iop_32Uto64, a), unop(Iop_32Uto64, b) );
            }
         } else {
            assign( rD, unop(Iop_64to32,
                             binop(Iop_MullU32,
                                   mkexpr(rA), mkexpr(rB))) );
            if (flag_OE) {
               set_XER_OV_OV32_SO( ty, PPCG_FLAG_OP_MULLW,
                                   mkexpr(rD), mkexpr(rA), mkexpr(rB) );
            }
         }
         break;

      case 0x068: // neg        (Negate, PPC32 p493)
         if (rB_addr != 0) {
            vex_printf("dis_int_arith(ppc)(neg,rB_addr)\n");
            return False;
         }
         DIP("neg%s%s r%u,r%u\n",
             flag_OE ? "o" : "", flag_rC ? ".":"",
             rD_addr, rA_addr);
         // rD = (~rA) + 1
         assign( rD, binop( mkSzOp(ty, Iop_Add8),
                            unop( mkSzOp(ty, Iop_Not8), mkexpr(rA) ),
                            mkSzImm(ty, 1)) );
         if (flag_OE) {
            set_XER_OV_OV32_SO( ty, PPCG_FLAG_OP_NEG,
                                mkexpr(rD), mkexpr(rA), mkexpr(rB) );
         }
         break;

      case 0x028: // subf       (Subtract From, PPC32 p537)
         DIP("subf%s%s r%u,r%u,r%u\n",
             flag_OE ? "o" : "", flag_rC ? ".":"",
             rD_addr, rA_addr, rB_addr);
         // rD = rB - rA
         assign( rD, binop( mkSzOp(ty, Iop_Sub8),
                            mkexpr(rB), mkexpr(rA)) );
         if (flag_OE) {
            set_XER_OV_OV32_SO( ty, PPCG_FLAG_OP_SUBF,
                                mkexpr(rD), mkexpr(rA), mkexpr(rB) );
         }
         break;

      case 0x008: // subfc      (Subtract from Carrying, PPC32 p538)
         DIP("subfc%s%s r%u,r%u,r%u\n",
             flag_OE ? "o" : "", flag_rC ? ".":"",
             rD_addr, rA_addr, rB_addr);
         // rD = rB - rA
         assign( rD, binop( mkSzOp(ty, Iop_Sub8),
                            mkexpr(rB), mkexpr(rA)) );
         set_XER_CA_CA32( ty, PPCG_FLAG_OP_SUBFC,
                          mkexpr(rD), mkexpr(rA), mkexpr(rB),
                          mkSzImm(ty, 0)/*old xer.ca, which is ignored*/ );
         if (flag_OE) {
            set_XER_OV_OV32_SO( ty, PPCG_FLAG_OP_SUBFC,
                                mkexpr(rD), mkexpr(rA), mkexpr(rB) );
         }
         break;
         
      case 0x088: {// subfe      (Subtract from Extended, PPC32 p539)
         IRTemp old_xer_ca = newTemp(ty);
         DIP("subfe%s%s r%u,r%u,r%u\n",
             flag_OE ? "o" : "", flag_rC ? ".":"",
             rD_addr, rA_addr, rB_addr);
         // rD = (log not)rA + rB + XER[CA]
         //    ==>
         // rD = rB - rA - (XER[CA] ^ 1)
         assign( old_xer_ca, mkWidenFrom32(ty, getXER_CA_32(), False) );
         assign( rD, binop( mkSzOp(ty, Iop_Sub8),
                            binop( mkSzOp(ty, Iop_Sub8),
                                   mkexpr(rB), mkexpr(rA)),
                            binop(mkSzOp(ty, Iop_Xor8),
                                  mkexpr(old_xer_ca),
                                  mkSzImm(ty, 1))) );
         set_XER_CA_CA32( ty, PPCG_FLAG_OP_SUBFE,
                          mkexpr(rD), mkexpr(rA), mkexpr(rB),
                          mkexpr(old_xer_ca) );
         if (flag_OE) {
            set_XER_OV_OV32_SO( ty, PPCG_FLAG_OP_SUBFE,
                                mkexpr(rD), mkexpr(rA), mkexpr(rB) );
         }
         break;
      }

      case 0x0E8: { // subfme    (Subtract from -1 Extended, PPC32 p541)
         IRTemp old_xer_ca = newTemp(ty);
         IRExpr *min_one;
         if (rB_addr != 0) {
            vex_printf("dis_int_arith(ppc)(subfme,rB_addr)\n");
            return False;
         }
         DIP("subfme%s%s r%u,r%u\n",
             flag_OE ? "o" : "", flag_rC ? ".":"",
             rD_addr, rA_addr);
         // rD = (log not)rA + (-1) + XER[CA]
         // => Just another form of subfe
         assign( old_xer_ca, mkWidenFrom32(ty, getXER_CA_32(), False) );
         min_one = mkSzImm(ty, (Long)-1);
         assign( rD, binop( mkSzOp(ty, Iop_Add8),
                            unop( mkSzOp(ty, Iop_Not8), mkexpr(rA)),
                            binop( mkSzOp(ty, Iop_Add8),
                                   min_one, mkexpr(old_xer_ca))) );
         set_XER_CA_CA32( ty, PPCG_FLAG_OP_SUBFE,
                          mkexpr(rD), mkexpr(rA), min_one,
                          mkexpr(old_xer_ca) );
         if (flag_OE) {
            set_XER_OV_OV32_SO( ty, PPCG_FLAG_OP_SUBFE,
                                mkexpr(rD), mkexpr(rA), min_one );
         }
         break;
      }

      case 0x0C8: { // subfze  (Subtract from Zero Extended, PPC32 p542)
         IRTemp old_xer_ca = newTemp(ty);
         if (rB_addr != 0) {
            vex_printf("dis_int_arith(ppc)(subfze,rB_addr)\n");
            return False;
         }
         DIP("subfze%s%s r%u,r%u\n",
             flag_OE ? "o" : "", flag_rC ? ".":"",
             rD_addr, rA_addr);
         // rD = (log not)rA + (0) + XER[CA]
         // => Just another form of subfe
         assign( old_xer_ca, mkWidenFrom32(ty, getXER_CA_32(), False) );
         assign( rD, binop( mkSzOp(ty, Iop_Add8),
                           unop( mkSzOp(ty, Iop_Not8),
                                 mkexpr(rA)), mkexpr(old_xer_ca)) );
         set_XER_CA_CA32( ty, PPCG_FLAG_OP_SUBFE,
                          mkexpr(rD), mkexpr(rA), mkSzImm(ty, 0),
                          mkexpr(old_xer_ca) );
         if (flag_OE) {
            set_XER_OV_OV32_SO( ty, PPCG_FLAG_OP_SUBFE,
                                mkexpr(rD), mkexpr(rA), mkSzImm(ty, 0) );
         }
         break;
      }


      /* 64bit Arithmetic */
      case 0x49:  // mulhd (Multiply High DWord, PPC64 p539)
         if (flag_OE != 0) {
            vex_printf("dis_int_arith(ppc)(mulhd,flagOE)\n");
            return False;
         }
         DIP("mulhd%s r%u,r%u,r%u\n", flag_rC ? ".":"",
             rD_addr, rA_addr, rB_addr);
         assign( rD, unop(Iop_128HIto64, 
                          binop(Iop_MullS64,
                                mkexpr(rA), mkexpr(rB))) );

         break;

      case 0x9:   // mulhdu  (Multiply High DWord Unsigned, PPC64 p540)
         if (flag_OE != 0) {
            vex_printf("dis_int_arith(ppc)(mulhdu,flagOE)\n");
            return False;
         }
         DIP("mulhdu%s r%u,r%u,r%u\n", flag_rC ? ".":"",
             rD_addr, rA_addr, rB_addr);
         assign( rD, unop(Iop_128HIto64, 
                          binop(Iop_MullU64,
                                mkexpr(rA), mkexpr(rB))) );
         break;

      case 0xE9:  // mulld (Multiply Low DWord, PPC64 p543)
         DIP("mulld%s%s r%u,r%u,r%u\n",
             flag_OE ? "o" : "", flag_rC ? ".":"",
             rD_addr, rA_addr, rB_addr);
         assign( rD, binop(Iop_Mul64, mkexpr(rA), mkexpr(rB)) );
         if (flag_OE) {
            set_XER_OV_64( PPCG_FLAG_OP_MULLD,
                           mkexpr(rD), mkexpr(rA), mkexpr(rB) );
            /* OV is set to 1 if product isn't representable.
             * In this case also need to set OV32 and SO to 1,
             * i.e. copy OV to OV32 and SO.
             */
            copy_OV_to_OV32();
            update_SO();
         }
         break;

      case 0x1E9: // divd (Divide DWord, PPC64 p419)
         DIP("divd%s%s r%u,r%u,r%u\n",
             flag_OE ? "o" : "", flag_rC ? ".":"",
             rD_addr, rA_addr, rB_addr);
         assign( rD, binop(Iop_DivS64, mkexpr(rA), mkexpr(rB)) );
         if (flag_OE) {
            set_XER_OV_OV32_SO( ty, PPCG_FLAG_OP_DIVW,
                                mkexpr(rD), mkexpr(rA), mkexpr(rB) );
         }
         break;
         /* Note:
            if (0x8000_0000_0000_0000 / -1) or (x / 0)
            => rD=undef, if(flag_rC) CR7=undef, if(flag_OE) XER_OV=1
            => But _no_ exception raised. */

      case 0x1C9: // divdu (Divide DWord Unsigned, PPC64 p420)
         DIP("divdu%s%s r%u,r%u,r%u\n",
             flag_OE ? "o" : "", flag_rC ? ".":"",
             rD_addr, rA_addr, rB_addr);
         assign( rD, binop(Iop_DivU64, mkexpr(rA), mkexpr(rB)) );
         if (flag_OE) {
            set_XER_OV_OV32_SO( ty, PPCG_FLAG_OP_DIVWU,
                                mkexpr(rD), mkexpr(rA), mkexpr(rB) );
         }
         break;
         /* Note: ditto comment divd, for (x / 0) */

      case 0x18B: // divweu (Divide Word Extended Unsigned)
      {
        /*
         *  If (RA) >= (RB), or if an attempt is made to perform the division
         *         <anything> / 0
         * then the contents of register RD are undefined as are (if Rc=1) the contents of
         * the LT, GT, and EQ bits of CR Field 0. In these cases, if OE=1 then OV is set
         * to 1.
         */
         IRTemp res = newTemp(Ity_I32);
         IRExpr * dividend, * divisor;
         DIP("divweu%s%s r%u,r%u,r%u\n",
             flag_OE ? "o" : "", flag_rC ? ".":"",
                                         rD_addr, rA_addr, rB_addr);
         if (mode64) {
            dividend = unop( Iop_64to32, mkexpr( rA ) );
            divisor = unop( Iop_64to32, mkexpr( rB ) );
            assign( res, binop( Iop_DivU32E, dividend, divisor ) );
            assign( rD, binop( Iop_32HLto64, mkU32( 0 ), mkexpr( res ) ) );
         } else {
            dividend = mkexpr( rA );
            divisor =  mkexpr( rB );
            assign( res, binop( Iop_DivU32E, dividend, divisor ) );
            assign( rD, mkexpr( res) );
         }

         if (flag_OE) {
            set_XER_OV_OV32_32( PPCG_FLAG_OP_DIVWEU,
                                mkexpr(res), dividend, divisor );
            update_SO( );
         }
         break;
      }

      case 0x1AB: // divwe (Divide Word Extended)
      {
         /*
          * If the quotient cannot be represented in 32 bits, or if an
          * attempt is made to perform the division
          *      <anything> / 0
          * then the contents of register RD are undefined as are (if
          * Rc=1) the contents of the LT, GT, and EQ bits of CR
          * Field 0. In these cases, if OE=1 then OV is set to 1.
          */

         IRTemp res = newTemp(Ity_I32);
         IRExpr * dividend, * divisor;
         DIP("divwe%s%s r%u,r%u,r%u\n",
             flag_OE ? "o" : "", flag_rC ? ".":"",
                                         rD_addr, rA_addr, rB_addr);
         if (mode64) {
            dividend = unop( Iop_64to32, mkexpr( rA ) );
            divisor = unop( Iop_64to32, mkexpr( rB ) );
            assign( res, binop( Iop_DivS32E, dividend, divisor ) );
            assign( rD, binop( Iop_32HLto64, mkU32( 0 ), mkexpr( res ) ) );
         } else {
            dividend = mkexpr( rA );
            divisor =  mkexpr( rB );
            assign( res, binop( Iop_DivS32E, dividend, divisor ) );
            assign( rD, mkexpr( res) );
         }

         if (flag_OE) {
            set_XER_OV_OV32_32( PPCG_FLAG_OP_DIVWE,
                                mkexpr(res), dividend, divisor );
            update_SO( );
         }
         break;
      }


      case 0x1A9: // divde (Divide Doubleword Extended)
        /*
         * If the quotient cannot be represented in 64 bits, or if an
         * attempt is made to perform the division
         *      <anything> / 0
         * then the contents of register RD are undefined as are (if
         * Rc=1) the contents of the LT, GT, and EQ bits of CR
         * Field 0. In these cases, if OE=1 then OV is set to 1.
         */
         DIP("divde%s%s r%u,r%u,r%u\n",
             flag_OE ? "o" : "", flag_rC ? ".":"",
             rD_addr, rA_addr, rB_addr);
         assign( rD, binop(Iop_DivS64E, mkexpr(rA), mkexpr(rB)) );
         if (flag_OE) {
            set_XER_OV_64( PPCG_FLAG_OP_DIVDE, mkexpr( rD ),
                           mkexpr( rA ), mkexpr( rB ) );
           copy_OV_to_OV32();
           update_SO();
         }
         break;

      case 0x189: //  divdeuo (Divide Doubleword Extended Unsigned)
        // Same CR and OV rules as given for divweu above
        DIP("divdeu%s%s r%u,r%u,r%u\n",
            flag_OE ? "o" : "", flag_rC ? ".":"",
            rD_addr, rA_addr, rB_addr);
        assign( rD, binop(Iop_DivU64E, mkexpr(rA), mkexpr(rB)) );
        if (flag_OE) {
           set_XER_OV_64( PPCG_FLAG_OP_DIVDEU, mkexpr( rD ),
                          mkexpr( rA ), mkexpr( rB ) );
           copy_OV_to_OV32();
           update_SO();
        }
        break;

      default:
         vex_printf("dis_int_arith(ppc)(opc2)\n");
         return False;
      }
      break;

   default:
      vex_printf("dis_int_arith(ppc)(opc1)\n");
      return False;
   }

   putIReg( rD_addr, mkexpr(rD) );

   if (do_rc && flag_rC) {
      set_CR0( mkexpr(rD) );
   }
   return True;
}

static Bool dis_modulo_int ( UInt prefix, UInt theInstr )
{
   /* X-Form */
   UChar opc1    = ifieldOPC( theInstr );
   UInt  opc2    = ifieldOPClo10( theInstr );
   UChar rA_addr = ifieldRegA( theInstr );
   UChar rB_addr = ifieldRegB( theInstr );
   UChar rD_addr = ifieldRegDS( theInstr );
   IRType ty     = mode64 ? Ity_I64 : Ity_I32;
   IRTemp rD     = newTemp( ty );

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   switch (opc1) {
   /* X-Form */
   case 0x1F:
      switch (opc2) {
      case 0x109: // modud  Modulo Unsigned Double Word
         {
            IRTemp rA = newTemp( Ity_I64 );
            IRTemp rB = newTemp( Ity_I64 );
            IRTemp quotient = newTemp( Ity_I64 );
            IRTemp quotientTimesDivisor = newTemp( Ity_I64 );
            IRTemp remainder = newTemp( Ity_I64 );
            IRTemp rB_0   = newTemp( Ity_I64 );    /* all 1's if rB = 0 */

            DIP("modud r%u,r%u,r%u\n", rD_addr, rA_addr, rB_addr);

            assign( rA, getIReg( rA_addr ) );
            assign( rB, getIReg( rB_addr ) );

            assign( quotient,
                    binop( Iop_DivU64, mkexpr( rA ), mkexpr( rB ) ) );

            assign( quotientTimesDivisor,
                    binop( Iop_Mul64,
                           mkexpr( quotient ),
                           mkexpr( rB ) ) );

            assign( remainder,
                    binop( Iop_Sub64,
                           mkexpr( rA ),
                           mkexpr( quotientTimesDivisor ) ) );

            /* Need to match the HW for these special cases
             * rB = 0         result all zeros
             */
            assign( rB_0, unop( Iop_1Sto64,
                                binop( Iop_CmpEQ64,
                                       mkexpr( rB ),
                                       mkU64( 0x0 ) ) ) );

            assign (rD, binop( Iop_And64,
                               unop( Iop_Not64, mkexpr( rB_0 ) ),
                               mkexpr( remainder ) ) );
            break;
         }

      case 0x10B: // moduw  Modulo Unsigned Word
         {
            IRTemp quotient = newTemp( Ity_I32 );
            IRTemp quotientTimesDivisor = newTemp( Ity_I32 );
            IRTemp remainder = newTemp( Ity_I32 );

            IRTemp rA     = newTemp( Ity_I32 );
            IRTemp rB     = newTemp( Ity_I32 );
            IRTemp rB_0   = newTemp( Ity_I32 );     /* all 1's if rB = 0 */

            DIP("moduw r%u,r%u,r%u\n", rD_addr, rA_addr, rB_addr);

            if ( ty == Ity_I64 ) {
               /* rA and rB are 32 bit values in bits 32:63 of the
                * 64-bit register.
                */
               assign( rA, unop( Iop_64to32, getIReg( rA_addr ) ) );
               assign( rB, unop( Iop_64to32, getIReg( rB_addr ) ) );

            } else {
               assign( rA, getIReg( rA_addr ) );
               assign( rB, getIReg( rB_addr ) );
            }

            assign( quotient,
                    binop( Iop_DivU32, mkexpr( rA ), mkexpr( rB ) ) );

            assign( quotientTimesDivisor,
                    unop( Iop_64to32,
                          binop( Iop_MullU32,
                                 mkexpr( quotient ),
                                 mkexpr( rB ) ) ) );

            assign( remainder,
                    binop( Iop_Sub32,
                           mkexpr( rA ),
                           mkexpr( quotientTimesDivisor ) ) );

            /* Need to match the HW for these special cases
             * rB = 0         result all zeros
             */
            assign( rB_0, unop( Iop_1Sto32,
                                binop( Iop_CmpEQ32,
                                       mkexpr( rB ),
                                       mkU32( 0x0 ) ) ) );

            assign (rD, binop( Iop_32HLto64,
                               mkU32( 0 ),
                               binop( Iop_And32,
                                      unop( Iop_Not32, mkexpr( rB_0 ) ),
                                      mkexpr( remainder ) ) ) );
            break;
         }

      case 0x21A: // cnttzw, cnttzw.   Count Trailing Zero Word
         {
            /* Note cnttzw RA, RS  - RA is dest, RS is source.  But the
             * order of the operands in theInst is opc1 RS RA opc2 which has
             * the operand fields backwards to what the standard order.
             */
            UChar rA_address = ifieldRegA(theInstr);
            UChar rS_address = ifieldRegDS(theInstr);
            IRTemp rA     = newTemp(Ity_I64);
            IRTemp rS     = newTemp(Ity_I64);
            UChar flag_rC = ifieldBIT0(theInstr);
            IRTemp result = newTemp(Ity_I32);

            DIP("cnttzw%s r%u,r%u\n",  flag_rC ? "." : "",
                rA_address, rS_address);

            assign( rS, getIReg( rS_address ) );
            assign( result, unop( Iop_CtzNat32,
                                  unop( Iop_64to32, mkexpr( rS ) ) ) );
            assign( rA, binop( Iop_32HLto64, mkU32( 0 ), mkexpr( result ) ) );

            if ( flag_rC )
               set_CR0( mkexpr( rA ) );

            putIReg( rA_address, mkexpr( rA ) );

            return True;  /* Return here since this inst is not consistent
                           * with the other instructions
                           */
         }
         break;

      case 0x23A: // cnttzd, cnttzd.   Count Trailing Zero Double word
         {
            /* Note cnttzd RA, RS  - RA is dest, RS is source.  But the
             * order of the operands in theInst is opc1 RS RA opc2 which has
             * the operand order listed backwards to what is standard.
             */
            UChar rA_address = ifieldRegA(theInstr);
            UChar rS_address = ifieldRegDS(theInstr);
            IRTemp rA     = newTemp(Ity_I64);
            IRTemp rS     = newTemp(Ity_I64);
            UChar flag_rC = ifieldBIT0(theInstr);

            DIP("cnttzd%s r%u,r%u\n",  flag_rC ? "." : "",
                rA_address, rS_address);

            assign( rS, getIReg( rS_address ) );
            assign( rA, unop( Iop_CtzNat64, mkexpr( rS ) ) );

            if ( flag_rC == 1 )
               set_CR0( mkexpr( rA ) );

            putIReg( rA_address, mkexpr( rA ) );

            return True;  /* Return here since this inst is not consistent
                           * with the other instructions
                           */
         }
         break;

      case 0x309: // modsd  Modulo Signed Double Word
         {
            IRTemp rA     = newTemp( Ity_I64 );
            IRTemp rB     = newTemp( Ity_I64 );
            IRTemp rA2_63 = newTemp( Ity_I64 );    /* all 1's if rA != -2^63 */
            IRTemp rB_0   = newTemp( Ity_I1 );     /* 1 if rB = 0 */
            IRTemp rB_1   = newTemp( Ity_I1 );     /* 1 if rB = 1 */
            IRTemp rB_m1  = newTemp( Ity_I1 );     /* 1 if rB = -1 */
            IRTemp rA_m1  = newTemp( Ity_I1 );     /* 1 if rA = -1 */
            IRTemp resultis0   = newTemp( Ity_I64 );
            IRTemp quotient = newTemp( Ity_I64 );
            IRTemp quotientTimesDivisor = newTemp( Ity_I64 );
            IRTemp remainder = newTemp( Ity_I64 );

            DIP("modsd r%u,r%u,r%u\n", rD_addr, rA_addr, rB_addr);

            assign( rA, getIReg( rA_addr ) );
            assign( rB, getIReg( rB_addr ) );

            assign( rA2_63, unop ( Iop_1Sto64,
                                   binop( Iop_CmpNE64,
                                          mkexpr( rA ),
                                          mkU64( 0x8000000000000000 ) ) ) );
            assign( rB_0, binop( Iop_CmpEQ64,
                                 mkexpr( rB ),
                                 mkU64( 0x0 ) ) );

            assign( rB_1, binop( Iop_CmpEQ64,
                                 mkexpr( rB ),
                                 mkU64( 0x1 ) ) );

            assign( rB_m1, binop( Iop_CmpEQ64,
                                  mkexpr( rB ),
                                  mkU64( 0xFFFFFFFFFFFFFFFF ) ) );

            assign( rA_m1, binop( Iop_CmpEQ64,
                                 mkexpr( rA ),
                                 mkU64( 0xFFFFFFFFFFFFFFFF ) ) );

            /* Need to match the HW for these special cases
               rA = -2^31 and rB = -1              result all zeros
               rA =  -1 and rB = -1                result all zeros

               if an attempt is made to perform any of the divisions:
                     0x80000000 % -1
                     <anything> % 0
               result is undefined.  Force result to zero to match the
               HW behaviour.  */

            assign( resultis0,
                    binop( Iop_Or64,
                           binop( Iop_Or64,
                                  /* -1 % 1 */
                                  binop( Iop_And64,
                                         unop( Iop_1Sto64, mkexpr( rA_m1 ) ),
                                         unop( Iop_1Sto64, mkexpr( rB_1 ) ) ),
                                  /* rA % 0 (division by zero) */
                                  unop( Iop_1Sto64, mkexpr( rB_0 ) ) ),
                           binop( Iop_Or64,
                                  binop( Iop_And64,
                                         unop( Iop_Not64,
                                               mkexpr( rA2_63 ) ),
                                         unop ( Iop_1Sto64,
                                                mkexpr( rB_m1 ) ) ),
                                  /* -1 % -1 */
                                  binop( Iop_And64,
                                         unop( Iop_1Sto64, mkexpr( rA_m1 ) ),
                                         unop( Iop_1Sto64, mkexpr( rB_m1 ) )
                                     ) ) ) );

            /* The following remainder computation works as long as
             * rA != -2^63 and rB != -1.
             */
            assign( quotient,
                    binop( Iop_DivS64, mkexpr( rA ), mkexpr( rB ) ) );

            assign( quotientTimesDivisor,
                    binop( Iop_Mul64,
                           mkexpr( quotient ),
                           mkexpr( rB ) ) );

            assign( remainder,
                    binop( Iop_Sub64,
                           mkexpr( rA ),
                           mkexpr( quotientTimesDivisor ) ) );

            assign( rD, binop( Iop_And64,
                                mkexpr( remainder ),
                                unop( Iop_Not64,
                                      mkexpr( resultis0 ) ) ) );
            break;
         }
      case 0x30B: // modsw  Modulo Signed Word
         {
            IRTemp rA     = newTemp( Ity_I32 );
            IRTemp rB     = newTemp( Ity_I32 );
            IRTemp rA2_32 = newTemp( Ity_I32 );  /* all 1's if rA = -2^32 */
            IRTemp rB_0   = newTemp( Ity_I1 );   /* 1 if rB = 0 */
            IRTemp rB_1   = newTemp( Ity_I1 );   /* 1 if rB = 1 */
            IRTemp rB_m1  = newTemp( Ity_I1 );   /* 1 if rB = -1, 0xFFFFFFFF */
            IRTemp rA_m1   = newTemp( Ity_I1 );  /* 1 if rA = -1, 0xFFFFFFFF */
            IRTemp resultis0   = newTemp( Ity_I32 );
            IRTemp quotient = newTemp( Ity_I32 );
            IRTemp quotientTimesDivisor = newTemp( Ity_I32 );
            IRTemp remainder = newTemp( Ity_I32 );

            DIP("modsw r%u,r%u,r%u\n", rD_addr, rA_addr, rB_addr);

            if ( ty == Ity_I64 ) {
               /* rA and rB are 32 bit values in bits 32:63 of the
                * 64-bit register.
                */
               assign( rA, unop(Iop_64to32, getIReg(rA_addr) ) );
               assign( rB, unop(Iop_64to32, getIReg(rB_addr) ) );

            } else {
               assign( rA, getIReg(rA_addr) );
               assign( rB, getIReg(rB_addr) );
            }

            assign( rA2_32, unop( Iop_1Sto32,
                                  binop( Iop_CmpEQ32,
                                         mkexpr( rA ),
                                         mkU32( 0x80000000 ) ) ) );
            /* If the divisor is zero, then the result is undefined.
             * However, we will make the result be zero to match what
             * the hardware does.
             */
            assign( rB_0, binop( Iop_CmpEQ32,
                                 mkexpr( rB ),
                                 mkU32( 0x0 ) ) );

            assign( rB_1, binop( Iop_CmpEQ32,
                                 mkexpr( rB ),
                                 mkU32( 0x00000001 ) ) );

            assign( rB_m1, binop( Iop_CmpEQ32,
                                  mkexpr( rB ),
                                  mkU32( 0xFFFFFFFF ) ) );

            assign( rA_m1, binop( Iop_CmpEQ32,
                                 mkexpr( rA ),
                                 mkU32( 0xFFFFFFFF ) ) );

            /* Need to match the HW for these special cases
               rA = -2^31 and rB = -1              result all zeros
               rA =  -1 and rB = -1                result all zeros
               rA =  -1 and rB = 1                 result all zeros

               if an attempt is made to perform any of the divisions:
                     0x80000000 % -1
                     <anything> % 0
               result is undefined.  Force result to zero to match the
               HW beaviour.  */

            assign( resultis0,
                    binop( Iop_Or32,
                           binop( Iop_Or32,
                                  /* -1 % 1 */
                                  binop( Iop_And32,
                                         unop( Iop_1Sto32, mkexpr( rA_m1 ) ),
                                         unop( Iop_1Sto32, mkexpr( rB_1 ) ) ),
                                  /* rA % 0 (division by zero) */
                                  unop( Iop_1Sto32, mkexpr( rB_0 ) ) ),

                           binop( Iop_Or32,
                                  /* 0x8000000 % -1 */
                                  binop( Iop_And32,
                                         mkexpr( rA2_32 ),
                                         unop( Iop_1Sto32,
                                               mkexpr( rB_m1 ) ) ),
                                  /* -1 % -1 */
                                  binop( Iop_And32,
                                         unop( Iop_1Sto32, mkexpr( rA_m1 ) ),
                                         unop( Iop_1Sto32, mkexpr( rB_m1 ) )
                                     ) ) ) );

            /* The following remainder computation works as long as
             * rA != -2^31 and rB != -1.
             */
            assign( quotient,
                    binop( Iop_DivS32, mkexpr( rA ), mkexpr( rB ) ) );

            assign( quotientTimesDivisor,
                    unop( Iop_64to32,
                          binop( Iop_MullS32,
                                 mkexpr( quotient ),
                                 mkexpr( rB ) ) ) );

            assign( remainder,
                    binop( Iop_Sub32,
                           mkexpr( rA ),
                           mkexpr( quotientTimesDivisor ) ) );

            assign( rD, binop( Iop_32HLto64,
                                mkU32( 0 ),
                                binop( Iop_And32,
                                       mkexpr( remainder ),
                                       unop( Iop_Not32,
                                             mkexpr( resultis0 ) ) ) ) );
            break;
         }

      default:
         vex_printf("dis_modulo_int(ppc)(opc2)\n");
         return False;
      }
      break;

   default:
      vex_printf("dis_modulo_int(ppc)(opc1)\n");
      return False;
   }

   putIReg( rD_addr, mkexpr( rD ) );

   return True;
}


/*
  Byte Compare Instructions
*/
static Bool dis_byte_cmp ( UInt prefix, UInt theInstr )
{
   /* X-Form */
   UChar opc1 = ifieldOPC(theInstr);
   UInt  opc2 = ifieldOPClo10(theInstr);
   UChar rA_addr = ifieldRegA(theInstr);
   UChar rB_addr = ifieldRegB(theInstr);
   IRTemp rA     = newTemp(Ity_I64);
   IRTemp rB     = newTemp(Ity_I64);
   UChar L    = toUChar( IFIELD( theInstr, 21, 1 ) );
   UChar BF   = toUChar( IFIELD( theInstr, 23, 3 ) );

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   assign( rA, getIReg(rA_addr) );
   assign( rB, getIReg(rB_addr) );

   if (opc1 != 0x1F) {
      vex_printf("dis_byte_cmp(ppc)(opc1)\n");
      return False;
   }

   switch (opc2) {
   case 0xc0: // cmprb (Compare Ranged Byte)
      {
         IRExpr *value;
         IRExpr *hi_1, *lo_1, *hi_2, *lo_2;
         IRExpr *inrange_1, *inrange_2;

         DIP("cmprb %u,%u,r%u,r%u\n", BF, L, rA_addr, rB_addr);

         hi_1 = binop( Iop_Shr64,
                       binop( Iop_And64,
                              mkexpr( rB ),
                              mkU64( 0xFF000000 ) ),
                       mkU8( 24 ) );
         lo_1 = binop( Iop_Shr64,
                       binop( Iop_And64,
                              mkexpr( rB ),
                              mkU64( 0xFF0000 ) ) ,
                       mkU8( 16 ) );
         hi_2 = binop( Iop_Shr64,
                       binop( Iop_And64,
                              mkexpr( rB ),
                              mkU64( 0xFF00 ) ),
                       mkU8( 8 ) );
         lo_2 = binop( Iop_And64,
                       mkexpr( rB ),
                       mkU64( 0xFF ) );
         value = binop( Iop_And64,
                        mkexpr( rA ),
                        mkU64( 0xFF ) );

         inrange_1 = mkAND1( binop( Iop_CmpLE64U, value, hi_1 ),
                             mkNOT1( binop( Iop_CmpLT64U, value, lo_1 ) ) );
         inrange_2 = mkAND1( binop( Iop_CmpLE64U, value, hi_2 ),
                             mkNOT1( binop( Iop_CmpLT64U, value, lo_2 ) ) );

         putGST_field( PPC_GST_CR,
                       binop( Iop_Shl32,
                              binop( Iop_Or32,
                                     unop( Iop_1Uto32, inrange_2 ),
                                     binop( Iop_And32,
                                            mkU32 ( L ),
                                            unop( Iop_1Uto32, inrange_1 ) ) ),
                              mkU8( 2 ) ),
                       BF );
      }
      break;

   case 0xE0: // cmpeqb (Compare Equal Byte)
      {
         Int i;
         IRTemp tmp[9];
         IRExpr *value;

         DIP("cmpeqb %u,r%u,r%u\n", BF, rA_addr, rB_addr);

         value = binop( Iop_And64,
                        mkexpr( rA ),
                        mkU64( 0xFF ) );

         tmp[0] = newTemp(Ity_I32);
         assign( tmp[0], mkU32( 0 ) );

         for(i = 0; i < 8; i++) {
            tmp[i+1] = newTemp(Ity_I32);
            assign( tmp[i+1], binop( Iop_Or32,
                                     unop( Iop_1Uto32,
                                           binop( Iop_CmpEQ64,
                                                  value,
                                                  binop( Iop_And64,
                                                         binop( Iop_Shr64,
                                                                mkexpr( rB ),
                                                                mkU8( i*8 ) ),
                                                         mkU64( 0xFF ) ) ) ),
                                     mkexpr( tmp[i] ) ) );
         }

         putGST_field( PPC_GST_CR,
                       binop( Iop_Shl32,
                              unop( Iop_1Uto32,
                                    mkNOT1( binop( Iop_CmpEQ32,
                                                   mkexpr( tmp[8] ),
                                                   mkU32( 0 ) ) ) ),
                              mkU8( 2 ) ),
                       BF );
      }
      break;

   default:
      vex_printf("dis_byte_cmp(ppc)(opc2)\n");
      return False;
   }
   return True;
}

/*
 * Integer Miscellaneous instructions
 */
static Bool dis_int_misc ( UInt prefix, UInt theInstr )
{
   Int wc = IFIELD(theInstr, 21, 2);
   UChar opc1 = ifieldOPC(theInstr);
   UInt  opc2 = ifieldOPClo10(theInstr);

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   if ( opc1 != 0x1F ) {
      vex_printf("dis_modulo_int(ppc)(opc1)\n");
      return False;
   }

   switch (opc2) {
   case 0x01E: // wait, (X-from)
      DIP("wait %d\n", wc);

      /* The wait instruction causes instruction fetching and execution
       * to be suspended.  Instruction fetching and execution are resumed
       * when the events specified by the WC field occur.
       *
       *    0b00   Resume instruction fetching and execution when an
       *           exception or an event-based branch exception occurs,
       *           or a resume signal from the platform is recieved.
       *
       *    0b01   Reserved.
       *
       *  For our purposes, we will just assume the contition is always
       * immediately satisfied.
       */
      break;
   default:
      vex_printf("dis_int_misc(ppc)(opc2)\n");
      return False;
}

   return True;
}

/*
  Integer Compare Instructions
*/
static Bool dis_int_cmp ( UInt prefix, UInt theInstr )
{
   /* D-Form, X-Form */
   UChar opc1    = ifieldOPC(theInstr);
   UChar crfD    = toUChar( IFIELD( theInstr, 23, 3 ) );
   UChar b22     = toUChar( IFIELD( theInstr, 22, 1 ) );
   UChar flag_L  = toUChar( IFIELD( theInstr, 21, 1 ) );
   UChar rA_addr = ifieldRegA(theInstr);
   UInt  uimm16  = ifieldUIMM16(theInstr);
   UChar rB_addr = ifieldRegB(theInstr);
   UInt  opc2    = ifieldOPClo10(theInstr);
   UChar b0      = ifieldBIT0(theInstr);

   IRType ty = mode64 ? Ity_I64 : Ity_I32;
   IRExpr *a = getIReg(rA_addr);
   IRExpr *b;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   if (!mode64 && flag_L==1) {  // L==1 invalid for 32 bit.
      vex_printf("dis_int_cmp(ppc)(flag_L)\n");
      return False;
   }
   
   if (( b22 != 0 ) && ( opc2 != 0x080 ) ) {   // setb case exception
      vex_printf("dis_int_cmp(ppc)(b22)\n");
      return False;
   }
   
   switch (opc1) {
   case 0x0B: // cmpi (Compare Immediate, PPC32 p368)
      DIP("cmpi cr%u,%u,r%u,%d\n", crfD, flag_L, rA_addr,
          (Int)extend_s_16to32(uimm16));
      b = mkSzExtendS16( ty, uimm16 );
      if (flag_L == 1) {
         putCR321(crfD, unop(Iop_64to8, binop(Iop_CmpORD64S, a, b)));
      } else {
         a = mkNarrowTo32( ty, a );
         b = mkNarrowTo32( ty, b );
         putCR321(crfD, unop(Iop_32to8, binop(Iop_CmpORD32S, a, b)));
      }
      putCR0( crfD, getXER_SO() );
      break;
      
   case 0x0A: // cmpli (Compare Logical Immediate, PPC32 p370)
      DIP("cmpli cr%u,%u,r%u,0x%x\n", crfD, flag_L, rA_addr, uimm16);
      b = mkSzImm( ty, uimm16 );
      if (flag_L == 1) {
         putCR321(crfD, unop(Iop_64to8, binop(Iop_CmpORD64U, a, b)));
      } else {
         a = mkNarrowTo32( ty, a );
         b = mkNarrowTo32( ty, b );
         putCR321(crfD, unop(Iop_32to8, binop(Iop_CmpORD32U, a, b)));
      }
      putCR0( crfD, getXER_SO() );
      break;
      
   /* X Form */
   case 0x1F:
      if (b0 != 0) {
         vex_printf("dis_int_cmp(ppc)(0x1F,b0)\n");
         return False;
      }
      b = getIReg(rB_addr);

      switch (opc2) {
      case 0x000: // cmp (Compare, PPC32 p367)
         DIP("cmp cr%u,%u,r%u,r%u\n", crfD, flag_L, rA_addr, rB_addr);
         /* Comparing a reg with itself produces a result which
            doesn't depend on the contents of the reg.  Therefore
            remove the false dependency, which has been known to cause
            memcheck to produce false errors. */
         if (rA_addr == rB_addr)
            a = b = typeOfIRExpr(irsb->tyenv,a) == Ity_I64
                    ? mkU64(0)  : mkU32(0);
         if (flag_L == 1) {
            putCR321(crfD, unop(Iop_64to8, binop(Iop_CmpORD64S, a, b)));
         } else {
            a = mkNarrowTo32( ty, a );
            b = mkNarrowTo32( ty, b );
            putCR321(crfD, unop(Iop_32to8,binop(Iop_CmpORD32S, a, b)));
         }
         putCR0( crfD, getXER_SO() );
         break;
         
      case 0x020: // cmpl (Compare Logical, PPC32 p369)
         DIP("cmpl cr%u,%u,r%u,r%u\n", crfD, flag_L, rA_addr, rB_addr);
         /* Comparing a reg with itself produces a result which
            doesn't depend on the contents of the reg.  Therefore
            remove the false dependency, which has been known to cause
            memcheck to produce false errors. */
         if (rA_addr == rB_addr)
            a = b = typeOfIRExpr(irsb->tyenv,a) == Ity_I64
                    ? mkU64(0)  : mkU32(0);
         if (flag_L == 1) {
            putCR321(crfD, unop(Iop_64to8, binop(Iop_CmpORD64U, a, b)));
         } else {
            a = mkNarrowTo32( ty, a );
            b = mkNarrowTo32( ty, b );
            putCR321(crfD, unop(Iop_32to8, binop(Iop_CmpORD32U, a, b)));
         }
         putCR0( crfD, getXER_SO() );
         break;

      case 0x080: // setb (Set Boolean)
         {
            /* Set Boolean Condition in result register.  The result register
               is set to all ones if the condition is true and all zeros
               otherwise.  */
            UChar rT_addr = ifieldRegDS(theInstr);
            Int bfa = IFIELD(theInstr, 18, 3);
            IRTemp cr = newTemp(Ity_I32);
            IRTemp cr0 = newTemp(Ity_I32);
            IRTemp cr1 = newTemp(Ity_I32);
            IRTemp result = newTemp(Ity_I64);

            DIP("setb r%u,%d\n", rT_addr, bfa);

            /* Fetch the entire condition code value */
            assign( cr, getGST( PPC_GST_CR ) );

            /* Get bit zero (IBM numbering) of the CR field specified
             * by bfa.
             */
            assign( cr0, binop( Iop_And32,
                                binop( Iop_Shr32,
                                       mkexpr( cr ),
                                       mkU8( (7-bfa)*4 ) ),
                                mkU32( 0x8 ) ) );
            assign( cr1, binop( Iop_And32,
                                binop( Iop_Shr32,
                                       mkexpr( cr ),
                                       mkU8( (7-bfa)*4 ) ),
                                mkU32( 0x4 ) ) );
            assign( result, binop( Iop_Or64,
                                   unop( Iop_1Sto64,
                                       binop( Iop_CmpEQ32,
                                              mkexpr( cr0 ),
                                              mkU32( 0x8 ) ) ),
                                   binop( Iop_32HLto64,
                                          mkU32( 0 ),
                                          unop( Iop_1Uto32,
                                                binop( Iop_CmpEQ32,
                                                       mkexpr( cr1 ),
                                                       mkU32( 0x4 ) ) ) ) ) );
            if ( ty == Ity_I64 )
               putIReg( rT_addr, mkexpr( result ) );
            else
               putIReg( rT_addr, unop( Iop_64to32, mkexpr(result ) ) );
         }
         break;
      default:
         vex_printf("dis_int_cmp(ppc)(opc2)\n");
         return False;
      }
      break;
      
   default:
      vex_printf("dis_int_cmp(ppc)(opc1)\n");
      return False;
   }
   
   return True;
}


/*
  Integer Logical Instructions
*/
static Bool dis_int_logic ( UInt prefix, UInt theInstr )
{
   /* D-Form, X-Form */
   UChar opc1    = ifieldOPC(theInstr);
   UChar rS_addr = ifieldRegDS(theInstr);
   UChar rA_addr = ifieldRegA(theInstr);
   UInt  uimm16  = ifieldUIMM16(theInstr);
   UChar rB_addr = ifieldRegB(theInstr);
   UInt  opc2    = ifieldOPClo10(theInstr);
   UChar flag_rC = ifieldBIT0(theInstr);
   
   IRType ty     = mode64 ? Ity_I64 : Ity_I32;
   IRTemp rS     = newTemp(ty);
   IRTemp rA     = newTemp(ty);
   IRTemp rB     = newTemp(ty);
   Bool do_rc    = False;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   assign( rS, getIReg(rS_addr) );
   assign( rB, getIReg(rB_addr) );
   
   switch (opc1) {
   case 0x1C: // andi. (AND Immediate, PPC32 p358)
      DIP("andi. r%u,r%u,0x%x\n", rA_addr, rS_addr, uimm16);
      assign( rA, binop( mkSzOp(ty, Iop_And8), mkexpr(rS),
                         mkSzImm(ty, uimm16)) );
      do_rc = True;  // Always record to CR
      flag_rC = 1;
      break;
      
   case 0x1D: // andis. (AND Immediate Shifted, PPC32 p359)
      DIP("andis r%u,r%u,0x%x\n", rA_addr, rS_addr, uimm16);
      assign( rA, binop( mkSzOp(ty, Iop_And8), mkexpr(rS),
                         mkSzImm(ty, uimm16 << 16)) );
      do_rc = True;  // Always record to CR
      flag_rC = 1;
      break;

   case 0x18: // ori (OR Immediate, PPC32 p497)
      DIP("ori r%u,r%u,0x%x\n", rA_addr, rS_addr, uimm16);
      assign( rA, binop( mkSzOp(ty, Iop_Or8), mkexpr(rS),
                         mkSzImm(ty, uimm16)) );
      break;

   case 0x19: // oris (OR Immediate Shifted, PPC32 p498)
      DIP("oris r%u,r%u,0x%x\n", rA_addr, rS_addr, uimm16);
      assign( rA, binop( mkSzOp(ty, Iop_Or8), mkexpr(rS),
                         mkSzImm(ty, uimm16 << 16)) );
      break;

   case 0x1A: // xori (XOR Immediate, PPC32 p550)
      DIP("xori r%u,r%u,0x%x\n", rA_addr, rS_addr, uimm16);
      assign( rA, binop( mkSzOp(ty, Iop_Xor8), mkexpr(rS),
                         mkSzImm(ty, uimm16)) );
      break;

   case 0x1B: // xoris (XOR Immediate Shifted, PPC32 p551)
      DIP("xoris r%u,r%u,0x%x\n", rA_addr, rS_addr, uimm16);
      assign( rA, binop( mkSzOp(ty, Iop_Xor8), mkexpr(rS),
                         mkSzImm(ty, uimm16 << 16)) );
      break;

   /* X Form */
   case 0x1F:

      opc2 = IFIELD( theInstr, 2, 9 );

      switch ( opc2 ) {
      case 0x1BD: // extswsli (Extend Sign Word shift left)
         {
            /* sh[5] is in bit 1, sh[0:4] is in bits [14:10] of theInstr */
            UChar sh = IFIELD( theInstr, 11, 5 ) | (IFIELD(theInstr, 1, 1) << 5);
            IRTemp temp = newTemp( ty );

            DIP("extswsli%s r%u,r%u,%u\n", flag_rC ? ".":"",
                rA_addr, rS_addr, sh);

            assign( temp, unop( Iop_32Sto64,
                                unop( Iop_64to32, mkexpr( rS ) ) ) );
            assign( rA, binop( Iop_Shl64, mkexpr( temp ), mkU8( sh ) ) );
            putIReg( rA_addr, mkexpr( rA ) );

            if ( flag_rC ) {
               set_CR0( mkexpr( rA ) );
            }
            return True;
         }
      default:
         break;  // drop to next opc2 check
      }

      do_rc = True; // All below record to CR, except for where we return at case end.

      opc2 = ifieldOPClo10( theInstr );

      switch (opc2) {
      case 0x01C: // and (AND, PPC32 p356)
         DIP("and%s r%u,r%u,r%u\n",
             flag_rC ? ".":"", rA_addr, rS_addr, rB_addr);
         assign(rA, binop( mkSzOp(ty, Iop_And8),
                           mkexpr(rS), mkexpr(rB)));
         break;
         
      case 0x03C: // andc (AND with Complement, PPC32 p357)
         DIP("andc%s r%u,r%u,r%u\n",
             flag_rC ? ".":"", rA_addr, rS_addr, rB_addr);
         assign(rA, binop( mkSzOp(ty, Iop_And8), mkexpr(rS),
                           unop( mkSzOp(ty, Iop_Not8),
                                 mkexpr(rB))));
         break;
         
      case 0x01A: { // cntlzw (Count Leading Zeros Word, PPC32 p371)
         if (rB_addr!=0) {
            vex_printf("dis_int_logic(ppc)(cntlzw,rB_addr)\n");
            return False;
         }
         DIP("cntlzw%s r%u,r%u\n", flag_rC ? ".":"", rA_addr, rS_addr);
         
         // mode64: count in low word only
         IRExpr* lo32 = mode64 ? unop(Iop_64to32, mkexpr(rS)) : mkexpr(rS);
         IRExpr* res32 = unop(Iop_ClzNat32, lo32);
         assign(rA, mode64 ? unop(Iop_32Uto64, res32) : res32);
         break;
      }
         
      case 0x11C: // eqv (Equivalent, PPC32 p396)
         DIP("eqv%s r%u,r%u,r%u\n",
             flag_rC ? ".":"", rA_addr, rS_addr, rB_addr);
         assign( rA, unop( mkSzOp(ty, Iop_Not8),
                           binop( mkSzOp(ty, Iop_Xor8),
                                  mkexpr(rS), mkexpr(rB))) );
         break;

      case 0x3BA: // extsb (Extend Sign Byte, PPC32 p397
         if (rB_addr!=0) {
            vex_printf("dis_int_logic(ppc)(extsb,rB_addr)\n");
            return False;
         }
         DIP("extsb%s r%u,r%u\n",
             flag_rC ? ".":"", rA_addr, rS_addr);
         if (mode64)
            assign( rA, unop(Iop_8Sto64, unop(Iop_64to8, mkexpr(rS))) );
         else
            assign( rA, unop(Iop_8Sto32, unop(Iop_32to8, mkexpr(rS))) );
         break;

      case 0x39A: // extsh (Extend Sign Half Word, PPC32 p398)
         if (rB_addr!=0) {
            vex_printf("dis_int_logic(ppc)(extsh,rB_addr)\n");
            return False;
         }
         DIP("extsh%s r%u,r%u\n",
             flag_rC ? ".":"", rA_addr, rS_addr);
         if (mode64)
            assign( rA, unop(Iop_16Sto64,
                             unop(Iop_64to16, mkexpr(rS))) );
         else
            assign( rA, unop(Iop_16Sto32,
                             unop(Iop_32to16, mkexpr(rS))) );
         break;

      case 0x1DC: // nand (NAND, PPC32 p492)
         DIP("nand%s r%u,r%u,r%u\n",
             flag_rC ? ".":"", rA_addr, rS_addr, rB_addr);
         assign( rA, unop( mkSzOp(ty, Iop_Not8),
                           binop( mkSzOp(ty, Iop_And8),
                                  mkexpr(rS), mkexpr(rB))) );
         break;
         
      case 0x07C: // nor (NOR, PPC32 p494)
         DIP("nor%s r%u,r%u,r%u\n",
             flag_rC ? ".":"", rA_addr, rS_addr, rB_addr);
         assign( rA, unop( mkSzOp(ty, Iop_Not8),
                           binop( mkSzOp(ty, Iop_Or8),
                                  mkexpr(rS), mkexpr(rB))) );
         break;

      case 0x1BC: // or (OR, PPC32 p495)
         if ((!flag_rC) && rS_addr == rB_addr) {
            DIP("mr r%u,r%u\n", rA_addr, rS_addr);
            assign( rA, mkexpr(rS) );
         } else {
            DIP("or%s r%u,r%u,r%u\n",
                flag_rC ? ".":"", rA_addr, rS_addr, rB_addr);
            assign( rA, binop( mkSzOp(ty, Iop_Or8),
                               mkexpr(rS), mkexpr(rB)) );
         }
         break;

      case 0x19C: // orc  (OR with Complement, PPC32 p496)
         DIP("orc%s r%u,r%u,r%u\n",
             flag_rC ? ".":"", rA_addr, rS_addr, rB_addr);
         assign( rA, binop( mkSzOp(ty, Iop_Or8), mkexpr(rS),
                            unop(mkSzOp(ty, Iop_Not8), mkexpr(rB))));
         break;
         
      case 0x13C: // xor (XOR, PPC32 p549)
         DIP("xor%s r%u,r%u,r%u\n",
             flag_rC ? ".":"", rA_addr, rS_addr, rB_addr);
         assign( rA, binop( mkSzOp(ty, Iop_Xor8),
                            mkexpr(rS), mkexpr(rB)) );
         break;


      /* 64bit Integer Logical Instructions */
      case 0x3DA: // extsw (Extend Sign Word, PPC64 p430)
         if (rB_addr!=0) {
            vex_printf("dis_int_logic(ppc)(extsw,rB_addr)\n");
            return False;
         }
         DIP("extsw%s r%u,r%u\n", flag_rC ? ".":"", rA_addr, rS_addr);
         assign(rA, unop(Iop_32Sto64, unop(Iop_64to32, mkexpr(rS))));
         break;

      case 0x03A: // cntlzd (Count Leading Zeros DWord, PPC64 p401)
         if (rB_addr!=0) {
            vex_printf("dis_int_logic(ppc)(cntlzd,rB_addr)\n");
            return False;
         }
         DIP("cntlzd%s r%u,r%u\n", flag_rC ? ".":"", rA_addr, rS_addr);
         assign(rA, unop(Iop_ClzNat64, mkexpr(rS)));
         break;

      case 0x1FC: // cmpb (Power6: compare bytes)
         DIP("cmpb r%u,r%u,r%u\n", rA_addr, rS_addr, rB_addr);

         if (mode64)
            assign( rA, unop( Iop_V128to64,
                              binop( Iop_CmpEQ8x16,
                                     binop( Iop_64HLtoV128, mkU64(0), mkexpr(rS) ),
                                     binop( Iop_64HLtoV128, mkU64(0), mkexpr(rB) )
                                     )) );
         else
            assign( rA, unop( Iop_V128to32,
                              binop( Iop_CmpEQ8x16,
                                     unop( Iop_32UtoV128, mkexpr(rS) ),
                                     unop( Iop_32UtoV128, mkexpr(rB) )
                                     )) );
         break;

      case 0x2DF: { // mftgpr (move floating-point to general purpose register)
         /* The mftgpr instruction was deprecated in Power 7, 2009 timeframe.
            Leaving support in Valgrind for now (9/10/2021).  Can remove the
            mftgpr support in Valgrind if the opcode ever gets reused.  */
         IRTemp frB = newTemp(Ity_F64);
         DIP("mftgpr r%u,fr%u\n", rS_addr, rB_addr);

         assign( frB, getFReg(rB_addr));  // always F64
         if (mode64)
            assign( rA, unop( Iop_ReinterpF64asI64, mkexpr(frB)) );
         else
            assign( rA, unop( Iop_64to32, unop( Iop_ReinterpF64asI64, mkexpr(frB))) );

         putIReg( rS_addr, mkexpr(rA));
         return True;
      }

      case 0x25F: { // mffgpr (move floating-point from general purpose register)
         /* The mffgpr instruction was deprecated in Power 7, 2009 timeframe.
            Leaving support in Valgrind for now (9/10/2021).  Can remove the
            mftgpr support in Valgrind if the opcode ever gets reused.  */
         IRTemp frA = newTemp(Ity_F64);
         DIP("mffgpr fr%u,r%u\n", rS_addr, rB_addr);

         if (mode64)
            assign( frA, unop( Iop_ReinterpI64asF64, mkexpr(rB)) );
         else
            assign( frA, unop( Iop_ReinterpI64asF64, unop( Iop_32Uto64, mkexpr(rB))) );

         putFReg( rS_addr, mkexpr(frA));
         return True;
      }
      case 0x1FA: // popcntd (population count doubleword)
      {
          vassert(mode64);
    	  DIP("popcntd r%u,r%u\n", rA_addr, rS_addr);
    	  IRTemp result = gen_POPCOUNT(ty, rS, DWORD);
    	  putIReg( rA_addr, mkexpr(result) );
    	  return True;
      }
      case 0x17A: // popcntw (Population Count Words)
      {
         DIP("popcntw r%u,r%u\n", rA_addr, rS_addr);
         if (mode64) {
            IRTemp resultHi, resultLo;
            IRTemp argLo = newTemp(Ity_I32);
            IRTemp argHi = newTemp(Ity_I32);
            assign(argLo, unop(Iop_64to32, mkexpr(rS)));
            assign(argHi, unop(Iop_64HIto32, mkexpr(rS)));
            resultLo = gen_POPCOUNT(Ity_I32, argLo, WORD);
            resultHi = gen_POPCOUNT(Ity_I32, argHi, WORD);
            putIReg( rA_addr, binop(Iop_32HLto64, mkexpr(resultHi), mkexpr(resultLo)));
         } else {
            IRTemp result = gen_POPCOUNT(ty, rS, WORD);
            putIReg( rA_addr, mkexpr(result) );
         }
         return True;
      }
      case 0x7A: // popcntb (Population Count Byte)
      {
         DIP("popcntb r%u,r%u\n", rA_addr, rS_addr);

         if (mode64) {
            IRTemp resultHi, resultLo;
            IRTemp argLo = newTemp(Ity_I32);
            IRTemp argHi = newTemp(Ity_I32);
            assign(argLo, unop(Iop_64to32, mkexpr(rS)));
            assign(argHi, unop(Iop_64HIto32, mkexpr(rS)));
            resultLo = gen_POPCOUNT(Ity_I32, argLo, BYTE);
            resultHi = gen_POPCOUNT(Ity_I32, argHi, BYTE);
            putIReg( rA_addr, binop(Iop_32HLto64, mkexpr(resultHi),
                                    mkexpr(resultLo)));
         } else {
            IRTemp result = gen_POPCOUNT(ty, rS, BYTE);
            putIReg( rA_addr, mkexpr(result) );
         }
         return True;
      }
       case 0x0FC: // bpermd (Bit Permute Doubleword)
       {
          /* This is a lot of rigmarole to emulate bpermd like this, as it
           * could be done much faster by implementing a call to the native
           * instruction.  However, where possible I want to avoid using new
           * native instructions so that we can use valgrind to emulate those
           * instructions on older PPC64 hardware.
           */
 #define BPERMD_IDX_MASK 0x00000000000000FFULL
 #define BPERMD_BIT_MASK 0x8000000000000000ULL
          int i;
          IRExpr * rS_expr = mkexpr(rS);
          IRExpr * res = binop(Iop_And64, mkU64(0), mkU64(0));
          DIP("bpermd r%u,r%u,r%u\n", rA_addr, rS_addr, rB_addr);
          for (i = 0; i < 8; i++) {
             IRTemp idx_tmp = newTemp( Ity_I64 );
             IRTemp perm_bit = newTemp( Ity_I64 );
             IRTemp idx = newTemp( Ity_I8 );
             IRTemp idx_LT64 = newTemp( Ity_I1 );
             IRTemp idx_LT64_ity64 = newTemp( Ity_I64 );

             assign( idx_tmp,
                     binop( Iop_And64, mkU64( BPERMD_IDX_MASK ), rS_expr ) );
             assign( idx_LT64,
                           binop( Iop_CmpLT64U, mkexpr( idx_tmp ), mkU64( 64 ) ) );
             assign( idx,
                           binop( Iop_And8,
                                  unop( Iop_1Sto8,
                                        mkexpr(idx_LT64) ),
                                  unop( Iop_64to8, mkexpr( idx_tmp ) ) ) );
             /* If idx_LT64 == 0, we must force the perm bit to '0'. Below, we se idx
              * to determine which bit of rB to use for the perm bit, and then we shift
              * that bit to the MSB position.  We AND that with a 64-bit-ized idx_LT64
              * to set the final perm bit.
              */
             assign( idx_LT64_ity64,
                           unop( Iop_32Uto64, unop( Iop_1Uto32, mkexpr(idx_LT64 ) ) ) );
             assign( perm_bit,
                           binop( Iop_And64,
                                  mkexpr( idx_LT64_ity64 ),
                                  binop( Iop_Shr64,
                                         binop( Iop_And64,
                                                mkU64( BPERMD_BIT_MASK ),
                                                binop( Iop_Shl64,
                                                       mkexpr( rB ),
                                                       mkexpr( idx ) ) ),
                                         mkU8( 63 ) ) ) );
             res = binop( Iop_Or64,
                                res,
                                binop( Iop_Shl64,
                                       mkexpr( perm_bit ),
                                       mkU8( i ) ) );
             rS_expr = binop( Iop_Shr64, rS_expr, mkU8( 8 ) );
          }
          putIReg(rA_addr, res);
          return True;
       }

      default:
         vex_printf("dis_int_logic(ppc)(opc2)\n");
         return False;
      }
      break;
      
   default:
      vex_printf("dis_int_logic(ppc)(opc1)\n");
      return False;
   }

   putIReg( rA_addr, mkexpr(rA) );

   if (do_rc && flag_rC) {
      set_CR0( mkexpr(rA) );
   }
   return True;
}

/*
  Integer Parity Instructions
*/
static Bool dis_int_parity ( UInt prefix, UInt theInstr )
{
   /* X-Form */
   UChar opc1    = ifieldOPC(theInstr);
   UChar rS_addr = ifieldRegDS(theInstr);
   UChar rA_addr = ifieldRegA(theInstr);
   UChar rB_addr = ifieldRegB(theInstr);
   UInt  opc2    = ifieldOPClo10(theInstr);
   UChar b0      = ifieldBIT0(theInstr);
   IRType ty     = mode64 ? Ity_I64 : Ity_I32;

   IRTemp rS     = newTemp(ty);
   IRTemp rA     = newTemp(ty);
   IRTemp iTot1  = newTemp(Ity_I32);
   IRTemp iTot2  = newTemp(Ity_I32);
   IRTemp iTot3  = newTemp(Ity_I32);
   IRTemp iTot4  = newTemp(Ity_I32);
   IRTemp iTot5  = newTemp(Ity_I32);
   IRTemp iTot6  = newTemp(Ity_I32);
   IRTemp iTot7  = newTemp(Ity_I32);
   IRTemp iTot8  = newTemp(Ity_I32);
   IRTemp rS1    = newTemp(ty);
   IRTemp rS2    = newTemp(ty);
   IRTemp rS3    = newTemp(ty);
   IRTemp rS4    = newTemp(ty);
   IRTemp rS5    = newTemp(ty);
   IRTemp rS6    = newTemp(ty);
   IRTemp rS7    = newTemp(ty);
   IRTemp iHi    = newTemp(Ity_I32);
   IRTemp iLo    = newTemp(Ity_I32);
   IROp to_bit   = (mode64 ? Iop_64to1 : Iop_32to1);
   IROp shr_op   = (mode64 ? Iop_Shr64 : Iop_Shr32);

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   if (opc1 != 0x1f || rB_addr || b0) {
      vex_printf("dis_int_parity(ppc)(0x1F,opc1:rB|b0)\n");
      return False;
   }

   assign( rS, getIReg(rS_addr) );

   switch (opc2) {
   case 0xba:  // prtyd (Parity Doubleword, ISA 2.05 p320)
      DIP("prtyd r%u,r%u\n", rA_addr, rS_addr);
      assign( iTot1, unop(Iop_1Uto32, unop(to_bit, mkexpr(rS))) );
      assign( rS1, binop(shr_op, mkexpr(rS), mkU8(8)) );
      assign( iTot2, binop(Iop_Add32,
                           unop(Iop_1Uto32, unop(to_bit, mkexpr(rS1))),
                           mkexpr(iTot1)) );
      assign( rS2, binop(shr_op, mkexpr(rS1), mkU8(8)) );
      assign( iTot3, binop(Iop_Add32,
                           unop(Iop_1Uto32, unop(to_bit, mkexpr(rS2))),
                           mkexpr(iTot2)) );
      assign( rS3, binop(shr_op, mkexpr(rS2), mkU8(8)) );
      assign( iTot4, binop(Iop_Add32,
                           unop(Iop_1Uto32, unop(to_bit, mkexpr(rS3))),
                           mkexpr(iTot3)) );
      if (mode64) {
         assign( rS4, binop(shr_op, mkexpr(rS3), mkU8(8)) );
         assign( iTot5, binop(Iop_Add32,
                              unop(Iop_1Uto32, unop(to_bit, mkexpr(rS4))),
                              mkexpr(iTot4)) );
         assign( rS5, binop(shr_op, mkexpr(rS4), mkU8(8)) );
         assign( iTot6, binop(Iop_Add32,
                              unop(Iop_1Uto32, unop(to_bit, mkexpr(rS5))),
                              mkexpr(iTot5)) );
         assign( rS6, binop(shr_op, mkexpr(rS5), mkU8(8)) );
         assign( iTot7, binop(Iop_Add32,
                              unop(Iop_1Uto32, unop(to_bit, mkexpr(rS6))),
                              mkexpr(iTot6)) );
         assign( rS7, binop(shr_op, mkexpr(rS6), mkU8(8)) );
         assign( iTot8, binop(Iop_Add32,
                              unop(Iop_1Uto32, unop(to_bit, mkexpr(rS7))),
                              mkexpr(iTot7)) );
         assign( rA, unop(Iop_32Uto64,
                          binop(Iop_And32, mkexpr(iTot8), mkU32(1))) );
      } else
         assign( rA, mkexpr(iTot4) );

      break;
   case 0x9a:  // prtyw (Parity Word, ISA 2.05 p320)
      assign( iTot1, unop(Iop_1Uto32, unop(to_bit, mkexpr(rS))) );
      assign( rS1, binop(shr_op, mkexpr(rS), mkU8(8)) );
      assign( iTot2, binop(Iop_Add32,
                           unop(Iop_1Uto32, unop(to_bit, mkexpr(rS1))),
                           mkexpr(iTot1)) );
      assign( rS2, binop(shr_op, mkexpr(rS1), mkU8(8)) );
      assign( iTot3, binop(Iop_Add32,
                           unop(Iop_1Uto32, unop(to_bit, mkexpr(rS2))),
                           mkexpr(iTot2)) );
      assign( rS3, binop(shr_op, mkexpr(rS2), mkU8(8)) );
      assign( iTot4, binop(Iop_Add32,
                           unop(Iop_1Uto32, unop(to_bit, mkexpr(rS3))),
                           mkexpr(iTot3)) );
      assign( iLo, unop(Iop_1Uto32, unop(Iop_32to1, mkexpr(iTot4) )) );

      if (mode64) {
         assign( rS4, binop(shr_op, mkexpr(rS3), mkU8(8)) );
         assign( iTot5, unop(Iop_1Uto32, unop(to_bit, mkexpr(rS4))) );
         assign( rS5, binop(shr_op, mkexpr(rS4), mkU8(8)) );
         assign( iTot6, binop(Iop_Add32,
                              unop(Iop_1Uto32, unop(to_bit, mkexpr(rS5))),
                              mkexpr(iTot5)) );
         assign( rS6, binop(shr_op, mkexpr(rS5), mkU8(8)) );
         assign( iTot7, binop(Iop_Add32,
                              unop(Iop_1Uto32, unop(to_bit, mkexpr(rS6))),
                              mkexpr(iTot6)) );
         assign( rS7, binop(shr_op, mkexpr(rS6), mkU8(8)));
         assign( iTot8, binop(Iop_Add32,
                              unop(Iop_1Uto32, unop(to_bit, mkexpr(rS7))),
                              mkexpr(iTot7)) );
         assign( iHi, binop(Iop_And32, mkU32(1), mkexpr(iTot8)) ),
            assign( rA, binop(Iop_32HLto64, mkexpr(iHi), mkexpr(iLo)) );
      } else
         assign( rA, binop(Iop_Or32, mkU32(0), mkexpr(iLo)) );
      break;
   default:
      vex_printf("dis_int_parity(ppc)(opc2)\n");
      return False;
   }

   putIReg( rA_addr, mkexpr(rA) );

   return True;
}


/*
  Integer Rotate Instructions
*/
static Bool dis_int_rot ( UInt prefix, UInt theInstr )
{
   /* M-Form, MDS-Form */
   UChar opc1    = ifieldOPC(theInstr);
   UChar rS_addr = ifieldRegDS(theInstr);
   UChar rA_addr = ifieldRegA(theInstr);
   UChar rB_addr = ifieldRegB(theInstr);
   UChar sh_imm  = rB_addr;
   UChar MaskBeg = toUChar( IFIELD( theInstr, 6, 5 ) );
   UChar MaskEnd = toUChar( IFIELD( theInstr, 1, 5 ) );
   UChar msk_imm = toUChar( IFIELD( theInstr, 5, 6 ) );
   UChar opc2    = toUChar( IFIELD( theInstr, 2, 3 ) );
   UChar b1      = ifieldBIT1(theInstr);
   UChar flag_rC = ifieldBIT0(theInstr);

   IRType ty     = mode64 ? Ity_I64 : Ity_I32;
   IRTemp rS     = newTemp(ty);
   IRTemp rA     = newTemp(ty);
   IRTemp rB     = newTemp(ty);
   IRTemp rot    = newTemp(ty);
   IRExpr *r;
   UInt   mask32;
   ULong  mask64;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   assign( rS, getIReg(rS_addr) );
   assign( rB, getIReg(rB_addr) );

   switch (opc1) {
   case 0x14: {
      // rlwimi (Rotate Left Word Imm then Mask Insert, PPC32 p500)
      DIP("rlwimi%s r%u,r%u,%d,%d,%d\n", flag_rC ? ".":"",
          rA_addr, rS_addr, sh_imm, MaskBeg, MaskEnd);
      if (mode64) {
         // tmp32 = (ROTL(rS_Lo32, Imm)
         // rA = ((tmp32 || tmp32) & mask64) | (rA & ~mask64)
         mask64 = MASK64(31-MaskEnd, 31-MaskBeg);
         r = ROTL( unop(Iop_64to32, mkexpr(rS) ), mkU8(sh_imm) );
         r = unop(Iop_32Uto64, r);
         assign( rot, binop(Iop_Or64, r,
                            binop(Iop_Shl64, r, mkU8(32))) );
         assign( rA,
            binop(Iop_Or64,
                  binop(Iop_And64, mkexpr(rot), mkU64(mask64)),
                  binop(Iop_And64, getIReg(rA_addr), mkU64(~mask64))) );
      }
      else {
         // rA = (ROTL(rS, Imm) & mask) | (rA & ~mask);
         mask32 = MASK32(31-MaskEnd, 31-MaskBeg);
         r = ROTL(mkexpr(rS), mkU8(sh_imm));
         assign( rA,
            binop(Iop_Or32,
                  binop(Iop_And32, mkU32(mask32), r),
                  binop(Iop_And32, getIReg(rA_addr), mkU32(~mask32))) );
      }
      break;
   }

   case 0x15: {
      // rlwinm (Rotate Left Word Imm then AND with Mask, PPC32 p501)
      vassert(MaskBeg < 32);
      vassert(MaskEnd < 32);
      vassert(sh_imm  < 32);

      if (mode64) {
         IRTemp rTmp = newTemp(Ity_I64);
         mask64 = MASK64(31-MaskEnd, 31-MaskBeg);
         DIP("rlwinm%s r%u,r%u,%d,%d,%d\n", flag_rC ? ".":"",
             rA_addr, rS_addr, sh_imm, MaskBeg, MaskEnd);
         // tmp32 = (ROTL(rS_Lo32, Imm)
         // rA = ((tmp32 || tmp32) & mask64)
         r = ROTL( unop(Iop_64to32, mkexpr(rS) ), mkU8(sh_imm) );
         r = unop(Iop_32Uto64, r);
         assign( rTmp, r );
         r = NULL;
         assign( rot, binop(Iop_Or64, mkexpr(rTmp),
                            binop(Iop_Shl64, mkexpr(rTmp), mkU8(32))) );
         assign( rA, binop(Iop_And64, mkexpr(rot), mkU64(mask64)) );
      }
      else {
         if (MaskBeg == 0 && sh_imm+MaskEnd == 31) {
            /* Special-case the ,n,0,31-n form as that is just n-bit
               shift left, PPC32 p501 */
            DIP("slwi%s r%u,r%u,%d\n", flag_rC ? ".":"",
                rA_addr, rS_addr, sh_imm);
            assign( rA, binop(Iop_Shl32, mkexpr(rS), mkU8(sh_imm)) );
         }
         else if (MaskEnd == 31 && sh_imm+MaskBeg == 32) {
            /* Special-case the ,32-n,n,31 form as that is just n-bit
               unsigned shift right, PPC32 p501 */
            DIP("srwi%s r%u,r%u,%d\n", flag_rC ? ".":"",
                rA_addr, rS_addr, MaskBeg);
            assign( rA, binop(Iop_Shr32, mkexpr(rS), mkU8(MaskBeg)) );
         }
         else {
            /* General case. */
            mask32 = MASK32(31-MaskEnd, 31-MaskBeg);
            DIP("rlwinm%s r%u,r%u,%d,%d,%d\n", flag_rC ? ".":"",
                rA_addr, rS_addr, sh_imm, MaskBeg, MaskEnd);
            // rA = ROTL(rS, Imm) & mask
            assign( rA, binop(Iop_And32,
                              ROTL(mkexpr(rS), mkU8(sh_imm)), 
                              mkU32(mask32)) );
         }
      }
      break;
   }

   case 0x17: {
      // rlwnm (Rotate Left Word then AND with Mask, PPC32 p503
      DIP("rlwnm%s r%u,r%u,r%u,%d,%d\n", flag_rC ? ".":"",
          rA_addr, rS_addr, rB_addr, MaskBeg, MaskEnd);
      if (mode64) {
         mask64 = MASK64(31-MaskEnd, 31-MaskBeg);
         /* weird insn alert!
            tmp32 = (ROTL(rS_Lo32, rB[0-4])
            rA = ((tmp32 || tmp32) & mask64)
         */
         // note, ROTL does the masking, so we don't do it here
         r = ROTL( unop(Iop_64to32, mkexpr(rS)),
                   unop(Iop_64to8, mkexpr(rB)) );
         r = unop(Iop_32Uto64, r);
         assign(rot, binop(Iop_Or64, r, binop(Iop_Shl64, r, mkU8(32))));
         assign( rA, binop(Iop_And64, mkexpr(rot), mkU64(mask64)) );
      } else {
         mask32 = MASK32(31-MaskEnd, 31-MaskBeg);
         // rA = ROTL(rS, rB[0-4]) & mask
         // note, ROTL does the masking, so we don't do it here
         assign( rA, binop(Iop_And32,
                           ROTL(mkexpr(rS),
                                unop(Iop_32to8, mkexpr(rB))),
                           mkU32(mask32)) );
      }
      break;
   }

   /* 64bit Integer Rotates */
   case 0x1E: {
      msk_imm = ((msk_imm & 1) << 5) | (msk_imm >> 1);
      sh_imm |= b1 << 5;

      vassert( msk_imm < 64 );
      vassert( sh_imm < 64 );

      switch (opc2) {
      case 0x4: {
         /* r = ROTL64( rS, rB_lo6) */
         r = ROTL( mkexpr(rS), unop(Iop_64to8, mkexpr(rB)) );

         if (b1 == 0) { // rldcl (Rotl DWord, Clear Left, PPC64 p555)
            DIP("rldcl%s r%u,r%u,r%u,%u\n", flag_rC ? ".":"",
                rA_addr, rS_addr, rB_addr, msk_imm);
            // note, ROTL does the masking, so we don't do it here
            mask64 = MASK64(0, 63-msk_imm);
            assign( rA, binop(Iop_And64, r, mkU64(mask64)) );
            break;
         } else {       // rldcr (Rotl DWord, Clear Right, PPC64 p556)
            DIP("rldcr%s r%u,r%u,r%u,%u\n", flag_rC ? ".":"",
                rA_addr, rS_addr, rB_addr, msk_imm);
            mask64 = MASK64(63-msk_imm, 63);
            assign( rA, binop(Iop_And64, r, mkU64(mask64)) );
            break;
         }
         break;
      }
      case 0x2: // rldic (Rotl DWord Imm, Clear, PPC64 p557)
         DIP("rldic%s r%u,r%u,%u,%u\n", flag_rC ? ".":"",
             rA_addr, rS_addr, sh_imm, msk_imm);
         r = ROTL(mkexpr(rS), mkU8(sh_imm));
         mask64 = MASK64(sh_imm, 63-msk_imm);
         assign( rA, binop(Iop_And64, r, mkU64(mask64)) );
         break;
         // later: deal with special case: (msk_imm==0) => SHL(sh_imm)
         /*
           Hmm... looks like this'll do the job more simply:
           r = SHL(rS, sh_imm)
           m = ~(1 << (63-msk_imm))
           assign(rA, r & m);
         */
         
      case 0x0: // rldicl (Rotl DWord Imm, Clear Left, PPC64 p558)
         if (mode64
             && sh_imm + msk_imm == 64 && msk_imm >= 1 && msk_imm <= 63) {
            /* special-case the ,64-n,n form as that is just
               unsigned shift-right by n */
            DIP("srdi%s r%u,r%u,%u\n",
                flag_rC ? ".":"", rA_addr, rS_addr, msk_imm);
            assign( rA, binop(Iop_Shr64, mkexpr(rS), mkU8(msk_imm)) );
         } else {
            DIP("rldicl%s r%u,r%u,%u,%u\n", flag_rC ? ".":"",
                rA_addr, rS_addr, sh_imm, msk_imm);
            r = ROTL(mkexpr(rS), mkU8(sh_imm));
            mask64 = MASK64(0, 63-msk_imm);
            assign( rA, binop(Iop_And64, r, mkU64(mask64)) );
         }
         break;
         
      case 0x1: // rldicr (Rotl DWord Imm, Clear Right, PPC64 p559)
         if (mode64 
             && sh_imm + msk_imm == 63 && sh_imm >= 1 && sh_imm <= 63) {
            /* special-case the ,n,63-n form as that is just
               shift-left by n */
            DIP("sldi%s r%u,r%u,%u\n",
                flag_rC ? ".":"", rA_addr, rS_addr, sh_imm);
            assign( rA, binop(Iop_Shl64, mkexpr(rS), mkU8(sh_imm)) );
         } else {
            DIP("rldicr%s r%u,r%u,%u,%u\n", flag_rC ? ".":"",
                rA_addr, rS_addr, sh_imm, msk_imm);
            r = ROTL(mkexpr(rS), mkU8(sh_imm));
            mask64 = MASK64(63-msk_imm, 63);
            assign( rA, binop(Iop_And64, r, mkU64(mask64)) );
         }
         break;
         
      case 0x3: { // rldimi (Rotl DWord Imm, Mask Insert, PPC64 p560)
         IRTemp rA_orig = newTemp(ty);
         DIP("rldimi%s r%u,r%u,%u,%u\n", flag_rC ? ".":"",
             rA_addr, rS_addr, sh_imm, msk_imm);
         r = ROTL(mkexpr(rS), mkU8(sh_imm));
         mask64 = MASK64(sh_imm, 63-msk_imm);
         assign( rA_orig, getIReg(rA_addr) );
         assign( rA, binop(Iop_Or64,
                           binop(Iop_And64, mkU64(mask64),  r),
                           binop(Iop_And64, mkU64(~mask64),
                                            mkexpr(rA_orig))) );
         break;
      }
      default:
         vex_printf("dis_int_rot(ppc)(opc2)\n");
         return False;
      }
      break;         
   }

   default:
      vex_printf("dis_int_rot(ppc)(opc1)\n");
      return False;
   }

   putIReg( rA_addr, mkexpr(rA) );

   if (flag_rC) {
      set_CR0( mkexpr(rA) );
   }
   return True;
}


/*
  Integer Load Instructions
*/
static Bool dis_int_load_ds_form_prefix ( UInt prefix,
                                          UInt theInstr )
{
   /* DS-Form  Prefixed versions */
   UChar opc1     = ifieldOPC(theInstr);
   UChar rT_addr  = ifieldRegDS(theInstr);
   UChar rA_addr  = ifieldRegA(theInstr);
   IRType  ty     = mode64 ? Ity_I64 : Ity_I32;
   UChar   b0     = ifieldBIT0(theInstr);
   UChar   b1     = ifieldBIT1(theInstr);
   IRTemp  EA     = newTemp(ty);
   UInt    ptype  = PrefixType(prefix);
   Bool    is_prefix = prefix_instruction( prefix );
   ULong   immediate_val = 0;
   UInt    R = 0;

   /* Some of these instructions have different encodings for their word
      versions and their prefix versions.  */

   if (opc1 == 0x29) {  //plwa
      pDIP( is_prefix, "lwa r%u,%llu(r%u)", rT_addr, immediate_val, rA_addr);
      DIPp( is_prefix, ",%u", R );
      assign( EA, calculate_prefix_EA( prefix, theInstr, rA_addr,
                                       ptype, DSFORM_IMMASK,
                                       &immediate_val, &R ) );

      putIReg( rT_addr,
               unop(Iop_32Sto64, load( Ity_I32, mkexpr( EA ) ) ) );
      return True;

   } else if (opc1 == 0x39) {  // pld
      pDIP( is_prefix, "ld r%u,%llu(r%u)", rT_addr, immediate_val, rA_addr);
      DIPn( is_prefix);
      assign( EA, calculate_prefix_EA( prefix, theInstr,
                                       rA_addr, ptype, DFORM_IMMASK,
                                       &immediate_val, &R ) );

      putIReg( rT_addr, load( Ity_I64, mkexpr( EA ) ) );
      return True;

   } else if (opc1 == 0x3A) {
      /* Word version DS Form - 64bit Loads.  In each case EA will have been
         formed with the lowest 2 bits masked off the immediate offset. */
      UInt uimm16 = ifieldUIMM16(theInstr);
      Int simm16 = extend_s_16to32(uimm16);

      simm16 = simm16 & DSFORM_IMMASK;
      assign( EA, ea_rAor0_simm( rA_addr, simm16  ) );

      switch ((b1<<1) | b0) {
      case 0x0: // ld (Load DWord, PPC64 p472)
         DIP("ld r%u,%llu(r%u)", rT_addr, immediate_val, rA_addr);
         putIReg( rT_addr, load( Ity_I64, mkexpr( EA ) ) );
         break;

      case 0x1: // ldu (Load DWord, Update, PPC64 p474)
         /* There is no prefixed version of this instructions.  */
         if (rA_addr == 0 || rA_addr == rT_addr) {
            vex_printf("dis_int_load_ds_form_prefix(ppc)(ldu,rA_addr|rT_addr)\n");
            return False;
         }
         DIP("ldu r%u,%llu(r%u)\n", rT_addr, immediate_val, rA_addr);

         putIReg( rT_addr, load( Ity_I64, mkexpr( EA ) ) );
         putIReg( rA_addr, mkexpr( EA ) );
         break;

      case 0x2: // lwa (Load Word Alg, PPC64 p499)
         pDIP( is_prefix, "lwa r%u,%llu(r%u)", rT_addr, immediate_val, rA_addr);
         DIPp( is_prefix, ",%u", R );

         putIReg( rT_addr,
                  unop(Iop_32Sto64, load( Ity_I32, mkexpr( EA ) ) ) );
         break;

      default:
         vex_printf("dis_int_load_ds_form_prefix(ppc)(0x3A, opc2)\n");
         return False;
      }
      return True;
   }
   return False;
}

static Bool dis_int_load_prefix ( UInt prefix, UInt theInstr )
{
   /* D-Form, X-Form, Prefixed versions */
   UChar opc1     = ifieldOPC(theInstr);
   UChar rT_addr  = ifieldRegDS(theInstr);
   UChar rA_addr  = ifieldRegA(theInstr);

   IRType  ty     = mode64 ? Ity_I64 : Ity_I32;
   IRTemp  EA     = newTemp(ty);
   UInt    ptype  = PrefixType(prefix);
   Bool    is_prefix = prefix_instruction( prefix );
   UInt    size   = 0;
   ULong   immediate_val = 0;
   UInt    R = 0;
   IRExpr* val;

   if (opc1 == 0x22) {
      // byte loads
      size = Ity_I8;
      assign( EA, calculate_prefix_EA( prefix, theInstr,
                                       rA_addr, ptype, DFORM_IMMASK,
                                       &immediate_val, &R ) );

   } else if ( opc1 == 0x28 ) {
      // half word loads lhz, plhz
      size = Ity_I16;
      assign( EA, calculate_prefix_EA( prefix, theInstr,
                                       rA_addr, ptype, DFORM_IMMASK,
                                       &immediate_val, &R ) );

   } else if ( opc1 == 0x2A ) {
      // half word loads lha, plha
      size = Ity_I16;
      assign( EA, calculate_prefix_EA( prefix, theInstr,
                                       rA_addr, ptype, DFORM_IMMASK,
                                       &immediate_val, &R ) );

   } else if (opc1 == 0x20 ) {
      // word load lwz, plwz
      size = Ity_I32;
      assign( EA, calculate_prefix_EA( prefix, theInstr,
                                       rA_addr, ptype, DFORM_IMMASK,
                                       &immediate_val, &R ) );

   } else if (opc1 == 0x38 ) {   // lq, plq
      // word load
      size = Ity_I64;

      if (!is_prefix)
         assign( EA, calculate_prefix_EA( prefix, theInstr,
                                          rA_addr, ptype, DQFORM_IMMASK,
                                          &immediate_val, &R ) );

      else
         assign( EA, calculate_prefix_EA( prefix, theInstr,
                                          rA_addr, ptype, DFORM_IMMASK,
                                          &immediate_val, &R ) );
   }

   val = load( size, mkexpr( EA ) );

   /* Store the load value in the destination and print the instruction
      details.  */
   switch (opc1) {
   case 0x20: // lwz (Load W & Zero, PPC32 p460)
      pDIP( is_prefix, "lwz r%u,%llu(r%u)", rT_addr, immediate_val, rA_addr);
      DIPp( is_prefix, ",%u", R );

      putIReg( rT_addr, mkWidenFrom32(ty, val, False) );
      break;

   case 0x22: // lbz (Load B & Zero, PPC32 p433)
      pDIP( is_prefix, "lbz r%u,%llu(r%u)", rT_addr, immediate_val, rA_addr );
      DIPp( is_prefix, ",%u", R );

      putIReg( rT_addr, mkWidenFrom8( ty, val, False ) );
      break;

   case 0x28: // lhz (Load HW & Zero, PPC32 p450)
      pDIP( is_prefix, "lhz r%u,%llu(r%u)", rT_addr, immediate_val, rA_addr );
      DIPp( is_prefix, ",%u", R );

      putIReg( rT_addr, mkWidenFrom16( ty, val, False ) );
      break;

   case 0x2A: // lha (Load HW Alg, PPC32 p445)
      pDIP( is_prefix, "lha r%u,%llu(r%u)", rT_addr, immediate_val, rA_addr);
      DIPp( is_prefix, ",%u", R );
      putIReg( rT_addr, mkWidenFrom16(ty, val, True) );
      break;

   case 0x38: { // lq, plq
      IRTemp  high = newTemp(ty);
      IRTemp  low  = newTemp(ty);
      /* DQ Form - 128bit Loads. Lowest bits [1:0] are the PT field. */
      pDIP( is_prefix, "lq r%u,%llu(r%u)", rT_addr, immediate_val, rA_addr);
      DIPp( is_prefix, ",%u", R );
      /* NOTE: there are some changes to XER[41:42] that have not been
       * implemented.
       */
      //trap if EA misaligned on 16 byte address
      if (mode64) {
         if (host_endness == VexEndnessBE) {
            assign(high, load(ty, mkexpr( EA ) ) );
            assign(low, load(ty, binop( Iop_Add64,
                                        mkexpr( EA ),
                                        mkU64( 8 ) ) ) );
         } else {
            assign(low, load(ty, mkexpr( EA ) ) );
            assign(high, load(ty, binop( Iop_Add64,
                                         mkexpr( EA ),
                                         mkU64( 8 ) ) ) );
         }
      } else {
         assign(high, load(ty, binop( Iop_Add32,
                                      mkexpr( EA ),
                                      mkU32( 4 ) ) ) );
         assign(low, load(ty, binop( Iop_Add32,
                                      mkexpr( EA ),
                                      mkU32( 12 ) ) ) );
      }

      /* Note, the load order for lq is the same for BE and LE.  However,
         plq does an endian aware load.  */
      if (is_prefix &&( host_endness == VexEndnessLE )) {
         putIReg( rT_addr,  mkexpr( low) );
         putIReg( rT_addr+1,  mkexpr( high) );
      } else {
         putIReg( rT_addr,  mkexpr( high) );
         putIReg( rT_addr+1,  mkexpr( low) );
      }
      break;
   }

   default:
      vex_printf("dis_int_load_prefix(ppc)(opc1)\n");
      return False;
   }
   return True;
}

static Bool dis_int_load ( UInt prefix, UInt theInstr )
{
   /* D-Form, X-Form, DS-Form */
   UChar opc1     = ifieldOPC(theInstr);
   UChar rD_addr  = ifieldRegDS(theInstr);
   UChar rA_addr  = ifieldRegA(theInstr);
   UInt  uimm16   = ifieldUIMM16(theInstr);
   UChar rB_addr  = ifieldRegB(theInstr);
   UInt  opc2     = ifieldOPClo10(theInstr);
   UChar b0       = ifieldBIT0(theInstr);

   Int     simm16 = extend_s_16to32(uimm16);
   IRType  ty     = mode64 ? Ity_I64 : Ity_I32;
   IRTemp  EA     = newTemp(ty);
   IRExpr* val;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   switch (opc1) {
   case 0x1F: // register offset
      assign( EA, ea_rAor0_idxd( rA_addr, rB_addr ) );
      break;
   case 0x38: // immediate offset: 64bit: lq: maskoff
              // lowest 4 bits of immediate before forming EA
      simm16 = simm16 & 0xFFFFFFF0;
      assign( EA, ea_rAor0_simm( rA_addr, simm16  ) );
      break;
   default:   // immediate offset
      assign( EA, ea_rAor0_simm( rA_addr, simm16  ) );
      break;
   }

   switch (opc1) {
   case 0x23: // lbzu (Load B & Zero, Update, PPC32 p434)
      if (rA_addr == 0 || rA_addr == rD_addr) {
         vex_printf("dis_int_load(ppc)(lbzu,rA_addr|rD_addr)\n");
         return False;
      }
      DIP("lbzu r%u,%d(r%u)\n", rD_addr, (Int)simm16, rA_addr);
      val = load(Ity_I8, mkexpr(EA));
      putIReg( rD_addr, mkWidenFrom8(ty, val, False) );
      putIReg( rA_addr, mkexpr(EA) );
      break;
      
   case 0x2B: // lhau (Load HW Alg, Update, PPC32 p446)
      if (rA_addr == 0 || rA_addr == rD_addr) {
         vex_printf("dis_int_load(ppc)(lhau,rA_addr|rD_addr)\n");
         return False;
      }
      DIP("lhau r%u,%d(r%u)\n", rD_addr, (Int)simm16, rA_addr);
      val = load(Ity_I16, mkexpr(EA));
      putIReg( rD_addr, mkWidenFrom16(ty, val, True) );
      putIReg( rA_addr, mkexpr(EA) );
      break;
      
   case 0x29: // lhzu (Load HW & and Zero, Update, PPC32 p451)
      if (rA_addr == 0 || rA_addr == rD_addr) {
         vex_printf("dis_int_load(ppc)(lhzu,rA_addr|rD_addr)\n");
         return False;
      }
      DIP("lhzu r%u,%d(r%u)\n", rD_addr, (Int)simm16, rA_addr);
      val = load(Ity_I16, mkexpr(EA));
      putIReg( rD_addr, mkWidenFrom16(ty, val, False) );
      putIReg( rA_addr, mkexpr(EA) );
      break;

   case 0x21: // lwzu (Load W & Zero, Update, PPC32 p461))
      if (rA_addr == 0 || rA_addr == rD_addr) {
         vex_printf("dis_int_load(ppc)(lwzu,rA_addr|rD_addr)\n");
         return False;
      }
      DIP("lwzu r%u,%d(r%u)\n", rD_addr, (Int)simm16, rA_addr);
      val = load(Ity_I32, mkexpr(EA));
      putIReg( rD_addr, mkWidenFrom32(ty, val, False) );
      putIReg( rA_addr, mkexpr(EA) );
      break;
      
   /* X Form */
   case 0x1F:
      if (b0 != 0) {
         vex_printf("dis_int_load(ppc)(Ox1F,b0)\n");
         return False;
      }

      switch (opc2) {
      case 0x077: // lbzux (Load B & Zero, Update Indexed, PPC32 p435)
         DIP("lbzux r%u,r%u,r%u\n", rD_addr, rA_addr, rB_addr);
         if (rA_addr == 0 || rA_addr == rD_addr) {
            vex_printf("dis_int_load(ppc)(lwzux,rA_addr|rD_addr)\n");
            return False;
         }
         val = load(Ity_I8, mkexpr(EA));
         putIReg( rD_addr, mkWidenFrom8(ty, val, False) );
         putIReg( rA_addr, mkexpr(EA) );
         break;
         
      case 0x057: // lbzx (Load B & Zero, Indexed, PPC32 p436)
         DIP("lbzx r%u,r%u,r%u\n", rD_addr, rA_addr, rB_addr);
         val = load(Ity_I8, mkexpr(EA));
         putIReg( rD_addr, mkWidenFrom8(ty, val, False) );
         break;
         
      case 0x177: // lhaux (Load HW Alg, Update Indexed, PPC32 p447)
         if (rA_addr == 0 || rA_addr == rD_addr) {
            vex_printf("dis_int_load(ppc)(lhaux,rA_addr|rD_addr)\n");
            return False;
         }
         DIP("lhaux r%u,r%u,r%u\n", rD_addr, rA_addr, rB_addr);
         val = load(Ity_I16, mkexpr(EA));
         putIReg( rD_addr, mkWidenFrom16(ty, val, True) );
         putIReg( rA_addr, mkexpr(EA) );
         break;
         
      case 0x157: // lhax (Load HW Alg, Indexed, PPC32 p448)
         DIP("lhax r%u,r%u,r%u\n", rD_addr, rA_addr, rB_addr);
         val = load(Ity_I16, mkexpr(EA));
         putIReg( rD_addr, mkWidenFrom16(ty, val, True) );
         break;
         
      case 0x137: // lhzux (Load HW & Zero, Update Indexed, PPC32 p452)
         if (rA_addr == 0 || rA_addr == rD_addr) {
            vex_printf("dis_int_load(ppc)(lhzux,rA_addr|rD_addr)\n");
            return False;
         }
         DIP("lhzux r%u,r%u,r%u\n", rD_addr, rA_addr, rB_addr);
         val = load(Ity_I16, mkexpr(EA));
         putIReg( rD_addr, mkWidenFrom16(ty, val, False) );
         putIReg( rA_addr, mkexpr(EA) );
         break;
         
      case 0x117: // lhzx (Load HW & Zero, Indexed, PPC32 p453)
         DIP("lhzx r%u,r%u,r%u\n", rD_addr, rA_addr, rB_addr);
         val = load(Ity_I16, mkexpr(EA));
         putIReg( rD_addr, mkWidenFrom16(ty, val, False) );
         break;

      case 0x037: // lwzux (Load W & Zero, Update Indexed, PPC32 p462)
         if (rA_addr == 0 || rA_addr == rD_addr) {
            vex_printf("dis_int_load(ppc)(lwzux,rA_addr|rD_addr)\n");
            return False;
         }
         DIP("lwzux r%u,r%u,r%u\n", rD_addr, rA_addr, rB_addr);
         val = load(Ity_I32, mkexpr(EA));
         putIReg( rD_addr, mkWidenFrom32(ty, val, False) );
         putIReg( rA_addr, mkexpr(EA) );
         break;
         
      case 0x017: // lwzx (Load W & Zero, Indexed, PPC32 p463)
         DIP("lwzx r%u,r%u,r%u\n", rD_addr, rA_addr, rB_addr);
         val = load(Ity_I32, mkexpr(EA));
         putIReg( rD_addr, mkWidenFrom32(ty, val, False) );
         break;


      /* 64bit Loads */
      case 0x035: // ldux (Load DWord, Update Indexed, PPC64 p475)
         if (rA_addr == 0 || rA_addr == rD_addr) {
            vex_printf("dis_int_load(ppc)(ldux,rA_addr|rD_addr)\n");
            return False;
         }
         DIP("ldux r%u,r%u,r%u\n", rD_addr, rA_addr, rB_addr);
         putIReg( rD_addr, load(Ity_I64, mkexpr(EA)) );
         putIReg( rA_addr, mkexpr(EA) );
         break;

      case 0x015: // ldx (Load DWord, Indexed, PPC64 p476)
         DIP("ldx r%u,r%u,r%u\n", rD_addr, rA_addr, rB_addr);
         putIReg( rD_addr, load(Ity_I64, mkexpr(EA)) );
         break;

      case 0x175: // lwaux (Load W Alg, Update Indexed, PPC64 p501)
         if (rA_addr == 0 || rA_addr == rD_addr) {
            vex_printf("dis_int_load(ppc)(lwaux,rA_addr|rD_addr)\n");
            return False;
         }
         DIP("lwaux r%u,r%u,r%u\n", rD_addr, rA_addr, rB_addr);
         putIReg( rD_addr,
                  unop(Iop_32Sto64, load(Ity_I32, mkexpr(EA))) );
         putIReg( rA_addr, mkexpr(EA) );
         break;

      case 0x155: // lwax (Load W Alg, Indexed, PPC64 p502)
         DIP("lwax r%u,r%u,r%u\n", rD_addr, rA_addr, rB_addr);
         putIReg( rD_addr,
                  unop(Iop_32Sto64, load(Ity_I32, mkexpr(EA))) );
         break;

      default:
         vex_printf("dis_int_load(ppc)(opc2)\n");
         return False;
      }
      break;

   default:
      vex_printf("dis_int_load(ppc)(opc1)\n");
      return False;
   }
   return True;
}

/*
  VSX Vector Splat Immediate Word 8RR:D-form
*/
static Bool dis_vector_splat_imm_prefix ( UInt prefix, UInt theInstr )
{
   UChar opc1 = ifieldOPC(theInstr);
   UChar opc2 = IFIELD(theInstr, (31-(46-32)), 4);  // bits[43:46]

   UInt imm0  = ifield_imm_8RR_D(prefix);
   UInt imm1  = ifield_imm_8RR_D(theInstr);
   UInt IMM32 = (imm0 << 16) | imm1;

   UInt XT_addr = ifieldRegXT_8RR_D(theInstr);

   if (opc1 != 0x20) return False;

   /* These are prefix instructions, no equivalent word instruction.   */
   switch(opc2) {
   case 0x0:
   case 0x1:
   {
      /* VSX Vector Splat Immediate32 Doubleword Indexed 8RR:D-form  */
      UInt IX = IFIELD(theInstr, (31-(46-32)), 1);  // bit[46]
      IRTemp tmp  = newTemp(Ity_V128);
      IRTemp mask = newTemp(Ity_V128);
      IRTemp new_elements = newTemp(Ity_V128);

      DIP("xxsplti32dx %u,%u,%u\n", XT_addr, IX, IMM32);

      assign( tmp, getVSReg( XT_addr ) );

      if (IX == 0) {
         assign( mask, binop( Iop_64HLtoV128,
                              binop( Iop_32HLto64,
                                     mkU32( 0 ), mkU32( 0xFFFFFFFF ) ),
                              binop( Iop_32HLto64,
                                     mkU32( 0 ), mkU32( 0xFFFFFFFF ) ) ) );
         assign( new_elements, binop( Iop_64HLtoV128,
                                      binop( Iop_32HLto64,
                                             mkU32( IMM32 ), mkU32( 0 ) ),
                                      binop( Iop_32HLto64,
                                             mkU32( IMM32 ), mkU32( 0 ) ) ) );
      } else {
         assign( mask, binop( Iop_64HLtoV128,
                              binop( Iop_32HLto64,
                                     mkU32( 0xFFFFFFFF ), mkU32( 0 ) ),
                              binop( Iop_32HLto64,
                                     mkU32( 0xFFFFFFFF ), mkU32( 0 ) ) ) );
         assign( new_elements, binop( Iop_64HLtoV128,
                                      binop( Iop_32HLto64,
                                             mkU32( 0 ), mkU32( IMM32 ) ),
                                      binop( Iop_32HLto64,
                                             mkU32( 0 ), mkU32( IMM32 ) ) ) );
      }

      putVSReg( XT_addr,
                binop( Iop_OrV128,
                       binop( Iop_AndV128, mkexpr( tmp ), mkexpr( mask) ),
                       mkexpr( new_elements ) ) );
      break;
   }
   case 0x2:
      {
         IRTemp  result = newTemp(Ity_I64);

         /* VSX Vector Splat Immediate Double-precision 8RR:D-form  */
         DIP("xxspltidp %u,%u\n", XT_addr, IMM32);

         assign( result,
                 unop( Iop_ReinterpF64asI64,
                       unop( Iop_F32toF64,
                             unop( Iop_ReinterpI32asF32,
                                   mkU32( IMM32 ) ) ) ) );
         putVSReg( XT_addr, binop( Iop_64HLtoV128,
                                   mkexpr( result ), mkexpr( result ) ) );
      }
      break;

   case 0x3:
      /* VSX Vector Splat Immediate Word 8RR:D-form  */
      DIP("xxspltiw %u,%u\n", XT_addr, IMM32);

      putVSReg( XT_addr,
                binop( Iop_64HLtoV128,
                       binop( Iop_32HLto64,
                              mkU32( IMM32 ), mkU32( IMM32 ) ),
                       binop( Iop_32HLto64,
                              mkU32( IMM32 ), mkU32( IMM32 ) ) ) );
      break;
   default:
      vex_printf("dis_vector_splat_imm_prefix (opc2)\n");
      return False;
   }

   return True;
}

 /*
  VSX Vector Permute Extended 8RR:D-form
 */
static Bool dis_vector_permute_prefix ( UInt prefix, UInt theInstr,
                                        const VexAbiInfo* vbi )
{
   #define MAX_ELE 16
   UChar opc1 = ifieldOPC(theInstr);
   UChar opc2 = IFIELD(theInstr, (63-59), 2);  // bits[58:59]
   UChar rXT_addr = ifieldRegXT_8RR_XX4( theInstr );
   UChar rXA_addr = ifieldRegXA_8RR_XX4( theInstr );
   UChar rXB_addr = ifieldRegXB_8RR_XX4( theInstr );
   UChar rXC_addr = ifieldRegXC_8RR_XX4( theInstr );

   Int i;
   IRTemp rXA  = newTemp(Ity_V128);
   IRTemp rXB  = newTemp(Ity_V128);
   IRTemp rXC  = newTemp(Ity_V128);
   IRTemp cmp_mask = newTemp(Ity_I64);
   IRTemp eidx_mask = newTemp(Ity_I64);
   IRTemp result[MAX_ELE+1];
   IRTemp result_mask[MAX_ELE];
   IRTemp byte[MAX_ELE];
   IRTemp eidx[MAX_ELE];

  /* These are prefix instructions, no equivalent word instruction.   */
   if ((opc1 != 0x22) && (opc2 != 0)) return False;

   assign( rXA, getVSReg( rXA_addr ) );
   assign( rXB, getVSReg( rXB_addr ) );
   assign( rXC, getVSReg( rXC_addr ) );

   switch(opc2) {
   case 0:
   {
      UInt  UIM = IFIELD(prefix, 0, 3);   // bit [29:31] of the prefix

      DIP("xxpermx v%u,v%u,v%u,v%u,%u\n",
          rXT_addr, rXA_addr, rXB_addr, rXC_addr, UIM);

      result[MAX_ELE] = newTemp(Ity_V128);
      assign( eidx_mask, mkU64( 0x1F ) );
      assign( cmp_mask, mkU64( 0x7 ) );
      assign( result[MAX_ELE], binop( Iop_64HLtoV128, mkU64( 0 ),
                                      mkU64( 0 ) ) );

      for (i = MAX_ELE-1; i >= 0; i--) {
         eidx[i] = newTemp( Ity_I64 );
         byte[i] = newTemp( Ity_I64 );
         result[i] = newTemp( Ity_V128 );
         result_mask[i] = newTemp( Ity_I64 );

         /* The eidx is left based, make index right based for
            extractBytefromV256().  */
         if ( i >= 8) {
            assign( eidx[i],
                    binop( Iop_Sub64,
                           mkU64( 31 ),
                           binop( Iop_And64,
                                  mkexpr( eidx_mask ),
                                  binop( Iop_Shr64,
                                         unop( Iop_V128HIto64, mkexpr( rXC ) ),
                                         mkU8( (i - 8)*8 ) ) ) ) );
            assign( result_mask[i],
                    unop( Iop_1Sto64,
                          binop( Iop_CmpEQ64,
                                 mkU64( UIM ),
                                 binop( Iop_And64,
                                        mkexpr ( cmp_mask ),
                                        // bits 0:2 of ith byte
                                        binop( Iop_Shr64,
                                               unop( Iop_V128HIto64,
                                                     mkexpr( rXC ) ),
                                               mkU8( (i - 8)*8 + 5 ) ) )
                             ) ) );
         } else {
            assign( eidx[i],
                    binop( Iop_Sub64,
                           mkU64( 31 ),
                           binop( Iop_And64,
                                  mkexpr( eidx_mask ),
                                  binop( Iop_Shr64,
                                         unop( Iop_V128to64, mkexpr( rXC ) ),
                                         mkU8( i*8 ) ) ) ) );
            assign( result_mask[i],
                    unop( Iop_1Sto64,
                          binop( Iop_CmpEQ64,
                                 mkU64( UIM ),
                                 binop( Iop_And64,
                                        mkexpr ( cmp_mask ),
                                        // bits 0:2 of ith byte
                                        binop( Iop_Shr64,
                                               unop( Iop_V128to64,
                                                     mkexpr( rXC ) ),
                                               mkU8( i*8 + 5 ) ) ) ) ) );
         }

         assign( byte[i],
                 binop( Iop_And64,
                        mkexpr( result_mask[i] ),
                        extractBytefromV256( rXA, rXB, eidx[i] ) ) );

         assign( result[i], insert_field_into_vector( result[i+1],
                                                      mkU64( i ),
                                                      mkexpr( byte[i] ),
                                                      mkU64( 0xFF ) ) );
      }
      putVSReg( rXT_addr, mkexpr( result[0] ) );
   }
   break;

   case 1:
   {
      UInt  IMM = IFIELD(prefix, 0, 8);  // bit [24:31] of the prefix
      DIP("xxeval v%u,v%u,v%u,v%u,%u\n",
          rXT_addr, rXA_addr, rXB_addr, rXC_addr, IMM);
      putVSReg( rXT_addr,
                vector_evaluate_inst ( vbi, mkexpr( rXA ), mkexpr( rXB ),
                                       mkexpr( rXC ), mkU64( IMM ) ) );
   }
   break;

   default:
      vex_printf("dis_vector_permute_prefix(ppc)(opc2)\n");
      return False;
   }

   return True;
#undef MAX_ELE
}

/*
  VSX Vector Splat Immediate Word 8RR:D-form
*/
static Bool dis_vector_blend_prefix ( UInt prefix, UInt theInstr )
{
   UChar opc1 = ifieldOPC(theInstr);
   UChar opc2 = IFIELD(theInstr, (63-59), 2);  // bits[58:59]
   UChar rXT_addr = ifieldRegXT_8RR_XX4( theInstr );
   UChar rXA_addr = ifieldRegXA_8RR_XX4( theInstr );
   UChar rXB_addr = ifieldRegXB_8RR_XX4( theInstr );
   UChar rXC_addr = ifieldRegXC_8RR_XX4( theInstr );

   IRTemp rXA  = newTemp(Ity_V128);
   IRTemp rXB  = newTemp(Ity_V128);
   IRTemp rXC  = newTemp(Ity_V128);
   IRTemp bit_mask  = newTemp(Ity_V128);
   IRTemp mask_gen  = newTemp(Ity_V128);
   IRTemp mask      = newTemp(Ity_V128);

   /* These are prefix instructions, no equivalent word instruction.   */
   if (opc1 != 0x21) return False;

   /* Generate the mask to select the elements from rXA or rXB.  Use a vector
      multiply to generate the mask to select the elments.  Take the selctor
      bit for the element (rXC & bit_mask) and multiply it by all 1's
      (mask_gen).  If the selector bit was 0, then we get zero bits for that
      element entry, otherwise we get 1's.

      Unfortunately, we don't have an integer vector multipy have to do it as
      an even and odd multiply for byt, halfword and word elements.  Note, the
      MK_Iop_MullOddXUxY shifts the operands right and uses the MullEven
      operator, so we have to move the result back to its correct lane
      position.  */

   assign( rXA, getVSReg( rXA_addr ) );
   assign( rXB, getVSReg( rXB_addr ) );
   assign( rXC, getVSReg( rXC_addr ) );

   assign( mask_gen,
           binop( Iop_64HLtoV128,
                  mkU64( 0xFFFFFFFFFFFFFFFFULL),
                  mkU64( 0xFFFFFFFFFFFFFFFFULL) ) );

   switch(opc2) {
   case 0:
      /* VSX Vector Blend Variable Byte 8RR:XX4-Form */
      DIP("xxblendvb v%u,v%u,v%u,v%u\n",
          rXT_addr, rXA_addr, rXB_addr, rXC_addr);

      assign( bit_mask,
              binop( Iop_ShrV128,
                     binop( Iop_AndV128,
                            mkexpr( rXC ),
                            binop( Iop_64HLtoV128,
                                   mkU64( 0x8080808080808080ULL ),
                                   mkU64( 0x8080808080808080ULL ) ) ),
                     mkU8 ( 7 ) ) );
      assign( mask,
              binop( Iop_OrV128,
                     binop( Iop_MullEven8Ux16,
                            mkexpr( mask_gen ),
                            mkexpr( bit_mask ) ),
                     binop( Iop_ShlV128,
                            MK_Iop_MullOdd8Ux16(
                                                mkexpr( mask_gen ),
                                                mkexpr( bit_mask ) ),
                            mkU8( 8 ) ) ) );
      break;

   case 1:
      /* VSX Vector Blend Variable Halfword 8RR:XX4-Form */
      DIP("xxblendvh v%u,v%u,v%u,v%u\n",
          rXT_addr, rXA_addr, rXB_addr, rXC_addr);

      assign( bit_mask,
              binop( Iop_ShrV128,
                     binop( Iop_AndV128,
                            mkexpr( rXC ),
                            binop( Iop_64HLtoV128,
                                   mkU64( 0x8000800080008000ULL ),
                                   mkU64( 0x8000800080008000ULL ) ) ),
                     mkU8 ( 15 ) ) );
      assign( mask,
              binop( Iop_OrV128,
                     binop( Iop_MullEven16Ux8,
                            mkexpr( mask_gen ),
                            mkexpr( bit_mask ) ),
                     binop( Iop_ShlV128,
                            MK_Iop_MullOdd16Ux8(
                                                mkexpr( mask_gen ),
                                                mkexpr( bit_mask ) ),
                            mkU8( 16 ) ) ) );
      break;

   case 2:
      /* VSX Vector Blend Variable Word 8RR:XX4-Form */
      DIP("xxblendvw v%u,v%u,v%u,v%u\n",
          rXT_addr, rXA_addr, rXB_addr, rXC_addr);

      assign( bit_mask,
              binop( Iop_ShrV128,
                     binop( Iop_AndV128,
                            mkexpr( rXC ),
                            binop( Iop_64HLtoV128,
                                   mkU64( 0x8000000080000000ULL ),
                                   mkU64( 0x8000000080000000ULL ) ) ),
                     mkU8 ( 31 ) ) );
      assign( mask,
              binop( Iop_OrV128,
                     binop( Iop_MullEven32Ux4,
                            mkexpr( mask_gen ),
                            mkexpr( bit_mask ) ),
                     binop( Iop_ShlV128,
                            MK_Iop_MullOdd32Ux4(
                                                mkexpr( mask_gen ),
                                                mkexpr( bit_mask ) ),
                            mkU8( 32 ) ) ) );
      break;

   case 3:
      /* VSX Vector Blend Variable Double 8RR:XX4-Form */
      DIP("xxblendvd v%u,v%u,v%u,v%u\n",
          rXT_addr, rXA_addr, rXB_addr, rXC_addr);

      /* Have to use a different trick here */
      assign( mask,
              binop( Iop_64HLtoV128,
                     unop( Iop_1Sto64,
                           unop( Iop_64to1,
                                 binop( Iop_Shr64,
                                        unop( Iop_V128HIto64,
                                              mkexpr( rXC ) ),
                                        mkU8( 63) ) ) ),
                     unop( Iop_1Sto64,
                           unop( Iop_64to1,
                                 binop( Iop_Shr64,
                                        unop( Iop_V128to64,
                                              mkexpr( rXC ) ),
                                        mkU8( 63) ) ) ) ) );
      break;

   default:
      vex_printf("dis_vector_blend_prefix (opc2)\n");
      return False;
   }
   putVSReg( rXT_addr, binop( Iop_OrV128,
                              binop( Iop_AndV128,
                                     unop( Iop_NotV128, mkexpr( mask ) ),
                                    mkexpr( rXA ) ),
                              binop( Iop_AndV128,
                                     mkexpr( mask ),
                                     mkexpr( rXB ) ) ) );
   return True;
}


/*
  Integer Store Instructions
*/
static Bool dis_int_store_ds_prefix ( UInt prefix,
                                      UInt theInstr, const VexAbiInfo* vbi)
{
   UChar opc1    = ifieldOPC(theInstr);
   UInt  rS_addr = ifieldRegDS(theInstr);
   UInt  rA_addr = ifieldRegA(theInstr);
   UChar b0      = ifieldBIT0(theInstr);
   UChar b1      = ifieldBIT1(theInstr);
   IRType ty     = mode64 ? Ity_I64 : Ity_I32;
   IRTemp rS     = newTemp(ty);
   IRTemp EA     = newTemp(ty);
   UInt   ptype  = PrefixType(prefix);
   Bool   is_prefix = prefix_instruction( prefix );
   UInt   R      = 0;                    // must be zero for word instruction
   ULong  immediate_val = 0;
   Int    simm16 = extend_s_16to32(ifieldUIMM16(theInstr));

   if (opc1 == 0x3C) {
      // force opc2 to 2 to map pstq to stq inst
      b0 = 0;
      b1 = 1;
      assign( EA, calculate_prefix_EA( prefix, theInstr, rA_addr,
                                       ptype, DSFORM_IMMASK, &immediate_val,
                                       &R ) );
   } else if (opc1 == 0x3D) {
      // force opc2 to 0 to map pstd to std inst
      b0 = 0;
      b1 = 0;
      assign( EA, calculate_prefix_EA( prefix, theInstr, rA_addr,
                                       ptype, DFORM_IMMASK, &immediate_val,
                                       &R ) );

   } else if ( opc1 == 0x3 ) {
      assign( EA, ea_rAor0_simm( rA_addr, simm16  ) );

   } else if ( opc1 == 0x3E ) {  // std, stdu, stq
      // lowest 2 bits of immediate before forming EA
      immediate_val = simm16 & 0xFFFFFFFC;
      assign( EA, ea_rAor0_simm( rA_addr, immediate_val ) );

   } else {
      return False;
   }

   assign( rS, getIReg(rS_addr) );

   /* DS Form - 64bit Stores.  In each case EA will have been formed
      with the lowest 2 bits masked off the immediate offset. */
  switch ((b1<<1) | b0) {
   case 0x0: // std (Store DWord, PPC64 p580)
      if (!mode64)
         return False;

      pDIP( is_prefix,"std r%u,%llu(r%u)", rS_addr, immediate_val, rA_addr );
      DIPp( is_prefix, ",%u", R );
      store( mkexpr(EA), mkexpr(rS) );
      break;

   case 0x1: // stdu (Store DWord, Update, PPC64 p583)
      /* Note this instruction is handled here but it isn't actually a
         prefix instruction.  Just makes the parsing easier to handle it
         here.  */
      if (!mode64)
         return False;

      DIP("stdu r%u,%llu(r%u)\n", rS_addr, immediate_val, rA_addr);
      putIReg( rA_addr, mkexpr(EA) );
      store( mkexpr(EA), mkexpr(rS) );
      break;

   case 0x2:  // stq, pstq (Store QuadWord, Update, PPC64 p583)
      {
         IRTemp EA_hi = newTemp(ty);
         IRTemp EA_lo = newTemp(ty);

         pDIP( is_prefix, "stq r%u,%llu(r%u)", rS_addr, immediate_val, rA_addr);
         DIPp( is_prefix, ",%u", R );

         if (mode64) {
            if (host_endness == VexEndnessBE) {

               /* upper 64-bits */
               assign( EA_hi, mkexpr(EA));

               /* lower 64-bits */
               assign( EA_lo, binop(Iop_Add64, mkexpr(EA), mkU64(8)));

            } else {
               /* upper 64-bits */
               assign( EA_hi, binop(Iop_Add64, mkexpr(EA), mkU64(8)));

               /* lower 64-bits */
               assign( EA_lo, mkexpr(EA));
            }
         } else {
            /* upper half of upper 64-bits */
            assign( EA_hi, binop(Iop_Add32, mkexpr(EA), mkU32(4)));

            /* lower half of upper 64-bits */
            assign( EA_lo, binop(Iop_Add32, mkexpr(EA), mkU32(12)));
         }

         /* Note, the store order for stq instruction is the same for BE
            and LE.  The store order for the pstq instruction is endian aware
            store.  */
         if (is_prefix &&( host_endness == VexEndnessLE )) {
            //  LE and pstq
            store( mkexpr(EA_hi), getIReg( rS_addr+1 ) );
            store( mkexpr(EA_lo), mkexpr(rS) );
         } else {
            store( mkexpr(EA_hi), mkexpr(rS) );
            store( mkexpr(EA_lo), getIReg( rS_addr+1 ) );
         }
         break;
      }
   default:
      vex_printf("dis_int_store_ds_prefix(ppc)(opc1)\n");
      return False;
   }
   return True;
}

static Bool dis_int_store_prefix ( UInt prefix,
                                   UInt theInstr, const VexAbiInfo* vbi)
{
   UChar opc1    = ifieldOPC(theInstr);
   UInt  rS_addr = ifieldRegDS(theInstr);
   UInt  rA_addr = ifieldRegA(theInstr);
   IRType ty     = mode64 ? Ity_I64 : Ity_I32;
   IRTemp rS     = newTemp(ty);
   IRTemp EA     = newTemp(ty);
   UInt   ptype  = PrefixType(prefix);
   Bool   is_prefix = prefix_instruction( prefix );
   ULong  immediate_val = 0;
   UInt   R      = 0;                    // must be zero for word instruction

   assign( rS, getIReg(rS_addr) );
   assign( EA, calculate_prefix_EA( prefix, theInstr, rA_addr,
                                    ptype, DFORM_IMMASK,
                                    &immediate_val, &R ) );

   switch (opc1) {
   case 0x24: // stw (Store W, PPC32 p530)
      pDIP( is_prefix, "stw r%u,%llu(r%u)\n", rS_addr, immediate_val, rA_addr );
      DIPp( is_prefix, ",%u", R );
      store( mkexpr(EA), mkNarrowTo32(ty, mkexpr(rS)) );
      break;

   case 0x26: // stb (Store B, PPC32 p509)
      pDIP( is_prefix, "stb r%u,%llu(r%u)", rS_addr, immediate_val, rA_addr );
      DIPp( is_prefix, ",%u", R );
      store( mkexpr(EA), mkNarrowTo8(ty, mkexpr(rS)) );
      break;

   case 0x2C: // sth (Store HW, PPC32 p522)
      pDIP( is_prefix, "sth r%u,%llu(r%u)", rS_addr, immediate_val, rA_addr );
      DIPp( is_prefix, ",%u", R );
      store( mkexpr(EA), mkNarrowTo16(ty, mkexpr(rS)) );
      break;

   default:
      vex_printf("dis_int_store_prefix(ppc)(opc1)\n");
      return False;
   }
   return True;
}

static Bool dis_int_store ( UInt prefix, UInt theInstr, const VexAbiInfo* vbi )
{
   /* D-Form, X-Form, DS-Form */
   UChar opc1    = ifieldOPC(theInstr);
   UInt  rS_addr = ifieldRegDS(theInstr);
   UInt  rA_addr = ifieldRegA(theInstr);
   UInt  uimm16  = ifieldUIMM16(theInstr);
   UInt  rB_addr = ifieldRegB(theInstr);
   UInt  opc2    = ifieldOPClo10(theInstr);
   UChar b0      = ifieldBIT0(theInstr);

   Int    simm16 = extend_s_16to32(uimm16);
   IRType ty     = mode64 ? Ity_I64 : Ity_I32;
   IRTemp rS     = newTemp(ty);
   IRTemp rB     = newTemp(ty);
   IRTemp EA     = newTemp(ty);

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   assign( rB, getIReg(rB_addr) );
   assign( rS, getIReg(rS_addr) );
   
   switch (opc1) {
   case 0x1F: // register offset
      assign( EA, ea_rAor0_idxd( rA_addr, rB_addr ) );
      break;

      /* fallthrough */
   default:   // immediate offset
      assign( EA, ea_rAor0_simm( rA_addr, simm16  ) );
      break;
   }

   switch (opc1) {
   case 0x27: // stbu (Store B, Update, PPC32 p510)
      if (rA_addr == 0 ) {
         vex_printf("dis_int_store(ppc)(stbu,rA_addr)\n");
         return False;
      }
      DIP("stbu r%u,%d(r%u)\n", rS_addr, simm16, rA_addr);
      putIReg( rA_addr, mkexpr(EA) );
      store( mkexpr(EA), mkNarrowTo8(ty, mkexpr(rS)) );
      break;

   case 0x2D: // sthu (Store HW, Update, PPC32 p524)
      if (rA_addr == 0) {
         vex_printf("dis_int_store(ppc)(sthu,rA_addr)\n");
         return False;
      }
      DIP("sthu r%u,%d(r%u)\n", rS_addr, simm16, rA_addr);
      putIReg( rA_addr, mkexpr(EA) );
      store( mkexpr(EA), mkNarrowTo16(ty, mkexpr(rS)) );
      break;

   case 0x24: // stw (Store W, PPC32 p530)
      {
      DIP("stw r%u,%d(r%u)\n", rS_addr, simm16, rA_addr);
      store( mkexpr(EA), mkNarrowTo32(ty, mkexpr(rS)) );
      }
      break;

   case 0x25: // stwu (Store W, Update, PPC32 p534)
      if (rA_addr == 0) {
         vex_printf("dis_int_store(ppc)(stwu,rA_addr)\n");
         return False;
      }
      DIP("stwu r%u,%d(r%u)\n", rS_addr, simm16, rA_addr);
      putIReg( rA_addr, mkexpr(EA) );
      store( mkexpr(EA), mkNarrowTo32(ty, mkexpr(rS)) );
      break;
      
   /* X Form : all these use EA_indexed */
   case 0x1F:
      if (b0 != 0) {
         vex_printf("dis_int_store(ppc)(0x1F,b0)\n");
         return False;
      }

      switch (opc2) {
      case 0x0F7: // stbux (Store B, Update Indexed, PPC32 p511)
         if (rA_addr == 0) {
            vex_printf("dis_int_store(ppc)(stbux,rA_addr)\n");
            return False;
         }
         DIP("stbux r%u,r%u,r%u\n", rS_addr, rA_addr, rB_addr);
         putIReg( rA_addr, mkexpr(EA) );
         store( mkexpr(EA), mkNarrowTo8(ty, mkexpr(rS)) );
         break;
         
      case 0x0D7: // stbx (Store B Indexed, PPC32 p512)
         DIP("stbx r%u,r%u,r%u\n", rS_addr, rA_addr, rB_addr);
         store( mkexpr(EA), mkNarrowTo8(ty, mkexpr(rS)) );
         break;
         
      case 0x1B7: // sthux (Store HW, Update Indexed, PPC32 p525)
         if (rA_addr == 0) {
            vex_printf("dis_int_store(ppc)(sthux,rA_addr)\n");
            return False;
         }
         DIP("sthux r%u,r%u,r%u\n", rS_addr, rA_addr, rB_addr);
         putIReg( rA_addr, mkexpr(EA) );
         store( mkexpr(EA), mkNarrowTo16(ty, mkexpr(rS)) );
         break;
         
      case 0x197: // sthx (Store HW Indexed, PPC32 p526)
         DIP("sthx r%u,r%u,r%u\n", rS_addr, rA_addr, rB_addr);
         store( mkexpr(EA), mkNarrowTo16(ty, mkexpr(rS)) );
         break;
         
      case 0x0B7: // stwux (Store W, Update Indexed, PPC32 p535)
         if (rA_addr == 0) {
            vex_printf("dis_int_store(ppc)(stwux,rA_addr)\n");
            return False;
         }
         DIP("stwux r%u,r%u,r%u\n", rS_addr, rA_addr, rB_addr);
         putIReg( rA_addr, mkexpr(EA) );
         store( mkexpr(EA), mkNarrowTo32(ty, mkexpr(rS)) );
         break;

      case 0x097: // stwx (Store W Indexed, PPC32 p536)
         DIP("stwx r%u,r%u,r%u\n", rS_addr, rA_addr, rB_addr);
         store( mkexpr(EA), mkNarrowTo32(ty, mkexpr(rS)) );
         break;
         

      /* 64bit Stores */
      case 0x0B5: // stdux (Store DWord, Update Indexed, PPC64 p584)
         if (rA_addr == 0) {
            vex_printf("dis_int_store(ppc)(stdux,rA_addr)\n");
            return False;
         }
         DIP("stdux r%u,r%u,r%u\n", rS_addr, rA_addr, rB_addr);
         putIReg( rA_addr, mkexpr(EA) );
         store( mkexpr(EA), mkexpr(rS) );
         break;

      case 0x095: // stdx (Store DWord Indexed, PPC64 p585)
         DIP("stdx r%u,r%u,r%u\n", rS_addr, rA_addr, rB_addr);
         store( mkexpr(EA), mkexpr(rS) );
         break;

      default:
         vex_printf("dis_int_store(ppc)(opc2)\n");
         return False;
      }
      break;

   default:
      vex_printf("dis_int_store(ppc)(opc1)\n");
      return False;
   }
   return True;
}



/*
  Integer Load/Store Multiple Instructions
*/
static Bool dis_int_ldst_mult ( UInt prefix, UInt theInstr )
{
   /* D-Form */
   UChar opc1     = ifieldOPC(theInstr);
   UChar rD_addr  = ifieldRegDS(theInstr);
   UChar rS_addr  = rD_addr;
   UChar rA_addr  = ifieldRegA(theInstr);
   UInt  uimm16   = ifieldUIMM16(theInstr);

   Int     simm16 = extend_s_16to32(uimm16);
   IRType  ty     = mode64 ? Ity_I64 : Ity_I32;
   IROp    mkAdd  = mode64 ? Iop_Add64 : Iop_Add32;
   IRTemp  EA     = newTemp(ty);
   UInt    r      = 0;
   UInt    ea_off = 0;
   IRExpr* irx_addr;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   assign( EA, ea_rAor0_simm( rA_addr, simm16 ) );

   switch (opc1) {
   case 0x2E: // lmw (Load Multiple Word, PPC32 p454)
      if (rA_addr >= rD_addr) {
         vex_printf("dis_int_ldst_mult(ppc)(lmw,rA_addr)\n");
         return False;
      }
      DIP("lmw r%u,%d(r%u)\n", rD_addr, simm16, rA_addr);
      for (r = rD_addr; r <= 31; r++) {
         irx_addr = binop(mkAdd, mkexpr(EA), mode64 ? mkU64(ea_off) : mkU32(ea_off));
         putIReg( r, mkWidenFrom32(ty, load(Ity_I32, irx_addr ),
                                       False) );
         ea_off += 4;
      }
      break;
      
   case 0x2F: // stmw (Store Multiple Word, PPC32 p527)
      DIP("stmw r%u,%d(r%u)\n", rS_addr, simm16, rA_addr);
      for (r = rS_addr; r <= 31; r++) {
         irx_addr = binop(mkAdd, mkexpr(EA), mode64 ? mkU64(ea_off) : mkU32(ea_off));
         store( irx_addr, mkNarrowTo32(ty, getIReg(r)) );
         ea_off += 4;
      }
      break;
      
   default:
      vex_printf("dis_int_ldst_mult(ppc)(opc1)\n");
      return False;
   }
   return True;
}



/*
  Integer Load/Store String Instructions
*/
static 
void generate_lsw_sequence ( IRTemp tNBytes,   // # bytes, :: Ity_I32
                             IRTemp EA,        // EA
                             Int    rD,        // first dst register
                             Int    maxBytes ) // 32 or 128
{
   Int     i, shift = 24;
   IRExpr* e_nbytes = mkexpr(tNBytes);
   IRExpr* e_EA     = mkexpr(EA);
   IRType  ty       = mode64 ? Ity_I64 : Ity_I32;

   vassert(rD >= 0 && rD < 32);
   rD--; if (rD < 0) rD = 31;

   for (i = 0; i < maxBytes; i++) {
      /* if (nBytes < (i+1)) goto NIA; */
      stmt( IRStmt_Exit( binop(Iop_CmpLT32U, e_nbytes, mkU32(i+1)),
                         Ijk_Boring, 
                         mkSzConst( ty, nextInsnAddr()), OFFB_CIA ));
      /* when crossing into a new dest register, set it to zero. */
      if ((i % 4) == 0) {
         rD++; if (rD == 32) rD = 0;
         putIReg(rD, mkSzImm(ty, 0));
         shift = 24;
      }
      /* rD |=  (8Uto32(*(EA+i))) << shift */
      vassert(shift == 0 || shift == 8 || shift == 16 || shift == 24);
      putIReg( 
         rD, 
         mkWidenFrom32(
            ty, 
            binop(
               Iop_Or32, 
               mkNarrowTo32(ty, getIReg(rD)),
               binop(
                  Iop_Shl32, 
                  unop(
                     Iop_8Uto32, 
                     load( Ity_I8,
                           binop( mkSzOp(ty,Iop_Add8),
                                  e_EA, mkSzImm(ty,i)))
                  ), 
                  mkU8(toUChar(shift))
               )
            ),
            /*Signed*/False
	 ) 
      ); 
      shift -= 8;
   }
}

static 
void generate_stsw_sequence ( IRTemp tNBytes,   // # bytes, :: Ity_I32
                              IRTemp EA,        // EA
                              Int    rS,        // first src register
                              Int    maxBytes ) // 32 or 128
{
   Int     i, shift = 24;
   IRExpr* e_nbytes = mkexpr(tNBytes);
   IRExpr* e_EA     = mkexpr(EA);
   IRType  ty       = mode64 ? Ity_I64 : Ity_I32;

   vassert(rS >= 0 && rS < 32);
   rS--; if (rS < 0) rS = 31;

   for (i = 0; i < maxBytes; i++) {
      /* if (nBytes < (i+1)) goto NIA; */
      stmt( IRStmt_Exit( binop(Iop_CmpLT32U, e_nbytes, mkU32(i+1)),
                         Ijk_Boring, 
                         mkSzConst( ty, nextInsnAddr() ), OFFB_CIA ));
      /* check for crossing into a new src register. */
      if ((i % 4) == 0) {
         rS++; if (rS == 32) rS = 0;
         shift = 24;
      }
      /* *(EA+i) = 32to8(rS >> shift) */
      vassert(shift == 0 || shift == 8 || shift == 16 || shift == 24);
      store(
            binop( mkSzOp(ty,Iop_Add8), e_EA, mkSzImm(ty,i)),
            unop( Iop_32to8,
                  binop( Iop_Shr32,
                         mkNarrowTo32( ty, getIReg(rS) ),
                         mkU8( toUChar(shift) )))
      );
      shift -= 8;
   }
}

static Bool dis_int_ldst_str ( UInt prefix, UInt theInstr, /*OUT*/Bool* stopHere )
{
   /* X-Form */
   UChar opc1     = ifieldOPC(theInstr);
   UChar rD_addr  = ifieldRegDS(theInstr);
   UChar rS_addr  = rD_addr;
   UChar rA_addr  = ifieldRegA(theInstr);
   UChar rB_addr  = ifieldRegB(theInstr);
   UChar NumBytes = rB_addr;
   UInt  opc2     = ifieldOPClo10(theInstr);
   UChar b0       = ifieldBIT0(theInstr);

   IRType ty      = mode64 ? Ity_I64 : Ity_I32;
   IRTemp t_EA    = newTemp(ty);
   IRTemp t_nbytes = IRTemp_INVALID;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   *stopHere = False;

   if (opc1 != 0x1F || b0 != 0) {
      vex_printf("dis_int_ldst_str(ppc)(opc1)\n");
      return False;
   }

   switch (opc2) {
   case 0x255: // lswi (Load String Word Immediate, PPC32 p455)
      /* NB: does not reject the case where RA is in the range of
         registers to be loaded.  It should. */
      DIP("lswi r%u,r%u,%d\n", rD_addr, rA_addr, NumBytes);
      assign( t_EA, ea_rAor0(rA_addr) );
      if (NumBytes == 8 && !mode64) {
         /* Special case hack */
         /* rD = Mem[EA]; (rD+1)%32 = Mem[EA+4] */
         putIReg( rD_addr,          
                  load(Ity_I32, mkexpr(t_EA)) );
         putIReg( (rD_addr+1) % 32, 
                  load(Ity_I32,
                       binop(Iop_Add32, mkexpr(t_EA), mkU32(4))) );
      } else {
         t_nbytes = newTemp(Ity_I32);
         assign( t_nbytes, mkU32(NumBytes==0 ? 32 : NumBytes) );
         generate_lsw_sequence( t_nbytes, t_EA, rD_addr, 32 );
         *stopHere = True;
      }
      return True;

   case 0x215: // lswx (Load String Word Indexed, PPC32 p456)
      /* NB: does not reject the case where RA is in the range of
         registers to be loaded.  It should.  Although considering
         that that can only be detected at run time, it's not easy to
         do so. */
      if (rD_addr == rA_addr || rD_addr == rB_addr)
         return False;
      if (rD_addr == 0 && rA_addr == 0)
         return False;
      DIP("lswx r%u,r%u,r%u\n", rD_addr, rA_addr, rB_addr);
      t_nbytes = newTemp(Ity_I32);
      assign( t_EA, ea_rAor0_idxd(rA_addr,rB_addr) );
      assign( t_nbytes, unop( Iop_8Uto32, getXER_BC() ) );
      generate_lsw_sequence( t_nbytes, t_EA, rD_addr, 128 );
      *stopHere = True;
      return True;

   case 0x2D5: // stswi (Store String Word Immediate, PPC32 p528)
      DIP("stswi r%u,r%u,%d\n", rS_addr, rA_addr, NumBytes);
      assign( t_EA, ea_rAor0(rA_addr) );
      if (NumBytes == 8 && !mode64) {
         /* Special case hack */
         /* Mem[EA] = rD; Mem[EA+4] = (rD+1)%32 */
         store( mkexpr(t_EA),
                getIReg(rD_addr) );
         store( binop(Iop_Add32, mkexpr(t_EA), mkU32(4)),
                getIReg((rD_addr+1) % 32) );
      } else {
         t_nbytes = newTemp(Ity_I32);
         assign( t_nbytes, mkU32(NumBytes==0 ? 32 : NumBytes) );
         generate_stsw_sequence( t_nbytes, t_EA, rD_addr, 32 );
         *stopHere = True;
      }
      return True;

   case 0x295: // stswx (Store String Word Indexed, PPC32 p529)
      DIP("stswx r%u,r%u,r%u\n", rS_addr, rA_addr, rB_addr);
      t_nbytes = newTemp(Ity_I32);
      assign( t_EA, ea_rAor0_idxd(rA_addr,rB_addr) );
      assign( t_nbytes, unop( Iop_8Uto32, getXER_BC() ) );
      generate_stsw_sequence( t_nbytes, t_EA, rS_addr, 128 );
      *stopHere = True;
      return True;

   default:
      vex_printf("dis_int_ldst_str(ppc)(opc2)\n");
      return False;
   }
   return True;
}


/* ------------------------------------------------------------------
   Integer Branch Instructions
   ------------------------------------------------------------------ */

/*
  Branch helper function
  ok = BO[2] | ((CTR[0] != 0) ^ BO[1])
  Returns an I32 which is 0x00000000 if the ctr condition failed
  and 0xFFFFFFFF otherwise.
*/
static IRExpr* /* :: Ity_I32 */ branch_ctr_ok( UInt BO )
{
   IRType ty = mode64 ? Ity_I64 : Ity_I32;
   IRTemp ok = newTemp(Ity_I32);

   if ((BO >> 2) & 1) {     // independent of ctr
      assign( ok, mkU32(0xFFFFFFFF) );
   } else {
      if ((BO >> 1) & 1) {  // ctr == 0 ?
         assign( ok, unop( Iop_1Sto32,
                           binop( mkSzOp(ty, Iop_CmpEQ8),
                                  getGST( PPC_GST_CTR ),
                                  mkSzImm(ty,0))) );
      } else {              // ctr != 0 ?
         assign( ok, unop( Iop_1Sto32,
                           binop( mkSzOp(ty, Iop_CmpNE8),
                                  getGST( PPC_GST_CTR ),
                                  mkSzImm(ty,0))) );
      }
   }
   return mkexpr(ok);
}


/*
  Branch helper function cond_ok = BO[4] | (CR[BI] == BO[3])
  Returns an I32 which is either 0 if the condition failed or 
  some arbitrary nonzero value otherwise. */

static IRExpr* /* :: Ity_I32 */ branch_cond_ok( UInt BO, UInt BI )
{
   Int where;
   IRTemp res   = newTemp(Ity_I32);
   IRTemp cr_bi = newTemp(Ity_I32);
   
   if ((BO >> 4) & 1) {
      assign( res, mkU32(1) );
   } else {
      // ok = (CR[BI] == BO[3]) Note, the following relies on
      // getCRbit_anywhere returning a value which
      // is either zero or has exactly 1 bit set.  
      assign( cr_bi, getCRbit_anywhere( BI, &where ) );

      if ((BO >> 3) & 1) {
         /* We can use cr_bi as-is. */
         assign( res, mkexpr(cr_bi) );
      } else {
         /* We have to invert the sense of the information held in
            cr_bi.  For that we need to know which bit
            getCRbit_anywhere regards as significant. */
         assign( res, binop(Iop_Xor32, mkexpr(cr_bi),
                                       mkU32(1<<where)) );
      }
   }
   return mkexpr(res);
}


/*
  Integer Branch Instructions
*/
static Bool dis_branch ( UInt prefix, UInt theInstr,
                         const VexAbiInfo* vbi,
                         /*OUT*/DisResult* dres )
{
   UChar opc1    = ifieldOPC(theInstr);
   UChar BO      = ifieldRegDS(theInstr);
   UChar BI      = ifieldRegA(theInstr);
   UInt  BD_u16  = ifieldUIMM16(theInstr) & 0xFFFFFFFC; /* mask off */
   UChar b11to15 = ifieldRegB(theInstr);
   UInt  opc2    = ifieldOPClo10(theInstr);
   UInt  LI_u26  = ifieldUIMM26(theInstr) & 0xFFFFFFFC; /* mask off */
   UChar flag_AA = ifieldBIT1(theInstr);
   UChar flag_LK = ifieldBIT0(theInstr);

   IRType   ty        = mode64 ? Ity_I64 : Ity_I32;
   Addr64   tgt       = 0;
   UInt     BD        = extend_s_16to32(BD_u16);
   IRTemp   do_branch = newTemp(Ity_I32);
   IRTemp   ctr_ok    = newTemp(Ity_I32);
   IRTemp   cond_ok   = newTemp(Ity_I32);
   IRExpr*  e_nia     = mkSzImm(ty, nextInsnAddr());
   IRConst* c_nia     = mkSzConst(ty, nextInsnAddr());
   IRTemp   lr_old    = newTemp(ty);

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   /* Hack to pass through code that just wants to read the PC */
   if (theInstr == 0x429F0005) {
      DIP("bcl 0x%x, 0x%x (a.k.a mr lr,cia+4)\n", BO, BI);
      putGST( PPC_GST_LR, e_nia );
      return True;
   }

   /* The default what-next.  Individual cases can override it. */    
   dres->whatNext = Dis_StopHere;
   vassert(dres->jk_StopHere == Ijk_INVALID);

   switch (opc1) {
   case 0x12: // b     (Branch, PPC32 p360)
      if (flag_AA) {
         tgt = mkSzAddr( ty, extend_s_26to64(LI_u26) );
      } else {
         tgt = mkSzAddr( ty, guest_CIA_curr_instr +
                             (Long)extend_s_26to64(LI_u26) );
      }
      if (mode64) {
         DIP("b%s%s 0x%llx\n",
             flag_LK ? "l" : "", flag_AA ? "a" : "", tgt);
      } else {
         DIP("b%s%s 0x%x\n",
             flag_LK ? "l" : "", flag_AA ? "a" : "", (Addr32)tgt);
      }

      if (flag_LK) {
         putGST( PPC_GST_LR, e_nia );
         if (vbi->guest_ppc_zap_RZ_at_bl
             && vbi->guest_ppc_zap_RZ_at_bl( (ULong)tgt) ) {
            IRTemp t_tgt = newTemp(ty);
            assign(t_tgt, mode64 ? mkU64(tgt) : mkU32(tgt) );
            make_redzone_AbiHint( vbi, t_tgt,
                                  "branch-and-link (unconditional call)" );
         }
      }

      dres->jk_StopHere = flag_LK ? Ijk_Call : Ijk_Boring; ;
      putGST( PPC_GST_CIA, mkSzImm(ty, tgt) );
      break;
      
   case 0x10: // bc    (Branch Conditional, PPC32 p361)
      DIP("bc%s%s 0x%x, 0x%x, 0x%x\n",
          flag_LK ? "l" : "", flag_AA ? "a" : "", BO, BI, BD);
      
      if (!(BO & 0x4)) {
         putGST( PPC_GST_CTR,
                 binop(mkSzOp(ty, Iop_Sub8),
                       getGST( PPC_GST_CTR ), mkSzImm(ty, 1)) );
      }

      /* This is a bit subtle.  ctr_ok is either all 0s or all 1s.
         cond_ok is either zero or nonzero, since that's the cheapest
         way to compute it.  Anding them together gives a value which
         is either zero or non zero and so that's what we must test
         for in the IRStmt_Exit. */
      assign( ctr_ok,  branch_ctr_ok( BO ) );
      assign( cond_ok, branch_cond_ok( BO, BI ) );
      assign( do_branch,
              binop(Iop_And32, mkexpr(cond_ok), mkexpr(ctr_ok)) );

      if (flag_AA) {
         tgt = mkSzAddr(ty, extend_s_16to64(BD_u16));
      } else {
         tgt = mkSzAddr(ty, guest_CIA_curr_instr +
                            (Long)extend_s_16to64(BD_u16));
      }
      if (flag_LK)
         putGST( PPC_GST_LR, e_nia );
      
      stmt( IRStmt_Exit(
               binop(Iop_CmpNE32, mkexpr(do_branch), mkU32(0)),
               flag_LK ? Ijk_Call : Ijk_Boring,
               mkSzConst(ty, tgt), OFFB_CIA ) );

      dres->jk_StopHere = Ijk_Boring;
      putGST( PPC_GST_CIA, e_nia );
      break;
      
   case 0x13:
      /* For bclr and bcctr, it appears that the lowest two bits of
         b11to15 are a branch hint, and so we only need to ensure it's
         of the form 000XX. */
      if ((b11to15 & ~3) != 0) {
         vex_printf("dis_int_branch(ppc)(0x13,b11to15)(%d)\n", b11to15);
         return False;
      }

      switch (opc2) {
      case 0x210: // bcctr (Branch Cond. to Count Register, PPC32 p363) 
         if ((BO & 0x4) == 0) { // "decr and test CTR" option invalid
            vex_printf("dis_int_branch(ppc)(bcctr,BO)\n");
            return False;
         }
         DIP("bcctr%s 0x%x, 0x%x\n", flag_LK ? "l" : "", BO, BI);
         
         assign( cond_ok, branch_cond_ok( BO, BI ) );

         /* FIXME: this is confusing.  lr_old holds the old value
            of ctr, not lr :-) */
         assign( lr_old, addr_align( getGST( PPC_GST_CTR ), 4 ));

         if (flag_LK)
            putGST( PPC_GST_LR, e_nia );
         
         stmt( IRStmt_Exit(
                  binop(Iop_CmpEQ32, mkexpr(cond_ok), mkU32(0)),
                  Ijk_Boring,
                  c_nia, OFFB_CIA ));

         if (flag_LK && vbi->guest_ppc_zap_RZ_at_bl) {
            make_redzone_AbiHint( vbi, lr_old,
                                  "b-ctr-l (indirect call)" );
	 }

         dres->jk_StopHere = flag_LK ? Ijk_Call : Ijk_Boring;;
         putGST( PPC_GST_CIA, mkexpr(lr_old) );
         break;
         
      case 0x010: { // bclr (Branch Cond. to Link Register, PPC32 p365) 
         Bool vanilla_return = False;
         if ((BO & 0x14 /* 1z1zz */) == 0x14 && flag_LK == 0) {
            DIP("blr\n");
            vanilla_return = True;
         } else {
            DIP("bclr%s 0x%x, 0x%x\n", flag_LK ? "l" : "", BO, BI);
         }

         if (!(BO & 0x4)) {
            putGST( PPC_GST_CTR,
                    binop(mkSzOp(ty, Iop_Sub8),
                          getGST( PPC_GST_CTR ), mkSzImm(ty, 1)) );
         }
         
         /* See comments above for 'bc' about this */
         assign( ctr_ok,  branch_ctr_ok( BO ) );
         assign( cond_ok, branch_cond_ok( BO, BI ) );
         assign( do_branch,
                 binop(Iop_And32, mkexpr(cond_ok), mkexpr(ctr_ok)) );
         
         assign( lr_old, addr_align( getGST( PPC_GST_LR ), 4 ));

         if (flag_LK)
            putGST( PPC_GST_LR,  e_nia );

         stmt( IRStmt_Exit(
                  binop(Iop_CmpEQ32, mkexpr(do_branch), mkU32(0)),
                  Ijk_Boring,
                  c_nia, OFFB_CIA ));

         if (vanilla_return && vbi->guest_ppc_zap_RZ_at_blr) {
            make_redzone_AbiHint( vbi, lr_old,
                                  "branch-to-lr (unconditional return)" );
         }

         /* blrl is pretty strange; it's like a return that sets the
            return address of its caller to the insn following this
            one.  Mark it as a return. */
         dres->jk_StopHere = Ijk_Ret;  /* was flag_LK ? Ijk_Call : Ijk_Ret; */
         putGST( PPC_GST_CIA, mkexpr(lr_old) );
         break;
      }
      default:
         vex_printf("dis_int_branch(ppc)(opc2)\n");
         return False;
      }
      break;
      
   default:
      vex_printf("dis_int_branch(ppc)(opc1)\n");
      return False;
   }
   
   return True;
}

/*
 *  PC relative instruction
 */
static Bool dis_pc_relative ( UInt prefix, UInt theInstr )
{
   /* DX-Form */
   UChar opc1 = ifieldOPC(theInstr);
   unsigned long long D;
   UInt d0 = IFIELD(theInstr,  6, 10);
   UInt d1 = IFIELD(theInstr, 16,  5);
   UInt d2 = IFIELD(theInstr,  0,  1);
   UChar rT_addr = ifieldRegDS(theInstr);
   UInt  opc2    = ifieldOPClo5(theInstr);
   IRType ty     = mode64 ? Ity_I64 : Ity_I32;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   if ( opc1 != 0x13) {
      vex_printf("dis_pc_relative(ppc)(opc1)\n");
      return False;
   }

   switch (opc2) {
   case 0x002:   // addpcis  (Add PC immediate Shifted DX-form)
      {
         IRExpr* nia     = mkSzImm(ty, nextInsnAddr());
         IRExpr* result;

         D = (d0 << 6) | (d1 << 1) | d2;
         DIP("addpcis %u,%llu\n", rT_addr, D);

         if ( (D & 0x8000) == 0x8000 )
            D = 0xFFFFFFFFFFFF0000UL | D;  // sign extend

         if ( ty == Ity_I32 ) {
            result = binop( Iop_Add32, nia, mkU32( D << 16 ) );
         } else {
            vassert( ty == Ity_I64 );
            result = binop( Iop_Add64, nia, mkU64( D << 16 ) );
         }

         putIReg( rT_addr, result);
      }
      break;

   default:
      vex_printf("dis_pc_relative(ppc)(opc2)\n");
      return False;
   }

   return True;
}

/*
  Condition Register Logical Instructions
 */
static Bool dis_cond_logic ( UInt prefix, UInt theInstr )
{
   /* XL-Form */
   UChar opc1      = ifieldOPC(theInstr);
   UChar crbD_addr = ifieldRegDS(theInstr);
   UChar crfD_addr = toUChar( IFIELD(theInstr, 23, 3) );
   UChar crbA_addr = ifieldRegA(theInstr);
   UChar crfS_addr = toUChar( IFIELD(theInstr, 18, 3) );
   UChar crbB_addr = ifieldRegB(theInstr);
   UInt  opc2      = ifieldOPClo10(theInstr);
   UChar b0        = ifieldBIT0(theInstr);

   IRTemp crbD     = newTemp(Ity_I32);
   IRTemp crbA     = newTemp(Ity_I32);
   IRTemp crbB     = newTemp(Ity_I32);

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   if (opc1 != 19 || b0 != 0) {
      vex_printf("dis_cond_logic(ppc)(opc1)\n");
      return False;
   }

   if (opc2 == 0) {  // mcrf    (Move Cond Reg Field, PPC32 p464)
      if (((crbD_addr & 0x3) != 0) ||
          ((crbA_addr & 0x3) != 0) || (crbB_addr != 0)) {
         vex_printf("dis_cond_logic(ppc)(crbD|crbA|crbB != 0)\n");
         return False;
      }
      DIP("mcrf cr%u,cr%u\n", crfD_addr, crfS_addr);
      putCR0(   crfD_addr, getCR0(  crfS_addr) );
      putCR321( crfD_addr, getCR321(crfS_addr) );
   } else {
      assign( crbA, getCRbit(crbA_addr) );
      if (crbA_addr == crbB_addr)
         crbB = crbA;
      else
         assign( crbB, getCRbit(crbB_addr) );

      switch (opc2) {
      case 0x101: // crand   (Cond Reg AND, PPC32 p372)
         DIP("crand crb%d,crb%d,crb%d\n", crbD_addr, crbA_addr, crbB_addr);
         assign( crbD, binop(Iop_And32, mkexpr(crbA), mkexpr(crbB)) );
         break;
      case 0x081: // crandc  (Cond Reg AND w. Complement, PPC32 p373)
         DIP("crandc crb%d,crb%d,crb%d\n", crbD_addr, crbA_addr, crbB_addr);
         assign( crbD, binop(Iop_And32, 
                             mkexpr(crbA),
                             unop(Iop_Not32, mkexpr(crbB))) );
         break;
      case 0x121: // creqv   (Cond Reg Equivalent, PPC32 p374)
         DIP("creqv crb%d,crb%d,crb%d\n", crbD_addr, crbA_addr, crbB_addr);
         assign( crbD, unop(Iop_Not32,
                            binop(Iop_Xor32, mkexpr(crbA), mkexpr(crbB))) );
         break;
      case 0x0E1: // crnand  (Cond Reg NAND, PPC32 p375)
         DIP("crnand crb%d,crb%d,crb%d\n", crbD_addr, crbA_addr, crbB_addr);
         assign( crbD, unop(Iop_Not32,
                            binop(Iop_And32, mkexpr(crbA), mkexpr(crbB))) );
         break;
      case 0x021: // crnor   (Cond Reg NOR, PPC32 p376)
         DIP("crnor crb%d,crb%d,crb%d\n", crbD_addr, crbA_addr, crbB_addr);
         assign( crbD, unop(Iop_Not32,
                            binop(Iop_Or32, mkexpr(crbA), mkexpr(crbB))) );
         break;
      case 0x1C1: // cror    (Cond Reg OR, PPC32 p377)
         DIP("cror crb%d,crb%d,crb%d\n", crbD_addr, crbA_addr, crbB_addr);
         assign( crbD, binop(Iop_Or32, mkexpr(crbA), mkexpr(crbB)) );
         break;
      case 0x1A1: // crorc   (Cond Reg OR w. Complement, PPC32 p378)
         DIP("crorc crb%d,crb%d,crb%d\n", crbD_addr, crbA_addr, crbB_addr);
         assign( crbD, binop(Iop_Or32, 
                             mkexpr(crbA),
                             unop(Iop_Not32, mkexpr(crbB))) );
         break;
      case 0x0C1: // crxor   (Cond Reg XOR, PPC32 p379)
         DIP("crxor crb%d,crb%d,crb%d\n", crbD_addr, crbA_addr, crbB_addr);
         assign( crbD, binop(Iop_Xor32, mkexpr(crbA), mkexpr(crbB)) );
         break;
      default:
         vex_printf("dis_cond_logic(ppc)(opc2)\n");
         return False;
      }

      putCRbit( crbD_addr, mkexpr(crbD) );
   }
   return True;
}

static Bool dis_set_bool_condition ( UInt prefixInstr, UInt theInstr )
{
   UInt  opc2    = ifieldOPClo10(theInstr);
   UChar BI      = toUChar( IFIELD( theInstr, 16, 5 ) );
   UInt  rT_addr = ifieldRegDS( theInstr );
   IRType ty     = mode64 ? Ity_I64 : Ity_I32;
   IROp  Iop_1XtoX;

   /* There is no prefixed version of these instructions.  */
   vassert( !prefix_instruction( prefixInstr ) );

   switch (opc2) {
   case 0x180: // setbc
      /* If bit BI of the CR contains a 1, register RT is set to 1.
         Otherwise, register RT is set to 0.  */
      DIP(" setbc %u,%u\n", rT_addr, BI);
      Iop_1XtoX = mode64 ? Iop_1Uto64 : Iop_1Uto32;
      putIReg( rT_addr, unop( Iop_1XtoX,
                              binop( Iop_CmpEQ32,
                                     getCRbit( BI ),
                                     mkU32( 1 ) ) ) );
      break;

   case 0x1A0: // setbcr
      /* If bit BI of the CR contains a 1, register RT is set to 0.
         Otherwise, register RT is set to 1.  */
      DIP(" setbcr %u,%u\n", rT_addr, BI);
      Iop_1XtoX = mode64 ? Iop_1Uto64 : Iop_1Uto32;
      putIReg( rT_addr, unop( Iop_1XtoX,
                                     binop( Iop_CmpNE32,
                                            getCRbit( BI ),
                                            mkU32( 1 ) ) ) );
      break;

   case 0x1C0: // setnbc
      /* If bit BI of the CR contains a 1, register RT is set to -1.
         Otherwise, register RT is set to 0.  */
      DIP(" setnbc %u,%u\n", rT_addr, BI);
      Iop_1XtoX = mode64 ? Iop_1Sto64 : Iop_1Sto32;
      putIReg( rT_addr, binop( mkSzOp(ty, Iop_And8),
                               mkSzImm( ty, -1 ),
                               unop( Iop_1XtoX,
                                     binop( Iop_CmpEQ32,
                                            getCRbit( BI ),
                                            mkU32( 1 ) ) ) ) );
      break;

   case 0x1E0: // setnbcr
      /* If bit BI of the CR contains a 1, register RT is set to -1.
         Otherwise, register RT is set to 0.  */
      DIP(" setnbcr %u,%u\n", rT_addr, BI);
      Iop_1XtoX = mode64 ? Iop_1Sto64 : Iop_1Sto32;
      putIReg( rT_addr, binop( mkSzOp(ty, Iop_And8),
                               mkSzImm( ty, -1 ),
                               unop( Iop_1XtoX,
                                     binop( Iop_CmpNE32,
                                            getCRbit( BI ),
                                            mkU32( 1 ) ) ) ) );
      break;

   default:
      vex_printf("dis_set_bool_condition(ppc)(opc2)\n");
      return False;
   }

   return True;
}

/* 
  Trap instructions
*/

/* Do the code generation for a trap.  Returned Bool is true iff
   this is an unconditional trap.  If the two arg IRExpr*s are 
   Ity_I32s then the comparison is 32-bit.  If they are Ity_I64s
   then they are 64-bit, and we must be disassembling 64-bit
   instructions. */
static Bool do_trap ( UChar TO, 
                      IRExpr* argL0, IRExpr* argR0, Addr64 cia )
{
   IRTemp argL, argR;
   IRExpr *argLe, *argRe, *cond, *tmp;

   Bool    is32bit = typeOfIRExpr(irsb->tyenv, argL0 ) == Ity_I32;

   IROp    opAND     = is32bit ? Iop_And32     : Iop_And64;
   IROp    opOR      = is32bit ? Iop_Or32      : Iop_Or64;
   IROp    opCMPORDS = is32bit ? Iop_CmpORD32S : Iop_CmpORD64S;
   IROp    opCMPORDU = is32bit ? Iop_CmpORD32U : Iop_CmpORD64U;
   IROp    opCMPNE   = is32bit ? Iop_CmpNE32   : Iop_CmpNE64;
   IROp    opCMPEQ   = is32bit ? Iop_CmpEQ32   : Iop_CmpEQ64;
   IRExpr* const0    = is32bit ? mkU32(0)      : mkU64(0);
   IRExpr* const2    = is32bit ? mkU32(2)      : mkU64(2);
   IRExpr* const4    = is32bit ? mkU32(4)      : mkU64(4);
   IRExpr* const8    = is32bit ? mkU32(8)      : mkU64(8);

   const UChar b11100 = 0x1C;
   const UChar b00111 = 0x07;

   if (is32bit) {
      vassert( typeOfIRExpr(irsb->tyenv, argL0) == Ity_I32 );
      vassert( typeOfIRExpr(irsb->tyenv, argR0) == Ity_I32 );
   } else {
      vassert( typeOfIRExpr(irsb->tyenv, argL0) == Ity_I64 );
      vassert( typeOfIRExpr(irsb->tyenv, argR0) == Ity_I64 );
      vassert( mode64 );
   }

   if ((TO & b11100) == b11100 || (TO & b00111) == b00111) {
      /* Unconditional trap.  Just do the exit without 
         testing the arguments. */
      stmt( IRStmt_Exit( 
               binop(opCMPEQ, const0, const0), 
               Ijk_SigTRAP,
               mode64 ? IRConst_U64(cia) : IRConst_U32((UInt)cia),
               OFFB_CIA
      ));
      return True; /* unconditional trap */
   }

   if (is32bit) {
      argL = newTemp(Ity_I32);
      argR = newTemp(Ity_I32);
   } else {
      argL = newTemp(Ity_I64);
      argR = newTemp(Ity_I64);
   }

   assign( argL, argL0 );
   assign( argR, argR0 );

   argLe = mkexpr(argL);
   argRe = mkexpr(argR);

   cond = const0;
   if (TO & 16) { // L <s R
      tmp = binop(opAND, binop(opCMPORDS, argLe, argRe), const8);
      cond = binop(opOR, tmp, cond);
   }
   if (TO & 8) { // L >s R
      tmp = binop(opAND, binop(opCMPORDS, argLe, argRe), const4);
      cond = binop(opOR, tmp, cond);
   }
   if (TO & 4) { // L == R
      tmp = binop(opAND, binop(opCMPORDS, argLe, argRe), const2);
      cond = binop(opOR, tmp, cond);
   }
   if (TO & 2) { // L <u R
      tmp = binop(opAND, binop(opCMPORDU, argLe, argRe), const8);
      cond = binop(opOR, tmp, cond);
   }
   if (TO & 1) { // L >u R
      tmp = binop(opAND, binop(opCMPORDU, argLe, argRe), const4);
      cond = binop(opOR, tmp, cond);
   }
   stmt( IRStmt_Exit( 
            binop(opCMPNE, cond, const0), 
            Ijk_SigTRAP,
            mode64 ? IRConst_U64(cia) : IRConst_U32((UInt)cia),
            OFFB_CIA
   ));
   return False; /* not an unconditional trap */
}

static Bool dis_trapi ( UInt prefix, UInt theInstr,
                        /*OUT*/DisResult* dres )
{
   /* D-Form */
   UChar  opc1    = ifieldOPC(theInstr);
   UChar  TO      = ifieldRegDS(theInstr);
   UChar  rA_addr = ifieldRegA(theInstr);
   UInt   uimm16  = ifieldUIMM16(theInstr);
   ULong  simm16  = extend_s_16to64(uimm16);
   Addr64 cia     = guest_CIA_curr_instr;
   IRType ty      = mode64 ? Ity_I64 : Ity_I32;
   Bool   uncond  = False;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   switch (opc1) {
   case 0x03: // twi  (Trap Word Immediate, PPC32 p548)
      uncond = do_trap( TO, 
                        mode64 ? unop(Iop_64to32, getIReg(rA_addr)) 
                               : getIReg(rA_addr),
                        mkU32( (UInt)simm16 ),
                        cia );
      if (TO == 4) {
         DIP("tweqi r%u,%d\n", rA_addr, (Int)simm16);
      } else {
         DIP("tw%di r%u,%d\n", TO, rA_addr, (Int)simm16);
      }
      break;
   case 0x02: // tdi
      if (!mode64)
         return False;
      uncond = do_trap( TO, getIReg(rA_addr), mkU64( (ULong)simm16 ), cia );
      if (TO == 4) {
         DIP("tdeqi r%u,%d\n", rA_addr, (Int)simm16);
      } else {
         DIP("td%di r%u,%d\n", TO, rA_addr, (Int)simm16);
      }
      break;
   default:
      return False;
   }

   if (uncond) {
      /* If the trap shows signs of being unconditional, don't
         continue decoding past it. */
      putGST( PPC_GST_CIA, mkSzImm( ty, nextInsnAddr() ));
      dres->jk_StopHere = Ijk_Boring;
      dres->whatNext    = Dis_StopHere;
   }

   return True;
}

static Bool dis_trap ( UInt prefix, UInt theInstr,
                        /*OUT*/DisResult* dres )
{
   /* X-Form */
   UInt   opc2    = ifieldOPClo10(theInstr);
   UChar  TO      = ifieldRegDS(theInstr);
   UChar  rA_addr = ifieldRegA(theInstr);
   UChar  rB_addr = ifieldRegB(theInstr);
   Addr64 cia     = guest_CIA_curr_instr;
   IRType ty      = mode64 ? Ity_I64 : Ity_I32;
   Bool   uncond  = False;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   if (ifieldBIT0(theInstr) != 0)
      return False;

   switch (opc2) {
   case 0x004: // tw  (Trap Word, PPC64 p540)
      uncond = do_trap( TO, 
                        mode64 ? unop(Iop_64to32, getIReg(rA_addr)) 
                               : getIReg(rA_addr),
                        mode64 ? unop(Iop_64to32, getIReg(rB_addr)) 
                               : getIReg(rB_addr),
                        cia );
      if (TO == 4) {
         DIP("tweq r%u,r%u\n", rA_addr, rB_addr);
      } else {
         DIP("tw%d r%u,r%u\n", TO, rA_addr, rB_addr);
      }
      break;
   case 0x044: // td (Trap Doubleword, PPC64 p534)
      if (!mode64)
         return False;
      uncond = do_trap( TO, getIReg(rA_addr), getIReg(rB_addr), cia );
      if (TO == 4) {
         DIP("tdeq r%u,r%u\n", rA_addr, rB_addr);
      } else {
         DIP("td%d r%u,r%u\n", TO, rA_addr, rB_addr);
      }
      break;
   default:
      return False;
   }

   if (uncond) {
      /* If the trap shows signs of being unconditional, don't
         continue decoding past it. */
      putGST( PPC_GST_CIA, mkSzImm( ty, nextInsnAddr() ));
      dres->jk_StopHere = Ijk_Boring;
      dres->whatNext    = Dis_StopHere;
   }

   return True;
}


/*
  System Linkage Instructions
*/

static Bool dis_syslink ( UInt prefix, UInt theInstr,
                          const VexAbiInfo* abiinfo, DisResult* dres,
                          Bool allow_scv, Bool sigill_diag )
{
   IRType ty = mode64 ? Ity_I64 : Ity_I32;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   if ((theInstr != 0x44000002)       // sc
       && (theInstr != 0x44000001)) { // scv
       vex_printf("dis_syslink(ppc)(theInstr)\n");
      return False;
   }

   /* The PPC syscall uses guest_GPR9 to pass a flag to indicate which
      system call instruction is to be used.  Arg7 = SC_FLAG for the sc
      instruction; Arg7 = SCV_FLAG for the scv instruction.  */
   if (theInstr == 0x44000002) {
      // sc  (System Call, PPC32 p504)
      DIP("sc\n");
      put_syscall_flag( mkU32(SC_FLAG) );
   } else if (theInstr == 0x44000001) {
      if (allow_scv) {   // scv
         DIP("scv\n");
         put_syscall_flag( mkU32(SCV_FLAG) );
      } else {
         if (sigill_diag)
            vex_printf("The scv instruction is not supported in this environment per the HWCAPS2 capability flags.\n");
         return False;
      }
   } else {
      /* Unknown instruction */
      return False;
   }

   /* Copy CIA into the IP_AT_SYSCALL pseudo-register, so that on Darwin
      Valgrind can back the guest up to this instruction if it needs
      to restart the syscall. */
   putGST( PPC_GST_IP_AT_SYSCALL, getGST( PPC_GST_CIA ) );

   /* It's important that all ArchRegs carry their up-to-date value
      at this point.  So we declare an end-of-block here, which
      forces any TempRegs caching ArchRegs to be flushed. */
   putGST( PPC_GST_CIA, mkSzImm( ty, nextInsnAddr() ));

   dres->whatNext    = Dis_StopHere;
   dres->jk_StopHere = Ijk_Sys_syscall;
   return True;
}


/*
  Memory Synchronization Instructions

  Note on Reservations:
  We rely on the assumption that V will in fact only allow one thread at
  once to run.  In effect, a thread can make a reservation, but we don't
  check any stores it does.  Instead, the reservation is cancelled when
  the scheduler switches to another thread (run_thread_for_a_while()).
*/
static Bool dis_memsync ( UInt prefix, UInt theInstr )
{
   /* X-Form, XL-Form */
   UChar opc1    = ifieldOPC(theInstr);
   UInt  b11to25 = IFIELD(theInstr, 11, 15);
   UChar flag_L  = IFIELD(theInstr, 21, 2);   //ISA 3.0
   UInt  b11to20 = IFIELD(theInstr, 11, 10);
   UInt  M0      = IFIELD(theInstr, 11, 5);
   UChar rD_addr = ifieldRegDS(theInstr);
   UChar rS_addr = rD_addr;
   UChar rA_addr = ifieldRegA(theInstr);
   UChar rB_addr = ifieldRegB(theInstr);
   UInt  opc2    = ifieldOPClo10(theInstr);
   UChar b0      = ifieldBIT0(theInstr);

   IRType ty     = mode64 ? Ity_I64 : Ity_I32;
   IRTemp EA     = newTemp(ty);

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   assign( EA, ea_rAor0_idxd( rA_addr, rB_addr ) );

   switch (opc1) {
   /* XL-Form */
   case 0x13:   // isync (Instruction Synchronize, PPC32 p432)
      if (opc2 != 0x096) {
         vex_printf("dis_memsync(ppc)(0x13,opc2)\n");
         return False;
      }
      if (b11to25 != 0 || b0 != 0) {
         vex_printf("dis_memsync(ppc)(0x13,b11to25|b0)\n");
         return False;
      }
      DIP("isync\n");
      stmt( IRStmt_MBE(Imbe_Fence) );
      break;

   /* X-Form */
   case 0x1F:
      switch (opc2) {
      case 0x356: // eieio or mbar (Enforce In-Order Exec of I/O, PPC32 p394)
         if (M0 == 0) {
            if (b11to20 != 0 || b0 != 0) {
               vex_printf("dis_memsync(ppc)(eieio,b11to20|b0)\n");
               return False;
            }
            DIP("eieio\n");
         } else {
            if (b11to20 != 0 || b0 != 0) {
               vex_printf("dis_memsync(ppc)(mbar,b11to20|b0)\n");
               return False;
            }
            DIP("mbar %u\n", M0);
         }
         /* Insert a memory fence, just to be on the safe side. */
         stmt( IRStmt_MBE(Imbe_Fence) );
         break;

      case 0x014: { // lwarx (Load Word and Reserve Indexed, PPC32 p458)
         IRTemp res;
         /* According to the PowerPC ISA version 2.05, b0 (called EH
            in the documentation) is merely a hint bit to the
            hardware, I think as to whether or not contention is
            likely.  So we can just ignore it. */
         DIP("lwarx r%u,r%u,r%u,EH=%u\n", rD_addr, rA_addr, rB_addr, b0);

         // trap if misaligned
         gen_SIGBUS_if_misaligned( EA, 4 );

         // and actually do the load
         res = newTemp(Ity_I32);
         stmt( stmt_load(res, mkexpr(EA), NULL/*this is a load*/) );

         putIReg( rD_addr, mkWidenFrom32(ty, mkexpr(res), False) );
         break;
      }

      case 0x034: { // lbarx (Load Word and Reserve Indexed)
         IRTemp res;
         /* According to the PowerPC ISA version 2.05, b0 (called EH
            in the documentation) is merely a hint bit to the
            hardware, I think as to whether or not contention is
            likely.  So we can just ignore it. */
         DIP("lbarx r%u,r%u,r%u,EH=%u\n", rD_addr, rA_addr, rB_addr, b0);

         // and actually do the load
         res = newTemp(Ity_I8);
         stmt( stmt_load(res, mkexpr(EA), NULL/*this is a load*/) );

         putIReg( rD_addr, mkWidenFrom8(ty, mkexpr(res), False) );
         break;
     }

      case 0x074: { // lharx (Load Word and Reserve Indexed)
         IRTemp res;
         /* According to the PowerPC ISA version 2.05, b0 (called EH
            in the documentation) is merely a hint bit to the
            hardware, I think as to whether or not contention is
            likely.  So we can just ignore it. */
         DIP("lharx r%u,r%u,r%u,EH=%u\n", rD_addr, rA_addr, rB_addr, b0);

         // trap if misaligned
         gen_SIGBUS_if_misaligned( EA, 2 );

         // and actually do the load
         res = newTemp(Ity_I16);
         stmt( stmt_load(res, mkexpr(EA), NULL/*this is a load*/) );

         putIReg( rD_addr, mkWidenFrom16(ty, mkexpr(res), False) );
         break;
      }

      case 0x096: { 
         // stwcx. (Store Word Conditional Indexed, PPC32 p532)
         // Note this has to handle stwcx. in both 32- and 64-bit modes,
         // so isn't quite as straightforward as it might otherwise be.
         IRTemp rS = newTemp(Ity_I32);
         IRTemp resSC;
         if (b0 != 1) {
            vex_printf("dis_memsync(ppc)(stwcx.,b0)\n");
            return False;
         }
         DIP("stwcx. r%u,r%u,r%u\n", rS_addr, rA_addr, rB_addr);

         // trap if misaligned
         gen_SIGBUS_if_misaligned( EA, 4 );

         // Get the data to be stored, and narrow to 32 bits if necessary
         assign( rS, mkNarrowTo32(ty, getIReg(rS_addr)) );

         // Do the store, and get success/failure bit into resSC
         resSC = newTemp(Ity_I1);
         stmt( stmt_load( resSC, mkexpr(EA), mkexpr(rS)) );

         // Set CR0[LT GT EQ S0] = 0b000 || XER[SO]  on failure
         // Set CR0[LT GT EQ S0] = 0b001 || XER[SO]  on success
         putCR321(0, binop(Iop_Shl8, unop(Iop_1Uto8, mkexpr(resSC)), mkU8(1)));
         putCR0(0, getXER_SO());

         /* Note:
            If resaddr != lwarx_resaddr, CR0[EQ] is undefined, and
            whether rS is stored is dependent on that value. */
         /* So I guess we can just ignore this case? */
         break;
      }

      case 0x2B6: {
         // stbcx. (Store Byte Conditional Indexed)
         // Note this has to handle stbcx. in both 32- and 64-bit modes,
         // so isn't quite as straightforward as it might otherwise be.
         IRTemp rS = newTemp(Ity_I8);
         IRTemp resSC;
         if (b0 != 1) {
            vex_printf("dis_memsync(ppc)(stbcx.,b0)\n");
            return False;
         }
         DIP("stbcx. r%u,r%u,r%u\n", rS_addr, rA_addr, rB_addr);

         // Get the data to be stored, and narrow to 32 bits if necessary
         assign( rS, mkNarrowTo8(ty, getIReg(rS_addr)) );

         // Do the store, and get success/failure bit into resSC
         resSC = newTemp(Ity_I1);
         stmt( stmt_load( resSC, mkexpr(EA), mkexpr(rS)) );

         // Set CR0[LT GT EQ S0] = 0b000 || XER[SO]  on failure
         // Set CR0[LT GT EQ S0] = 0b001 || XER[SO]  on success
         putCR321(0, binop(Iop_Shl8, unop(Iop_1Uto8, mkexpr(resSC)), mkU8(1)));
         putCR0(0, getXER_SO());

         /* Note:
            If resaddr != lbarx_resaddr, CR0[EQ] is undefined, and
            whether rS is stored is dependent on that value. */
         /* So I guess we can just ignore this case? */
         break;
      }

      case 0x2D6: {
         // sthcx. (Store Word Conditional Indexed, PPC32 p532)
         // Note this has to handle sthcx. in both 32- and 64-bit modes,
         // so isn't quite as straightforward as it might otherwise be.
         IRTemp rS = newTemp(Ity_I16);
         IRTemp resSC;
         if (b0 != 1) {
            vex_printf("dis_memsync(ppc)(stwcx.,b0)\n");
            return False;
         }
         DIP("sthcx. r%u,r%u,r%u\n", rS_addr, rA_addr, rB_addr);

         // trap if misaligned
         gen_SIGBUS_if_misaligned( EA, 2 );

         // Get the data to be stored, and narrow to 16 bits if necessary
         assign( rS, mkNarrowTo16(ty, getIReg(rS_addr)) );

         // Do the store, and get success/failure bit into resSC
         resSC = newTemp(Ity_I1);
         stmt( stmt_load( resSC, mkexpr(EA), mkexpr(rS)) );

         // Set CR0[LT GT EQ S0] = 0b000 || XER[SO]  on failure
         // Set CR0[LT GT EQ S0] = 0b001 || XER[SO]  on success
         putCR321(0, binop(Iop_Shl8, unop(Iop_1Uto8, mkexpr(resSC)), mkU8(1)));
         putCR0(0, getXER_SO());

         /* Note:
            If resaddr != lharx_resaddr, CR0[EQ] is undefined, and
            whether rS is stored is dependent on that value. */
         /* So I guess we can just ignore this case? */
         break;
      }

      case 0x256: // sync (Synchronize, PPC32 p543), 
                  // also lwsync (L==1), ptesync (L==2)
         /* http://sources.redhat.com/ml/binutils/2000-12/msg00311.html

            The PowerPC architecture used in IBM chips has expanded
            the sync instruction into two variants: lightweight sync
            and heavyweight sync.  The original sync instruction is
            the new heavyweight sync and lightweight sync is a strict
            subset of the heavyweight sync functionality. This allows
            the programmer to specify a less expensive operation on
            high-end systems when the full sync functionality is not
            necessary.

            The basic "sync" mnemonic now utilizes an operand. "sync"
            without an operand now becomes a extended mnemonic for
            heavyweight sync.  Processors without the lwsync
            instruction will not decode the L field and will perform a
            heavyweight sync.  Everything is backward compatible.

            sync    =       sync 0
            lwsync  =       sync 1
            ptesync =       sync 2    *** TODO - not implemented ***
         */
         if (b11to20 != 0 || b0 != 0) {
            vex_printf("dis_memsync(ppc)(sync/lwsync,b11to20|b0)\n");
            return False;
         }
         if (flag_L != 0/*sync*/ && flag_L != 1/*lwsync*/) {
            vex_printf("dis_memsync(ppc)(sync/lwsync,flag_L)\n");
            return False;
         }
         DIP("%ssync\n", flag_L == 1 ? "lw" : "");
         /* Insert a memory fence.  It's sometimes important that these
            are carried through to the generated code. */
         stmt( IRStmt_MBE(Imbe_Fence) );
         break;

      /* 64bit Memsync */
      case 0x054: { // ldarx (Load DWord and Reserve Indexed, PPC64 p473)
         IRTemp res;
         /* According to the PowerPC ISA version 2.05, b0 (called EH
            in the documentation) is merely a hint bit to the
            hardware, I think as to whether or not contention is
            likely.  So we can just ignore it. */
         if (!mode64)
            return False;
         DIP("ldarx r%u,r%u,r%u,EH=%u\n", rD_addr, rA_addr, rB_addr, b0);

         // trap if misaligned
         gen_SIGBUS_if_misaligned( EA, 8 );

         // and actually do the load
         res = newTemp(Ity_I64);
         stmt( stmt_load( res, mkexpr(EA), NULL/*this is a load*/) );

         putIReg( rD_addr, mkexpr(res) );
         break;
      }
      
      case 0x0D6: { // stdcx. (Store DWord Condition Indexd, PPC64 p581)
         // A marginally simplified version of the stwcx. case
         IRTemp rS = newTemp(Ity_I64);
         IRTemp resSC;
         if (b0 != 1) {
            vex_printf("dis_memsync(ppc)(stdcx.,b0)\n");
            return False;
         }
         if (!mode64)
            return False;
         DIP("stdcx. r%u,r%u,r%u\n", rS_addr, rA_addr, rB_addr);

         // trap if misaligned
         gen_SIGBUS_if_misaligned( EA, 8 );

         // Get the data to be stored
         assign( rS, getIReg(rS_addr) );

         // Do the store, and get success/failure bit into resSC
         resSC = newTemp(Ity_I1);
         stmt( stmt_load( resSC, mkexpr(EA), mkexpr(rS)) );

         // Set CR0[LT GT EQ S0] = 0b000 || XER[SO]  on failure
         // Set CR0[LT GT EQ S0] = 0b001 || XER[SO]  on success
         putCR321(0, binop(Iop_Shl8, unop(Iop_1Uto8, mkexpr(resSC)), mkU8(1)));
         putCR0(0, getXER_SO());

         /* Note:
            If resaddr != lwarx_resaddr, CR0[EQ] is undefined, and
            whether rS is stored is dependent on that value. */
         /* So I guess we can just ignore this case? */
         break;
      }

      /* 128bit Memsync */
      case 0x114: { // lqarx (Load QuadWord and Reserve Indexed)
         IRTemp res_hi = newTemp(ty);
         IRTemp res_lo = newTemp(ty);

         /* According to the PowerPC ISA version 2.07, b0 (called EH
            in the documentation) is merely a hint bit to the
            hardware, I think as to whether or not contention is
            likely.  So we can just ignore it. */
         DIP("lqarx r%u,r%u,r%u,EH=%u\n", rD_addr, rA_addr, rB_addr, b0);

         // trap if misaligned
         gen_SIGBUS_if_misaligned( EA, 16 );

         // and actually do the load
         if (mode64) {
            if (host_endness == VexEndnessBE) {
               stmt( stmt_load( res_hi,
                                mkexpr(EA), NULL/*this is a load*/) );
               stmt( stmt_load( res_lo,
                                binop(Iop_Add64, mkexpr(EA), mkU64(8) ),
                                NULL/*this is a load*/) );
	    } else {
               stmt( stmt_load( res_lo,
                                mkexpr(EA), NULL/*this is a load*/) );
               stmt( stmt_load( res_hi,
                                binop(Iop_Add64, mkexpr(EA), mkU64(8) ),
                                NULL/*this is a load*/) );
            }
         } else {
            stmt( stmt_load( res_hi,
                             binop( Iop_Add32, mkexpr(EA), mkU32(4) ),
                             NULL/*this is a load*/) );
            stmt( stmt_load( res_lo,
                             binop( Iop_Add32, mkexpr(EA), mkU32(12) ),
                             NULL/*this is a load*/) );
         }
         putIReg( rD_addr,   mkexpr(res_hi) );
         putIReg( rD_addr+1, mkexpr(res_lo) );
         break;
      }

      case 0x0B6: { // stqcx. (Store QuadWord Condition Indexd, PPC64)
         // A marginally simplified version of the stwcx. case
         IRTemp rS_hi = newTemp(ty);
         IRTemp rS_lo = newTemp(ty);
         IRTemp resSC;
         if (b0 != 1) {
            vex_printf("dis_memsync(ppc)(stqcx.,b0)\n");
            return False;
         }

         DIP("stqcx. r%u,r%u,r%u\n", rS_addr, rA_addr, rB_addr);

         // trap if misaligned
         gen_SIGBUS_if_misaligned( EA, 16 );
         // Get the data to be stored
         assign( rS_hi, getIReg(rS_addr) );
         assign( rS_lo, getIReg(rS_addr+1) );

         // Do the store, and get success/failure bit into resSC
         resSC = newTemp(Ity_I1);

         if (mode64) {
            if (host_endness == VexEndnessBE) {
               stmt( stmt_load( resSC, mkexpr(EA), mkexpr(rS_hi) ) );
               store( binop( Iop_Add64, mkexpr(EA), mkU64(8) ),
                      mkexpr(rS_lo) );
	    } else {
               stmt( stmt_load( resSC, mkexpr(EA), mkexpr(rS_lo) ) );
               store( binop( Iop_Add64, mkexpr(EA), mkU64(8) ),
                      mkexpr(rS_hi) );
	    }
         } else {
            stmt( stmt_load( resSC, binop( Iop_Add32,
                                           mkexpr(EA),
                                           mkU32(4) ),
                                           mkexpr(rS_hi) ) );
            store( binop(Iop_Add32, mkexpr(EA), mkU32(12) ), mkexpr(rS_lo) );
         }

         // Set CR0[LT GT EQ S0] = 0b000 || XER[SO]  on failure
         // Set CR0[LT GT EQ S0] = 0b001 || XER[SO]  on success
         putCR321(0, binop( Iop_Shl8,
                            unop(Iop_1Uto8, mkexpr(resSC) ),
                            mkU8(1)));
         putCR0(0, getXER_SO());
         break;
      }

      default:
         vex_printf("dis_memsync(ppc)(opc2)\n");
         return False;
      }
      break;

   default:
      vex_printf("dis_memsync(ppc)(opc1)\n");
      return False;
   }
   return True;
}



/*
  Integer Shift Instructions
*/
static Bool dis_int_shift ( UInt prefix, UInt theInstr, UInt allow_isa_3_0 )
{
   /* X-Form, XS-Form */
   UChar opc1    = ifieldOPC(theInstr);
   UChar rS_addr = ifieldRegDS(theInstr);
   UChar rA_addr = ifieldRegA(theInstr);
   UChar rB_addr = ifieldRegB(theInstr);
   UChar sh_imm  = rB_addr;
   UInt  opc2    = ifieldOPClo10(theInstr);
   UChar b1      = ifieldBIT1(theInstr);
   UChar flag_rC = ifieldBIT0(theInstr);

   IRType  ty         = mode64 ? Ity_I64 : Ity_I32;
   IRTemp  rA         = newTemp(ty);
   IRTemp  rS         = newTemp(ty);
   IRTemp  rB         = newTemp(ty);
   IRTemp  outofrange = newTemp(Ity_I1);
   IRTemp  rS_lo32    = newTemp(Ity_I32);
   IRTemp  rB_lo32    = newTemp(Ity_I32);
   IRExpr* e_tmp;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   assign( rS, getIReg(rS_addr) );
   assign( rB, getIReg(rB_addr) );
   assign( rS_lo32, mkNarrowTo32(ty, mkexpr(rS)) );
   assign( rB_lo32, mkNarrowTo32(ty, mkexpr(rB)) );
   
   if (opc1 == 0x1F) {
      switch (opc2) {
      case 0x018: { // slw (Shift Left Word, PPC32 p505)
         DIP("slw%s r%u,r%u,r%u\n", flag_rC ? ".":"",
             rA_addr, rS_addr, rB_addr);
         /* rA = rS << rB */
         /* ppc32 semantics are: 
            slw(x,y) = (x << (y & 31))         -- primary result
                       & ~((y << 26) >>s 31)   -- make result 0 
                                                  for y in 32 .. 63
         */
         e_tmp =
            binop( Iop_And32,
               binop( Iop_Shl32,
                      mkexpr(rS_lo32), 
                      unop( Iop_32to8,
                            binop(Iop_And32,
                                  mkexpr(rB_lo32), mkU32(31)))),
               unop( Iop_Not32,
                     binop( Iop_Sar32,
                            binop(Iop_Shl32, mkexpr(rB_lo32), mkU8(26)),
                            mkU8(31))) );
         assign( rA, mkWidenFrom32(ty, e_tmp, /* Signed */False) );
         break;
      }

      case 0x318: { // sraw (Shift Right Alg Word, PPC32 p506)
         IRTemp sh_amt = newTemp(Ity_I32);
         DIP("sraw%s r%u,r%u,r%u\n", flag_rC ? ".":"",
             rA_addr, rS_addr, rB_addr);
         /* JRS: my reading of the (poorly worded) PPC32 doc p506 is:
            amt = rB & 63
            rA = Sar32( rS, amt > 31 ? 31 : amt )
            XER.CA = amt > 31 ? sign-of-rS : (computation as per srawi)
         */
         assign( sh_amt, binop(Iop_And32, mkU32(0x3F),
                                          mkexpr(rB_lo32)) );
         assign( outofrange,
                 binop(Iop_CmpLT32U, mkU32(31), mkexpr(sh_amt)) );
         e_tmp = binop( Iop_Sar32, 
                        mkexpr(rS_lo32), 
                        unop( Iop_32to8, 
                              IRExpr_ITE( mkexpr(outofrange), 
                                          mkU32(31),
                                          mkexpr(sh_amt)) ) );
         assign( rA, mkWidenFrom32(ty, e_tmp, /* Signed */True) );

         /* Set CA bit */
         set_XER_CA_CA32( ty, PPCG_FLAG_OP_SRAW,
                          mkexpr(rA),
                          mkWidenFrom32(ty, mkexpr(rS_lo32), True),
                          mkWidenFrom32(ty, mkexpr(sh_amt), True ),
                          mkWidenFrom32(ty, getXER_CA_32(), True) );

         if (allow_isa_3_0)
            /* copy CA to CA32 */
            putXER_CA32( unop(Iop_32to8, getXER_CA_32()));
         break;
      }
         
      case 0x338: // srawi (Shift Right Alg Word Immediate, PPC32 p507)
         DIP("srawi%s r%u,r%u,%d\n", flag_rC ? ".":"",
             rA_addr, rS_addr, sh_imm);
         vassert(sh_imm < 32);
         if (mode64) {
            assign( rA, binop(Iop_Sar64,
                              binop(Iop_Shl64, getIReg(rS_addr),
                                               mkU8(32)),
                              mkU8(32 + sh_imm)) );
         } else {
            assign( rA, binop(Iop_Sar32, mkexpr(rS_lo32),
                                         mkU8(sh_imm)) );
         }

         /* Set CA bit */
         set_XER_CA_CA32( ty, PPCG_FLAG_OP_SRAWI,
                          mkexpr(rA),
                          mkWidenFrom32(ty, mkexpr(rS_lo32), /* Syned */True),
                          mkSzImm(ty, sh_imm),
                          mkWidenFrom32(ty, getXER_CA_32(), /* Syned */False) );

         if (allow_isa_3_0)
            /* copy CA to CA32 */
            putXER_CA32( unop(Iop_32to8, getXER_CA_32()));
         break;
      
      case 0x218: // srw (Shift Right Word, PPC32 p508)
         DIP("srw%s r%u,r%u,r%u\n", flag_rC ? ".":"",
             rA_addr, rS_addr, rB_addr);
         /* rA = rS >>u rB */
         /* ppc32 semantics are: 
            srw(x,y) = (x >>u (y & 31))        -- primary result
                       & ~((y << 26) >>s 31)   -- make result 0 
                                                  for y in 32 .. 63
         */
         e_tmp = 
            binop(
               Iop_And32,
               binop( Iop_Shr32, 
                      mkexpr(rS_lo32), 
                      unop( Iop_32to8, 
                            binop(Iop_And32, mkexpr(rB_lo32),
                                             mkU32(31)))),
               unop( Iop_Not32, 
                     binop( Iop_Sar32, 
                            binop(Iop_Shl32, mkexpr(rB_lo32),
                                             mkU8(26)), 
                            mkU8(31))));
         assign( rA, mkWidenFrom32(ty, e_tmp, /* Signed */False) );
         break;


      /* 64bit Shifts */
      case 0x01B: // sld (Shift Left DWord, PPC64 p568)
         DIP("sld%s r%u,r%u,r%u\n",
             flag_rC ? ".":"", rA_addr, rS_addr, rB_addr);
         /* rA = rS << rB */
         /* ppc64 semantics are: 
            slw(x,y) = (x << (y & 63))         -- primary result
                       & ~((y << 57) >>s 63)   -- make result 0 
                                                  for y in 64 .. 
         */
         assign( rA,
            binop(
               Iop_And64,
               binop( Iop_Shl64,
                      mkexpr(rS), 
                      unop( Iop_64to8, 
                            binop(Iop_And64, mkexpr(rB), mkU64(63)))),
               unop( Iop_Not64,
                     binop( Iop_Sar64,
                            binop(Iop_Shl64, mkexpr(rB), mkU8(57)), 
                            mkU8(63)))) );
         break;
      
      case 0x31A: { // srad (Shift Right Alg DWord, PPC64 p570)
         IRTemp sh_amt = newTemp(Ity_I64);
         DIP("srad%s r%u,r%u,r%u\n",
             flag_rC ? ".":"", rA_addr, rS_addr, rB_addr);
         /* amt = rB & 127
            rA = Sar64( rS, amt > 63 ? 63 : amt )
            XER.CA = amt > 63 ? sign-of-rS : (computation as per srawi)
         */
         assign( sh_amt, binop(Iop_And64, mkU64(0x7F), mkexpr(rB)) );
         assign( outofrange,
                 binop(Iop_CmpLT64U, mkU64(63), mkexpr(sh_amt)) );
         assign( rA,
                 binop( Iop_Sar64, 
                        mkexpr(rS), 
                        unop( Iop_64to8, 
                              IRExpr_ITE( mkexpr(outofrange), 
                                          mkU64(63),
                                          mkexpr(sh_amt)) ))
               );
         /* Set CA bit */
         set_XER_CA_CA32( ty, PPCG_FLAG_OP_SRAD,
                          mkexpr(rA), mkexpr(rS), mkexpr(sh_amt),
                          mkWidenFrom32(ty, getXER_CA_32(), /* Syned */False) );

         if (allow_isa_3_0)
            /* copy CA to CA32 */
            putXER_CA32( unop(Iop_32to8, getXER_CA_32()));
         break;
      }

      case 0x33A: case 0x33B: // sradi (Shr Alg DWord Imm, PPC64 p571)
         sh_imm |= b1<<5;
         vassert(sh_imm < 64);
         DIP("sradi%s r%u,r%u,%u\n",
             flag_rC ? ".":"", rA_addr, rS_addr, sh_imm);
         assign( rA, binop(Iop_Sar64, getIReg(rS_addr), mkU8(sh_imm)) );

         /* Set CA bit */
         set_XER_CA_CA32( ty, PPCG_FLAG_OP_SRADI,
                          mkexpr(rA),
                          getIReg(rS_addr),
                          mkU64(sh_imm),
                          mkWidenFrom32(ty, getXER_CA_32(), /* Syned */False) );

         if (allow_isa_3_0)
            /* copy CA to CA32 */
            putXER_CA32( unop(Iop_32to8, getXER_CA_32()));
         break;

      case 0x21B: // srd (Shift Right DWord, PPC64 p574)
         DIP("srd%s r%u,r%u,r%u\n",
             flag_rC ? ".":"", rA_addr, rS_addr, rB_addr);
         /* rA = rS >>u rB */
         /* ppc semantics are: 
            srw(x,y) = (x >>u (y & 63))        -- primary result
                       & ~((y << 57) >>s 63)   -- make result 0 
                                                  for y in 64 .. 127
         */
         assign( rA,
            binop(
               Iop_And64,
               binop( Iop_Shr64, 
                      mkexpr(rS), 
                      unop( Iop_64to8, 
                            binop(Iop_And64, mkexpr(rB), mkU64(63)))),
               unop( Iop_Not64, 
                     binop( Iop_Sar64, 
                            binop(Iop_Shl64, mkexpr(rB), mkU8(57)), 
                            mkU8(63)))) );
         break;
     
      default:
         vex_printf("dis_int_shift(ppc)(opc2)\n");
         return False;
      }
   } else {
      vex_printf("dis_int_shift(ppc)(opc1)\n");
      return False;
   }

   putIReg( rA_addr, mkexpr(rA) );
   
   if (flag_rC) {
      set_CR0( mkexpr(rA) );
   }
   return True;
}



/*
  Integer Load/Store Reverse Instructions
*/
/* Generates code to swap the byte order in an Ity_I32. */
static IRExpr* /* :: Ity_I32 */ gen_byterev32 ( IRTemp t )
{
   vassert(typeOfIRTemp(irsb->tyenv, t) == Ity_I32);
   return unop(Iop_Reverse8sIn32_x1, mkexpr(t));
}

/* Generates code to swap the byte order in the lower half of an Ity_I32,
   and zeroes the upper half. */
static IRExpr* /* :: Ity_I32 */ gen_byterev16 ( IRTemp t )
{
   vassert(typeOfIRTemp(irsb->tyenv, t) == Ity_I32);
   return
      binop(Iop_Or32,
         binop(Iop_And32, binop(Iop_Shl32, mkexpr(t), mkU8(8)),
                          mkU32(0x0000FF00)),
         binop(Iop_And32, binop(Iop_Shr32, mkexpr(t), mkU8(8)),
                          mkU32(0x000000FF))
      );
}

static Bool dis_int_ldst_rev ( UInt prefix, UInt theInstr )
{
   /* X-Form */
   UChar opc1    = ifieldOPC(theInstr);
   UChar rD_addr = ifieldRegDS(theInstr);
   UChar rS_addr = rD_addr;
   UChar rA_addr = ifieldRegA(theInstr);
   UChar rB_addr = ifieldRegB(theInstr);
   UInt  opc2    = ifieldOPClo10(theInstr);
   UChar b0      = ifieldBIT0(theInstr);

   IRType ty = mode64 ? Ity_I64 : Ity_I32;
   IRTemp EA = newTemp(ty);
   IRTemp w1 = newTemp(Ity_I32);
   IRTemp w2 = newTemp(Ity_I32);

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   if (opc1 != 0x1F || b0 != 0) {
      vex_printf("dis_int_ldst_rev(ppc)(opc1|b0)\n");
      return False;
   }

   assign( EA, ea_rAor0_idxd( rA_addr, rB_addr ) );
   
   switch (opc2) {

      case 0x316: // lhbrx (Load Halfword Byte-Reverse Indexed, PPC32 p449)
         DIP("lhbrx r%u,r%u,r%u\n", rD_addr, rA_addr, rB_addr);
         assign( w1, unop(Iop_16Uto32, load(Ity_I16, mkexpr(EA))) );
         assign( w2, gen_byterev16(w1) );
         putIReg( rD_addr, mkWidenFrom32(ty, mkexpr(w2),
                                         /* Signed */False) );
         break;

      case 0x216: // lwbrx (Load Word Byte-Reverse Indexed, PPC32 p459)
         DIP("lwbrx r%u,r%u,r%u\n", rD_addr, rA_addr, rB_addr);
         assign( w1, load(Ity_I32, mkexpr(EA)) );
         assign( w2, gen_byterev32(w1) );
         putIReg( rD_addr, mkWidenFrom32(ty, mkexpr(w2),
                                         /* Signed */False) );
         break;

      case 0x214: // ldbrx (Load Doubleword Byte-Reverse Indexed)
      {
         /* Caller makes sure we are only called in mode64. */

         /* If we supported swapping LE/BE loads in the backend then we could
            just load the value with the bytes reversed by doing a BE load
            on an LE machine and a LE load on a BE machine.

               IRTemp dw1 = newTemp(Ity_I64);
               if (host_endness == VexEndnessBE)
                  assign( dw1, IRExpr_Load(Iend_LE, Ity_I64, mkexpr(EA)));
               else
                  assign( dw1, IRExpr_Load(Iend_BE, Ity_I64, mkexpr(EA)));
               putIReg( rD_addr, mkexpr(dw1) );

            But since we currently don't we load the value as is and then
            switch it around with Iop_Reverse8sIn64_x1. */

         IRTemp dw1 = newTemp(Ity_I64);
         IRTemp dw2 = newTemp(Ity_I64);
         DIP("ldbrx r%u,r%u,r%u\n", rD_addr, rA_addr, rB_addr);
         assign( dw1, load(Ity_I64, mkexpr(EA)) );
         assign( dw2, unop(Iop_Reverse8sIn64_x1, mkexpr(dw1)) );
         putIReg( rD_addr, mkexpr(dw2) );
         break;
      }

      case 0x396: // sthbrx (Store Half Word Byte-Reverse Indexed, PPC32 p523)
         DIP("sthbrx r%u,r%u,r%u\n", rS_addr, rA_addr, rB_addr);
         assign( w1, mkNarrowTo32(ty, getIReg(rS_addr)) );
         store( mkexpr(EA), unop(Iop_32to16, gen_byterev16(w1)) );
         break;
      
      case 0x296: // stwbrx (Store Word Byte-Reverse Indxd, PPC32 p531)
         DIP("stwbrx r%u,r%u,r%u\n", rS_addr, rA_addr, rB_addr);
         assign( w1, mkNarrowTo32(ty, getIReg(rS_addr)) );
         store( mkexpr(EA), gen_byterev32(w1) );
         break;

      case 0x294: // stdbrx (Store Doubleword Byte-Reverse Indexed)
      {
         IRTemp lo = newTemp(Ity_I32);
         IRTemp hi = newTemp(Ity_I32);
         IRTemp rS = newTemp(Ity_I64);
         assign( rS, getIReg( rS_addr ) );
         DIP("stdbrx r%u,r%u,r%u\n", rS_addr, rA_addr, rB_addr);
         assign(lo, unop(Iop_64HIto32, mkexpr(rS)));
         assign(hi, unop(Iop_64to32, mkexpr(rS)));
         store( mkexpr( EA ),
                binop( Iop_32HLto64, gen_byterev32( hi ),
                       gen_byterev32( lo ) ) );
         break;
      }

      default:
         vex_printf("dis_int_ldst_rev(ppc)(opc2)\n");
         return False;
   }
   return True;
}



/*
  Processor Control Instructions
*/
static Bool dis_proc_ctl ( const VexAbiInfo* vbi, UInt prefix, UInt theInstr )
{
   UChar opc1     = ifieldOPC(theInstr);
   
   /* X-Form */
   UChar crfD     = toUChar( IFIELD( theInstr, 23, 3 ) );
   UChar b21to22  = toUChar( IFIELD( theInstr, 21, 2 ) );
   UChar rD_addr  = ifieldRegDS(theInstr);
   UInt  b11to20  = IFIELD( theInstr, 11, 10 );

   /* XFX-Form */
   UChar rS_addr  = rD_addr;
   UInt  SPR      = b11to20;
   UInt  TBR      = b11to20;
   UChar b20      = toUChar( IFIELD( theInstr, 20, 1 ) );
   UInt  CRM      = IFIELD( theInstr, 12, 8 );
   UChar b11      = toUChar( IFIELD( theInstr, 11, 1 ) );

   UInt  opc2     = ifieldOPClo10(theInstr);
   UChar b0       = ifieldBIT0(theInstr);

   IRType ty = mode64 ? Ity_I64 : Ity_I32;
   IRTemp rS = newTemp(ty);

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   assign( rS, getIReg(rS_addr) );

   /* Reorder SPR field as per PPC32 p470 */
   SPR = ((SPR & 0x1F) << 5) | ((SPR >> 5) & 0x1F);
   /* Reorder TBR field as per PPC32 p475 */
   TBR = ((TBR & 31) << 5) | ((TBR >> 5) & 31);
   
   /* b0 = 0, inst is treated as floating point inst for reservation purposes
    * b0 = 1, inst is treated as vector inst for reservation purposes
    */
   if (opc1 != 0x1F) {
      vex_printf("dis_proc_ctl(ppc)(opc1|b%d)\n", b0);
      return False;
   }
   
   switch (opc2) {
   /* X-Form */
   case 0x200: { // mcrxr (Move to Cond Register from XER, PPC32 p466)
      if (b21to22 != 0 || b11to20 != 0) {
         vex_printf("dis_proc_ctl(ppc)(mcrxr,b21to22|b11to20)\n");
         return False;
      }
      DIP("mcrxr crf%d\n", crfD);
      /* Move XER[0-3] (the top 4 bits of XER) to CR[crfD] */
      putGST_field( PPC_GST_CR,
                    getGST_field( PPC_GST_XER, 7 ),
                    crfD );

      // Clear XER[0-3]
      putXER_SO( mkU8(0) );
      putXER_OV( mkU8(0) );
      putXER_CA( mkU8(0) );
      break;
   }
      
   case 0x240: { // mcrxrx (Move to Cond Register from XER)
      IRTemp  OV   = newTemp(Ity_I32);
      IRTemp  CA   = newTemp(Ity_I32);
      IRTemp  OV32 = newTemp(Ity_I32);
      IRTemp  CA32 = newTemp(Ity_I32);
      IRTemp  tmp  = newTemp(Ity_I32);

      if (b21to22 != 0 || b11to20 != 0) {
         vex_printf("dis_proc_ctl(ppc)(mcrxrx,b21to22|b11to20)\n");
         return False;
      }
      DIP("mcrxrx crf%d\n", crfD);
      /* Move OV, OV32, CA, CA32 to condition register field BF */
      assign( OV, binop( Iop_Shl32, getXER_OV_32(), mkU8( 3 ) ));
      assign( CA, binop( Iop_Shl32, getXER_CA_32(), mkU8( 1 ) ));
      assign( OV32, binop( Iop_Shl32, getXER_OV32_32(), mkU8( 2 ) ));
      assign( CA32, getXER_CA32_32() );

      /* Put [OV | OV32 | CA | CA32] into the condition code register */
      assign( tmp,
              binop( Iop_Or32,
                     binop( Iop_Or32, mkexpr ( OV ),  mkexpr ( OV32 ) ),
                     binop( Iop_Or32, mkexpr ( CA ),  mkexpr ( CA32 ) )
                 ) );

      putGST_field( PPC_GST_CR,  mkexpr( tmp ), crfD );
      break;
   }

   case 0x013: 
      // b11to20==0:      mfcr (Move from Cond Register, PPC32 p467)
      // b20==1 & b11==0: mfocrf (Move from One CR Field)
      // However it seems that the 'mfcr' behaviour is an acceptable
      // implementation of mfocr (from the 2.02 arch spec)
      if (b11to20 == 0) {
         DIP("mfcr r%u\n", rD_addr);
         putIReg( rD_addr, mkWidenFrom32(ty, getGST( PPC_GST_CR ),
                                         /* Signed */False) );
         break;
      }
      if (b20 == 1 && b11 == 0) {
         DIP("mfocrf r%u,%u\n", rD_addr, CRM);
         putIReg( rD_addr, mkWidenFrom32(ty, getGST( PPC_GST_CR ),
                                         /* Signed */False) );
         break;
      }
      /* not decodable */
      return False;

   /* XFX-Form */
   case 0x153: // mfspr (Move from Special-Purpose Register, PPC32 p470)
      
      switch (SPR) {  // Choose a register...
      case 0x1:
         DIP("mfxer r%u\n", rD_addr);
         putIReg( rD_addr, mkWidenFrom32(ty, getGST( PPC_GST_XER ),
                                         /* Signed */False) );
         break;
      case 0x3:  // 131
         DIP("mfspr r%u (DSCR)\n", rD_addr);
         putIReg( rD_addr, getGST( PPC_GST_DSCR) );
         break;
      case 0x8:
         DIP("mflr r%u\n", rD_addr);
         putIReg( rD_addr, getGST( PPC_GST_LR ) ); 
         break;
      case 0x9:
         DIP("mfctr r%u\n", rD_addr);
         putIReg( rD_addr, getGST( PPC_GST_CTR ) ); 
         break;
      case 0x80:  // 128
         DIP("mfspr r%u (TFHAR)\n", rD_addr);
         putIReg( rD_addr, getGST( PPC_GST_TFHAR) );
         break;
      case 0x81:  // 129
         DIP("mfspr r%u (TFIAR)\n", rD_addr);
         putIReg( rD_addr, getGST( PPC_GST_TFIAR) );
         break;
      case 0x82:  // 130
         DIP("mfspr r%u (TEXASR)\n", rD_addr);
         putIReg( rD_addr, getGST( PPC_GST_TEXASR) );
         break;
      case 0x83:  // 131
         DIP("mfspr r%u (TEXASRU)\n", rD_addr);
         putIReg( rD_addr, getGST( PPC_GST_TEXASRU) );
         break;
      case 0x9F:  // 159
         DIP("mfspr r%u (PSPB)\n", rD_addr);
         putIReg( rD_addr, getGST( PPC_GST_PSPB) );
         break;
      case 0x380:  // 896
         DIP("mfspr r%u (PPR)\n", rD_addr);
         putIReg( rD_addr, getGST( PPC_GST_PPR) );
         break;
      case 0x382:  // 898
         DIP("mfspr r%u (PPR)32\n", rD_addr);
         putIReg( rD_addr, getGST( PPC_GST_PPR32) );
         break;
      case 0x100: 
         DIP("mfvrsave r%u\n", rD_addr);
         putIReg( rD_addr, mkWidenFrom32(ty, getGST( PPC_GST_VRSAVE ),
                                         /* Signed */False) );
         break;

      case 0x103:
         DIP("mfspr r%u, SPRG3(readonly)\n", rD_addr);
         putIReg( rD_addr, getGST( PPC_GST_SPRG3_RO ) );
         break;

      case 268 /* 0x10C  TB  - 64 bit time base register */:
      {
         IRTemp   val  = newTemp(Ity_I64);
         IRExpr** args = mkIRExprVec_0();
         IRDirty* d    = unsafeIRDirty_1_N(
                                           val,
                                           0/*regparms*/,
                                           "ppcg_dirtyhelper_MFTB",
                                           fnptr_to_fnentry(vbi,
                                                            &ppcg_dirtyhelper_MFTB),
                                           args );
         /* execute the dirty call, dumping the result in val. */
         stmt( IRStmt_Dirty(d) );
         putIReg( rD_addr, (mode64) ? mkexpr(val) :
                                      unop(Iop_64to32, mkexpr(val)) );

         break;
      }
      case 269 /* 0x10D  TBU - upper 32-bits of time base register */:
      {
         DIP("mfspr r%u,%u", rD_addr, SPR);
         IRTemp   val  = newTemp(Ity_I64);
         IRExpr** args = mkIRExprVec_0();
         IRDirty* d    = unsafeIRDirty_1_N(
                                           val,
                                           0/*regparms*/,
                                           "ppcg_dirtyhelper_MFTB",
                                           fnptr_to_fnentry(vbi,
                                                            &ppcg_dirtyhelper_MFTB),
                                           args );
         /* execute the dirty call, dumping the result in val. */
         stmt( IRStmt_Dirty(d) );
         putIReg( rD_addr,
                  mkWidenFrom32(ty, unop(Iop_64HIto32, mkexpr(val)),
                                /* Signed */False) );
         break;
      }
      case 284 /* 0x1  TBL - lower 32-bits of time base register */:
      {
         DIP("mfspr r%u,%u", rD_addr, SPR);
         IRTemp   val  = newTemp(Ity_I64);
         IRExpr** args = mkIRExprVec_0();
         IRDirty* d    = unsafeIRDirty_1_N(
                                           val,
                                           0/*regparms*/,
                                           "ppcg_dirtyhelper_MFTB",
                                           fnptr_to_fnentry(vbi,
                                                            &ppcg_dirtyhelper_MFTB),
                                           args );
         /* execute the dirty call, dumping the result in val. */
         stmt( IRStmt_Dirty(d) );
         putIReg( rD_addr,
                  mkWidenFrom32(ty, unop(Iop_64to32, mkexpr(val)),
                                /* Signed */False) );
         break;
      }

      /* Again, runs natively on PPC7400 (7447, really).  Not
         bothering with a feature test. */
      case 287: /* 0x11F */ {
         IRTemp   val  = newTemp(Ity_I32);
         IRExpr** args = mkIRExprVec_0();
         IRDirty* d    = unsafeIRDirty_1_N(
                            val,
                            0/*regparms*/,
                            "ppc32g_dirtyhelper_MFSPR_287",
                            fnptr_to_fnentry
                               (vbi, &ppc32g_dirtyhelper_MFSPR_287),
                            args
                         );
         /* execute the dirty call, dumping the result in val. */
         stmt( IRStmt_Dirty(d) );
         putIReg( rD_addr,
                  mkWidenFrom32(ty, mkexpr(val), False/*unsigned*/) );
         DIP("mfspr r%u,%u", rD_addr, SPR);
         break;
      }

      default:
         vex_printf("dis_proc_ctl(ppc)(mfspr,SPR)(0x%x)\n", SPR);
         return False;
      }
      break;
      
   case 0x173: { // mftb (Move from Time Base, PPC32 p475)
      IRTemp   val  = newTemp(Ity_I64);
      IRExpr** args = mkIRExprVec_0();
      IRDirty* d    = unsafeIRDirty_1_N(
                              val, 
                              0/*regparms*/, 
                              "ppcg_dirtyhelper_MFTB", 
                              fnptr_to_fnentry(vbi, &ppcg_dirtyhelper_MFTB), 
                              args );
      /* execute the dirty call, dumping the result in val. */
      stmt( IRStmt_Dirty(d) );

      switch (TBR) {
      case 269: 
         DIP("mftbu r%u", rD_addr);
         putIReg( rD_addr,
                  mkWidenFrom32(ty, unop(Iop_64HIto32, mkexpr(val)),
                                /* Signed */False) );
         break;
      case 268: 
         DIP("mftb r%u", rD_addr);
         putIReg( rD_addr, (mode64) ? mkexpr(val) :
                                      unop(Iop_64to32, mkexpr(val)) );
         break;
      case 284:
         DIP("mftbl r%u", rD_addr);
         putIReg( rD_addr,
                  mkWidenFrom32(ty, unop(Iop_64to32, mkexpr(val)),
                                /* Signed */False) );
         break;
      default:
         return False; /* illegal instruction */
      }
      break;
   }

   case 0x090: { 
      // b20==0: mtcrf (Move to Cond Register Fields, PPC32 p477)
      // b20==1: mtocrf (Move to One Cond Reg Field)
      Int   cr;
      UChar shft;
      if (b11 != 0)
         return False;
      if (b20 == 1) {
         /* ppc64 v2.02 spec says mtocrf gives undefined outcome if >
            1 field is written.  It seems more robust to decline to
            decode the insn if so. */
         switch (CRM) {
            case 0x01: case 0x02: case 0x04: case 0x08:
            case 0x10: case 0x20: case 0x40: case 0x80:
               break;
            default: 
               return False; 
         }
      }
      DIP("%s 0x%x,r%u\n", b20==1 ? "mtocrf" : "mtcrf", 
                           CRM, rS_addr);
      /* Write to each field specified by CRM */
      for (cr = 0; cr < 8; cr++) {
         if ((CRM & (1 << (7-cr))) == 0)
            continue;
         shft = 4*(7-cr);
         putGST_field( PPC_GST_CR,
                       binop(Iop_Shr32,
                             mkNarrowTo32(ty, mkexpr(rS)),
                             mkU8(shft)), cr );
      }
      break;
   }

   case 0x1D3: // mtspr (Move to Special-Purpose Register, PPC32 p483)
      
      switch (SPR) {  // Choose a register...
      case 0x1:
         DIP("mtxer r%u\n", rS_addr);
         putGST( PPC_GST_XER, mkNarrowTo32(ty, mkexpr(rS)) );
         break;
      case 0x3:
         DIP("mtspr r%u (DSCR)\n", rS_addr);
         putGST( PPC_GST_DSCR, mkexpr(rS) );
         break;
      case 0x8:
         DIP("mtlr r%u\n", rS_addr);
         putGST( PPC_GST_LR, mkexpr(rS) ); 
         break;
      case 0x9:
         DIP("mtctr r%u\n", rS_addr);
         putGST( PPC_GST_CTR, mkexpr(rS) ); 
         break;
      case 0x100:
         DIP("mtvrsave r%u\n", rS_addr);
         putGST( PPC_GST_VRSAVE, mkNarrowTo32(ty, mkexpr(rS)) );
         break;
      case 0x80:  // 128
         DIP("mtspr r%u (TFHAR)\n", rS_addr);
         putGST( PPC_GST_TFHAR, mkexpr(rS) );
         break;
      case 0x81:  // 129
         DIP("mtspr r%u (TFIAR)\n", rS_addr);
         putGST( PPC_GST_TFIAR, mkexpr(rS) );
         break;
      case 0x82:  // 130
         DIP("mtspr r%u (TEXASR)\n", rS_addr);
         putGST( PPC_GST_TEXASR, mkexpr(rS) );
         break;
      case 0x9F:  // 159
         DIP("mtspr r%u (PSPB)\n", rS_addr);
         putGST( PPC_GST_PSPB, mkexpr(rS) );
         break;
      case 0x380:  // 896
         DIP("mtspr r%u (PPR)\n", rS_addr);
         putGST( PPC_GST_PPR, mkexpr(rS) );
         break;
      case 0x382:  // 898
         DIP("mtspr r%u (PPR32)\n", rS_addr);
         putGST( PPC_GST_PPR32, mkexpr(rS) );
         break;
      default:
         vex_printf("dis_proc_ctl(ppc)(mtspr,SPR)(%u)\n", SPR);
         return False;
      }
      break;

   case 0x33:                // mfvsrd
   {
      UChar XS = ifieldRegXS( theInstr );
      UChar rA_addr = ifieldRegA(theInstr);
      IRExpr * high64;
      IRTemp vS = newTemp( Ity_V128 );
      DIP("mfvsrd r%u,vsr%d\n", rA_addr, XS);

      /*  XS = SX || S
       *  For SX=0, mfvsrd is treated as a Floating-Point
       *            instruction in terms of resource availability.
       *  For SX=1, mfvsrd is treated as a Vector instruction in
       *            terms of resource availability.
       * FIXME: NEED TO FIGURE OUT HOW TO IMPLEMENT THE RESOURCE AVAILABILITY PART
       */
      assign( vS, getVSReg( XS ) );
      high64 = unop( Iop_V128HIto64, mkexpr( vS ) );
      putIReg( rA_addr, (mode64) ? high64 :
      unop( Iop_64to32, high64 ) );
      break;
   }

   case 0x73:                // mfvsrwz
   {
      UChar XS = ifieldRegXS( theInstr );
      UChar rA_addr = ifieldRegA(theInstr);
      IRExpr * high64;
      IRTemp vS = newTemp( Ity_V128 );
      DIP("mfvsrwz r%u,vsr%d\n", rA_addr, XS);
      /*  XS = SX || S
       *  For SX=0, mfvsrwz is treated as a Floating-Point
       *            instruction in terms of resource availability.
       *  For SX=1, mfvsrwz is treated as a Vector instruction in
       *            terms of resource availability.
       * FIXME: NEED TO FIGURE OUT HOW TO IMPLEMENT THE RESOURCE AVAILABILITY PART
       */

      assign( vS, getVSReg( XS ) );
      high64 = unop( Iop_V128HIto64, mkexpr( vS ) );
      /* move value to the destination setting the upper 32-bits to zero */
      putIReg( rA_addr, (mode64) ?
                                  binop( Iop_And64, high64, mkU64( 0xFFFFFFFF ) ) :
                                  unop(  Iop_64to32,
                                         binop( Iop_And64, high64, mkU64( 0xFFFFFFFF ) ) ) );
      break;
   }

   case 0xB3:                // mtvsrd
   {
      UChar XT = ifieldRegXT( theInstr );
      UChar rA_addr = ifieldRegA(theInstr);
      IRTemp rA = newTemp(ty);
      DIP("mtvsrd vsr%d,r%u\n", XT, rA_addr);
      /*  XS = SX || S
       *  For SX=0, mfvsrd is treated as a Floating-Point
       *            instruction in terms of resource availability.
       *  For SX=1, mfvsrd is treated as a Vector instruction in
       *            terms of resource availability.
       * FIXME: NEED TO FIGURE OUT HOW TO IMPLEMENT THE RESOURCE AVAILABILITY PART
       */
      assign( rA, getIReg(rA_addr) );

      if (mode64)
         putVSReg( XT, binop( Iop_64HLtoV128, mkexpr( rA ), mkU64( 0 ) ) );
      else
         putVSReg( XT, binop( Iop_64HLtoV128,
                              binop( Iop_32HLto64,
                                     mkU32( 0 ),
                                     mkexpr( rA ) ),
                                     mkU64( 0 ) ) );
      break;
   }

   case 0xD3:                // mtvsrwa
   {
      UChar XT = ifieldRegXT( theInstr );
      UChar rA_addr = ifieldRegA(theInstr);
      IRTemp rA = newTemp( Ity_I32 );
      DIP("mtvsrwa vsr%d,r%u\n", XT, rA_addr);
      /*  XS = SX || S
       *  For SX=0, mtvsrwa is treated as a Floating-Point
       *            instruction in terms of resource availability.
       *  For SX=1, mtvsrwa is treated as a Vector instruction in
       *            terms of resource availability.
       * FIXME: NEED TO FIGURE OUT HOW TO IMPLEMENT THE RESOURCE AVAILABILITY PART
       */
      if (mode64)
         assign( rA, unop( Iop_64to32, getIReg( rA_addr ) ) );
      else
         assign( rA, getIReg(rA_addr) );

      putVSReg( XT, binop( Iop_64HLtoV128,
                           unop( Iop_32Sto64, mkexpr( rA ) ),
                           mkU64( 0 ) ) );
      break;
   }

   case 0xF3:                // mtvsrwz
      {
         UChar XT = ifieldRegXT( theInstr );
         UChar rA_addr = ifieldRegA(theInstr);
         IRTemp rA = newTemp( Ity_I32 );
         DIP("mtvsrwz vsr%d,r%u\n", rA_addr, XT);
         /*  XS = SX || S
          *  For SX=0, mtvsrwz is treated as a Floating-Point
          *            instruction in terms of resource availability.
          *  For SX=1, mtvsrwz is treated as a Vector instruction in
          *            terms of resource availability.
          * FIXME: NEED TO FIGURE OUT HOW TO IMPLEMENT THE RESOURCE AVAILABILITY PART
          */
         if (mode64)
             assign( rA, unop( Iop_64to32, getIReg( rA_addr ) ) );
         else
             assign( rA, getIReg(rA_addr) );

         putVSReg( XT, binop( Iop_64HLtoV128,
                              binop( Iop_32HLto64, mkU32( 0 ), mkexpr ( rA ) ),
                              mkU64( 0 ) ) );
         break;
      }

   default:
      vex_printf("dis_proc_ctl(ppc)(opc2)\n");
      return False;
   }
   return True;
}


/*
  Cache Management Instructions
*/
static Bool dis_cache_manage ( UInt prefix, UInt theInstr,
                               DisResult*   dres,
                               const VexArchInfo* guest_archinfo )
{
   /* X-Form */
   UChar opc1    = ifieldOPC(theInstr);
   UChar b21to25 = ifieldRegDS(theInstr);
   UChar rA_addr = ifieldRegA(theInstr);
   UChar rB_addr = ifieldRegB(theInstr);
   UInt  opc2    = ifieldOPClo10(theInstr);
   UChar b0      = ifieldBIT0(theInstr);
   UInt  lineszB = guest_archinfo->ppc_icache_line_szB;
   Bool  is_dcbzl = False;

   IRType ty     = mode64 ? Ity_I64 : Ity_I32;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   // Check for valid hint values for dcbt and dcbtst as currently described in
   // ISA 2.07.  If valid, then we simply set b21to25 to zero since we have no
   // means of modeling the hint anyway.
   if (opc1 == 0x1F && ((opc2 == 0x116) || (opc2 == 0xF6))) {
      if (b21to25 == 0x10 || b21to25 < 0x10)
         b21to25 = 0;
   }
   if (opc1 == 0x1F && opc2 == 0x116 && b21to25 == 0x11)
      b21to25 = 0;

   if (opc1 == 0x1F && opc2 == 0x3F6) { // dcbz
      if (b21to25 == 1) {
         is_dcbzl = True;
         b21to25 = 0;
         if (!(guest_archinfo->ppc_dcbzl_szB)) {
            vex_printf("dis_cache_manage(ppc)(dcbzl not supported by host)\n");
            return False;
         }
      }
   }

   if (opc1 != 0x1F || b0 != 0) {
      if (0) vex_printf("dis_cache_manage %d %d\n", 
                        opc1, b0);
      vex_printf("dis_cache_manage(ppc)(opc1|b0)\n");
      return False;
   }

   /* stay sane .. */
   vassert(lineszB == 16 || lineszB == 32 || lineszB == 64 || lineszB == 128);
   
   switch (opc2) {
//zz    case 0x2F6: // dcba (Data Cache Block Allocate, PPC32 p380)
//zz       vassert(0); /* AWAITING TEST CASE */
//zz       DIP("dcba r%u,r%u\n", rA_addr, rB_addr);
//zz       if (0) vex_printf("vex ppc->IR: kludged dcba\n");
//zz       break;
      
   case 0x056: // dcbf (Data Cache Block Flush, PPC32 p382)
      DIP("dcbf r%u,r%u\n", rA_addr, rB_addr);
      /* nop as far as vex is concerned */
      break;
      
   case 0x036: // dcbst (Data Cache Block Store, PPC32 p384)
      DIP("dcbst r%u,r%u\n", rA_addr, rB_addr);
      /* nop as far as vex is concerned */
      break;

   case 0x116: // dcbt (Data Cache Block Touch, PPC32 p385)
      DIP("dcbt r%u,r%u\n", rA_addr, rB_addr);
      /* nop as far as vex is concerned */
      break;
      
   case 0x0F6: // dcbtst (Data Cache Block Touch for Store, PPC32 p386)
      DIP("dcbtst r%u,r%u\n", rA_addr, rB_addr);
      /* nop as far as vex is concerned */
      break;
      
   case 0x3F6: { // dcbz (Data Cache Block Clear to Zero, PPC32 p387)
                 // dcbzl (Data Cache Block Clear to Zero Long, bug#135264)
      /* Clear all bytes in cache block at (rA|0) + rB. */
      IRTemp  EA   = newTemp(ty);
      IRTemp  addr = newTemp(ty);
      IRExpr* irx_addr;
      UInt    i;
      UInt clearszB;
      if (is_dcbzl) {
          clearszB = guest_archinfo->ppc_dcbzl_szB;
          DIP("dcbzl r%u,r%u\n", rA_addr, rB_addr);
      }
      else {
          clearszB = guest_archinfo->ppc_dcbz_szB;
          DIP("dcbz r%u,r%u\n", rA_addr, rB_addr);
      }

      assign( EA, ea_rAor0_idxd(rA_addr, rB_addr) );

      if (mode64) {
         /* Round EA down to the start of the containing block. */
         assign( addr, binop( Iop_And64,
                              mkexpr(EA),
                              mkU64( ~((ULong)clearszB-1) )) );
         
         for (i = 0; i < clearszB / 8; i++) {
            irx_addr = binop( Iop_Add64, mkexpr(addr), mkU64(i*8) );
            store( irx_addr, mkU64(0) );
         }
      } else {
         /* Round EA down to the start of the containing block. */
         assign( addr, binop( Iop_And32,
                              mkexpr(EA),
                              mkU32( ~(clearszB-1) )) );
         
         for (i = 0; i < clearszB / 4; i++) {
            irx_addr = binop( Iop_Add32, mkexpr(addr), mkU32(i*4) );
            store( irx_addr, mkU32(0) );
         }
      }
      break;
   }

   case 0x3D6: { 
      // icbi (Instruction Cache Block Invalidate, PPC32 p431)
      /* Invalidate all translations containing code from the cache
         block at (rA|0) + rB. */
      IRTemp EA   = newTemp(ty);
      IRTemp addr = newTemp(ty);
      DIP("icbi r%u,r%u\n", rA_addr, rB_addr);
      assign( EA, ea_rAor0_idxd(rA_addr, rB_addr) );

      /* Round EA down to the start of the containing block. */
      assign( addr, binop( mkSzOp(ty, Iop_And8),
                           mkexpr(EA),
                           mkSzImm(ty, ~(((ULong)lineszB)-1) )) );
      putGST( PPC_GST_CMSTART, mkexpr(addr) );
      putGST( PPC_GST_CMLEN, mkSzImm(ty, lineszB) );

      /* be paranoid ... */
      stmt( IRStmt_MBE(Imbe_Fence) );

      putGST( PPC_GST_CIA, mkSzImm(ty, nextInsnAddr()));
      dres->jk_StopHere = Ijk_InvalICache;
      dres->whatNext    = Dis_StopHere;
      break;
   }

   default:
      vex_printf("dis_cache_manage(ppc)(opc2)\n");
      return False;
   }
   return True;
}


/*------------------------------------------------------------*/
/*--- Floating Point Helpers                               ---*/
/*------------------------------------------------------------*/

/* --------- Synthesise a 2-bit FPU rounding mode. --------- */
/* Produces a value in 0 .. 3, which is encoded as per the type
   IRRoundingMode.  PPCRoundingMode encoding is different to
   IRRoundingMode, so need to map it.
*/

static IRExpr* /* :: Ity_I32 */ set_round_to_Oddmode ( void )
{
/* PPC/ valgrind have two-bits to designate the rounding mode.
   ISA 3.0 adds instructions than can use a round to odd mode
   but did not change the number of bits for the rm.  Basically,
   they added two instructions that only differ by the rounding
   mode the operation uses.  In essesce, they encoded the rm
   in the name.  In order to avoid having to create Iops, that
   encode the rm in th name, we will "expand" the definition of
   the rounding mode bits.  We will just pass the rm and then
   map the to odd mode to the appropriate PPCFpOp name that
   will tell us which instruction to map to.

   rounding mode | PPC | IR
   ------------------------
   to nearest    | 000  | 00
   to zero       | 001  | 11
   to +infinity  | 010  | 10
   to -infinity  | 011  | 01
   to odd        | 1xx  | xx
*/
   return mkU32(8);
}

static IRExpr* /* :: Ity_I32 */ get_IR_roundingmode ( void )
{
/* 
   rounding mode | PPC | IR
   ------------------------
   to nearest    | 00  | 00
   to zero       | 01  | 11
   to +infinity  | 10  | 10
   to -infinity  | 11  | 01
*/
   IRTemp rm_PPC32 = newTemp(Ity_I32);
   assign( rm_PPC32, getGST_masked( PPC_GST_FPSCR, MASK_FPSCR_RN ) );

   // rm_IR = XOR( rm_PPC32, (rm_PPC32 << 1) & 2)
   return binop( Iop_Xor32, 
                 mkexpr(rm_PPC32),
                 binop( Iop_And32, 
                        binop(Iop_Shl32, mkexpr(rm_PPC32), mkU8(1)),
                        mkU32(2) ));
}

/* The DFP IR rounding modes were chosen such that the existing PPC to IR
 * mapping would still work with the extended three bit DFP rounding 
 * mode designator.

 *  rounding mode                     | PPC  |  IR
 *  -----------------------------------------------
 *  to nearest, ties to even          | 000  | 000
 *  to zero                           | 001  | 011
 *  to +infinity                      | 010  | 010
 *  to -infinity                      | 011  | 001
 *  to nearest, ties away from 0      | 100  | 100
 *  to nearest, ties toward 0         | 101  | 111
 *  to away from 0                    | 110  | 110
 *  to prepare for shorter precision  | 111  | 101
 */
static IRExpr* /* :: Ity_I32 */ get_IR_roundingmode_DFP( void )
{
   IRTemp rm_PPC32 = newTemp( Ity_I32 );
   assign( rm_PPC32, getGST_masked_upper( PPC_GST_FPSCR, MASK_FPSCR_DRN ) );

   // rm_IR = XOR( rm_PPC32, (rm_PPC32 << 1) & 2)
   return binop( Iop_Xor32,
                 mkexpr( rm_PPC32 ),
                 binop( Iop_And32,
                        binop( Iop_Shl32, mkexpr( rm_PPC32 ), mkU8( 1 ) ),
                        mkU32( 2 ) ) );
}

#define NANmaskSingle   0x7F800000
#define NANmaskDouble   0x7FF00000

static IRExpr * Check_NaN( IRExpr * value, IRExpr * Hi32Mask )
{
   IRTemp exp_zero  = newTemp(Ity_I8);
   IRTemp frac_mask = newTemp(Ity_I32);
   IRTemp frac_not_zero = newTemp(Ity_I8);

   /* Check if the result is QNAN or SNAN and not +infinity or -infinity.
    * The input value is always 64-bits, for single precision values, the
    * lower 32 bits must be zero.
    *
    * Single Pricision 
    *  [62:54] exponent field is equal to 0xFF for NAN and Infinity.
    *  [53:32] fraction field is zero for Infinity and non-zero for NAN
    *  [31:0]  unused for single precision representation
    *
    * Double Pricision 
    *  [62:51] exponent field is equal to 0xFF for NAN and Infinity.
    *  [50:0]  fraction field is zero for Infinity and non-zero for NAN
    *
    * Returned result is a U32 value of 0xFFFFFFFF for NaN and 0 otherwise.
    */
   assign( frac_mask, unop( Iop_Not32,
                            binop( Iop_Or32,
                                   mkU32( 0x80000000ULL ), Hi32Mask) ) );

   assign( exp_zero,
           unop( Iop_1Sto8,
                 binop( Iop_CmpEQ32,
                        binop( Iop_And32,
                               unop( Iop_64HIto32,
                                     unop( Iop_ReinterpF64asI64,
                                           value ) ),
                               Hi32Mask ),
                        Hi32Mask ) ) );
   assign( frac_not_zero,
           binop( Iop_Or8,
                  unop( Iop_1Sto8,
                        binop( Iop_CmpNE32,
                               binop( Iop_And32,
                                      unop( Iop_64HIto32,
                                            unop( Iop_ReinterpF64asI64,
                                                  value ) ),
                                      mkexpr( frac_mask ) ),
                               mkU32( 0x0 ) ) ),
                  unop( Iop_1Sto8,
                        binop( Iop_CmpNE32,
                               binop( Iop_And32,
                                      unop( Iop_64to32,
                                            unop( Iop_ReinterpF64asI64,
                                                  value ) ),
                                      mkU32( 0xFFFFFFFF ) ),
                               mkU32( 0x0 ) ) ) ) );
   return unop( Iop_8Sto32,
                binop( Iop_And8,
                       mkexpr( exp_zero ),
                       mkexpr( frac_not_zero ) ) );
}

static IRExpr * Complement_non_NaN( IRExpr * value, IRExpr * nan_mask )
{
   /* This function will only complement the 64-bit floating point value if it
    * is not Nan.  NaN is not a signed value.  Need to do computations using
    * 32-bit operands to ensure it will run in 32-bit mode.
    */
   return  binop( Iop_32HLto64,
                  binop( Iop_Or32,
                         binop( Iop_And32,
                                nan_mask,
                                unop( Iop_64HIto32,
                                      unop( Iop_ReinterpF64asI64,
                                            value ) ) ),
                         binop( Iop_And32,
                                unop( Iop_Not32,
                                      nan_mask ),
                                unop( Iop_64HIto32,
                                      unop( Iop_ReinterpF64asI64,
                                            unop( Iop_NegF64,
                                                  value ) ) ) ) ),
                  unop( Iop_64to32,
                        unop( Iop_ReinterpF64asI64, value ) ) );
}

/*------------------------------------------------------------*/
/*--- Floating Point Instruction Translation               ---*/
/*------------------------------------------------------------*/

/*
  Floating Point Load Instructions
*/
static Bool dis_fp_load_prefix ( UInt prefix, UInt theInstr )
{
   /* X-Form, D-Form */
   UChar opc1      = ifieldOPC(theInstr);
   UChar frT_addr  = ifieldRegDS(theInstr);
   UChar rA_addr   = ifieldRegA(theInstr);

   IRType  ty     = mode64 ? Ity_I64 : Ity_I32;
   IRTemp  EA     = newTemp(ty);
   IRTemp  rA     = newTemp(ty);
   UInt    ptype  = PrefixType(prefix);
   Bool    is_prefix = prefix_instruction( prefix );
   UInt    R      = 0;                    // must be zero for word instruction
   ULong   immediate_val = 0;

   assign( rA, getIReg(rA_addr) );
   assign( EA, calculate_prefix_EA( prefix, theInstr, rA_addr,
                                    ptype, DFORM_IMMASK,
                                    &immediate_val, &R ) );

   switch (opc1) {
   case 0x30: // lfs (Load Float Single, PPC32 p441)
      pDIP( is_prefix, "lfs fr%u,%llu(r%u)\n", frT_addr, immediate_val, rA_addr );
      DIPp( is_prefix, ",%u", R );
      putFReg( frT_addr,
               unop(Iop_F32toF64, load(Ity_F32, mkexpr(EA))) );
      break;

   case 0x32: // lfd (Load Float Double, PPC32 p437)
      pDIP( prefix, "lfd fr%u,%llu(r%u)", frT_addr, immediate_val, rA_addr );
      DIPp( is_prefix, ",%u", R );
      putFReg( frT_addr, load(Ity_F64, mkexpr(EA)) );
      break;

   default:
      vex_printf("dis_fp_load_prefix(ppc)(opc1)\n");
      return False;
   }
   return True;
}

static Bool dis_fp_load ( UInt prefix, UInt theInstr )
{
   /* X-Form, D-Form */
   UChar opc1      = ifieldOPC(theInstr);
   UChar frD_addr  = ifieldRegDS(theInstr);
   UChar rA_addr   = ifieldRegA(theInstr);
   UChar rB_addr   = ifieldRegB(theInstr);
   UInt  opc2      = ifieldOPClo10(theInstr);
   UChar b0        = ifieldBIT0(theInstr);
   UInt  uimm16    = ifieldUIMM16(theInstr);

   Int    simm16 = extend_s_16to32(uimm16);
   IRType ty     = mode64 ? Ity_I64 : Ity_I32;
   IRTemp EA     = newTemp(ty);
   IRTemp rA     = newTemp(ty);
   IRTemp rB     = newTemp(ty);
   IRTemp iHi    = newTemp(Ity_I32);
   IRTemp iLo    = newTemp(Ity_I32);

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   assign( rA, getIReg(rA_addr) );
   assign( rB, getIReg(rB_addr) );

   /* These are completely straightforward from a rounding and status
      bits perspective: no rounding involved and no funny status or CR
      bits affected. */

   switch (opc1) {
   case 0x30: // lfs (Load Float Single, PPC32 p441)
      DIP("lfs fr%u,%d(r%u)\n", frD_addr, simm16, rA_addr);
      assign( EA, ea_rAor0_simm(rA_addr, simm16) );
      putFReg( frD_addr,
               unop(Iop_F32toF64, load(Ity_F32, mkexpr(EA))) );
      break;

   case 0x31: // lfsu (Load Float Single, Update, PPC32 p442)
      if (rA_addr == 0)
         return False;
      DIP("lfsu fr%u,%d(r%u)\n", frD_addr, simm16, rA_addr);
      assign( EA, ea_rA_simm(rA_addr, simm16) );
      putFReg( frD_addr,
               unop(Iop_F32toF64, load(Ity_F32, mkexpr(EA))) );
      putIReg( rA_addr, mkexpr(EA) );
      break;
      
   case 0x32: // lfd (Load Float Double, PPC32 p437)
      DIP("lfd fr%u,%d(r%u)\n", frD_addr, simm16, rA_addr);
      assign( EA, ea_rAor0_simm(rA_addr, simm16) );
      putFReg( frD_addr, load(Ity_F64, mkexpr(EA)) );
      break;

   case 0x33: // lfdu (Load Float Double, Update, PPC32 p438)
      if (rA_addr == 0)
         return False;
      DIP("lfdu fr%u,%d(r%u)\n", frD_addr, simm16, rA_addr);
      assign( EA, ea_rA_simm(rA_addr, simm16) );
      putFReg( frD_addr, load(Ity_F64, mkexpr(EA)) );
      putIReg( rA_addr, mkexpr(EA) );
      break;

   case 0x1F:
      if (b0 != 0) {
         vex_printf("dis_fp_load(ppc)(instr,b0)\n");
         return False;
      }

      switch(opc2) {
      case 0x217: // lfsx (Load Float Single Indexed, PPC32 p444)
         DIP("lfsx fr%u,r%u,r%u\n", frD_addr, rA_addr, rB_addr);
         assign( EA, ea_rAor0_idxd(rA_addr, rB_addr) );
         putFReg( frD_addr, unop( Iop_F32toF64, 
                                  load(Ity_F32, mkexpr(EA))) );
         break;
         
      case 0x237: // lfsux (Load Float Single, Update Indxd, PPC32 p443)
         if (rA_addr == 0)
            return False;
         DIP("lfsux fr%u,r%u,r%u\n", frD_addr, rA_addr, rB_addr);
         assign( EA, ea_rA_idxd(rA_addr, rB_addr) );
         putFReg( frD_addr,
                  unop(Iop_F32toF64, load(Ity_F32, mkexpr(EA))) );
         putIReg( rA_addr, mkexpr(EA) );
         break;
         
      case 0x257: // lfdx (Load Float Double Indexed, PPC32 p440)
         DIP("lfdx fr%u,r%u,r%u\n", frD_addr, rA_addr, rB_addr);
         assign( EA, ea_rAor0_idxd(rA_addr, rB_addr) );
         putFReg( frD_addr, load(Ity_F64, mkexpr(EA)) );
         break;
         
      case 0x277: // lfdux (Load Float Double, Update Indxd, PPC32 p439)
         if (rA_addr == 0)
            return False;
         DIP("lfdux fr%u,r%u,r%u\n", frD_addr, rA_addr, rB_addr);
         assign( EA, ea_rA_idxd(rA_addr, rB_addr) );
         putFReg( frD_addr, load(Ity_F64, mkexpr(EA)) );
         putIReg( rA_addr, mkexpr(EA) );
         break;
         
      case 0x357: // lfiwax (Load Float As Integer, Indxd, ISA 2.05 p120)
         DIP("lfiwax fr%u,r%u,r%u\n", frD_addr, rA_addr, rB_addr);
         assign( EA, ea_rAor0_idxd( rA_addr, rB_addr ) );
         assign( iLo, load(Ity_I32, mkexpr(EA)) );
         assign( iHi, binop(Iop_Sub32,
                            mkU32(0),
                            binop(Iop_Shr32, mkexpr(iLo), mkU8(31)))  );
         putFReg( frD_addr, unop(Iop_ReinterpI64asF64,
                                 binop(Iop_32HLto64, mkexpr(iHi), mkexpr(iLo))) );
         break;

      case 0x377: // lfiwzx (Load floating-point as integer word, zero indexed
      {
         IRTemp dw = newTemp( Ity_I64 );
         DIP("lfiwzx fr%u,r%u,r%u\n", frD_addr, rA_addr, rB_addr);
         assign( EA, ea_rAor0_idxd( rA_addr, rB_addr ) );
         assign( iLo, load(Ity_I32, mkexpr(EA)) );
         assign( dw, binop( Iop_32HLto64, mkU32( 0 ), mkexpr( iLo ) ) );
         putFReg( frD_addr, unop( Iop_ReinterpI64asF64, mkexpr( dw ) ) );
         break;
      }

      default:
         vex_printf("dis_fp_load(ppc)(opc2)\n");
         return False;
      }
      break;

   default:
      vex_printf("dis_fp_load(ppc)(opc1)\n");
      return False;
   }
   return True;
}



/*
  Floating Point Store Instructions
*/
static Bool dis_fp_store_prefix ( UInt prefix, UInt theInstr )
{
   /* X-Form, D-Form */
   UChar opc1      = ifieldOPC(theInstr);
   UChar frS_addr  = ifieldRegDS(theInstr);
   UChar rA_addr   = ifieldRegA(theInstr);

   IRType ty     = mode64 ? Ity_I64 : Ity_I32;
   IRTemp frS    = newTemp(Ity_F64);
   IRTemp EA     = newTemp(ty);
   IRTemp rA     = newTemp(ty);
   UInt   ptype  = PrefixType(prefix);
   Bool   is_prefix = prefix_instruction( prefix );
   UInt   R      = 0;                    // must be zero for word instruction
   ULong  immediate_val = 0;

   assign( frS, getFReg( frS_addr ) );
   assign( rA,  getIReg( rA_addr ) );
   assign( EA, calculate_prefix_EA( prefix, theInstr, rA_addr,
                                    ptype, DFORM_IMMASK,
                                    &immediate_val, &R ) );

  /* These are straightforward from a status bits perspective: no
      funny status or CR bits affected.  For single precision stores,
      the values are truncated and denormalised (not rounded) to turn
      them into single precision values. */

   switch (opc1) {

   case 0x34: // stfs (Store Float Single, PPC32 p518)
      pDIP( is_prefix, "stfs fr%u,%llu(r%u)\n", frS_addr, immediate_val, rA_addr );
      DIPp( is_prefix, ",%u", R );
      /* Use Iop_TruncF64asF32 to truncate and possible denormalise
         the value to be stored in the correct way, without any
         rounding. */
      store( mkexpr(EA), unop(Iop_TruncF64asF32, mkexpr(frS)) );
      break;

   case 0x36: // stfd (Store Float Double, PPC32 p513)
      pDIP( is_prefix, "stfd fr%u,%llu(r%u)", frS_addr, immediate_val, rA_addr );
      DIPp( is_prefix, ",%u", R );
      store( mkexpr(EA), mkexpr(frS) );
      break;

 default:
      vex_printf("dis_fp_store_prefix(ppc)(opc1)\n");
      return False;
   }
   return True;
}

static Bool dis_fp_store ( UInt prefix, UInt theInstr )
{
   /* X-Form, D-Form */
   UChar opc1      = ifieldOPC(theInstr);
   UChar frS_addr  = ifieldRegDS(theInstr);
   UChar rA_addr   = ifieldRegA(theInstr);
   UChar rB_addr   = ifieldRegB(theInstr);
   UInt  opc2      = ifieldOPClo10(theInstr);
   UChar b0        = ifieldBIT0(theInstr);
   Int   uimm16    = ifieldUIMM16(theInstr);

   Int    simm16 = extend_s_16to32(uimm16);
   IRTemp frS    = newTemp(Ity_F64);
   IRType ty     = mode64 ? Ity_I64 : Ity_I32;
   IRTemp EA     = newTemp(ty);
   IRTemp rA     = newTemp(ty);
   IRTemp rB     = newTemp(ty);

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   assign( frS, getFReg(frS_addr) );
   assign( rA,  getIReg(rA_addr) );
   assign( rB,  getIReg(rB_addr) );

   /* These are straightforward from a status bits perspective: no
      funny status or CR bits affected.  For single precision stores,
      the values are truncated and denormalised (not rounded) to turn
      them into single precision values. */

   switch (opc1) {
   case 0x35: // stfsu (Store Float Single, Update, PPC32 p519)
      if (rA_addr == 0)
         return False;
      DIP("stfsu fr%u,%d(r%u)\n", frS_addr, simm16, rA_addr);
      assign( EA, ea_rA_simm(rA_addr, simm16) );
      /* See comment for stfs */
      store( mkexpr(EA), unop(Iop_TruncF64asF32, mkexpr(frS)) );
      putIReg( rA_addr, mkexpr(EA) );
      break;

   case 0x37: // stfdu (Store Float Double, Update, PPC32 p514)
      if (rA_addr == 0)
         return False;
      DIP("stfdu fr%u,%d(r%u)\n", frS_addr, simm16, rA_addr);
      assign( EA, ea_rA_simm(rA_addr, simm16) );
      store( mkexpr(EA), mkexpr(frS) );
      putIReg( rA_addr, mkexpr(EA) );
      break;

   case 0x1F:
      if (b0 != 0) {
         vex_printf("dis_fp_store(ppc)(instr,b0)\n");
         return False;
      }
      switch(opc2) {
      case 0x297: // stfsx (Store Float Single Indexed, PPC32 p521)
         DIP("stfsx fr%u,r%u,r%u\n", frS_addr, rA_addr, rB_addr);
         assign( EA, ea_rAor0_idxd(rA_addr, rB_addr) );
         /* See note for stfs */
         store( mkexpr(EA),
                unop(Iop_TruncF64asF32, mkexpr(frS)) );
         break;
         
      case 0x2B7: // stfsux (Store Float Sgl, Update Indxd, PPC32 p520)
         if (rA_addr == 0)
            return False;
         DIP("stfsux fr%u,r%u,r%u\n", frS_addr, rA_addr, rB_addr);
         assign( EA, ea_rA_idxd(rA_addr, rB_addr) );
         /* See note for stfs */
         store( mkexpr(EA), unop(Iop_TruncF64asF32, mkexpr(frS)) );
         putIReg( rA_addr, mkexpr(EA) );
         break;

      case 0x2D7: // stfdx (Store Float Double Indexed, PPC32 p516)
         DIP("stfdx fr%u,r%u,r%u\n", frS_addr, rA_addr, rB_addr);
         assign( EA, ea_rAor0_idxd(rA_addr, rB_addr) );
         store( mkexpr(EA), mkexpr(frS) );
         break;
         
      case 0x2F7: // stfdux (Store Float Dbl, Update Indxd, PPC32 p515)
         if (rA_addr == 0)
            return False;
         DIP("stfdux fr%u,r%u,r%u\n", frS_addr, rA_addr, rB_addr);
         assign( EA, ea_rA_idxd(rA_addr, rB_addr) );
         store( mkexpr(EA), mkexpr(frS) );
         putIReg( rA_addr, mkexpr(EA) );
         break;

      case 0x3D7: // stfiwx (Store Float as Int, Indexed, PPC32 p517)
         // NOTE: POWERPC OPTIONAL, "Graphics Group" (PPC32_GX)
         DIP("stfiwx fr%u,r%u,r%u\n", frS_addr, rA_addr, rB_addr);
         assign( EA, ea_rAor0_idxd(rA_addr, rB_addr) );
         store( mkexpr(EA),
                unop(Iop_64to32, unop(Iop_ReinterpF64asI64, mkexpr(frS))) );
         break;

      default:
         vex_printf("dis_fp_store(ppc)(opc2)\n");
         return False;
      }
      break;

   default:
      vex_printf("dis_fp_store(ppc)(opc1)\n");
      return False;
   }
   return True;
}



/*
  Floating Point Arith Instructions
*/
static Bool dis_fp_arith ( UInt prefix, UInt theInstr )
{
   /* A-Form */
   UChar opc1     = ifieldOPC(theInstr);
   UChar frD_addr = ifieldRegDS(theInstr);
   UChar frA_addr = ifieldRegA(theInstr);
   UChar frB_addr = ifieldRegB(theInstr);
   UChar frC_addr = ifieldRegC(theInstr);
   UChar opc2     = ifieldOPClo5(theInstr);
   UChar flag_rC  = ifieldBIT0(theInstr);

   IRTemp  frD = newTemp(Ity_F64);
   IRTemp  frA = newTemp(Ity_F64);
   IRTemp  frB = newTemp(Ity_F64);
   IRTemp  frC = newTemp(Ity_F64);
   IRExpr* rm  = get_IR_roundingmode();

   /* By default, we will examine the results of the operation and set
      fpscr[FPRF] accordingly. */
   Bool set_FPRF = True;

   /* By default, if flag_RC is set, we will clear cr1 after the
      operation.  In reality we should set cr1 to indicate the
      exception status of the operation, but since we're not
      simulating exceptions, the exception status will appear to be
      zero.  Hence cr1 should be cleared if this is a . form insn. */
   Bool clear_CR1 = True;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   assign( frA, getFReg(frA_addr));
   assign( frB, getFReg(frB_addr));
   assign( frC, getFReg(frC_addr));

   switch (opc1) {
   case 0x3B:
      switch (opc2) {
      case 0x12: // fdivs (Floating Divide Single, PPC32 p407)
         if (frC_addr != 0)
            return False;
         DIP("fdivs%s fr%u,fr%u,fr%u\n", flag_rC ? ".":"",
             frD_addr, frA_addr, frB_addr);
         assign( frD, triop( Iop_DivF64r32, 
                             rm, mkexpr(frA), mkexpr(frB) ));
         break;

      case 0x14: // fsubs (Floating Subtract Single, PPC32 p430)
         if (frC_addr != 0)
            return False;
         DIP("fsubs%s fr%u,fr%u,fr%u\n", flag_rC ? ".":"",
             frD_addr, frA_addr, frB_addr);
         assign( frD, triop( Iop_SubF64r32, 
                             rm, mkexpr(frA), mkexpr(frB) ));
         break;

      case 0x15: // fadds (Floating Add Single, PPC32 p401)
         if (frC_addr != 0)
            return False;
         DIP("fadds%s fr%u,fr%u,fr%u\n", flag_rC ? ".":"",
             frD_addr, frA_addr, frB_addr);
         assign( frD, triop( Iop_AddF64r32, 
                             rm, mkexpr(frA), mkexpr(frB) ));
         break;

      case 0x16: // fsqrts (Floating SqRt (Single-Precision), PPC32 p428)
         // NOTE: POWERPC OPTIONAL, "General-Purpose Group" (PPC32_FX)
         if (frA_addr != 0 || frC_addr != 0)
            return False;
         DIP("fsqrts%s fr%u,fr%u\n", flag_rC ? ".":"",
             frD_addr, frB_addr);
         // however illogically, on ppc970 this insn behaves identically
         // to fsqrt (double-precision).  So use SqrtF64, not SqrtF64r32.
         assign( frD, binop( Iop_SqrtF64, rm, mkexpr(frB) ));
         break;

      case 0x18: // fres (Floating Reciprocal Estimate Single, PPC32 p421)
         // NOTE: POWERPC OPTIONAL, "Graphics Group" (PPC32_GX)
         if (frA_addr != 0 || frC_addr != 0)
            return False;
         DIP("fres%s fr%u,fr%u\n", flag_rC ? ".":"",
             frD_addr, frB_addr);
         { IRExpr* ieee_one
              = IRExpr_Const(IRConst_F64i(0x3ff0000000000000ULL));
           assign( frD, triop( Iop_DivF64r32, 
                               rm,
                               ieee_one, mkexpr(frB) ));
         }
         break;

      case 0x19: // fmuls (Floating Multiply Single, PPC32 p414)
         if (frB_addr != 0)
            return False;
         DIP("fmuls%s fr%u,fr%u,fr%u\n", flag_rC ? ".":"",
             frD_addr, frA_addr, frC_addr);
         assign( frD, triop( Iop_MulF64r32,
                             rm, mkexpr(frA), mkexpr(frC) ));
         break;

      case 0x1A: // frsqrtes (Floating Recip SqRt Est Single)
         // NOTE: POWERPC OPTIONAL, "Graphics Group" (PPC32_GX)
         // Undocumented instruction?
         if (frA_addr != 0 || frC_addr != 0)
            return False;
         DIP("frsqrtes%s fr%u,fr%u\n", flag_rC ? ".":"",
             frD_addr, frB_addr);
         assign( frD, unop(Iop_RSqrtEst5GoodF64, mkexpr(frB)) );
         break;

      default:
         vex_printf("dis_fp_arith(ppc)(3B: opc2)\n");
         return False;
      }
      break;

   case 0x3F:
      switch (opc2) {           
      case 0x12: // fdiv (Floating Div (Double-Precision), PPC32 p406)
         if (frC_addr != 0)
            return False;
         DIP("fdiv%s fr%u,fr%u,fr%u\n", flag_rC ? ".":"",
             frD_addr, frA_addr, frB_addr);
         assign( frD, triop(Iop_DivF64, rm, mkexpr(frA), mkexpr(frB)) );
         break;

      case 0x14: // fsub (Floating Sub (Double-Precision), PPC32 p429)
         if (frC_addr != 0)
            return False;
         DIP("fsub%s fr%u,fr%u,fr%u\n", flag_rC ? ".":"",
             frD_addr, frA_addr, frB_addr);
         assign( frD, triop(Iop_SubF64, rm, mkexpr(frA), mkexpr(frB)) );
         break;

      case 0x15: // fadd (Floating Add (Double-Precision), PPC32 p400)
         if (frC_addr != 0)
            return False;
         DIP("fadd%s fr%u,fr%u,fr%u\n", flag_rC ? ".":"",
             frD_addr, frA_addr, frB_addr);
         assign( frD, triop(Iop_AddF64, rm, mkexpr(frA), mkexpr(frB)) );
         break;

      case 0x16: // fsqrt (Floating SqRt (Double-Precision), PPC32 p427)
         // NOTE: POWERPC OPTIONAL, "General-Purpose Group" (PPC32_FX)
         if (frA_addr != 0 || frC_addr != 0)
            return False;
         DIP("fsqrt%s fr%u,fr%u\n", flag_rC ? ".":"",
             frD_addr, frB_addr);
         assign( frD, binop(Iop_SqrtF64, rm, mkexpr(frB)) );
         break;

      case 0x17: { // fsel (Floating Select, PPC32 p426)
         // NOTE: POWERPC OPTIONAL, "Graphics Group" (PPC32_GX)
         IRTemp cc    = newTemp(Ity_I32);
         IRTemp cc_b0 = newTemp(Ity_I32);

         DIP("fsel%s fr%u,fr%u,fr%u,fr%u\n", flag_rC ? ".":"",
             frD_addr, frA_addr, frC_addr, frB_addr);

         // cc: UN == 0x41, LT == 0x01, GT == 0x00, EQ == 0x40
         // => GT|EQ == (cc & 0x1 == 0)
         assign( cc, binop(Iop_CmpF64, mkexpr(frA),
                                       IRExpr_Const(IRConst_F64(0))) );
         assign( cc_b0, binop(Iop_And32, mkexpr(cc), mkU32(1)) );

         // frD = (frA >= 0.0) ? frC : frB
         //     = (cc_b0 == 0) ? frC : frB
         assign( frD,
                 IRExpr_ITE(
                    binop(Iop_CmpEQ32, mkexpr(cc_b0), mkU32(0)),
                    mkexpr(frC),
                    mkexpr(frB) ));

         /* One of the rare ones which don't mess with FPRF */
         set_FPRF = False;
         break;
      }

      case 0x18: // fre (Floating Reciprocal Estimate)
         // NOTE: POWERPC OPTIONAL, "Graphics Group" (PPC32_GX)
         // Note: unclear whether this insn really exists or not
         // ppc970 doesn't have it, but POWER5 does
         if (frA_addr != 0 || frC_addr != 0)
            return False;
         DIP("fre%s fr%u,fr%u\n", flag_rC ? ".":"",
             frD_addr, frB_addr);
         { IRExpr* ieee_one
              = IRExpr_Const(IRConst_F64i(0x3ff0000000000000ULL));
           assign( frD, triop( Iop_DivF64, 
                               rm,
                               ieee_one, mkexpr(frB) ));
         }
         break;

      case 0x19: // fmul (Floating Mult (Double Precision), PPC32 p413)
         if (frB_addr != 0)
            vex_printf("dis_fp_arith(ppc)(instr,fmul)\n");
         DIP("fmul%s fr%u,fr%u,fr%u\n", flag_rC ? ".":"",
             frD_addr, frA_addr, frC_addr);
         assign( frD, triop(Iop_MulF64, rm, mkexpr(frA), mkexpr(frC)) );
         break;

      case 0x1A: // frsqrte (Floating Recip SqRt Est., PPC32 p424)
         // NOTE: POWERPC OPTIONAL, "Graphics Group" (PPC32_GX)
         if (frA_addr != 0 || frC_addr != 0)
            return False;
         DIP("frsqrte%s fr%u,fr%u\n", flag_rC ? ".":"",
             frD_addr, frB_addr);
         assign( frD, unop(Iop_RSqrtEst5GoodF64, mkexpr(frB)) );
         break;

      default:
         vex_printf("dis_fp_arith(ppc)(3F: opc2)\n");
         return False;
      }
      break;

   default:
      vex_printf("dis_fp_arith(ppc)(opc1)\n");
      return False;
   }

   putFReg( frD_addr, mkexpr(frD) );

   if (set_FPRF) {
      // XXX XXX XXX FIXME
      // set FPRF from frD
   }

   if (flag_rC && clear_CR1) {
      putCR321( 1, mkU8(0) );
      putCR0( 1, mkU8(0) );
   }

   return True;
}



/*
  Floating Point Mult-Add Instructions
*/
static Bool dis_fp_multadd ( UInt prefix, UInt theInstr )
{
   /* A-Form */
   UChar opc1     = ifieldOPC(theInstr);
   UChar frD_addr = ifieldRegDS(theInstr);
   UChar frA_addr = ifieldRegA(theInstr);
   UChar frB_addr = ifieldRegB(theInstr);
   UChar frC_addr = ifieldRegC(theInstr);
   UChar opc2     = ifieldOPClo5(theInstr);
   UChar flag_rC  = ifieldBIT0(theInstr);

   IRTemp  frD = newTemp(Ity_F64);
   IRTemp  frA = newTemp(Ity_F64);
   IRTemp  frB = newTemp(Ity_F64);
   IRTemp  frC = newTemp(Ity_F64);
   IRTemp  rmt = newTemp(Ity_I32);
   IRTemp  tmp = newTemp(Ity_F64);
   IRTemp  sign_tmp = newTemp(Ity_I64);
   IRTemp  nan_mask = newTemp(Ity_I32);
   IRExpr* rm;

   /* By default, we will examine the results of the operation and set
      fpscr[FPRF] accordingly. */
   Bool set_FPRF = True;

   /* By default, if flag_RC is set, we will clear cr1 after the
      operation.  In reality we should set cr1 to indicate the
      exception status of the operation, but since we're not
      simulating exceptions, the exception status will appear to be
      zero.  Hence cr1 should be cleared if this is a . form insn. */
   Bool clear_CR1 = True;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   /* Bind the rounding mode expression to a temp; there's no
      point in creating gratuitous CSEs, as we know we'll need 
      to use it twice. */
   assign( rmt, get_IR_roundingmode() );
   rm = mkexpr(rmt);

   assign( frA, getFReg(frA_addr));
   assign( frB, getFReg(frB_addr));
   assign( frC, getFReg(frC_addr));

   /* The rounding in this is all a bit dodgy.  The idea is to only do
      one rounding.  That clearly isn't achieveable without dedicated
      four-input IR primops, although in the single precision case we
      can sort-of simulate it by doing the inner multiply in double
      precision. 

      In the negated cases, the negation happens after rounding. */

   switch (opc1) {
   case 0x3B:
      switch (opc2) {
      case 0x1C: // fmsubs (Floating Mult-Subtr Single, PPC32 p412)
         DIP("fmsubs%s fr%u,fr%u,fr%u,fr%u\n", flag_rC ? ".":"",
             frD_addr, frA_addr, frC_addr, frB_addr);
         assign( frD, qop( Iop_MSubF64r32, rm,
                           mkexpr(frA), mkexpr(frC), mkexpr(frB) ));
         break;

      case 0x1D: // fmadds (Floating Mult-Add Single, PPC32 p409)
         DIP("fmadds%s fr%u,fr%u,fr%u,fr%u\n", flag_rC ? ".":"",
             frD_addr, frA_addr, frC_addr, frB_addr);
         assign( frD, qop( Iop_MAddF64r32, rm,
                           mkexpr(frA), mkexpr(frC), mkexpr(frB) ));
         break;

      case 0x1E: // fnmsubs (Float Neg Mult-Subtr Single, PPC32 p420)
      case 0x1F: // fnmadds (Floating Negative Multiply-Add Single, PPC32 p418)

         if (opc2 == 0x1E) {
            DIP("fnmsubs%s fr%u,fr%u,fr%u,fr%u\n", flag_rC ? ".":"",
                     frD_addr, frA_addr, frC_addr, frB_addr);
            assign( tmp, qop( Iop_MSubF64r32, rm,
                              mkexpr(frA), mkexpr(frC), mkexpr(frB) ) );
         } else {
            DIP("fnmadds%s fr%u,fr%u,fr%u,fr%u\n", flag_rC ? ".":"",
                     frD_addr, frA_addr, frC_addr, frB_addr);
            assign( tmp, qop( Iop_MAddF64r32, rm,
                              mkexpr(frA), mkexpr(frC), mkexpr(frB) ) );
         }

         assign( nan_mask, Check_NaN( mkexpr( tmp ),
                                      mkU32( NANmaskSingle ) ) );
         assign( sign_tmp, Complement_non_NaN( mkexpr( tmp ),
                                               mkexpr( nan_mask ) ) );
         assign( frD, unop( Iop_ReinterpI64asF64, mkexpr( sign_tmp ) ) ); 
         break;

      default:
         vex_printf("dis_fp_multadd(ppc)(3B: opc2)\n");
         return False;
      }
      break;

   case 0x3F:
      switch (opc2) {           
      case 0x1C: // fmsub (Float Mult-Sub (Dbl Precision), PPC32 p411)
         DIP("fmsub%s fr%u,fr%u,fr%u,fr%u\n", flag_rC ? ".":"",
             frD_addr, frA_addr, frC_addr, frB_addr);
         assign( frD, qop( Iop_MSubF64, rm,
                           mkexpr(frA), mkexpr(frC), mkexpr(frB) ));
         break;

      case 0x1D: // fmadd (Float Mult-Add (Dbl Precision), PPC32 p408)
         DIP("fmadd%s fr%u,fr%u,fr%u,fr%u\n", flag_rC ? ".":"",
             frD_addr, frA_addr, frC_addr, frB_addr);
         assign( frD, qop( Iop_MAddF64, rm,
                           mkexpr(frA), mkexpr(frC), mkexpr(frB) ));
         break;

      case 0x1E: // fnmsub (Float Neg Mult-Subtr (Dbl Precision), PPC32 p419)
      case 0x1F: // fnmadd (Float Neg Mult-Add (Dbl Precision), PPC32 p417)

         if (opc2 == 0x1E) {
            DIP("fnmsub%s fr%u,fr%u,fr%u,fr%u\n", flag_rC ? ".":"",
                     frD_addr, frA_addr, frC_addr, frB_addr);
            assign( tmp, qop( Iop_MSubF64, rm,
                              mkexpr(frA), mkexpr(frC), mkexpr(frB) ) );
         } else {
            DIP("fnmadd%s fr%u,fr%u,fr%u,fr%u\n", flag_rC ? ".":"",
                     frD_addr, frA_addr, frC_addr, frB_addr);
            assign( tmp, qop( Iop_MAddF64, rm,
                              mkexpr(frA), mkexpr(frC), mkexpr(frB) ));
         }

         assign( nan_mask, Check_NaN( mkexpr( tmp ),
                                      mkU32( NANmaskDouble ) ) );
         assign( sign_tmp, Complement_non_NaN( mkexpr( tmp ),
                                               mkexpr( nan_mask ) ) );
         assign( frD, unop( Iop_ReinterpI64asF64, mkexpr( sign_tmp ) ) ); 
         break;

      default:
         vex_printf("dis_fp_multadd(ppc)(3F: opc2)\n");
         return False;
      }
      break;

   default:
      vex_printf("dis_fp_multadd(ppc)(opc1)\n");
      return False;
   }

   putFReg( frD_addr, mkexpr(frD) );

   if (set_FPRF) {
      // XXX XXX XXX FIXME
      // set FPRF from frD
   }

   if (flag_rC && clear_CR1) {
      putCR321( 1, mkU8(0) );
      putCR0( 1, mkU8(0) );
   }

   return True;
}

/*
 * fe_flag is set to 1 if any of the following conditions occurs:
 *  - The floating-point operand in register FRB is a Zero, a
 *    NaN, an Infinity, or a negative value.
 *  - e_b is less than or equal to: -970 for double precision; -103 for single precision
 *  Otherwise fe_flag is set to 0.
 *
 * fg_flag is set to 1 if either of the following conditions occurs.
 *   - The floating-point operand in register FRB is a Zero, an
 *     Infinity, or a denormalized value.
 *  Otherwise fg_flag is set to 0.
 *
 */

static void do_fp_tsqrt(IRTemp frB_Int, Bool sp, IRTemp * fe_flag_tmp, IRTemp * fg_flag_tmp)
{
   // The following temps are for holding intermediate results
   IRTemp e_b = newTemp(Ity_I32);
   IRExpr * fe_flag,  * fg_flag;
   IRTemp frB_exp_shR = newTemp(Ity_I32);
   UInt bias = sp? 127 : 1023;
   IRExpr * frbNaN, * frbDenorm, * frBNeg;
   IRExpr * eb_LTE;
   IRTemp  frbZero_tmp = newTemp(Ity_I1);
   IRTemp  frbInf_tmp = newTemp(Ity_I1);
   *fe_flag_tmp = newTemp(Ity_I32);
   *fg_flag_tmp = newTemp(Ity_I32);

   if ( sp )
      assign( frB_exp_shR, fp_exp_part( Ity_I32, frB_Int ) );
   else
      assign( frB_exp_shR, fp_exp_part( Ity_I64, frB_Int ) );

   assign(e_b, binop( Iop_Sub32, mkexpr(frB_exp_shR), mkU32( bias ) ));

   //////////////////  fe_flag tests BEGIN //////////////////////
   /* We first do all tests that may result in setting fe_flag to '1'.
    * (NOTE: These tests are similar to those used for ftdiv.  See do_fp_tdiv()
    * for details.)
    */
   if ( sp ) {
      frbNaN = is_NaN( Ity_I32, frB_Int );
      assign( frbInf_tmp, is_Inf( Ity_I32, frB_Int ) );
      assign( frbZero_tmp, is_Zero( Ity_I32, frB_Int ) );

   } else {
      frbNaN = is_NaN( Ity_I64, frB_Int );
      assign( frbInf_tmp, is_Inf( Ity_I64, frB_Int ) );
      assign( frbZero_tmp, is_Zero( Ity_I64, frB_Int ) );
   }

   {
      // Test_value = -970 for double precision
      UInt test_value = sp ? 0xffffff99 : 0xfffffc36;
      eb_LTE = binop( Iop_CmpLE32S, mkexpr( e_b ), mkU32( test_value ) );
   }
   frBNeg = binop( Iop_CmpEQ32,
                   binop( Iop_Shr32,
                          sp ? mkexpr( frB_Int ) : unop( Iop_64HIto32, mkexpr( frB_Int ) ),
                          mkU8( 31 ) ),
                   mkU32( 1 ) );
   //////////////////  fe_flag tests END //////////////////////

   //////////////////  fg_flag tests BEGIN //////////////////////
   /*
    * The following tests were already performed above in the fe_flag
    * tests.  So these conditions will result in both fe_ and fg_ flags
    * being set.
    *   - Test if FRB is Zero
    *   - Test if FRB is an Infinity
    */

   /*
    * Test if FRB holds a denormalized value.  A denormalized value is one where
    * the exp is 0 and the fraction is non-zero.
    */
   if (sp) {
      IRTemp frac_part = newTemp(Ity_I32);
      assign( frac_part, binop( Iop_And32, mkexpr(frB_Int), mkU32(0x007fffff)) );
      frbDenorm
               = mkAND1( binop( Iop_CmpEQ32, mkexpr( frB_exp_shR ), mkU32( 0 ) ),
                         binop( Iop_CmpNE32, mkexpr( frac_part ), mkU32( 0 ) ) );
   } else {
      IRExpr * hi32, * low32, * fraction_is_nonzero;
      IRTemp frac_part = newTemp(Ity_I64);

      assign( frac_part, FP_FRAC_PART(frB_Int) );
      hi32 = unop( Iop_64HIto32, mkexpr( frac_part ) );
      low32 = unop( Iop_64to32, mkexpr( frac_part ) );
      fraction_is_nonzero = binop( Iop_CmpNE32, binop( Iop_Or32, low32, hi32 ),
                                                mkU32( 0 ) );
      frbDenorm
               = mkAND1( binop( Iop_CmpEQ32, mkexpr( frB_exp_shR ), mkU32( 0 ) ),
                         fraction_is_nonzero );
   }
   //////////////////  fg_flag tests END //////////////////////

   /////////////////////////
   fe_flag = mkOR1( mkexpr( frbZero_tmp ),
                    mkOR1( frbNaN,
                           mkOR1( mkexpr( frbInf_tmp ),
                                  mkOR1( frBNeg, eb_LTE ) ) ) );

   fe_flag = unop(Iop_1Uto32, fe_flag);

   fg_flag = mkOR1( mkexpr( frbZero_tmp ),
                    mkOR1( mkexpr( frbInf_tmp ), frbDenorm ) );
   fg_flag = unop(Iop_1Uto32, fg_flag);
   assign (*fg_flag_tmp, fg_flag);
   assign (*fe_flag_tmp, fe_flag);
}
/*
 * fe_flag is set to 1 if any of the following conditions occurs:
 *  - The double-precision floating-point operand in register FRA is a NaN or an
 *    Infinity.
 *  - The double-precision floating-point operand in register FRB is a Zero, a
 *    NaN, or an Infinity.
 *  - e_b is less than or equal to -1022.
 *  - e_b is greater than or equal to 1021.
 *  - The double-precision floating-point operand in register FRA is not a zero
 *    and the difference, e_a - e_b, is greater than or equal to 1023.
 *  - The double-precision floating-point operand in register FRA is not a zero
 *    and the difference, e_a - e_b, is less than or equal to -1021.
 *  - The double-precision floating-point operand in register FRA is not a zero
 *    and e_a is less than or equal to -970
 *  Otherwise fe_flag is set to 0.
 *
 * fg_flag is set to 1 if either of the following conditions occurs.
 *   - The double-precision floating-point operand in register FRA is an Infinity.
 *   - The double-precision floating-point operand in register FRB is a Zero, an
 *     Infinity, or a denormalized value.
 *  Otherwise fg_flag is set to 0.
 *
 */
static void _do_fp_tdiv(IRTemp frA_int, IRTemp frB_int, Bool sp, IRTemp * fe_flag_tmp, IRTemp * fg_flag_tmp)
{
   // The following temps are for holding intermediate results
   IRTemp e_a = newTemp(Ity_I32);
   IRTemp e_b = newTemp(Ity_I32);
   IRTemp frA_exp_shR = newTemp(Ity_I32);
   IRTemp frB_exp_shR = newTemp(Ity_I32);

   UInt bias = sp? 127 : 1023;
   *fe_flag_tmp = newTemp(Ity_I32);
   *fg_flag_tmp = newTemp(Ity_I32);

   /* The following variables hold boolean results from tests
    * that are OR'ed together for setting the fe_ and fg_ flags.
    * For some cases, the booleans are used more than once, so
    * I make those IRTemp's instead of IRExpr's.
    */
   IRExpr * fraNaN, * frbNaN, * frbDenorm;
   IRExpr * eb_LTE, * eb_GTE, * ea_eb_GTE, * ea_eb_LTE, * ea_LTE;
   IRTemp  fraInf_tmp = newTemp(Ity_I1);
   IRTemp  frbZero_tmp = newTemp(Ity_I1);
   IRTemp  frbInf_tmp = newTemp(Ity_I1);
   IRTemp  fraNotZero_tmp = newTemp(Ity_I1);

/* The following are the flags that are set by OR'ing the results of
 * all the tests done for tdiv.  These flags are the input to the specified CR.
 */
   IRExpr * fe_flag, * fg_flag;

   // Create temps that will be used throughout the following tests.
   if ( sp ) {
      assign( frA_exp_shR, fp_exp_part( Ity_I32, frA_int ) );
      assign( frB_exp_shR, fp_exp_part( Ity_I32, frB_int ) );
   } else{
      assign( frA_exp_shR, fp_exp_part( Ity_I64, frA_int ) );
      assign( frB_exp_shR, fp_exp_part( Ity_I64, frB_int ) );
   }

   /* Let e_[a|b] be the unbiased exponent: i.e. exp - 1023. */
   assign(e_a, binop( Iop_Sub32, mkexpr(frA_exp_shR), mkU32( bias ) ));
   assign(e_b, binop( Iop_Sub32, mkexpr(frB_exp_shR), mkU32( bias ) ));


   //////////////////  fe_flag tests BEGIN //////////////////////
   /* We first do all tests that may result in setting fe_flag to '1'. */

   /*
    * Test if the double-precision floating-point operand in register FRA is
    * a NaN:
    */
   fraNaN = sp ? is_NaN( Ity_I32, frA_int ) : is_NaN( Ity_I64, frA_int );
   /*
    * Test if the double-precision floating-point operands in register FRA
    * and FRB is an Infinity.  Test if FRB is zero.
    */
   if ( sp ) {
      assign(fraInf_tmp, is_Inf( Ity_I32, frA_int ) );
      assign( frbInf_tmp, is_Inf( Ity_I32, frB_int ) );
      assign( frbZero_tmp, is_Zero( Ity_I32, frB_int ) );

   } else {
      assign(fraInf_tmp, is_Inf( Ity_I64, frA_int ) );
      assign( frbInf_tmp, is_Inf( Ity_I64, frB_int ) );
      assign( frbZero_tmp, is_Zero( Ity_I64, frB_int ) );
   }
   /*
    * Test if the double-precision floating-point operand in register FRB is
    * a NaN:
    */
   frbNaN = sp ? is_NaN( Ity_I32, frB_int ) : is_NaN( Ity_I64, frB_int );

   /*
    * Test if e_b <= -1022 for double precision;
    * or e_b <= -126 for single precision
    */
   {
      UInt test_value = sp ? 0xffffff82 : 0xfffffc02;
      eb_LTE = binop(Iop_CmpLE32S, mkexpr(e_b), mkU32(test_value));
   }

   /*
    * Test if e_b >= 1021 (i.e., 1021 < e_b) for double precision;
    * or e_b >= -125 (125 < e_b) for single precision
    */
   {
      Int test_value = sp ? 125 : 1021;
      eb_GTE = binop(Iop_CmpLT32S, mkU32(test_value), mkexpr(e_b));
   }

   /*
    * Test if FRA != Zero and (e_a - e_b) >= bias
    */
   if ( sp )
      assign( fraNotZero_tmp, unop( Iop_Not1, is_Zero( Ity_I32, frA_int ) ) );
   else
      assign( fraNotZero_tmp, unop( Iop_Not1, is_Zero( Ity_I64, frA_int ) ) );

   ea_eb_GTE = mkAND1( mkexpr( fraNotZero_tmp ),
                       binop( Iop_CmpLT32S, mkU32( bias ),
                              binop( Iop_Sub32, mkexpr( e_a ),
                                     mkexpr( e_b ) ) ) );

   /*
    * Test if FRA != Zero and (e_a - e_b) <= [-1021 (double precision) or -125 (single precision)]
    */
   {
      UInt test_value = sp ? 0xffffff83 : 0xfffffc03;

      ea_eb_LTE = mkAND1( mkexpr( fraNotZero_tmp ),
                          binop( Iop_CmpLE32S,
                                 binop( Iop_Sub32,
                                        mkexpr( e_a ),
                                        mkexpr( e_b ) ),
                                        mkU32( test_value ) ) );
   }

   /*
    * Test if FRA != Zero and e_a <= [-970 (double precision) or -103 (single precision)]
    */
   {
      UInt test_value = 0xfffffc36;  //Int test_value = -970;

      ea_LTE = mkAND1( mkexpr( fraNotZero_tmp ), binop( Iop_CmpLE32S,
                                                        mkexpr( e_a ),
                                                        mkU32( test_value ) ) );
   }
   //////////////////  fe_flag tests END //////////////////////

   //////////////////  fg_flag tests BEGIN //////////////////////
   /*
    * The following tests were already performed above in the fe_flag
    * tests.  So these conditions will result in both fe_ and fg_ flags
    * being set.
    *   - Test if FRA is an Infinity
    *   - Test if FRB ix Zero
    *   - Test if FRB is an Infinity
    */

   /*
    * Test if FRB holds a denormalized value.  A denormalized value is one where
    * the exp is 0 and the fraction is non-zero.
    */
   {
      IRExpr * fraction_is_nonzero;

      if (sp) {
         fraction_is_nonzero = binop( Iop_CmpNE32, FP_FRAC_PART32(frB_int),
                                      mkU32( 0 ) );
      } else {
         IRExpr * hi32, * low32;
         IRTemp frac_part = newTemp(Ity_I64);
         assign( frac_part, FP_FRAC_PART(frB_int) );

         hi32 = unop( Iop_64HIto32, mkexpr( frac_part ) );
         low32 = unop( Iop_64to32, mkexpr( frac_part ) );
         fraction_is_nonzero = binop( Iop_CmpNE32, binop( Iop_Or32, low32, hi32 ),
                                      mkU32( 0 ) );
      }
      frbDenorm = mkAND1( binop( Iop_CmpEQ32, mkexpr( frB_exp_shR ),
                                 mkU32( 0x0 ) ), fraction_is_nonzero );

   }
   //////////////////  fg_flag tests END //////////////////////

   fe_flag
   = mkOR1(
            fraNaN,
            mkOR1(
                   mkexpr( fraInf_tmp ),
                   mkOR1(
                          mkexpr( frbZero_tmp ),
                          mkOR1(
                                 frbNaN,
                                 mkOR1(
                                        mkexpr( frbInf_tmp ),
                                        mkOR1( eb_LTE,
                                               mkOR1( eb_GTE,
                                                      mkOR1( ea_eb_GTE,
                                                             mkOR1( ea_eb_LTE,
                                                                    ea_LTE ) ) ) ) ) ) ) ) );

   fe_flag = unop(Iop_1Uto32, fe_flag);

   fg_flag = mkOR1( mkexpr( fraInf_tmp ), mkOR1( mkexpr( frbZero_tmp ),
                                                 mkOR1( mkexpr( frbInf_tmp ),
                                                        frbDenorm ) ) );
   fg_flag = unop(Iop_1Uto32, fg_flag);
   assign(*fe_flag_tmp, fe_flag);
   assign(*fg_flag_tmp, fg_flag);
}

/* See description for _do_fp_tdiv() above. */
static IRExpr * do_fp_tdiv(IRTemp frA_int, IRTemp frB_int)
{
   IRTemp  fe_flag, fg_flag;
   /////////////////////////
   /* The CR field consists of fl_flag || fg_flag || fe_flag || 0b0
    * where fl_flag == 1 on ppc64.
    */
   IRExpr * fl_flag = unop(Iop_Not32, mkU32(0xFFFFFE));
   fe_flag = fg_flag = IRTemp_INVALID;
   _do_fp_tdiv(frA_int, frB_int, False/*not single precision*/, &fe_flag, &fg_flag);
   return binop( Iop_Or32,
                 binop( Iop_Or32,
                        binop( Iop_Shl32, fl_flag, mkU8( 3 ) ),
                        binop( Iop_Shl32, mkexpr(fg_flag), mkU8( 2 ) ) ),
                 binop( Iop_Shl32, mkexpr(fe_flag), mkU8( 1 ) ) );
}

static Bool dis_fp_tests ( UInt prefix, UInt theInstr )
{
   UChar opc1     = ifieldOPC(theInstr);
   UChar crfD     = toUChar( IFIELD( theInstr, 23, 3 ) );
   UChar frB_addr = ifieldRegB(theInstr);
   UChar b0       = ifieldBIT0(theInstr);
   UInt  opc2     = ifieldOPClo10(theInstr);
   IRTemp frB_I64     = newTemp(Ity_I64);

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   if (opc1 != 0x3F || b0 != 0 ){
      vex_printf("dis_fp_tests(ppc)(ftdiv)\n");
      return False;
   }
   assign( frB_I64, unop( Iop_ReinterpF64asI64, getFReg( frB_addr ) ) );

   switch (opc2) {
      case 0x080: // ftdiv
      {
         UChar frA_addr = ifieldRegA(theInstr);
         IRTemp frA_I64     = newTemp(Ity_I64);
         UChar b21to22  = toUChar( IFIELD( theInstr, 21, 2 ) );
         if (b21to22 != 0 ) {
            vex_printf("dis_fp_tests(ppc)(ftdiv)\n");
            return False;
         }

         assign( frA_I64, unop( Iop_ReinterpF64asI64, getFReg( frA_addr ) ) );
         putGST_field( PPC_GST_CR, do_fp_tdiv(frA_I64, frB_I64), crfD );

         DIP("ftdiv crf%d,fr%u,fr%u\n", crfD, frA_addr, frB_addr);
         break;
      }
      case 0x0A0: // ftsqrt
      {
         IRTemp flags = newTemp(Ity_I32);
         IRTemp  fe_flag, fg_flag;
         fe_flag = fg_flag = IRTemp_INVALID;
         UChar b18to22  = toUChar( IFIELD( theInstr, 18, 5 ) );
         if ( b18to22 != 0) {
            vex_printf("dis_fp_tests(ppc)(ftsqrt)\n");
            return False;
         }
         DIP("ftsqrt crf%d,fr%u\n", crfD, frB_addr);
         do_fp_tsqrt(frB_I64, False /* not single precision*/, &fe_flag, &fg_flag);
         /* The CR field consists of fl_flag || fg_flag || fe_flag || 0b0
          * where fl_flag == 1 on ppc64.
          */
         assign( flags,
                 binop( Iop_Or32,
                        binop( Iop_Or32, mkU32( 8 ), // fl_flag
                               binop( Iop_Shl32, mkexpr(fg_flag), mkU8( 2 ) ) ),
                        binop( Iop_Shl32, mkexpr(fe_flag), mkU8( 1 ) ) ) );
         putGST_field( PPC_GST_CR, mkexpr(flags), crfD );
         break;
      }

      default:
         vex_printf("dis_fp_tests(ppc)(opc2)\n");
         return False;

   }
   return True;
}

/*
  Floating Point Compare Instructions
*/
static Bool dis_fp_cmp ( UInt prefix, UInt theInstr )
{   
   /* X-Form */
   UChar opc1     = ifieldOPC(theInstr);
   UChar crfD     = toUChar( IFIELD( theInstr, 23, 3 ) );
   UChar b21to22  = toUChar( IFIELD( theInstr, 21, 2 ) );
   UChar frA_addr = ifieldRegA(theInstr);
   UChar frB_addr = ifieldRegB(theInstr);
   UInt  opc2     = ifieldOPClo10(theInstr);
   UChar b0       = ifieldBIT0(theInstr);

   IRTemp ccIR    = newTemp(Ity_I32);
   IRTemp ccPPC32 = newTemp(Ity_I32);

   IRTemp frA     = newTemp(Ity_F64);
   IRTemp frB     = newTemp(Ity_F64);

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   if (opc1 != 0x3F || b21to22 != 0 || b0 != 0) {
      vex_printf("dis_fp_cmp(ppc)(instr)\n");
      return False;
   }

   assign( frA, getFReg(frA_addr));
   assign( frB, getFReg(frB_addr));

   assign( ccIR, binop(Iop_CmpF64, mkexpr(frA), mkexpr(frB)) );
   
   /* Map compare result from IR to PPC32 */
   /*
     FP cmp result | PPC | IR
     --------------------------
     UN            | 0x1 | 0x45
     EQ            | 0x2 | 0x40
     GT            | 0x4 | 0x00
     LT            | 0x8 | 0x01
   */

   // ccPPC32 = Shl(1, (~(ccIR>>5) & 2) 
   //                    | ((ccIR ^ (ccIR>>6)) & 1)
   assign(
      ccPPC32,
      binop(
         Iop_Shl32, 
         mkU32(1),
         unop(
            Iop_32to8, 
            binop(
               Iop_Or32,
               binop(
                  Iop_And32, 
                  unop(
                     Iop_Not32,
                     binop(Iop_Shr32, mkexpr(ccIR), mkU8(5))
                  ),
                  mkU32(2)
               ),
               binop(
                  Iop_And32, 
                  binop(
                     Iop_Xor32, 
                     mkexpr(ccIR),
                     binop(Iop_Shr32, mkexpr(ccIR), mkU8(6))
                  ),
                  mkU32(1)
               )
            )
         )
      )
   );

   putGST_field( PPC_GST_CR, mkexpr(ccPPC32), crfD );
   putFPCC( mkexpr( ccPPC32 ) );

   // XXX XXX XXX FIXME
   // Also write the result into FPRF (it's not entirely clear how)

   /* Note: Differences between fcmpu and fcmpo are only in exception
      flag settings, which aren't supported anyway. */
   switch (opc2) {
   case 0x000: // fcmpu (Floating Compare Unordered, PPC32 p403)
      DIP("fcmpu crf%d,fr%u,fr%u\n", crfD, frA_addr, frB_addr);
      break;
   case 0x020: // fcmpo (Floating Compare Ordered, PPC32 p402)
      DIP("fcmpo crf%d,fr%u,fr%u\n", crfD, frA_addr, frB_addr);
      break;
   default:
      vex_printf("dis_fp_cmp(ppc)(opc2)\n");
      return False;
   }
   return True;
}



/*
  Floating Point Rounding/Conversion Instructions
*/
static Bool dis_fp_round ( UInt prefix, UInt theInstr )
{
   /* X-Form */
   UChar opc1     = ifieldOPC(theInstr);
   UChar b16to20  = ifieldRegA(theInstr);
   UChar frD_addr = ifieldRegDS(theInstr);
   UChar frB_addr = ifieldRegB(theInstr);
   UInt  opc2     = ifieldOPClo10(theInstr);
   UChar flag_rC  = ifieldBIT0(theInstr);

   IRTemp  frD     = newTemp(Ity_F64);
   IRTemp  frB     = newTemp(Ity_F64);
   IRTemp  r_tmp32 = newTemp(Ity_I32);
   IRTemp  r_tmp64 = newTemp(Ity_I64);
   IRExpr* rm      = get_IR_roundingmode();

   /* By default, we will examine the results of the operation and set
      fpscr[FPRF] accordingly. */
   Bool set_FPRF = True;

   /* By default, if flag_RC is set, we will clear cr1 after the
      operation.  In reality we should set cr1 to indicate the
      exception status of the operation, but since we're not
      simulating exceptions, the exception status will appear to be
      zero.  Hence cr1 should be cleared if this is a . form insn. */
   Bool clear_CR1 = True;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   if ((!(opc1 == 0x3F || opc1 == 0x3B)) || b16to20 != 0) {
      vex_printf("dis_fp_round(ppc)(instr)\n");
      return False;
   }

   assign( frB, getFReg(frB_addr));
   if (opc1 == 0x3B) {
      /* The fcfid[u]s instructions (from ISA 2.06) are a bit odd because
       * they're very similar to the other instructions handled here, but have
       * a different primary opcode.
       */
      switch (opc2) {
         case 0x34E: // fcfids (Float convert from signed DWord to single precision)
            DIP("fcfids%s fr%u,fr%u\n", flag_rC ? ".":"", frD_addr, frB_addr);
            assign( r_tmp64, unop( Iop_ReinterpF64asI64, mkexpr(frB)) );
            assign( frD, binop( Iop_RoundF64toF32, rm, binop( Iop_I64StoF64, rm,
                                                              mkexpr( r_tmp64 ) ) ) );
            goto putFR;

         case 0x3Ce: // fcfidus (Float convert from unsigned DWord to single precision)
            DIP("fcfidus%s fr%u,fr%u\n", flag_rC ? ".":"", frD_addr, frB_addr);
            assign( r_tmp64, unop( Iop_ReinterpF64asI64, mkexpr(frB)) );
            assign( frD, unop( Iop_F32toF64, binop( Iop_I64UtoF32, rm, mkexpr( r_tmp64 ) ) ) );
            goto putFR;
      }
      return True;
   }


   switch (opc2) {
   case 0x00C: // frsp (Float Round to Single, PPC32 p423)
      DIP("frsp%s fr%u,fr%u\n", flag_rC ? ".":"", frD_addr, frB_addr);
      assign( frD, binop( Iop_RoundF64toF32, rm, mkexpr(frB) ));
      break;
      
   case 0x00E: // fctiw (Float Conv to Int, PPC32 p404)
      DIP("fctiw%s fr%u,fr%u\n", flag_rC ? ".":"", frD_addr, frB_addr);
      assign( r_tmp32,
              binop(Iop_F64toI32S, rm, mkexpr(frB)) );
      assign( frD, unop( Iop_ReinterpI64asF64,
                         unop( Iop_32Uto64, mkexpr(r_tmp32))));
      /* FPRF is undefined after fctiw.  Leave unchanged. */
      set_FPRF = False;
      break;
      
   case 0x00F: // fctiwz (Float Conv to Int, Round to Zero, PPC32 p405)
      DIP("fctiwz%s fr%u,fr%u\n", flag_rC ? ".":"", frD_addr, frB_addr);
      assign( r_tmp32, 
              binop(Iop_F64toI32S, mkU32(Irrm_ZERO), mkexpr(frB) ));
      assign( frD, unop( Iop_ReinterpI64asF64,
                         unop( Iop_32Uto64, mkexpr(r_tmp32))));
      /* FPRF is undefined after fctiwz.  Leave unchanged. */
      set_FPRF = False;
      break;

   case 0x08F: case 0x08E: // fctiwu[z]
      DIP("fctiwu%s%s fr%u,fr%u\n", opc2 == 0x08F ? "z" : "",
               flag_rC ? ".":"", frD_addr, frB_addr);
      assign( r_tmp32,
              binop( Iop_F64toI32U,
                     opc2 == 0x08F ? mkU32( Irrm_ZERO ) : rm,
                     mkexpr( frB ) ) );
      assign( frD, unop( Iop_ReinterpI64asF64,
                         unop( Iop_32Uto64, mkexpr(r_tmp32))));
      /* FPRF is undefined after fctiwz.  Leave unchanged. */
      set_FPRF = False;
      break;


   case 0x32E: // fctid (Float Conv to Int DWord, PPC64 p437)
      DIP("fctid%s fr%u,fr%u\n", flag_rC ? ".":"", frD_addr, frB_addr);
      assign( r_tmp64,
              binop(Iop_F64toI64S, rm, mkexpr(frB)) );
      assign( frD, unop( Iop_ReinterpI64asF64, mkexpr(r_tmp64)) );
      /* FPRF is undefined after fctid.  Leave unchanged. */
      set_FPRF = False;
      break;

   case 0x32F: // fctidz (Float Conv to Int DWord, Round to Zero, PPC64 p437)
      DIP("fctidz%s fr%u,fr%u\n", flag_rC ? ".":"", frD_addr, frB_addr);
      assign( r_tmp64, 
              binop(Iop_F64toI64S, mkU32(Irrm_ZERO), mkexpr(frB)) );
      assign( frD, unop( Iop_ReinterpI64asF64, mkexpr(r_tmp64)) );
      /* FPRF is undefined after fctidz.  Leave unchanged. */
      set_FPRF = False;
      break;

   case 0x3AE: case 0x3AF: // fctidu[z] (Float Conv to Int DWord Unsigned [Round to Zero])
   {
      DIP("fctidu%s%s fr%u,fr%u\n", opc2 == 0x3AE ? "" : "z",
               flag_rC ? ".":"", frD_addr, frB_addr);
      assign( r_tmp64,
              binop(Iop_F64toI64U, opc2 == 0x3AE ? rm : mkU32(Irrm_ZERO), mkexpr(frB)) );
      assign( frD, unop( Iop_ReinterpI64asF64, mkexpr(r_tmp64)) );
      /* FPRF is undefined after fctidz.  Leave unchanged. */
      set_FPRF = False;
      break;
   }
   case 0x34E: // fcfid (Float Conv from Int DWord, PPC64 p434)
      DIP("fcfid%s fr%u,fr%u\n", flag_rC ? ".":"", frD_addr, frB_addr);
      assign( r_tmp64, unop( Iop_ReinterpF64asI64, mkexpr(frB)) );
      assign( frD, 
              binop(Iop_I64StoF64, rm, mkexpr(r_tmp64)) );
      break;

   case 0x3CE: // fcfidu (Float convert from unsigned DWord)
      DIP("fcfidu%s fr%u,fr%u\n", flag_rC ? ".":"", frD_addr, frB_addr);
      assign( r_tmp64, unop( Iop_ReinterpF64asI64, mkexpr(frB)) );
      assign( frD, binop( Iop_I64UtoF64, rm, mkexpr( r_tmp64 ) ) );
      break;

   case 0x188: case 0x1A8: case 0x1C8: case 0x1E8: // frin, friz, frip, frim
      switch(opc2) {
      case 0x188: // frin (Floating Round to Integer Nearest)
         DIP("frin%s fr%u,fr%u\n", flag_rC ? ".":"", frD_addr, frB_addr);
         assign( r_tmp64,
                 binop(Iop_F64toI64S, mkU32(Irrm_NEAREST), mkexpr(frB)) );
         break;
      case 0x1A8: // friz (Floating Round to Integer Toward Zero)
         DIP("friz%s fr%u,fr%u\n", flag_rC ? ".":"", frD_addr, frB_addr);
         assign( r_tmp64,
                 binop(Iop_F64toI64S, mkU32(Irrm_ZERO), mkexpr(frB)) );
         break;
      case 0x1C8: // frip (Floating Round to Integer Plus)
         DIP("frip%s fr%u,fr%u\n", flag_rC ? ".":"", frD_addr, frB_addr);
         assign( r_tmp64,
                 binop(Iop_F64toI64S, mkU32(Irrm_PosINF), mkexpr(frB)) );
         break;
      case 0x1E8: // frim (Floating Round to Integer Minus)
         DIP("frim%s fr%u,fr%u\n", flag_rC ? ".":"", frD_addr, frB_addr);
         assign( r_tmp64,
                 binop(Iop_F64toI64S, mkU32(Irrm_NegINF), mkexpr(frB)) );
         break;
      }

      /* don't use the rounded integer if frB is outside -9e18..9e18 */
      /* F64 has only log10(2**52) significant digits anyway */
      /* need to preserve sign of zero */
      /*   frD = (fabs(frB) > 9e18) ? frB :
               (sign(frB)) ? -fabs((double)r_tmp64) : (double)r_tmp64  */
      assign(frD, IRExpr_ITE(
                     binop(Iop_CmpNE8,
                           unop(Iop_32to8,
                                binop(Iop_CmpF64,
                                      IRExpr_Const(IRConst_F64(9e18)),
                                      unop(Iop_AbsF64, mkexpr(frB)))),
                           mkU8(0)),
                     mkexpr(frB),
                     IRExpr_ITE(
                        binop(Iop_CmpNE32,
                              binop(Iop_Shr32,
                                    unop(Iop_64HIto32,
                                         unop(Iop_ReinterpF64asI64,
                                              mkexpr(frB))),
                                    mkU8(31)),
                              mkU32(0)),
                        unop(Iop_NegF64,
                             unop( Iop_AbsF64,
                                   binop(Iop_I64StoF64, mkU32(0),
                                         mkexpr(r_tmp64)) )),
                        binop(Iop_I64StoF64, mkU32(0), mkexpr(r_tmp64) )
                     )
      ));
      break;

   default:
      vex_printf("dis_fp_round(ppc)(opc2)\n");
      return False;
   }
putFR:
   putFReg( frD_addr, mkexpr(frD) );

   if (set_FPRF) {
      // XXX XXX XXX FIXME
      // set FPRF from frD
   }

   if (flag_rC && clear_CR1) {
      putCR321( 1, mkU8(0) );
      putCR0( 1, mkU8(0) );
   }

   return True;
}

/*
  Floating Point Pair Instructions
*/
static Bool dis_fp_pair_prefix ( UInt prefix, UInt theInstr )
{
   /* X-Form/DS-Form */
   UChar  opc1         = ifieldOPC(theInstr);
   UChar  rA_addr      = ifieldRegA(theInstr);
   IRType ty           = mode64 ? Ity_I64 : Ity_I32;
   IRTemp EA           = newTemp(ty);
   IRTemp EA_16        = newTemp(ty);
   UInt ptype          = PrefixType(prefix);
   Bool is_prefix      = prefix_instruction( prefix );
   UInt R              = 0;
   ULong immediate_val  = 0;
   UInt opc2;

   switch (opc1) {
   case 0x6:
   {
      UChar XTp = ifieldRegXTp(theInstr);
      opc2 = ifieldOPClo4(theInstr);

      assign( EA, calculate_prefix_EA( prefix, theInstr,
                                       rA_addr, ptype, DQFORM_IMMASK,
                                       &immediate_val, &R ) );

      switch (opc2) {

      case 0:
      {
         /* Endian aware load */
         DIP( "lxvp %u,%llu(%u)\n", XTp, immediate_val, rA_addr );

         // address of next 128bits
         assign( EA_16, binop( Iop_Add64, mkU64( 16 ), mkexpr( EA ) ) );
         if (host_endness == VexEndnessBE) {
            putVSReg( XTp, load( Ity_V128, mkexpr( EA ) ) );
            putVSReg( XTp+1, load( Ity_V128, mkexpr( EA_16 ) ) );
         } else {
            putVSReg( XTp+1, load( Ity_V128, mkexpr( EA ) ) );
            putVSReg( XTp, load( Ity_V128, mkexpr( EA_16 ) ) );
         }
         break;
      }

      case 1:
      {
         IRTemp EA_8  = newTemp(ty);
         IRTemp EA_24 = newTemp(ty);
         /* Endian aware store */
         DIP("stxvp %u,%llu(%u)\n", XTp, immediate_val, rA_addr );

         // address of next 128bits
         assign( EA_8, binop( Iop_Add64, mkU64( 8 ), mkexpr( EA ) ) );
         assign( EA_16, binop( Iop_Add64, mkU64( 16 ), mkexpr( EA ) ) );
         assign( EA_24, binop( Iop_Add64, mkU64( 24 ), mkexpr( EA ) ) );

         if (host_endness == VexEndnessBE) {
            store( mkexpr( EA ), unop( Iop_V128to64, getVSReg( XTp ) ) );
            store( mkexpr( EA_8 ), unop( Iop_V128HIto64, getVSReg( XTp ) ) );
            store( mkexpr( EA_16 ), unop( Iop_V128to64, getVSReg( XTp+1 ) ) );
            store( mkexpr( EA_24 ), unop( Iop_V128HIto64, getVSReg( XTp+1 ) ) );
         } else {
            store( mkexpr( EA ), unop( Iop_V128to64, getVSReg( XTp+1 ) ) );
            store( mkexpr( EA_8 ), unop( Iop_V128HIto64, getVSReg( XTp+1 ) ) );
            store( mkexpr( EA_16 ), unop( Iop_V128to64, getVSReg( XTp ) ) );
            store( mkexpr( EA_24 ), unop( Iop_V128HIto64, getVSReg( XTp ) ) );
         }
         break;
      }

      default:
         vex_printf("dis_fp_pair_prefix\n");
         return False;
      }
      return True;
   }
   break;

   case 0x2A:   // plxsd
   {
      UChar vRT = ifieldRegDS(theInstr);
      /* The prefixed word version uses the D-form.  */
      assign( EA, calculate_prefix_EA( prefix, theInstr, rA_addr,
                                       ptype, DFORM_IMMASK,
                                       &immediate_val, &R ) );

      pDIP( is_prefix, "lxsd v%u,%llu(r%u)\n", vRT, immediate_val, rA_addr );
      DIPp( is_prefix, ",%u", R );

      putVSReg( vRT+32, binop( Iop_64HLtoV128,
                               load( Ity_I64, mkexpr( EA ) ),
                               mkU64( 0 ) ) );
      return True;
   }

   case 0x2B:   // plxssp
   {
      UChar vRT = ifieldRegDS(theInstr);
      /* The prefixed word version uses the D-form.  */
      assign( EA, calculate_prefix_EA( prefix, theInstr, rA_addr,
                                       ptype, DFORM_IMMASK,
                                       &immediate_val, &R ) );


      pDIP( is_prefix, "lxssp v%u,%llu(r%u)\n", vRT, immediate_val, rA_addr );
      DIPp( is_prefix, ",%u", R );
      putVSReg( vRT+32,
                binop( Iop_64HLtoV128,
                       unop( Iop_ReinterpF64asI64,
                             unop( Iop_F32toF64,
                                   unop( Iop_ReinterpI32asF32,
                                         load( Ity_I32, mkexpr( EA ) )
                                      ) ) ),
                       mkU64( 0 ) ) );
      return True;
   }

   case 0x32:   // plxv0
   case 0x33:   // plxv1  These are both plxv, but bit 5 is used for TX
   {
      IRExpr* irx_addr;
      IRTemp word[2];
      UInt ea_off = 8;
      UChar vRS  = ifieldRegDS(theInstr);
      UInt  T  = IFIELD( theInstr, 21, 5);
      UInt  TX = IFIELD( theInstr, 26, 1);

      assign( EA,
              calculate_prefix_EA( prefix, theInstr, rA_addr, ptype,
                                   DFORM_IMMASK, &immediate_val, &R ) );

      // plxv (Load VSX Vector)
      pDIP( is_prefix, "lxv v%u,%llu(r%u)\n", vRS, immediate_val, rA_addr );
      DIPp( is_prefix, ",%u", R );

      word[0] = newTemp(Ity_I64);
      assign( word[0], load( Ity_I64, mkexpr( EA ) ) );

      irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( EA ),
                        ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );

      word[1] = newTemp(Ity_I64);
      assign( word[1], load( Ity_I64, irx_addr ) );

      if (host_endness == VexEndnessBE)
         putVSReg( TX*32+T, binop( Iop_64HLtoV128,
                                   mkexpr( word[0] ),
                                   mkexpr( word[1] ) ) );
      else
         putVSReg( TX*32+T, binop( Iop_64HLtoV128,
                                   mkexpr( word[1] ),
                                   mkexpr( word[0] ) ) );
      return True;
   }
   break;

   case 0x39:   // lxsd, plxsd, lxssp, plxssp
   {
      UChar vRT = ifieldRegDS(theInstr);
      opc2 = ifieldOPC0o2(theInstr);

      if (opc1 == 0x2A) {  // plxsd
         opc2 = 0x2;  // map plxsd to lxsd inst
         /* The prefixed word version uses the D-form.  */
         assign( EA, calculate_prefix_EA( prefix, theInstr, rA_addr,
                                          ptype, DFORM_IMMASK,
                                          &immediate_val, &R ) );

      } else if (opc1 == 0x2B) { // plxssp
         opc2 = 0x3;  // map plxssp to lxssp inst
         /* The prefixed word version uses the D-form.  */
         assign( EA, calculate_prefix_EA( prefix, theInstr, rA_addr,
                                          ptype, DFORM_IMMASK,
                                          &immediate_val, &R ) );
      } else {
         /* The word version uses the DS-form.  */
         assign( EA, calculate_prefix_EA( prefix, theInstr, rA_addr,
                                          ptype, DSFORM_IMMASK,
                                          &immediate_val, &R ) );
      }

      switch(opc2) {
      case 0x2:     // lxsd, plxsd (Load VSX Scalar Doubleword)
      {
         pDIP( is_prefix, "lxsd v%u,%llu(r%u)\n", vRT, immediate_val,
               rA_addr );
         DIPp( is_prefix, ",%u", R );
         putVSReg( vRT+32, binop( Iop_64HLtoV128,
                                  load( Ity_I64, mkexpr( EA ) ),
                                  mkU64( 0 ) ) );
         return True;
      }
      break;

      case 0x3:     // lxssp (Load VSX Scalar Single from memory,
                    // store as double in register)
         pDIP( is_prefix, "lxssp v%u,%llu(r%u)\n", vRT, immediate_val,
               rA_addr );
         DIPp( is_prefix, ",%u", R );

         putVSReg( vRT+32,
                   binop( Iop_64HLtoV128,
                          unop( Iop_ReinterpF64asI64,
                                unop( Iop_F32toF64,
                                      unop( Iop_ReinterpI32asF32,
                                            load( Ity_I32, mkexpr( EA ) )
                                         ) ) ),
                          mkU64( 0 ) ) );
         return True;

      default:
         vex_printf("dis_fp_pair_prefix(ppc) : DS-form wrong opc2\n");
         return False;
      }
      break;
   }

   case 0x2E:  // pstxsd
   {
      // pstxsd (Store VSX Scalar Doubleword)
      UChar vRS  = ifieldRegDS(theInstr);

      assign( EA, calculate_prefix_EA( prefix, theInstr,
                                       rA_addr, ptype, DFORM_IMMASK,
                                       &immediate_val, &R ) );
      pDIP( is_prefix, "stxsd v%u,%llu(r%u)\n", vRS, immediate_val, rA_addr);
      DIPp( is_prefix, ",%u", R );
      store( mkexpr(EA), unop( Iop_V128HIto64,
                               getVSReg( vRS+32 ) ) );
      /* HW is clearing vector element 1.  Don't see that in the ISA but
       * matching the HW.
       */
      putVSReg( vRS+32, binop( Iop_64HLtoV128,
                               unop( Iop_V128HIto64,
                                     getVSReg( vRS+32 ) ),
                               mkU64( 0 ) ) );
      return True;
   }
   break;

   case 0x2F:
   {
      // pstxssp (Store VSX Scalar Single - store double precision
      // value from register into memory in single precision format)
      UChar vRS  = ifieldRegDS(theInstr);
      IRTemp high64 = newTemp(Ity_F64);
      IRTemp val32  = newTemp(Ity_I32);

      assign( EA, calculate_prefix_EA( prefix, theInstr,
                                       rA_addr, ptype, DFORM_IMMASK,
                                       &immediate_val, &R ) );
      pDIP( is_prefix, "stxssp v%u,%llu(r%u)\n", vRS, immediate_val, rA_addr);
      DIPp( is_prefix, ",%u", R );

      assign(high64, unop( Iop_ReinterpI64asF64,
                           unop( Iop_V128HIto64, getVSReg( vRS+32 ) ) ) );

      assign(val32, unop( Iop_ReinterpF32asI32,
                          unop( Iop_TruncF64asF32,
                                mkexpr(high64) ) ) );
      store( mkexpr(EA), mkexpr( val32 ) );

      return True;
   }
   break;

   case 0x3d:  // lxv
   {
      IRExpr* irx_addr;
      IRTemp word[2];
      UInt ea_off = 8;
      UChar vRS = ifieldRegDS(theInstr);
      UInt T = IFIELD( theInstr, 21, 5);
      UInt TX = IFIELD( theInstr, 3, 1);

      opc2 = IFIELD(theInstr, 0, 3);

      if ( IFIELD( theInstr, 0, 3) == 1) {
         // lxv (Load VSX Vector)
         assign( EA, calculate_prefix_EA( prefix, theInstr,
                                          rA_addr, ptype, DQFORM_IMMASK,
                                          &immediate_val, &R ) );

         DIP("lxv v%u,%llu(r%u)\n", vRS, immediate_val, rA_addr );
         word[0] = newTemp(Ity_I64);
         assign( word[0], load( Ity_I64, mkexpr( EA ) ) );

         irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( EA ),
                           ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );
         word[1] = newTemp(Ity_I64);
         assign( word[1], load( Ity_I64, irx_addr ) );

         if (host_endness == VexEndnessBE)
            putVSReg( TX*32+T, binop( Iop_64HLtoV128,
                                      mkexpr( word[0] ),
                                      mkexpr( word[1] ) ) );
         else
            putVSReg( TX*32+T, binop( Iop_64HLtoV128,
                                      mkexpr( word[1] ),
                                      mkexpr( word[0] ) ) );
         return True;

       } else if ((opc2 & 0x3) == 0x2) {
         // stxsd (Store VSX Scalar Doubleword)
         R = 0;  // must be zero for word instruction
         assign( EA, calculate_prefix_EA( prefix, theInstr,
                                          rA_addr, ptype, DSFORM_IMMASK,
                                          &immediate_val, &R ) );

         DIP("stxsd v%u,%llu(r%u)\n", vRS, immediate_val, rA_addr);
         store( mkexpr(EA), unop( Iop_V128HIto64,
                                  getVSReg( vRS+32 ) ) );
         /* HW is clearing vector element 1.  Don't see that in the ISA but
          * matching the HW.
          */
         putVSReg( vRS+32, binop( Iop_64HLtoV128,
                                  unop( Iop_V128HIto64,
                                        getVSReg( vRS+32 ) ),
                                  mkU64( 0 ) ) );
         return True;

      } else if ((opc2 & 0x3) == 0x3) {
         // stxssp (Store VSX Scalar Single - store double precision
         // value from register into memory in single precision format)
         IRTemp high64 = newTemp(Ity_F64);
         IRTemp val32  = newTemp(Ity_I32);

         assign( EA,
                 calculate_prefix_EA( prefix, theInstr, rA_addr, ptype,
                                      DSFORM_IMMASK, &immediate_val, &R ) );
         DIP("stxssp v%u,%llu(r%u)\n", vRS, immediate_val, rA_addr);
         assign(high64, unop( Iop_ReinterpI64asF64,
                              unop( Iop_V128HIto64, getVSReg( vRS+32 ) ) ) );

         assign(val32, unop( Iop_ReinterpF32asI32,
                             unop( Iop_TruncF64asF32,
                                   mkexpr(high64) ) ) );
         store( mkexpr(EA), mkexpr( val32 ) );

         return True;

      } else if (opc2  == 0x5) {
         // stxv (Store VSX Vector)
         assign( EA, calculate_prefix_EA( prefix, theInstr,
                                          rA_addr, ptype, DQFORM_IMMASK,
                                          &immediate_val, &R ) );
         DIP("stxv v%u,%llu(r%u)\n", vRS, immediate_val, rA_addr );

         if (host_endness == VexEndnessBE) {
            store( mkexpr(EA), unop( Iop_V128HIto64,
                                     getVSReg( TX*32+T ) ) );
            irx_addr
               = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( EA ),
                        ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );
            store( irx_addr, unop( Iop_V128to64,
                                   getVSReg( TX*32+T ) ) );
         } else {
            store( mkexpr(EA), unop( Iop_V128to64,
                                     getVSReg( TX*32+T ) ) );
            irx_addr
               = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( EA ),
                        ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );
               store( irx_addr, unop( Iop_V128HIto64,
                                      getVSReg( TX*32+T ) ) );
         }
         return True;

      } else {
         vex_printf("dis_fp_pair vector load/store (ppc) : DS-form wrong opc2\n");
         return False;
      }
   }
   break;

   case 0x3A:  // plxvp
   {
      UChar XTp = ifieldRegXTp(theInstr);

      assign( EA, calculate_prefix_EA( prefix, theInstr,
                                       rA_addr, ptype, DFORM_IMMASK,
                                       &immediate_val, &R ) );

      /* Endian aware prefixed load */
      pDIP( is_prefix, "lxvp %u,%llu(%u)", XTp, immediate_val, rA_addr );
      DIPp( is_prefix, ",%u", R );

      // address of next 128bits
      assign( EA_16, binop( Iop_Add64, mkU64( 16 ), mkexpr( EA ) ) );
      if (host_endness == VexEndnessBE) {
         putVSReg( XTp, load( Ity_V128, mkexpr( EA ) ) );
         putVSReg( XTp+1, load( Ity_V128, mkexpr( EA_16 ) ) );
      } else {
         putVSReg( XTp+1, load( Ity_V128, mkexpr( EA ) ) );
         putVSReg( XTp, load( Ity_V128, mkexpr( EA_16 ) ) );
      }
      return True;
   }

   case 0x3E:  // pstxvp
   {
      IRTemp EA_8  = newTemp(ty);
      IRTemp EA_24 = newTemp(ty);
      UChar XTp = ifieldRegXTp(theInstr);

      assign( EA, calculate_prefix_EA( prefix, theInstr,
                                       rA_addr, ptype, DFORM_IMMASK,
                                       &immediate_val, &R ) );

      /* Endian aware prefixed load */
      pDIP( is_prefix, "stxvp %u,%llu(%u)\n", XTp, immediate_val, rA_addr );
      DIPp( is_prefix, ",%u", R );

      assign( EA_8, binop( Iop_Add64, mkU64( 8 ), mkexpr( EA ) ) );
      assign( EA_16, binop( Iop_Add64, mkU64( 16 ), mkexpr( EA ) ) );
      assign( EA_24, binop( Iop_Add64, mkU64( 24 ), mkexpr( EA ) ) );

      if (host_endness == VexEndnessBE) {
         store( mkexpr( EA ), unop( Iop_V128to64, getVSReg( XTp ) ) );
         store( mkexpr( EA_8 ), unop( Iop_V128HIto64, getVSReg( XTp ) ) );
         store( mkexpr( EA_16 ), unop( Iop_V128to64, getVSReg( XTp+1 ) ) );
         store( mkexpr( EA_24 ), unop( Iop_V128HIto64, getVSReg( XTp+1 ) ) );
     } else {
         store( mkexpr( EA ), unop( Iop_V128to64, getVSReg( XTp+1 ) ) );
         store( mkexpr( EA_8 ), unop( Iop_V128HIto64, getVSReg( XTp+1 ) ) );
         store( mkexpr( EA_16 ), unop( Iop_V128to64, getVSReg( XTp ) ) );
         store( mkexpr( EA_24 ), unop( Iop_V128HIto64, getVSReg( XTp ) ) );
      }
      return True;
   }

   case 0x36:  // pstxv0
   case 0x37:  // pstxv1, pstxv inst where bit 5 is SX
   {
      // pstxv (Prefixed store VSX Vector 1 8LS:D-form)
      // AKA pstxv0, pstxv1
      UInt  S  = IFIELD( theInstr, 21, 5);
      UInt  SX = IFIELD( theInstr, 26, 1);
      UInt  XS = 32*SX+S;
      UChar vRS  = ifieldRegDS(theInstr);
      IRTemp tmpV128  = newTemp(Ity_V128);
      IRExpr* irx_addr;
      UInt ea_off = 8;

      DIP("pstxv v%u,%llu(r%u)", vRS, immediate_val, rA_addr );
      DIPp( is_prefix, ",%u", R );

      assign( tmpV128, getVSReg( XS ) );

      assign( EA,
              calculate_prefix_EA( prefix, theInstr,
                                   rA_addr, ptype, DFORM_IMMASK,
                                   &immediate_val, &R ) );

      if (host_endness == VexEndnessBE) {
         store( mkexpr(EA), unop( Iop_V128HIto64,
                                  mkexpr( tmpV128 ) ) );
         irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( EA ),
                           ty == Ity_I64 ? mkU64( ea_off ):
                           mkU32( ea_off ) );
         store( irx_addr, unop( Iop_V128to64,
                                mkexpr( tmpV128 ) ) );

      } else {
         store( mkexpr(EA), unop( Iop_V128to64,
                                  mkexpr( tmpV128 ) ) );
         irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( EA ),
                           ty == Ity_I64 ? mkU64( ea_off ):
                           mkU32( ea_off ) );
         store( irx_addr, unop( Iop_V128HIto64,
                                mkexpr( tmpV128 ) ) );
      }
      return True;
   }

   default:
      vex_printf("dis_fp_pair_prefix(ppc)(instr)\n");
      return False;
   }
   return False;
}

static Bool dis_fp_pair ( UInt prefix, UInt theInstr )
{
   /* X-Form/DS-Form */
   UChar  opc1         = ifieldOPC(theInstr);
   UChar  frT_hi_addr  = ifieldRegDS(theInstr);
   UChar  frT_lo_addr  = frT_hi_addr + 1;
   UChar  rA_addr      = ifieldRegA(theInstr);
   UChar  rB_addr      = ifieldRegB(theInstr);
   UInt  uimm16        = ifieldUIMM16(theInstr);
   Int    simm16       = extend_s_16to32(uimm16);
   UInt   opc2         = ifieldOPClo10(theInstr);
   IRType ty           = mode64 ? Ity_I64 : Ity_I32;
   IRTemp EA_hi        = newTemp(ty);
   IRTemp EA_lo        = newTemp(ty);
   IRTemp frT_hi       = newTemp(Ity_F64);
   IRTemp frT_lo       = newTemp(Ity_F64);
   UChar b0            = ifieldBIT0(theInstr);
   Bool is_load        = 0;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   switch (opc1) {
   case 0x1F: // register offset
      /* These instructions work on a pair of registers.  The specified
       * register must be even.
       */
      if ((frT_hi_addr %2) != 0) {
         vex_printf("dis_fp_pair(ppc) ldpx or stdpx: odd frT register\n");
         return False;
      }

      switch(opc2) {
      case 0x317:     // lfdpx (FP Load Double Pair X-form, ISA 2.05  p125)
         DIP("ldpx fr%u,r%u,r%u\n", frT_hi_addr, rA_addr, rB_addr);
         is_load = 1;
         break;
      case 0x397:     // stfdpx (FP STORE Double Pair X-form, ISA 2.05  p125)
         DIP("stdpx fr%u,r%u,r%u\n", frT_hi_addr, rA_addr, rB_addr);
         break;
      default:
         vex_printf("dis_fp_pair(ppc) : X-form wrong opc2\n");
         return False;
      }

      if (b0 != 0) {
         vex_printf("dis_fp_pair(ppc)(0x1F,b0)\n");
         return False;
      }
      assign( EA_hi, ea_rAor0_idxd( rA_addr, rB_addr ) );
      break;
   case 0x39:
   {
      opc2 = ifieldOPC0o2(theInstr);

      switch(opc2) {
      case 0x0:     // lfdp (FP Load Double Pair DS-form, ISA 2.05  p125)
         /* This instruction works on a pair of registers.  The specified
          * register must be even.
          */
         if ((frT_hi_addr %2) != 0) {
            vex_printf("dis_fp_pair(ppc) lfdp : odd frT register\n");
            return False;
         }

         DIP("lfdp fr%u,%d(r%u)\n", frT_hi_addr, simm16, rA_addr);
         assign( EA_hi, ea_rAor0_simm( rA_addr, simm16  ) );
         is_load = 1;
         break;

      default:
         vex_printf("dis_fp_pair(ppc) : DS-form wrong opc2\n");
         return False;
      }
      break;
   }
   case 0x3d:
   {
      opc2 = ifieldOPC0o2(theInstr);

      switch(opc2) {
      case 0x0:
         // stfdp (FP Store Double Pair DS-form, ISA 2.05  p125)
         /* This instruction works on a pair of registers.  The specified
          * register must be even.
          */
         if ((frT_hi_addr %2) != 0) {
            vex_printf("dis_fp_pair(ppc) stfdp : odd frT register\n");
            return False;
         }

         DIP("stfdp fr%u,%d(r%u)\n", frT_hi_addr, simm16, rA_addr);
         assign( EA_hi, ea_rAor0_simm( rA_addr, simm16  ) );
         break;

      default:
         vex_printf("dis_fp_pair(ppc) : DS-form wrong opc2\n");
         return False;
      }
      break;
   }
   default:   // immediate offset
      vex_printf("dis_fp_pair(ppc)(instr)\n");
      return False;
   }

   if (mode64)
      assign( EA_lo, binop(Iop_Add64, mkexpr(EA_hi), mkU64(8)) );
   else
      assign( EA_lo, binop(Iop_Add32, mkexpr(EA_hi), mkU32(8)) );

   assign( frT_hi, getFReg(frT_hi_addr) );
   assign( frT_lo, getFReg(frT_lo_addr) );

   if (is_load) {
      putFReg( frT_hi_addr, load(Ity_F64, mkexpr(EA_hi)) );
      putFReg( frT_lo_addr, load(Ity_F64, mkexpr(EA_lo)) );
   } else {
      store( mkexpr(EA_hi), mkexpr(frT_hi) );
      store( mkexpr(EA_lo), mkexpr(frT_lo) );
   }

   return True;
}


/*
  Floating Point Merge Instructions
*/
static Bool dis_fp_merge ( UInt prefix, UInt theInstr )
{
   /* X-Form */
   UInt  opc2     = ifieldOPClo10(theInstr);
   UChar frD_addr = ifieldRegDS(theInstr);
   UChar frA_addr = ifieldRegA(theInstr);
   UChar frB_addr = ifieldRegB(theInstr);

   IRTemp frD = newTemp(Ity_F64);
   IRTemp frA = newTemp(Ity_F64);
   IRTemp frB = newTemp(Ity_F64);

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   assign( frA, getFReg(frA_addr));
   assign( frB, getFReg(frB_addr));

   switch (opc2) {
   case 0x3c6: // fmrgew floating merge even word
      DIP("fmrgew fr%u,fr%u,fr%u\n", frD_addr, frA_addr, frB_addr);

      assign( frD, unop( Iop_ReinterpI64asF64,
                         binop( Iop_32HLto64,
                                unop( Iop_64HIto32,
                                      unop( Iop_ReinterpF64asI64,
                                            mkexpr(frA) ) ),
                                unop( Iop_64HIto32,
                                      unop( Iop_ReinterpF64asI64,
                                            mkexpr(frB) ) ) ) ) );
   break;

   case 0x346: // fmrgow floating merge odd word
      DIP("fmrgow fr%u,fr%u,fr%u\n", frD_addr, frA_addr, frB_addr);

      assign( frD, unop( Iop_ReinterpI64asF64,
                         binop( Iop_32HLto64,
                                unop( Iop_64to32,
                                      unop( Iop_ReinterpF64asI64,
                                            mkexpr(frA) ) ),
                                unop( Iop_64to32,
                                      unop( Iop_ReinterpF64asI64,
                                            mkexpr(frB) ) ) ) ) );
   break;

   default:
      vex_printf("dis_fp_merge(ppc)(opc2)\n");
      return False;
   }

   putFReg( frD_addr, mkexpr(frD) );
   return True;
}

/*
  Floating Point Move Instructions
*/
static Bool dis_fp_move ( UInt prefix, UInt theInstr )
{
   /* X-Form */
   UChar opc1     = ifieldOPC(theInstr);
   UChar frD_addr = ifieldRegDS(theInstr);
   UChar frA_addr = ifieldRegA(theInstr);
   UChar frB_addr = ifieldRegB(theInstr);
   UInt  opc2     = ifieldOPClo10(theInstr);
   UChar flag_rC  = ifieldBIT0(theInstr);

   IRTemp frD = newTemp(Ity_F64);
   IRTemp frB = newTemp(Ity_F64);
   IRTemp itmpB = newTemp(Ity_F64);
   IRTemp frA;
   IRTemp signA;
   IRTemp hiD;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   if (opc1 != 0x3F || (frA_addr != 0 && opc2 != 0x008)) {
      vex_printf("dis_fp_move(ppc)(instr)\n");
      return False;
   }

   assign( frB, getFReg(frB_addr));

   switch (opc2) {
   case 0x008: // fcpsgn (Floating Copy Sign, ISA_V2.05 p126)
      DIP("fcpsgn%s fr%u,fr%u,fr%u\n", flag_rC ? ".":"", frD_addr, frA_addr,
          frB_addr);
      signA = newTemp(Ity_I32);
      hiD = newTemp(Ity_I32);
      itmpB = newTemp(Ity_I64);
      frA = newTemp(Ity_F64);
      assign( frA, getFReg(frA_addr) );

      /* get A's sign bit */
      assign(signA, binop(Iop_And32,
                          unop(Iop_64HIto32, unop(Iop_ReinterpF64asI64,
                                                  mkexpr(frA))),
                          mkU32(0x80000000)) );

      assign( itmpB, unop(Iop_ReinterpF64asI64, mkexpr(frB)) );

      /* mask off B's sign bit and or in A's sign bit */
      assign(hiD, binop(Iop_Or32,
                        binop(Iop_And32,
                              unop(Iop_64HIto32,
                                   mkexpr(itmpB)),  /* frB's high 32 bits */
                              mkU32(0x7fffffff)),
                        mkexpr(signA)) );

      /* combine hiD/loB into frD */
      assign( frD, unop(Iop_ReinterpI64asF64,
                        binop(Iop_32HLto64,
                              mkexpr(hiD),
                              unop(Iop_64to32,
                                   mkexpr(itmpB)))) );   /* frB's low 32 bits */
      break;

   case 0x028: // fneg (Floating Negate, PPC32 p416)
      DIP("fneg%s fr%u,fr%u\n", flag_rC ? ".":"", frD_addr, frB_addr);
      assign( frD, unop( Iop_NegF64, mkexpr(frB) ));
      break;
      
   case 0x048: // fmr (Floating Move Register, PPC32 p410)
      DIP("fmr%s fr%u,fr%u\n", flag_rC ? ".":"", frD_addr, frB_addr);
      assign( frD, mkexpr(frB) );
      break;
      
   case 0x088: // fnabs (Floating Negative Absolute Value, PPC32 p415)
      DIP("fnabs%s fr%u,fr%u\n", flag_rC ? ".":"", frD_addr, frB_addr);
      assign( frD, unop( Iop_NegF64, unop( Iop_AbsF64, mkexpr(frB) )));
      break;
      
   case 0x108: // fabs (Floating Absolute Value, PPC32 p399)
      DIP("fabs%s fr%u,fr%u\n", flag_rC ? ".":"", frD_addr, frB_addr);
      assign( frD, unop( Iop_AbsF64, mkexpr(frB) ));
      break;
      
   default:
      vex_printf("dis_fp_move(ppc)(opc2)\n");
      return False;
   }

   putFReg( frD_addr, mkexpr(frD) );

   /* None of these change FPRF.  cr1 is set in the usual way though,
      if flag_rC is set. */

   if (flag_rC) {
      putCR321( 1, mkU8(0) );
      putCR0( 1, mkU8(0) );
   }

   return True;
}



/*
  Floating Point Status/Control Register Instructions
*/
static Bool dis_fp_scr ( UInt prefix, UInt theInstr, Bool GX_level )
{
   /* Many forms - see each switch case */
   UChar opc1    = ifieldOPC(theInstr);
   UInt  opc2    = ifieldOPClo10(theInstr);
   UChar flag_rC = ifieldBIT0(theInstr);

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   if (opc1 != 0x3F) {
      vex_printf("dis_fp_scr(ppc)(instr)\n");
      return False;
   }

   switch (opc2) {
   case 0x026: { // mtfsb1 (Move to FPSCR Bit 1, PPC32 p479)
      // Bit crbD of the FPSCR is set.
      UChar crbD    = ifieldRegDS(theInstr);
      UInt  b11to20 = IFIELD(theInstr, 11, 10);

      if (b11to20 != 0) {
         vex_printf("dis_fp_scr(ppc)(instr,mtfsb1)\n");
         return False;
      }
      DIP("mtfsb1%s crb%d \n", flag_rC ? ".":"", crbD);
      putGST_masked( PPC_GST_FPSCR, mkU64( 1 <<( 31 - crbD ) ),
		     1ULL << ( 31 - crbD ) );
      break;
   }

   case 0x040: { // mcrfs (Move to Condition Register from FPSCR, PPC32 p465)
      UChar   crfD    = toUChar( IFIELD( theInstr, 23, 3 ) );
      UChar   b21to22 = toUChar( IFIELD( theInstr, 21, 2 ) );
      UChar   crfS    = toUChar( IFIELD( theInstr, 18, 3 ) );
      UChar   b11to17 = toUChar( IFIELD( theInstr, 11, 7 ) );
      IRTemp  tmp     = newTemp(Ity_I32);
      IRExpr* fpscr_all;
      if (b21to22 != 0 || b11to17 != 0 || flag_rC != 0) {
         vex_printf("dis_fp_scr(ppc)(instr,mcrfs)\n");
         return False;
      }
      DIP("mcrfs crf%d,crf%d\n", crfD, crfS);
      vassert(crfD < 8);
      vassert(crfS < 8);
      fpscr_all = getGST_masked( PPC_GST_FPSCR, MASK_FPSCR_RN );
      assign( tmp, binop(Iop_And32,
                         binop(Iop_Shr32,fpscr_all,mkU8(4 * (7-crfS))),
                        mkU32(0xF)) );
      putGST_field( PPC_GST_CR, mkexpr(tmp), crfD );
      break;
   }

   case 0x046: { // mtfsb0 (Move to FPSCR Bit 0, PPC32 p478)
      // Bit crbD of the FPSCR is cleared.
      UChar crbD    = ifieldRegDS(theInstr);
      UInt  b11to20 = IFIELD(theInstr, 11, 10);

      if (b11to20 != 0) {
         vex_printf("dis_fp_scr(ppc)(instr,mtfsb0)\n");
         return False;
      }      
      DIP("mtfsb0%s crb%d\n", flag_rC ? ".":"", crbD);
      putGST_masked( PPC_GST_FPSCR, mkU64( 0 ), 1ULL << ( 31 - crbD ) );
      break;
   }

   case 0x086: { // mtfsfi (Move to FPSCR Field Immediate, PPC32 p481)
      UInt crfD     = IFIELD( theInstr, 23, 3 );
      UChar b17to22 = toUChar( IFIELD( theInstr, 17, 6 ) );
      UChar IMM     = toUChar( IFIELD( theInstr, 12, 4 ) );
      UChar b11     = toUChar( IFIELD( theInstr, 11, 1 ) );
      UChar Wbit    = toUChar( IFIELD( theInstr, 16, 1 ) );

      if (b17to22 != 0 || b11 != 0 || (Wbit && !GX_level)) {
         vex_printf("dis_fp_scr(ppc)(instr,mtfsfi)\n");
         return False;
      }
      DIP("mtfsfi%s crf%u,%d%s\n", flag_rC ? ".":"", crfD, IMM, Wbit ? ",1":"");
      crfD = crfD + (8 * (1 - Wbit) );
      putGST_field( PPC_GST_FPSCR, mkU32( IMM ), crfD );
      break;
   }

   case 0x247: { // mffs (Move from FPSCR, PPC32 p468)
      UChar   frD_addr  = ifieldRegDS(theInstr);
      UChar   frB_addr  = ifieldRegB(theInstr);
      IRTemp  frB = newTemp(Ity_F64);
      UInt    b11to12   = IFIELD(theInstr, 19, 2);
      UInt    b13to15   = IFIELD(theInstr, 16, 3);
      UInt    RN        = IFIELD(theInstr, 11, 2);
      UInt    DRN       = IFIELD(theInstr, 11, 3);

      /* The FPSCR_DRN, FPSCR_RN and FPSCR_FPCC are all stored in
       * their own 8-bit entries with distinct offsets.  The FPSCR
       * register is handled as two 32-bit values.  We need to
       * assemble the pieces into the single 64-bit value to return.
       */
      IRExpr* fpscr_lower
         = binop( Iop_Or32,
                  getGST_masked( PPC_GST_FPSCR, (MASK_FPSCR_RN | MASK_FPSCR_C_FPCC) ),
                  binop( Iop_Or32,
                         binop( Iop_Shl32,
                                getC(),
                                mkU8(63-47) ) ,
                         binop( Iop_Shl32,
                                getFPCC(),
                                mkU8(63-51) ) ) );
      IRExpr* fpscr_upper = getGST_masked_upper( PPC_GST_FPSCR, MASK_FPSCR_DRN );

      if ((b11to12 == 0) && (b13to15 == 0)) {
            DIP("mffs%s fr%u\n", flag_rC ? ".":"", frD_addr);
            putFReg( frD_addr,
                     unop( Iop_ReinterpI64asF64,
                           binop( Iop_32HLto64, fpscr_upper, fpscr_lower ) ) );

      } else if ((b11to12 == 0) && (b13to15 == 1)) {
            DIP("mffsce fr%u\n", frD_addr);
            /* Technically as of 4/5/2017 we are not tracking VE, OE, UE, ZE,
               or XE but in case that changes in the future, do the masking. */
            putFReg( frD_addr,
                     unop( Iop_ReinterpI64asF64,
                           binop( Iop_32HLto64, fpscr_upper,
                                  binop( Iop_And32, fpscr_lower,
                                         mkU32( 0xFFFFFF07 ) ) ) ) );

      } else if ((b11to12 == 2) && (b13to15 == 4)) {
            IRTemp  frB_int = newTemp(Ity_I64);

            DIP("mffscdrn fr%u,fr%u\n", frD_addr, frB_addr);

            assign( frB, getFReg(frB_addr));
            assign( frB_int, unop( Iop_ReinterpF64asI64, mkexpr( frB ) ) );

            /* Clear all of the FPSCR bits except for the DRN field, VE,
               OE, UE, ZE and XE bits and write the result to the frD
               register. Note, currently the exception bits are not tracked but
               will mask anyway in case that changes in the future. */
            putFReg( frD_addr,
                     unop( Iop_ReinterpI64asF64,
                           binop( Iop_32HLto64,
                                  binop( Iop_And32, mkU32(0x7), fpscr_upper ),
                                  binop( Iop_And32, mkU32(0xFF), fpscr_lower ) ) ) );

            /* Put new_DRN bits into the FPSCR register */
            putGST_masked( PPC_GST_FPSCR, mkexpr( frB_int ), MASK_FPSCR_DRN );

      } else if ((b11to12 == 2) && (b13to15 == 5)) {
            DIP("mffscdrni fr%u,%u\n", frD_addr, DRN);

            /* Clear all of the FPSCR bits except for the DRN field, VE,
               OE, UE, ZE and XE bits and write the result to the frD
               register. Note, currently the exception bits are not tracked but
               will mask anyway in case that changes in the future. */
            putFReg( frD_addr,
                     unop( Iop_ReinterpI64asF64,
                           binop( Iop_32HLto64,
                                  binop( Iop_And32, mkU32(0x7), fpscr_upper ),
                                  binop( Iop_And32, mkU32(0xFF), fpscr_lower ) ) ) );

            /* Put new_DRN bits into the FPSCR register */
            putGST_masked( PPC_GST_FPSCR, binop( Iop_32HLto64, mkU32( DRN ),
                                                 mkU32( 0 ) ), MASK_FPSCR_DRN );

      } else if ((b11to12 == 2) && (b13to15 == 6)) {
            IRTemp  frB_int = newTemp(Ity_I64);

            DIP("mffscrn fr%u,fr%u\n", frD_addr,frB_addr);

            assign( frB, getFReg(frB_addr));
            assign( frB_int, unop( Iop_ReinterpF64asI64, mkexpr( frB ) ) );

            /* Clear all of the FPSCR bits except for the DRN field, VE,
               OE, UE, ZE and XE bits and write the result to the frD
               register. Note, currently the exception bits are not tracked but
               will mask anyway in case that changes in the future. */
            putFReg( frD_addr,
                     unop( Iop_ReinterpI64asF64,
                           binop( Iop_32HLto64,
                                  binop( Iop_And32, mkU32(0x7), fpscr_upper ),
                                  binop( Iop_And32, mkU32(0xFF), fpscr_lower ) ) ) );

            /* Put new_CRN bits into the FPSCR register */
            putGST_masked( PPC_GST_FPSCR, mkexpr( frB_int ), MASK_FPSCR_RN );

      } else if ((b11to12 == 2) && (b13to15 == 7)) {
            DIP("mffscrni fr%u,%u\n", frD_addr, RN);

            /* Clear all of the FPSCR bits except for the DRN field, VE,
               OE, UE, ZE and XE bits and write the result to the frD
               register. Note, currently the exception bits are not tracked but
               will mask anyway in case that changes in the future. */
            putFReg( frD_addr,
                     unop( Iop_ReinterpI64asF64,
                           binop( Iop_32HLto64,
                                  binop( Iop_And32, mkU32(0x7), fpscr_upper ),
                                  binop( Iop_And32, mkU32(0xFF), fpscr_lower ) ) ) );

            /* Put new_RN bits into the FPSCR register */
            putGST_masked( PPC_GST_FPSCR, binop( Iop_32HLto64, mkU32( 0 ),
                                                 mkU32( RN ) ), MASK_FPSCR_RN );

      } else if ((b11to12 == 3) && (b13to15 == 0)) {
            DIP("mffsl fr%u\n", frD_addr);
            /* Technically as of 4/5/2017 we are not tracking VE, OE, UE, ZE,
               XE, FR, FI, C, FL, FG, FE, FU.  Also only track DRN in the upper
               bits but in case that changes in the future we will do the
               masking. */
            putFReg( frD_addr,
                     unop( Iop_ReinterpI64asF64,
                           binop( Iop_32HLto64,
                                  binop( Iop_And32, fpscr_upper,
                                         mkU32( 0x7 ) ),
                                  binop( Iop_And32, fpscr_lower,
                                         mkU32( 0x7F0FF ) ) ) ) );
      } else {
         vex_printf("dis_fp_scr(ppc)(mff**) Unrecognized instruction.\n");
         return False;
      }
      break;
   }

   case 0x2C7: { // mtfsf (Move to FPSCR Fields, PPC32 p480)
      UChar b25      = toUChar( IFIELD(theInstr, 25, 1) );
      UChar FM       = toUChar( IFIELD(theInstr, 17, 8) );
      UChar frB_addr = ifieldRegB(theInstr);
      IRTemp frB   = newTemp(Ity_F64);
      IRTemp rB_64 = newTemp( Ity_I64 );
      Int i;
      ULong mask;
      UChar Wbit;
#define BFP_MASK_SEED 0x3000000000000000ULL
#define DFP_MASK_SEED 0x7000000000000000ULL

      if (GX_level) {
         /* This implies that Decimal Floating Point is supported, and the
          * FPSCR must be managed as a 64-bit register.
          */
         Wbit = toUChar( IFIELD(theInstr, 16, 1) );
      } else {
         Wbit = 0;
      }

      if (b25 == 1) {
         /* new 64 bit move variant for power 6.  If L field (bit 25) is
          * a one do a full 64 bit move.  Note, the FPSCR is not really
          * properly modeled.  This instruciton only changes the value of
          * the rounding mode bit fields RN, FPCC and DRN.  The HW exception bits
          * do not get set in the simulator.  1/12/09
          */
         DIP("mtfsf%s %d,fr%u (L=1)\n", flag_rC ? ".":"", FM, frB_addr);
         mask = 0x1F0001F003;

      } else {
         DIP("mtfsf%s %d,fr%u\n", flag_rC ? ".":"", FM, frB_addr);
         // Build 32bit mask from FM:
         mask = 0;
         for (i=0; i<8; i++) {
            if ((FM & (1<<(7-i))) == 1) {
               /* FPSCR field k is set to the contents of the corresponding
                * field of register FRB, where k = i+8x(1-W).  In the Power
                * ISA, register field numbering is from left to right, so field
                * 15 is the least significant field in a 64-bit register.  To
                * generate the mask, we set all the appropriate rounding mode
                * bits in the highest order nibble (field 0) and shift right 
                * 'k x nibble length'.
                */
               if (Wbit)
                  mask |= DFP_MASK_SEED >> ( 4 * ( i + 8 * ( 1 - Wbit ) ) );
               else
                  mask |= BFP_MASK_SEED >> ( 4 * ( i + 8 * ( 1 - Wbit ) ) );
            }
            if ((FM & (1<<(7-i))) == 0x2) { //set the FPCC bits
               mask |= 0xF000;
            }
            if ((FM & (1<<(7-i))) == 0x4) { //set the Floating-Point Class Descriptor (C) bit
               mask |= 0x10000;
            }
         }
      }
      assign( frB, getFReg(frB_addr));
      assign( rB_64, unop( Iop_ReinterpF64asI64, mkexpr( frB ) ) );
      putGST_masked( PPC_GST_FPSCR, mkexpr( rB_64 ), mask );
      break;
   }

   default:
      vex_printf("dis_fp_scr(ppc)(opc2)\n");
      return False;
   }
   return True;
}

/*------------------------------------------------------------*/
/*--- Decimal Floating Point (DFP)  Helper functions       ---*/
/*------------------------------------------------------------*/
#define DFP_LONG  1
#define DFP_EXTND 2
#define DFP_LONG_BIAS   398
#define DFP_LONG_ENCODED_FIELD_MASK  0x1F00
#define DFP_EXTND_BIAS  6176
#define DFP_EXTND_ENCODED_FIELD_MASK 0x1F000
#define DFP_LONG_EXP_MSK   0XFF
#define DFP_EXTND_EXP_MSK  0XFFF

#define DFP_G_FIELD_LONG_MASK     0x7FFC0000  // upper 32-bits only
#define DFP_LONG_GFIELD_RT_SHIFT  (63 - 13 - 32) // adj for upper 32-bits 
#define DFP_G_FIELD_EXTND_MASK    0x7FFFC000  // upper 32-bits only
#define DFP_EXTND_GFIELD_RT_SHIFT (63 - 17 - 32) //adj for upper 32 bits
#define DFP_T_FIELD_LONG_MASK     0x3FFFF  // mask for upper 32-bits
#define DFP_T_FIELD_EXTND_MASK    0x03FFFF // mask for upper 32-bits
#define DFP_LONG_EXP_MAX          369      // biased max
#define DFP_LONG_EXP_MIN          0        // biased min
#define DFP_EXTND_EXP_MAX         6111     // biased max
#define DFP_EXTND_EXP_MIN         0        // biased min
#define DFP_LONG_MAX_SIG_DIGITS   16
#define DFP_EXTND_MAX_SIG_DIGITS  34
#define MAX_DIGITS_IN_STRING      8


#define  AND(x, y) binop( Iop_And32, x, y )
#define AND4(w, x, y, z) AND( AND( w, x ), AND( y, z ) )
#define   OR(x, y) binop( Iop_Or32,  x, y )
#define  OR3(x, y, z)    OR( x, OR( y, z ) )
#define  OR4(w, x, y, z) OR( OR( w, x ), OR( y, z ) )
#define  NOT(x) unop( Iop_1Uto32, unop( Iop_Not1, unop( Iop_32to1,  mkexpr( x ) ) ) )

#define  SHL(value, by) binop( Iop_Shl32, value, mkU8( by ) )
#define  SHR(value, by) binop( Iop_Shr32, value, mkU8( by ) )

#define BITS5(_b4,_b3,_b2,_b1,_b0) \
   (((_b4) << 4) | ((_b3) << 3) | ((_b2) << 2) | \
    ((_b1) << 1) | ((_b0) << 0))

static void generate_store_DFP_FPRF_value( ULong irType, IRExpr *src,
                                           const VexAbiInfo* vbi )
{
   /* This function takes a DFP value and computes the value of the FPRF
      field in the FPCC register and store it.  It is done as a clean helper.
      The FPRF[0:4]:
        bits[0:4] =
                    0b00001   Signaling NaN (DFP only)
                    0b10001   Quite NaN
                    0b01001   negative infinity
                    0b01000   negative normal number
                    0b11000   negative subnormal number
                    0b10010   negative zero
                    0b00010   positive zero
                    0b10100   positive subnormal number
                    0b00100   positive normal number
                    0b00101   positive infinity
   */

   IRTemp sign = newTemp( Ity_I32 );
   IRTemp gfield = newTemp( Ity_I32 );
   IRTemp gfield_mask = newTemp( Ity_I32 );
   IRTemp exponent = newTemp( Ity_I64 );
   UInt   exponent_bias = 0;
   IRTemp T_value_is_zero = newTemp( Ity_I32 );
   IRTemp fprf_value = newTemp( Ity_I32 );
   IRTemp lmd            = newTemp( Ity_I32 );
   IRTemp lmd_zero_true  = newTemp( Ity_I1 );
   Int    min_norm_exp = 0;

   vassert( irType == Ity_D128);

   if (irType == Ity_D128) {
      assign( gfield_mask, mkU32( DFP_G_FIELD_EXTND_MASK ) );
      /* The gfield bits are left justified.  */
      assign( gfield, binop( Iop_And32,
                            mkexpr( gfield_mask ),
                            unop( Iop_64HIto32,
                                  unop( Iop_ReinterpD64asI64,
                                        unop( Iop_D128HItoD64, src ) ) ) ) );
      assign( exponent, unop( Iop_ExtractExpD128, src ) );
      exponent_bias = 6176;
      min_norm_exp = -6143;

      /* The significand is zero if the T field and LMD are all zeros */
      /* Check if LMD is zero */
      Get_lmd( &lmd, binop( Iop_Shr32,
                            mkexpr( gfield ), mkU8( 31 - 5 ) ) );

      assign( lmd_zero_true, binop( Iop_CmpEQ32,
                                    mkexpr( lmd ),
                                    mkU32( 0 ) ) );
      /* The T value and the LMD are the BCD value of the significand.
         If the upper and lower T value fields and the LMD are all zero
         then the significand is zero.  */
      assign( T_value_is_zero,
              unop( Iop_1Uto32,
                    mkAND1 (
                       mkexpr( lmd_zero_true ),
                       mkAND1 ( binop( Iop_CmpEQ64,
                                       binop( Iop_And64,
                                              mkU64( DFP_T_FIELD_EXTND_MASK ),
                                              unop( Iop_ReinterpD64asI64,
                                                    unop( Iop_D128HItoD64,
                                                          src ) ) ),
                                       mkU64( 0 ) ),
                                binop( Iop_CmpEQ64,
                                       unop( Iop_ReinterpD64asI64,
                                             unop( Iop_D128LOtoD64,
                                                   src ) ),
                                       mkU64( 0 ) ) ) ) ) );

      assign( sign,
              unop( Iop_64to32,
                    binop( Iop_Shr64,
                           unop( Iop_ReinterpD64asI64,
                                 unop( Iop_D128HItoD64, src ) ),
                           mkU8( 63 ) ) ) );
   } else {
      /* generate_store_DFP_FPRF_value, unknown value for irType */
      vassert(0);
   }

   /* Determine what the type of the number is. */
   assign( fprf_value,
           mkIRExprCCall( Ity_I32, 0 /*regparms*/,
                          "generate_DFP_FPRF_value_helper",
                          fnptr_to_fnentry( vbi,
                                            &generate_DFP_FPRF_value_helper ),
                             mkIRExprVec_6( mkexpr( gfield ),
                                            mkexpr( exponent ),
                                            mkU32( exponent_bias ),
                                            mkU32( min_norm_exp ),
                                            mkexpr( sign ),
                                            mkexpr( T_value_is_zero ) ) ) );
   /* fprf[0:4] = (C | FPCC[0:3])  */
   putC( binop( Iop_Shr32, mkexpr( fprf_value ), mkU8( 4 ) ) );
   putFPCC( binop( Iop_And32, mkexpr( fprf_value ), mkU32 (0xF ) ) );
   return;
}

static IRExpr * Gfield_encoding( IRExpr * lmexp, IRExpr * lmd32 )
{
   IRTemp lmd_07_mask   = newTemp( Ity_I32 );
   IRTemp lmd_8_mask    = newTemp( Ity_I32 );
   IRTemp lmd_9_mask    = newTemp( Ity_I32 );
   IRTemp lmexp_00_mask = newTemp( Ity_I32 );
   IRTemp lmexp_01_mask = newTemp( Ity_I32 );
   IRTemp lmexp_10_mask = newTemp( Ity_I32 );
   IRTemp lmd_07_val    = newTemp( Ity_I32 );
   IRTemp lmd_8_val     = newTemp( Ity_I32 );
   IRTemp lmd_9_val     = newTemp( Ity_I32 );

   /* The encodig is as follows:
    * lmd - left most digit
    * lme - left most 2-bits of the exponent
    *
    *    lmd
    *   0 - 7    (lmexp << 3) | lmd
    *     8      0b11000 (24 decimal) if lme=0b00;
    *            0b11010 (26 decimal) if lme=0b01;
    *            0b11100 (28 decimal) if lme=0b10;
    *     9      0b11001 (25 decimal) if lme=0b00;
    *            0b11011 (27 decimal) if lme=0b01;
    *            0b11101 (29 decimal) if lme=0b10;
    */

   /* Generate the masks for each condition */
   assign( lmd_07_mask,
           unop( Iop_1Sto32, binop( Iop_CmpLE32U, lmd32, mkU32( 7 ) ) ) );
   assign( lmd_8_mask,
           unop( Iop_1Sto32, binop( Iop_CmpEQ32, lmd32, mkU32( 8 ) ) ) );
   assign( lmd_9_mask,
           unop( Iop_1Sto32, binop( Iop_CmpEQ32, lmd32, mkU32( 9 ) ) ) );
   assign( lmexp_00_mask,
           unop( Iop_1Sto32, binop( Iop_CmpEQ32, lmexp, mkU32( 0 ) ) ) );
   assign( lmexp_01_mask,
           unop( Iop_1Sto32, binop( Iop_CmpEQ32, lmexp, mkU32( 1 ) ) ) );
   assign( lmexp_10_mask,
           unop( Iop_1Sto32, binop( Iop_CmpEQ32, lmexp, mkU32( 2 ) ) ) );

   /* Generate the values for each LMD condition, assuming the condition
    * is TRUE.
    */
   assign( lmd_07_val,
           binop( Iop_Or32, binop( Iop_Shl32, lmexp, mkU8( 3 ) ), lmd32 ) );
   assign( lmd_8_val,
           binop( Iop_Or32,
                  binop( Iop_Or32,
                         binop( Iop_And32,
                                mkexpr( lmexp_00_mask ),
                                mkU32( 24 ) ),
                         binop( Iop_And32,
                                mkexpr( lmexp_01_mask ),
                                mkU32( 26 ) ) ),
                  binop( Iop_And32, mkexpr( lmexp_10_mask ), mkU32( 28 ) ) ) );
   assign( lmd_9_val,
           binop( Iop_Or32,
                  binop( Iop_Or32,
                         binop( Iop_And32,
                                mkexpr( lmexp_00_mask ),
                                mkU32( 25 ) ),
                         binop( Iop_And32,
                                mkexpr( lmexp_01_mask ),
                                mkU32( 27 ) ) ),
                  binop( Iop_And32, mkexpr( lmexp_10_mask ), mkU32( 29 ) ) ) );

   /* generate the result from the possible LMD values */
   return binop( Iop_Or32,
                 binop( Iop_Or32,
                        binop( Iop_And32,
                               mkexpr( lmd_07_mask ),
                               mkexpr( lmd_07_val ) ),
                        binop( Iop_And32,
                               mkexpr( lmd_8_mask ),
                               mkexpr( lmd_8_val ) ) ),
                 binop( Iop_And32, mkexpr( lmd_9_mask ), mkexpr( lmd_9_val ) ) );
}

static void Get_lmd( IRTemp * lmd, IRExpr * gfield_0_4 )
{
   /* Extract the exponent and the left most digit of the mantissa
    * from the G field bits [0:4].
    */
   IRTemp lmd_07_mask   = newTemp( Ity_I32 );
   IRTemp lmd_8_00_mask = newTemp( Ity_I32 );
   IRTemp lmd_8_01_mask = newTemp( Ity_I32 );
   IRTemp lmd_8_10_mask = newTemp( Ity_I32 );
   IRTemp lmd_9_00_mask = newTemp( Ity_I32 );
   IRTemp lmd_9_01_mask = newTemp( Ity_I32 );
   IRTemp lmd_9_10_mask = newTemp( Ity_I32 );

   IRTemp lmd_07_val = newTemp( Ity_I32 );
   IRTemp lmd_8_val  = newTemp( Ity_I32 );
   IRTemp lmd_9_val  = newTemp( Ity_I32 );

   /* The left most digit (LMD) encoding is as follows:
    *    lmd
    *   0 - 7    (lmexp << 3) | lmd
    *     8      0b11000 (24 decimal) if lme=0b00;
    *            0b11010 (26 decimal) if lme=0b01;
    *            0b11100 (28 decimal) if lme=0b10
    *     9      0b11001 (25 decimal) if lme=0b00;
    *            0b11011 (27 decimal) if lme=0b01;
    *            0b11101 (29 decimal) if lme=0b10;
    */

   /* Generate the masks for each condition of LMD and exponent bits */
   assign( lmd_07_mask,
           unop( Iop_1Sto32, binop( Iop_CmpLE32U,
                                    gfield_0_4,
                                    mkU32( BITS5(1,0,1,1,1) ) ) ) );
   assign( lmd_8_00_mask,
           unop( Iop_1Sto32, binop( Iop_CmpEQ32,
                                    gfield_0_4,
                                    mkU32( BITS5(1,1,0,0,0) ) ) ) );
   assign( lmd_8_01_mask,
           unop( Iop_1Sto32, binop( Iop_CmpEQ32,
                                    gfield_0_4,
                                    mkU32( BITS5(1,1,0,1,0) ) ) ) );
   assign( lmd_8_10_mask,
           unop( Iop_1Sto32, binop( Iop_CmpEQ32,
                                    gfield_0_4,
                                    mkU32( BITS5(1,1,1,0,0) ) ) ) );
   assign( lmd_9_00_mask,
           unop( Iop_1Sto32, binop( Iop_CmpEQ32,
                                    gfield_0_4,
                                    mkU32( BITS5(1,1,0,0,1) ) ) ) );
   assign( lmd_9_01_mask,
           unop( Iop_1Sto32, binop( Iop_CmpEQ32,
                                    gfield_0_4,
                                    mkU32( BITS5(1,1,0,1,1) ) ) ) );
   assign( lmd_9_10_mask,
           unop( Iop_1Sto32, binop( Iop_CmpEQ32,
                                    gfield_0_4,
                                    mkU32( BITS5(1,1,1,0,1) ) ) ) );

   /* Generate the values for each LMD condition, assuming the condition
    * is TRUE.
    */
   assign( lmd_07_val, binop( Iop_And32, gfield_0_4, mkU32( 0x7 ) ) );
   assign( lmd_8_val, mkU32( 0x8 ) );
   assign( lmd_9_val, mkU32( 0x9 ) );

   assign( *lmd,
           OR( OR3 ( AND( mkexpr( lmd_07_mask ), mkexpr( lmd_07_val ) ),
                     AND( mkexpr( lmd_8_00_mask ), mkexpr( lmd_8_val ) ),
                     AND( mkexpr( lmd_8_01_mask ), mkexpr( lmd_8_val ) )),
                     OR4( AND( mkexpr( lmd_8_10_mask ), mkexpr( lmd_8_val ) ),
                          AND( mkexpr( lmd_9_00_mask ), mkexpr( lmd_9_val ) ),
                          AND( mkexpr( lmd_9_01_mask ), mkexpr( lmd_9_val ) ),
                          AND( mkexpr( lmd_9_10_mask ), mkexpr( lmd_9_val ) )
                     ) ) );
}

#define DIGIT1_SHR 4    // shift digit 1 to bottom 4 bits
#define DIGIT2_SHR 8    // shift digit 2 to bottom 4 bits
#define DIGIT3_SHR 12
#define DIGIT4_SHR 16
#define DIGIT5_SHR 20
#define DIGIT6_SHR 24
#define DIGIT7_SHR 28

static IRExpr * bcd_digit_inval( IRExpr * bcd_u, IRExpr * bcd_l )
{
   /* 60-bit BCD string stored in two 32-bit values.  Check that each,
    * digit is a valid BCD number, i.e. less then 9.
    */
   IRTemp valid = newTemp( Ity_I32 );

   assign( valid,
           AND4( AND4 ( unop( Iop_1Sto32,
                              binop( Iop_CmpLE32U,
                                     binop( Iop_And32,
                                            bcd_l,
                                            mkU32 ( 0xF ) ),
                                      mkU32( 0x9 ) ) ),
                        unop( Iop_1Sto32,
                              binop( Iop_CmpLE32U,
                                     binop( Iop_And32,
                                            binop( Iop_Shr32,
                                                   bcd_l,
                                                   mkU8 ( DIGIT1_SHR ) ),
                                             mkU32 ( 0xF ) ),
                                      mkU32( 0x9 ) ) ),
                        unop( Iop_1Sto32,
                              binop( Iop_CmpLE32U,
                                     binop( Iop_And32,
                                            binop( Iop_Shr32,
                                                   bcd_l,
                                                   mkU8 ( DIGIT2_SHR ) ),
                                            mkU32 ( 0xF ) ),
                                      mkU32( 0x9 ) ) ),
                        unop( Iop_1Sto32,
                              binop( Iop_CmpLE32U,
                                     binop( Iop_And32,
                                            binop( Iop_Shr32,
                                                   bcd_l,
                                                   mkU8 ( DIGIT3_SHR ) ),
                                             mkU32 ( 0xF ) ),
                                      mkU32( 0x9 ) ) ) ),
                 AND4 ( unop( Iop_1Sto32,
                              binop( Iop_CmpLE32U,
                                     binop( Iop_And32,
                                            binop( Iop_Shr32,
                                                   bcd_l,
                                                   mkU8 ( DIGIT4_SHR ) ),
                                            mkU32 ( 0xF ) ),
                                     mkU32( 0x9 ) ) ),
                        unop( Iop_1Sto32,
                              binop( Iop_CmpLE32U,
                                     binop( Iop_And32,
                                            binop( Iop_Shr32,
                                                   bcd_l,
                                                   mkU8 ( DIGIT5_SHR ) ),
                                            mkU32 ( 0xF ) ),
                                     mkU32( 0x9 ) ) ),
                        unop( Iop_1Sto32,
                              binop( Iop_CmpLE32U,
                                     binop( Iop_And32,
                                            binop( Iop_Shr32,
                                                   bcd_l,
                                                   mkU8 ( DIGIT6_SHR ) ),
                                            mkU32 ( 0xF ) ),
                                     mkU32( 0x9 ) ) ),
                        unop( Iop_1Sto32,
                              binop( Iop_CmpLE32U,
                                     binop( Iop_And32,
                                            binop( Iop_Shr32,
                                                   bcd_l,
                                                   mkU8 ( DIGIT7_SHR ) ),
                                            mkU32 ( 0xF ) ),
                                     mkU32( 0x9 ) ) ) ),
                 AND4( unop( Iop_1Sto32,
                             binop( Iop_CmpLE32U,
                                    binop( Iop_And32,
                                           bcd_u,
                                           mkU32 ( 0xF ) ),
                                    mkU32( 0x9 ) ) ),
                       unop( Iop_1Sto32,
                             binop( Iop_CmpLE32U,
                                    binop( Iop_And32,
                                           binop( Iop_Shr32,
                                                  bcd_u,
                                                  mkU8 ( DIGIT1_SHR ) ),
                                           mkU32 ( 0xF ) ),
                                    mkU32( 0x9 ) ) ),
                       unop( Iop_1Sto32,
                             binop( Iop_CmpLE32U,
                                    binop( Iop_And32,
                                           binop( Iop_Shr32,
                                                  bcd_u,
                                                  mkU8 ( DIGIT2_SHR ) ),
                                           mkU32 ( 0xF ) ),
                                    mkU32( 0x9 ) ) ),
                       unop( Iop_1Sto32,
                             binop( Iop_CmpLE32U,
                                    binop( Iop_And32,
                                           binop( Iop_Shr32,
                                                  bcd_u,
                                                  mkU8 ( DIGIT3_SHR ) ),
                                           mkU32 ( 0xF ) ),
                                    mkU32( 0x9 ) ) ) ),
                 AND4( unop( Iop_1Sto32,
                             binop( Iop_CmpLE32U,
                                    binop( Iop_And32,
                                           binop( Iop_Shr32,
                                                  bcd_u,
                                                  mkU8 ( DIGIT4_SHR ) ),
                                           mkU32 ( 0xF ) ),
                                    mkU32( 0x9 ) ) ),
                       unop( Iop_1Sto32,
                             binop( Iop_CmpLE32U,
                                    binop( Iop_And32,
                                           binop( Iop_Shr32,
                                                  bcd_u,
                                                  mkU8 ( DIGIT5_SHR ) ),
                                           mkU32 ( 0xF ) ),
                                    mkU32( 0x9 ) ) ),
                       unop( Iop_1Sto32,
                             binop( Iop_CmpLE32U,
                                    binop( Iop_And32,
                                           binop( Iop_Shr32,
                                                  bcd_u,
                                                  mkU8 ( DIGIT6_SHR ) ),
                                           mkU32 ( 0xF ) ),
                                    mkU32( 0x9 ) ) ),
                       unop( Iop_1Sto32,
                             binop( Iop_CmpLE32U,
                                    binop( Iop_And32,
                                           binop( Iop_Shr32,
                                                  bcd_u,
                                                  mkU8 ( DIGIT7_SHR ) ),
                                           mkU32 ( 0xF ) ),
                                    mkU32( 0x9 ) ) ) ) ) );
           
   return unop( Iop_Not32, mkexpr( valid ) );
}
#undef DIGIT1_SHR
#undef DIGIT2_SHR
#undef DIGIT3_SHR
#undef DIGIT4_SHR
#undef DIGIT5_SHR
#undef DIGIT6_SHR
#undef DIGIT7_SHR

static IRExpr * Generate_neg_sign_mask( IRExpr * sign )
{
   return binop( Iop_Or32,
                 unop( Iop_1Sto32, binop( Iop_CmpEQ32, sign, mkU32( 0xB ) ) ),
                 unop( Iop_1Sto32, binop( Iop_CmpEQ32, sign, mkU32( 0xD ) ) )
               );
}

static IRExpr * Generate_pos_sign_mask( IRExpr * sign )
{
   return binop( Iop_Or32,
                 binop( Iop_Or32,
                        unop( Iop_1Sto32,
                              binop( Iop_CmpEQ32, sign, mkU32( 0xA ) ) ),
                        unop( Iop_1Sto32,
                              binop( Iop_CmpEQ32, sign, mkU32( 0xC ) ) ) ),
                 binop( Iop_Or32,
                        unop( Iop_1Sto32,
                              binop( Iop_CmpEQ32, sign, mkU32( 0xE ) ) ),
                        unop( Iop_1Sto32,
                              binop( Iop_CmpEQ32, sign, mkU32( 0xF ) ) ) ) );
}

static IRExpr * Generate_sign_bit( IRExpr * pos_sign_mask,
                                   IRExpr * neg_sign_mask )
{
   return binop( Iop_Or32,
                 binop( Iop_And32, neg_sign_mask, mkU32( 0x80000000 ) ),
                 binop( Iop_And32, pos_sign_mask, mkU32( 0x00000000 ) ) );
}

static IRExpr * Generate_inv_mask( IRExpr * invalid_bcd_mask,
                                   IRExpr * pos_sign_mask,
                                   IRExpr * neg_sign_mask )
/* first argument is all 1's if the BCD string had an invalid digit in it. */
{
   return binop( Iop_Or32,
                 invalid_bcd_mask,
                 unop( Iop_1Sto32,
                       binop( Iop_CmpEQ32,
                              binop( Iop_Or32, pos_sign_mask, neg_sign_mask ),
                              mkU32( 0x0 ) ) ) );
}

static void Generate_132_bit_bcd_string( IRExpr * frBI64_hi, IRExpr * frBI64_lo,
                                         IRTemp * top_12_l, IRTemp * mid_60_u,
                                         IRTemp * mid_60_l, IRTemp * low_60_u,
                                         IRTemp * low_60_l)
{
   IRTemp tmplow60 = newTemp( Ity_I64 );
   IRTemp tmpmid60 = newTemp( Ity_I64 );
   IRTemp tmptop12 = newTemp( Ity_I64 );
   IRTemp low_50   = newTemp( Ity_I64 );
   IRTemp mid_50   = newTemp( Ity_I64 );
   IRTemp top_10   = newTemp( Ity_I64 );
   IRTemp top_12_u = newTemp( Ity_I32 ); // only needed for a dummy arg

   /* Convert the 110-bit densely packed BCD string to a 128-bit BCD string */

   /* low_50[49:0] = ((frBI64_lo[49:32]  << 14) | frBI64_lo[31:0]) */
   assign( low_50,
           binop( Iop_32HLto64,
                  binop( Iop_And32,
                         unop( Iop_64HIto32, frBI64_lo ),
                         mkU32( 0x3FFFF ) ),
                         unop( Iop_64to32, frBI64_lo ) ) );

   /* Convert the 50 bit densely packed BCD string to a 60 bit
    * BCD string.
    */
   assign( tmplow60, unop( Iop_DPBtoBCD, mkexpr( low_50 ) ) );
   assign( *low_60_u, unop( Iop_64HIto32, mkexpr( tmplow60 ) ) );
   assign( *low_60_l, unop( Iop_64to32, mkexpr( tmplow60 ) ) );

   /* mid_50[49:0] =  ((frBI64_hi[35:32] << 14) | frBI64_hi[31:18]) |
    *                 ((frBI64_hi[17:0]  << 14) | frBI64_lo[63:50])
    */
   assign( mid_50,
           binop( Iop_32HLto64,
                  binop( Iop_Or32,
                         binop( Iop_Shl32,
                                binop( Iop_And32,
                                       unop( Iop_64HIto32, frBI64_hi ),
                                       mkU32( 0xF ) ),
                                mkU8( 14 ) ),
                         binop( Iop_Shr32,
                                unop( Iop_64to32, frBI64_hi ),
                                mkU8( 18 ) ) ),
                  binop( Iop_Or32,
                         binop( Iop_Shl32,
                                unop( Iop_64to32, frBI64_hi ),
                                mkU8( 14 ) ),
                         binop( Iop_Shr32,
                                unop( Iop_64HIto32, frBI64_lo ),
                                mkU8( 18 ) ) ) ) );

   /* Convert the 50 bit densely packed BCD string to a 60 bit
    * BCD string.
    */
   assign( tmpmid60, unop( Iop_DPBtoBCD, mkexpr( mid_50 ) ) );
   assign( *mid_60_u, unop( Iop_64HIto32, mkexpr( tmpmid60 ) ) );
   assign( *mid_60_l, unop( Iop_64to32, mkexpr( tmpmid60 ) ) );

   /* top_10[49:0] = frBI64_hi[45:36]) |  */
   assign( top_10,
           binop( Iop_32HLto64,
                  mkU32( 0 ),
                  binop( Iop_And32,
                         binop( Iop_Shr32,
                                unop( Iop_64HIto32, frBI64_hi ),
                                mkU8( 4 ) ),
                         mkU32( 0x3FF ) ) ) );

   /* Convert the 10 bit densely packed BCD string to a 12 bit
    * BCD string.
    */
   assign( tmptop12, unop( Iop_DPBtoBCD, mkexpr( top_10 ) ) );
   assign( top_12_u, unop( Iop_64HIto32, mkexpr( tmptop12 ) ) );
   assign( *top_12_l, unop( Iop_64to32, mkexpr( tmptop12 ) ) );
}

static void Count_zeros( int start, IRExpr * init_cnt, IRExpr * init_flag,
                         IRTemp * final_cnt, IRTemp * final_flag,
                         IRExpr * string )
{
   IRTemp cnt[MAX_DIGITS_IN_STRING + 1];IRTemp flag[MAX_DIGITS_IN_STRING+1];
   int digits = MAX_DIGITS_IN_STRING;
   int i;

   cnt[start-1] = newTemp( Ity_I8 );
   flag[start-1] = newTemp( Ity_I8 );
   assign( cnt[start-1], init_cnt);
   assign( flag[start-1], init_flag);

   for ( i = start; i <= digits; i++) {
      cnt[i] = newTemp( Ity_I8 );
      flag[i] = newTemp( Ity_I8 );
      assign( cnt[i],
              binop( Iop_Add8,
                     mkexpr( cnt[i-1] ),
                     binop(Iop_And8,
                           unop( Iop_1Uto8,
                                 binop(Iop_CmpEQ32,
                                       binop(Iop_And32,
                                             string,
                                             mkU32( 0xF <<
                                                    ( ( digits - i ) * 4) ) ),
                                       mkU32( 0 ) ) ),
                           binop( Iop_Xor8, /* complement flag */
                                  mkexpr( flag[i - 1] ),
                                  mkU8( 0xFF ) ) ) ) );

      /* set flag to 1 if digit was not a zero */
      assign( flag[i],
              binop(Iop_Or8,
                    unop( Iop_1Sto8,
                          binop(Iop_CmpNE32,
                                binop(Iop_And32,
                                      string,
                                      mkU32( 0xF <<
                                             ( (digits - i) * 4) ) ),
                                mkU32( 0 ) ) ),
                    mkexpr( flag[i - 1] ) ) );
   }

   *final_cnt = cnt[digits];
   *final_flag = flag[digits];
}

static IRExpr * Count_leading_zeros_60( IRExpr * lmd, IRExpr * upper_28,
                                        IRExpr * low_32 )
{
   IRTemp num_lmd    = newTemp( Ity_I8 );
   IRTemp num_upper  = newTemp( Ity_I8 );
   IRTemp num_low    = newTemp( Ity_I8 );
   IRTemp lmd_flag   = newTemp( Ity_I8 );
   IRTemp upper_flag = newTemp( Ity_I8 );
   IRTemp low_flag   = newTemp( Ity_I8 );

   assign( num_lmd, unop( Iop_1Uto8, binop( Iop_CmpEQ32, lmd, mkU32( 0 ) ) ) );
   assign( lmd_flag, unop( Iop_Not8, mkexpr( num_lmd ) ) );

   Count_zeros( 2,
                mkexpr( num_lmd ),
                mkexpr( lmd_flag ),
                &num_upper,
                &upper_flag,
                upper_28 );

   Count_zeros( 1,
                mkexpr( num_upper ),
                mkexpr( upper_flag ),
                &num_low,
                &low_flag,
                low_32 );

   return mkexpr( num_low );
}

static IRExpr * Count_leading_zeros_128( IRExpr * lmd, IRExpr * top_12_l,
                                         IRExpr * mid_60_u, IRExpr * mid_60_l,
                                         IRExpr * low_60_u, IRExpr * low_60_l)
{
   IRTemp num_lmd   = newTemp( Ity_I8 );
   IRTemp num_top   = newTemp( Ity_I8 );
   IRTemp num_mid_u = newTemp( Ity_I8 );
   IRTemp num_mid_l = newTemp( Ity_I8 );
   IRTemp num_low_u = newTemp( Ity_I8 );
   IRTemp num_low_l = newTemp( Ity_I8 );

   IRTemp lmd_flag   = newTemp( Ity_I8 );
   IRTemp top_flag   = newTemp( Ity_I8 );
   IRTemp mid_u_flag = newTemp( Ity_I8 );
   IRTemp mid_l_flag = newTemp( Ity_I8 );
   IRTemp low_u_flag = newTemp( Ity_I8 );
   IRTemp low_l_flag = newTemp( Ity_I8 );

   /* Check the LMD, digit 34, to see if it is zero. */
   assign( num_lmd, unop( Iop_1Uto8, binop( Iop_CmpEQ32, lmd, mkU32( 0 ) ) ) );

   assign( lmd_flag, unop( Iop_Not8, mkexpr( num_lmd ) ) );

   Count_zeros( 6,
                mkexpr( num_lmd ),
                mkexpr( lmd_flag ),
                &num_top,
                &top_flag,
                top_12_l );

   Count_zeros( 2,
                mkexpr( num_top ),
                mkexpr( top_flag ),
                &num_mid_u,
                &mid_u_flag,
                binop( Iop_Or32,
                       binop( Iop_Shl32, mid_60_u, mkU8( 2 ) ),
                       binop( Iop_Shr32, mid_60_l, mkU8( 30 ) ) ) );

   Count_zeros( 1,
                mkexpr( num_mid_u ),
                mkexpr( mid_u_flag ),
                &num_mid_l,
                &mid_l_flag,
                mid_60_l );

   Count_zeros( 2,
                mkexpr( num_mid_l ),
                mkexpr( mid_l_flag ),
                &num_low_u,
                &low_u_flag,
                binop( Iop_Or32,
                       binop( Iop_Shl32, low_60_u, mkU8( 2 ) ),
                       binop( Iop_Shr32, low_60_l, mkU8( 30 ) ) ) );

   Count_zeros( 1,
                mkexpr( num_low_u ),
                mkexpr( low_u_flag ),
                &num_low_l,
                &low_l_flag,
                low_60_l );

   return mkexpr( num_low_l );
}

static IRExpr * Check_unordered(IRExpr * val)
{
   IRTemp gfield0to5 = newTemp( Ity_I32 );

   /* Extract G[0:4] */
   assign( gfield0to5,
           binop( Iop_And32,
                  binop( Iop_Shr32, unop( Iop_64HIto32, val ), mkU8( 26 ) ),
                  mkU32( 0x1F ) ) );

   /* Check for unordered, return all 1'x if true */
   return binop( Iop_Or32, /* QNaN check */
                 unop( Iop_1Sto32,
                       binop( Iop_CmpEQ32,
                              mkexpr( gfield0to5 ),
                              mkU32( 0x1E ) ) ),
                 unop( Iop_1Sto32, /* SNaN check */
                       binop( Iop_CmpEQ32,
                              mkexpr( gfield0to5 ),
                              mkU32( 0x1F ) ) ) );
}

#undef AND
#undef AND4
#undef OR
#undef OR3
#undef OR4
#undef NOT
#undef SHR
#undef SHL
#undef BITS5

/*------------------------------------------------------------*/
/*--- Decimal Floating Point (DFP) instruction translation ---*/
/*------------------------------------------------------------*/

/* DFP Arithmetic instructions */
static Bool dis_dfp_arith( UInt prefix, UInt theInstr )
{
   UInt opc2 = ifieldOPClo10( theInstr );
   UChar frS_addr = ifieldRegDS( theInstr );
   UChar frA_addr = ifieldRegA( theInstr );
   UChar frB_addr = ifieldRegB( theInstr );
   UChar flag_rC = ifieldBIT0( theInstr );

   IRTemp frA = newTemp( Ity_D64 );
   IRTemp frB = newTemp( Ity_D64 );
   IRTemp frS = newTemp( Ity_D64 );
   IRExpr* round = get_IR_roundingmode_DFP();

   /* By default, if flag_RC is set, we will clear cr1 after the
    * operation.  In reality we should set cr1 to indicate the
    * exception status of the operation, but since we're not
    * simulating exceptions, the exception status will appear to be
    * zero.  Hence cr1 should be cleared if this is a . form insn.
    */
   Bool clear_CR1 = True;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   assign( frA, getDReg( frA_addr ) );
   assign( frB, getDReg( frB_addr ) );

   switch (opc2) {
   case 0x2: // dadd
      DIP( "dadd%s fr%u,fr%u,fr%u\n",
           flag_rC ? ".":"", frS_addr, frA_addr, frB_addr );
      assign( frS, triop( Iop_AddD64, round, mkexpr( frA ), mkexpr( frB ) ) );
      break;
   case 0x202: // dsub
      DIP( "dsub%s fr%u,fr%u,fr%u\n",
           flag_rC ? ".":"", frS_addr, frA_addr, frB_addr );
      assign( frS, triop( Iop_SubD64, round, mkexpr( frA ), mkexpr( frB ) ) );
      break;
   case 0x22: // dmul
      DIP( "dmul%s fr%u,fr%u,fr%u\n",
           flag_rC ? ".":"", frS_addr, frA_addr, frB_addr );
      assign( frS, triop( Iop_MulD64, round, mkexpr( frA ), mkexpr( frB ) ) );
      break;
   case 0x222: // ddiv
      DIP( "ddiv%s fr%u,fr%u,fr%u\n",
           flag_rC ? ".":"", frS_addr, frA_addr, frB_addr );
      assign( frS, triop( Iop_DivD64, round, mkexpr( frA ), mkexpr( frB ) ) );
      break;
   }

   putDReg( frS_addr, mkexpr( frS ) );

   if (flag_rC && clear_CR1) {
      putCR321( 1, mkU8( 0 ) );
      putCR0( 1, mkU8( 0 ) );
   }

   return True;
}

/* Quad DFP Arithmetic instructions */
static Bool dis_dfp_arithq( UInt prefix, UInt theInstr )
{
   UInt opc2 = ifieldOPClo10( theInstr );
   UChar frS_addr = ifieldRegDS( theInstr );
   UChar frA_addr = ifieldRegA( theInstr );
   UChar frB_addr = ifieldRegB( theInstr );
   UChar flag_rC = ifieldBIT0( theInstr );

   IRTemp frA = newTemp( Ity_D128 );
   IRTemp frB = newTemp( Ity_D128 );
   IRTemp frS = newTemp( Ity_D128 );
   IRExpr* round = get_IR_roundingmode_DFP();

   /* By default, if flag_RC is set, we will clear cr1 after the
    * operation.  In reality we should set cr1 to indicate the
    * exception status of the operation, but since we're not
    * simulating exceptions, the exception status will appear to be
    * zero.  Hence cr1 should be cleared if this is a . form insn.
    */
   Bool clear_CR1 = True;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   assign( frA, getDReg_pair( frA_addr ) );
   assign( frB, getDReg_pair( frB_addr ) );

   switch (opc2) {
   case 0x2: // daddq
      DIP( "daddq%s fr%u,fr%u,fr%u\n",
           flag_rC ? ".":"", frS_addr, frA_addr, frB_addr );
      assign( frS, triop( Iop_AddD128, round, mkexpr( frA ), mkexpr( frB ) ) );
      break;
   case 0x202: // dsubq
      DIP( "dsubq%s fr%u,fr%u,fr%u\n",
           flag_rC ? ".":"", frS_addr, frA_addr, frB_addr );
      assign( frS, triop( Iop_SubD128, round, mkexpr( frA ), mkexpr( frB ) ) );
      break;
   case 0x22: // dmulq
      DIP( "dmulq%s fr%u,fr%u,fr%u\n",
           flag_rC ? ".":"", frS_addr, frA_addr, frB_addr );
      assign( frS, triop( Iop_MulD128, round, mkexpr( frA ), mkexpr( frB ) ) );
      break;
   case 0x222: // ddivq
      DIP( "ddivq%s fr%u,fr%u,fr%u\n",
           flag_rC ? ".":"", frS_addr, frA_addr, frB_addr );
      assign( frS, triop( Iop_DivD128, round, mkexpr( frA ), mkexpr( frB ) ) );
      break;
   }

   putDReg_pair( frS_addr, mkexpr( frS ) );

   if (flag_rC && clear_CR1) {
      putCR321( 1, mkU8( 0 ) );
      putCR0( 1, mkU8( 0 ) );
   }

   return True;
}

/* DFP 64-bit logical shift instructions  */
static Bool dis_dfp_shift( UInt prefix, UInt theInstr ) {
   UInt opc2       = ifieldOPClo9( theInstr );
   UChar frS_addr  = ifieldRegDS( theInstr );
   UChar frA_addr  = ifieldRegA( theInstr );
   UChar shift_val = IFIELD(theInstr, 10, 6);
   UChar flag_rC   = ifieldBIT0( theInstr );

   IRTemp frA = newTemp( Ity_D64 );
   IRTemp frS = newTemp( Ity_D64 );
   Bool clear_CR1 = True;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   assign( frA, getDReg( frA_addr ) );

   switch (opc2) {
   case 0x42: // dscli
      DIP( "dscli%s fr%u,fr%u,%u\n",
           flag_rC ? ".":"", frS_addr, frA_addr, shift_val );
      assign( frS, binop( Iop_ShlD64, mkexpr( frA ), mkU8( shift_val ) ) );
      break;
   case 0x62: // dscri
      DIP( "dscri%s fr%u,fr%u,%u\n",
           flag_rC ? ".":"", frS_addr, frA_addr, shift_val );
      assign( frS, binop( Iop_ShrD64, mkexpr( frA ), mkU8( shift_val ) ) );
      break;
   }

   putDReg( frS_addr, mkexpr( frS ) );

   if (flag_rC && clear_CR1) {
      putCR321( 1, mkU8( 0 ) );
      putCR0( 1, mkU8( 0 ) );
   }

   return True;
}

/* Quad DFP  logical shift instructions  */
static Bool dis_dfp_shiftq( UInt prefix, UInt theInstr ) {
   UInt opc2       = ifieldOPClo9( theInstr );
   UChar frS_addr  = ifieldRegDS( theInstr );
   UChar frA_addr  = ifieldRegA( theInstr );
   UChar shift_val = IFIELD(theInstr, 10, 6);
   UChar flag_rC   = ifieldBIT0( theInstr );

   IRTemp frA = newTemp( Ity_D128 );
   IRTemp frS = newTemp( Ity_D128 );
   Bool clear_CR1 = True;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   assign( frA, getDReg_pair( frA_addr ) );

   switch (opc2) {
   case 0x42: // dscliq
      DIP( "dscliq%s fr%u,fr%u,%u\n",
           flag_rC ? ".":"", frS_addr, frA_addr, shift_val );
      assign( frS, binop( Iop_ShlD128, mkexpr( frA ), mkU8( shift_val ) ) );
      break;
   case 0x62: // dscriq
      DIP( "dscriq%s fr%u,fr%u,%u\n",
           flag_rC ? ".":"", frS_addr, frA_addr, shift_val );
      assign( frS, binop( Iop_ShrD128, mkexpr( frA ), mkU8( shift_val ) ) );
      break;
   }

   putDReg_pair( frS_addr, mkexpr( frS ) );

   if (flag_rC && clear_CR1) {
      putCR321( 1, mkU8( 0 ) );
      putCR0( 1, mkU8( 0 ) );
   }

   return True;
}

/* DFP 64-bit format conversion instructions */
static Bool dis_dfp_fmt_conv( UInt prefix, UInt theInstr ) {
   UInt opc2      = ifieldOPClo10( theInstr );
   UChar frS_addr = ifieldRegDS( theInstr );
   UChar frB_addr = ifieldRegB( theInstr );
   IRExpr* round  = get_IR_roundingmode_DFP();
   UChar flag_rC  = ifieldBIT0( theInstr );
   IRTemp frB;
   IRTemp frS;
   Bool clear_CR1 = True;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   switch (opc2) {
   case 0x102: //dctdp
      DIP( "dctdp%s fr%u,fr%u\n",
           flag_rC ? ".":"", frS_addr, frB_addr );

      frB = newTemp( Ity_D32 );
      frS = newTemp( Ity_D64 );
      assign( frB, getDReg32( frB_addr ) );
      assign( frS, unop( Iop_D32toD64, mkexpr( frB ) ) );
      putDReg( frS_addr, mkexpr( frS ) );
      break;
   case 0x302: // drsp
      DIP( "drsp%s fr%u,fr%u\n",
           flag_rC ? ".":"", frS_addr, frB_addr );
      frB = newTemp( Ity_D64 );
      frS = newTemp( Ity_D32 );
      assign( frB, getDReg( frB_addr ) );
      assign( frS, binop( Iop_D64toD32, round, mkexpr( frB ) ) );
      putDReg32( frS_addr, mkexpr( frS ) );
      break;
   case 0x122: // dctfix
      {
         IRTemp tmp = newTemp( Ity_I64 );

         DIP( "dctfix%s fr%u,fr%u\n",
              flag_rC ? ".":"", frS_addr, frB_addr );
         frB = newTemp( Ity_D64 );
         frS = newTemp( Ity_D64 );
         assign( frB, getDReg( frB_addr ) );
         assign( tmp, binop( Iop_D64toI64S, round, mkexpr( frB ) ) );
         assign( frS, unop( Iop_ReinterpI64asD64, mkexpr( tmp ) ) );
         putDReg( frS_addr, mkexpr( frS ) );
      }
      break;
   case 0x322: // dcffix
      DIP( "dcffix%s fr%u,fr%u\n",
           flag_rC ? ".":"", frS_addr, frB_addr );
      frB = newTemp( Ity_D64 );
      frS = newTemp( Ity_D64 );
      assign( frB, getDReg( frB_addr ) );
      assign( frS, binop( Iop_I64StoD64,
                          round,
                          unop( Iop_ReinterpD64asI64, mkexpr( frB ) ) ) );
      putDReg( frS_addr, mkexpr( frS ) );
      break;
   }

   if (flag_rC && clear_CR1) {
      putCR321( 1, mkU8( 0 ) );
      putCR0( 1, mkU8( 0 ) );
   }

   return True;
}

/* Quad DFP format conversion instructions */
static Bool dis_dfp_fmt_convq( UInt prefix, UInt theInstr,
                               const VexAbiInfo* vbi ) {
   UInt opc2      = ifieldOPClo10( theInstr );
   UChar frS_addr = ifieldRegDS( theInstr );
   UChar frB_addr = ifieldRegB( theInstr );
   IRExpr* round  = get_IR_roundingmode_DFP();
   IRTemp frB64   = newTemp( Ity_D64 );
   IRTemp frB128  = newTemp( Ity_D128 );
   IRTemp frS64   = newTemp( Ity_D64 );
   IRTemp frS128  = newTemp( Ity_D128 );
   UChar flag_rC  = ifieldBIT0( theInstr );
   Bool clear_CR1 = True;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   switch (opc2) {
   case 0x102: // dctqpq
      DIP( "dctqpq%s fr%u,fr%u\n",
           flag_rC ? ".":"", frS_addr, frB_addr );
      assign( frB64, getDReg( frB_addr ) );
      assign( frS128, unop( Iop_D64toD128, mkexpr( frB64 ) ) );
      putDReg_pair( frS_addr, mkexpr( frS128 ) );
      break;
   case 0x122: // dctfixq
      {
         IRTemp tmp = newTemp( Ity_I64 );

         DIP( "dctfixq%s fr%u,fr%u\n",
              flag_rC ? ".":"", frS_addr, frB_addr );
         assign( frB128, getDReg_pair( frB_addr ) );
         assign( tmp, binop( Iop_D128toI64S, round, mkexpr( frB128 ) ) );
         assign( frS64, unop( Iop_ReinterpI64asD64, mkexpr( tmp ) ) );
         putDReg( frS_addr, mkexpr( frS64 ) );
      }
      break;
   case 0x302: //drdpq
      DIP( "drdpq%s fr%u,fr%u\n",
           flag_rC ? ".":"", frS_addr, frB_addr );
      assign( frB128, getDReg_pair( frB_addr ) );
      assign( frS64, binop( Iop_D128toD64, round, mkexpr( frB128 ) ) );
      putDReg( frS_addr, mkexpr( frS64 ) );
      break;
   case 0x322: // dcffixq
     {
      /* Have to introduce an IOP for this instruction so it will work
       * on POWER 6 because emulating the instruction requires a POWER 7
       * DFP instruction in the emulation code.
       */
      DIP( "dcffixq%s fr%u,fr%u\n",
           flag_rC ? ".":"", frS_addr, frB_addr );
      assign( frB64, getDReg( frB_addr ) );
      assign( frS128, unop( Iop_I64StoD128,
                            unop( Iop_ReinterpD64asI64,
                                  mkexpr( frB64 ) ) ) );
      putDReg_pair( frS_addr, mkexpr( frS128 ) );
      break;
     }

   case 0x3E2:
     {
        Int opc3 = IFIELD( theInstr, 16, 5 );

        flag_rC = 0;  // These instructions do not set condition codes.

        if (opc3 == 0) {           // dcffixqq
           IRTemp tmpD128 = newTemp( Ity_D128 );
           IRTemp vB_src  = newTemp( Ity_V128 );

           DIP( "dcffixqq fr%u,v%u\n", frS_addr, frB_addr );

           assign( vB_src, getVReg( frB_addr ));
           assign( tmpD128, binop( Iop_I128StoD128, round,
                                   unop( Iop_ReinterpV128asI128,
                                         mkexpr( vB_src ) ) ) );
           /* tmp128 is a Dfp 128 value which is held in a hi/lo 64-bit values.
            */
           generate_store_DFP_FPRF_value( Ity_D128, mkexpr( tmpD128 ), vbi);
           putDReg_pair( frS_addr, mkexpr( tmpD128 ) );

        } else if (opc3 == 1) {    // dctfixqq
           IRTemp tmp128 = newTemp(Ity_I128);

           DIP( "dctfixqq v%u,fr%u\n", frS_addr, frB_addr );
           assign( tmp128, binop( Iop_D128toI128S, round,
                                  getDReg_pair( frB_addr ) ) );

           putVReg( frS_addr,
                    unop( Iop_ReinterpI128asV128, mkexpr( tmp128 ) ) );

        } else {
           vex_printf("ERROR: dis_dfp_fmt_convq unknown opc3 = %d value.\n",
                      opc3);
           return False;
        }
     }
     break;
   }

   if (flag_rC && clear_CR1) {
      putCR321( 1, mkU8( 0 ) );
      putCR0( 1, mkU8( 0 ) );
   }

   return True;
}

static Bool dis_dfp_round( UInt prefix, UInt theInstr ) {
   UChar frS_addr = ifieldRegDS(theInstr);
   UChar R        = IFIELD(theInstr, 16, 1);
   UChar RMC      = IFIELD(theInstr, 9, 2);
   UChar frB_addr = ifieldRegB( theInstr );
   UChar flag_rC  = ifieldBIT0( theInstr );
   IRTemp frB     = newTemp( Ity_D64 );
   IRTemp frS     = newTemp( Ity_D64 );
   UInt opc2      = ifieldOPClo8( theInstr );
   Bool clear_CR1 = True;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   switch (opc2) {
   /* drintn, is the same as drintx.  The only difference is this
    * instruction does not generate an exception for an inexact operation.
    * Currently not supporting inexact exceptions.
    */
   case 0x63: // drintx
   case 0xE3: // drintn
      DIP( "drintx/drintn%s fr%u,fr%u\n",
           flag_rC ? ".":"", frS_addr, frB_addr );

      /* NOTE, this instruction takes a DFP value and rounds to the
       * neares floating point integer value, i.e. fractional part
       * is zero.  The result is a floating point number.
       */
      /* pass the value of R and RMC in the same field */
      assign( frB, getDReg( frB_addr ) );
      assign( frS, binop( Iop_RoundD64toInt,
                          mkU32( ( R << 3 ) | RMC ),
                          mkexpr( frB ) ) );
      putDReg( frS_addr, mkexpr( frS ) );
      break;
   default:
      vex_printf("dis_dfp_round(ppc)(opc2)\n");
      return False;
   }

   if (flag_rC && clear_CR1) {
      putCR321( 1, mkU8( 0 ) );
      putCR0( 1, mkU8( 0 ) );
   }

   return True;
}

static Bool dis_dfp_roundq( UInt prefix, UInt theInstr ) {
   UChar frS_addr = ifieldRegDS( theInstr );
   UChar frB_addr = ifieldRegB( theInstr );
   UChar R = IFIELD(theInstr, 16, 1);
   UChar RMC = IFIELD(theInstr, 9, 2);
   UChar flag_rC = ifieldBIT0( theInstr );
   IRTemp frB = newTemp( Ity_D128 );
   IRTemp frS = newTemp( Ity_D128 );
   Bool clear_CR1 = True;
   UInt opc2 = ifieldOPClo8( theInstr );

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   switch (opc2) {
   /* drintnq, is the same as drintxq.  The only difference is this
    * instruction does not generate an exception for an inexact operation.
    * Currently not supporting inexact exceptions.
    */
   case 0x63: // drintxq
   case 0xE3: // drintnq
      DIP( "drintxq/drintnq%s fr%u,fr%u\n",
           flag_rC ? ".":"", frS_addr, frB_addr );

      /* pass the value of R and RMC in the same field */
      assign( frB, getDReg_pair( frB_addr ) );
      assign( frS, binop( Iop_RoundD128toInt,
                          mkU32( ( R << 3 ) | RMC ),
                          mkexpr( frB ) ) );
      putDReg_pair( frS_addr, mkexpr( frS ) );
      break;
   default:
      vex_printf("dis_dfp_roundq(ppc)(opc2)\n");
      return False;
   }

   if (flag_rC && clear_CR1) {
      putCR321( 1, mkU8( 0 ) );
      putCR0( 1, mkU8( 0 ) );
   }

   return True;
}

static Bool dis_dfp_quantize_sig_rrnd( UInt prefix, UInt theInstr ) {
   UInt opc2 = ifieldOPClo8( theInstr );
   UChar frS_addr = ifieldRegDS( theInstr );
   UChar frA_addr = ifieldRegA( theInstr );
   UChar frB_addr = ifieldRegB( theInstr );
   UChar flag_rC = ifieldBIT0( theInstr );
   UInt TE_value = IFIELD(theInstr, 16, 4);
   UInt TE_sign  = IFIELD(theInstr, 20, 1);
   UInt RMC = IFIELD(theInstr, 9, 2);
   IRTemp frA = newTemp( Ity_D64 );
   IRTemp frB = newTemp( Ity_D64 );
   IRTemp frS = newTemp( Ity_D64 );
   Bool clear_CR1 = True;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   assign( frB, getDReg( frB_addr ) );

   switch (opc2) {
   case 0x43: // dquai
      DIP( "dquai%s fr%u,fr%u,fr%u\n",
           flag_rC ? ".":"", frS_addr, frA_addr, frB_addr );
      IRTemp TE_I64 = newTemp( Ity_I64 );

      /* Generate a reference DFP value frA with the desired exponent
       * given by TE using significand from frB.  Need to add the bias
       * 398 to TE.  TE is stored as a 2's complement number.
       */
      if (TE_sign == 1) {
         /* Take 2's complement of the 5-bit value and subtract from bias. 
          *  Bias is adjusted for the +1 required when taking 2's complement.
          */
         assign( TE_I64,
                 unop( Iop_32Uto64,
                       binop( Iop_Sub32, mkU32( 397 ),
                              binop( Iop_And32, mkU32( 0xF ),
                                     unop( Iop_Not32, mkU32( TE_value ) )
                                     ) ) ) );

      } else {
          assign( TE_I64,
                  unop( Iop_32Uto64,
                        binop( Iop_Add32, mkU32( 398 ), mkU32( TE_value ) )
                        ) );
      }

      assign( frA, binop( Iop_InsertExpD64, mkexpr( TE_I64 ),
                          unop( Iop_ReinterpI64asD64, mkU64( 1 ) ) ) );

      assign( frS, triop( Iop_QuantizeD64,
                          mkU32( RMC ),
                          mkexpr( frA ),
                          mkexpr( frB ) ) );
      break;

   case 0x3: // dqua
      DIP( "dqua%s fr%u,fr%u,fr%u\n",
           flag_rC ? ".":"", frS_addr, frA_addr, frB_addr );
      assign( frA, getDReg( frA_addr ) );
      assign( frS, triop( Iop_QuantizeD64,
                          mkU32( RMC ),
                          mkexpr( frA ),
                          mkexpr( frB ) ) );
      break;
   case 0x23: // drrnd
      {
         IRTemp tmp = newTemp( Ity_I8 );

         DIP( "drrnd%s fr%u,fr%u,fr%u\n",
              flag_rC ? ".":"", frS_addr, frA_addr, frB_addr );
         assign( frA, getDReg( frA_addr ) );
         /* Iop_64to8 not supported in 32 bit mode, do it in two steps. */
         assign( tmp, unop( Iop_32to8,
                            unop( Iop_64to32,
                                  unop( Iop_ReinterpD64asI64,
                                        mkexpr( frA ) ) ) ) );
         assign( frS, triop( Iop_SignificanceRoundD64,
                             mkU32( RMC ),
                             mkexpr( tmp ),
                             mkexpr( frB ) ) );
      }
      break;
   default:
      vex_printf("dis_dfp_quantize_sig_rrnd(ppc)(opc2)\n");
      return False;
   }
   putDReg( frS_addr, mkexpr( frS ) );

   if (flag_rC && clear_CR1) {
      putCR321( 1, mkU8( 0 ) );
      putCR0( 1, mkU8( 0 ) );
   }

   return True;
}

static Bool dis_dfp_quantize_sig_rrndq( UInt prefix, UInt theInstr ) {
   UInt opc2 = ifieldOPClo8( theInstr );
   UChar frS_addr = ifieldRegDS( theInstr );
   UChar frA_addr = ifieldRegA( theInstr );
   UChar frB_addr = ifieldRegB( theInstr );
   UChar flag_rC = ifieldBIT0( theInstr );
   UInt TE_value = IFIELD(theInstr, 16, 4);
   UInt TE_sign  = IFIELD(theInstr, 20, 1);
   UInt RMC = IFIELD(theInstr, 9, 2);
   IRTemp frA = newTemp( Ity_D128 );
   IRTemp frB = newTemp( Ity_D128 );
   IRTemp frS = newTemp( Ity_D128 );
   Bool clear_CR1 = True;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   assign( frB, getDReg_pair( frB_addr ) );

   switch (opc2) {
   case 0x43: // dquaiq
      DIP( "dquaiq%s fr%u,fr%u,fr%u\n",
           flag_rC ? ".":"", frS_addr, frA_addr, frB_addr );
      IRTemp TE_I64 = newTemp( Ity_I64 );

      /* Generate a reference DFP value frA with the desired exponent
       * given by TE using significand of 1.  Need to add the bias
       * 6176 to TE.
       */
      if (TE_sign == 1) {
         /* Take 2's complement of the 5-bit value and subtract from bias. 
          *  Bias adjusted for the +1 required when taking 2's complement.
          */
         assign( TE_I64,
                 unop( Iop_32Uto64,
                       binop( Iop_Sub32, mkU32( 6175 ),
                              binop( Iop_And32, mkU32( 0xF ),
                                     unop( Iop_Not32, mkU32( TE_value ) )
                                     ) ) ) );

      } else {
         assign( TE_I64,
                 unop( Iop_32Uto64,
                       binop( Iop_Add32,
                             mkU32( 6176 ),
                             mkU32( TE_value ) ) ) );
      }

      assign( frA,
              binop( Iop_InsertExpD128, mkexpr( TE_I64 ),
                     unop( Iop_D64toD128,
                           unop( Iop_ReinterpI64asD64, mkU64( 1 ) ) ) ) );
      assign( frS, triop( Iop_QuantizeD128,
                          mkU32( RMC ),
                          mkexpr( frA ),
                          mkexpr( frB ) ) );
      break;
   case 0x3: // dquaq
      DIP( "dquaiq%s fr%u,fr%u,fr%u\n",
           flag_rC ? ".":"", frS_addr, frA_addr, frB_addr );
      assign( frA, getDReg_pair( frA_addr ) );
      assign( frS, triop( Iop_QuantizeD128,
                          mkU32( RMC ),
                          mkexpr( frA ),
                          mkexpr( frB ) ) );
      break;
   case 0x23: // drrndq
      {
         IRTemp tmp = newTemp( Ity_I8 );

         DIP( "drrndq%s fr%u,fr%u,fr%u\n",
              flag_rC ? ".":"", frS_addr, frA_addr, frB_addr );
         assign( frA, getDReg_pair( frA_addr ) );
         assign( tmp, unop( Iop_32to8,
                            unop( Iop_64to32,
                                  unop( Iop_ReinterpD64asI64,
                                        unop( Iop_D128HItoD64,
                                              mkexpr( frA ) ) ) ) ) );
         assign( frS, triop( Iop_SignificanceRoundD128,
                             mkU32( RMC ),
                             mkexpr( tmp ),
                             mkexpr( frB ) ) );
      }
      break;
   default:
      vex_printf("dis_dfp_quantize_sig_rrndq(ppc)(opc2)\n");
      return False;
   }
   putDReg_pair( frS_addr, mkexpr( frS ) );

   if (flag_rC && clear_CR1) {
      putCR321( 1, mkU8( 0 ) );
      putCR0( 1, mkU8( 0 ) );
   }

   return True;
}

static Bool dis_dfp_extract_insert( UInt prefix, UInt theInstr ) {
   UInt opc2 = ifieldOPClo10( theInstr );
   UChar frS_addr = ifieldRegDS( theInstr );
   UChar frA_addr = ifieldRegA( theInstr );
   UChar frB_addr = ifieldRegB( theInstr );
   UChar flag_rC = ifieldBIT0( theInstr );
   Bool clear_CR1 = True;

   IRTemp frA = newTemp( Ity_D64 );
   IRTemp frB = newTemp( Ity_D64 );
   IRTemp frS = newTemp( Ity_D64 );
   IRTemp tmp = newTemp( Ity_I64 );

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   assign( frA, getDReg( frA_addr ) );
   assign( frB, getDReg( frB_addr ) );

   switch (opc2) {
   case 0x162: // dxex
      DIP( "dxex%s fr%u,fr%u,fr%u\n",
           flag_rC ? ".":"", frS_addr, frA_addr, frB_addr );
      assign( tmp, unop( Iop_ExtractExpD64, mkexpr( frB ) ) );
      assign( frS, unop( Iop_ReinterpI64asD64, mkexpr( tmp ) ) );
      break;
   case 0x362: // diex
      DIP( "diex%s fr%u,fr%u,fr%u\n",
           flag_rC ? ".":"", frS_addr, frA_addr, frB_addr );
      assign( frS, binop( Iop_InsertExpD64,
                          unop( Iop_ReinterpD64asI64,
                                mkexpr( frA ) ),
                          mkexpr( frB ) ) );
      break;
   default:
      vex_printf("dis_dfp_extract_insert(ppc)(opc2)\n");
      return False;
   }

   putDReg( frS_addr, mkexpr( frS ) );

   if (flag_rC && clear_CR1) {
      putCR321( 1, mkU8( 0 ) );
      putCR0( 1, mkU8( 0 ) );
   }

   return True;
}

static Bool dis_dfp_extract_insertq( UInt prefix, UInt theInstr ) {
   UInt opc2 = ifieldOPClo10( theInstr );
   UChar frS_addr = ifieldRegDS( theInstr );
   UChar frA_addr = ifieldRegA( theInstr );
   UChar frB_addr = ifieldRegB( theInstr );
   UChar flag_rC = ifieldBIT0( theInstr );

   IRTemp frA   = newTemp( Ity_D64 );
   IRTemp frB   = newTemp( Ity_D128 );
   IRTemp frS64 = newTemp( Ity_D64 );
   IRTemp frS   = newTemp( Ity_D128 );
   IRTemp tmp   = newTemp( Ity_I64 );
   Bool clear_CR1 = True;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   assign( frB, getDReg_pair( frB_addr ) );

   switch (opc2) {
   case 0x162:  // dxexq
      DIP( "dxexq%s fr%u,fr%u\n",
           flag_rC ? ".":"", frS_addr,  frB_addr );
      /* Instruction actually returns a 64-bit result.  So as to be
       * consistent and not have to add a new struct, the emulation returns
       * the 64-bit result in the upper and lower register.
       */
      assign( tmp, unop( Iop_ExtractExpD128, mkexpr( frB ) ) );
      assign( frS64, unop( Iop_ReinterpI64asD64, mkexpr( tmp ) ) );
      putDReg( frS_addr, mkexpr( frS64 ) );
      break;
   case 0x362:  // diexq
      DIP( "diexq%s fr%u,fr%u,fr%u\n",
           flag_rC ? ".":"", frS_addr, frA_addr, frB_addr );
      assign( frA, getDReg( frA_addr ) );
      assign( frS, binop( Iop_InsertExpD128,
                          unop( Iop_ReinterpD64asI64, mkexpr( frA ) ),
                          mkexpr( frB ) ) );
      putDReg_pair( frS_addr, mkexpr( frS ) );
      break;
   default:
      vex_printf("dis_dfp_extract_insertq(ppc)(opc2)\n");
      return False;
   }

   if (flag_rC && clear_CR1) {
      putCR321( 1, mkU8( 0 ) );
      putCR0( 1, mkU8( 0 ) );
   }

   return True;
}

/* DFP 64-bit comparison instructions */
static Bool dis_dfp_compare( UInt prefix, UInt theInstr ) {
   /* X-Form */
   UChar crfD = toUChar( IFIELD( theInstr, 23, 3 ) ); // AKA BF
   UChar frA_addr = ifieldRegA( theInstr );
   UChar frB_addr = ifieldRegB( theInstr );
   UInt opc1 = ifieldOPC( theInstr );
   IRTemp frA;
   IRTemp frB;

   IRTemp ccIR = newTemp( Ity_I32 );
   IRTemp ccPPC32 = newTemp( Ity_I32 );

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   /* Note: Differences between dcmpu and dcmpo are only in exception
    flag settings, which aren't supported anyway. */
   switch (opc1) {
   case 0x3B: /* dcmpo and dcmpu, DFP 64-bit */
      DIP( "dcmpo %u,fr%u,fr%u\n", crfD, frA_addr, frB_addr );
      frA = newTemp( Ity_D64 );
      frB = newTemp( Ity_D64 );

      assign( frA, getDReg( frA_addr ) );
      assign( frB, getDReg( frB_addr ) );

      assign( ccIR, binop( Iop_CmpD64, mkexpr( frA ), mkexpr( frB ) ) );
      break;
   case 0x3F: /* dcmpoq and dcmpuq,DFP 128-bit */
      DIP( "dcmpoq %u,fr%u,fr%u\n", crfD, frA_addr, frB_addr );
      frA = newTemp( Ity_D128 );
      frB = newTemp( Ity_D128 );

      assign( frA, getDReg_pair( frA_addr ) );
      assign( frB, getDReg_pair( frB_addr ) );
      assign( ccIR, binop( Iop_CmpD128, mkexpr( frA ), mkexpr( frB ) ) );
      break;
   default:
      vex_printf("dis_dfp_compare(ppc)(opc2)\n");
      return False;
   }

   /* Map compare result from IR to PPC32 */
   /*
    FP cmp result | PPC | IR
    --------------------------
    UN            | 0x1 | 0x45
    EQ            | 0x2 | 0x40
    GT            | 0x4 | 0x00
    LT            | 0x8 | 0x01
    */

   assign( ccPPC32,
           binop( Iop_Shl32,
                  mkU32( 1 ),
                  unop( Iop_32to8,
                        binop( Iop_Or32,
                               binop( Iop_And32,
                                      unop( Iop_Not32,
                                            binop( Iop_Shr32,
                                                   mkexpr( ccIR ),
                                                   mkU8( 5 ) ) ),
                                      mkU32( 2 ) ),
                               binop( Iop_And32,
                                      binop( Iop_Xor32,
                                             mkexpr( ccIR ),
                                             binop( Iop_Shr32,
                                                    mkexpr( ccIR ),
                                                    mkU8( 6 ) ) ),
                                      mkU32( 1 ) ) ) ) ) );

   putGST_field( PPC_GST_CR, mkexpr( ccPPC32 ), crfD );
   putFPCC( mkexpr( ccPPC32 ) );
   return True;
}

/* Test class/group/exponent/significance instructions. */
static Bool dis_dfp_exponent_test ( UInt prefix, UInt theInstr )
{
   UChar frA_addr   = ifieldRegA( theInstr );
   UChar frB_addr   = ifieldRegB( theInstr );
   UChar crfD       = toUChar( IFIELD( theInstr, 23, 3 ) );
   IRTemp frA       = newTemp( Ity_D64 );
   IRTemp frB       = newTemp( Ity_D64 );
   IRTemp frA128    = newTemp( Ity_D128 );
   IRTemp frB128    = newTemp( Ity_D128 );
   UInt opc1        = ifieldOPC( theInstr );
   IRTemp gfield_A  = newTemp( Ity_I32 );
   IRTemp gfield_B  = newTemp( Ity_I32 );
   IRTemp gfield_mask   = newTemp( Ity_I32 );
   IRTemp exponent_A    = newTemp( Ity_I32 );
   IRTemp exponent_B    = newTemp( Ity_I32 );
   IRTemp A_NaN_true    = newTemp( Ity_I32 );
   IRTemp B_NaN_true    = newTemp( Ity_I32 );
   IRTemp A_inf_true    = newTemp( Ity_I32 );
   IRTemp B_inf_true    = newTemp( Ity_I32 );
   IRTemp A_equals_B    = newTemp( Ity_I32 );
   IRTemp finite_number = newTemp( Ity_I32 );
   IRTemp cc0 = newTemp( Ity_I32 );
   IRTemp cc1 = newTemp( Ity_I32 );
   IRTemp cc2 = newTemp( Ity_I32 );
   IRTemp cc3 = newTemp( Ity_I32 );
   IRTemp cc  = newTemp( Ity_I32 );

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   /* The dtstex and dtstexg instructions only differ in the size of the
    * exponent field.  The following switch statement takes care of the size
    * specific setup.  Once the value of the exponents, the G-field shift
    * and mask is setup the remaining code is identical.
    */
   switch (opc1) {
   case 0x3b: // dtstex       Extended instruction setup
      DIP("dtstex %u,r%u,r%d\n", crfD, frA_addr, frB_addr);
      assign( frA, getDReg( frA_addr ) );
      assign( frB, getDReg( frB_addr ) );
      assign( gfield_mask, mkU32( DFP_G_FIELD_LONG_MASK ) );
      assign(exponent_A, unop( Iop_64to32,
                               unop( Iop_ExtractExpD64,
                                     mkexpr( frA ) ) ) );
      assign(exponent_B, unop( Iop_64to32,
                               unop( Iop_ExtractExpD64,
                                     mkexpr( frB ) ) ) );
      break;

   case 0x3F: //  dtstexq      Quad instruction setup
      DIP("dtstexq %u,r%u,r%d\n", crfD, frA_addr, frB_addr);
      assign( frA128, getDReg_pair( frA_addr ) );
      assign( frB128, getDReg_pair( frB_addr ) );
      assign( frA, unop( Iop_D128HItoD64, mkexpr( frA128 ) ) );
      assign( frB, unop( Iop_D128HItoD64, mkexpr( frB128 ) ) );
      assign( gfield_mask, mkU32( DFP_G_FIELD_EXTND_MASK ) );
      assign( exponent_A, unop( Iop_64to32,
                                unop( Iop_ExtractExpD128,
                                      mkexpr( frA128 ) ) ) );
      assign( exponent_B, unop( Iop_64to32,
                                unop( Iop_ExtractExpD128,
                                      mkexpr( frB128 ) ) ) );
      break;
   default:
      vex_printf("dis_dfp_exponent_test(ppc)(opc2)\n");
      return False;
   }

   /* Extract the Gfield */
   assign( gfield_A, binop( Iop_And32,
                            mkexpr( gfield_mask ),
                            unop( Iop_64HIto32,
                                  unop( Iop_ReinterpD64asI64,
                                        mkexpr(frA) ) ) ) );

   assign( gfield_B, binop( Iop_And32,
                            mkexpr( gfield_mask ),
                            unop( Iop_64HIto32,
                                  unop( Iop_ReinterpD64asI64,
                                        mkexpr(frB) ) ) ) );

   /* check for NAN */
   assign( A_NaN_true, binop(Iop_Or32,
                             unop( Iop_1Sto32,
                                   binop( Iop_CmpEQ32,
                                          mkexpr( gfield_A ),
                                          mkU32( 0x7C000000 ) ) ),
                             unop( Iop_1Sto32,
                                   binop( Iop_CmpEQ32,
                                          mkexpr( gfield_A ),
                                          mkU32( 0x7E000000 ) )
                                   ) ) );
   assign( B_NaN_true, binop(Iop_Or32,
                             unop( Iop_1Sto32,
                                   binop( Iop_CmpEQ32,
                                          mkexpr( gfield_B ),
                                          mkU32( 0x7C000000 ) ) ),
                             unop( Iop_1Sto32,
                                   binop( Iop_CmpEQ32,
                                          mkexpr( gfield_B ),
                                          mkU32( 0x7E000000 ) )
                             ) ) );

   /* check for infinity */ 
   assign( A_inf_true,
           unop( Iop_1Sto32,
                 binop( Iop_CmpEQ32,
                        mkexpr( gfield_A ),
                        mkU32( 0x78000000 ) ) ) );

   assign( B_inf_true,
           unop( Iop_1Sto32,
                 binop( Iop_CmpEQ32,
                        mkexpr( gfield_B ),
                        mkU32( 0x78000000 ) ) ) );

   assign( finite_number,
           unop( Iop_Not32,
                 binop( Iop_Or32,
                        binop( Iop_Or32,
                               mkexpr( A_NaN_true ),
                               mkexpr( B_NaN_true ) ),
                        binop( Iop_Or32,
                               mkexpr( A_inf_true ),
                               mkexpr( B_inf_true ) ) ) ) );

   /* Calculate the condition code bits
    * If QNaN,SNaN, +infinity, -infinity then cc0, cc1 and cc2 are zero
    * regardless of the value of the comparisons and cc3 is 1.  Otherwise,
    * cc0, cc1 and cc0 reflect the results of the comparisons.
    */
   assign( A_equals_B,
           binop( Iop_Or32,
                  unop( Iop_1Uto32,
                  binop( Iop_CmpEQ32,
                         mkexpr( exponent_A ),
                         mkexpr( exponent_B ) ) ),
                  binop( Iop_Or32,
                         binop( Iop_And32,
                                mkexpr( A_inf_true ),
                                mkexpr( B_inf_true ) ),
                         binop( Iop_And32,
                                mkexpr( A_NaN_true ),
                                mkexpr( B_NaN_true ) ) ) ) );

   assign( cc0, binop( Iop_And32,
                       mkexpr( finite_number ),
                       binop( Iop_Shl32,
                              unop( Iop_1Uto32,
                                    binop( Iop_CmpLT32U,
                                           mkexpr( exponent_A ),
                                           mkexpr( exponent_B ) ) ),
                                           mkU8( 3 ) ) ) );

   assign( cc1, binop( Iop_And32,
                       mkexpr( finite_number ),
                       binop( Iop_Shl32,
                              unop( Iop_1Uto32,
                                    binop( Iop_CmpLT32U,
                                           mkexpr( exponent_B ),
                                           mkexpr( exponent_A ) ) ),
                                           mkU8( 2 ) ) ) );

   assign( cc2, binop( Iop_Shl32, 
                       binop( Iop_And32,
                              mkexpr( A_equals_B ),
                              mkU32( 1 ) ),
                              mkU8( 1 ) ) );

   assign( cc3, binop( Iop_And32,
                       unop( Iop_Not32, mkexpr( A_equals_B ) ),
                       binop( Iop_And32,
                              mkU32( 0x1 ),
                              binop( Iop_Or32,
                                     binop( Iop_Or32,
                                            mkexpr ( A_inf_true ),
                                            mkexpr ( B_inf_true ) ),
                                            binop( Iop_Or32,
                                                   mkexpr ( A_NaN_true ),
                                                   mkexpr ( B_NaN_true ) ) )
                              ) ) );

   /* store the condition code */
   assign( cc, binop( Iop_Or32,
                      mkexpr( cc0 ),
                      binop( Iop_Or32,
                             mkexpr( cc1 ),
                             binop( Iop_Or32,
                                    mkexpr( cc2 ),
                                    mkexpr( cc3 ) ) ) ) );
   putGST_field( PPC_GST_CR, mkexpr( cc ), crfD );
   putFPCC( mkexpr( cc ) );
   return True;
}

/* Test class/group/exponent/significance instructions. */
static Bool dis_dfp_class_test ( UInt prefix, UInt theInstr )
{
   UChar frA_addr   = ifieldRegA( theInstr );
   IRTemp frA       = newTemp( Ity_D64 );
   IRTemp abs_frA   = newTemp( Ity_D64 );
   IRTemp frAI64_hi = newTemp( Ity_I64 );
   IRTemp frAI64_lo = newTemp( Ity_I64 );
   UInt opc1        = ifieldOPC( theInstr );
   UInt opc2        = ifieldOPClo9( theInstr );
   UChar crfD       = toUChar( IFIELD( theInstr, 23, 3 ) );  // AKA BF
   UInt DCM         = IFIELD( theInstr, 10, 6 );
   IRTemp DCM_calc  = newTemp( Ity_I32 );
   UInt max_exp     = 0;
   UInt min_exp     = 0;
   IRTemp min_subnormalD64  = newTemp( Ity_D64 );
   IRTemp min_subnormalD128 = newTemp( Ity_D128 );
   IRTemp significand64  = newTemp( Ity_D64 );
   IRTemp significand128 = newTemp( Ity_D128 );
   IRTemp exp_min_normal = newTemp( Ity_I64 );
   IRTemp exponent       = newTemp( Ity_I32 );

   IRTemp infinity_true  = newTemp( Ity_I32 );
   IRTemp SNaN_true      = newTemp( Ity_I32 );
   IRTemp QNaN_true      = newTemp( Ity_I32 );
   IRTemp subnormal_true = newTemp( Ity_I32 );
   IRTemp normal_true    = newTemp( Ity_I32 );
   IRTemp extreme_true   = newTemp( Ity_I32 );
   IRTemp lmd            = newTemp( Ity_I32 );
   IRTemp lmd_zero_true  = newTemp( Ity_I32 );
   IRTemp zero_true      = newTemp( Ity_I32 );
   IRTemp sign           = newTemp( Ity_I32 );
   IRTemp field          = newTemp( Ity_I32 );
   IRTemp ccIR_zero      = newTemp( Ity_I32 );
   IRTemp ccIR_subnormal = newTemp( Ity_I32 );

   /* UInt size     = DFP_LONG;  JRS:unused */
   IRTemp gfield = newTemp( Ity_I32 );
   IRTemp gfield_0_4_shift  = newTemp( Ity_I8 );
   IRTemp gfield_mask       = newTemp( Ity_I32 );
   IRTemp dcm0 = newTemp( Ity_I32 );
   IRTemp dcm1 = newTemp( Ity_I32 );
   IRTemp dcm2 = newTemp( Ity_I32 );
   IRTemp dcm3 = newTemp( Ity_I32 );
   IRTemp dcm4 = newTemp( Ity_I32 );
   IRTemp dcm5 = newTemp( Ity_I32 );

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   /* The only difference between the dtstdc and dtstdcq instructions is
    * size of the T and G fields.  The calculation of the 4 bit field
    * is the same.  Setup the parameters and values that are DFP size
    * specific.  The rest of the code is independent of the DFP size.
    *
    * The Io_CmpD64 is used below.  The instruction sets the ccIR values.
    * The interpretation of the ccIR values is as follows:
    *
    *    DFP cmp result | IR
    * --------------------------
    *	 UN             | 0x45
    *	 EQ             | 0x40
    *	 GT             | 0x00
    *	 LT             | 0x01
    */

   assign( frA, getDReg( frA_addr ) );
   assign( frAI64_hi, unop( Iop_ReinterpD64asI64, mkexpr( frA ) ) );

   assign( abs_frA, unop( Iop_ReinterpI64asD64,
                          binop( Iop_And64,
                                 unop( Iop_ReinterpD64asI64,
                                       mkexpr( frA ) ),
                                 mkU64( 0x7FFFFFFFFFFFFFFFULL ) ) ) );
   assign( gfield_0_4_shift, mkU8( 31 - 5 ) );  // G-field[0:4]
   switch (opc1) {
   case 0x3b: // dtstdc, dtstdg
      DIP("dtstd%s %u,r%u,%u\n", opc2 == 0xc2 ? "c" : "g",
               crfD, frA_addr, DCM);
      /* setup the parameters for the long format of the two instructions */
      assign( frAI64_lo, mkU64( 0 ) );
      assign( gfield_mask, mkU32( DFP_G_FIELD_LONG_MASK ) );
      max_exp = DFP_LONG_EXP_MAX;
      min_exp = DFP_LONG_EXP_MIN;

      assign( exponent, unop( Iop_64to32,
                              unop( Iop_ExtractExpD64,
                                    mkexpr( frA ) ) ) );
      assign( significand64,
              unop( Iop_ReinterpI64asD64,
                    mkU64( 0x2234000000000001ULL ) ) );  // dfp 1.0
      assign( exp_min_normal,mkU64( 398 - 383 ) );
      assign( min_subnormalD64,
              binop( Iop_InsertExpD64,
                     mkexpr( exp_min_normal ),
                     mkexpr( significand64 ) ) );

      assign( ccIR_subnormal,
              binop( Iop_CmpD64,
                     mkexpr( abs_frA ),
                     mkexpr( min_subnormalD64 ) ) );

      /* compare absolute value of frA with zero */
      assign( ccIR_zero,
              binop( Iop_CmpD64,
                     mkexpr( abs_frA ),
                     unop( Iop_ReinterpI64asD64,
                           mkU64( 0x2238000000000000ULL ) ) ) );

      /* size = DFP_LONG; JRS: unused */
      break;

   case 0x3F:   // dtstdcq, dtstdgq
      DIP("dtstd%sq %u,r%u,%u\n", opc2 == 0xc2 ? "c" : "g",
               crfD, frA_addr, DCM);
      /* setup the parameters for the extended format of the
       * two instructions
       */
      assign( frAI64_lo, unop( Iop_ReinterpD64asI64,
                               getDReg( frA_addr+1 ) ) );

      assign( gfield_mask, mkU32( DFP_G_FIELD_EXTND_MASK ) );
      max_exp = DFP_EXTND_EXP_MAX;
      min_exp = DFP_EXTND_EXP_MIN;
      assign( exponent, unop( Iop_64to32, 
                              unop( Iop_ExtractExpD128,
                                    getDReg_pair( frA_addr) ) ) );

      /* create quand exponent for minimum normal number */
      assign( exp_min_normal, mkU64( 6176 - 6143 ) );
      assign( significand128,
              unop( Iop_D64toD128,
                    unop( Iop_ReinterpI64asD64,
                          mkU64( 0x2234000000000001ULL ) ) ) );  // dfp 1.0

      assign( min_subnormalD128,
              binop( Iop_InsertExpD128,
                     mkexpr( exp_min_normal ),
                     mkexpr( significand128 ) ) );

      assign( ccIR_subnormal, 
              binop( Iop_CmpD128,
                     binop( Iop_D64HLtoD128,
                            unop( Iop_ReinterpI64asD64,
                                  binop( Iop_And64,
                                         unop( Iop_ReinterpD64asI64,
                                               mkexpr( frA ) ),
                                         mkU64( 0x7FFFFFFFFFFFFFFFULL ) ) ),
                            getDReg( frA_addr+1 ) ),
                     mkexpr( min_subnormalD128 ) ) );
      assign( ccIR_zero,
              binop( Iop_CmpD128,
                     binop( Iop_D64HLtoD128,
                            mkexpr( abs_frA ),
                            getDReg( frA_addr+1 ) ),
                     unop( Iop_D64toD128,
                           unop( Iop_ReinterpI64asD64,
                                 mkU64( 0x0ULL ) ) ) ) );

      /* size = DFP_EXTND; JRS:unused */
      break;
   default:
      vex_printf("dis_dfp_class_test(ppc)(opc2)\n");
      return False;
   }

   /* The G-field is in the upper 32-bits.  The I64 logical operations
    * do not seem to be supported in 32-bit mode so keep things as 32-bit
    * operations.
    */
   assign( gfield, binop( Iop_And32,
                          mkexpr( gfield_mask ),
                          unop( Iop_64HIto32,
                                mkexpr(frAI64_hi) ) ) );

   /* There is a lot of code that is the same to do the class and group
    * instructions.  Later there is an if statement to handle the specific
    * instruction.
    *
    * Will be using I32 values, compares, shifts and logical operations for
    * this code as the 64-bit compare, shifts, logical operations are not 
    * supported in 32-bit mode.
    */

   /* Check the bits for Infinity, QNaN or Signaling NaN */
   assign( infinity_true,
           unop( Iop_1Sto32,
                 binop( Iop_CmpEQ32,
                        binop( Iop_And32,
                               mkU32( 0x7C000000 ),
                               mkexpr( gfield ) ),
                        mkU32( 0x78000000 ) ) ) );

   assign( SNaN_true,
           unop( Iop_1Sto32,
                 binop( Iop_CmpEQ32,
                        binop( Iop_And32,
                               mkU32( 0x7E000000 ),
                               mkexpr( gfield ) ),
                        mkU32( 0x7E000000 ) ) ) );

   assign( QNaN_true,
           binop( Iop_And32,
                  unop( Iop_1Sto32,
                       binop( Iop_CmpEQ32,
                              binop( Iop_And32,
                                     mkU32( 0x7E000000 ),
                                     mkexpr( gfield ) ),
                              mkU32( 0x7C000000 ) ) ),
                  unop( Iop_Not32,
                        mkexpr( SNaN_true ) ) ) );

   assign( zero_true,
           binop( Iop_And32,
                  unop(Iop_1Sto32,
                       binop( Iop_CmpEQ32,
                              mkexpr( ccIR_zero ),
                              mkU32( 0x40 ) ) ),  // ccIR code for Equal
                  unop( Iop_Not32,
                        binop( Iop_Or32,
                               mkexpr( infinity_true ),
                               binop( Iop_Or32,
                                      mkexpr( QNaN_true ),
                                      mkexpr( SNaN_true ) ) ) ) ) );

   /* Do compare of frA the minimum normal value.  Comparison is size
    * depenent and was done above to get the ccIR value.
    */
   assign( subnormal_true, 
           binop( Iop_And32,
                  binop( Iop_Or32,
                         unop( Iop_1Sto32,
                               binop( Iop_CmpEQ32,
                                      mkexpr( ccIR_subnormal ),
                                      mkU32( 0x40 ) ) ), // ccIR code for Equal
                         unop( Iop_1Sto32,
                               binop( Iop_CmpEQ32,
                                      mkexpr( ccIR_subnormal ),
                                      mkU32( 0x1 ) ) ) ), // ccIR code for LT
           unop( Iop_Not32,
                 binop( Iop_Or32,
                        binop( Iop_Or32,
                               mkexpr( infinity_true ),
                               mkexpr( zero_true) ),
                        binop( Iop_Or32,
                               mkexpr( QNaN_true ),
                               mkexpr( SNaN_true ) ) ) ) ) );

   /* Normal number is not subnormal, infinity, NaN or Zero */
   assign( normal_true,
           unop( Iop_Not32,
                 binop( Iop_Or32,
                        binop( Iop_Or32,
                               mkexpr( infinity_true ),
                               mkexpr( zero_true ) ),
                        binop( Iop_Or32,
                               mkexpr( subnormal_true ),
                               binop( Iop_Or32,
                                      mkexpr( QNaN_true ),
                                      mkexpr( SNaN_true ) ) ) ) ) );

   /* Calculate the DCM bit field based on the tests for the specific
    * instruction
    */
   if (opc2 == 0xC2) {    // dtstdc, dtstdcq
      /* DCM[0:5] Bit   Data Class definition
       *   0   Zero
       *   1   Subnormal
       *   2   Normal
       *   3   Infinity
       *   4   Quiet NaN
       *   5   Signaling NaN
       */

      assign( dcm0, binop( Iop_Shl32,
                           mkexpr( zero_true ),
                           mkU8( 5 ) ) );
      assign( dcm1, binop( Iop_Shl32,
                           binop( Iop_And32,
                                  mkexpr( subnormal_true ),
                                  mkU32( 1 ) ),
                           mkU8( 4 ) ) );
      assign( dcm2, binop( Iop_Shl32,
                           binop( Iop_And32,
                                  mkexpr( normal_true ),
                                  mkU32( 1 ) ),
                           mkU8( 3 ) ) );
      assign( dcm3, binop( Iop_Shl32,
                           binop( Iop_And32,
                                  mkexpr( infinity_true),
                                  mkU32( 1 ) ),
                           mkU8( 2 ) ) );
      assign( dcm4, binop( Iop_Shl32,
                           binop( Iop_And32,
                                  mkexpr( QNaN_true ),
                                  mkU32( 1 ) ),
                           mkU8( 1 ) ) );
      assign( dcm5, binop( Iop_And32, mkexpr( SNaN_true), mkU32( 1 ) ) );

   } else if (opc2 == 0xE2) {   // dtstdg, dtstdgq
      /* check if the exponent is extreme */
      assign( extreme_true, binop( Iop_Or32,
                                   unop( Iop_1Sto32,
                                         binop( Iop_CmpEQ32,
                                                mkexpr( exponent ),
                                                mkU32( max_exp ) ) ),
                                   unop( Iop_1Sto32,
                                         binop( Iop_CmpEQ32,
                                                mkexpr( exponent ),
                                                mkU32( min_exp ) ) ) ) );

      /* Check if LMD is zero */
      Get_lmd( &lmd, binop( Iop_Shr32,
                            mkexpr( gfield ), mkU8( 31 - 5 ) ) );

      assign( lmd_zero_true, unop( Iop_1Sto32,
                                   binop( Iop_CmpEQ32,
                                          mkexpr( lmd ),
                                          mkU32( 0 ) ) ) );

      /* DCM[0:5] Bit   Data Class definition
       *  0   Zero with non-extreme exponent
       *  1   Zero with extreme exponent
       *  2   Subnormal or (Normal with extreme exponent)
       *  3   Normal with non-extreme exponent and
       *      leftmost zero digit in significand
       *  4   Normal with non-extreme exponent and
       *      leftmost nonzero digit in significand
       *  5   Special symbol (Infinity, QNaN, or SNaN)
       */
      assign( dcm0, binop( Iop_Shl32,
                           binop( Iop_And32,
                                  binop( Iop_And32,
                                         unop( Iop_Not32,
                                               mkexpr( extreme_true ) ),
                                         mkexpr( zero_true ) ),
                                  mkU32( 0x1 ) ),
                           mkU8( 5 ) ) );

      assign( dcm1, binop( Iop_Shl32,
                           binop( Iop_And32,
                                  binop( Iop_And32,
                                         mkexpr( extreme_true ),
                                         mkexpr( zero_true ) ),
                                  mkU32( 0x1 ) ),
                           mkU8( 4 ) ) );

      assign( dcm2, binop( Iop_Shl32,
                           binop( Iop_And32,
                                  binop( Iop_Or32,
                                         binop( Iop_And32,
                                                mkexpr( extreme_true ),
                                                mkexpr( normal_true ) ),
                                         mkexpr( subnormal_true ) ),
                                  mkU32( 0x1 ) ),
                           mkU8( 3 ) ) );

      assign( dcm3, binop( Iop_Shl32,
                           binop( Iop_And32,
                                  binop( Iop_And32,
                                         binop( Iop_And32,
                                                unop( Iop_Not32,
                                                      mkexpr( extreme_true ) ),
                                                      mkexpr( normal_true ) ),
                                         unop( Iop_1Sto32,
                                               binop( Iop_CmpEQ32,
                                                      mkexpr( lmd ),
                                                      mkU32( 0 ) ) ) ),
                                  mkU32( 0x1 ) ),
                           mkU8( 2 ) ) );

      assign( dcm4, binop( Iop_Shl32,
                           binop( Iop_And32,
                                  binop( Iop_And32,
                                         binop( Iop_And32,
                                                unop( Iop_Not32,
                                                      mkexpr( extreme_true ) ),
                                                mkexpr( normal_true ) ),
                                          unop( Iop_1Sto32,
                                                binop( Iop_CmpNE32,
                                                       mkexpr( lmd ),
                                                       mkU32( 0 ) ) ) ),
                                  mkU32( 0x1 ) ),
                           mkU8( 1 ) ) );

      assign( dcm5, binop( Iop_And32,
                           binop( Iop_Or32,
                                  mkexpr( SNaN_true),
                                  binop( Iop_Or32,
                                         mkexpr( QNaN_true),
                                         mkexpr( infinity_true) ) ),
                           mkU32( 0x1 ) ) );
   }

   /* create DCM field */
   assign( DCM_calc,
           binop( Iop_Or32,
                  mkexpr( dcm0 ),
                  binop( Iop_Or32,
                         mkexpr( dcm1 ),
                         binop( Iop_Or32,
                                mkexpr( dcm2 ),
                                binop( Iop_Or32,
                                       mkexpr( dcm3 ),
                                       binop( Iop_Or32,
                                              mkexpr( dcm4 ),
                                              mkexpr( dcm5 ) ) ) ) ) ) );

   /* Get the sign of the DFP number, ignore sign for QNaN */
   assign( sign,
           unop( Iop_1Uto32,
                 binop( Iop_CmpEQ32,
                        binop( Iop_Shr32,
                               unop( Iop_64HIto32, mkexpr( frAI64_hi ) ),
                               mkU8( 63 - 32 ) ),
                        mkU32( 1 ) ) ) );

   /* This instruction generates a four bit field to be stored in the
    * condition code register.  The condition code register consists of 7
    * fields.  The field to be written to is specified by the BF (AKA crfD)
    * field.
    *
    * The field layout is as follows:
    *
    *      Field          Meaning
    *      0000           Operand positive with no match
    *      0100           Operand positive with at least one match
    *      0001           Operand negative with no match
    *      0101           Operand negative with at least one match
    */
   assign( field, binop( Iop_Or32,
                         binop( Iop_Shl32,
                                mkexpr( sign ),
                                mkU8( 3 ) ),
                                binop( Iop_Shl32,
                                       unop( Iop_1Uto32,
                                             binop( Iop_CmpNE32,
                                                    binop( Iop_And32,
                                                           mkU32( DCM ),
                                                           mkexpr( DCM_calc ) ),
                                                     mkU32( 0 ) ) ),
                                       mkU8( 1 ) ) ) );

   putGST_field( PPC_GST_CR, mkexpr( field ), crfD );
   putFPCC( mkexpr( field ) );
   return True;
}

static Bool dis_dfp_bcd( UInt prefix, UInt theInstr ) {
   UInt opc2        = ifieldOPClo10( theInstr );
   ULong sp         = IFIELD(theInstr, 19, 2);
   ULong s          = IFIELD(theInstr, 20, 1);
   UChar frT_addr   = ifieldRegDS( theInstr );
   UChar frB_addr   = ifieldRegB( theInstr );
   IRTemp frB       = newTemp( Ity_D64 );
   IRTemp frBI64    = newTemp( Ity_I64 );
   IRTemp result    = newTemp( Ity_I64 );
   IRTemp resultD64 = newTemp( Ity_D64 );
   IRTemp bcd64     = newTemp( Ity_I64 );
   IRTemp bcd_u     = newTemp( Ity_I32 );
   IRTemp bcd_l     = newTemp( Ity_I32 );
   IRTemp dbcd_u    = newTemp( Ity_I32 );
   IRTemp dbcd_l    = newTemp( Ity_I32 );
   IRTemp lmd       = newTemp( Ity_I32 );

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   assign( frB, getDReg( frB_addr ) );
   assign( frBI64, unop( Iop_ReinterpD64asI64, mkexpr( frB ) ) );

   switch ( opc2 ) {
   case 0x142: // ddedpd   DFP Decode DPD to BCD
      DIP( "ddedpd %llu,r%u,r%u\n", sp, frT_addr, frB_addr );

         assign( bcd64, unop( Iop_DPBtoBCD, mkexpr( frBI64 ) ) );
         assign( bcd_u, unop( Iop_64HIto32, mkexpr( bcd64 ) ) );
         assign( bcd_l, unop( Iop_64to32, mkexpr( bcd64 ) ) );

      if ( ( sp == 0 ) || ( sp == 1 ) ) {
         /* Unsigned BCD string */
         Get_lmd( &lmd,
                  binop( Iop_Shr32,
                         unop( Iop_64HIto32, mkexpr( frBI64 ) ),
                         mkU8( 31 - 5 ) ) ); // G-field[0:4]

         assign( result,
                 binop( Iop_32HLto64,
                        binop( Iop_Or32,
                               binop( Iop_Shl32, mkexpr( lmd ), mkU8( 28 ) ),
                               mkexpr( bcd_u ) ),
                        mkexpr( bcd_l ) ) );

      } else {
         /* Signed BCD string, the cases for sp 2 and 3 only differ in how
          * the positive and negative values are encoded in the least
          * significant bits.
          */
         IRTemp sign = newTemp( Ity_I32 );

         if (sp == 2) {
            /* Positive sign = 0xC, negative sign = 0xD */

            assign( sign,
                    binop( Iop_Or32,
                           binop( Iop_Shr32,
                                  unop( Iop_64HIto32, mkexpr( frBI64 ) ),
                                  mkU8( 31 ) ),
                           mkU32( 0xC ) ) );

         } else if ( sp == 3 ) {
            /* Positive sign = 0xF, negative sign = 0xD */
            IRTemp tmp32 = newTemp( Ity_I32 );

            /* Complement sign bit then OR into bit position 1 */
            assign( tmp32,
                    binop( Iop_Xor32,
                           binop( Iop_Shr32,
                                  unop( Iop_64HIto32, mkexpr( frBI64 ) ),
                                  mkU8( 30 ) ),
                           mkU32( 0x2 ) ) );

            assign( sign, binop( Iop_Or32, mkexpr( tmp32 ), mkU32( 0xD ) ) );

         } else {
            vpanic( "The impossible happened: dis_dfp_bcd(ppc), undefined SP field" );
         }

         /* Put sign in bottom 4 bits, move most significant 4-bits from
          * bcd_l to bcd_u.
          */
         assign( result,
                 binop( Iop_32HLto64,
                        binop( Iop_Or32,
                               binop( Iop_Shr32,
                                      mkexpr( bcd_l ),
                                      mkU8( 28 ) ),
                               binop( Iop_Shl32,
                                      mkexpr( bcd_u ),
                                      mkU8( 4 ) ) ),
                        binop( Iop_Or32,
                                      mkexpr( sign ),
                               binop( Iop_Shl32,
                                      mkexpr( bcd_l ),
                                      mkU8( 4 ) ) ) ) );
      }

      putDReg( frT_addr, unop( Iop_ReinterpI64asD64, mkexpr( result ) ) );
      break;

   case 0x342: // denbcd   DFP Encode BCD to DPD
   {
      IRTemp valid_mask   = newTemp( Ity_I32 );
      IRTemp invalid_mask = newTemp( Ity_I32 );
      IRTemp without_lmd  = newTemp( Ity_I64 );
      IRTemp tmp64        = newTemp( Ity_I64 );
      IRTemp dbcd64       = newTemp( Ity_I64 );
      IRTemp left_exp     = newTemp( Ity_I32 );
      IRTemp g0_4         = newTemp( Ity_I32 );

      DIP( "denbcd %llu,r%u,r%u\n", s, frT_addr, frB_addr );

      if ( s == 0 ) {
         /* Unsigned BCD string */
         assign( dbcd64, unop( Iop_BCDtoDPB, mkexpr(frBI64 ) ) );
         assign( dbcd_u, unop( Iop_64HIto32, mkexpr( dbcd64 ) ) );
         assign( dbcd_l, unop( Iop_64to32, mkexpr( dbcd64 ) ) );

         assign( lmd,
                 binop( Iop_Shr32,
                        binop( Iop_And32,
                               unop( Iop_64HIto32, mkexpr( frBI64 ) ),
                               mkU32( 0xF0000000 ) ),
                        mkU8( 28 ) ) );

         assign( invalid_mask,
                 bcd_digit_inval( unop( Iop_64HIto32, mkexpr( frBI64 ) ),
                                  unop( Iop_64to32, mkexpr( frBI64 ) ) ) );
         assign( valid_mask, unop( Iop_Not32, mkexpr( invalid_mask ) ) );

         assign( without_lmd,
                 unop( Iop_ReinterpD64asI64,
                       binop( Iop_InsertExpD64,
                              mkU64( DFP_LONG_BIAS ),
                              unop( Iop_ReinterpI64asD64,
                                    binop( Iop_32HLto64,
                                           mkexpr( dbcd_u ),
                                           mkexpr( dbcd_l ) ) ) ) ) );
         assign( left_exp,
                 binop( Iop_Shr32,
                        binop( Iop_And32,
                               unop( Iop_64HIto32, mkexpr( without_lmd ) ),
                               mkU32( 0x60000000 ) ),
                        mkU8( 29 ) ) );

         assign( g0_4,
                 binop( Iop_Shl32,
                        Gfield_encoding( mkexpr( left_exp ), mkexpr( lmd ) ),
                        mkU8( 26 ) ) );

         assign( tmp64,
                 binop( Iop_32HLto64,
                        binop( Iop_Or32,
                               binop( Iop_And32,
                                      unop( Iop_64HIto32,
                                            mkexpr( without_lmd ) ),
                                      mkU32( 0x83FFFFFF ) ),
                               mkexpr( g0_4 ) ),
                        unop( Iop_64to32, mkexpr( without_lmd ) ) ) );

      } else if ( s == 1 ) {
         IRTemp sign = newTemp( Ity_I32 );
         IRTemp sign_bit = newTemp( Ity_I32 );
         IRTemp pos_sign_mask = newTemp( Ity_I32 );
         IRTemp neg_sign_mask = newTemp( Ity_I32 );
         IRTemp tmp = newTemp( Ity_I64 );

         /* Signed BCD string, least significant 4 bits are sign bits
          * positive sign = 0xC, negative sign = 0xD
          */
         assign( tmp, unop( Iop_BCDtoDPB,
                            binop( Iop_32HLto64,
                                   binop( Iop_Shr32,
                                          unop( Iop_64HIto32,
                                                mkexpr( frBI64 ) ),
                                                mkU8( 4 ) ),
                                   binop( Iop_Or32,
                                          binop( Iop_Shr32,
                                                 unop( Iop_64to32,
                                                       mkexpr( frBI64 ) ),
                                                  mkU8( 4 ) ),
                                          binop( Iop_Shl32,
                                                 unop( Iop_64HIto32,
                                                       mkexpr( frBI64 ) ),
                                                       mkU8( 28 ) ) ) ) ) );

         assign( dbcd_u, unop( Iop_64HIto32, mkexpr( tmp ) ) );
         assign( dbcd_l, unop( Iop_64to32, mkexpr( tmp ) ) );

         /* Get the sign of the BCD string. */
         assign( sign,
                 binop( Iop_And32,
                        unop( Iop_64to32, mkexpr( frBI64 ) ),
                        mkU32( 0xF ) ) );

         assign( neg_sign_mask, Generate_neg_sign_mask( mkexpr( sign ) ) );
         assign( pos_sign_mask, Generate_pos_sign_mask( mkexpr( sign ) ) );
         assign( sign_bit,
                 Generate_sign_bit( mkexpr( pos_sign_mask ),
                                    mkexpr( neg_sign_mask ) ) );

         /* Check for invalid sign and BCD digit.  Don't check the bottom
          * four bits of bcd_l as that is the sign value.
          */
         assign( invalid_mask,
                 Generate_inv_mask(
                                   bcd_digit_inval( unop( Iop_64HIto32,
                                                          mkexpr( frBI64 ) ),
                                                    binop( Iop_Shr32,
                                                           unop( Iop_64to32,
                                                                 mkexpr( frBI64 ) ),
                                                           mkU8( 4 ) ) ),
                                   mkexpr( pos_sign_mask ),
                                   mkexpr( neg_sign_mask ) ) );

         assign( valid_mask, unop( Iop_Not32, mkexpr( invalid_mask ) ) );

         /* Generate the result assuming the sign value was valid. */
         assign( tmp64,
                 unop( Iop_ReinterpD64asI64,
                       binop( Iop_InsertExpD64,
                              mkU64( DFP_LONG_BIAS ),
                              unop( Iop_ReinterpI64asD64,
                                    binop( Iop_32HLto64,
                                           binop( Iop_Or32,
                                                  mkexpr( dbcd_u ),
                                                  mkexpr( sign_bit ) ),
                                           mkexpr( dbcd_l ) ) ) ) ) );
      }

      /* Generate the value to store depending on the validity of the
       * sign value and the validity of the BCD digits.
       */
      assign( resultD64,
              unop( Iop_ReinterpI64asD64,
                    binop( Iop_32HLto64,
                           binop( Iop_Or32,
                                  binop( Iop_And32,
                                         mkexpr( valid_mask ),
                                         unop( Iop_64HIto32,
                                               mkexpr( tmp64 ) ) ),
                                  binop( Iop_And32,
                                         mkU32( 0x7C000000 ),
                                         mkexpr( invalid_mask ) ) ),
                           binop( Iop_Or32,
                                  binop( Iop_And32,
                                         mkexpr( valid_mask ),
                                         unop( Iop_64to32, mkexpr( tmp64 ) ) ),
                                  binop( Iop_And32,
                                         mkU32( 0x0 ),
                                         mkexpr( invalid_mask ) ) ) ) ) );
      putDReg( frT_addr, mkexpr( resultD64 ) );
   }
   break;
   default:
      vpanic( "ERROR: dis_dfp_bcd(ppc), undefined opc2 case " );
      return False;
   }
   return True;
}

static Bool dis_dfp_bcdq( UInt prefix, UInt theInstr )
{
   UInt opc2        = ifieldOPClo10( theInstr );
   ULong sp         = IFIELD(theInstr, 19, 2);
   ULong s          = IFIELD(theInstr, 20, 1);
   IRTemp frB_hi    = newTemp( Ity_D64 );
   IRTemp frB_lo    = newTemp( Ity_D64 );
   IRTemp frBI64_hi = newTemp( Ity_I64 );
   IRTemp frBI64_lo = newTemp( Ity_I64 );
   UChar frT_addr   = ifieldRegDS( theInstr );
   UChar frB_addr   = ifieldRegB( theInstr );

   IRTemp lmd       = newTemp( Ity_I32 );
   IRTemp result_hi = newTemp( Ity_I64 );
   IRTemp result_lo = newTemp( Ity_I64 );

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   assign( frB_hi, getDReg( frB_addr ) );
   assign( frB_lo, getDReg( frB_addr + 1 ) );
   assign( frBI64_hi, unop( Iop_ReinterpD64asI64, mkexpr( frB_hi ) ) );
   assign( frBI64_lo, unop( Iop_ReinterpD64asI64, mkexpr( frB_lo ) ) );

   switch ( opc2 ) {
   case 0x142: // ddedpdq   DFP Decode DPD to BCD
   {
      IRTemp low_60_u = newTemp( Ity_I32 );
      IRTemp low_60_l = newTemp( Ity_I32 );
      IRTemp mid_60_u = newTemp( Ity_I32 );
      IRTemp mid_60_l = newTemp( Ity_I32 );
      IRTemp top_12_l = newTemp( Ity_I32 );

      DIP( "ddedpdq %llu,r%u,r%u\n", sp, frT_addr, frB_addr );

      /* Note, instruction only stores the lower 32 BCD digits in
       * the result
       */
      Generate_132_bit_bcd_string( mkexpr( frBI64_hi ),
                                   mkexpr( frBI64_lo ),
                                   &top_12_l,
                                   &mid_60_u,
                                   &mid_60_l,
                                   &low_60_u,
                                   &low_60_l );

      if ( ( sp == 0 ) || ( sp == 1 ) ) {
         /* Unsigned BCD string */
         assign( result_hi,
                 binop( Iop_32HLto64,
                        binop( Iop_Or32,
                               binop( Iop_Shl32,
                                      mkexpr( top_12_l ),
                                      mkU8( 24 ) ),
                               binop( Iop_Shr32,
                                      mkexpr( mid_60_u ),
                                      mkU8( 4 ) ) ),
                        binop( Iop_Or32,
                               binop( Iop_Shl32,
                                      mkexpr( mid_60_u ),
                                      mkU8( 28 ) ),
                               binop( Iop_Shr32,
                                      mkexpr( mid_60_l ),
                                      mkU8( 4 ) ) ) ) );

         assign( result_lo,
                 binop( Iop_32HLto64,
                        binop( Iop_Or32,
                               binop( Iop_Shl32,
                                      mkexpr( mid_60_l ),
                                      mkU8( 28 ) ),
                               mkexpr( low_60_u ) ),
                        mkexpr( low_60_l ) ) );

      } else {
         /* Signed BCD string, the cases for sp 2 and 3 only differ in how
          * the positive and negative values are encoded in the least
          * significant bits.
          */
         IRTemp sign = newTemp( Ity_I32 );

         if ( sp == 2 ) {
            /* Positive sign = 0xC, negative sign = 0xD */
            assign( sign,
                    binop( Iop_Or32,
                           binop( Iop_Shr32,
                                  unop( Iop_64HIto32, mkexpr( frBI64_hi ) ),
                                  mkU8( 31 ) ),
                           mkU32( 0xC ) ) );

         } else if ( sp == 3 ) {
            IRTemp tmp32 = newTemp( Ity_I32 );

            /* Positive sign = 0xF, negative sign = 0xD.
             * Need to complement sign bit then OR into bit position 1.
             */
            assign( tmp32,
                    binop( Iop_Xor32,
                           binop( Iop_Shr32,
                                  unop( Iop_64HIto32, mkexpr( frBI64_hi ) ),
                                  mkU8( 30 ) ),
                           mkU32( 0x2 ) ) );

            assign( sign, binop( Iop_Or32, mkexpr( tmp32 ), mkU32( 0xD ) ) );

         } else {
            vpanic( "The impossible happened: dis_dfp_bcd(ppc), undefined SP field" );
         }

         assign( result_hi,
                 binop( Iop_32HLto64,
                        binop( Iop_Or32,
                               binop( Iop_Shl32,
                                      mkexpr( top_12_l ),
                                      mkU8( 28 ) ),
                               mkexpr( mid_60_u ) ),
                        mkexpr( mid_60_l ) ) );

         assign( result_lo,
                 binop( Iop_32HLto64,
                        binop( Iop_Or32,
                               binop( Iop_Shl32,
                                      mkexpr( low_60_u ),
                                      mkU8( 4 ) ),
                               binop( Iop_Shr32,
                                      mkexpr( low_60_l ),
                                      mkU8( 28 ) ) ),
                        binop( Iop_Or32,
                               binop( Iop_Shl32,
                                      mkexpr( low_60_l ),
                                      mkU8( 4 ) ),
                               mkexpr( sign ) ) ) );
      }

      putDReg( frT_addr, unop( Iop_ReinterpI64asD64, mkexpr( result_hi ) ) );
      putDReg( frT_addr + 1,
               unop( Iop_ReinterpI64asD64, mkexpr( result_lo ) ) );
   }
   break;
   case 0x342: // denbcdq   DFP Encode BCD to DPD
   {
      IRTemp valid_mask      = newTemp( Ity_I32 );
      IRTemp invalid_mask    = newTemp( Ity_I32 );
      IRTemp result128       = newTemp( Ity_D128 );
      IRTemp dfp_significand = newTemp( Ity_D128 );
      IRTemp tmp_hi          = newTemp( Ity_I64 );
      IRTemp tmp_lo          = newTemp( Ity_I64 );
      IRTemp dbcd_top_l      = newTemp( Ity_I32 );
      IRTemp dbcd_mid_u      = newTemp( Ity_I32 );
      IRTemp dbcd_mid_l      = newTemp( Ity_I32 );
      IRTemp dbcd_low_u      = newTemp( Ity_I32 );
      IRTemp dbcd_low_l      = newTemp( Ity_I32 );
      IRTemp bcd_top_8       = newTemp( Ity_I64 );
      IRTemp bcd_mid_60      = newTemp( Ity_I64 );
      IRTemp bcd_low_60      = newTemp( Ity_I64 );
      IRTemp sign_bit        = newTemp( Ity_I32 );
      IRTemp tmptop10        = newTemp( Ity_I64 );
      IRTemp tmpmid50        = newTemp( Ity_I64 );
      IRTemp tmplow50        = newTemp( Ity_I64 );
      IRTemp inval_bcd_digit_mask = newTemp( Ity_I32 );

      DIP( "denbcd %llu,r%u,r%u\n", s, frT_addr, frB_addr );

      if ( s == 0 ) {
         /* Unsigned BCD string */
         assign( sign_bit, mkU32( 0 ) ); // set to zero for unsigned string

         assign( bcd_top_8,
                 binop( Iop_32HLto64,
                        mkU32( 0 ),
                        binop( Iop_And32,
                               binop( Iop_Shr32,
                                      unop( Iop_64HIto32,
                                            mkexpr( frBI64_hi ) ),
                                      mkU8( 24 ) ),
                               mkU32( 0xFF ) ) ) );
         assign( bcd_mid_60,
                 binop( Iop_32HLto64,
                        binop( Iop_Or32,
                               binop( Iop_Shr32,
                                      unop( Iop_64to32,
                                            mkexpr( frBI64_hi ) ),
                                      mkU8( 28 ) ),
                               binop( Iop_Shl32,
                                      unop( Iop_64HIto32,
                                            mkexpr( frBI64_hi ) ),
                                      mkU8( 4 ) ) ),
                        binop( Iop_Or32,
                               binop( Iop_Shl32,
                                      unop( Iop_64to32,
                                            mkexpr( frBI64_hi ) ),
                                      mkU8( 4 ) ),
                               binop( Iop_Shr32,
                                      unop( Iop_64HIto32,
                                            mkexpr( frBI64_lo ) ),
                                      mkU8( 28 ) ) ) ) );

         /* Note, the various helper functions ignores top 4-bits */
         assign( bcd_low_60, mkexpr( frBI64_lo ) );

         assign( tmptop10, unop( Iop_BCDtoDPB, mkexpr( bcd_top_8 ) ) );
         assign( dbcd_top_l, unop( Iop_64to32, mkexpr( tmptop10 ) ) );

         assign( tmpmid50, unop( Iop_BCDtoDPB, mkexpr( bcd_mid_60 ) ) );
         assign( dbcd_mid_u, unop( Iop_64HIto32, mkexpr( tmpmid50 ) ) );
         assign( dbcd_mid_l, unop( Iop_64to32, mkexpr( tmpmid50 ) ) );

         assign( tmplow50, unop( Iop_BCDtoDPB, mkexpr( bcd_low_60 ) ) );
         assign( dbcd_low_u, unop( Iop_64HIto32, mkexpr( tmplow50 ) ) );
         assign( dbcd_low_l, unop( Iop_64to32, mkexpr( tmplow50 ) ) );

         /* The entire BCD string fits in lower 110-bits.  The LMD = 0,
          * value is not part of the final result. Only the right most
          * BCD digits are stored.
          */
         assign( lmd, mkU32( 0 ) );

         assign( invalid_mask,
                 binop( Iop_Or32,
                        bcd_digit_inval( mkU32( 0 ),
                                         unop( Iop_64to32,
                                               mkexpr( bcd_top_8 ) ) ),
                        binop( Iop_Or32,
                               bcd_digit_inval( unop( Iop_64HIto32,
                                                      mkexpr( bcd_mid_60 ) ),
                                                unop( Iop_64to32,
                                                      mkexpr( bcd_mid_60 ) ) ),
                               bcd_digit_inval( unop( Iop_64HIto32,
                                                      mkexpr( bcd_low_60 ) ),
                                                unop( Iop_64to32,
                                                      mkexpr( bcd_low_60 ) )
                                                ) ) ) );

      } else if ( s == 1 ) {
         IRTemp sign          = newTemp( Ity_I32 );
         IRTemp zero          = newTemp( Ity_I32 );
         IRTemp pos_sign_mask = newTemp( Ity_I32 );
         IRTemp neg_sign_mask = newTemp( Ity_I32 );

         /* The sign of the BCD string is stored in lower 4 bits */
         assign( sign,
                 binop( Iop_And32,
                        unop( Iop_64to32, mkexpr( frBI64_lo ) ),
                        mkU32( 0xF ) ) );
         assign( neg_sign_mask, Generate_neg_sign_mask( mkexpr( sign ) ) );
         assign( pos_sign_mask, Generate_pos_sign_mask( mkexpr( sign ) ) );
         assign( sign_bit,
                 Generate_sign_bit( mkexpr( pos_sign_mask ),
                                    mkexpr( neg_sign_mask ) ) );

         /* Generate the value assuminig the sign and BCD digits are vaild */
         assign( bcd_top_8,
                 binop( Iop_32HLto64,
                        mkU32( 0x0 ),
                        binop( Iop_Shr32,
                               unop( Iop_64HIto32, mkexpr( frBI64_hi ) ),
                               mkU8( 28 ) ) ) );

         /* The various helper routines ignore the upper 4-bits */
         assign( bcd_mid_60, mkexpr( frBI64_hi ) );

         /* Remove bottom four sign bits */
         assign( bcd_low_60,
                 binop( Iop_32HLto64,
                        binop( Iop_Shr32,
                               unop( Iop_64HIto32,
                                     mkexpr( frBI64_lo ) ),
                               mkU8( 4 ) ),
                               binop( Iop_Or32,
                                      binop( Iop_Shl32,
                                             unop( Iop_64HIto32,
                                                   mkexpr( frBI64_lo ) ),
                                             mkU8( 28 ) ),
                                      binop( Iop_Shr32,
                                             unop( Iop_64to32,
                                                   mkexpr( frBI64_lo ) ),
                                             mkU8( 4 ) ) ) ) );
         assign( tmptop10, unop( Iop_BCDtoDPB, mkexpr(bcd_top_8 ) ) );
         assign( dbcd_top_l, unop( Iop_64to32, mkexpr( tmptop10 ) ) );

         assign( tmpmid50, unop( Iop_BCDtoDPB, mkexpr(bcd_mid_60 ) ) );
         assign( dbcd_mid_u, unop( Iop_64HIto32, mkexpr( tmpmid50 ) ) );
         assign( dbcd_mid_l, unop( Iop_64to32, mkexpr( tmpmid50 ) ) );

         assign( tmplow50, unop( Iop_BCDtoDPB, mkexpr( bcd_low_60 ) ) );
         assign( dbcd_low_u, unop( Iop_64HIto32, mkexpr( tmplow50 ) ) );
         assign( dbcd_low_l, unop( Iop_64to32, mkexpr( tmplow50 ) ) );

         /* The entire BCD string fits in lower 110-bits.  The LMD value
          * is not stored in the final result for the DFP Long instruction.
          */
         assign( lmd, mkU32( 0 ) );

         /* Check for invalid sign and invalid BCD digit.  Don't check the
          *  bottom four bits of frBI64_lo as that is the sign value.
          */
         assign( zero, mkU32( 0 ) );
         assign( inval_bcd_digit_mask,
                 binop( Iop_Or32,
                        bcd_digit_inval( mkexpr( zero ),
                                         unop( Iop_64to32,
                                               mkexpr( bcd_top_8 ) ) ),
                        binop( Iop_Or32,
                               bcd_digit_inval( unop( Iop_64HIto32,
                                                     mkexpr( bcd_mid_60 ) ),
                                               unop( Iop_64to32,
                                                     mkexpr( bcd_mid_60 ) ) ),
                               bcd_digit_inval( unop( Iop_64HIto32,
                                                     mkexpr( frBI64_lo ) ),
                                               binop( Iop_Shr32,
                                                      unop( Iop_64to32,
                                                            mkexpr( frBI64_lo ) ),
                                                        mkU8( 4 ) ) ) ) ) );
         assign( invalid_mask,
                 Generate_inv_mask( mkexpr( inval_bcd_digit_mask ),
                                    mkexpr( pos_sign_mask ),
                                    mkexpr( neg_sign_mask ) ) );

      }

      assign( valid_mask, unop( Iop_Not32, mkexpr( invalid_mask ) ) );

      /* Calculate the value of the result assuming sign and BCD digits
       * are all valid.
       */
      assign( dfp_significand,
              binop( Iop_D64HLtoD128,
                     unop( Iop_ReinterpI64asD64,
                           binop( Iop_32HLto64,
                                  binop( Iop_Or32,
                                         mkexpr( sign_bit ),
                                         mkexpr( dbcd_top_l ) ),
                                  binop( Iop_Or32,
                                         binop( Iop_Shl32,
                                                mkexpr( dbcd_mid_u ),
                                                mkU8( 18 ) ),
                                         binop( Iop_Shr32,
                                                mkexpr( dbcd_mid_l ),
                                                mkU8( 14 ) ) ) ) ),
                     unop( Iop_ReinterpI64asD64,
                           binop( Iop_32HLto64,
                                  binop( Iop_Or32,
                                         mkexpr( dbcd_low_u ),
                                         binop( Iop_Shl32,
                                                mkexpr( dbcd_mid_l ),
                                                mkU8( 18 ) ) ),
                                  mkexpr( dbcd_low_l ) ) ) ) );

      /* Break the result back down to 32-bit chunks and replace chunks.
       * If there was an invalid BCD digit or invalid sign value, replace
       * the calculated result with the invalid bit string.
       */
      assign( result128,
              binop( Iop_InsertExpD128,
                     mkU64( DFP_EXTND_BIAS ),
                     mkexpr( dfp_significand ) ) );

      assign( tmp_hi,
              unop( Iop_ReinterpD64asI64,
                    unop( Iop_D128HItoD64, mkexpr( result128 ) ) ) );

      assign( tmp_lo,
              unop( Iop_ReinterpD64asI64,
                    unop( Iop_D128LOtoD64, mkexpr( result128 ) ) ) );

      assign( result_hi,
              binop( Iop_32HLto64,
                     binop( Iop_Or32,
                            binop( Iop_And32,
                                   mkexpr( valid_mask ),
                                   unop( Iop_64HIto32, mkexpr( tmp_hi ) ) ),
                            binop( Iop_And32,
                                   mkU32( 0x7C000000 ),
                                   mkexpr( invalid_mask ) ) ),
                     binop( Iop_Or32,
                            binop( Iop_And32,
                                   mkexpr( valid_mask ),
                                   unop( Iop_64to32, mkexpr( tmp_hi ) ) ),
                            binop( Iop_And32,
                                   mkU32( 0x0 ),
                                   mkexpr( invalid_mask ) ) ) ) );

      assign( result_lo,
              binop( Iop_32HLto64,
                     binop( Iop_Or32,
                            binop( Iop_And32,
                                   mkexpr( valid_mask ),
                                   unop( Iop_64HIto32, mkexpr( tmp_lo ) ) ),
                            binop( Iop_And32,
                                   mkU32( 0x0 ),
                                   mkexpr( invalid_mask ) ) ),
                     binop( Iop_Or32,
                            binop( Iop_And32,
                                   mkexpr( valid_mask ),
                                   unop( Iop_64to32, mkexpr( tmp_lo ) ) ),
                            binop( Iop_And32,
                                   mkU32( 0x0 ),
                                   mkexpr( invalid_mask ) ) ) ) );

      putDReg( frT_addr, unop( Iop_ReinterpI64asD64, mkexpr( result_hi ) ) );
      putDReg( frT_addr + 1,
               unop( Iop_ReinterpI64asD64, mkexpr( result_lo ) ) );

   }
   break;
   default:
      vpanic( "ERROR: dis_dfp_bcdq(ppc), undefined opc2 case " );
      break;
   }
   return True;
}

static Bool dis_dfp_significant_digits( UInt prefix, UInt theInstr )
{
   UInt opc1      = ifieldOPC( theInstr );
   UInt opc2      = ifieldOPClo10(theInstr);
   UChar frA_addr = ifieldRegA( theInstr );
   UChar frB_addr = ifieldRegB( theInstr );
   IRTemp frA     = newTemp( Ity_D64 );
   IRTemp B_sig   = newTemp( Ity_I8 );
   IRTemp K       = newTemp( Ity_I8 );
   IRTemp lmd_B   = newTemp( Ity_I32 );
   IRTemp field   = newTemp( Ity_I32 );
   UChar crfD     = toUChar( IFIELD( theInstr, 23, 3 ) ); // AKA BF
   IRTemp Unordered_true     = newTemp( Ity_I32 );
   IRTemp Eq_true_mask       = newTemp( Ity_I32 );
   IRTemp Lt_true_mask       = newTemp( Ity_I32 );
   IRTemp Gt_true_mask       = newTemp( Ity_I32 );
   IRTemp KisZero_true_mask  = newTemp( Ity_I32 );
   IRTemp KisZero_false_mask = newTemp( Ity_I32 );
   IRTemp cc = newTemp( Ity_I32 );
   UChar  UIM     = toUChar( IFIELD( theInstr, 16, 6 ) );
   IRTemp BCD_valid  = newTemp( Ity_I32 );

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   if (opc2 == 0x2A2) {        // dtstsf   DFP Test Significance
                               // dtstsfq  DFP Test Significance Quad
      /* Get the reference singificance stored in frA */
      assign( frA, getDReg( frA_addr ) );

      /* Convert from 64 bit to 8 bits in two steps.  The Iop_64to8 is not
       * supported in 32-bit mode.
       */
      assign( K, unop( Iop_32to8,
                       binop( Iop_And32,
                              unop( Iop_64to32,
                                    unop( Iop_ReinterpD64asI64,
                                          mkexpr( frA ) ) ),
                              mkU32( 0x3F ) ) ) );

   } else if (opc2 == 0x2A3) { // dtstsfi  DFP Test Significance Immediate
                               // dtstsfiq DFP Test Significance Quad Immediate
      /* get the significane from the immediate field */
      assign( K, mkU8( UIM) );

   } else {
      vex_printf("dis_dfp_significant_digits(ppc)(opc2) wrong\n");
      return False;
   }

   switch ( opc1 ) {
   case 0x3b: // dtstsf   DFP Test Significance
              // dtstsfi  DFP Test Significance Immediate
   {
      IRTemp frB     = newTemp( Ity_D64 );
      IRTemp frBI64  = newTemp( Ity_I64 );
      IRTemp B_bcd_u = newTemp( Ity_I32 );
      IRTemp B_bcd_l = newTemp( Ity_I32 );
      IRTemp tmp64   = newTemp( Ity_I64 );

      if (opc2 == 0x2A2) {
         DIP( "dtstsf %u,r%u,r%u\n", crfD, frA_addr, frB_addr );
      } else {
         DIP( "dtstsfi %u,%u,r%u\n", crfD, UIM, frB_addr );
      }

      assign( frB, getDReg( frB_addr ) );
      assign( frBI64, unop( Iop_ReinterpD64asI64, mkexpr( frB ) ) );

      /* Get the BCD string for the value stored in a series of I32 values.
       * Count the number of leading zeros.  Subtract the number of leading
       * zeros from 16 (maximum number of significant digits in DFP
       * Long).
       */
      Get_lmd( &lmd_B,
               binop( Iop_Shr32,
                      unop( Iop_64HIto32, mkexpr( frBI64 ) ),
                      mkU8( 31 - 5 ) ) ); // G-field[0:4]

      assign( tmp64, unop( Iop_DPBtoBCD, mkexpr( frBI64 ) ) );
      assign( B_bcd_u, unop( Iop_64HIto32, mkexpr( tmp64 ) ) );
      assign( B_bcd_l, unop( Iop_64to32, mkexpr( tmp64 ) ) );

      assign( B_sig,
              binop( Iop_Sub8,
                     mkU8( DFP_LONG_MAX_SIG_DIGITS ),
                     Count_leading_zeros_60( mkexpr( lmd_B ),
                                             mkexpr( B_bcd_u ),
                                             mkexpr( B_bcd_l ) ) ) );

      assign( BCD_valid,
              binop( Iop_Or32,
                     bcd_digit_inval( mkexpr( B_bcd_u), mkexpr( B_bcd_l) ),
                     bcd_digit_inval( mkexpr( lmd_B), mkU32( 0 ) ) ) );

      /* Set unordered to True if the number is NaN, Inf or an invalid
       * digit.
       */
      assign( Unordered_true,
              binop( Iop_Or32,
                     Check_unordered( mkexpr( frBI64 ) ),
                     mkexpr( BCD_valid) ) );
   }
   break;
   case 0x3F: // dtstsfq     DFP Test Significance
              // dtstsfqi    DFP Test Significance Immediate
   {
      IRTemp frB_hi     = newTemp( Ity_D64 );
      IRTemp frB_lo     = newTemp( Ity_D64 );
      IRTemp frBI64_hi  = newTemp( Ity_I64 );
      IRTemp frBI64_lo  = newTemp( Ity_I64 );
      IRTemp B_low_60_u = newTemp( Ity_I32 );
      IRTemp B_low_60_l = newTemp( Ity_I32 );
      IRTemp B_mid_60_u = newTemp( Ity_I32 );
      IRTemp B_mid_60_l = newTemp( Ity_I32 );
      IRTemp B_top_12_l = newTemp( Ity_I32 );

      if (opc2 == 0x2A2) {
         DIP( "dtstsfq %u,r%u,r%u\n", crfD, frA_addr, frB_addr );
      } else {
         DIP( "dtstsfiq %u,%u,r%u\n", crfD, UIM, frB_addr );
      }

      assign( frB_hi, getDReg( frB_addr ) );
      assign( frB_lo, getDReg( frB_addr + 1 ) );

      assign( frBI64_hi, unop( Iop_ReinterpD64asI64, mkexpr( frB_hi ) ) );
      assign( frBI64_lo, unop( Iop_ReinterpD64asI64, mkexpr( frB_lo ) ) );

      /* Get the BCD string for the value stored in a series of I32 values.
       * Count the number of leading zeros.  Subtract the number of leading
       * zeros from 32 (maximum number of significant digits in DFP
       * extended).
       */
      Get_lmd( &lmd_B,
               binop( Iop_Shr32,
                      unop( Iop_64HIto32, mkexpr( frBI64_hi ) ),
                      mkU8( 31 - 5 ) ) ); // G-field[0:4]

      Generate_132_bit_bcd_string( mkexpr( frBI64_hi ),
                                   mkexpr( frBI64_lo ),
                                   &B_top_12_l,
                                   &B_mid_60_u,
                                   &B_mid_60_l,
                                   &B_low_60_u,
                                   &B_low_60_l );

      assign( BCD_valid,
              binop( Iop_Or32,
                     binop( Iop_Or32,
                            bcd_digit_inval( mkexpr( lmd_B ),
                                             mkexpr( B_top_12_l ) ),
                            bcd_digit_inval( mkexpr( B_mid_60_u ),
                                             mkexpr( B_mid_60_l ) ) ),
                     bcd_digit_inval( mkexpr( B_low_60_u ),
                                      mkexpr( B_low_60_l ) ) ) );

      assign( B_sig,
              binop( Iop_Sub8,
                     mkU8( DFP_EXTND_MAX_SIG_DIGITS ),
                     Count_leading_zeros_128( mkexpr( lmd_B ),
                                              mkexpr( B_top_12_l ),
                                              mkexpr( B_mid_60_u ),
                                              mkexpr( B_mid_60_l ),
                                              mkexpr( B_low_60_u ),
                                              mkexpr( B_low_60_l ) ) ) );

      /* Set unordered to True if the number is NaN, Inf or an invalid
       * digit.
       */
      assign( Unordered_true,
              binop( Iop_Or32,
                     Check_unordered( mkexpr( frBI64_hi ) ),
                     mkexpr( BCD_valid) ) );
   }
   break;
   }

   /* Compare (16 - cnt[0]) against K and set the condition code field
    * accordingly.
    *
    * The field layout is as follows:
    *
    * bit[3:0]    Description
    *    3     K != 0 and K < Number of significant digits if FRB
    *    2     K != 0 and K > Number of significant digits if FRB OR K = 0
    *    1     K != 0 and K = Number of significant digits if FRB
    *    0     K ? Number of significant digits if FRB
    */
   assign( Eq_true_mask,
           unop( Iop_1Sto32,
                 binop( Iop_CmpEQ32,
                        unop( Iop_8Uto32, mkexpr( K ) ),
                        unop( Iop_8Uto32, mkexpr( B_sig ) ) ) ) );
   assign( Lt_true_mask,
           unop( Iop_1Sto32,
                 binop( Iop_CmpLT32U,
                        unop( Iop_8Uto32, mkexpr( K ) ),
                        unop( Iop_8Uto32, mkexpr( B_sig ) ) ) ) );
   assign( Gt_true_mask,
           unop( Iop_1Sto32,
                 binop( Iop_CmpLT32U,
                        unop( Iop_8Uto32, mkexpr( B_sig ) ),
                        unop( Iop_8Uto32, mkexpr( K ) ) ) ) );

   assign( KisZero_true_mask,
           unop( Iop_1Sto32,
                 binop( Iop_CmpEQ32,
                        unop( Iop_8Uto32, mkexpr( K ) ),
                        mkU32( 0 ) ) ) );
   assign( KisZero_false_mask,
           unop( Iop_1Sto32,
                 binop( Iop_CmpNE32,
                        unop( Iop_8Uto32, mkexpr( K ) ),
                        mkU32( 0 ) ) ) );

   assign( field,
           binop( Iop_Or32,
                  binop( Iop_And32,
                         mkexpr( KisZero_false_mask ),
                         binop( Iop_Or32,
                                binop( Iop_And32,
                                       mkexpr( Lt_true_mask ),
                                       mkU32( 0x8 ) ),
                                binop( Iop_Or32,
                                       binop( Iop_And32,
                                              mkexpr( Gt_true_mask ),
                                              mkU32( 0x4 ) ),
                                       binop( Iop_And32,
                                              mkexpr( Eq_true_mask ),
                                              mkU32( 0x2 ) ) ) ) ),
                  binop( Iop_And32,
                         mkexpr( KisZero_true_mask ),
                         mkU32( 0x4 ) ) ) );

   assign( cc, binop( Iop_Or32,
                      binop( Iop_And32,
                             mkexpr( Unordered_true ),
                             mkU32( 0x1 ) ),
                      binop( Iop_And32,
                             unop( Iop_Not32, mkexpr( Unordered_true ) ),
                             mkexpr( field ) ) ) );

   putGST_field( PPC_GST_CR, mkexpr( cc ), crfD );
   putFPCC( mkexpr( cc ) );

   return True;
}
/*------------------------------------------------------------*/
/*--- AltiVec Instruction Translation                      ---*/
/*------------------------------------------------------------*/

/*
  Altivec Cache Control Instructions (Data Streams)
*/
static Bool dis_av_datastream ( UInt prefix, UInt theInstr )
{
   /* X-Form */
   UChar opc1     = ifieldOPC(theInstr);
   UChar flag_T   = toUChar( IFIELD( theInstr, 25, 1 ) );
   UChar flag_A   = flag_T;
   UChar b23to24  = toUChar( IFIELD( theInstr, 23, 2 ) );
   UChar STRM     = toUChar( IFIELD( theInstr, 21, 2 ) );
   UChar rA_addr  = ifieldRegA(theInstr);
   UChar rB_addr  = ifieldRegB(theInstr);
   UInt  opc2     = ifieldOPClo10(theInstr);
   UChar b0       = ifieldBIT0(theInstr);

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   if (opc1 != 0x1F || b23to24 != 0 || b0 != 0) {
      vex_printf("dis_av_datastream(ppc)(instr)\n");
      return False;
   }

   switch (opc2) {
   case 0x156: // dst (Data Stream Touch, AV p115)
      DIP("dst%s r%u,r%u,%d\n", flag_T ? "t" : "",
                                rA_addr, rB_addr, STRM);
      break;

   case 0x176: // dstst (Data Stream Touch for Store, AV p117)
      DIP("dstst%s r%u,r%u,%d\n", flag_T ? "t" : "",
                                  rA_addr, rB_addr, STRM);
      break;

   case 0x336: // dss (Data Stream Stop, AV p114)
      if (rA_addr != 0 || rB_addr != 0) {
         vex_printf("dis_av_datastream(ppc)(opc2,dst)\n");
         return False;
      }
      if (flag_A == 0) {
         DIP("dss %d\n", STRM);
      } else {
         DIP("dssall\n");
      }
      break;

   default:
      vex_printf("dis_av_datastream(ppc)(opc2)\n");
      return False;
   }
   return True;
}

/*
  AltiVec Processor Control Instructions
*/
static Bool dis_av_procctl ( UInt prefix, UInt theInstr )
{
   /* VX-Form */
   UChar opc1    = ifieldOPC(theInstr);
   UChar vD_addr = ifieldRegDS(theInstr);
   UChar vA_addr = ifieldRegA(theInstr);
   UChar vB_addr = ifieldRegB(theInstr);
   UInt  opc2    = IFIELD( theInstr, 0, 11 );

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   if (opc1 != 0x4) {
      vex_printf("dis_av_procctl(ppc)(instr)\n");
      return False;
   }

   switch (opc2) {
   case 0x604: // mfvscr (Move from VSCR, AV p129)
      if (vA_addr != 0 || vB_addr != 0) {
         vex_printf("dis_av_procctl(ppc)(opc2,dst)\n");
         return False;
      }
      DIP("mfvscr v%d\n", vD_addr);
      putVReg( vD_addr, unop(Iop_32UtoV128, getGST( PPC_GST_VSCR )) ); 
      break;

   case 0x644: { // mtvscr (Move to VSCR, AV p130)
      IRTemp vB = newTemp(Ity_V128);
      if (vD_addr != 0 || vA_addr != 0) {
         vex_printf("dis_av_procctl(ppc)(opc2,dst)\n");
         return False;
      }
      DIP("mtvscr v%d\n", vB_addr);
      assign( vB, getVReg(vB_addr));
      putGST( PPC_GST_VSCR, unop(Iop_V128to32, mkexpr(vB)) ); 
      break;
   }
   default:
      vex_printf("dis_av_procctl(ppc)(opc2)\n");
      return False;
   }
   return True;
}

/*
Vector Extend Sign Instructions
*/
static Bool dis_av_extend_sign_count_zero ( UInt prefix, UInt theInstr,
					    UInt allow_isa_3_0 )
{
   /* VX-Form, sort of, the A register field is used to select the specific
    * sign extension instruction or count leading/trailing zero LSB
    * instruction.
    */

   UChar opc1    = ifieldOPC( theInstr );
   UChar rT_addr = ifieldRegDS (theInstr );
   UChar rA_addr = ifieldRegA( theInstr );
   UChar vB_addr = ifieldRegB( theInstr );
   UInt  opc2    = IFIELD( theInstr, 0, 11 );

   IRTemp vB    = newTemp( Ity_V128 );
   IRTemp vT    = newTemp( Ity_V128 );

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   assign( vB, getVReg ( vB_addr ) );

   if ( ( opc1 != 0x4 ) && ( opc2 != 0x602 ) )  {
      vex_printf("dis_av_extend_sign(ppc)(instr)\n");
      return False;
   }

   switch ( rA_addr ) {
   case 0:
   case 1:
   {
      UInt i;
      IRTemp count[17];
      IRTemp bit_zero[16];
      IRTemp byte_mask[17];

      /* These instructions store the result in the general purpose
       * register in the rT_addr field.
       */

      byte_mask[0] = newTemp( Ity_I32 );
      count[0] = newTemp( Ity_I32 );
      assign( count[0], mkU32( 0 ) );
      assign( byte_mask[0], mkU32( 0x1 ) );

      if ( rA_addr == 0 ) {
         // vclzlsbb (Vector Count Leading Zero Least-Significant Bits Byte)
         DIP("vclzlsbb %d,v%d\n", rT_addr, vB_addr);

      } else {
         // vctzlsbb (Vector Count Trailing Zero Least-Significant Bits Byte)
         DIP("vctzlsbb %d,v%d\n", rT_addr, vB_addr);
      }

      for( i = 0; i < 16; i++ ) {
         byte_mask[i+1] = newTemp( Ity_I32 );
         count[i+1] = newTemp( Ity_I32 );
         bit_zero[i] = newTemp( Ity_I1 );

         /* bit_zero[i] = 0x0 until the first 1 bit is found in lsb of
          * byte.  When the first 1 bit is found it causes the byte_mask
          * to change from 0x1 to 0x0.  Thus the AND of the lsb and byte_mask
          * will be zero  which will be equal to the zero byte_mask causing
          * the value of bit_zero[i] to be equal to 0x1 for all remaining bits.
          */

         if ( rA_addr == 0 )
            /* leading zero bit in byte count,
               work bytes from left to right
            */
            assign( bit_zero[i],
                    binop( Iop_CmpEQ32,
                           binop( Iop_And32,
                                  unop( Iop_V128to32,
                                        binop( Iop_ShrV128,
                                               mkexpr( vB ),
                                               mkU8( ( 15 - i) * 8 ) ) ),
                                  mkexpr( byte_mask[i] ) ),
                           mkexpr( byte_mask[i] ) ) );

         else if ( rA_addr == 1 )
            /* trailing zero bit in byte count,
             * work bytes from right to left
             */
            assign( bit_zero[i],
                    binop( Iop_CmpEQ32,
                           binop( Iop_And32,
                                  unop( Iop_V128to32,
                                        binop( Iop_ShrV128,
                                               mkexpr( vB ),
                                               mkU8( i * 8 ) ) ),
                                  mkexpr( byte_mask[i] ) ),
                           mkexpr( byte_mask[i] ) ) );

         /* Increment count as long as bit_zero = 0 */
         assign( count[i+1], binop( Iop_Add32,
                                    mkexpr( count[i] ),
                                    unop( Iop_1Uto32,
                                          unop( Iop_Not1,
                                                mkexpr( bit_zero[i] ) ) ) ) );

         /* If comparison fails to find a zero bit, set the byte_mask to zero
          * for all future comparisons so there will be no more matches.
          */
         assign( byte_mask[i+1],
                 binop( Iop_And32,
                        unop( Iop_1Uto32,
                              unop( Iop_Not1,
                                    mkexpr( bit_zero[i] ) ) ),
                        mkexpr( byte_mask[i] )  ) );
      }
      putIReg( rT_addr, unop( Iop_32Uto64, mkexpr( count[16] ) ) );
      return True;
   }

   case 6: // vnegw,  Vector Negate Word
      DIP("vnegw  v%u,v%u", rT_addr, vB_addr);

      /* multiply each word by -1 */
      assign( vT, binop( Iop_Mul32x4, mkexpr( vB ), mkV128( 0xFFFF ) ) );
      break;

   case 7: // vnegd,  Vector Negate Doubleword
      DIP("vnegd  v%u,v%u", rT_addr, vB_addr);

      /* multiply each word by -1 */
      assign( vT, binop( Iop_64HLtoV128,
                         binop( Iop_Mul64,
                                unop( Iop_V128HIto64,
                                      mkexpr( vB ) ),
                                mkU64( 0xFFFFFFFFFFFFFFFF ) ),
                         binop( Iop_Mul64,
                               unop( Iop_V128to64,
                                      mkexpr( vB ) ),
                                mkU64( 0xFFFFFFFFFFFFFFFF ) ) ) );
      break;

   case 8:  // vprtybw,  Vector Parity Byte Word
   case 9:  // vprtybd,  Vector Parity Byte Doubleword
   case 10: // vprtybq,  Vector Parity Byte Quadword
      {
         UInt i;
         IRTemp bit_in_byte[16];
         IRTemp word_parity[4];

         for( i = 0; i < 16; i++ ) {
            bit_in_byte[i] = newTemp( Ity_I32 );
            assign( bit_in_byte[i],
                    binop( Iop_And32,
                           unop( Iop_V128to32,
                                 binop( Iop_ShrV128,
                                        mkexpr( vB ),
                                        mkU8( ( 15 - i ) * 8 ) ) ),
                           mkU32( 0x1 ) ) );
         }

         for( i = 0; i < 4; i++ ) {
            word_parity[i] = newTemp(Ity_I32);
            assign( word_parity[i],
                    mkXOr4_32( bit_in_byte[0 + i * 4],
                               bit_in_byte[1 + i * 4],
                               bit_in_byte[2 + i * 4],
                               bit_in_byte[3 + i * 4] ) );
         }

         if ( rA_addr == 8 ) {
            DIP("vprtybw  v%d,v%d", rT_addr, vB_addr);

            assign( vT, mkV128from32( word_parity[0], word_parity[1],
                                      word_parity[2], word_parity[3] ) );

         } else if ( rA_addr == 9 ) {
            DIP("vprtybd  v%d,v%d", rT_addr, vB_addr);

            assign( vT,
                    binop( Iop_64HLtoV128,
                           binop( Iop_32HLto64,
                                  mkU32( 0 ),
                                  binop( Iop_Xor32,
                                         mkexpr( word_parity[0] ),
                                         mkexpr( word_parity[1] ) ) ),
                           binop( Iop_32HLto64,
                                  mkU32( 0 ),
                                  binop( Iop_Xor32,
                                         mkexpr( word_parity[2] ),
                                         mkexpr( word_parity[3] ) ) ) ) );

         } else if ( rA_addr == 10 ) {
            DIP("vprtybq  v%d,v%d", rT_addr, vB_addr);

            assign( vT,
                    binop( Iop_64HLtoV128,
                           mkU64( 0 ),
                           unop( Iop_32Uto64,
                                 mkXOr4_32( word_parity[0],
                                            word_parity[1],
                                            word_parity[2],
                                            word_parity[3] ) ) ) );
         }
      }
      break;

   case 16: // vextsb2w,  Vector Extend Sign Byte to Word
      DIP("vextsb2w  v%u,v%u", rT_addr, vB_addr);

      /* Iop_MullEven8Sx16 does a signed widening multiplication of byte to
       * two byte sign extended result.  Then do a two byte to four byte sign
       * extended multiply.  Note contents of upper three bytes in word are
       * "over written". So just take source and multiply by 1.
       */
      assign( vT, binop( Iop_MullEven16Sx8,
                         binop( Iop_64HLtoV128,
                                mkU64( 0x0000000100000001 ),
                                mkU64( 0x0000000100000001 ) ),
                         binop( Iop_MullEven8Sx16,
                                mkexpr( vB ),
                                binop( Iop_64HLtoV128,
                                       mkU64( 0x0001000100010001 ),
                                       mkU64( 0x0001000100010001 ) ) ) ) );
      break;

   case 17: // vextsh2w,  Vector Extend Sign Halfword to Word
      DIP("vextsh2w  v%u,v%u", rT_addr, vB_addr);

      /* Iop_MullEven16Sx8 does a signed widening multiply of four byte
       * 8 bytes.  Note contents of upper two bytes in word are
       * "over written". So just take source and multiply by 1.
       */
      assign( vT, binop( Iop_MullEven16Sx8,
                         binop( Iop_64HLtoV128,
                                mkU64( 0x0000000100000001 ),
                                mkU64( 0x0000000100000001 ) ),
                        mkexpr( vB ) ) );

      break;

   case 24: // vextsb2d,  Vector Extend Sign Byte to Doubleword
      DIP("vextsb2d  v%u,v%u", rT_addr, vB_addr);

      /* Iop_MullEven8Sx16 does a signed widening multiplication of byte to
       * two byte sign extended result.  Then do a two byte to four byte sign
       * extended multiply. Then do four byte to eight byte multiply.
       */
      assign( vT, binop( Iop_MullEven32Sx4,
                         binop( Iop_64HLtoV128,
                                mkU64( 0x0000000000000001 ),
                                mkU64( 0x0000000000000001 ) ),
                        binop( Iop_MullEven16Sx8,
                               binop( Iop_64HLtoV128,
                                      mkU64( 0x0000000100000001 ),
                                      mkU64( 0x0000000100000001 ) ),
                              binop( Iop_MullEven8Sx16,
                                     binop( Iop_64HLtoV128,
                                            mkU64( 0x0001000100010001 ),
                                            mkU64( 0x0001000100010001 ) ),
                                    mkexpr( vB ) ) ) ) );
      break;

   case 25: // vextsh2d,  Vector Extend Sign Halfword to Doubleword
      DIP("vextsh2d  v%u,v%u", rT_addr, vB_addr);

      assign( vT, binop( Iop_MullEven32Sx4,
                         binop( Iop_64HLtoV128,
                                mkU64( 0x0000000000000001 ),
                                mkU64( 0x0000000000000001 ) ),
                        binop( Iop_MullEven16Sx8,
                               binop( Iop_64HLtoV128,
                                      mkU64( 0x0000000100000001 ),
                                      mkU64( 0x0000000100000001 ) ),
                               mkexpr( vB ) ) ) );
      break;

   case 26: // vextsw2d,  Vector Extend Sign Word to Doubleword
      DIP("vextsw2d  v%u,v%u", rT_addr, vB_addr);

      assign( vT, binop( Iop_MullEven32Sx4,
                         binop( Iop_64HLtoV128,
                                mkU64( 0x0000000000000001 ),
                                mkU64( 0x0000000000000001 ) ),
                        mkexpr( vB ) ) );
      break;
   case 27:  // vextsd2q  Vector Extend Sign Doubleword to Quadword
      {
         IRTemp sb  = newTemp(Ity_I64);   // sign bit extended
         IRTemp tmp = newTemp(Ity_I64);

         DIP("vextsd2q v%u,v%u\n", rT_addr, vB_addr);
         assign( tmp, unop( Iop_V128to64, mkexpr( vB ) ) );
         assign( sb,  unop( Iop_1Sto64,
                            unop( Iop_64to1,
                                  binop( Iop_Shr64,
                                         mkexpr( tmp ),
                                         mkU8( 63 ) ) ) ) );

         assign( vT, binop( Iop_64HLtoV128, mkexpr( sb ), mkexpr( tmp ) ) );
      }
      break;


   case 28: // vctzb,  Vector Count Trailing Zeros Byte
      {
         DIP("vctzb  v%d,v%d", rT_addr, vB_addr);

         /* This instruction is only available in the ISA 3.0 */
         if ( !mode64 || !allow_isa_3_0 ) {
            vex_printf("\n vctzb instruction not supported on non ISA 3.0 platform\n\n");
            return False;
         }
         assign( vT, unop( Iop_Ctz8x16, mkexpr( vB ) ) );
      }
      break;

   case 29: // vctzh,  Vector Count Trailing Zeros Halfword
      {
         DIP("vctzh  v%d,v%d", rT_addr, vB_addr);

         /* This instruction is only available in the ISA 3.0 */
         if ( !mode64 || !allow_isa_3_0 ) {
            vex_printf("\n vctzh instruction not supported on non ISA 3.0 platform\n\n");
            return False;
         }
         assign( vT, unop( Iop_Ctz16x8, mkexpr( vB ) ) );
      }
      break;

   case 30: // vctzw,  Vector Count Trailing Zeros Word
      {
         DIP("vctzw  v%d,v%d", rT_addr, vB_addr);

         /* This instruction is only available in the ISA 3.0 */
         if ( !mode64 || !allow_isa_3_0 ) {
            vex_printf("\n vctzw instruction not supported on non ISA 3.0 platform\n\n");
            return False;
         }
         assign( vT, unop( Iop_Ctz32x4, mkexpr( vB ) ) );
      }
      break;

   case 31: // vctzd,  Vector Count Trailing Zeros Double word
      {
         DIP("vctzd  v%d,v%d", rT_addr, vB_addr);

         /* This instruction is only available in the ISA 3.0 */
         if ( !mode64 || !allow_isa_3_0 ) {
            vex_printf("\n vctzd instruction not supported on non ISA 3.0 platform\n\n");
            return False;
         }
         assign( vT, unop( Iop_Ctz64x2, mkexpr( vB ) ) );
      }
      break;

   default:
      vex_printf("dis_av_extend_sign(ppc)(Unsupported vector extend sign instruction)\n");
      return False;
   }

   putVReg( rT_addr, mkexpr( vT ) );
   return True;
}

/*
Vector Rotate Instructions
*/
static Bool dis_av_rotate ( UInt prefix, UInt theInstr )
{
   /* VX-Form */

   UChar opc1    = ifieldOPC( theInstr );
   UChar vT_addr = ifieldRegDS( theInstr );
   UChar vA_addr = ifieldRegA( theInstr );
   UChar vB_addr = ifieldRegB( theInstr );
   UInt  opc2    = IFIELD( theInstr, 0, 11 );

   IRTemp vA    = newTemp( Ity_V128 );
   IRTemp vB    = newTemp( Ity_V128 );
   IRTemp src3  = newTemp( Ity_V128 );
   IRTemp vT    = newTemp( Ity_V128 );
   IRTemp field_mask = newTemp( Ity_V128 );
   IRTemp mask128 = newTemp( Ity_V128 );
   IRTemp vA_word[4];
   IRTemp left_bits[4];
   IRTemp right_bits[4];
   IRTemp mb[4];
   IRTemp me[4];
   IRTemp shift[4];
   IRTemp mask[4];
   IRTemp tmp_mask[4];
   IRTemp invert_mask[4];
   IRTemp tmp128[4];
   UInt i;
   UInt num_words;
   UInt word_size;
   unsigned long long word_mask;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   if ( opc1 != 0x4 ) {
      vex_printf("dis_av_rotate(ppc)(instr)\n");
      return False;
   }

   assign( vA, getVReg( vA_addr ) );
   assign( vB, getVReg( vB_addr ) );

   switch (opc2) {
   case 0x85: // vrlwmi,  Vector Rotate Left Word then Mask Insert
   case 0x185: // vrlwnm,  Vector Rotate Left Word then AND with Mask
      num_words = 4;
      word_size = 32;
      assign( field_mask, binop( Iop_64HLtoV128,
                                 mkU64( 0 ),
                                 mkU64( 0x1F ) ) );
      word_mask = 0xFFFFFFFF;
      break;

   case 0x0C5: // vrldmi,  Vector Rotate Left Doubleword then Mask Insert
   case 0x1C5: // vrldnm,  Vector Rotate Left Doubleword then AND with Mask
      num_words = 2;
      word_size = 64;
      assign( field_mask, binop( Iop_64HLtoV128,
                                 mkU64( 0 ),
                                 mkU64( 0x3F ) ) );
      word_mask = 0xFFFFFFFFFFFFFFFFULL;
      break;
   default:
      vex_printf("dis_av_rotate(ppc)(opc2)\n");
      return False;
   }

   for( i = 0; i < num_words; i++ ) {
      left_bits[i]  = newTemp( Ity_I8 );
      right_bits[i] = newTemp( Ity_I8 );
      shift[i] = newTemp( Ity_I8 );
      mb[i] = newTemp( Ity_I64 );
      me[i] = newTemp( Ity_I64 );
      tmp_mask[i] = newTemp( Ity_I64 );
      invert_mask[i] = newTemp( Ity_I64 );
      mask[i] = newTemp( Ity_V128 );
      tmp128[i] = newTemp( Ity_V128 );
      vA_word[i] = newTemp( Ity_V128 );

      assign( shift[i],
              unop( Iop_64to8,
                    unop( Iop_V128to64,
                          binop( Iop_AndV128,
                                 binop( Iop_ShrV128,
                                        mkexpr( vB ),
                                        mkU8( (num_words - 1 - i )
                                              * word_size ) ),
                                 mkexpr( field_mask ) ) ) ) );

      assign( mb[i], unop( Iop_V128to64,
                           binop( Iop_AndV128,
                                  binop( Iop_ShrV128,
                                         mkexpr( vB ),
                                         mkU8( ( num_words - 1 - i )
                                               * word_size + 16 ) ),
                                  mkexpr( field_mask ) ) ) );

      assign( me[i], unop( Iop_V128to64,
                           binop( Iop_AndV128,
                                  binop( Iop_ShrV128,
                                         mkexpr( vB ),
                                         mkU8( ( num_words - 1 - i )
                                               * word_size + 8 ) ),
                                  mkexpr( field_mask ) ) ) );

      /* If me < mb, we have to flip things around and invert the mask */
      assign( invert_mask[i],
              unop( Iop_1Sto64, binop( Iop_CmpLT64U,
                                       mkexpr( me[i] ), mkexpr( mb[i] ) ) ) );

      /* left_bits = 63 - mb.  Tells us how many bits to the left
       * of mb to clear. Note for a word left_bits = 32+mb, for a double
       * word left_bits = mb
       */
      assign( left_bits[i],
              unop( Iop_64to8,
                    binop( Iop_Or64,
                           binop( Iop_And64,     // mb < me
                                  unop( Iop_Not64, mkexpr( invert_mask[i] ) ),
                                  binop( Iop_Add64,
                                         mkU64( 64 - word_size ),
                                         mkexpr( mb[i] ) ) ),
                           binop( Iop_And64,     // me < mb
                                  mkexpr( invert_mask[i] ),
                                  binop( Iop_Add64,
                                         mkU64( 64 + 1 - word_size ),
                                         mkexpr( me[i] ) ) ) ) ) );

      /* right_bits = 63 - me.  Tells us how many bits to the right
       * of me to clear. Note for a word, left_bits = me+32, for a double
       * word left_bits = me
       */
      assign( right_bits[i],
              unop( Iop_64to8,
                    binop( Iop_Or64,
                           binop( Iop_And64,   // mb < me
                                  unop( Iop_Not64, mkexpr( invert_mask[i] ) ),
                                  binop( Iop_Sub64,
                                         mkU64( word_size - 1 ),
                                         mkexpr( me[i] ) ) ),
                           binop( Iop_And64,   // me < mb
                                  mkexpr( invert_mask[i] ),
                                  binop( Iop_Sub64,
                                         mkU64( word_size - 1 + 1),
                                         mkexpr( mb[i] ) ) ) ) ) );

      /* create mask for 32-bit word or 64-bit word */
      assign( tmp_mask[i],
              binop( Iop_Shl64,
                     binop( Iop_Shr64,
                            binop( Iop_Shr64,
                                   binop( Iop_Shl64,
                                          mkU64( 0xFFFFFFFFFFFFFFFF ),
                                          mkexpr( left_bits[i] ) ),
                                   mkexpr( left_bits[i] ) ),
                            mkexpr( right_bits[i] ) ),
                     mkexpr( right_bits[i] ) ) );

      assign( mask[i],
              binop( Iop_64HLtoV128,
                     mkU64( 0 ),
                     binop( Iop_Or64,
                            binop( Iop_And64,
                                   unop( Iop_Not64, mkexpr( invert_mask[i] ) ),
                                   mkexpr( tmp_mask[i] ) ),
                            binop( Iop_And64,
                                   mkexpr( invert_mask[i] ),
                                   /* Need to make sure mask is only the size
                                      desired word.*/
                                   binop( Iop_And64,
                                          mkU64( word_mask ),
                                          unop( Iop_Not64,
                                                mkexpr( tmp_mask[i] ) ) ) ))));

      /* Need to rotate vA using a left and right shift of vA OR'd together
       * then ANDed with the mask.
       */
      assign( vA_word[i], binop( Iop_AndV128,
                                 mkexpr( vA ),
                                 binop( Iop_ShlV128,
                                        binop( Iop_64HLtoV128,
                                               mkU64( 0 ),
                                               mkU64( word_mask ) ),
                                        mkU8( ( num_words - 1 - i )
                                              * word_size ) ) ) );
      assign( tmp128[i],
              binop( Iop_AndV128,
                     binop( Iop_ShlV128,
                            mkexpr( mask[i] ),
                            mkU8( ( num_words - 1 - i) * word_size ) ),
                     binop( Iop_OrV128,
                            binop( Iop_ShlV128,
                                   mkexpr( vA_word[i] ),
                                   mkexpr( shift[i] ) ),
                            binop( Iop_ShrV128,
                                   mkexpr( vA_word[i] ),
                                   unop( Iop_32to8,
                                         binop(Iop_Sub32,
                                               mkU32( word_size ),
                                               unop( Iop_8Uto32,
                                                     mkexpr( shift[i] ) ) )
                                      ) ) ) ) );
   }

   switch (opc2) {
   case 0x85: // vrlwmi,  Vector Rotate Left Word then Mask Insert
      DIP("vrlwmi %d,%d,v%d", vT_addr, vA_addr, vB_addr);

      assign( src3, getVReg( vT_addr ) );
      assign( mask128, unop( Iop_NotV128,
                             mkOr4_V128_expr( binop( Iop_ShlV128,
                                                     mkexpr( mask[0] ),
                                                     mkU8( 96 ) ),
                                              binop( Iop_ShlV128,
                                                     mkexpr( mask[1] ),
                                                     mkU8( 64 ) ),
                                              binop( Iop_ShlV128,
                                                     mkexpr( mask[2] ),
                                                     mkU8( 32 ) ),
                                              mkexpr( mask[3] ) ) ) );
      assign( vT, binop( Iop_OrV128,
                         binop( Iop_AndV128,
                                mkexpr( src3 ),
                                mkexpr( mask128 ) ),
                         mkOr4_V128( tmp128[0], tmp128[1],
                                     tmp128[2], tmp128[3] ) ) );
      break;

   case 0xC5: // vrldmi,  Vector Rotate Left Double word then Mask Insert
      DIP("vrldmi %d,%d,v%d", vT_addr, vA_addr, vB_addr);

      assign( src3, getVReg( vT_addr ) );
      assign( mask128, unop( Iop_NotV128,
                             binop( Iop_OrV128,
                                    binop( Iop_ShlV128,
                                           mkexpr( mask[0] ),
                                           mkU8( 64 ) ),
                                    mkexpr( mask[1] ) ) ) );

      assign( vT, binop( Iop_OrV128,
                         binop( Iop_AndV128,
                                mkexpr( src3 ),
                                mkexpr( mask128 ) ),
                         binop( Iop_OrV128,
                                mkexpr( tmp128[0] ),
                                mkexpr( tmp128[1] ) ) ) );
      break;

   case 0x185: // vrlwnm,  Vector Rotate Left Word then AND with Mask
      DIP("vrlwnm %d,%d,v%d", vT_addr, vA_addr, vB_addr);
      assign( vT, mkOr4_V128( tmp128[0], tmp128[1], tmp128[2], tmp128[3] ) );
      break;

   case 0x1C5: // vrldnm,  Vector Rotate Left Doubleword then AND with Mask
      DIP("vrldnm %d,%d,v%d", vT_addr, vA_addr, vB_addr);
      assign( vT, binop( Iop_OrV128,
                         mkexpr( tmp128[0] ),
                         mkexpr( tmp128[1] ) ) );
      break;
   }

   putVReg( vT_addr, mkexpr( vT ) );
   return True;
}

/*
  AltiVec Vector Extract Element Instructions
*/
static Bool dis_av_insert_element ( UInt prefix, UInt theInstr )
{
   /* VX-Form,
    * Source, index and value are GPR, destination is a vector register.
    */
   UChar opc1    = ifieldOPC( theInstr );
   UChar VRT = ifieldRegDS( theInstr );
   UChar rA_addr = ifieldRegA( theInstr );
   UChar VRB = ifieldRegB( theInstr );
   UInt  opc2    = IFIELD( theInstr, 0, 11 );
   UChar rVT_addr = VRT;
   UChar rVB_addr = VRB;

   IRTemp rA = newTemp( Ity_I64 );
   IRTemp vTmp = newTemp( Ity_V128 );
   IRTemp index = newTemp( Ity_I64 );
   UInt max_index_in_src = 15;

   /* There is no prefixed version of these instructions.  */
   vassert( !prefix_instruction( prefix ) );

   assign( vTmp, getVReg( rVT_addr ) );
   assign( rA, getIReg( rA_addr ) );
   assign ( index, binop( Iop_Sub64,
                          mkU64( 15 ),
                          mkexpr( rA ) ) );

   if ( opc1 != 0x4 ) {
      vex_printf("dis_av_insert_element(ppc)(instr)\n");
      return False;
   }

   switch ( opc2 ) {
   case 0x00F: // vinsbvlx, vector insert Byte from VSR Left-indexed VX form
   {
      IRTemp src = newTemp( Ity_I64 );
      IRTemp adj_index = newTemp( Ity_I64 );
      IRTemp rVB = newTemp( Ity_V128 );

      DIP("vinsbvlx v%d,%d,v%d", VRT, rA_addr, VRB);

      assign( rVB, getVReg( rVB_addr ) );
      assign( adj_index, binop( Iop_Sub64,
                                mkU64( max_index_in_src ),
                                binop( Iop_And64,
                                       mkU64( 0xF),
                                       mkexpr( rA ) ) ) );

      /* Extract byte in rVB[56:63], that is byte 8 counting from the right */
      assign( src, extract_field_from_vector( rVB, mkU64( 8 ), 0xFF ) );
      putVReg( rVT_addr,
               insert_field_into_vector( vTmp, mkexpr( adj_index ),
                                         mkexpr( src), mkU64( 0xFF ) ) );
   }
   break;

   case 0x10F: // vinsbvrx, vector insert Byte from VSR Right-indexed VX form
   {
      IRTemp src = newTemp( Ity_I64 );
      IRTemp rVB = newTemp( Ity_V128 );
      IRTemp adj_index = newTemp( Ity_I64 );

      DIP("vinsbvrx v%d,%d,v%d", VRT, rA_addr, VRB);

      assign( rVB, getVReg( rVB_addr ) );

      assign( adj_index, binop( Iop_And64, mkexpr( rA ), mkU64( 0xF ) ) );
      /* Extract byte in rVB[56:63], that is byte 8 counting from the right */
      assign( src, extract_field_from_vector( rVB, mkU64( 8 ), 0xFF ) );
      putVReg( rVT_addr,
               insert_field_into_vector( vTmp, mkexpr( rA ),
                                         mkexpr( src), mkU64( 0xFF ) ) );
   }
   break;

   case 0x04F:
   // vinshvlx, vector insert Halfword from VSR Left-indexed VX form
   {
      IRTemp src = newTemp( Ity_I64 );
      IRTemp adj_index = newTemp( Ity_I64 );
      IRTemp rVB = newTemp( Ity_V128 );

      DIP("vinshvlx v%d,%d,v%d", VRT, rA_addr, VRB);

      assign( rVB, getVReg( rVB_addr ) );
      assign( adj_index, binop( Iop_Sub64,
                                mkU64( max_index_in_src - 1 ),
                                binop( Iop_And64,
                                       mkexpr( rA ),
                                       mkU64( 0xF ) ) ) );

      /* Extract half word rVB[48:63], bytes [9:8] counting from the right */
      assign( src, extract_field_from_vector( rVB, mkU64( 8 ), 0xFFFF ) );
      putVReg( rVT_addr,
               insert_field_into_vector( vTmp, mkexpr( adj_index ),
                                         mkexpr( src), mkU64( 0xFFFF ) ) );
   }
   break;

   case 0x14F:
   // vinshvrx, vector insert Halfword from VSR Right-indexed VX form
   {
      IRTemp src = newTemp( Ity_I64 );
      IRTemp rVB = newTemp( Ity_V128 );
      IRTemp adj_index = newTemp( Ity_I64 );

      DIP("vinshvrx v%d,%d,v%d", VRT, rA_addr, VRB);

      assign( rVB, getVReg( rVB_addr ) );

      assign( adj_index, binop( Iop_And64, mkexpr( rA ), mkU64( 0xF ) ) );

      /* Extract half word rVB[48:63], bytes [9:8] counting from the right */
      assign( src, extract_field_from_vector( rVB, mkU64( 8 ), 0xFFFF ) );
      putVReg( rVT_addr,
               insert_field_into_vector( vTmp, mkexpr( rA ), mkexpr( src),
                                         mkU64( 0xFFFF ) ) );
   }
   break;

   case 0x08F:
   // vinswvlx, vector insert Word from VSR Left-indexed VX form
   {
      IRTemp src = newTemp( Ity_I64 );
      IRTemp adj_index = newTemp( Ity_I64 );
      IRTemp rVB = newTemp( Ity_V128 );

      DIP("vinswvlx v%u,%u,v%u", VRT, rA_addr, VRB);

      assign( rVB, getVReg( rVB_addr ) );
      assign( adj_index, binop( Iop_Sub64,
                                mkU64( max_index_in_src - 3 ),
                                binop( Iop_And64,
                                       mkU64( 0xF ),
                                       mkexpr( rA ) ) ) );

      /* Extract word rVB[32:63], bytes [15:8] counting from the right */
      assign( src, extract_field_from_vector( rVB, mkU64( 8 ), 0xFFFFFFFF ) );
      putVReg( rVT_addr,
               insert_field_into_vector( vTmp, mkexpr( adj_index ),
                                         mkexpr( src), mkU64( 0xFFFFFFFF ) ) );
   }
   break;

   case 0x18F:
   // vinswvrx, vector insert Word from VSR Right-indexed VX form
   {
      IRTemp src = newTemp( Ity_I64 );
      IRTemp rVB = newTemp( Ity_V128 );
      IRTemp adj_index = newTemp( Ity_I64 );

      DIP("vinswvrx v%u,%u,v%u", VRT, rA_addr, VRB);

      assign( rVB, getVReg( rVB_addr ) );

      assign( adj_index, binop( Iop_And64, mkexpr( rA ), mkU64( 0xF ) ) );
      /* Extract word in rVB[32:63], bytes [15:8] counting from the right */
      assign( src, extract_field_from_vector( rVB, mkU64( 8 ), 0xFFFFFFFF ) );

      putVReg( rVT_addr,
               insert_field_into_vector( vTmp, mkexpr( rA ),
                                         mkexpr( src), mkU64( 0xFFFFFFFF ) ) );
   }
   break;

   case 0x0CF:
   // vinsw, vector insert Word from GPR VX form
   {
      IRTemp rB = newTemp( Ity_I64 );
      UChar rB_addr = ifieldRegB( theInstr );
      UInt UIM  = IFIELD( theInstr, 16, 4 );
      UInt max_bytes_in_src = 15;

      DIP("vinsw v%u,%u,%u", VRT, rB_addr, UIM);

      assign( rB, getIReg( rB_addr ) );

      putVReg( rVT_addr,
               insert_field_into_vector( vTmp,
                                         mkU64( max_bytes_in_src - 3 - UIM ),
                                         mkexpr( rB), mkU64( 0xFFFFFFFF ) ) );
   }
   break;

   case 0x1CF:
   // vinsd, vector insert Doubleword from GPR VX form
   {
      IRTemp rB = newTemp( Ity_I64 );
      UChar rB_addr = ifieldRegB( theInstr );
      UInt UIM  = IFIELD( theInstr, 16, 4 );
      UInt max_bytes_in_src = 15;

      DIP("vinsd v%u,%u,%u", VRT, rB_addr, UIM);

      assign( rB, getIReg( rB_addr ) );

      putVReg( rVT_addr,
               insert_field_into_vector( vTmp,
                                         mkU64( max_bytes_in_src - 7 - UIM ),
                                         mkexpr( rB ),
                                         mkU64( 0xFFFFFFFFFFFFFFFFULL ) ) );
   }
   break;

   case 0x20F: // vinsblx, vector insert Byte from GPR Left-indexed VX form
   {
      IRTemp rB = newTemp( Ity_I64 );
      UChar rB_addr = ifieldRegB( theInstr );

      DIP("vinsblx v%u,%u,%u", VRT, rA_addr, rB_addr);

      assign( rB, getIReg( rB_addr ) );
      putVReg( rVT_addr,
               insert_field_into_vector( vTmp,
                                         binop( Iop_Sub64,
                                                mkU64( max_index_in_src ),
                                                mkexpr( rA ) ),
                                         mkexpr( rB ), mkU64( 0xFF ) ) );
      break;
   }
   case 0x30F: // vinsbrx, vector insert Byte from GPR Right-indexed VX form
   {
      IRTemp rB = newTemp( Ity_I64 );
      UChar rB_addr = ifieldRegB( theInstr );

      DIP("vinsbrx v%u,%u,%u", VRT, rA_addr, rB_addr);

      assign( rB, getIReg( rB_addr ) );
      putVReg( rVT_addr,
               insert_field_into_vector( vTmp, mkexpr( rA ),
                                         mkexpr( rB ), mkU64( 0xFF ) ) );
      break;
   }
   case 0x24F: // vinshlx, vector insert Halfword from GPR Left-indexed VX form
   {
      IRTemp rB = newTemp( Ity_I64 );
      UChar rB_addr = ifieldRegB( theInstr );

      DIP("vinshlx v%u,%u,%u", VRT, rA_addr, rB_addr);

      /* insert_field_into_vector assumes right-indexed, convert argument  */
      assign( rB, getIReg( rB_addr ) );
      putVReg( rVT_addr,
               insert_field_into_vector( vTmp,
                                         binop( Iop_Sub64,
                                                mkU64( max_index_in_src-1 ),
                                                mkexpr( rA ) ),
                                         mkexpr( rB ), mkU64( 0xFFFF ) ) );
      break;
   }
   case 0x34F:// vinshrx, vector insert Halfword from GPR Right-indexed VX form
   {
      IRTemp rB = newTemp( Ity_I64 );
      UChar rB_addr = ifieldRegB( theInstr );

      DIP("vinshrx v%u,%u,%u", VRT, rA_addr, rB_addr);

      assign( rB, getIReg( rB_addr ) );
      putVReg( rVT_addr,
               insert_field_into_vector( vTmp, mkexpr( rA ),
                                         mkexpr( rB ), mkU64( 0xFFFF ) ) );
      break;
   }
   case 0x28F: // vinswlx, vector insert Word from GPR Left-indexed VX form
   {
      IRTemp rB = newTemp( Ity_I64 );
      UChar rB_addr = ifieldRegB( theInstr );

      DIP("vinswlx v%u,%u,%u", VRT, rA_addr, rB_addr);

      /* insert_field_into_vector assumes right-indexed, convert argument  */
      assign( rB, getIReg( rB_addr ) );
      putVReg( rVT_addr,
               insert_field_into_vector( vTmp,
                                         binop( Iop_Sub64,
                                                mkU64( max_index_in_src-3 ),
                                                mkexpr( rA ) ),
                                         mkexpr( rB ), mkU64( 0xFFFFFFFF ) ) );
      break;
   }
   case 0x38F:// vinswrx, vector insert Word from GPR Right-indexed VX form
   {
      IRTemp rB = newTemp( Ity_I64 );
      UChar rB_addr = ifieldRegB( theInstr );

      DIP("vinswrx v%u,%u,%u", VRT, rA_addr, rB_addr);

      assign( rB, getIReg( rB_addr ) );
      putVReg( rVT_addr,
               insert_field_into_vector( vTmp, mkexpr( rA ),
                                         mkexpr( rB ), mkU64( 0xFFFFFFFF ) ) );
      break;
   }
   case 0x2CF:
   {
      // vinsdlx, vector insert Doubleword from GPR Left-indexed VX form
      IRTemp rB = newTemp( Ity_I64 );
      UChar rB_addr = ifieldRegB( theInstr );

      DIP("vinsdlx v%u,%u,%u", VRT, rA_addr, rB_addr);

      /* insert_field_into_vector assumes right-indexed, convert argument  */
      assign( rB, getIReg( rB_addr ) );
      putVReg( rVT_addr,
               insert_field_into_vector( vTmp,
                                         binop( Iop_Sub64,
                                                mkU64( max_index_in_src-7 ),
                                                mkexpr( rA ) ),
                                         mkexpr( rB ),
                                         mkU64( 0xFFFFFFFFFFFFFFFFULL ) ) );
      break;
   }
   case 0x3CF:
   {
      // vinsdrx, vector insert Doubleword from GPR Right-indexed VX form
      IRTemp rB = newTemp( Ity_I64 );
      UChar rB_addr = ifieldRegB( theInstr );

      DIP("vinsdrx v%u,%u,%u", VRT, rA_addr, rB_addr);

      assign( rB, getIReg( rB_addr ) );
      putVReg( rVT_addr,
               insert_field_into_vector( vTmp, mkexpr( rA ),
                                         mkexpr( rB ),
                                         mkU64( 0xFFFFFFFFFFFFFFFFULL ) ) );
      break;
   }
   default:
      vex_printf("dis_av_extract_element(ppc)(opc2)\n");
      return False;
   }
   return True;
}

/*
  AltiVec Vector Extract Element Instructions
*/
static Bool dis_av_extract_element ( UInt prefix, UInt theInstr )
{
   /* VX-Form,
    * sorta destination and first source are GPR not vector registers
    */

   UChar opc1    = ifieldOPC( theInstr );
   UChar rT_addr = ifieldRegDS( theInstr );
   UChar rA_addr = ifieldRegA( theInstr );
   UChar vB_addr = ifieldRegB( theInstr );
   UInt  opc2    = IFIELD( theInstr, 0, 11 );

   IRTemp vB = newTemp( Ity_V128 );
   IRTemp rA = newTemp( Ity_I64 );
   IRTemp rT = newTemp( Ity_I64 );

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   assign( vB, getVReg( vB_addr ) );
   assign( rA, getIReg( rA_addr ) );

   if ( opc1 != 0x4 ) {
      vex_printf("dis_av_extract_element(ppc)(instr)\n");
      return False;
   }

   switch ( opc2 ) {
   case 0x60D: // vextublx, vector extract unsigned Byte Left-indexed
      DIP("vextublx %d,%d,v%d", rT_addr, rA_addr, vB_addr);

      assign( rT, extract_field_from_vector( vB,
                                             binop( Iop_Sub64,
                                                    mkU64( 15 ),
                                                    mkexpr( rA ) ),
                                             0xFF ) );

      break;

   case 0x64D: // vextuhlx, vector extract unsigned Halfword Left-indexed
      DIP("vextuhlx %d,%d,v%d", rT_addr, rA_addr, vB_addr);

      assign( rT, extract_field_from_vector( vB,
                                             binop( Iop_Sub64,
                                                    mkU64( 14 ),
                                                    mkexpr( rA ) ),
                                             0xFFFF ) );
      break;

   case 0x68D: // vextuwlx, vector extract unsigned Word Left-indexed
      DIP("vextuwlx %d,%d,v%d", rT_addr, rA_addr, vB_addr);

      assign( rT, extract_field_from_vector( vB,
                                             binop( Iop_Sub64,
                                                    mkU64( 12 ),
                                                    mkexpr( rA ) ),
                                             0xFFFFFFFF ) );
      break;

   case 0x70D: // vextubrx, vector extract unsigned Byte Right-indexed
      DIP("vextubrx %d,%d,v%d", rT_addr, rA_addr, vB_addr);

      assign( rT, extract_field_from_vector( vB, mkexpr( rA ), 0xFF ) );
      break;

   case 0x74D: // vextuhrx, vector extract unsigned Halfword Right-indexed
      DIP("vextuhrx %d,%d,v%d", rT_addr, rA_addr, vB_addr);

      assign( rT, extract_field_from_vector( vB, mkexpr( rA ), 0xFFFF ) );
      break;

   case 0x78D: // vextuwrx, vector extract unsigned Word Right-indexed
      DIP("vextuwrx %d,%d,v%d", rT_addr, rA_addr, vB_addr);

      assign( rT, extract_field_from_vector( vB, mkexpr( rA ), 0xFFFFFFFF ) );
      break;

   default:
      vex_printf("dis_av_extract_element(ppc)(opc2)\n");
      return False;
   }
   putIReg( rT_addr, mkexpr( rT ) );
   return True;
}

/*
 * VSX scalar and vector convert instructions
 */
static Bool
dis_vx_conv ( UInt prefix, UInt theInstr, UInt opc2 )
{
   /* XX2-Form */
   UChar opc1 = ifieldOPC( theInstr );
   UChar XT = ifieldRegXT( theInstr );
   UChar XB = ifieldRegXB( theInstr );
   IRTemp xB, xB2;
   IRTemp b3, b2, b1, b0;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   xB = xB2 = IRTemp_INVALID;

   if (opc1 != 0x3C) {
      vex_printf( "dis_vx_conv(ppc)(instr)\n" );
      return False;
   }

   /* Create and assign temps only as needed for the given instruction. */
   switch (opc2) {
      // scalar double-precision floating point argument
      case 0x2B0: case 0x0b0: case 0x290: case 0x212: case 0x216: case 0x090:
         xB = newTemp(Ity_F64);
         assign( xB,
                 unop( Iop_ReinterpI64asF64,
                       unop( Iop_V128HIto64, getVSReg( XB ) ) ) );
         break;
      // vector double-precision floating point arguments
      case 0x1b0: case 0x312: case 0x390: case 0x190: case 0x3B0:

         xB = newTemp(Ity_F64);
         xB2 = newTemp(Ity_F64);
         assign( xB,
                 unop( Iop_ReinterpI64asF64,
                       unop( Iop_V128HIto64, getVSReg( XB ) ) ) );
         assign( xB2,
                 unop( Iop_ReinterpI64asF64,
                       unop( Iop_V128to64, getVSReg( XB ) ) ) );
         break;
      // vector single precision or [un]signed integer word arguments
      case 0x130: case 0x392: case 0x330: case 0x310: case 0x110:
      case 0x1f0: case 0x1d0:
         b3 = b2 = b1 = b0 = IRTemp_INVALID;
         breakV128to4x32(getVSReg(XB), &b3, &b2, &b1, &b0);
         break;
         // vector [un]signed integer doubleword argument
      case 0x3f0: case 0x370: case 0x3d0: case 0x350:
         xB = newTemp(Ity_I64);
         assign( xB, unop( Iop_V128HIto64, getVSReg( XB ) ) );
         xB2 = newTemp(Ity_I64);
         assign( xB2, unop( Iop_V128to64, getVSReg( XB ) ) );
         break;
      // scalar [un]signed integer doubleword argument
      case 0x250: case 0x270: case 0x2D0: case 0x2F0:
         xB = newTemp(Ity_I64);
         assign( xB, unop( Iop_V128HIto64, getVSReg( XB ) ) );
         break;
      // scalar single precision argument
      case 0x292: // xscvspdp
         xB  = newTemp(Ity_I32);

         assign( xB, handle_SNaN_to_QNaN_32(unop( Iop_64HIto32,
                                                  unop( Iop_V128HIto64,
                                                        getVSReg( XB ) ) ) ) );
         break;
      case 0x296: // xscvspdpn (non signaling version of xscvpdp)
         xB = newTemp(Ity_I32);
         assign( xB,
                 unop( Iop_64HIto32, unop( Iop_V128HIto64, getVSReg( XB ) ) ) );
         break;

      /* Certain instructions have their complete implementation in the main switch statement
       * that follows this one; thus we have a "do nothing" case for those instructions here.
       */
      case 0x170: case 0x150:
         break; // do nothing

      default:
         vex_printf( "dis_vx_conv(ppc)(opc2)\n" );
         return False;
   }


   switch (opc2) {
      case 0x2B0:
         // xscvdpsxds (VSX Scalar truncate Double-Precision to integer and Convert
         //             to Signed Integer Doubleword format with Saturate)
         DIP("xscvdpsxds v%u,v%u\n", XT, XB);
         putVSReg( XT,
                   binop( Iop_64HLtoV128, binop( Iop_F64toI64S,
                                                 mkU32( Irrm_ZERO ),
                                                 mkexpr( xB ) ), mkU64( 0 ) ) );
         break;
      case 0x0b0: // xscvdpsxws (VSX Scalar truncate Double-Precision to integer and
                  //             Convert to Signed Integer Word format with Saturate)
         DIP("xscvdpsxws v%u,v%u\n", XT, XB);
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          unop( Iop_32Sto64,
                                binop( Iop_F64toI32S,
                                       mkU32( Irrm_ZERO ),
                                       mkexpr( xB ) ) ),
                                       mkU64( 0ULL ) ) );
         break;
      case 0x290: // xscvdpuxds (VSX Scalar truncate Double-Precision integer and Convert
                  //             to Unsigned Integer Doubleword format with Saturate)
         DIP("xscvdpuxds v%u,v%u\n", XT, XB);
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          binop( Iop_F64toI64U,
                                 mkU32( Irrm_ZERO ),
                                 mkexpr( xB ) ),
                                 mkU64( 0ULL ) ) );
         break;
      case 0x270:
         // xscvsxdsp (VSX Scalar Convert and round Signed Integer Doubleword
         //             to Single-Precision format)
         DIP("xscvsxdsp v%u,v%u\n", XT, XB);
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          unop( Iop_ReinterpF64asI64,
                                binop( Iop_RoundF64toF32,
                                       get_IR_roundingmode(),
                                       binop( Iop_I64StoF64,
                                              get_IR_roundingmode(),
                                              mkexpr( xB ) ) ) ),
                          mkU64( 0 ) ) );
         break;
      case 0x2F0:
         // xscvsxddp (VSX Scalar Convert and round Signed Integer Doubleword to
         //            Double-Precision format)
         DIP("xscvsxddp v%u,v%u\n", XT, XB);
         putVSReg( XT,
                   binop( Iop_64HLtoV128, unop( Iop_ReinterpF64asI64,
                                                binop( Iop_I64StoF64, get_IR_roundingmode(),
                                                       mkexpr( xB ) ) ),
                                                       mkU64( 0 ) ) );
         break;
      case 0x250:
         // xscvuxdsp (VSX Scalar Convert and round Unsigned Integer
         //            Doubleword to Singel-Precision format)
         DIP("xscvuxdsp v%u,v%u\n", XT, XB);
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          unop( Iop_ReinterpF64asI64,
                                binop( Iop_RoundF64toF32,
                                       get_IR_roundingmode(),
                                       binop( Iop_I64UtoF64,
                                              get_IR_roundingmode(),
                                              mkexpr( xB ) ) ) ),
                          mkU64( 0 ) ) );
         break;
      case 0x2D0:
         // xscvuxddp (VSX Scalar Convert and round Unsigned Integer Doubleword to
         //            Double-Precision format)
         DIP("xscvuxddp v%u,v%u\n", XT, XB);
         putVSReg( XT,
                   binop( Iop_64HLtoV128, unop( Iop_ReinterpF64asI64,
                                                binop( Iop_I64UtoF64, get_IR_roundingmode(),
                                                       mkexpr( xB ) ) ),
                                                       mkU64( 0 ) ) );
         break;
      case 0x1b0: // xvcvdpsxws (VSX Vector truncate Double-Precision to integer and Convert
                  //             to Signed Integer Word format with Saturate)
      case 0x190: // xvcvdpuxws (VSX Vector truncate Double-Precision to integer and
                  //             Convert to Unsigned Integer Word format with Saturate)
      {
         IRTemp value_f64[2];
         IRTemp Result_32[2];
         IRTemp Result_32_tmp[2];
         IRTemp nan_mask[2];
         IRTemp underflow_mask[2];
         IRTemp overflow_mask[2];
         IRTemp error_mask[2];
         IRTemp error_value[2];
         IRTemp tmp_64[2];

         Int i;
         Int underflow_value;
         Int overflow_value;
         IRExpr* rmZero = mkU32(Irrm_ZERO);

         value_f64[0] = newTemp(Ity_F64);
         assign( value_f64[0], mkexpr( xB ) );

         value_f64[1] = newTemp(Ity_F64);
         assign( value_f64[1], mkexpr( xB2 ) );

         for ( i = 0; i < 2; i++) {
            Result_32[i] = newTemp(Ity_I32);
            Result_32_tmp[i] = newTemp(Ity_I32);
            nan_mask[i] = newTemp(Ity_I32);
            underflow_mask[i] = newTemp(Ity_I32);
            overflow_mask[i] = newTemp(Ity_I32);
            error_mask[i] = newTemp(Ity_I32);
            error_value[i] = newTemp(Ity_I32);
            tmp_64[i]  = newTemp(Ity_I64);

            if ( opc2 == 0x1b0 ) {    // xvcvdpsxws
               assign(Result_32_tmp[i], binop(Iop_F64toI32S,
                                              rmZero, mkexpr( value_f64[i] ) ) );

               /* result of Iop_CmpF64 is 0x01 if A < -2^31.  */
               assign( underflow_mask[i],
                       unop( Iop_1Sto32,
                             unop( Iop_32to1,
                                   binop( Iop_CmpF64,
                                          mkexpr( value_f64[i] ),
                                          unop( Iop_ReinterpI64asF64,
                                                mkU64( 0xC1E0000000000000 ))))));
               overflow_value  = 0x7FFFFFFF;
               underflow_value = 0x80000000;

            } else {                  // xvcvdpuxws
               assign( Result_32_tmp[i],
                       binop( Iop_F64toI32U,
                              mkU32( Irrm_ZERO ),
                              mkexpr( value_f64[i] ) ) );

               /* result of Iop_CmpF64 is 0x01 if A < 0.  */
               assign( underflow_mask[i],
                       unop( Iop_1Sto32,
                             unop( Iop_32to1,
                                   binop( Iop_CmpF64,
                                          mkexpr( value_f64[i] ),
                                          unop( Iop_ReinterpI64asF64,
                                                mkU64( 0x0 ) ) ) ) ) );
               overflow_value  = 0xFFFFFFFF;
               underflow_value = 0;
            }

            /* Check if input is NaN, output is 0x80000000.
               if input < -2^31,  output is 0x80000000.
               if input >  2^31 - 1, output is 0x7FFFFFFF  */
            assign( tmp_64[i], unop (Iop_ReinterpF64asI64,
                                     mkexpr( value_f64[i] ) ) );

            assign( nan_mask[i], unop( Iop_1Sto32,
                                    is_NaN( Ity_I64, tmp_64[i] ) ) );

            /* result of Iop_CmpF64 is 0x00 if A > 2^31 - 1.  */
            assign( overflow_mask[i],
                    unop( Iop_1Sto32,
                          binop( Iop_CmpEQ32,
                                 mkU32( 0 ),
                                 binop( Iop_CmpF64,
                                        mkexpr( value_f64[i] ),
                                        unop( Iop_ReinterpI64asF64,
                                              mkU64( 0x41DFFFFFFFC00000 ))))));

            assign( error_mask[i], binop( Iop_Or32, mkexpr( overflow_mask[i] ),
                                          binop( Iop_Or32,
                                                 mkexpr( underflow_mask[i] ),
                                                 mkexpr( nan_mask[i] ) ) ) );

            if ( opc2 == 0x1b0 ) {    // xvcvdpsxws
               /* NaN takes precedence over underflow/overflow for vxcvdpsxws */
               assign( error_value[i],
                       binop( Iop_Or32,
                              binop( Iop_And32,
                                     unop( Iop_Not32, mkexpr( nan_mask[i] ) ),
                                     binop( Iop_Or32,
                                            binop( Iop_And32,
                                                   mkexpr( overflow_mask[i] ),
                                                   mkU32( overflow_value ) ),
                                            binop( Iop_And32,
                                                   mkexpr( underflow_mask[i] ),
                                                   mkU32( underflow_value ) ) ) ),
                              binop( Iop_And32,
                                     mkexpr( nan_mask[i] ),
                                     mkU32( 0x80000000 ) ) ) );
            } else {
               /* Less then zeo takes precedence over NaN/overflow
                  for vxcvdpuxws in the hardware. Matching the HW here
                  but it does not appear to match ISA.  */
               assign( error_value[i],
                       binop( Iop_Or32,
                              binop( Iop_And32,
                                     unop( Iop_Not32,
                                           mkexpr( underflow_mask[i] ) ),
                                     binop( Iop_Or32,
                                            binop( Iop_And32,
                                                   mkexpr( overflow_mask[i] ),
                                                   mkU32( overflow_value ) ),
                                            binop( Iop_And32,
                                                   mkexpr( nan_mask[i] ),
                                                   mkU32( 0x80000000 ) ) ) ),
                              binop( Iop_And32,
                                     mkexpr( underflow_mask[i] ),
                                     mkU32( underflow_value ) ) ) );
            }

            assign( Result_32[i], binop( Iop_Or32,
                                         binop( Iop_And32,
                                                mkexpr( Result_32_tmp[i] ),
                                                unop( Iop_Not32,
                                                      mkexpr( error_mask[i] ) ) ),
                                         binop( Iop_And32,
                                                mkexpr( error_value[i] ),
                                                mkexpr( error_mask[i] ) ) ) );
         }

         if ( opc2 == 0x1b0 ) {
            DIP("xvcvdpsxws v%u,v%u\n",  XT, XB);

         } else {
            DIP("xvcvdpuxws v%u,v%u", XT, XB);
         }

         /* Result is put in the hi and low 32-bits of the double word result.  */
         putVSReg( XT,
                binop( Iop_64HLtoV128,
                       binop( Iop_32HLto64,
                              mkexpr( Result_32[0] ),
                              mkexpr( Result_32[0] ) ),
                       binop( Iop_32HLto64,
                              mkexpr( Result_32[1] ),
                              mkexpr( Result_32[1] ) ) ) );
         break;
      }
      case 0x130: case 0x110: // xvcvspsxws, xvcvspuxws
         //  (VSX Vector truncate Single-Precision to integer and
         //   Convert to [Un]signed Integer Word format with Saturate)
      {
         IRExpr * b0_result, * b1_result, * b2_result, * b3_result;
         IRTemp tempResult = newTemp(Ity_V128);
         IRTemp res0 = newTemp(Ity_I32);
         IRTemp res1 = newTemp(Ity_I32);
         IRTemp res2 = newTemp(Ity_I32);
         IRTemp res3 = newTemp(Ity_I32);
         IRTemp hi64 = newTemp(Ity_I64);
         IRTemp lo64 = newTemp(Ity_I64);
         Bool un_signed = (opc2 == 0x110);
         IROp op = un_signed ? Iop_QF32toI32Ux4_RZ : Iop_QF32toI32Sx4_RZ;

         DIP("xvcvsp%sxws v%u,v%u\n", un_signed ? "u" : "s", XT, XB);
         /* The xvcvsp{s|u}xws instruction is similar to vct{s|u}xs, except if src is a NaN,
          * then result is set to 0x80000000.  */
         assign(tempResult, unop(op, getVSReg(XB)));
         assign( hi64, unop(Iop_V128HIto64, mkexpr(tempResult)) );
         assign( lo64, unop(Iop_V128to64,   mkexpr(tempResult)) );
         assign( res3, unop(Iop_64HIto32, mkexpr(hi64)) );
         assign( res2, unop(Iop_64to32,   mkexpr(hi64)) );
         assign( res1, unop(Iop_64HIto32, mkexpr(lo64)) );
         assign( res0, unop(Iop_64to32,   mkexpr(lo64)) );

         b3_result = IRExpr_ITE(is_NaN(Ity_I32, b3),
                                // then: result is 0x{8|0}80000000
                                mkU32(un_signed ? 0x00000000 : 0x80000000),
                                // else: result is from the Iop_QFtoI32{s|u}x4_RZ
                                mkexpr(res3));
         b2_result = IRExpr_ITE(is_NaN(Ity_I32, b2),
                                // then: result is 0x{8|0}80000000
                                mkU32(un_signed ? 0x00000000 : 0x80000000),
                                // else: result is from the Iop_QFtoI32{s|u}x4_RZ
                                mkexpr(res2));
         b1_result = IRExpr_ITE(is_NaN(Ity_I32, b1),
                                // then: result is 0x{8|0}80000000
                                mkU32(un_signed ? 0x00000000 : 0x80000000),
                                // else: result is from the Iop_QFtoI32{s|u}x4_RZ
                                mkexpr(res1));
         b0_result = IRExpr_ITE(is_NaN(Ity_I32, b0),
                                // then: result is 0x{8|0}80000000
                                mkU32(un_signed ? 0x00000000 : 0x80000000),
                                // else: result is from the Iop_QFtoI32{s|u}x4_RZ
                                mkexpr(res0));

         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          binop( Iop_32HLto64, b3_result, b2_result ),
                          binop( Iop_32HLto64, b1_result, b0_result ) ) );
         break;
      }
      case 0x212: // xscvdpsp (VSX Scalar round Double-Precision to single-precision and
                  //           Convert to Single-Precision format
                  // Apr 2019 update - write the result to both halves of the
                  // target VSR.  (see bug 401827,401828).
         DIP("xscvdpsp v%u,v%u\n", XT, XB);
         IRTemp ResultI32a = newTemp(Ity_I32);
         assign(ResultI32a,  unop( Iop_ReinterpF32asI32,
                                  unop( Iop_TruncF64asF32,
                                        binop( Iop_RoundF64toF32,
                                               get_IR_roundingmode(),
                                               mkexpr( xB ) ) ) ) );
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          binop( Iop_32HLto64,
                                 mkexpr(ResultI32a ),
                                 mkexpr(ResultI32a ) ),
                          mkU64( 0ULL ) ) );
         break;
      case 0x216: /* xscvdpspn (VSX Scalar convert scalar Single-Precision to
                              vector Single-Precision non-signalling */
                  // Apr 2019 update - write the result to both halves of the
                  // target VSR.  (see bug 401827,401828).
         DIP("xscvdpspn v%u,v%u\n", XT, XB);
         IRTemp ResultI32b = newTemp(Ity_I32);
         assign(ResultI32b, unop( Iop_ReinterpF32asI32,
                                 unop( Iop_TruncF64asF32,
                                       mkexpr( xB ) ) ) );
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          binop( Iop_32HLto64,
                                 mkexpr(ResultI32b ),
                                 mkexpr(ResultI32b ) ),
                          mkU64( 0ULL ) ) );
         break;
      case 0x090: // xscvdpuxws (VSX Scalar truncate Double-Precision to integer
                  //             and Convert to Unsigned Integer Word format with Saturate)
         DIP("xscvdpuxws v%u,v%u\n", XT, XB);
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          binop( Iop_32HLto64,
                                 mkU32( 0 ),
                                 binop( Iop_F64toI32U,
                                        mkU32( Irrm_ZERO ),
                                        mkexpr( xB ) ) ),
                          mkU64( 0ULL ) ) );
         break;
      case 0x292: // xscvspdp (VSX Scalar Convert Single-Precision to Double-Precision format, signaling)
         DIP("xscvspdp v%u,v%u\n", XT, XB);
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          unop( Iop_ReinterpF64asI64,
                                unop( Iop_F32toF64,
                                      unop( Iop_ReinterpI32asF32, mkexpr( xB ) ) ) ),
                          mkU64( 0ULL ) ) );
         break;
      case 0x296: // xscvspdpn (VSX Scalar Convert Single-Precision to Double-Precision format Non signaling)
         DIP("xscvspdpn v%u,v%u\n", XT, XB);
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          unop( Iop_ReinterpF64asI64,
                                unop( Iop_F32toF64,
                                      unop( Iop_ReinterpI32asF32, mkexpr( xB ) ) ) ),
                                      mkU64( 0ULL ) ) );
         break;
      case 0x312: // xvcvdpsp (VSX Vector round Double-Precision to single-precision
                  //           and Convert to Single-Precision format)
         DIP("xvcvdpsp v%u,v%u\n", XT, XB);

         /* Note, the 32-bit result is put into the upper and lower bits of the
            doubleword result.  */
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          binop( Iop_32HLto64,
                                 unop( Iop_ReinterpF32asI32,
                                       unop( Iop_TruncF64asF32,
                                             binop( Iop_RoundF64toF32,
                                                    get_IR_roundingmode(),
                                                    mkexpr( xB ) ) ) ),
                                 unop( Iop_ReinterpF32asI32,
                                       unop( Iop_TruncF64asF32,
                                             binop( Iop_RoundF64toF32,
                                                    get_IR_roundingmode(),
                                                    mkexpr( xB ) ) ) ) ),
                          binop( Iop_32HLto64,
                                 unop( Iop_ReinterpF32asI32,
                                       unop( Iop_TruncF64asF32,
                                             binop( Iop_RoundF64toF32,
                                                    get_IR_roundingmode(),
                                                    mkexpr( xB2 ) ) ) ),
                                 unop( Iop_ReinterpF32asI32,
                                       unop( Iop_TruncF64asF32,
                                             binop( Iop_RoundF64toF32,
                                                    get_IR_roundingmode(),
                                                    mkexpr( xB2 ) ) ) ) ) ) );
         break;
      case 0x390: // xvcvdpuxds (VSX Vector truncate Double-Precision to integer
                  //             and Convert to Unsigned Integer Doubleword format
                  //             with Saturate)
         DIP("xvcvdpuxds v%u,v%u\n", XT, XB);
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          binop( Iop_F64toI64U, mkU32( Irrm_ZERO ), mkexpr( xB ) ),
                          binop( Iop_F64toI64U, mkU32( Irrm_ZERO ), mkexpr( xB2 ) ) ) );
         break;
      case 0x392: // xvcvspdp (VSX Vector Convert Single-Precision to Double-Precision format)
         DIP("xvcvspdp v%u,v%u\n", XT, XB);
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          unop( Iop_ReinterpF64asI64,
                                unop( Iop_F32toF64,
                                      unop( Iop_ReinterpI32asF32,
                                            handle_SNaN_to_QNaN_32( mkexpr( b3 ) ) ) ) ),
                          unop( Iop_ReinterpF64asI64,
                                unop( Iop_F32toF64,
                                      unop( Iop_ReinterpI32asF32,
                                            handle_SNaN_to_QNaN_32( mkexpr( b1 ) ) ) ) ) ) );
         break;
      case 0x330: // xvcvspsxds (VSX Vector truncate Single-Precision to integer and
                  //           Convert to Signed Integer Doubleword format with Saturate)
         DIP("xvcvspsxds v%u,v%u\n", XT, XB);
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          binop( Iop_F64toI64S,
                                 mkU32( Irrm_ZERO ),
                                 unop( Iop_F32toF64,
                                       unop( Iop_ReinterpI32asF32, mkexpr( b3 ) ) ) ),
                          binop( Iop_F64toI64S,
                                 mkU32( Irrm_ZERO ),
                                 unop( Iop_F32toF64,
                                       unop( Iop_ReinterpI32asF32, mkexpr( b1 ) ) ) ) ) );
         break;
      case 0x310: // xvcvspuxds (VSX Vector truncate Single-Precision to integer and
                  //            Convert to Unsigned Integer Doubleword format with Saturate)
         DIP("xvcvspuxds v%u,v%u\n", XT, XB);
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          binop( Iop_F64toI64U,
                                 mkU32( Irrm_ZERO ),
                                 unop( Iop_F32toF64,
                                       unop( Iop_ReinterpI32asF32, mkexpr( b3 ) ) ) ),
                          binop( Iop_F64toI64U,
                                 mkU32( Irrm_ZERO ),
                                 unop( Iop_F32toF64,
                                       unop( Iop_ReinterpI32asF32, mkexpr( b1 ) ) ) ) ) );
         break;
      case 0x3B0: // xvcvdpsxds (VSX Vector truncate Double-Precision to integer and
                  //             Convert to Signed Integer Doubleword format with Saturate)
         DIP("xvcvdpsxds v%u,v%u\n", XT, XB);
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          binop( Iop_F64toI64S, mkU32( Irrm_ZERO ), mkexpr( xB ) ),
                          binop( Iop_F64toI64S, mkU32( Irrm_ZERO ), mkexpr( xB2 ) ) ) );
         break;
      case 0x3f0: // xvcvsxddp (VSX Vector Convert and round Signed Integer Doubleword
                  //            to Double-Precision format)
         DIP("xvcvsxddp v%u,v%u\n", XT, XB);
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          unop( Iop_ReinterpF64asI64,
                                binop( Iop_I64StoF64,
                                       get_IR_roundingmode(),
                                       mkexpr( xB ) ) ),
                          unop( Iop_ReinterpF64asI64,
                                binop( Iop_I64StoF64,
                                       get_IR_roundingmode(),
                                       mkexpr( xB2 ) ) ) ) );
         break;
      case 0x3d0: // xvcvuxddp (VSX Vector Convert and round Unsigned Integer Doubleword
                  //            to Double-Precision format)
         DIP("xvcvuxddp v%u,v%u\n", XT, XB);
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          unop( Iop_ReinterpF64asI64,
                                binop( Iop_I64UtoF64,
                                       get_IR_roundingmode(),
                                       mkexpr( xB ) ) ),
                          unop( Iop_ReinterpF64asI64,
                                binop( Iop_I64UtoF64,
                                       get_IR_roundingmode(),
                                       mkexpr( xB2 ) ) ) ) );

         break;
      case 0x370: // xvcvsxdsp (VSX Vector Convert and round Signed Integer Doubleword
                  //            to Single-Precision format)
      {
         IRTemp result32hi = newTemp(Ity_I32);
         IRTemp result32lo = newTemp(Ity_I32);

         DIP("xvcvsxdsp v%u,v%u\n",  XT, XB);
         assign( result32hi,
                 unop( Iop_ReinterpF32asI32,
                       unop( Iop_TruncF64asF32,
                             binop( Iop_RoundF64toF32,
                                    get_IR_roundingmode(),
                                    binop( Iop_I64StoF64,
                                           get_IR_roundingmode(),
                                           mkexpr( xB ) ) ) ) ) );
         assign( result32lo,
                 unop( Iop_ReinterpF32asI32,
                       unop( Iop_TruncF64asF32,
                             binop( Iop_RoundF64toF32,
                                    get_IR_roundingmode(),
                                    binop( Iop_I64StoF64,
                                           get_IR_roundingmode(),
                                           mkexpr( xB2 ) ) ) ) ) );

         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          binop( Iop_32HLto64,
                                 mkexpr( result32hi ),
                                 mkexpr( result32hi ) ),
                          binop( Iop_32HLto64,
                                 mkexpr( result32lo ),
                                 mkexpr( result32lo ) ) ) );
      }
      break;
      case 0x350: // xvcvuxdsp (VSX Vector Convert and round Unsigned Integer Doubleword
                  //            to Single-Precision format)
      {
         IRTemp result32hi = newTemp(Ity_I32);
         IRTemp result32lo = newTemp(Ity_I32);

         DIP("xvcvuxdsp v%u,v%u\n", XT, XB);
         assign( result32hi,
                 unop( Iop_ReinterpF32asI32,
                       unop( Iop_TruncF64asF32,
                             binop( Iop_RoundF64toF32,
                                    get_IR_roundingmode(),
                                    binop( Iop_I64UtoF64,
                                           get_IR_roundingmode(),
                                           mkexpr( xB ) ) ) ) ) );
         assign( result32lo,
                 unop( Iop_ReinterpF32asI32,
                       unop( Iop_TruncF64asF32,
                             binop( Iop_RoundF64toF32,
                                    get_IR_roundingmode(),
                                    binop( Iop_I64UtoF64,
                                           get_IR_roundingmode(),
                                           mkexpr( xB2 ) ) ) ) ) );
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          binop( Iop_32HLto64,
                                 mkexpr( result32hi ),
                                 mkexpr( result32hi ) ),
                          binop( Iop_32HLto64,
                                 mkexpr( result32lo ),
                                 mkexpr( result32lo ) ) ) );
      }
      break;

      case 0x1f0: // xvcvsxwdp (VSX Vector Convert Signed Integer Word to Double-Precision format)
         DIP("xvcvsxwdp v%u,v%u\n",  XT, XB);
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          unop( Iop_ReinterpF64asI64,
                                binop( Iop_I64StoF64, get_IR_roundingmode(),
                                       unop( Iop_32Sto64, mkexpr( b3 ) ) ) ),
                          unop( Iop_ReinterpF64asI64,
                                binop( Iop_I64StoF64, get_IR_roundingmode(),
                                       unop( Iop_32Sto64, mkexpr( b1 ) ) ) ) ) );
         break;
      case 0x1d0: // xvcvuxwdp (VSX Vector Convert Unsigned Integer Word to Double-Precision format)
         DIP("xvcvuxwdp v%u,v%u\n",  XT, XB);
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          unop( Iop_ReinterpF64asI64,
                                binop( Iop_I64UtoF64, get_IR_roundingmode(),
                                       unop( Iop_32Uto64, mkexpr( b3 ) ) ) ),
                          unop( Iop_ReinterpF64asI64,
                                binop( Iop_I64UtoF64, get_IR_roundingmode(),
                                       unop( Iop_32Uto64, mkexpr( b1 ) ) ) ) ) );
         break;
      case 0x170: // xvcvsxwsp (VSX Vector Convert Signed Integer Word to Single-Precision format)
         DIP("xvcvsxwsp v%u,v%u\n",  XT, XB);
         putVSReg( XT, unop( Iop_I32StoF32x4_DEP, getVSReg( XB ) ) );
         break;
      case 0x150: // xvcvuxwsp (VSX Vector Convert Unsigned Integer Word to Single-Precision format)
         DIP("xvcvuxwsp v%u,v%u\n",  XT, XB);
         putVSReg( XT, unop( Iop_I32UtoF32x4_DEP, getVSReg( XB ) ) );
         break;

      default:
         vex_printf( "dis_vx_conv(ppc)(opc2)\n" );
         return False;
   }
   return True;
}

/*
 * VSX vector Double Precision Floating Point Arithmetic Instructions
 */
static Bool
dis_vxv_dp_arith ( UInt prefix, UInt theInstr, UInt opc2 )
{
   /* XX3-Form */
   UChar opc1 = ifieldOPC( theInstr );
   UChar XT = ifieldRegXT( theInstr );
   UChar XA = ifieldRegXA( theInstr );
   UChar XB = ifieldRegXB( theInstr );
   IRExpr* rm = get_IR_roundingmode();
   IRTemp frA = newTemp(Ity_F64);
   IRTemp frB = newTemp(Ity_F64);
   IRTemp frA2 = newTemp(Ity_F64);
   IRTemp frB2 = newTemp(Ity_F64);

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   if (opc1 != 0x3C) {
      vex_printf( "dis_vxv_dp_arith(ppc)(instr)\n" );
      return False;
   }

   assign(frA,  unop(Iop_ReinterpI64asF64, unop(Iop_V128HIto64, getVSReg( XA ))));
   assign(frB,  unop(Iop_ReinterpI64asF64, unop(Iop_V128HIto64, getVSReg( XB ))));
   assign(frA2, unop(Iop_ReinterpI64asF64, unop(Iop_V128to64, getVSReg( XA ))));
   assign(frB2, unop(Iop_ReinterpI64asF64, unop(Iop_V128to64, getVSReg( XB ))));

   switch (opc2) {
      case 0x1E0: // xvdivdp (VSX Vector Divide Double-Precision)
      case 0x1C0: // xvmuldp (VSX Vector Multiply Double-Precision)
      case 0x180: // xvadddp (VSX Vector Add Double-Precision)
      case 0x1A0: // xvsubdp (VSX Vector Subtract Double-Precision)
      {
         IROp mOp;
         const HChar * oper_name;
         switch (opc2) {
            case 0x1E0:
               mOp = Iop_DivF64;
               oper_name = "div";
               break;
            case 0x1C0:
               mOp = Iop_MulF64;
               oper_name = "mul";
               break;
            case 0x180:
               mOp = Iop_AddF64;
               oper_name = "add";
               break;
            case 0x1A0:
               mOp = Iop_SubF64;
               oper_name = "sub";
               break;

            default:
               vpanic("The impossible happened: dis_vxv_dp_arith(ppc)");
         }
         IRTemp hiResult = newTemp(Ity_I64);
         IRTemp loResult = newTemp(Ity_I64);
         DIP("xv%sdp v%d,v%d,v%d\n", oper_name, XT, XA, XB);

         assign( hiResult,
                 unop( Iop_ReinterpF64asI64,
                       triop( mOp, rm, mkexpr( frA ), mkexpr( frB ) ) ) );
         assign( loResult,
                 unop( Iop_ReinterpF64asI64,
                       triop( mOp, rm, mkexpr( frA2 ), mkexpr( frB2 ) ) ) );
         putVSReg( XT,
                   binop( Iop_64HLtoV128, mkexpr( hiResult ), mkexpr( loResult ) ) );
         break;
      }
      case 0x196: // xvsqrtdp
      {
         IRTemp hiResult = newTemp(Ity_I64);
         IRTemp loResult = newTemp(Ity_I64);
         DIP("xvsqrtdp v%d,v%d\n", XT, XB);

         assign( hiResult,
                 unop( Iop_ReinterpF64asI64,
                       binop( Iop_SqrtF64, rm, mkexpr( frB ) ) ) );
         assign( loResult,
                 unop( Iop_ReinterpF64asI64,
                       binop( Iop_SqrtF64, rm, mkexpr( frB2 ) ) ) );
         putVSReg( XT,
                   binop( Iop_64HLtoV128, mkexpr( hiResult ), mkexpr( loResult ) ) );
         break;
      }
      case 0x184: case 0x1A4: // xvmaddadp, xvmaddmdp (VSX Vector Multiply-Add Double-Precision)
      case 0x1C4: case 0x1E4: // xvmsubadp, xvmsubmdp (VSX Vector Multiply-Subtract Double-Precision)
      case 0x384: case 0x3A4: // xvnmaddadp, xvnmaddmdp (VSX Vector Negate Multiply-Add Double-Precision)
      case 0x3C4: case 0x3E4: // xvnmsubadp, xvnmsubmdp (VSX Vector Negate Multiply-Subtract Double-Precision)
      {
         /* xvm{add|sub}mdp XT,XA,XB is element-wise equivalent to fm{add|sub} FRT,FRA,FRC,FRB with . . .
          *    XT == FRC
          *    XA == FRA
          *    XB == FRB
          *
          * and for xvm{add|sub}adp . . .
          *    XT == FRB
          *    XA == FRA
          *    XB == FRC
          */
         Bool negate;
         IROp mOp = Iop_INVALID;
         const HChar * oper_name = NULL;
         Bool mdp = False;

         switch (opc2) {
            case 0x184: case 0x1A4:
            case 0x384: case 0x3A4:
               mOp = Iop_MAddF64;
               oper_name = "add";
               mdp = (opc2 & 0x0FF) == 0x0A4;
               break;

            case 0x1C4: case 0x1E4:
            case 0x3C4: case 0x3E4:
               mOp = Iop_MSubF64;
               oper_name = "sub";
               mdp = (opc2 & 0x0FF) == 0x0E4;
               break;

            default:
               vpanic("The impossible happened: dis_vxv_sp_arith(ppc)");
         }

         switch (opc2) {
            case 0x384: case 0x3A4:
            case 0x3C4: case 0x3E4:
               negate = True;
               break;
            default:
               negate = False;
         }
         IRTemp hiResult = newTemp(Ity_I64);
         IRTemp loResult = newTemp(Ity_I64);
         IRTemp frT = newTemp(Ity_F64);
         IRTemp frT2 = newTemp(Ity_F64);
         DIP("xv%sm%s%s v%d,v%d,v%d\n", negate ? "n" : "", oper_name, mdp ? "mdp" : "adp",
             XT, XA, XB);
         assign(frT,  unop(Iop_ReinterpI64asF64, unop(Iop_V128HIto64, getVSReg( XT ) ) ) );
         assign(frT2, unop(Iop_ReinterpI64asF64, unop(Iop_V128to64, getVSReg( XT ) ) ) );

         assign( hiResult,
                 unop( Iop_ReinterpF64asI64,
                       qop( mOp,
                            rm,
                            mkexpr( frA ),
                            mkexpr( mdp ? frT : frB ),
                            mkexpr( mdp ? frB : frT ) ) ) );
         assign( loResult,
                 unop( Iop_ReinterpF64asI64,
                       qop( mOp,
                            rm,
                            mkexpr( frA2 ),
                            mkexpr( mdp ? frT2 : frB2 ),
                            mkexpr( mdp ? frB2 : frT2 ) ) ) );
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          mkexpr( negate ? getNegatedResult( hiResult )
                                         : hiResult ),
                          mkexpr( negate ? getNegatedResult( loResult )
                                         : loResult ) ) );
         break;
      }
      case 0x1D4: // xvtsqrtdp (VSX Vector Test for software Square Root Double-Precision)
      {
         IRTemp frBHi_I64 = newTemp(Ity_I64);
         IRTemp frBLo_I64 = newTemp(Ity_I64);
         IRTemp flagsHi = newTemp(Ity_I32);
         IRTemp flagsLo = newTemp(Ity_I32);
         UChar crfD     = toUChar( IFIELD( theInstr, 23, 3 ) );
         IRTemp  fe_flagHi, fg_flagHi, fe_flagLo, fg_flagLo;
         fe_flagHi = fg_flagHi = fe_flagLo = fg_flagLo = IRTemp_INVALID;

         DIP("xvtsqrtdp cr%d,v%d\n", crfD, XB);
         assign( frBHi_I64, unop(Iop_V128HIto64, getVSReg( XB )) );
         assign( frBLo_I64, unop(Iop_V128to64, getVSReg( XB )) );
         do_fp_tsqrt(frBHi_I64, False /*not single precision*/, &fe_flagHi, &fg_flagHi);
         do_fp_tsqrt(frBLo_I64, False /*not single precision*/, &fe_flagLo, &fg_flagLo);
         /* The CR field consists of fl_flag || fg_flag || fe_flag || 0b0
          * where fl_flag == 1 on ppc64.
          */
         assign( flagsHi,
                 binop( Iop_Or32,
                        binop( Iop_Or32, mkU32( 8 ), // fl_flag
                               binop( Iop_Shl32, mkexpr(fg_flagHi), mkU8( 2 ) ) ),
                        binop( Iop_Shl32, mkexpr(fe_flagHi), mkU8( 1 ) ) ) );
         assign( flagsLo,
                 binop( Iop_Or32,
                        binop( Iop_Or32, mkU32( 8 ), // fl_flag
                               binop( Iop_Shl32, mkexpr(fg_flagLo), mkU8( 2 ) ) ),
                        binop( Iop_Shl32, mkexpr(fe_flagLo), mkU8( 1 ) ) ) );
         putGST_field( PPC_GST_CR,
                       binop( Iop_Or32, mkexpr( flagsHi ), mkexpr( flagsLo ) ),
                       crfD );
         break;
      }
      case 0x1F4: // xvtdivdp (VSX Vector Test for software Divide Double-Precision)
      {
         IRTemp frBHi_I64 = newTemp(Ity_I64);
         IRTemp frBLo_I64 = newTemp(Ity_I64);
         IRTemp frAHi_I64 = newTemp(Ity_I64);
         IRTemp frALo_I64 = newTemp(Ity_I64);
         IRTemp flagsHi = newTemp(Ity_I32);
         IRTemp flagsLo = newTemp(Ity_I32);
         UChar crfD     = toUChar( IFIELD( theInstr, 23, 3 ) );
         IRTemp  fe_flagHi, fg_flagHi, fe_flagLo, fg_flagLo;
         fe_flagHi = fg_flagHi = fe_flagLo = fg_flagLo = IRTemp_INVALID;

         DIP("xvtdivdp cr%d,v%d,v%d\n", crfD, XA, XB);
         assign( frAHi_I64, unop(Iop_V128HIto64, getVSReg( XA )) );
         assign( frALo_I64, unop(Iop_V128to64, getVSReg( XA )) );
         assign( frBHi_I64, unop(Iop_V128HIto64, getVSReg( XB )) );
         assign( frBLo_I64, unop(Iop_V128to64, getVSReg( XB )) );

         _do_fp_tdiv(frAHi_I64, frBHi_I64, False/*dp*/, &fe_flagHi, &fg_flagHi);
         _do_fp_tdiv(frALo_I64, frBLo_I64, False/*dp*/, &fe_flagLo, &fg_flagLo);
         /* The CR field consists of fl_flag || fg_flag || fe_flag || 0b0
          * where fl_flag == 1 on ppc64.
          */
         assign( flagsHi,
                 binop( Iop_Or32,
                        binop( Iop_Or32, mkU32( 8 ), // fl_flag
                               binop( Iop_Shl32, mkexpr(fg_flagHi), mkU8( 2 ) ) ),
                        binop( Iop_Shl32, mkexpr(fe_flagHi), mkU8( 1 ) ) ) );
         assign( flagsLo,
                 binop( Iop_Or32,
                        binop( Iop_Or32, mkU32( 8 ), // fl_flag
                               binop( Iop_Shl32, mkexpr(fg_flagLo), mkU8( 2 ) ) ),
                        binop( Iop_Shl32, mkexpr(fe_flagLo), mkU8( 1 ) ) ) );
         putGST_field( PPC_GST_CR,
                       binop( Iop_Or32, mkexpr( flagsHi ), mkexpr( flagsLo ) ),
                       crfD );
         break;
      }

      default:
         vex_printf( "dis_vxv_dp_arith(ppc)(opc2)\n" );
         return False;
   }
   return True;
}

/*
 * VSX vector Single Precision Floating Point Arithmetic Instructions
 */
static Bool
dis_vxv_sp_arith ( UInt prefix, UInt theInstr, UInt opc2 )
{
   /* XX3-Form */
   UChar opc1 = ifieldOPC( theInstr );
   UChar XT = ifieldRegXT( theInstr );
   UChar XA = ifieldRegXA( theInstr );
   UChar XB = ifieldRegXB( theInstr );
   IRExpr* rm = get_IR_roundingmode();
   IRTemp a3, a2, a1, a0;
   IRTemp b3, b2, b1, b0;
   IRTemp res0 = newTemp(Ity_I32);
   IRTemp res1 = newTemp(Ity_I32);
   IRTemp res2 = newTemp(Ity_I32);
   IRTemp res3 = newTemp(Ity_I32);

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   a3 = a2 = a1 = a0 = IRTemp_INVALID;
   b3 = b2 = b1 = b0 = IRTemp_INVALID;

   if (opc1 != 0x3C) {
      vex_printf( "dis_vxv_sp_arith(ppc)(instr)\n" );
      return False;
   }

   switch (opc2) {
      case 0x100: // xvaddsp (VSX Vector Add Single-Precision)
         DIP("xvaddsp v%d,v%d,v%d\n", XT, XA, XB);
         // WARNING: BOGUS! The backend ignores rm on Iop_Add32Fx4
         putVSReg( XT, triop(Iop_Add32Fx4, rm,
                             getVSReg( XA ), getVSReg( XB )) );
         break;

      case 0x140: // xvmulsp (VSX Vector Multiply Single-Precision)
         DIP("xvmulsp v%d,v%d,v%d\n", XT, XA, XB);
         // WARNING: BOGUS! The backend ignores rm on Iop_Mul32Fx4
         putVSReg( XT, triop(Iop_Mul32Fx4, rm,
                             getVSReg( XA ), getVSReg( XB )) );
         break;

      case 0x120: // xvsubsp (VSX Vector Subtract Single-Precision)
         DIP("xvsubsp v%d,v%d,v%d\n", XT, XA, XB);
         // WARNING: BOGUS! The backend ignores rm on Iop_Sub32Fx4
         putVSReg( XT, triop(Iop_Sub32Fx4, rm,
                             getVSReg( XA ), getVSReg( XB )) );
         break;

      case 0x160: // xvdivsp (VSX Vector Divide Single-Precision)
      {
         /* Iop_Div32Fx4 is not implemented for ppc64 (in host_ppc_{isel|defs}.c.
          * So there are two choices:
          *   1. Implement the xvdivsp with a native insn; or
          *   2. Extract the 4 single precision floats from each vector
          *      register inputs and perform fdivs on each pair
          * I will do the latter, due to the general philosophy of
          * reusing existing implementations when practical.
          */
         DIP("xvdivsp v%d,v%d,v%d\n", XT, XA, XB);
         breakV128to4xF64( getVSReg( XA ), &a3, &a2, &a1, &a0 );
         breakV128to4xF64( getVSReg( XB ), &b3, &b2, &b1, &b0 );

         assign( res0,
              unop( Iop_ReinterpF32asI32,
                    unop( Iop_TruncF64asF32,
                          triop( Iop_DivF64r32, rm, mkexpr( a0 ), mkexpr( b0 ) ) ) ) );
         assign( res1,
                 unop( Iop_ReinterpF32asI32,
                       unop( Iop_TruncF64asF32,
                             triop( Iop_DivF64r32, rm, mkexpr( a1 ), mkexpr( b1 ) ) ) ) );
         assign( res2,
                 unop( Iop_ReinterpF32asI32,
                       unop( Iop_TruncF64asF32,
                             triop( Iop_DivF64r32, rm, mkexpr( a2 ), mkexpr( b2 ) ) ) ) );
         assign( res3,
                 unop( Iop_ReinterpF32asI32,
                       unop( Iop_TruncF64asF32,
                             triop( Iop_DivF64r32, rm, mkexpr( a3 ), mkexpr( b3 ) ) ) ) );

         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          binop( Iop_32HLto64, mkexpr( res3 ), mkexpr( res2 ) ),
                          binop( Iop_32HLto64, mkexpr( res1 ), mkexpr( res0 ) ) ) );
         break;
      }
      case 0x116: // xvsqrtsp (VSX Vector Square Root Single-Precision)
      {
         DIP("xvsqrtsp v%d,v%d\n", XT, XB);
         breakV128to4xF64( getVSReg( XB ), &b3, &b2, &b1, &b0 );
         /* Note: The native xvsqrtsp insruction does not always give the same precision
          * as what we get with Iop_SqrtF64.  But it doesn't seem worthwhile to implement
          * an Iop_SqrtF32 that would give us a lower precision result, albeit more true
          * to the actual instruction.
          */

         assign( res0,
                 unop( Iop_ReinterpF32asI32,
                       unop( Iop_TruncF64asF32,
                             binop(Iop_SqrtF64, rm, mkexpr( b0 ) ) ) ) );
         assign( res1,
                 unop( Iop_ReinterpF32asI32,
                       unop( Iop_TruncF64asF32,
                             binop(Iop_SqrtF64, rm, mkexpr( b1 ) ) ) ) );
         assign( res2,
                 unop( Iop_ReinterpF32asI32,
                       unop( Iop_TruncF64asF32,
                             binop(Iop_SqrtF64, rm, mkexpr( b2) ) ) ) );
         assign( res3,
                 unop( Iop_ReinterpF32asI32,
                       unop( Iop_TruncF64asF32,
                             binop(Iop_SqrtF64, rm, mkexpr( b3 ) ) ) ) );

         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          binop( Iop_32HLto64, mkexpr( res3 ), mkexpr( res2 ) ),
                          binop( Iop_32HLto64, mkexpr( res1 ), mkexpr( res0 ) ) ) );
         break;
      }

      case 0x104: case 0x124: // xvmaddasp, xvmaddmsp (VSX Vector Multiply-Add Single-Precision)
      case 0x144: case 0x164: // xvmsubasp, xvmsubmsp (VSX Vector Multiply-Subtract Single-Precision)
      case 0x304: case 0x324: // xvnmaddasp, xvnmaddmsp (VSX Vector Negate Multiply-Add Single-Precision)
      case 0x344: case 0x364: // xvnmsubasp, xvnmsubmsp (VSX Vector Negate Multiply-Subtract Single-Precision)
      {
         IRTemp t3, t2, t1, t0;
         Bool msp = False;
         Bool negate;
         const HChar * oper_name = NULL;
         IROp mOp = Iop_INVALID;
         switch (opc2) {
            case 0x104: case 0x124:
            case 0x304: case 0x324:
               msp = (opc2 & 0x0FF) == 0x024;
               mOp = Iop_MAddF64r32;
               oper_name = "madd";
               break;

            case 0x144: case 0x164:
            case 0x344: case 0x364:
               msp = (opc2 & 0x0FF) == 0x064;
               mOp = Iop_MSubF64r32;
               oper_name = "sub";
               break;

            default:
               vpanic("The impossible happened: dis_vxv_sp_arith(ppc)");
         }

         switch (opc2) {
            case 0x304: case 0x324:
            case 0x344: case 0x364:
               negate = True;
               break;

            default:
               negate = False;
         }

         DIP("xv%sm%s%s v%d,v%d,v%d\n", negate ? "n" : "", oper_name,
             msp ? "msp" : "asp", XT, XA, XB);

         t3 = t2 = t1 = t0 = IRTemp_INVALID;
         breakV128to4xF64( getVSReg( XA ), &a3, &a2, &a1, &a0 );
         breakV128to4xF64( getVSReg( XB ), &b3, &b2, &b1, &b0 );
         breakV128to4xF64( getVSReg( XT ), &t3, &t2, &t1, &t0 );

         assign( res0,
                 unop( Iop_ReinterpF32asI32,
                       unop( Iop_TruncF64asF32,
                             qop( mOp,
                                  rm,
                                  mkexpr( a0 ),
                                  mkexpr( msp ? t0 : b0 ),
                                  mkexpr( msp ? b0 : t0 ) ) ) ) );
         assign( res1,
                 unop( Iop_ReinterpF32asI32,
                       unop( Iop_TruncF64asF32,
                             qop( mOp,
                                  rm,
                                  mkexpr( a1 ),
                                  mkexpr( msp ? t1 : b1 ),
                                  mkexpr( msp ? b1 : t1 ) ) ) ) );
         assign( res2,
                 unop( Iop_ReinterpF32asI32,
                       unop( Iop_TruncF64asF32,
                             qop( mOp,
                                  rm,
                                  mkexpr( a2 ),
                                  mkexpr( msp ? t2 : b2 ),
                                  mkexpr( msp ? b2 : t2 ) ) ) ) );
         assign( res3,
                 unop( Iop_ReinterpF32asI32,
                       unop( Iop_TruncF64asF32,
                             qop( mOp,
                                  rm,
                                  mkexpr( a3 ),
                                  mkexpr( msp ? t3 : b3 ),
                                  mkexpr( msp ? b3 : t3 ) ) ) ) );

         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          binop( Iop_32HLto64, mkexpr( negate ? getNegatedResult_32( res3 ) : res3 ),
                                 mkexpr( negate ? getNegatedResult_32( res2 ) : res2 ) ),
                          binop( Iop_32HLto64, mkexpr( negate ? getNegatedResult_32( res1 ) : res1 ),
                                 mkexpr( negate ? getNegatedResult_32( res0 ) : res0 ) ) ) );

         break;
      }
      case 0x154: // xvtsqrtsp (VSX Vector Test for software Square Root Single-Precision)
      {
         IRTemp flags0 = newTemp(Ity_I32);
         IRTemp flags1 = newTemp(Ity_I32);
         IRTemp flags2 = newTemp(Ity_I32);
         IRTemp flags3 = newTemp(Ity_I32);
         UChar crfD     = toUChar( IFIELD( theInstr, 23, 3 ) );
         IRTemp  fe_flag0, fg_flag0, fe_flag1, fg_flag1;
         IRTemp  fe_flag2, fg_flag2, fe_flag3, fg_flag3;
         fe_flag0 = fg_flag0 = fe_flag1 = fg_flag1 = IRTemp_INVALID;
         fe_flag2 = fg_flag2 = fe_flag3 = fg_flag3 = IRTemp_INVALID;
         DIP("xvtsqrtsp cr%d,v%d\n", crfD, XB);

         breakV128to4x32( getVSReg( XB ), &b3, &b2, &b1, &b0 );
         do_fp_tsqrt(b0, True /* single precision*/, &fe_flag0, &fg_flag0);
         do_fp_tsqrt(b1, True /* single precision*/, &fe_flag1, &fg_flag1);
         do_fp_tsqrt(b2, True /* single precision*/, &fe_flag2, &fg_flag2);
         do_fp_tsqrt(b3, True /* single precision*/, &fe_flag3, &fg_flag3);

         /* The CR field consists of fl_flag || fg_flag || fe_flag || 0b0
          * where fl_flag == 1 on ppc64.
          */
         assign( flags0,
                 binop( Iop_Or32,
                        binop( Iop_Or32, mkU32( 8 ), // fl_flag
                               binop( Iop_Shl32, mkexpr(fg_flag0), mkU8( 2 ) ) ),
                        binop( Iop_Shl32, mkexpr(fe_flag0), mkU8( 1 ) ) ) );
         assign( flags1,
                 binop( Iop_Or32,
                        binop( Iop_Or32, mkU32( 8 ), // fl_flag
                               binop( Iop_Shl32, mkexpr(fg_flag1), mkU8( 2 ) ) ),
                        binop( Iop_Shl32, mkexpr(fe_flag1), mkU8( 1 ) ) ) );
         assign( flags2,
                 binop( Iop_Or32,
                        binop( Iop_Or32, mkU32( 8 ), // fl_flag
                               binop( Iop_Shl32, mkexpr(fg_flag2), mkU8( 2 ) ) ),
                        binop( Iop_Shl32, mkexpr(fe_flag2), mkU8( 1 ) ) ) );
         assign( flags3,
                 binop( Iop_Or32,
                        binop( Iop_Or32, mkU32( 8 ), // fl_flag
                               binop( Iop_Shl32, mkexpr(fg_flag3), mkU8( 2 ) ) ),
                        binop( Iop_Shl32, mkexpr(fe_flag3), mkU8( 1 ) ) ) );
         putGST_field( PPC_GST_CR,
                       binop( Iop_Or32,
                              mkexpr( flags0 ),
                              binop( Iop_Or32,
                                     mkexpr( flags1 ),
                                     binop( Iop_Or32,
                                            mkexpr( flags2 ),
                                            mkexpr( flags3 ) ) ) ),
                       crfD );
         break;
      }

   case 0x174: // xvtdivsp (VSX Vector Test for software Divide Single-Precision)
      {
         IRTemp flags0 = newTemp(Ity_I32);
         IRTemp flags1 = newTemp(Ity_I32);
         IRTemp flags2 = newTemp(Ity_I32);
         IRTemp flags3 = newTemp(Ity_I32);
         UChar crfD     = toUChar( IFIELD( theInstr, 23, 3 ) );
         IRTemp  fe_flag0, fg_flag0, fe_flag1, fg_flag1;
         IRTemp  fe_flag2, fg_flag2, fe_flag3, fg_flag3;
         fe_flag0 = fg_flag0 = fe_flag1 = fg_flag1 = IRTemp_INVALID;
         fe_flag2 = fg_flag2 = fe_flag3 = fg_flag3 = IRTemp_INVALID;
         DIP("xvtdivsp cr%d,v%d,v%d\n", crfD, XA, XB);

         breakV128to4x32( getVSReg( XA ), &a3, &a2, &a1, &a0 );
         breakV128to4x32( getVSReg( XB ), &b3, &b2, &b1, &b0 );
         _do_fp_tdiv(a0, b0, True /* single precision*/, &fe_flag0, &fg_flag0);
         _do_fp_tdiv(a1, b1, True /* single precision*/, &fe_flag1, &fg_flag1);
         _do_fp_tdiv(a2, b2, True /* single precision*/, &fe_flag2, &fg_flag2);
         _do_fp_tdiv(a3, b3, True /* single precision*/, &fe_flag3, &fg_flag3);

         /* The CR field consists of fl_flag || fg_flag || fe_flag || 0b0
          * where fl_flag == 1 on ppc64.
          */
         assign( flags0,
                 binop( Iop_Or32,
                        binop( Iop_Or32, mkU32( 8 ), // fl_flag
                               binop( Iop_Shl32, mkexpr(fg_flag0), mkU8( 2 ) ) ),
                        binop( Iop_Shl32, mkexpr(fe_flag0), mkU8( 1 ) ) ) );
         assign( flags1,
                 binop( Iop_Or32,
                        binop( Iop_Or32, mkU32( 8 ), // fl_flag
                               binop( Iop_Shl32, mkexpr(fg_flag1), mkU8( 2 ) ) ),
                        binop( Iop_Shl32, mkexpr(fe_flag1), mkU8( 1 ) ) ) );
         assign( flags2,
                 binop( Iop_Or32,
                        binop( Iop_Or32, mkU32( 8 ), // fl_flag
                               binop( Iop_Shl32, mkexpr(fg_flag2), mkU8( 2 ) ) ),
                        binop( Iop_Shl32, mkexpr(fe_flag2), mkU8( 1 ) ) ) );
         assign( flags3,
                 binop( Iop_Or32,
                        binop( Iop_Or32, mkU32( 8 ), // fl_flag
                               binop( Iop_Shl32, mkexpr(fg_flag3), mkU8( 2 ) ) ),
                        binop( Iop_Shl32, mkexpr(fe_flag3), mkU8( 1 ) ) ) );
         putGST_field( PPC_GST_CR,
                       binop( Iop_Or32,
                              mkexpr( flags0 ),
                              binop( Iop_Or32,
                                     mkexpr( flags1 ),
                                     binop( Iop_Or32,
                                            mkexpr( flags2 ),
                                            mkexpr( flags3 ) ) ) ),
                       crfD );

         break;
      }

      default:
         vex_printf( "dis_vxv_sp_arith(ppc)(opc2)\n" );
         return False;
   }
   return True;
}

/*
 * Vector Population Count/bit matrix transpose
 */
static Bool
dis_av_count_bitTranspose ( UInt prefix, UInt theInstr, UInt opc2 )
{
   UChar vRB_addr = ifieldRegB(theInstr);
   UChar vRT_addr = ifieldRegDS(theInstr);
   UChar opc1 = ifieldOPC( theInstr );
   IRTemp vB = newTemp(Ity_V128);

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   assign( vB, getVReg(vRB_addr));

   if (opc1 != 0x4) {
      vex_printf( "dis_av_count_bitTranspose(ppc)(instr)\n" );
      return False;
   }

   switch (opc2) {
      case 0x702:    // vclzb
         DIP("vclzb v%d,v%d\n", vRT_addr, vRB_addr);
         putVReg( vRT_addr, unop(Iop_Clz8x16, mkexpr( vB ) ) );
         break;

      case 0x742:    // vclzh
         DIP("vclzh v%d,v%d\n", vRT_addr, vRB_addr);
         putVReg( vRT_addr, unop(Iop_Clz16x8, mkexpr( vB ) ) );
         break;

      case 0x782:    // vclzw
         DIP("vclzw v%d,v%d\n", vRT_addr, vRB_addr);
         putVReg( vRT_addr, unop(Iop_Clz32x4, mkexpr( vB ) ) );
         break;

      case 0x7C2:    // vclzd
         DIP("vclzd v%d,v%d\n", vRT_addr, vRB_addr);
         putVReg( vRT_addr, unop(Iop_Clz64x2, mkexpr( vB ) ) );
         break;

      case 0x703:    // vpopcntb
      {
         /* Break vector into 32-bit words and do the population count
          * on byte in the words
          */
         IRType ty = Ity_I32;
         IRTemp bits0_31, bits32_63, bits64_95, bits96_127;
         bits0_31 = bits32_63 = bits64_95 = bits96_127 = IRTemp_INVALID;
         IRTemp cnt_bits0_31, cnt_bits32_63, cnt_bits64_95, cnt_bits96_127;
         cnt_bits0_31 = cnt_bits32_63 = cnt_bits64_95 = cnt_bits96_127 = IRTemp_INVALID;

         DIP("vpopcntb v%d,v%d\n", vRT_addr, vRB_addr);
         breakV128to4x32(mkexpr( vB), &bits96_127, &bits64_95, &bits32_63, &bits0_31 );
         cnt_bits0_31   = gen_POPCOUNT(ty, bits0_31,   BYTE);
         cnt_bits32_63  = gen_POPCOUNT(ty, bits32_63,  BYTE);
         cnt_bits64_95  = gen_POPCOUNT(ty, bits64_95,  BYTE);
         cnt_bits96_127 = gen_POPCOUNT(ty, bits96_127, BYTE);

         putVReg( vRT_addr, mkV128from32(cnt_bits96_127, cnt_bits64_95,
                                         cnt_bits32_63, cnt_bits0_31) );
         break;
      }

      case 0x743:    // vpopcnth
      {
         /* Break vector into 32-bit words and do the population count
          * for each half word
          */
         IRType ty = Ity_I32;
         IRTemp bits0_31, bits32_63, bits64_95, bits96_127;
         bits0_31 = bits32_63 = bits64_95 = bits96_127 = IRTemp_INVALID;
         IRTemp cnt_bits0_31, cnt_bits32_63, cnt_bits64_95, cnt_bits96_127;
         cnt_bits0_31 = cnt_bits32_63 = cnt_bits64_95 = cnt_bits96_127 = IRTemp_INVALID;

         DIP("vpopcnth v%d,v%d\n", vRT_addr, vRB_addr);
         breakV128to4x32(mkexpr( vB), &bits96_127, &bits64_95, &bits32_63, &bits0_31 );

         cnt_bits0_31   = gen_POPCOUNT(ty, bits0_31,   HWORD);
         cnt_bits32_63  = gen_POPCOUNT(ty, bits32_63,  HWORD);
         cnt_bits64_95  = gen_POPCOUNT(ty, bits64_95,  HWORD);
         cnt_bits96_127 = gen_POPCOUNT(ty, bits96_127, HWORD);

         putVReg( vRT_addr, mkV128from32(cnt_bits96_127, cnt_bits64_95,
                                         cnt_bits32_63, cnt_bits0_31) );
         break;
      }

      case 0x783:    // vpopcntw
      {
         /* Break vector into 32-bit words and do the population count
          * on each word.
          */
         IRType ty = Ity_I32;
         IRTemp bits0_31, bits32_63, bits64_95, bits96_127;
         bits0_31 = bits32_63 = bits64_95 = bits96_127 = IRTemp_INVALID;
         IRTemp cnt_bits0_31, cnt_bits32_63, cnt_bits64_95, cnt_bits96_127;
         cnt_bits0_31 = cnt_bits32_63 = cnt_bits64_95 = cnt_bits96_127 = IRTemp_INVALID;

         DIP("vpopcntw v%d,v%d\n", vRT_addr, vRB_addr);
         breakV128to4x32(mkexpr( vB), &bits96_127, &bits64_95, &bits32_63, &bits0_31 );

         cnt_bits0_31   = gen_POPCOUNT(ty, bits0_31,   WORD);
         cnt_bits32_63  = gen_POPCOUNT(ty, bits32_63,  WORD);
         cnt_bits64_95  = gen_POPCOUNT(ty, bits64_95,  WORD);
         cnt_bits96_127 = gen_POPCOUNT(ty, bits96_127, WORD);

         putVReg( vRT_addr, mkV128from32(cnt_bits96_127, cnt_bits64_95,
                                         cnt_bits32_63, cnt_bits0_31) );
         break;
      }

      case 0x7C3:    // vpopcntd
      {
         if (mode64) {
            /* Break vector into 64-bit double words and do the population
               count on each double word.
             */
            IRType ty = Ity_I64;
            IRTemp bits0_63   = newTemp(Ity_I64);
            IRTemp bits64_127 = newTemp(Ity_I64);
            IRTemp cnt_bits0_63   = newTemp(Ity_I64);
            IRTemp cnt_bits64_127 = newTemp(Ity_I64);

            DIP("vpopcntd v%d,v%d\n", vRT_addr, vRB_addr);

            assign(bits0_63,   unop( Iop_V128to64,   mkexpr( vB ) ) );
            assign(bits64_127, unop( Iop_V128HIto64, mkexpr( vB ) ) );
            cnt_bits0_63   = gen_POPCOUNT(ty, bits0_63,   DWORD);
            cnt_bits64_127 = gen_POPCOUNT(ty, bits64_127, DWORD);

            putVReg( vRT_addr, binop( Iop_64HLtoV128,
                                      mkexpr( cnt_bits64_127 ),
                                      mkexpr( cnt_bits0_63 ) ) );
         } else {
            /* Break vector into 32-bit words and do the population count
               on each 32-bit word.
             */
            IRTemp bits0_31, bits32_63, bits64_95, bits96_127;
            bits0_31 = bits32_63 = bits64_95 = bits96_127 = IRTemp_INVALID;
            IRTemp cnt_bits0_63    = newTemp(Ity_I64);
            IRTemp cnt_bits64_127  = newTemp(Ity_I64);

            DIP("vpopcntd v%d,v%d\n", vRT_addr, vRB_addr);
            breakV128to4x32(mkexpr( vB), &bits96_127, &bits64_95,
                                         &bits32_63, &bits0_31 );

            cnt_bits0_63   = gen_vpopcntd_mode32(bits0_31, bits32_63);
            cnt_bits64_127 = gen_vpopcntd_mode32(bits64_95, bits96_127);

            putVReg( vRT_addr, binop( Iop_64HLtoV128,
                                      mkexpr( cnt_bits64_127 ),
                                      mkexpr( cnt_bits0_63 ) ) );
         }
         break;
      }

      case 0x50C:  // vgbbd Vector Gather Bits by Bytes by Doubleword
         DIP("vgbbd v%d,v%d\n", vRT_addr, vRB_addr);
         putVReg( vRT_addr, unop( Iop_PwBitMtxXpose64x2, mkexpr( vB ) ) );
         break;

      case 0x5CC:  // vbpermd Vector Bit Permute Doubleword
      {
         UChar vRA_addr = ifieldRegA( theInstr );
         IRTemp vA = newTemp( Ity_V128 );
         UInt j;
         IRTemp index_dword_hi[8]; // index in double word
         IRTemp index_dword_lo[8];
         IRTemp index_dword_hi_valid[8];
         IRTemp index_dword_lo_valid[8];
         IRTemp pb_dword_hi[8];  // permute bit
         IRTemp pb_dword_lo[8];
         IRTemp tmp_hi[9];
         IRTemp tmp_lo[9];

         DIP("vbpermd v%d,v%d,v%d\n", vRT_addr, vRA_addr, vRB_addr);

         tmp_hi[0] = newTemp( Ity_I64 );
         tmp_lo[0] = newTemp( Ity_I64 );

         assign( vA, getVReg(vRA_addr) );
         assign( tmp_hi[0], mkU64( 0 ) );
         assign( tmp_lo[0], mkU64( 0 ) );

         for (j=0; j<8; j++) {
            index_dword_hi[j] = newTemp( Ity_I64 );
            index_dword_lo[j] = newTemp( Ity_I64 );
            index_dword_hi_valid[j] = newTemp( Ity_I64 );
            index_dword_lo_valid[j] = newTemp( Ity_I64 );
            pb_dword_hi[j] = newTemp( Ity_I64 );
            pb_dword_lo[j] = newTemp( Ity_I64 );
            tmp_hi[j+1] = newTemp( Ity_I64 );
            tmp_lo[j+1] = newTemp( Ity_I64 );

            assign( index_dword_hi[j],
                    binop( Iop_And64,
                           binop( Iop_Shr64,
                                  unop( Iop_V128HIto64,
                                        mkexpr( vB ) ),
                                  mkU8( ( 7 - j ) * 8 ) ),
                           mkU64( 0xFF ) ) );

            assign( index_dword_lo[j],
                    binop( Iop_And64,
                           binop( Iop_Shr64,
                                  unop( Iop_V128to64,
                                        mkexpr( vB ) ),
                                  mkU8( ( 7 - j ) * 8 ) ),
                           mkU64( 0xFF ) ) );

            assign( index_dword_hi_valid[j],
                    unop( Iop_1Sto64,
                          binop( Iop_CmpLT64U,
                                 mkexpr( index_dword_hi[j] ),
                                 mkU64( 64 ) ) ) );

            assign( index_dword_lo_valid[j],
                    unop( Iop_1Sto64,
                          binop( Iop_CmpLT64U,
                                 mkexpr( index_dword_lo[j] ),
                                 mkU64( 64 ) ) ) );
            assign( pb_dword_hi[j],
                    binop( Iop_And64,
                           binop( Iop_Shr64,
                                  unop( Iop_V128HIto64,
                                        mkexpr( vA ) ),
                                  unop( Iop_64to8,
                                        binop( Iop_Sub64,
                                               mkU64( 63 ),
                                               mkexpr( index_dword_hi[j] )
                                               ) ) ),
                           mkU64( 0x1 ) ) );

            assign( pb_dword_lo[j],
                    binop( Iop_And64,
                           binop( Iop_Shr64,
                                  unop( Iop_V128to64,
                                        mkexpr( vA ) ),
                                  unop( Iop_64to8,
                                        binop( Iop_Sub64,
                                               mkU64( 63 ),
                                               mkexpr( index_dword_lo[j] )
                                               ) ) ),
                           mkU64( 0x1 ) ) );

            assign( tmp_hi[j+1],
                    binop( Iop_Or64,
                           binop( Iop_And64,
                                  mkexpr( index_dword_hi_valid[j] ),
                                  binop( Iop_Shl64,
                                         mkexpr( pb_dword_hi[j] ),
                                         mkU8( 7 - j ) ) ),
                           mkexpr( tmp_hi[j] ) ) );

            assign( tmp_lo[j+1],
                    binop( Iop_Or64,
                           binop( Iop_And64,
                                  mkexpr( index_dword_lo_valid[j] ),
                                  binop( Iop_Shl64,
                                         mkexpr( pb_dword_lo[j] ),
                                         mkU8( 7 - j ) ) ),
                           mkexpr( tmp_lo[j] ) ) );
         }

         putVReg( vRT_addr,
                  binop( Iop_64HLtoV128,
                         mkexpr( tmp_hi[8] ),
                         mkexpr( tmp_lo[8] ) ) );
      }
      break;

      default:
         vex_printf("dis_av_count_bitTranspose(ppc)(opc2)\n");
         return False;
      break;
   }
   return True;
}

/*
 * Scalar / Vector Population Count/bit matrix transpose
 */
static Bool dis_logical_mask_bits ( UInt prefix, UInt theInstr,
                                    const VexAbiInfo* vbi )
{
   UChar opc1    = ifieldOPC(theInstr);
   UInt  opc2    = ifieldOPClo10(theInstr);
   UChar rS_addr = ifieldRegDS(theInstr);
   UChar rA_addr = ifieldRegA(theInstr);
   UChar rB_addr = ifieldRegB(theInstr);

   IRTemp rS     = newTemp( Ity_I64 );
   IRTemp rA     = newTemp( Ity_I64 );
   IRTemp rB     = newTemp( Ity_I64 );

   /* There are no prefixed version of these instructions.  */
   vassert( !prefix_instruction( prefix ) );

   assign( rS, getIReg(rS_addr) );
   assign( rB, getIReg(rB_addr) );

   if (opc1 != 0x1F) {
      vex_printf( "dis_logical_mask_bits(ppc)(instr)\n" );
      return False;
   }

   switch (opc2) {

      /* X-form instructions */
      case 0x03B: // cntlzdm, Count Leading Zeros Doubleword Under bitmask
      case 0x0BC: // pextd, Parallel Bits Extract Doubleword
      case 0x0DC: // cfuged, Centrifuge Doubleword
      case 0x23B: // cnttzdm, Count Trailing Zeros Doubleword Under bit mask
         {
            UInt max_bits = mode64 ? 64 : 32;
            IRTemp ones = newTemp( Ity_I64 );
            IRTemp all_ones = newTemp( Ity_I64 );

            /* Get the bits corresponding to 1's in the mask */
            assign( ones, extract_bits_under_mask ( vbi,
                                                    mkexpr( rS ),
                                                    mkexpr( rB ),
                                                    mkU64( 1 ) ) );

            if ( opc2 == 0x03b ) {  // cntlzdm
               IRTemp cnt = newTemp( Ity_I64 );

               DIP("cntlzdm r%u,r%u,r%u\n", rA_addr, rS_addr, rB_addr);
               assign( cnt, popcnt64( vbi, mkexpr( rB ) ) );

               assign( all_ones, binop( Iop_Shr64,
                                        mkU64( 0xFFFFFFFFFFFFFFFF ),
                                        unop( Iop_64to8, mkexpr( cnt ) ) ) );

               assign( rA,
                       unop( Iop_ClzNat64,
                             binop( Iop_Or64,
                                    binop( Iop_Shl64,
                                           mkexpr( ones ),
                                           binop( Iop_Sub8,
                                                  mkU8( max_bits ),
                                                  unop( Iop_64to8,
                                                        mkexpr( cnt ) ) ) ),
                                    mkexpr( all_ones ) ) ) );

            } else if ( opc2 == 0x0BC ) { // pextd
               DIP("pextd r%u,r%u,r%u\n", rA_addr, rS_addr, rB_addr);
               assign( rA, mkexpr( ones ) );

            } else if ( opc2 == 0x0DC ) { // cfuged
               IRTemp zeros = newTemp( Ity_I64 );
               IRTemp cnt = newTemp( Ity_I64 );

               DIP("cfuged r%u,r%u,r%u\n", rA_addr, rS_addr, rB_addr);
               assign( cnt, popcnt64( vbi, mkexpr( rB ) ) );

               /* Get the bits corresponding to 0's in the mask */
               assign( zeros, extract_bits_under_mask ( vbi,
                                                       mkexpr( rS ),
                                                       mkexpr( rB ),
                                                       mkU64( 0 ) ) );

               assign( rA,
                       binop( Iop_Or64,
                              binop( Iop_Shl64,
                                     mkexpr( zeros ),
                                     unop( Iop_64to8,
                                           mkexpr( cnt ) ) ),
                              mkexpr( ones ) ) );

            } else if ( opc2 == 0x23B ) {   //cnttzdm
               DIP("cnttzdm r%u,r%u,r%u\n", rA_addr, rS_addr, rB_addr);
               assign( all_ones, binop( Iop_Shl64,
                                        mkU64( 0xFFFFFFFFFFFFFFFF ),
                                        unop( Iop_64to8,
                                              popcnt64( vbi,
                                                        mkexpr( rB ) ) ) ) );

               assign( rA,
                       unop( Iop_CtzNat64,
                             binop( Iop_Or64,
                                    mkexpr( all_ones ), mkexpr( ones ) ) ) );

            } else {   //pexld
               DIP("pexld r%u,r%u,r%u\n", rA_addr, rS_addr, rB_addr);
               assign( rA, mkexpr( ones ) );
            }
            break;
         }

      case 0x09C: // pdepd, Parallel Bits Deposit Doubleword X-form
         {
            IRTemp ones = newTemp( Ity_I64 );

            DIP("pdepd r%u,r%u,r%u\n", rA_addr, rS_addr, rB_addr);
            assign( ones, deposit_bits_under_mask ( vbi, mkexpr( rS ),
                                                    mkexpr( rB ) ) );
            assign( rA, mkexpr( ones ) );
            break;
         }

      default:
         vex_printf("dis_logical_mask_bits)(ppc)\n");
         return False;
      }

   putIReg( rA_addr, mkexpr( rA ) );
   return True;
}

static Bool
dis_vector_logical_mask_bits ( UInt prefix, UInt theInstr, UInt opc2,
                               const VexAbiInfo* vbi )
{
   UChar vRA_addr = ifieldRegA(theInstr);
   UChar vRB_addr = ifieldRegB(theInstr);
   UChar vRT_addr = ifieldRegDS(theInstr);
   UChar opc1 = ifieldOPC( theInstr );
   IRTemp vA = newTemp(Ity_V128);
   IRTemp vB = newTemp(Ity_V128);

   /* There are no prefixed version of these instructions.  */
   vassert( !prefix_instruction( prefix ) );

   if (opc1 != 4) {
      vex_printf( "dis_vector_logical_mask_bits(ppc)(instr)\n" );
      return False;
   }

   assign( vA, getVReg(vRA_addr));
   assign( vB, getVReg(vRB_addr));

   switch (opc2) {
   case 0x4CC: // vgnb, Vector Gather every Nth Bit VX-form
      {
         IRTemp vB_hi = newTemp( Ity_I64 );
         IRTemp vB_lo = newTemp( Ity_I64 );
         IRTemp ones_hi, ones_lo;
         UChar N = toUChar( IFIELD( theInstr, 16, 3 ) );
         ULong extract_mask_hi, extract_mask_lo, byte_mask;
         UInt i, num_bits_hi, num_bits_lo;

         /* Note, the return register number is actually for a GPR not a
         vector register.  */
         DIP("vgnb %u,v%u,%u\n", vRT_addr, vRB_addr, N);

         if ((N < 2) || (N>7)) {
            /* The value of N can be any value between 2 and 7, inclusive.  */
            vex_printf("\nERROR: vgnb RT,VRB,N; N is out of range.\n\n");
            return False;
         }

         /* Create 32-bit extract mask, starting with bit 0 (IBM numbering),
            every Nth bit going right will be a 1.  */
         extract_mask_hi = 0;
         extract_mask_lo = 0;

         byte_mask = 1;

         i = 0;
         num_bits_hi = 0;
         while( i < 64) {
            extract_mask_hi = extract_mask_hi | (byte_mask << (63 - i));
            i = i + N;
            num_bits_hi++;
         }

         num_bits_lo = 0;
         while( i < 128) {
            extract_mask_lo = extract_mask_lo | (byte_mask << (127 - i));
            i = i + N;
            num_bits_lo++;
         }

         ones_hi = newTemp( Ity_I64 );
         ones_lo = newTemp( Ity_I64 );

         assign( vB_hi, unop( Iop_V128HIto64, mkexpr( vB ) ) );
         assign( vB_lo, unop( Iop_V128to64, mkexpr( vB ) ) );

         assign( ones_hi, extract_bits_under_mask ( vbi, mkexpr( vB_hi ),
                                                    mkU64( extract_mask_hi ),
                                                    mkU64( 1 ) ) );
         assign( ones_lo, extract_bits_under_mask ( vbi, mkexpr( vB_lo ),
                                                    mkU64( extract_mask_lo ),
                                                    mkU64( 1 ) ) );

         /* Concatenate the extracted bits from ones_hi and ones_lo and
            store in GPR.  Make sure the hi and low bits are left aligned per
            IBM numbering */
         putIReg( vRT_addr, binop( Iop_Or64,
                                   binop( Iop_Shl64,
                                          mkexpr( ones_hi ),
                                          mkU8( 64 - num_bits_hi ) ),
                                   binop( Iop_Shl64,
                                          mkexpr( ones_lo ),
                                          mkU8( 64 - num_bits_hi
                                          - num_bits_lo ) ) ) );
      }
      return True;

   case 0x54D: // vcfuged, Centrifuge Doubleword VX-form
      {
         IRTemp vA_hi = newTemp( Ity_I64 );
         IRTemp vA_lo = newTemp( Ity_I64 );
         IRTemp vB_hi = newTemp( Ity_I64 );
         IRTemp vB_lo = newTemp( Ity_I64 );
         IRTemp zeros[2];
         IRTemp ones[2];
         IRTemp count[2];

         DIP("vcfuged v%u,v%u,v%u\n", vRT_addr, vRA_addr, vRB_addr);

         zeros[0] = newTemp( Ity_I64 );
         zeros[1] = newTemp( Ity_I64 );
         ones[0] = newTemp( Ity_I64 );
         ones[1] = newTemp( Ity_I64 );
         count[0] = newTemp( Ity_I64 );
         count[1] = newTemp( Ity_I64 );

         assign( vA_hi, unop( Iop_V128HIto64, mkexpr( vA ) ) );
         assign( vB_hi, unop( Iop_V128HIto64, mkexpr( vB ) ) );
         assign( vA_lo, unop( Iop_V128to64, mkexpr( vA ) ) );
         assign( vB_lo, unop( Iop_V128to64, mkexpr( vB ) ) );

         assign( count[0], popcnt64( vbi, mkexpr( vB_hi ) ) );
         assign( count[1], popcnt64( vbi, mkexpr( vB_lo ) ) );

         assign( ones[0], extract_bits_under_mask ( vbi, mkexpr( vA_hi ),
                                                    mkexpr( vB_hi ),
                                                    mkU64( 1 ) ) );
         assign( ones[1], extract_bits_under_mask ( vbi, mkexpr( vA_lo ),
                                                    mkexpr( vB_lo ),
                                                    mkU64( 1 ) ) );
         assign( zeros[0], extract_bits_under_mask ( vbi, mkexpr( vA_hi ),
                                                     mkexpr( vB_hi ),
                                                     mkU64( 0 ) ) );
         assign( zeros[1], extract_bits_under_mask ( vbi, mkexpr( vA_lo ),
                                                     mkexpr( vB_lo ),
                                                     mkU64( 0 ) ) );

         /* Put the bits corresponding to zero mask bits to the left of the
             bits corresponding to one mask bits for the upper and lower 64-bit
             words.  */
         putVReg( vRT_addr, binop( Iop_64HLtoV128,
                                   binop( Iop_Or64,
                                          binop( Iop_Shl64,
                                                 mkexpr( zeros[0] ),
                                                 unop( Iop_64to8,
                                                       mkexpr( count[0] ) ) ),
                                          mkexpr( ones[0] ) ),
                                   binop( Iop_Or64,
                                          binop( Iop_Shl64,
                                                 mkexpr( zeros[1] ),
                                                 unop( Iop_64to8,
                                                       mkexpr( count[1] ) ) ),
                                          mkexpr( ones[1] ) ) ) );
      }
      break;

   case 0x58D: // vpextd, Vector Parallel Bits Extract Doubleword VX-form
      {
         IRTemp vA_hi = newTemp( Ity_I64 );
         IRTemp vA_lo = newTemp( Ity_I64 );
         IRTemp vB_hi = newTemp( Ity_I64 );
         IRTemp vB_lo = newTemp( Ity_I64 );
         IRTemp ones[2];

         DIP("vpextd v%u,v%u,v%u\n", vRT_addr, vRA_addr, vRB_addr);

         ones[0] = newTemp( Ity_I64 );
         ones[1] = newTemp( Ity_I64 );

         assign( vA_hi, unop( Iop_V128HIto64, mkexpr( vA ) ) );
         assign( vB_hi, unop( Iop_V128HIto64, mkexpr( vB ) ) );
         assign( vA_lo, unop( Iop_V128to64, mkexpr( vA ) ) );
         assign( vB_lo, unop( Iop_V128to64, mkexpr( vB ) ) );

         assign( ones[0], extract_bits_under_mask ( vbi, mkexpr( vA_hi ),
                                                    mkexpr( vB_hi ),
                                                    mkU64( 1 ) ) );
         assign( ones[1], extract_bits_under_mask ( vbi, mkexpr( vA_lo ),
                                                    mkexpr( vB_lo ),
                                                    mkU64( 1 ) ) );
         putVReg( vRT_addr, binop( Iop_64HLtoV128,
                                   mkexpr( ones[0] ), mkexpr( ones[1] ) ) );
      }
      break;

   case 0x5CD: // vpdepd, Vector Parallel Bits Deposit Doubleword VX-form
      {
         IRTemp vA_hi = newTemp( Ity_I64 );
         IRTemp vA_lo = newTemp( Ity_I64 );
         IRTemp vB_hi = newTemp( Ity_I64 );
         IRTemp vB_lo = newTemp( Ity_I64 );
         IRTemp ones[2];

         DIP("vpdepd v%u,v%u,v%u\n", vRT_addr, vRA_addr, vRB_addr);

         ones[0] = newTemp( Ity_I64 );
         ones[1] = newTemp( Ity_I64 );

         assign( vA_hi, unop( Iop_V128HIto64, mkexpr( vA ) ) );
         assign( vB_hi, unop( Iop_V128HIto64, mkexpr( vB ) ) );
         assign( vA_lo, unop( Iop_V128to64, mkexpr( vA ) ) );
         assign( vB_lo, unop( Iop_V128to64, mkexpr( vB ) ) );

         assign( ones[0], deposit_bits_under_mask ( vbi, mkexpr( vA_hi ),
                                                    mkexpr( vB_hi ) ) );
         assign( ones[1], deposit_bits_under_mask ( vbi, mkexpr( vA_lo ),
                                                    mkexpr( vB_lo ) ) );
         putVReg( vRT_addr, binop( Iop_64HLtoV128,
                                   mkexpr( ones[0] ), mkexpr( ones[1] ) ) );
      }
      break;

   case 0x784:    // vclzdm,
      {
         /* Vector Count Leading Zeros Doubleword under bit mask */

         IRTemp extracted_bits[2];
         IRTemp clz[2];
         IRTemp ones[2];
         IRTemp cnt_extract_bits[2];
         UInt max_bits = 64;
         IRTemp vA_hi = newTemp( Ity_I64 );
         IRTemp vA_lo = newTemp( Ity_I64 );
         IRTemp vB_hi = newTemp( Ity_I64 );
         IRTemp vB_lo = newTemp( Ity_I64 );

         DIP("vclzdm v%u,v%u,v%u\n", vRT_addr, vRA_addr, vRB_addr);

         ones[0] = newTemp( Ity_I64 );
         ones[1] = newTemp( Ity_I64 );
         clz[0] = newTemp( Ity_I64 );
         clz[1] = newTemp( Ity_I64 );
         extracted_bits[0] = newTemp( Ity_I64 );
         extracted_bits[1] = newTemp( Ity_I64 );
         cnt_extract_bits[0] = newTemp( Ity_I8 );
         cnt_extract_bits[1] = newTemp( Ity_I8 );

         /* Gather bits in each vector element, then count leading zeros.  */
         assign( vA_hi, unop( Iop_V128HIto64, mkexpr( vA ) ) );
         assign( vB_hi, unop( Iop_V128HIto64, mkexpr( vB ) ) );
         assign( vA_lo, unop( Iop_V128to64, mkexpr( vA ) ) );
         assign( vB_lo, unop( Iop_V128to64, mkexpr( vB ) ) );

         assign( ones[0], extract_bits_under_mask ( vbi,
                                                    mkexpr( vA_hi ),
                                                    mkexpr( vB_hi ),
                                                    mkU64( 1 ) ) );

         assign( ones[1], extract_bits_under_mask ( vbi,
                                                    mkexpr( vA_lo ),
                                                    mkexpr( vB_lo ),
                                                    mkU64( 1 ) ) );

         assign( cnt_extract_bits[0],
                 unop( Iop_16to8,
                       unop( Iop_32to16,
                             count_bits_under_mask ( vbi,
                                                     mkexpr( vA_hi ),
                                                     mkexpr( vB_hi ),
                                                     mkU64( 1 ) ) ) ) );

         assign( cnt_extract_bits[1],
                 unop( Iop_16to8,
                       unop( Iop_32to16,
                             count_bits_under_mask ( vbi,
                                                     mkexpr( vA_lo ),
                                                     mkexpr( vB_lo ),
                                                     mkU64( 1 ) ) ) ) );

         /* Shift extracted bits to High order bits, filling lower order bits
            with 1's so we only count zeros in extracted bits.  */
         assign( extracted_bits[0],
                 binop( Iop_Or64,
                        binop( Iop_Shr64,
                               mkU64( 0xFFFFFFFFFFFFFFFF ),
                               mkexpr( cnt_extract_bits[0] ) ),
                        binop( Iop_Shl64,
                               mkexpr( ones[0] ),
                               binop( Iop_Sub8,
                                      mkU8( max_bits ),
                                      mkexpr( cnt_extract_bits[0] )
                                  ) ) ) );

         assign( clz[0],
                 unop( Iop_Clz64,
                       mkexpr( extracted_bits[0] ) ) );

         assign( extracted_bits[1],
                 binop( Iop_Or64,
                        binop( Iop_Shr64,
                               mkU64( 0xFFFFFFFFFFFFFFFF ),
                               mkexpr( cnt_extract_bits[1] ) ),
                        binop( Iop_Shl64,
                               mkexpr( ones[1] ),
                               binop( Iop_Sub8,
                                      mkU8( max_bits ),
                                      mkexpr( cnt_extract_bits[1] )
                                  ) ) ) );
         assign( clz[1],
                 unop( Iop_Clz64,
                       mkexpr( extracted_bits[1] ) ) );

         putVReg( vRT_addr, binop( Iop_64HLtoV128,
                                   mkexpr( clz[0] ), mkexpr( clz[1] ) ) );
         break;
      }

   case 0x7C4:    // vctzdm
      {
         /* Vector Count Trailing Zeros Doubleword under bit mask */
         IRTemp ctz[2];
         IRTemp ones[2];
         IRTemp all_ones_hi = newTemp( Ity_I64 );
         IRTemp all_ones_lo = newTemp( Ity_I64 );
         IRTemp vA_hi = newTemp( Ity_I64 );
         IRTemp vA_lo = newTemp( Ity_I64 );
         IRTemp vB_hi = newTemp( Ity_I64 );
         IRTemp vB_lo = newTemp( Ity_I64 );

         DIP("vctzdm v%u,v%u,v%u\n", vRT_addr, vRA_addr, vRB_addr);

         ones[0] = newTemp( Ity_I64 );
         ones[1] = newTemp( Ity_I64 );
         ctz[0] = newTemp( Ity_I64 );
         ctz[1] = newTemp( Ity_I64 );

         /* Gather bits in each vector element, then count trailing zeros.  */
         assign( vA_hi, unop( Iop_V128HIto64, mkexpr( vA ) ) );
         assign( vB_hi, unop( Iop_V128HIto64, mkexpr( vB ) ) );
         assign( vA_lo, unop( Iop_V128to64, mkexpr( vA ) ) );
         assign( vB_lo, unop( Iop_V128to64, mkexpr( vB ) ) );

         /* Shift all 1's value left by the count of the number of bits in the
            mask.  OR this with the extracted bits so the trailing zero count
            will only count zeros in extracted field. */
         assign( all_ones_hi,
                 binop( Iop_Shl64,
                        mkU64( 0xFFFFFFFFFFFFFFFF ),
                        unop( Iop_64to8,
                              popcnt64( vbi, mkexpr( vB_hi ) ) ) ) );
         assign( all_ones_lo,
                 binop( Iop_Shl64,
                        mkU64( 0xFFFFFFFFFFFFFFFF ),
                        unop( Iop_64to8,
                              popcnt64( vbi, mkexpr( vB_lo ) ) ) ) );

         assign( ones[0],
                 binop( Iop_Or64,
                        mkexpr( all_ones_hi ),
                        extract_bits_under_mask ( vbi,
                                                  mkexpr( vA_hi ),
                                                  mkexpr( vB_hi ),
                                                  mkU64( 1 ) ) ) );

         assign( ones[1],
                 binop( Iop_Or64,
                        mkexpr( all_ones_lo ),
                        extract_bits_under_mask ( vbi,
                                                  mkexpr( vA_lo ),
                                                  mkexpr( vB_lo ),
                                                  mkU64( 1 ) ) ) );

         assign( ctz[0], unop( Iop_CtzNat64, mkexpr( ones[0] ) ) );
         assign( ctz[1], unop( Iop_CtzNat64, mkexpr( ones[1] ) ) );

         putVReg( vRT_addr, binop( Iop_64HLtoV128,
                                   mkexpr( ctz[0] ), mkexpr( ctz[1] ) ) );
         break;
      }

   default:
         vex_printf("dis_vector_logical_mask_bits(ppc)(opc2)\n");
         return False;
   }
   return True;
}

typedef enum {
   PPC_CMP_EQ = 2,
   PPC_CMP_GT = 4,
   PPC_CMP_GE = 6,
   PPC_CMP_LT = 8
} ppc_cmp_t;


/*
  This helper function takes as input the IRExpr returned
  from a binop( Iop_CmpF64, fpA, fpB), whose result is returned
  in IR form.  This helper function converts it to PPC form.

  Map compare result from IR to PPC

  FP cmp result | PPC | IR
  --------------------------
  UN            | 0x1 | 0x45
  EQ            | 0x2 | 0x40
  GT            | 0x4 | 0x00
  LT            | 0x8 | 0x01

 condcode = Shl(1, (~(ccIR>>5) & 2)
                    | ((ccIR ^ (ccIR>>6)) & 1)
*/
static IRTemp
get_fp_cmp_CR_val (IRExpr * ccIR_expr)
{
   IRTemp condcode = newTemp( Ity_I32 );
   IRTemp ccIR = newTemp( Ity_I32 );

   assign(ccIR, ccIR_expr);
   assign( condcode,
           binop( Iop_Shl32,
                  mkU32( 1 ),
                  unop( Iop_32to8,
                        binop( Iop_Or32,
                               binop( Iop_And32,
                                      unop( Iop_Not32,
                                            binop( Iop_Shr32,
                                                   mkexpr( ccIR ),
                                                   mkU8( 5 ) ) ),
                                      mkU32( 2 ) ),
                               binop( Iop_And32,
                                      binop( Iop_Xor32,
                                             mkexpr( ccIR ),
                                             binop( Iop_Shr32,
                                                    mkexpr( ccIR ),
                                                    mkU8( 6 ) ) ),
                                      mkU32( 1 ) ) ) ) ) );
   return condcode;
}

/*
 * Helper function for get_max_min_fp for ascertaining the max or min between two doubles
 * following these special rules:
 *   - The max/min of a QNaN and any value is that value
 *     (When two QNaNs are being compared, the frA QNaN is the return value.)
 *   - The max/min of any value and an SNaN is that SNaN converted to a QNaN
 *     (When two SNaNs are being compared, the frA SNaN is converted to a QNaN.)
 */
static IRExpr * _get_maxmin_fp_NaN(IRTemp frA_I64, IRTemp frB_I64)
{
   IRTemp frA_isNaN = newTemp(Ity_I1);
   IRTemp frB_isNaN = newTemp(Ity_I1);
   IRTemp frA_isSNaN = newTemp(Ity_I1);
   IRTemp frB_isSNaN = newTemp(Ity_I1);
   IRTemp frA_isQNaN = newTemp(Ity_I1);
   IRTemp frB_isQNaN = newTemp(Ity_I1);

   assign( frA_isNaN, is_NaN( Ity_I64, frA_I64 ) );
   assign( frB_isNaN, is_NaN( Ity_I64, frB_I64 ) );
   // If operand is a NAN and bit 12 is '0', then it's an SNaN
   assign( frA_isSNaN,
           mkAND1( mkexpr(frA_isNaN),
                   binop( Iop_CmpEQ32,
                          binop( Iop_And32,
                                 unop( Iop_64HIto32, mkexpr( frA_I64 ) ),
                                 mkU32( 0x00080000 ) ),
                          mkU32( 0 ) ) ) );
   assign( frB_isSNaN,
           mkAND1( mkexpr(frB_isNaN),
                   binop( Iop_CmpEQ32,
                          binop( Iop_And32,
                                 unop( Iop_64HIto32, mkexpr( frB_I64 ) ),
                                 mkU32( 0x00080000 ) ),
                          mkU32( 0 ) ) ) );
   assign( frA_isQNaN,
           mkAND1( mkexpr( frA_isNaN ), unop( Iop_Not1, mkexpr( frA_isSNaN ) ) ) );
   assign( frB_isQNaN,
           mkAND1( mkexpr( frB_isNaN ), unop( Iop_Not1, mkexpr( frB_isSNaN ) ) ) );

   /* Based on the rules specified in the function prologue, the algorithm is as follows:
    *  <<<<<<<<<>>>>>>>>>>>>>>>>>>
    *   if frA is a SNaN
    *     result = frA converted to QNaN
    *   else if frB is a SNaN
    *     if (frA is QNan)
    *        result = frA
    *     else
    *        result = frB converted to QNaN
    *   else if frB is a QNaN
    *     result = frA
    *   // One of frA or frB was a NaN in order for this function to be called, so
    *   // if we get to this point, we KNOW that frA must be a QNaN.
    *   else // frA is a QNaN
    *     result = frB
    *  <<<<<<<<<>>>>>>>>>>>>>>>>>>
    */

#define SNAN_MASK 0x0008000000000000ULL

   return
      IRExpr_ITE(mkexpr(frA_isSNaN),
              /* then: result = frA converted to QNaN */
              binop(Iop_Or64, mkexpr(frA_I64), mkU64(SNAN_MASK)),
              /* else:  if frB is a SNaN */
              IRExpr_ITE(mkexpr(frB_isSNaN),
                         IRExpr_ITE(mkexpr(frA_isQNaN),
                                    /* then: result = frA  */
                                    mkexpr(frA_I64),
                                    /* else: result = frB converted to QNaN */
                                    binop(Iop_Or64, mkexpr(frB_I64),
                                          mkU64(SNAN_MASK))),
                         /* else:  if frB is a QNaN */
                         IRExpr_ITE(mkexpr(frB_isQNaN),
                                    /* then: result = frA */
                                    mkexpr(frA_I64),
                                    /* else:  frA is a QNaN, so result = frB */
                                    mkexpr(frB_I64))));
}

/*
 * Helper function for get_max_min_fp.
 */
static IRExpr * _get_maxmin_fp_cmp(IRTemp src1, IRTemp src2, Bool isMin)
{
   IRTemp src1cmpsrc2 = get_fp_cmp_CR_val( binop( Iop_CmpF64,
                                                  unop( Iop_ReinterpI64asF64,
                                                        mkexpr( src1 ) ),
                                                  unop( Iop_ReinterpI64asF64,
                                                        mkexpr( src2 ) ) ) );

   return IRExpr_ITE( binop( Iop_CmpEQ32,
                               mkexpr( src1cmpsrc2 ),
                               mkU32( isMin ? PPC_CMP_LT : PPC_CMP_GT ) ),
                      /* then: use src1 */
                      mkexpr( src1 ),
                      /* else: use src2 */
                      mkexpr( src2 ) );
}

/*
 * Helper function for "Maximum/Minimum Double Precision" operations.
 * Arguments: frA and frb are Ity_I64
 * Returns Ity_I64 IRExpr that answers the "which is Maxiumum/Minimum" question
 */
static IRExpr * get_max_min_fp(IRTemp frA_I64, IRTemp frB_I64, Bool isMin)
{
   /* There are three special cases where get_fp_cmp_CR_val is not helpful
    * for ascertaining the maximum between two doubles:
    *   1. The max/min of +0 and -0 is +0.
    *   2. The max/min of a QNaN and any value is that value.
    *   3. The max/min of any value and an SNaN is that SNaN converted to a QNaN.
    * We perform the check for [+/-]0 here in this function and use the
    * _get_maxmin_fp_NaN helper for the two NaN cases; otherwise we call _get_maxmin_fp_cmp
    * to do the standard comparison function.
    */
   IRTemp anyNaN = newTemp(Ity_I1);
   IRTemp frA_isZero = newTemp(Ity_I1);
   IRTemp frB_isZero = newTemp(Ity_I1);
   assign( frA_isZero, is_Zero( Ity_I64, frA_I64 ) );
   assign( frB_isZero, is_Zero( Ity_I64, frB_I64 ) );
   assign( anyNaN, mkOR1( is_NaN( Ity_I64, frA_I64 ),
                          is_NaN(Ity_I64, frB_I64 ) ) );
#define MINUS_ZERO 0x8000000000000000ULL

   return IRExpr_ITE( /* If both arguments are zero . . . */
                     mkAND1( mkexpr( frA_isZero ), mkexpr( frB_isZero ) ),
                     /* then: if frA is -0 and isMin==True, return -0;
                      *     else if frA is +0 and isMin==False; return +0;
                      *     otherwise, simply return frB. */
                     IRExpr_ITE( binop( Iop_CmpEQ32,
                                        unop( Iop_64HIto32,
                                              mkexpr( frA_I64 ) ),
                                        mkU32( isMin ? 0x80000000 : 0 ) ),
                                 mkU64( isMin ? MINUS_ZERO : 0ULL ),
                                 mkexpr( frB_I64 ) ),
                     /* else: check if either input is a NaN*/
                     IRExpr_ITE( mkexpr( anyNaN ),
                                 /* then: use "NaN helper" */
                                 _get_maxmin_fp_NaN( frA_I64, frB_I64 ),
                                 /* else: use "comparison helper" */
                                 _get_maxmin_fp_cmp( frB_I64, frA_I64, isMin ) ));
}

static const HChar * _get_vsx_rdpi_suffix(UInt opc2)
{
   switch (opc2 & 0x7F) {
      case 0x72:
         return "m";
      case 0x52:
         return "p";
      case 0x56:
         return "c";
      case 0x32:
         return "z";
      case 0x12:
         return "";

      default: // Impossible to get here
         vex_printf("Unrecognized opcode %x\n", opc2);
         vpanic("_get_vsx_rdpi_suffix(ppc)(opc2)");
   }
}

/*
 * Helper function for vector/scalar double precision fp round to integer instructions.
 */
static IRExpr * _do_vsx_fp_roundToInt(IRTemp frB_I64, UInt opc2)
{

   /* The same rules apply for x{s|v}rdpi{m|p|c|z} as for floating point round operations (fri{m|n|p|z}). */
   IRTemp frB = newTemp(Ity_F64);
   IRTemp frD = newTemp(Ity_F64);
   IRTemp intermediateResult = newTemp(Ity_I64);
   IRTemp is_SNAN = newTemp(Ity_I1);
   IRExpr * hi32;
   IRExpr * rxpi_rm;
   switch (opc2 & 0x7F) {
      case 0x72:
         rxpi_rm = mkU32(Irrm_NegINF);
         break;
      case 0x52:
         rxpi_rm = mkU32(Irrm_PosINF);
         break;
      case 0x56:
         rxpi_rm = get_IR_roundingmode();
         break;
      case 0x32:
         rxpi_rm = mkU32(Irrm_ZERO);
         break;
      case 0x12:
         rxpi_rm = mkU32(Irrm_NEAREST);
         break;

      default: // Impossible to get here
         vex_printf("Unrecognized opcode %x\n", opc2);
         vpanic("_do_vsx_fp_roundToInt(ppc)(opc2)");
   }
   assign(frB, unop(Iop_ReinterpI64asF64, mkexpr(frB_I64)));
   assign( intermediateResult,
           binop( Iop_F64toI64S, rxpi_rm,
                  mkexpr( frB ) ) );

   /* don't use the rounded integer if frB is outside -9e18..9e18 */
   /* F64 has only log10(2**52) significant digits anyway */
   /* need to preserve sign of zero */
   /*   frD = (fabs(frB) > 9e18) ? frB :
            (sign(frB)) ? -fabs((double)intermediateResult) : (double)intermediateResult  */
   assign( frD,
           IRExpr_ITE(
              binop( Iop_CmpNE8,
                     unop( Iop_32to8,
                           binop( Iop_CmpF64,
                                  IRExpr_Const( IRConst_F64( 9e18 ) ),
                                  unop( Iop_AbsF64, mkexpr( frB ) ) ) ),
                     mkU8(0) ),
              mkexpr( frB ),
              IRExpr_ITE(
                 binop( Iop_CmpNE32,
                        binop( Iop_Shr32,
                               unop( Iop_64HIto32,
                                     mkexpr( frB_I64 ) ),
                               mkU8( 31 ) ),
                        mkU32(0) ),
                 unop( Iop_NegF64,
                       unop( Iop_AbsF64,
                             binop( Iop_I64StoF64,
                                    mkU32( 0 ),
                                    mkexpr( intermediateResult ) ) ) ),
                 binop( Iop_I64StoF64,
                        mkU32( 0 ),
                        mkexpr( intermediateResult ) )
              )
           )
   );

   /* See Appendix "Floating-Point Round to Integer Model" in ISA doc.
    * If frB is a SNAN, then frD <- frB, with bit 12 set to '1'.
    */
#define SNAN_MASK 0x0008000000000000ULL
   hi32 = unop( Iop_64HIto32, mkexpr(frB_I64) );
   assign( is_SNAN,
           mkAND1( is_NaN( Ity_I64, frB_I64 ),
                   binop( Iop_CmpEQ32,
                          binop( Iop_And32, hi32, mkU32( 0x00080000 ) ),
                          mkU32( 0 ) ) ) );

   return IRExpr_ITE( mkexpr( is_SNAN ),
                        unop( Iop_ReinterpI64asF64,
                              binop( Iop_Xor64,
                                     mkU64( SNAN_MASK ),
                                     mkexpr( frB_I64 ) ) ),
                      mkexpr( frD ));
}

/*
 * Miscellaneous VSX vector instructions
 */
static Bool
dis_vxv_misc ( UInt prefix, UInt theInstr, UInt opc2 )
{
   /* XX3-Form */
   UChar opc1 = ifieldOPC( theInstr );
   UChar XT = ifieldRegXT( theInstr );
   UChar XB = ifieldRegXB( theInstr );

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   if (opc1 != 0x3C) {
      vex_printf( "dis_vxv_misc(ppc)(instr)\n" );
      return False;
   }

   switch (opc2) {
      case 0x1B4:  // xvredp (VSX Vector Reciprocal Estimate Double-Precision)
      case 0x194:  // xvrsqrtedp (VSX Vector Reciprocal Square Root Estimate
                   //             Double-Precision)
      {
         IRExpr* ieee_one = IRExpr_Const(IRConst_F64i(0x3ff0000000000000ULL));
         IRExpr* rm  = get_IR_roundingmode();
         IRTemp frB = newTemp(Ity_I64);
         IRTemp frB2 = newTemp(Ity_I64);
         Bool redp = opc2 == 0x1B4;
         IRTemp sqrtHi = newTemp(Ity_F64);
         IRTemp sqrtLo = newTemp(Ity_F64);
         assign(frB,  unop(Iop_V128HIto64, getVSReg( XB )));
         assign(frB2, unop(Iop_V128to64, getVSReg( XB )));

         DIP("%s v%d,v%d\n", redp ? "xvredp" : "xvrsqrtedp", XT, XB);

         if (!redp) {
            assign( sqrtHi,
                    binop( Iop_SqrtF64,
                           rm,
                           unop( Iop_ReinterpI64asF64, mkexpr( frB ) ) ) );
            assign( sqrtLo,
                    binop( Iop_SqrtF64,
                           rm,
                           unop( Iop_ReinterpI64asF64, mkexpr( frB2 ) ) ) );
         }
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          unop( Iop_ReinterpF64asI64,
                                triop( Iop_DivF64,
                                       rm,
                                       ieee_one,
                                       redp ? unop( Iop_ReinterpI64asF64,
                                                    mkexpr( frB ) )
                                            : mkexpr( sqrtHi ) ) ),
                          unop( Iop_ReinterpF64asI64,
                                triop( Iop_DivF64,
                                       rm,
                                       ieee_one,
                                       redp ? unop( Iop_ReinterpI64asF64,
                                                    mkexpr( frB2 ) )
                                            : mkexpr( sqrtLo ) ) ) ) );
         break;

      }
      case 0x134: // xvresp (VSX Vector Reciprocal Estimate Single-Precision)
      case 0x114: // xvrsqrtesp (VSX Vector Reciprocal Square Root Estimate Single-Precision)
      {
         IRTemp b3, b2, b1, b0;
         IRTemp res0 = newTemp(Ity_I32);
         IRTemp res1 = newTemp(Ity_I32);
         IRTemp res2 = newTemp(Ity_I32);
         IRTemp res3 = newTemp(Ity_I32);
         IRTemp sqrt3 = newTemp(Ity_F64);
         IRTemp sqrt2 = newTemp(Ity_F64);
         IRTemp sqrt1 = newTemp(Ity_F64);
         IRTemp sqrt0 = newTemp(Ity_F64);
         IRExpr* rm  = get_IR_roundingmode();
         Bool resp = opc2 == 0x134;

         IRExpr* ieee_one = IRExpr_Const(IRConst_F64i(0x3ff0000000000000ULL));

         b3 = b2 = b1 = b0 = IRTemp_INVALID;
         DIP("%s v%d,v%d\n", resp ? "xvresp" : "xvrsqrtesp", XT, XB);
         breakV128to4xF64( getVSReg( XB ), &b3, &b2, &b1, &b0 );

         if (!resp) {
            assign( sqrt3, binop( Iop_SqrtF64, rm, mkexpr( b3 ) ) );
            assign( sqrt2, binop( Iop_SqrtF64, rm, mkexpr( b2 ) ) );
            assign( sqrt1, binop( Iop_SqrtF64, rm, mkexpr( b1 ) ) );
            assign( sqrt0, binop( Iop_SqrtF64, rm, mkexpr( b0 ) ) );
         }

         assign( res0,
                 unop( Iop_ReinterpF32asI32,
                       unop( Iop_TruncF64asF32,
                             triop( Iop_DivF64r32,
                                    rm,
                                    ieee_one,
                                    resp ? mkexpr( b0 ) : mkexpr( sqrt0 ) ) ) ) );
         assign( res1,
                 unop( Iop_ReinterpF32asI32,
                       unop( Iop_TruncF64asF32,
                             triop( Iop_DivF64r32,
                                    rm,
                                    ieee_one,
                                    resp ? mkexpr( b1 ) : mkexpr( sqrt1 ) ) ) ) );
         assign( res2,
                 unop( Iop_ReinterpF32asI32,
                       unop( Iop_TruncF64asF32,
                             triop( Iop_DivF64r32,
                                    rm,
                                    ieee_one,
                                    resp ? mkexpr( b2 ) : mkexpr( sqrt2 ) ) ) ) );
         assign( res3,
                 unop( Iop_ReinterpF32asI32,
                       unop( Iop_TruncF64asF32,
                             triop( Iop_DivF64r32,
                                    rm,
                                    ieee_one,
                                    resp ? mkexpr( b3 ) : mkexpr( sqrt3 ) ) ) ) );
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          binop( Iop_32HLto64, mkexpr( res3 ), mkexpr( res2 ) ),
                          binop( Iop_32HLto64, mkexpr( res1 ), mkexpr( res0 ) ) ) );
         break;
      }
      case 0x300: // xvmaxsp (VSX Vector Maximum Single-Precision)
      case 0x320: // xvminsp (VSX Vector Minimum Single-Precision)
      {
         UChar XA = ifieldRegXA( theInstr );
         IRTemp a3, a2, a1, a0;
         IRTemp b3, b2, b1, b0;
         IRTemp res0 = newTemp( Ity_I32 );
         IRTemp res1 = newTemp( Ity_I32 );
         IRTemp res2 = newTemp( Ity_I32 );
         IRTemp res3 = newTemp( Ity_I32 );
         IRTemp a0_I64 = newTemp( Ity_I64 );
         IRTemp a1_I64 = newTemp( Ity_I64 );
         IRTemp a2_I64 = newTemp( Ity_I64 );
         IRTemp a3_I64 = newTemp( Ity_I64 );
         IRTemp b0_I64 = newTemp( Ity_I64 );
         IRTemp b1_I64 = newTemp( Ity_I64 );
         IRTemp b2_I64 = newTemp( Ity_I64 );
         IRTemp b3_I64 = newTemp( Ity_I64 );

         Bool isMin = opc2 == 0x320 ? True : False;

         a3 = a2 = a1 = a0 = IRTemp_INVALID;
         b3 = b2 = b1 = b0 = IRTemp_INVALID;
         DIP("%s v%d,v%d v%d\n", isMin ? "xvminsp" : "xvmaxsp", XT, XA, XB);
         breakV128to4xF64( getVSReg( XA ), &a3, &a2, &a1, &a0 );
         breakV128to4xF64( getVSReg( XB ), &b3, &b2, &b1, &b0 );
         assign( a0_I64, unop( Iop_ReinterpF64asI64, mkexpr( a0 ) ) );
         assign( b0_I64, unop( Iop_ReinterpF64asI64, mkexpr( b0 ) ) );
         assign( a1_I64, unop( Iop_ReinterpF64asI64, mkexpr( a1 ) ) );
         assign( b1_I64, unop( Iop_ReinterpF64asI64, mkexpr( b1 ) ) );
         assign( a2_I64, unop( Iop_ReinterpF64asI64, mkexpr( a2 ) ) );
         assign( b2_I64, unop( Iop_ReinterpF64asI64, mkexpr( b2 ) ) );
         assign( a3_I64, unop( Iop_ReinterpF64asI64, mkexpr( a3 ) ) );
         assign( b3_I64, unop( Iop_ReinterpF64asI64, mkexpr( b3 ) ) );
         assign( res0,
                 unop( Iop_ReinterpF32asI32,
                       unop( Iop_TruncF64asF32,
                             unop( Iop_ReinterpI64asF64,
                                   get_max_min_fp( a0_I64, b0_I64, isMin ) ) ) ) );
         assign( res1,
                 unop( Iop_ReinterpF32asI32,
                       unop( Iop_TruncF64asF32,
                             unop( Iop_ReinterpI64asF64,
                                   get_max_min_fp( a1_I64, b1_I64, isMin ) ) ) ) );
         assign( res2,
                 unop( Iop_ReinterpF32asI32,
                       unop( Iop_TruncF64asF32,
                             unop( Iop_ReinterpI64asF64,
                                   get_max_min_fp( a2_I64, b2_I64, isMin ) ) ) ) );
         assign( res3,
                 unop( Iop_ReinterpF32asI32,
                       unop( Iop_TruncF64asF32,
                             unop( Iop_ReinterpI64asF64,
                                   get_max_min_fp( a3_I64, b3_I64, isMin ) ) ) ) );
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          binop( Iop_32HLto64, mkexpr( res3 ), mkexpr( res2 ) ),
                          binop( Iop_32HLto64, mkexpr( res1 ), mkexpr( res0 ) ) ) );
         break;
      }
      case 0x380: // xvmaxdp (VSX Vector Maximum Double-Precision)
      case 0x3A0: // xvmindp (VSX Vector Minimum Double-Precision)
      {
         UChar XA = ifieldRegXA( theInstr );
         IRTemp frA = newTemp(Ity_I64);
         IRTemp frB = newTemp(Ity_I64);
         IRTemp frA2 = newTemp(Ity_I64);
         IRTemp frB2 = newTemp(Ity_I64);
         Bool isMin = opc2 == 0x3A0 ? True : False;

         assign(frA,  unop(Iop_V128HIto64, getVSReg( XA )));
         assign(frB,  unop(Iop_V128HIto64, getVSReg( XB )));
         assign(frA2, unop(Iop_V128to64, getVSReg( XA )));
         assign(frB2, unop(Iop_V128to64, getVSReg( XB )));
         DIP("%s v%d,v%d v%d\n", isMin ? "xvmindp" : "xvmaxdp", XT, XA, XB);
         putVSReg( XT, binop( Iop_64HLtoV128, get_max_min_fp(frA, frB, isMin), get_max_min_fp(frA2, frB2, isMin) ) );

         break;
      }
      case 0x3c0: // xvcpsgndp (VSX Vector Copy Sign Double-Precision)
      {
         UChar XA = ifieldRegXA( theInstr );
         IRTemp frA = newTemp(Ity_I64);
         IRTemp frB = newTemp(Ity_I64);
         IRTemp frA2 = newTemp(Ity_I64);
         IRTemp frB2 = newTemp(Ity_I64);
         assign(frA,  unop(Iop_V128HIto64, getVSReg( XA )));
         assign(frB,  unop(Iop_V128HIto64, getVSReg( XB )));
         assign(frA2, unop(Iop_V128to64, getVSReg( XA )));
         assign(frB2, unop(Iop_V128to64, getVSReg( XB )));

         DIP("xvcpsgndp v%d,v%d,v%d\n", XT, XA, XB);
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          binop( Iop_Or64,
                                 binop( Iop_And64,
                                        mkexpr( frA ),
                                        mkU64( SIGN_BIT ) ),
                                 binop( Iop_And64,
                                        mkexpr( frB ),
                                        mkU64( SIGN_MASK ) ) ),
                          binop( Iop_Or64,
                                 binop( Iop_And64,
                                        mkexpr( frA2 ),
                                        mkU64( SIGN_BIT ) ),
                                 binop( Iop_And64,
                                        mkexpr( frB2 ),
                                        mkU64( SIGN_MASK ) ) ) ) );
         break;
      }
      case 0x340: // xvcpsgnsp
      {
         UChar XA = ifieldRegXA( theInstr );
         IRTemp a3_I64, a2_I64, a1_I64, a0_I64;
         IRTemp b3_I64, b2_I64, b1_I64, b0_I64;
         IRTemp resHi = newTemp(Ity_I64);
         IRTemp resLo = newTemp(Ity_I64);

         a3_I64 = a2_I64 = a1_I64 = a0_I64 = IRTemp_INVALID;
         b3_I64 = b2_I64 = b1_I64 = b0_I64 = IRTemp_INVALID;
         DIP("xvcpsgnsp v%d,v%d v%d\n",XT, XA, XB);
         breakV128to4x64U( getVSReg( XA ), &a3_I64, &a2_I64, &a1_I64, &a0_I64 );
         breakV128to4x64U( getVSReg( XB ), &b3_I64, &b2_I64, &b1_I64, &b0_I64 );

         assign( resHi,
                 binop( Iop_32HLto64,
                        binop( Iop_Or32,
                               binop( Iop_And32,
                                      unop(Iop_64to32, mkexpr( a3_I64 ) ),
                                      mkU32( SIGN_BIT32 ) ),
                               binop( Iop_And32,
                                      unop(Iop_64to32, mkexpr( b3_I64 ) ),
                                      mkU32( SIGN_MASK32) ) ),

                        binop( Iop_Or32,
                               binop( Iop_And32,
                                      unop(Iop_64to32, mkexpr( a2_I64 ) ),
                                      mkU32( SIGN_BIT32 ) ),
                               binop( Iop_And32,
                                      unop(Iop_64to32, mkexpr( b2_I64 ) ),
                                      mkU32( SIGN_MASK32 ) ) ) ) );
         assign( resLo,
                 binop( Iop_32HLto64,
                        binop( Iop_Or32,
                               binop( Iop_And32,
                                      unop(Iop_64to32, mkexpr( a1_I64 ) ),
                                      mkU32( SIGN_BIT32 ) ),
                               binop( Iop_And32,
                                      unop(Iop_64to32, mkexpr( b1_I64 ) ),
                                      mkU32( SIGN_MASK32 ) ) ),

                        binop( Iop_Or32,
                               binop( Iop_And32,
                                      unop(Iop_64to32, mkexpr( a0_I64 ) ),
                                      mkU32( SIGN_BIT32 ) ),
                               binop( Iop_And32,
                                      unop(Iop_64to32, mkexpr( b0_I64 ) ),
                                      mkU32( SIGN_MASK32 ) ) ) ) );
         putVSReg( XT, binop( Iop_64HLtoV128, mkexpr( resHi ), mkexpr( resLo ) ) );
         break;
      }
      case 0x3B2: // xvabsdp (VSX Vector Absolute Value Double-Precision)
      case 0x3D2: // xvnabsdp VSX Vector Negative Absolute Value Double-Precision)
      {
         IRTemp frB = newTemp(Ity_F64);
         IRTemp frB2 = newTemp(Ity_F64);
         IRTemp abs_resultHi = newTemp(Ity_F64);
         IRTemp abs_resultLo = newTemp(Ity_F64);
         Bool make_negative = (opc2 == 0x3D2) ? True : False;
         assign(frB,  unop(Iop_ReinterpI64asF64, unop(Iop_V128HIto64, getVSReg( XB ))));
         assign(frB2, unop(Iop_ReinterpI64asF64, unop(Iop_V128to64, getVSReg(XB))));

         DIP("xv%sabsdp v%d,v%d\n", make_negative ? "n" : "", XT, XB);
         if (make_negative) {
            assign(abs_resultHi, unop( Iop_NegF64, unop( Iop_AbsF64, mkexpr( frB ) ) ) );
            assign(abs_resultLo, unop( Iop_NegF64, unop( Iop_AbsF64, mkexpr( frB2 ) ) ) );

         } else {
            assign(abs_resultHi, unop( Iop_AbsF64, mkexpr( frB ) ) );
            assign(abs_resultLo, unop( Iop_AbsF64, mkexpr( frB2 ) ) );
         }
         putVSReg( XT, binop( Iop_64HLtoV128,
                              unop( Iop_ReinterpF64asI64, mkexpr( abs_resultHi ) ),
                              unop( Iop_ReinterpF64asI64, mkexpr( abs_resultLo ) ) ) );
         break;
      }
      case 0x332: // xvabssp (VSX Vector Absolute Value Single-Precision)
      case 0x352: // xvnabssp (VSX Vector Negative Absolute Value Single-Precision)
      {
         /*
          * The Iop_AbsF32 IRop is not implemented for ppc64 since, up until introduction
          * of xvabssp, there has not been an abs(sp) type of instruction.  But since emulation
          * of this function is so easy using shifts, I choose to emulate this instruction that
          * way versus a native instruction method of implementation.
          */
         Bool make_negative = (opc2 == 0x352) ? True : False;
         IRTemp shiftVector = newTemp(Ity_V128);
         IRTemp absVal_vector = newTemp(Ity_V128);
         assign( shiftVector,
                 binop( Iop_64HLtoV128,
                        binop( Iop_32HLto64, mkU32( 1 ), mkU32( 1 ) ),
                        binop( Iop_32HLto64, mkU32( 1 ), mkU32( 1 ) ) ) );
         assign( absVal_vector,
                   binop( Iop_Shr32x4,
                          binop( Iop_Shl32x4,
                                 getVSReg( XB ),
                                 mkexpr( shiftVector ) ),
                          mkexpr( shiftVector ) ) );
         if (make_negative) {
            IRTemp signBit_vector = newTemp(Ity_V128);
            assign( signBit_vector,
                    binop( Iop_64HLtoV128,
                           binop( Iop_32HLto64,
                                  mkU32( 0x80000000 ),
                                  mkU32( 0x80000000 ) ),
                           binop( Iop_32HLto64,
                                  mkU32( 0x80000000 ),
                                  mkU32( 0x80000000 ) ) ) );
            putVSReg( XT,
                      binop( Iop_OrV128,
                             mkexpr( absVal_vector ),
                             mkexpr( signBit_vector ) ) );
         } else {
            putVSReg( XT, mkexpr( absVal_vector ) );
         }
         break;
      }
      case 0x372: // xvnegsp (VSX Vector Negate Single-Precision)
      {
         IRTemp B0 = newTemp(Ity_I32);
         IRTemp B1 = newTemp(Ity_I32);
         IRTemp B2 = newTemp(Ity_I32);
         IRTemp B3 = newTemp(Ity_I32);

         DIP("xvnegsp v%d,v%d\n",  XT, XB);

	 /* Don't support NegF32, so just XOR the sign bit in the int value */
         assign(B0, unop( Iop_64HIto32,
			  unop( Iop_V128HIto64, getVSReg( XB ) ) ) );
         assign(B1, unop( Iop_64to32,
			  unop( Iop_V128HIto64, getVSReg( XB ) ) ) );
         assign(B2, unop( Iop_64HIto32,
			  unop( Iop_V128to64, getVSReg( XB ) ) ) );
         assign(B3, unop( Iop_64to32,
			  unop( Iop_V128to64, getVSReg( XB ) ) ) );

         putVSReg( XT,
		   binop( Iop_64HLtoV128,
			  binop( Iop_32HLto64,
				 binop( Iop_Xor32, mkexpr( B0 ), mkU32( 0x80000000 ) ),
				 binop( Iop_Xor32, mkexpr( B1 ), mkU32( 0x80000000 ) ) ),
			  binop( Iop_32HLto64,
				 binop( Iop_Xor32, mkexpr( B2 ), mkU32( 0x80000000 ) ),
				 binop( Iop_Xor32, mkexpr( B3 ), mkU32( 0x80000000 ) ) ) ) );
         break;
      }
      case 0x3F2: // xvnegdp (VSX Vector Negate Double-Precision)
      {
         IRTemp frB = newTemp(Ity_F64);
         IRTemp frB2 = newTemp(Ity_F64);
         assign(frB,  unop(Iop_ReinterpI64asF64, unop(Iop_V128HIto64, getVSReg( XB ))));
         assign(frB2, unop(Iop_ReinterpI64asF64, unop(Iop_V128to64, getVSReg(XB))));
         DIP("xvnegdp v%d,v%d\n",  XT, XB);
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          unop( Iop_ReinterpF64asI64,
                                unop( Iop_NegF64, mkexpr( frB ) ) ),
                          unop( Iop_ReinterpF64asI64,
                                unop( Iop_NegF64, mkexpr( frB2 ) ) ) ) );
         break;
      }
      case 0x192: // xvrdpi  (VSX Vector Round to Double-Precision Integer using round toward Nearest Away)
      case 0x1D6: // xvrdpic (VSX Vector Round to Double-Precision Integer using Current rounding mode)
      case 0x1F2: // xvrdpim (VSX Vector Round to Double-Precision Integer using round toward -Infinity)
      case 0x1D2: // xvrdpip (VSX Vector Round to Double-Precision Integer using round toward +Infinity)
      case 0x1B2: // xvrdpiz (VSX Vector Round to Double-Precision Integer using round toward Zero)
      {
         IRTemp frBHi_I64 = newTemp(Ity_I64);
         IRTemp frBLo_I64 = newTemp(Ity_I64);
         IRExpr * frD_fp_roundHi = NULL;
         IRExpr * frD_fp_roundLo = NULL;

         assign( frBHi_I64, unop( Iop_V128HIto64, getVSReg( XB ) ) );
         frD_fp_roundHi = _do_vsx_fp_roundToInt(frBHi_I64, opc2);
         assign( frBLo_I64, unop( Iop_V128to64, getVSReg( XB ) ) );
         frD_fp_roundLo = _do_vsx_fp_roundToInt(frBLo_I64, opc2);

         DIP("xvrdpi%s v%d,v%d\n", _get_vsx_rdpi_suffix(opc2), XT, XB);
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          unop( Iop_ReinterpF64asI64, frD_fp_roundHi ),
                          unop( Iop_ReinterpF64asI64, frD_fp_roundLo ) ) );
         break;
      }
      case 0x112: // xvrspi  (VSX Vector Round to Single-Precision Integer using round toward Nearest Away)
      case 0x156: // xvrspic (VSX Vector Round to SinglePrecision Integer using Current rounding mode)
      case 0x172: // xvrspim (VSX Vector Round to SinglePrecision Integer using round toward -Infinity)
      case 0x152: // xvrspip (VSX Vector Round to SinglePrecision Integer using round toward +Infinity)
      case 0x132: // xvrspiz (VSX Vector Round to SinglePrecision Integer using round toward Zero)
      {
         const HChar * insn_suffix = NULL;
         IROp op;
         if (opc2 != 0x156) {
            // Use pre-defined IRop's for vrfi{m|n|p|z}
            switch (opc2) {
               case 0x112:
                  insn_suffix = "";
                  op = Iop_RoundF32x4_RN;
                  break;
               case 0x172:
                  insn_suffix = "m";
                  op = Iop_RoundF32x4_RM;
                  break;
               case 0x152:
                  insn_suffix = "p";
                  op = Iop_RoundF32x4_RP;
                  break;
               case 0x132:
                  insn_suffix = "z";
                  op = Iop_RoundF32x4_RZ;
                  break;

               default:
                  vex_printf("Unrecognized opcode %x\n", opc2);
                  vpanic("dis_vxv_misc(ppc)(vrspi<x>)(opc2)\n");
            }
            DIP("xvrspi%s v%d,v%d\n", insn_suffix, XT, XB);
            putVSReg( XT, unop( op, getVSReg(XB) ) );
         } else {
            // Handle xvrspic.  Unfortunately there is no corresponding "vfric" instruction.
            IRExpr * frD_fp_roundb3, * frD_fp_roundb2, * frD_fp_roundb1, * frD_fp_roundb0;
            IRTemp b3_F64, b2_F64, b1_F64, b0_F64;
            IRTemp b3_I64 = newTemp(Ity_I64);
            IRTemp b2_I64 = newTemp(Ity_I64);
            IRTemp b1_I64 = newTemp(Ity_I64);
            IRTemp b0_I64 = newTemp(Ity_I64);

            b3_F64 = b2_F64 = b1_F64 = b0_F64 = IRTemp_INVALID;
            frD_fp_roundb3 = frD_fp_roundb2 = frD_fp_roundb1 = frD_fp_roundb0 = NULL;
            breakV128to4xF64( getVSReg(XB), &b3_F64, &b2_F64, &b1_F64, &b0_F64);
            assign(b3_I64, unop(Iop_ReinterpF64asI64, mkexpr(b3_F64)));
            assign(b2_I64, unop(Iop_ReinterpF64asI64, mkexpr(b2_F64)));
            assign(b1_I64, unop(Iop_ReinterpF64asI64, mkexpr(b1_F64)));
            assign(b0_I64, unop(Iop_ReinterpF64asI64, mkexpr(b0_F64)));
            frD_fp_roundb3 = unop(Iop_TruncF64asF32,
                                  _do_vsx_fp_roundToInt(b3_I64, opc2));
            frD_fp_roundb2 = unop(Iop_TruncF64asF32,
                                  _do_vsx_fp_roundToInt(b2_I64, opc2));
            frD_fp_roundb1 = unop(Iop_TruncF64asF32,
                                  _do_vsx_fp_roundToInt(b1_I64, opc2));
            frD_fp_roundb0 = unop(Iop_TruncF64asF32,
                                  _do_vsx_fp_roundToInt(b0_I64, opc2));
            DIP("xvrspic v%d,v%d\n", XT, XB);
            putVSReg( XT,
                      binop( Iop_64HLtoV128,
                             binop( Iop_32HLto64,
                                    unop( Iop_ReinterpF32asI32, frD_fp_roundb3 ),
                                    unop( Iop_ReinterpF32asI32, frD_fp_roundb2 ) ),
                             binop( Iop_32HLto64,
                                    unop( Iop_ReinterpF32asI32, frD_fp_roundb1 ),
                                    unop( Iop_ReinterpF32asI32, frD_fp_roundb0 ) ) ) );
         }
         break;
      }

      default:
         vex_printf( "dis_vxv_misc(ppc)(opc2)\n" );
         return False;
   }
   return True;
}


/*
 * VSX Scalar Floating Point Arithmetic Instructions
 */
static Bool
dis_vxs_arith ( UInt prefix, UInt theInstr, UInt opc2 )
{
   /* XX3-Form */
   UChar opc1 = ifieldOPC( theInstr );
   UChar XT = ifieldRegXT( theInstr );
   UChar XA = ifieldRegXA( theInstr );
   UChar XB = ifieldRegXB( theInstr );
   IRExpr* rm = get_IR_roundingmode();
   IRTemp frA = newTemp(Ity_F64);
   IRTemp frB = newTemp(Ity_F64);

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   if (opc1 != 0x3C) {
      vex_printf( "dis_vxs_arith(ppc)(instr)\n" );
      return False;
   }

   assign(frA, unop(Iop_ReinterpI64asF64, unop(Iop_V128HIto64, getVSReg( XA ))));
   assign(frB, unop(Iop_ReinterpI64asF64, unop(Iop_V128HIto64, getVSReg( XB ))));

   /* For all the VSX sclar arithmetic instructions, the contents of doubleword element 1
    * of VSX[XT] are undefined after the operation; therefore, we can simply set
    * element to zero where it makes sense to do so.
    */
   switch (opc2) {
      case 0x000: // xsaddsp  (VSX Scalar Add Single-Precision)
         DIP("xsaddsp v%d,v%d,v%d\n", XT, XA, XB);
         putVSReg( XT, binop( Iop_64HLtoV128,
                              unop( Iop_ReinterpF64asI64,
                                    binop( Iop_RoundF64toF32, rm,
                                           triop( Iop_AddF64, rm,
                                                  mkexpr( frA ),
                                                  mkexpr( frB ) ) ) ),
                              mkU64( 0 ) ) );
         break;
      case 0x020: // xssubsp  (VSX Scalar Subtract Single-Precision)
         DIP("xssubsp v%d,v%d,v%d\n", XT, XA, XB);
         putVSReg( XT, binop( Iop_64HLtoV128,
                              unop( Iop_ReinterpF64asI64,
                                    binop( Iop_RoundF64toF32, rm,
                                           triop( Iop_SubF64, rm,
                                                  mkexpr( frA ),
                                                  mkexpr( frB ) ) ) ),
                              mkU64( 0 ) ) );
         break;
      case 0x080: // xsadddp (VSX scalar add double-precision)
         DIP("xsadddp v%d,v%d,v%d\n", XT, XA, XB);
         putVSReg( XT, binop( Iop_64HLtoV128, unop( Iop_ReinterpF64asI64,
                                                    triop( Iop_AddF64, rm,
                                                           mkexpr( frA ),
                                                           mkexpr( frB ) ) ),
                              mkU64( 0 ) ) );
         break;
      case 0x060: // xsdivsp (VSX scalar divide single-precision)
         DIP("xsdivsp v%d,v%d,v%d\n", XT, XA, XB);
         putVSReg( XT, binop( Iop_64HLtoV128,
                              unop( Iop_ReinterpF64asI64,
                                    binop( Iop_RoundF64toF32, rm,
                                           triop( Iop_DivF64, rm,
                                                  mkexpr( frA ),
                                                  mkexpr( frB ) ) ) ),
                               mkU64( 0 ) ) );
         break;
      case 0x0E0: // xsdivdp (VSX scalar divide double-precision)
         DIP("xsdivdp v%d,v%d,v%d\n", XT, XA, XB);
         putVSReg( XT, binop( Iop_64HLtoV128, unop( Iop_ReinterpF64asI64,
                                                    triop( Iop_DivF64, rm,
                                                           mkexpr( frA ),
                                                           mkexpr( frB ) ) ),
                              mkU64( 0 ) ) );
         break;
      case 0x004: case 0x024: /* xsmaddasp, xsmaddmsp (VSX scalar multiply-add
                               * single-precision)
                               */
      {
         IRTemp frT = newTemp(Ity_F64);
         Bool mdp = opc2 == 0x024;
         DIP("xsmadd%ssp v%d,v%d,v%d\n", mdp ? "m" : "a", XT, XA, XB);
         assign( frT, unop( Iop_ReinterpI64asF64, unop( Iop_V128HIto64,
                                                        getVSReg( XT ) ) ) );
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          unop( Iop_ReinterpF64asI64,
                                binop( Iop_RoundF64toF32, rm,
                                       qop( Iop_MAddF64, rm,
                                            mkexpr( frA ),
                                            mkexpr( mdp ? frT : frB ),
                                            mkexpr( mdp ? frB : frT ) ) ) ),
                          mkU64( 0 ) ) );
         break;
      }
      case 0x084: case 0x0A4: // xsmaddadp, xsmaddmdp (VSX scalar multiply-add double-precision)
      {
         IRTemp frT = newTemp(Ity_F64);
         Bool mdp = opc2 == 0x0A4;
         DIP("xsmadd%sdp v%d,v%d,v%d\n", mdp ? "m" : "a", XT, XA, XB);
         assign( frT, unop( Iop_ReinterpI64asF64, unop( Iop_V128HIto64,
                                                        getVSReg( XT ) ) ) );
         putVSReg( XT, binop( Iop_64HLtoV128, unop( Iop_ReinterpF64asI64,
                                                    qop( Iop_MAddF64, rm,
                                                         mkexpr( frA ),
                                                         mkexpr( mdp ? frT : frB ),
                                                         mkexpr( mdp ? frB : frT ) ) ),
                              mkU64( 0 ) ) );
         break;
      }
      case 0x044: case 0x064: /* xsmsubasp, xsmsubmsp (VSX scalar
                               * multiply-subtract single-precision)
			       */
      {
         IRTemp frT = newTemp(Ity_F64);
         Bool mdp = opc2 == 0x064;
         DIP("xsmsub%ssp v%d,v%d,v%d\n", mdp ? "m" : "a", XT, XA, XB);
         assign( frT, unop( Iop_ReinterpI64asF64, unop( Iop_V128HIto64,
                                                        getVSReg( XT ) ) ) );
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          unop( Iop_ReinterpF64asI64,
                                binop( Iop_RoundF64toF32, rm,
                                       qop( Iop_MSubF64, rm,
                                            mkexpr( frA ),
                                            mkexpr( mdp ? frT : frB ),
                                            mkexpr( mdp ? frB : frT ) ) ) ),
                          mkU64( 0 ) ) );
         break;
      }
      case 0x0C4: case 0x0E4: // xsmsubadp, xsmsubmdp (VSX scalar multiply-subtract double-precision)
      {
         IRTemp frT = newTemp(Ity_F64);
         Bool mdp = opc2 == 0x0E4;
         DIP("xsmsub%sdp v%d,v%d,v%d\n", mdp ? "m" : "a", XT, XA, XB);
         assign( frT, unop( Iop_ReinterpI64asF64, unop( Iop_V128HIto64,
                                                        getVSReg( XT ) ) ) );
         putVSReg( XT, binop( Iop_64HLtoV128, unop( Iop_ReinterpF64asI64,
                                                    qop( Iop_MSubF64, rm,
                                                         mkexpr( frA ),
                                                         mkexpr( mdp ? frT : frB ),
                                                         mkexpr( mdp ? frB : frT ) ) ),
                              mkU64( 0 ) ) );
         break;
      }
      case 0x284: case 0x2A4: // xsnmaddadp, xsnmaddmdp (VSX scalar multiply-add double-precision)
      {
         /* TODO: mpj -- Naturally, I expected to be able to leverage the implementation
          * of fnmadd and use pretty much the same code. However, that code has a bug in the
          * way it blindly negates the signbit, even if the floating point result is a NaN.
          * So, the TODO is to fix fnmadd (which I'll do in a different patch).
          * FIXED 7/1/2012: carll fnmadd and fnmsubs fixed to not negate sign
          * bit for NaN result.
          */
         Bool mdp = opc2 == 0x2A4;
         IRTemp frT = newTemp(Ity_F64);
         IRTemp maddResult = newTemp(Ity_I64);

         DIP("xsnmadd%sdp v%d,v%d,v%d\n", mdp ? "m" : "a", XT, XA, XB);
         assign( frT, unop( Iop_ReinterpI64asF64, unop( Iop_V128HIto64,
                                                        getVSReg( XT ) ) ) );
         assign( maddResult, unop( Iop_ReinterpF64asI64, qop( Iop_MAddF64, rm,
                                                              mkexpr( frA ),
                                                              mkexpr( mdp ? frT : frB ),
                                                              mkexpr( mdp ? frB : frT ) ) ) );

         putVSReg( XT, binop( Iop_64HLtoV128, mkexpr( getNegatedResult(maddResult) ),
                              mkU64( 0 ) ) );
         break;
      }
      case 0x204: case 0x224: /* xsnmaddasp, xsnmaddmsp (VSX scalar
                               * multiply-add single-precision)
                               */
      {
         Bool mdp = opc2 == 0x224;
         IRTemp frT = newTemp(Ity_F64);
         IRTemp maddResult = newTemp(Ity_I64);

         DIP("xsnmadd%ssp v%d,v%d,v%d\n", mdp ? "m" : "a", XT, XA, XB);
         assign( frT, unop( Iop_ReinterpI64asF64, unop( Iop_V128HIto64,
                                                        getVSReg( XT ) ) ) );
         assign( maddResult,
                 unop( Iop_ReinterpF64asI64,
                       binop( Iop_RoundF64toF32, rm,
                              qop( Iop_MAddF64, rm,
                                   mkexpr( frA ),
                                   mkexpr( mdp ? frT : frB ),
                                   mkexpr( mdp ? frB : frT ) ) ) ) );

         putVSReg( XT, binop( Iop_64HLtoV128,
                              mkexpr( getNegatedResult(maddResult) ),
                              mkU64( 0 ) ) );
         break;
      }
      case 0x244: case 0x264: /* xsnmsubasp, xsnmsubmsp (VSX Scalar Negative
                               * Multiply-Subtract Single-Precision)
                               */
      {
         IRTemp frT = newTemp(Ity_F64);
         Bool mdp = opc2 == 0x264;
         IRTemp msubResult = newTemp(Ity_I64);

         DIP("xsnmsub%ssp v%d,v%d,v%d\n", mdp ? "m" : "a", XT, XA, XB);
         assign( frT, unop( Iop_ReinterpI64asF64, unop( Iop_V128HIto64,
                                                        getVSReg( XT ) ) ) );
         assign( msubResult,
                 unop( Iop_ReinterpF64asI64,
                       binop( Iop_RoundF64toF32, rm,
                              qop( Iop_MSubF64, rm,
                                   mkexpr( frA ),
                                   mkexpr( mdp ? frT : frB ),
                                   mkexpr( mdp ? frB : frT ) ) ) ) );

         putVSReg( XT, binop( Iop_64HLtoV128,
                              mkexpr( getNegatedResult(msubResult) ),
                              mkU64( 0 ) ) );

         break;
      }

      case 0x2C4: case 0x2E4: // xsnmsubadp, xsnmsubmdp (VSX Scalar Negative Multiply-Subtract Double-Precision)
      {
         IRTemp frT = newTemp(Ity_F64);
         Bool mdp = opc2 == 0x2E4;
         IRTemp msubResult = newTemp(Ity_I64);

         DIP("xsnmsub%sdp v%d,v%d,v%d\n", mdp ? "m" : "a", XT, XA, XB);
         assign( frT, unop( Iop_ReinterpI64asF64, unop( Iop_V128HIto64,
                                                        getVSReg( XT ) ) ) );
         assign(msubResult, unop( Iop_ReinterpF64asI64,
                                      qop( Iop_MSubF64,
                                           rm,
                                           mkexpr( frA ),
                                           mkexpr( mdp ? frT : frB ),
                                           mkexpr( mdp ? frB : frT ) ) ));

         putVSReg( XT, binop( Iop_64HLtoV128, mkexpr( getNegatedResult(msubResult) ), mkU64( 0 ) ) );

         break;
      }

      case 0x040: // xsmulsp (VSX Scalar Multiply Single-Precision)
         DIP("xsmulsp v%d,v%d,v%d\n", XT, XA, XB);
         putVSReg( XT, binop( Iop_64HLtoV128,
                              unop( Iop_ReinterpF64asI64,
                                    binop( Iop_RoundF64toF32, rm,
                                           triop( Iop_MulF64, rm,
                                                   mkexpr( frA ),
                                                   mkexpr( frB ) ) ) ),
                              mkU64( 0 ) ) );
         break;

      case 0x0C0: // xsmuldp (VSX Scalar Multiply Double-Precision)
         DIP("xsmuldp v%d,v%d,v%d\n", XT, XA, XB);
         putVSReg( XT, binop( Iop_64HLtoV128, unop( Iop_ReinterpF64asI64,
                                                    triop( Iop_MulF64, rm,
                                                           mkexpr( frA ),
                                                           mkexpr( frB ) ) ),
                              mkU64( 0 ) ) );
         break;
      case 0x0A0: // xssubdp (VSX Scalar Subtract Double-Precision)
         DIP("xssubdp v%d,v%d,v%d\n", XT, XA, XB);
         putVSReg( XT, binop( Iop_64HLtoV128, unop( Iop_ReinterpF64asI64,
                                                    triop( Iop_SubF64, rm,
                                                           mkexpr( frA ),
                                                           mkexpr( frB ) ) ),
                              mkU64( 0 ) ) );
         break;

      case 0x016: // xssqrtsp (VSX Scalar Square Root Single-Precision)
         DIP("xssqrtsp v%d,v%d\n", XT, XB);
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          unop( Iop_ReinterpF64asI64,
                                binop( Iop_RoundF64toF32, rm,
                                       binop( Iop_SqrtF64, rm,
                                              mkexpr( frB ) ) ) ),
                          mkU64( 0 ) ) );
         break;

      case 0x096: // xssqrtdp (VSX Scalar Square Root Double-Precision)
         DIP("xssqrtdp v%d,v%d\n", XT, XB);
         putVSReg( XT,  binop( Iop_64HLtoV128, unop( Iop_ReinterpF64asI64,
                                                     binop( Iop_SqrtF64, rm,
                                                            mkexpr( frB ) ) ),
                               mkU64( 0 ) ) );
         break;

      case 0x0F4: // xstdivdp (VSX Scalar Test for software Divide Double-Precision)
      {
         UChar crfD     = toUChar( IFIELD( theInstr, 23, 3 ) );
         IRTemp frA_I64 = newTemp(Ity_I64);
         IRTemp frB_I64 = newTemp(Ity_I64);
         DIP("xstdivdp crf%d,v%d,v%d\n", crfD, XA, XB);
         assign( frA_I64, unop( Iop_ReinterpF64asI64, mkexpr( frA ) ) );
         assign( frB_I64, unop( Iop_ReinterpF64asI64, mkexpr( frB ) ) );
         putGST_field( PPC_GST_CR, do_fp_tdiv(frA_I64, frB_I64), crfD );
         break;
      }
      case 0x0D4: // xstsqrtdp (VSX Vector Test for software Square Root Double-Precision)
      {
         IRTemp frB_I64 = newTemp(Ity_I64);
         UChar crfD     = toUChar( IFIELD( theInstr, 23, 3 ) );
         IRTemp flags = newTemp(Ity_I32);
         IRTemp  fe_flag, fg_flag;
         fe_flag = fg_flag = IRTemp_INVALID;
         DIP("xstsqrtdp v%d,v%d\n", XT, XB);
         assign( frB_I64, unop(Iop_V128HIto64, getVSReg( XB )) );
         do_fp_tsqrt(frB_I64, False /*not single precision*/, &fe_flag, &fg_flag);
         /* The CR field consists of fl_flag || fg_flag || fe_flag || 0b0
          * where fl_flag == 1 on ppc64.
          */
         assign( flags,
                 binop( Iop_Or32,
                        binop( Iop_Or32, mkU32( 8 ), // fl_flag
                               binop( Iop_Shl32, mkexpr(fg_flag), mkU8( 2 ) ) ),
                        binop( Iop_Shl32, mkexpr(fe_flag), mkU8( 1 ) ) ) );
         putGST_field( PPC_GST_CR, mkexpr(flags), crfD );
         break;
      }

      default:
         vex_printf( "dis_vxs_arith(ppc)(opc2)\n" );
         return False;
   }

   return True;
}


/*
 * VSX Floating Point Compare Instructions
 */
static Bool
dis_vx_cmp( UInt prefix, UInt theInstr, UInt opc2 )
{
   /* XX3-Form and XX2-Form */
   UChar opc1 = ifieldOPC( theInstr );
   UChar crfD     = toUChar( IFIELD( theInstr, 23, 3 ) );
   IRTemp ccPPC32;
   UChar XA       = ifieldRegXA ( theInstr );
   UChar XB       = ifieldRegXB ( theInstr );
   IRTemp frA     = newTemp(Ity_F64);
   IRTemp frB     = newTemp(Ity_F64);

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   if (opc1 != 0x3C) {
      vex_printf( "dis_vx_cmp(ppc)(instr)\n" );
      return False;
   }

   assign(frA, unop(Iop_ReinterpI64asF64, unop(Iop_V128HIto64, getVSReg( XA ))));
   assign(frB, unop(Iop_ReinterpI64asF64, unop(Iop_V128HIto64, getVSReg( XB ))));
   switch (opc2) {
      case 0x08C: case 0x0AC: // xscmpudp, xscmpodp
         /* Note: Differences between xscmpudp and xscmpodp are only in
          * exception flag settings, which aren't supported anyway. */
         DIP("xscmp%sdp crf%d,fr%u,fr%u\n", opc2 == 0x08c ? "u" : "o",
                                           crfD, XA, XB);
         ccPPC32 = get_fp_cmp_CR_val( binop(Iop_CmpF64, mkexpr(frA), mkexpr(frB)));
         putGST_field( PPC_GST_CR, mkexpr(ccPPC32), crfD );
         putFPCC( mkexpr( ccPPC32 ) );
         break;

      default:
         vex_printf( "dis_vx_cmp(ppc)(opc2)\n" );
         return False;
   }
   return True;
}

static void
do_vvec_fp_cmp ( IRTemp vA, IRTemp vB, UChar XT, UChar flag_rC,
                 ppc_cmp_t cmp_type )
{
   IRTemp frA_hi     = newTemp(Ity_F64);
   IRTemp frB_hi     = newTemp(Ity_F64);
   IRTemp frA_lo     = newTemp(Ity_F64);
   IRTemp frB_lo     = newTemp(Ity_F64);
   IRTemp ccPPC32    = newTemp(Ity_I32);
   IRTemp ccIR_hi;
   IRTemp ccIR_lo;

   IRTemp hiResult = newTemp(Ity_I64);
   IRTemp loResult = newTemp(Ity_I64);
   IRTemp hiEQlo = newTemp(Ity_I1);
   IRTemp all_elem_true = newTemp(Ity_I32);
   IRTemp all_elem_false = newTemp(Ity_I32);

   assign(frA_hi, unop(Iop_ReinterpI64asF64, unop(Iop_V128HIto64, mkexpr( vA ))));
   assign(frB_hi, unop(Iop_ReinterpI64asF64, unop(Iop_V128HIto64, mkexpr( vB ))));
   assign(frA_lo, unop(Iop_ReinterpI64asF64, unop(Iop_V128to64, mkexpr( vA ))));
   assign(frB_lo, unop(Iop_ReinterpI64asF64, unop(Iop_V128to64, mkexpr( vB ))));

   ccIR_hi = get_fp_cmp_CR_val( binop( Iop_CmpF64,
                                       mkexpr( frA_hi ),
                                       mkexpr( frB_hi ) ) );
   ccIR_lo = get_fp_cmp_CR_val( binop( Iop_CmpF64,
                                       mkexpr( frA_lo ),
                                       mkexpr( frB_lo ) ) );

   if (cmp_type != PPC_CMP_GE) {
      assign( hiResult,
              unop( Iop_1Sto64,
                    binop( Iop_CmpEQ32, mkexpr( ccIR_hi ), mkU32( cmp_type ) ) ) );
      assign( loResult,
              unop( Iop_1Sto64,
                    binop( Iop_CmpEQ32, mkexpr( ccIR_lo ), mkU32( cmp_type ) ) ) );
   } else {
      // For PPC_CMP_GE, one element compare may return "4" (for "greater than") and
      // the other element compare may return "2" (for "equal to").
      IRTemp lo_GE = newTemp(Ity_I1);
      IRTemp hi_GE = newTemp(Ity_I1);

      assign(hi_GE, mkOR1( binop( Iop_CmpEQ32, mkexpr( ccIR_hi ), mkU32( 2 ) ),
                           binop( Iop_CmpEQ32, mkexpr( ccIR_hi ), mkU32( 4 ) ) ) );
      assign( hiResult,unop( Iop_1Sto64, mkexpr( hi_GE ) ) );

      assign(lo_GE, mkOR1( binop( Iop_CmpEQ32, mkexpr( ccIR_lo ), mkU32( 2 ) ),
                           binop( Iop_CmpEQ32, mkexpr( ccIR_lo ), mkU32( 4 ) ) ) );
      assign( loResult, unop( Iop_1Sto64, mkexpr( lo_GE ) ) );
   }

   // The [hi/lo]Result will be all 1's or all 0's.  We just look at the lower word.
   assign( hiEQlo,
           binop( Iop_CmpEQ32,
                  unop( Iop_64to32, mkexpr( hiResult ) ),
                  unop( Iop_64to32, mkexpr( loResult ) ) ) );
   putVSReg( XT,
             binop( Iop_64HLtoV128, mkexpr( hiResult ), mkexpr( loResult ) ) );

   assign( all_elem_true,
           unop( Iop_1Uto32,
                 mkAND1( mkexpr( hiEQlo ),
                         binop( Iop_CmpEQ32,
                                mkU32( 0xffffffff ),
                                unop( Iop_64to32,
                                mkexpr( hiResult ) ) ) ) ) );

   assign( all_elem_false,
           unop( Iop_1Uto32,
                 mkAND1( mkexpr( hiEQlo ),
                         binop( Iop_CmpEQ32,
                                mkU32( 0 ),
                                unop( Iop_64to32,
                                mkexpr( hiResult ) ) ) ) ) );
   assign( ccPPC32,
           binop( Iop_Or32,
                  binop( Iop_Shl32, mkexpr( all_elem_false ), mkU8( 1 ) ),
                  binop( Iop_Shl32, mkexpr( all_elem_true ), mkU8( 3 ) ) ) );

   if (flag_rC) {
      putGST_field( PPC_GST_CR, mkexpr(ccPPC32), 6 );
   }
}

/*
 * VSX Vector Compare Instructions
 */
static Bool
dis_vvec_cmp( UInt prefix, UInt theInstr, UInt opc2 )
{
   /* XX3-Form */
   UChar opc1 = ifieldOPC( theInstr );
   UChar XT = ifieldRegXT ( theInstr );
   UChar XA = ifieldRegXA ( theInstr );
   UChar XB = ifieldRegXB ( theInstr );
   UChar flag_rC  = ifieldBIT10(theInstr);
   IRTemp vA = newTemp( Ity_V128 );
   IRTemp vB = newTemp( Ity_V128 );

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   if (opc1 != 0x3C) {
      vex_printf( "dis_vvec_cmp(ppc)(instr)\n" );
      return False;
   }

   assign( vA, getVSReg( XA ) );
   assign( vB, getVSReg( XB ) );

   switch (opc2) {
      case 0x18C:  // xvcmpeqdp[.] (VSX Vector Compare Equal To Double-Precision [ & Record ])
      {
         DIP("xvcmpeqdp%s crf%d,fr%u,fr%u\n", (flag_rC ? ".":""),
             XT, XA, XB);
         do_vvec_fp_cmp(vA, vB, XT, flag_rC, PPC_CMP_EQ);
         break;
      }

      case 0x1CC:  // xvcmpgedp[.] (VSX Vector Compare Greater Than or Equal To Double-Precision [ & Record ])
      {
         DIP("xvcmpgedp%s crf%d,fr%u,fr%u\n", (flag_rC ? ".":""),
             XT, XA, XB);
         do_vvec_fp_cmp(vA, vB, XT, flag_rC, PPC_CMP_GE);
         break;
      }

      case 0x1AC:  // xvcmpgtdp[.] (VSX Vector Compare Greater Than Double-Precision [ & Record ])
      {
         DIP("xvcmpgtdp%s crf%d,fr%u,fr%u\n", (flag_rC ? ".":""),
             XT, XA, XB);
         do_vvec_fp_cmp(vA, vB, XT, flag_rC, PPC_CMP_GT);
         break;
      }

      case 0x10C:  // xvcmpeqsp[.] (VSX Vector Compare Equal To Single-Precision [ & Record ])
      {
         IRTemp vD = newTemp(Ity_V128);

         DIP("xvcmpeqsp%s crf%d,fr%u,fr%u\n", (flag_rC ? ".":""),
             XT, XA, XB);
         assign( vD, binop(Iop_CmpEQ32Fx4, mkexpr(vA), mkexpr(vB)) );
         putVSReg( XT, mkexpr(vD) );
         if (flag_rC) {
            set_AV_CR6( mkexpr(vD), True );
         }
         break;
      }

      case 0x14C:  // xvcmpgesp[.] (VSX Vector Compare Greater Than or Equal To Single-Precision [ & Record ])
      {
         IRTemp vD = newTemp(Ity_V128);

         DIP("xvcmpgesp%s crf%d,fr%u,fr%u\n", (flag_rC ? ".":""),
             XT, XA, XB);
         assign( vD, binop(Iop_CmpGE32Fx4, mkexpr(vA), mkexpr(vB)) );
         putVSReg( XT, mkexpr(vD) );
         if (flag_rC) {
            set_AV_CR6( mkexpr(vD), True );
         }
         break;
      }

      case 0x12C:  //xvcmpgtsp[.] (VSX Vector Compare Greater Than Single-Precision [ & Record ])
      {
         IRTemp vD = newTemp(Ity_V128);

         DIP("xvcmpgtsp%s crf%d,fr%u,fr%u\n", (flag_rC ? ".":""),
             XT, XA, XB);
         assign( vD, binop(Iop_CmpGT32Fx4, mkexpr(vA), mkexpr(vB)) );
         putVSReg( XT, mkexpr(vD) );
         if (flag_rC) {
            set_AV_CR6( mkexpr(vD), True );
         }
         break;
      }

      default:
         vex_printf( "dis_vvec_cmp(ppc)(opc2)\n" );
         return False;
   }
   return True;
}
/*
 * Miscellaneous VSX Scalar Instructions
 */
static Bool
dis_load_vector_special( UInt prefix, UInt theInstr,
                         const VexAbiInfo* vbi, UInt opc2, int allow_isa_3_0 )
{
   UChar opc1 = ifieldOPC( theInstr );
   UChar XT = ifieldRegXT ( theInstr );
   UInt uim = IFIELD( theInstr, 11, 5 );    // inst[16:20]

   if (opc1 != 0x3C) {
      vex_printf( "dis_load_special(ppc)(instr)\n" );
      return False;
   }

   DIP("lxvkq v%u,%u\n", (UInt)XT, uim);

   switch( uim ) {
   case 0b00001: putVSReg( XT, binop( Iop_64HLtoV128,
                                      mkU64( 0x3FFF000000000000 ),
                                      mkU64( 0x0000000000000000 ) ) );
      break;
   case 0b00010: putVSReg( XT, binop( Iop_64HLtoV128,
                                      mkU64( 0x4000000000000000 ),
                                      mkU64( 0x0000000000000000 ) ) );
      break;
   case 0b00011: putVSReg( XT, binop( Iop_64HLtoV128,
                                      mkU64( 0x4000800000000000 ),
                                      mkU64( 0x0000000000000000 ) ) );
      break;
   case 0b00100: putVSReg( XT, binop( Iop_64HLtoV128,
                                      mkU64( 0x4001000000000000 ),
                                      mkU64( 0x0000000000000000 ) ) );
      break;
   case 0b00101: putVSReg( XT, binop( Iop_64HLtoV128,
                                      mkU64( 0x4001400000000000 ),
                                      mkU64( 0x0000000000000000 ) ) );
      break;
   case 0b00110: putVSReg( XT, binop( Iop_64HLtoV128,
                                      mkU64( 0x4001800000000000 ),
                                      mkU64( 0x0000000000000000 ) ) );
      break;
   case 0b00111: putVSReg( XT, binop( Iop_64HLtoV128,
                                      mkU64( 0x4001C00000000000 ),
                                      mkU64( 0x0000000000000000 ) ) );
      break;
   case 0b01000: putVSReg( XT, binop( Iop_64HLtoV128,
                                      mkU64( 0x7FFF000000000000 ),
                                      mkU64( 0x0000000000000000 ) ) );
      break;
   case 0b01001: putVSReg( XT, binop( Iop_64HLtoV128,
                                      mkU64( 0x7FFF800000000000 ),
                                      mkU64( 0x0000000000000000 ) ) );
      break;
   case 0b10000: putVSReg( XT, binop( Iop_64HLtoV128,
                                      mkU64( 0x8000000000000000 ),
                                      mkU64( 0x0000000000000000 ) ) );
      break;
   case 0b10001: putVSReg( XT, binop( Iop_64HLtoV128,
                                      mkU64( 0xBFFF000000000000 ),
                                      mkU64( 0x0000000000000000 ) ) );
      break;
   case 0b10010: putVSReg( XT, binop( Iop_64HLtoV128,
                                      mkU64( 0xC000000000000000 ),
                                      mkU64( 0x0000000000000000 ) ) );
      break;
   case 0b10011: putVSReg( XT, binop( Iop_64HLtoV128,
                                      mkU64( 0xC000800000000000 ),
                                      mkU64( 0x0000000000000000 ) ) );
      break;
   case 0b10100: putVSReg( XT, binop( Iop_64HLtoV128,
                                      mkU64( 0xC001000000000000 ),
                                      mkU64( 0x0000000000000000 ) ) );
      break;
   case 0b10101: putVSReg( XT, binop( Iop_64HLtoV128,
                                      mkU64( 0xC001400000000000 ),
                                      mkU64( 0x0000000000000000 ) ) );
      break;
   case 0b10110: putVSReg( XT, binop( Iop_64HLtoV128,
                                      mkU64( 0xC001800000000000 ),
                                      mkU64( 0x0000000000000000 ) ) );
      break;
   case 0b10111: putVSReg( XT, binop( Iop_64HLtoV128,
                                      mkU64( 0xC001C00000000000 ),
                                      mkU64( 0x0000000000000000 ) ) );
      break;
   case 0b11000: putVSReg( XT, binop( Iop_64HLtoV128,
                                      mkU64( 0xFFFF000000000000 ),
                                      mkU64( 0x0000000000000000 ) ) );
      break;
   default:  vex_printf( "dis_load_special(ppc)(lxvkq XT, UIM not valid)\n" );
      putVSReg( XT, binop( Iop_64HLtoV128,
                                      mkU64( 0x0000000000000000 ),
                                      mkU64( 0x0000000000000000 ) ) );
      return True;   /* print message, continue */
   }
   return True;
}

static Bool
dis_vxs_misc( UInt prefix, UInt theInstr, const VexAbiInfo* vbi, UInt opc2,
              int allow_isa_3_0 )
{
#define VG_PPC_SIGN_MASK 0x7fffffffffffffffULL
   /* XX3-Form and XX2-Form */
   UChar opc1 = ifieldOPC( theInstr );
   UChar XT = ifieldRegXT ( theInstr );
   UChar XA = ifieldRegXA ( theInstr );
   UChar XB = ifieldRegXB ( theInstr );
   IRTemp vA = newTemp( Ity_V128 );
   IRTemp vB = newTemp( Ity_V128 );

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   if (opc1 != 0x3C) {
      vex_printf( "dis_vxs_misc(ppc)(instr)\n" );
      return False;
   }

   assign( vA, getVSReg( XA ) );
   assign( vB, getVSReg( XB ) );

   /* For all the VSX move instructions, the contents of doubleword element 1
    * of VSX[XT] are undefined after the operation; therefore, we can simply
    * move the entire array element where it makes sense to do so.
    */
   if (( opc2 == 0x168 ) && ( IFIELD( theInstr, 19, 2 ) == 0 ) )
      {
         /* Special case of XX1-Form with immediate value
          *  xxspltib (VSX Vector Splat Immediate Byte)
          */
         UInt uim = IFIELD( theInstr, 11, 8 );
         UInt word_value = ( uim << 24 ) | ( uim << 16 ) | ( uim << 8 ) | uim;

         DIP("xxspltib v%u,%u\n", (UInt)XT, uim);
         putVSReg(XT, binop( Iop_64HLtoV128,
                             binop( Iop_32HLto64,
                                    mkU32( word_value ),
                                    mkU32( word_value ) ),
                             binop( Iop_32HLto64,
                                    mkU32( word_value ),
                                    mkU32( word_value ) ) ) );
         return True;
      }

   switch ( opc2 ) {
      case 0x0ec: // xscmpexpdp (VSX Scalar Compare Exponents Double-Precision)
      {
         /* Compare 64-bit data, 128-bit layout:
            src1[0:63] is double word, src1[64:127] is unused
            src2[0:63] is double word, src2[64:127] is unused
         */
         IRExpr *bit4, *bit5, *bit6, *bit7;
         UInt BF = IFIELD( theInstr, 23, 3 );
         IRTemp eq_lt_gt = newTemp( Ity_I32 );
         IRTemp CC = newTemp( Ity_I32 );
         IRTemp vA_hi = newTemp( Ity_I64 );
         IRTemp vB_hi = newTemp( Ity_I64 );
         IRExpr *mask = mkU64( 0x7FF0000000000000 );

         DIP("xscmpexpdp %u,v%u,v%u\n", BF, XA, XB);

         assign( vA_hi, unop( Iop_V128HIto64, mkexpr( vA ) ) );
         assign( vB_hi, unop( Iop_V128HIto64, mkexpr( vB ) ) );

         /* A exp < B exp */
         bit4 = binop( Iop_CmpLT64U,
                      binop( Iop_And64,
                             mkexpr( vA_hi ),
                             mask ),
                      binop( Iop_And64,
                             mkexpr( vB_hi ),
                             mask ) );
         /*  A exp > B exp */
         bit5 = binop( Iop_CmpLT64U,
                      binop( Iop_And64,
                             mkexpr( vB_hi ),
                             mask ),
                      binop( Iop_And64,
                             mkexpr( vA_hi ),
                             mask ) );
         /* test equal */
         bit6 = binop( Iop_CmpEQ64,
                      binop( Iop_And64,
                             mkexpr( vA_hi ),
                             mask ),
                      binop( Iop_And64,
                             mkexpr( vB_hi ),
                             mask ) );

         /* exp A or exp B is NaN */
         bit7 = mkOR1( is_NaN( Ity_I64, vA_hi ),
                       is_NaN( Ity_I64, vB_hi ) );

         assign( eq_lt_gt, binop( Iop_Or32,
                                  binop( Iop_Shl32,
                                         unop( Iop_1Uto32, bit4 ),
                                         mkU8( 3) ),
                                  binop( Iop_Or32,
                                         binop( Iop_Shl32,
                                                unop( Iop_1Uto32, bit5 ),
                                                mkU8( 2) ),
                                         binop( Iop_Shl32,
                                                unop( Iop_1Uto32, bit6 ),
                                                mkU8( 1 ) ) ) ) );
         assign(CC, binop( Iop_Or32,
                           binop( Iop_And32,
                                  mkexpr( eq_lt_gt ) ,
                                  unop( Iop_Not32, unop( Iop_1Sto32, bit7 ) ) ),
                           unop( Iop_1Uto32, bit7 ) ) );

         putGST_field( PPC_GST_CR, mkexpr( CC ), BF );
         putFPCC( mkexpr( CC ) );
         return True;
      }
      break;

      case 0x14A: // xxextractuw (VSX Vector Extract Unsigned Word)
      {
         UInt uim = IFIELD( theInstr, 16, 4 );

         DIP("xxextractuw v%u,v%u,%u\n", (UInt)XT, (UInt)XB, uim);

         putVSReg( XT,
                   binop( Iop_ShlV128,
                          binop( Iop_AndV128,
                                 binop( Iop_ShrV128,
                                        mkexpr( vB ),
                                        mkU8( ( 12 - uim ) * 8 ) ),
                                 binop(Iop_64HLtoV128,
                                       mkU64( 0 ),
                                       mkU64( 0xFFFFFFFF ) ) ),
                          mkU8( ( 32*2 ) ) ) );
         break;
      }
      case 0x16A: // xxinsertw (VSX Vector insert Word)
      {
         UInt uim = IFIELD( theInstr, 16, 4 );
         IRTemp vT = newTemp( Ity_V128 );
         IRTemp tmp = newTemp( Ity_V128 );

         DIP("xxinsertw v%u,v%u,%u\n", (UInt)XT, (UInt)XB, uim);

         assign( vT, getVSReg( XT ) );
         assign( tmp, binop( Iop_AndV128,
                             mkexpr( vT ),
                             unop( Iop_NotV128,
                                   binop( Iop_ShlV128,
                                          binop( Iop_64HLtoV128,
                                                 mkU64( 0x0 ),
                                                 mkU64( 0xFFFFFFFF) ),
                                          mkU8( ( 12 - uim ) * 8 ) ) ) ) );

         putVSReg( XT,
                   binop( Iop_OrV128,
                          binop( Iop_ShlV128,
                                 binop( Iop_AndV128,
                                        binop( Iop_ShrV128,
                                               mkexpr( vB ),
                                               mkU8( 32 * 2 ) ),
                                        binop( Iop_64HLtoV128,
                                               mkU64( 0 ),
                                               mkU64( 0xFFFFFFFF ) ) ),
                                 mkU8( ( 12 - uim ) * 8 ) ),
                          mkexpr( tmp ) ) );
         break;
      }

      case 0x2B2: // xsabsdp (VSX scalar absolute value double-precision
      {
         /* Move abs val of dw 0 of VSX[XB] to dw 0 of VSX[XT]. */
         IRTemp absVal = newTemp(Ity_V128);
         if (host_endness == VexEndnessLE) {
            IRTemp hi64 = newTemp(Ity_I64);
            IRTemp lo64 = newTemp(Ity_I64);
            assign( hi64, unop( Iop_V128HIto64, mkexpr(vB) ) );
            assign( lo64, unop( Iop_V128to64, mkexpr(vB) ) );
            assign( absVal, binop( Iop_64HLtoV128,
                                   binop( Iop_And64, mkexpr(hi64),
                                          mkU64(VG_PPC_SIGN_MASK) ),
                                   mkexpr(lo64) ) );
         } else {
            assign(absVal, binop(Iop_ShrV128,
                                 binop(Iop_ShlV128, mkexpr(vB),
                                       mkU8(1)), mkU8(1)));
         }
         DIP("xsabsdp v%u,v%u\n", XT, XB);
         putVSReg(XT, mkexpr(absVal));
         break;
      }

      case 0x2b6: // xsxexpdp (VSX Scalar Extract Exponent Double-Precision)
                  // xsxsigdp (VSX Scalar Extract Significand Doulbe-Precision)
                  // xsvhpdp  (VSX Scalar Convert Half-Precision format
                  //           to Double-Precision format)
                  // xscvdphp (VSX Scalar round & convert Double-precision
                  //           format to Half-precision format)
      {
         IRTemp rT = newTemp( Ity_I64 );
         UInt inst_select = IFIELD( theInstr, 16, 5);

         if (inst_select == 0) {
            DIP("xsxexpd %u,v%u\n", (UInt)XT, (UInt)XB);

            assign( rT, binop( Iop_Shr64,
                               binop( Iop_And64,
                                      unop( Iop_V128HIto64, mkexpr( vB ) ),
                                      mkU64( 0x7FF0000000000000 ) ),
                               mkU8 ( 52 ) ) );
         } else if (inst_select == 1) {
            IRExpr *normal;
            IRTemp tmp = newTemp(Ity_I64);

            DIP("xsxsigdp v%u,v%u\n",  (UInt)XT, (UInt)XB);

            assign( tmp, unop( Iop_V128HIto64, mkexpr( vB ) ) );

            /* Value is normal if it isn't infinite, zero or denormalized */
            normal = mkNOT1( mkOR1(
                                   mkOR1( is_NaN( Ity_I64, tmp ),
                                          is_Inf( Ity_I64, tmp ) ),
                                   mkOR1( is_Zero( Ity_I64, tmp ),
                                          is_Denorm( Ity_I64, tmp ) ) ) );

            assign( rT, binop( Iop_Or64,
                               binop( Iop_And64,
                                      mkexpr( tmp ),
                                      mkU64( 0xFFFFFFFFFFFFF ) ),
                               binop( Iop_Shl64,
                                      unop( Iop_1Uto64, normal),
                                      mkU8( 52 ) ) ) );
            putIReg( XT, mkexpr( rT ) );

        } else if (inst_select == 16) {
            IRTemp result = newTemp( Ity_V128 );
            IRTemp value = newTemp( Ity_I64 );
            /* Note: PPC only coverts the 16-bit value in the upper 64-bits
             * of the source V128 to a 64-bit value stored in the upper
             * 64-bits of the V128 result.  The contents of the lower 64-bits
             * is undefined.
             */

            DIP("xscvhpdp v%u, v%u\n", (UInt)XT, (UInt)XB);
            assign( result, unop( Iop_F16toF64x2, mkexpr( vB ) ) );

            putVSReg( XT, mkexpr( result ) );

            assign( value, unop( Iop_V128HIto64, mkexpr( result ) ) );
            generate_store_FPRF( Ity_I64, value, vbi );
            return True;

         } else if (inst_select == 17) {   // xscvdphp
            IRTemp value = newTemp( Ity_I32 );
            IRTemp result = newTemp( Ity_V128 );
            /* Note: PPC only coverts the 64-bit value in the upper 64-bits of
             * the V128 and stores the 16-bit result in the upper word of the
             * V128 result.  The contents of the lower 64-bits is undefined.
             */
            DIP("xscvdphp v%u, v%u\n", (UInt)XT, (UInt)XB);
            assign( result,  unop( Iop_F64toF16x2_DEP, mkexpr( vB ) ) );
            assign( value, unop( Iop_64to32, unop( Iop_V128HIto64,
                                                   mkexpr( result ) ) ) );
            putVSReg( XT, mkexpr( result ) );
            generate_store_FPRF( Ity_I16, value, vbi );
            return True;

         } else {
            vex_printf( "dis_vxv_scalar_extract_exp_sig invalid inst_select (ppc)(opc2)\n" );
            vex_printf("inst_select = %u\n", inst_select);
            return False;
         }
      }
      break;

      case 0x254:  // xststdcsp (VSX Scalar Test Data Class Single-Precision)
      case 0x2D4:  // xststdcdp (VSX Scalar Test Data Class Double-Precision)
      {
         /* These instructions only differ in that the single precision
            instruction, xststdcsp, has the additional constraint on the
            denormal test that the exponent be greater then zero and
            less then 0x381. */
         IRTemp vB_hi = newTemp( Ity_I64 );
         UInt BF = IFIELD( theInstr, 23, 3 );
         UInt DCMX_mask = IFIELD( theInstr, 16, 7 );
         IRTemp NaN = newTemp( Ity_I64 );
         IRTemp inf = newTemp( Ity_I64 );
         IRTemp zero = newTemp( Ity_I64 );
         IRTemp dnorm = newTemp( Ity_I64 );
         IRTemp pos = newTemp( Ity_I64 );
         IRTemp not_sp = newTemp( Ity_I64 );
         IRTemp DCM = newTemp( Ity_I64 );
         IRTemp CC = newTemp( Ity_I64 );
         IRTemp exponent = newTemp( Ity_I64 );
         IRTemp tmp = newTemp( Ity_I64 );

         assign( vB_hi, unop( Iop_V128HIto64, mkexpr( vB ) ) );

         assign( pos, unop( Iop_1Uto64,
                            binop( Iop_CmpEQ64,
                                   binop( Iop_Shr64,
                                          mkexpr( vB_hi ),
                                          mkU8( 63 ) ),
                                   mkU64( 0 ) ) ) );

         assign( NaN, unop( Iop_1Uto64, is_NaN( Ity_I64, vB_hi ) ) );
         assign( inf, unop( Iop_1Uto64, is_Inf( Ity_I64, vB_hi ) ) );
         assign( zero, unop( Iop_1Uto64, is_Zero( Ity_I64, vB_hi ) ) );

         if (opc2 == 0x254) {
            DIP("xststdcsp %u,v%u,%u\n", BF, (UInt)XB, DCMX_mask);

            /* The least significant bit of the CC is set to 1 if the double
               precision value is not representable as a single precision
               value. The spec says the bit is set if:
                  src != convert_SPtoDP(convert_DPtoSP(src))
            */
            assign( tmp,
                    unop( Iop_ReinterpF64asI64,
                                unop( Iop_F32toF64,
                                      unop( Iop_TruncF64asF32,
                                            unop( Iop_ReinterpI64asF64,
                                                  mkexpr( vB_hi ) ) ) ) ) );
            assign( not_sp, unop( Iop_1Uto64,
                                  mkNOT1( binop( Iop_CmpEQ64,
                                                 mkexpr( vB_hi ),
                                                 mkexpr( tmp ) ) ) ) );
            assign( exponent,
                    binop( Iop_Shr64,
                           binop( Iop_And64,
                                  mkexpr( vB_hi ),
                                  mkU64( 0x7ff0000000000000 ) ),
                           mkU8( 52 ) ) );
            assign( dnorm, unop( Iop_1Uto64,
                                 mkOR1( is_Denorm( Ity_I64, vB_hi ),
                                        mkAND1( binop( Iop_CmpLT64U,
                                                       mkexpr( exponent ),
                                                       mkU64( 0x381 ) ),
                                                binop( Iop_CmpNE64,
                                                       mkexpr( exponent ),
                                                       mkU64( 0x0 ) ) ) ) ) );

         } else {
            DIP("xststdcdp %u,v%u,%u\n", BF, (UInt)XB, DCMX_mask);
            assign( not_sp,  mkU64( 0 ) );
            assign( dnorm, unop( Iop_1Uto64, is_Denorm( Ity_I64, vB_hi ) ) );
         }

         assign( DCM, create_DCM( Ity_I64, NaN, inf, zero, dnorm, pos ) );
         assign( CC,
                 binop( Iop_Or64,
                        binop( Iop_And64,    /* vB sign bit */
                               binop( Iop_Shr64,
                                      mkexpr( vB_hi ),
                                      mkU8( 60 ) ),
                               mkU64( 0x8 ) ),
                        binop( Iop_Or64,
                               binop( Iop_Shl64,
                                      unop( Iop_1Uto64,
                                            binop( Iop_CmpNE64,
                                                   binop( Iop_And64,
                                                          mkexpr( DCM ),
                                                          mkU64( DCMX_mask ) ),
                                                   mkU64( 0 ) ) ),
                                      mkU8( 1 ) ),
                               mkexpr( not_sp ) ) ) );
         putGST_field( PPC_GST_CR, unop( Iop_64to32, mkexpr( CC ) ), BF );
         putFPCC( unop( Iop_64to32, mkexpr( CC ) ) );
      }
      return True;

      case 0x2C0: // xscpsgndp
      {
         /* Scalar copy sign double-precision */
         IRTemp vecA_signed = newTemp(Ity_I64);
         IRTemp vecB_unsigned = newTemp(Ity_I64);
         IRTemp vec_result = newTemp(Ity_V128);
         DIP("xscpsgndp v%d,v%d v%d\n", XT, XA, XB);
         assign( vecA_signed, binop( Iop_And64,
                                     unop( Iop_V128HIto64,
                                           mkexpr(vA)),
                                           mkU64(~VG_PPC_SIGN_MASK) ) );
         assign( vecB_unsigned, binop( Iop_And64,
                                       unop( Iop_V128HIto64,
                                             mkexpr(vB) ),
                                             mkU64(VG_PPC_SIGN_MASK) ) );
         assign( vec_result, binop( Iop_64HLtoV128,
                                    binop( Iop_Or64,
                                           mkexpr(vecA_signed),
                                           mkexpr(vecB_unsigned) ),
                                    mkU64(0x0ULL)));
         putVSReg(XT, mkexpr(vec_result));
         break;
      }
      case 0x2D2: // xsnabsdp
      {
         /* Scalar negative absolute value double-precision */
         IRTemp BHi_signed = newTemp(Ity_I64);
         DIP("xsnabsdp v%d,v%d\n", XT, XB);
         assign( BHi_signed, binop( Iop_Or64,
                                    unop( Iop_V128HIto64,
                                          mkexpr(vB) ),
                                          mkU64(~VG_PPC_SIGN_MASK) ) );
         putVSReg(XT, binop( Iop_64HLtoV128,
                             mkexpr(BHi_signed), mkU64(0x0ULL) ) );
         break;
      }
      case 0x2F2: // xsnegdp
      {
         /* Scalar negate double-precision */
         IRTemp BHi_signed = newTemp(Ity_I64);
         IRTemp BHi_unsigned = newTemp(Ity_I64);
         IRTemp BHi_negated = newTemp(Ity_I64);
         IRTemp BHi_negated_signbit = newTemp(Ity_I1);
         IRTemp vec_result = newTemp(Ity_V128);
         DIP("xsnabsdp v%d,v%d\n", XT, XB);
         assign( BHi_signed, unop( Iop_V128HIto64, mkexpr(vB) ) );
         assign( BHi_unsigned, binop( Iop_And64, mkexpr(BHi_signed),
                                      mkU64(VG_PPC_SIGN_MASK) ) );
         assign( BHi_negated_signbit,
                 unop( Iop_Not1,
                       unop( Iop_32to1,
                             binop( Iop_Shr32,
                                    unop( Iop_64HIto32,
                                          binop( Iop_And64,
                                                 mkexpr(BHi_signed),
                                                 mkU64(~VG_PPC_SIGN_MASK) )
                                          ),
                                    mkU8(31) ) ) ) );
         assign( BHi_negated,
                 binop( Iop_Or64,
                        binop( Iop_32HLto64,
                               binop( Iop_Shl32,
                                      unop( Iop_1Uto32,
                                            mkexpr(BHi_negated_signbit) ),
                                      mkU8(31) ),
                               mkU32(0) ),
                        mkexpr(BHi_unsigned) ) );
         assign( vec_result, binop( Iop_64HLtoV128, mkexpr(BHi_negated),
                                    mkU64(0x0ULL)));
         putVSReg( XT, mkexpr(vec_result));
         break;
      }
      case 0x280: // xsmaxdp (VSX Scalar Maximum Double-Precision)
      case 0x2A0: // xsmindp (VSX Scalar Minimum Double-Precision)
      {
         IRTemp frA     = newTemp(Ity_I64);
         IRTemp frB     = newTemp(Ity_I64);
         Bool isMin = opc2 == 0x2A0 ? True : False;
         DIP("%s v%d,v%d v%d\n", isMin ? "xsmaxdp" : "xsmindp", XT, XA, XB);

         assign(frA, unop(Iop_V128HIto64, mkexpr( vA )));
         assign(frB, unop(Iop_V128HIto64, mkexpr( vB )));
         putVSReg( XT, binop( Iop_64HLtoV128, get_max_min_fp(frA, frB, isMin), mkU64( 0 ) ) );

         break;
      }
      case 0x0F2: // xsrdpim (VSX Scalar Round to Double-Precision Integer using round toward -Infinity)
      case 0x0D2: // xsrdpip (VSX Scalar Round to Double-Precision Integer using round toward +Infinity)
      case 0x0D6: // xsrdpic (VSX Scalar Round to Double-Precision Integer using Current rounding mode)
      case 0x0B2: // xsrdpiz (VSX Scalar Round to Double-Precision Integer using round toward Zero)
      case 0x092: // xsrdpi  (VSX Scalar Round to Double-Precision Integer using round toward Nearest Away)
      {
         IRTemp frB_I64 = newTemp(Ity_I64);
         IRExpr * frD_fp_round = NULL;

         assign(frB_I64, unop(Iop_V128HIto64, mkexpr( vB )));
         frD_fp_round = _do_vsx_fp_roundToInt(frB_I64, opc2);

         DIP("xsrdpi%s v%d,v%d\n", _get_vsx_rdpi_suffix(opc2), XT, XB);
         putVSReg( XT,
                   binop( Iop_64HLtoV128,
                          unop( Iop_ReinterpF64asI64, frD_fp_round),
                          mkU64( 0 ) ) );
         break;
      }
      case 0x034: // xsresp (VSX Scalar Reciprocal Estimate single-Precision)
      case 0x014: /* xsrsqrtesp (VSX Scalar Reciprocal Square Root Estimate
                   * single-Precision)
                   */
      {
         IRTemp frB = newTemp(Ity_F64);
         IRTemp sqrt = newTemp(Ity_F64);
         IRExpr* ieee_one = IRExpr_Const(IRConst_F64i(0x3ff0000000000000ULL));
         IRExpr* rm  = get_IR_roundingmode();
         Bool redp = opc2 == 0x034;
         DIP("%s v%d,v%d\n", redp ? "xsresp" : "xsrsqrtesp", XT,
             XB);

         assign( frB,
                 unop( Iop_ReinterpI64asF64,
                       unop( Iop_V128HIto64, mkexpr( vB ) ) ) );

         if (!redp)
            assign( sqrt,
                    binop( Iop_SqrtF64,
                           rm,
                           mkexpr(frB) ) );
         putVSReg( XT,
                      binop( Iop_64HLtoV128,
                             unop( Iop_ReinterpF64asI64,
                                   binop( Iop_RoundF64toF32, rm,
                                          triop( Iop_DivF64,
                                                 rm,
                                                 ieee_one,
                                                 redp ? mkexpr( frB ) :
                                                        mkexpr( sqrt ) ) ) ),
                             mkU64( 0 ) ) );
         break;
      }

      case 0x0B4: // xsredp (VSX Scalar Reciprocal Estimate Double-Precision)
      case 0x094: // xsrsqrtedp (VSX Scalar Reciprocal Square Root Estimate Double-Precision)

      {
         IRTemp frB = newTemp(Ity_F64);
         IRTemp sqrt = newTemp(Ity_F64);
         IRExpr* ieee_one = IRExpr_Const(IRConst_F64i(0x3ff0000000000000ULL));
         IRExpr* rm  = get_IR_roundingmode();
         Bool redp = opc2 == 0x0B4;
         DIP("%s v%d,v%d\n", redp ? "xsredp" : "xsrsqrtedp", XT, XB);
         assign( frB,
                 unop( Iop_ReinterpI64asF64,
                       unop( Iop_V128HIto64, mkexpr( vB ) ) ) );

         if (!redp)
            assign( sqrt,
                    binop( Iop_SqrtF64,
                           rm,
                           mkexpr(frB) ) );
         putVSReg( XT,
                      binop( Iop_64HLtoV128,
                             unop( Iop_ReinterpF64asI64,
                                   triop( Iop_DivF64,
                                          rm,
                                          ieee_one,
                                          redp ? mkexpr( frB ) : mkexpr( sqrt ) ) ),
                             mkU64( 0 ) ) );
         break;
      }

      case 0x232: // xsrsp (VSX Scalar Round to Single-Precision)
      {
         IRTemp frB = newTemp(Ity_F64);
         IRExpr* rm  = get_IR_roundingmode();
         DIP("xsrsp v%d, v%d\n", XT, XB);
         assign( frB,
                 unop( Iop_ReinterpI64asF64,
                       unop( Iop_V128HIto64, mkexpr( vB ) ) ) );

         putVSReg( XT, binop( Iop_64HLtoV128,
                              unop( Iop_ReinterpF64asI64,
                                    binop( Iop_RoundF64toF32,
                                           rm,
                                           mkexpr( frB ) ) ),
                              mkU64( 0 ) ) );
         break;
      }

      case 0x354: // xvtstdcsp (VSX Test Data Class Single-Precision)
      {
         UInt DX_mask = IFIELD( theInstr, 16, 5 );
         UInt DC_mask = IFIELD( theInstr, 6, 1 );
         UInt DM_mask = IFIELD( theInstr, 2, 1 );
         UInt DCMX_mask = (DC_mask << 6) | (DM_mask << 5) | DX_mask;

         IRTemp match_value[4];
         IRTemp value[4];
         IRTemp NaN[4];
         IRTemp inf[4];
         IRTemp pos[4];
         IRTemp DCM[4];
         IRTemp zero[4];
         IRTemp dnorm[4];
         Int i;

         DIP("xvtstdcsp v%u,v%u,%u\n", (UInt)XT, (UInt)XB, DCMX_mask);

         for (i = 0; i < 4; i++) {
            NaN[i]   = newTemp(Ity_I32);
            inf[i]   = newTemp(Ity_I32);
            pos[i]   = newTemp(Ity_I32);
            DCM[i]   = newTemp(Ity_I32);
            zero[i]  = newTemp(Ity_I32);
            dnorm[i] = newTemp(Ity_I32);

            value[i] = newTemp(Ity_I32);
            match_value[i] = newTemp(Ity_I32);

            assign( value[i],
                    unop( Iop_64to32,
                          unop( Iop_V128to64,
                                binop( Iop_AndV128,
                                       binop( Iop_ShrV128,
                                              mkexpr( vB ),
                                              mkU8( (3-i)*32 ) ),
                                       binop( Iop_64HLtoV128,
                                              mkU64( 0x0 ),
                                              mkU64( 0xFFFFFFFF ) ) ) ) ) );

            assign( pos[i], unop( Iop_1Uto32,
                                  binop( Iop_CmpEQ32,
                                         binop( Iop_Shr32,
                                                mkexpr( value[i] ),
                                                mkU8( 31 ) ),
                                         mkU32( 0 ) ) ) );

            assign( NaN[i], unop( Iop_1Uto32, is_NaN( Ity_I32, value[i] ) ));
            assign( inf[i], unop( Iop_1Uto32, is_Inf( Ity_I32, value[i] ) ) );
            assign( zero[i], unop( Iop_1Uto32, is_Zero( Ity_I32, value[i] ) ) );

            assign( dnorm[i], unop( Iop_1Uto32, is_Denorm( Ity_I32,
                                                           value[i] ) ) );
            assign( DCM[i], create_DCM( Ity_I32, NaN[i], inf[i], zero[i],
                                        dnorm[i], pos[i] ) );

            assign( match_value[i],
                    unop( Iop_1Sto32,
                          binop( Iop_CmpNE32,
                                 binop( Iop_And32,
                                        mkU32( DCMX_mask ),
                                        mkexpr( DCM[i] ) ),
                                 mkU32( 0 ) ) ) );
         }

         putVSReg( XT, binop( Iop_64HLtoV128,
                              binop( Iop_32HLto64,
                                     mkexpr( match_value[0] ),
                                     mkexpr( match_value[1] ) ),
                              binop( Iop_32HLto64,
                                     mkexpr( match_value[2] ),
                                     mkexpr( match_value[3] ) ) ) );
      }
      break;

      case 0x360:  // xviexpsp  (VSX Vector Insert Exponent Single-Precision)
      {
            Int i;
            IRTemp new_XT[5];
            IRTemp A_value[4];
            IRTemp B_value[4];
            IRExpr *sign[4], *expr[4], *fract[4];

            DIP("xviexpsp v%d,v%d\n", XT, XB);
            new_XT[0] = newTemp(Ity_V128);
            assign( new_XT[0], binop( Iop_64HLtoV128,
                                      mkU64( 0x0 ),
                                      mkU64( 0x0 ) ) );

            for (i = 0; i < 4; i++) {
               A_value[i] = newTemp(Ity_I32);
               B_value[i] = newTemp(Ity_I32);

               assign( A_value[i],
                       unop( Iop_64to32,
                             unop( Iop_V128to64,
                                   binop( Iop_AndV128,
                                          binop( Iop_ShrV128,
                                                 mkexpr( vA ),
                                                 mkU8( (3-i)*32 ) ),
                                          binop( Iop_64HLtoV128,
                                                 mkU64( 0x0 ),
                                                 mkU64( 0xFFFFFFFF ) ) ) ) ) );
               assign( B_value[i],
                       unop( Iop_64to32,
                             unop( Iop_V128to64,
                                   binop( Iop_AndV128,
                                          binop( Iop_ShrV128,
                                                 mkexpr( vB ),
                                                 mkU8( (3-i)*32 ) ),
                                          binop( Iop_64HLtoV128,
                                                 mkU64( 0x0 ),
                                                 mkU64( 0xFFFFFFFF ) ) ) ) ) );

               sign[i] = binop( Iop_And32, mkexpr( A_value[i] ),
                                mkU32( 0x80000000 ) );
               expr[i] = binop( Iop_Shl32,
                                binop( Iop_And32, mkexpr( B_value[i] ),
                                       mkU32( 0xFF ) ),
                                mkU8( 23 ) );
               fract[i] = binop( Iop_And32, mkexpr( A_value[i] ),
                                 mkU32( 0x007FFFFF ) );

               new_XT[i+1] = newTemp(Ity_V128);
               assign( new_XT[i+1],
                       binop( Iop_OrV128,
                              binop( Iop_ShlV128,
                                     binop( Iop_64HLtoV128,
                                            mkU64( 0 ),
                                            binop( Iop_32HLto64,
                                                   mkU32( 0 ),
                                                   binop( Iop_Or32,
                                                         binop( Iop_Or32,
                                                                 sign[i],
                                                                 expr[i] ),
                                                          fract[i] ) ) ),
                                     mkU8( (3-i)*32 ) ),
                              mkexpr( new_XT[i] ) ) );
            }
            putVSReg( XT, mkexpr( new_XT[4] ) );
      }
      break;

      case 0x396:  // xsiexpdp (VSX Scalar Insert Exponent Double-Precision)
      {
         IRExpr *sign, *expr, *fract;
         UChar rA_addr = ifieldRegA(theInstr);
         UChar rB_addr = ifieldRegB(theInstr);
         IRTemp rA = newTemp( Ity_I64 );
         IRTemp rB = newTemp( Ity_I64 );

         DIP("xsiexpdp v%u,%u,%u\n", (UInt)XT, (UInt)rA_addr, (UInt)rB_addr);
         assign( rA, getIReg(rA_addr));
         assign( rB, getIReg(rB_addr));

         sign = binop( Iop_And64, mkexpr( rA ), mkU64( 0x8000000000000000 ) );
         expr = binop( Iop_Shl64,
                       binop( Iop_And64, mkexpr( rB ), mkU64( 0x7FF ) ),
                       mkU8( 52 ) );
         fract = binop( Iop_And64, mkexpr( rA ), mkU64( 0x000FFFFFFFFFFFFF ) );

         putVSReg( XT, binop( Iop_64HLtoV128,
                              binop( Iop_Or64,
                                     binop( Iop_Or64, sign, expr ),
                                     fract ),
                              mkU64( 0 ) ) );
      }
      break;

      case 0x3B6: // xvxexpdp (VSX Vector Extract Exponent Double-Precision)
                  // xvxsigdp (VSX Vector Extract Significand Double-Precision)
                  // xxbrh
                  // xvxexpsp (VSX Vector Extract Exponent Single-Precision)
                  // xvxsigsp (VSX Vector Extract Significand Single-Precision)
                  // xxbrw
                  // xxbrd
                  // xxbrq
                  // xvcvbf16spn (VSX Convert 16-bit bfloat to 32-bit float)
                  // xvcvspbf16 (VSX Convert 32-bit float to 16-bit bfloat)
                  // xvcvhpsp (VSX Vector Convert Half-Precision format to Single-Precision format)
                  // xvcvsphp (VSX Vector round and convert Single-Precision format to Half-Precision format)
      {
         UInt inst_select = IFIELD( theInstr, 16, 5);

         if (inst_select == 0) {
            DIP("xvxexpdp v%d,v%d\n", XT, XB);

            putVSReg( XT, binop( Iop_ShrV128,
                                 binop( Iop_AndV128,
                                        mkexpr( vB ),
                                        binop( Iop_64HLtoV128,
                                               mkU64( 0x7FF0000000000000 ),
                                               mkU64( 0x7FF0000000000000 ) ) ),
                                 mkU8( 52 ) ) );

         } else if (inst_select == 1) {
            Int i;
            IRExpr *normal[2];
            IRTemp value[2];
            IRTemp new_XT[3];

            DIP("xvxsigdp v%d,v%d\n", XT, XB);
            new_XT[0] = newTemp(Ity_V128);
            assign( new_XT[0], binop( Iop_64HLtoV128,
                                      mkU64( 0x0 ),
                                      mkU64( 0x0 ) ) );

            for (i = 0; i < 2; i++) {
               value[i] = newTemp(Ity_I64);
               assign( value[i],
                       unop( Iop_V128to64,
                            binop( Iop_AndV128,
                                    binop( Iop_ShrV128,
                                           mkexpr( vB ),
                                           mkU8( (1-i)*64 ) ),
                                    binop( Iop_64HLtoV128,
                                           mkU64( 0x0 ),
                                           mkU64( 0xFFFFFFFFFFFFFFFF ) ) ) ) );

               /* Value is normal if it isn't infinite, zero or denormalized */
               normal[i] = mkNOT1( mkOR1(
                                         mkOR1( is_NaN( Ity_I64, value[i] ),
                                                is_Inf( Ity_I64, value[i] ) ),
                                         mkOR1( is_Zero( Ity_I64, value[i] ),
                                                is_Denorm( Ity_I64,
                                                           value[i] ) ) ) );
               new_XT[i+1] = newTemp(Ity_V128);

               assign( new_XT[i+1],
                       binop( Iop_OrV128,
                              binop( Iop_ShlV128,
                                     binop( Iop_64HLtoV128,
                                            mkU64( 0x0 ),
                                            binop( Iop_Or64,
                                                   binop( Iop_And64,
                                                          mkexpr( value[i] ),
                                                          mkU64( 0xFFFFFFFFFFFFF ) ),
                                                   binop( Iop_Shl64,
                                                          unop( Iop_1Uto64,
                                                                normal[i]),
                                                          mkU8( 52 ) ) ) ),
                                     mkU8( (1-i)*64 ) ),
                              mkexpr( new_XT[i] ) ) );
            }
            putVSReg( XT, mkexpr( new_XT[2] ) );

         } else if (inst_select == 7) {
            IRTemp sub_element0 = newTemp( Ity_V128 );
            IRTemp sub_element1 = newTemp( Ity_V128 );

            DIP("xxbrh v%u, v%u\n", (UInt)XT, (UInt)XB);

            assign( sub_element0,
                    binop( Iop_ShrV128,
                           binop( Iop_AndV128,
                                  binop(Iop_64HLtoV128,
                                        mkU64( 0xFF00FF00FF00FF00 ),
                                        mkU64( 0xFF00FF00FF00FF00 ) ),
                                  mkexpr( vB ) ),
                           mkU8( 8 ) ) );
            assign( sub_element1,
                    binop( Iop_ShlV128,
                           binop( Iop_AndV128,
                                  binop(Iop_64HLtoV128,
                                        mkU64( 0x00FF00FF00FF00FF ),
                                        mkU64( 0x00FF00FF00FF00FF ) ),
                                  mkexpr( vB ) ),
                           mkU8( 8 ) ) );

            putVSReg(XT, binop( Iop_OrV128,
                                mkexpr( sub_element1 ),
                                mkexpr( sub_element0 ) ) );

         } else if (inst_select == 8) {
            DIP("xvxexpsp v%d,v%d\n", XT, XB);

            putVSReg( XT, binop( Iop_ShrV128,
                                 binop( Iop_AndV128,
                                        mkexpr( vB ),
                                        binop( Iop_64HLtoV128,
                                               mkU64( 0x7F8000007F800000 ),
                                               mkU64( 0x7F8000007F800000 ) ) ),
                                 mkU8( 23 ) ) );
         } else if (inst_select == 9) {
            Int i;
            IRExpr *normal[4];
            IRTemp value[4];
            IRTemp new_value[4];
            IRTemp new_XT[5];

            DIP("xvxsigsp v%d,v%d\n", XT, XB);
            new_XT[0] = newTemp(Ity_V128);
            assign( new_XT[0], binop( Iop_64HLtoV128,
                                      mkU64( 0x0 ),
                                      mkU64( 0x0 ) ) );

            for (i = 0; i < 4; i++) {
               value[i] = newTemp(Ity_I32);
               assign( value[i],
                       unop( Iop_64to32,
                             unop( Iop_V128to64,
                                   binop( Iop_AndV128,
                                          binop( Iop_ShrV128,
                                                 mkexpr( vB ),
                                                 mkU8( (3-i)*32 ) ),
                                          binop( Iop_64HLtoV128,
                                                 mkU64( 0x0 ),
                                                 mkU64( 0xFFFFFFFF ) ) ) ) ) );

               new_XT[i+1] = newTemp(Ity_V128);

               /* Value is normal if it isn't infinite, zero or denormalized */
               normal[i] = mkNOT1( mkOR1(
                                         mkOR1( is_NaN( Ity_I32, value[i] ),
                                                is_Inf( Ity_I32, value[i] ) ),
                                         mkOR1( is_Zero( Ity_I32, value[i] ),
                                                is_Denorm( Ity_I32,
                                                           value[i] ) ) ) );
               new_value[i] = newTemp(Ity_I32);
               assign( new_value[i],
                       binop( Iop_Or32,
                              binop( Iop_And32,
                                     mkexpr( value[i] ),
                                     mkU32( 0x7FFFFF ) ),
                              binop( Iop_Shl32,
                                     unop( Iop_1Uto32,
                                           normal[i]),
                                     mkU8( 23 ) ) ) );

               assign( new_XT[i+1],
                       binop( Iop_OrV128,
                              binop( Iop_ShlV128,
                                     binop( Iop_64HLtoV128,
                                            mkU64( 0x0 ),
                                            binop( Iop_32HLto64,
                                                   mkU32( 0x0 ),
                                                   mkexpr( new_value[i] ) ) ),
                                     mkU8( (3-i)*32 ) ),
                              mkexpr( new_XT[i] ) ) );
            }
            putVSReg( XT, mkexpr( new_XT[4] ) );

         } else if (inst_select == 15) {
            IRTemp sub_element0 = newTemp( Ity_V128 );
            IRTemp sub_element1 = newTemp( Ity_V128 );
            IRTemp sub_element2 = newTemp( Ity_V128 );
            IRTemp sub_element3 = newTemp( Ity_V128 );

            DIP("xxbrw v%u, v%u\n", (UInt)XT, (UInt)XB);

            assign( sub_element0,
                    binop( Iop_ShrV128,
                           binop( Iop_AndV128,
                                  binop(Iop_64HLtoV128,
                                        mkU64( 0xFF000000FF000000 ),
                                        mkU64( 0xFF000000FF000000 ) ),
                                  mkexpr( vB ) ),
                           mkU8( 24 ) ) );
            assign( sub_element1,
                    binop( Iop_ShrV128,
                           binop( Iop_AndV128,
                                  binop(Iop_64HLtoV128,
                                        mkU64( 0x00FF000000FF0000 ),
                                        mkU64( 0x00FF000000FF0000 ) ),
                                  mkexpr( vB ) ),
                           mkU8( 8 ) ) );
            assign( sub_element2,
                    binop( Iop_ShlV128,
                           binop( Iop_AndV128,
                                  binop(Iop_64HLtoV128,
                                        mkU64( 0x0000FF000000FF00 ),
                                        mkU64( 0x0000FF000000FF00 ) ),
                                  mkexpr( vB ) ),
                           mkU8( 8 ) ) );
            assign( sub_element3,
                    binop( Iop_ShlV128,
                           binop( Iop_AndV128,
                                  binop(Iop_64HLtoV128,
                                        mkU64( 0x00000000FF000000FF ),
                                        mkU64( 0x00000000FF000000FF ) ),
                                  mkexpr( vB ) ),
                           mkU8( 24 ) ) );

            putVSReg( XT,
                      binop( Iop_OrV128,
                             binop( Iop_OrV128,
                                    mkexpr( sub_element3 ),
                                    mkexpr( sub_element2 ) ),
                             binop( Iop_OrV128,
                                    mkexpr( sub_element1 ),
                                    mkexpr( sub_element0 ) ) ) );

         } else if ((inst_select == 16) && !prefix) {
            IRTemp result = newTemp(Ity_V128);
            UChar xT_addr = ifieldRegXT ( theInstr );
            UChar xB_addr = ifieldRegXB ( theInstr );
            /* Convert 16-bit bfloat to 32-bit float, not a prefix inst */
            DIP("xvcvbf16spn v%u,v%u\n", xT_addr, xB_addr);
            assign( result, vector_convert_bf16tofloat( vbi, mkexpr( vB ) ) );
            putVSReg( XT, mkexpr( result) );

         } else if ((inst_select == 17) && !prefix) {
            IRTemp result = newTemp(Ity_V128);
            UChar xT_addr = ifieldRegXT ( theInstr );
            UChar xB_addr = ifieldRegXB ( theInstr );
            /* Convert 32-bit float to 16-bit bfloat, not a prefix inst */
            DIP("xvcvspbf16 v%u,v%u\n", xT_addr, xB_addr);
            assign( result, vector_convert_floattobf16( vbi, mkexpr( vB ) ) );
            putVSReg( XT, mkexpr( result) );

         } else if (inst_select == 23) {
            DIP("xxbrd v%u, v%u\n", (UInt)XT, (UInt)XB);

            int i;
            int shift = 56;
            IRTemp sub_element[16];
            IRTemp new_xT[17];

            new_xT[0] = newTemp( Ity_V128 );
            assign( new_xT[0], binop( Iop_64HLtoV128,
                                      mkU64( 0 ),
                                      mkU64( 0 ) ) );

            for ( i = 0; i < 4; i++ ) {
               new_xT[i+1] = newTemp( Ity_V128 );
               sub_element[i]   = newTemp( Ity_V128 );
               sub_element[i+4] = newTemp( Ity_V128 );

               assign( sub_element[i],
                       binop( Iop_ShrV128,
                              binop( Iop_AndV128,
                                     binop( Iop_64HLtoV128,
                                            mkU64( (0xFFULL << (7 - i) * 8) ),
                                            mkU64( (0xFFULL << (7 - i) * 8) ) ),
                                     mkexpr( vB ) ),
                              mkU8( shift ) ) );

               assign( sub_element[i+4],
                       binop( Iop_ShlV128,
                              binop( Iop_AndV128,
                                     binop( Iop_64HLtoV128,
                                            mkU64( (0xFFULL << i*8) ),
                                            mkU64( (0xFFULL << i*8) ) ),
                                     mkexpr( vB ) ),
                              mkU8( shift ) ) );
               shift = shift - 16;

               assign( new_xT[i+1],
                       binop( Iop_OrV128,
                              mkexpr( new_xT[i] ),
                              binop( Iop_OrV128,
                                     mkexpr ( sub_element[i] ),
                                     mkexpr ( sub_element[i+4] ) ) ) );
               }

            putVSReg( XT, mkexpr( new_xT[4] ) );

         } else if (inst_select == 24) {
            // xvcvhpsp, (VSX Vector Convert half-precision format to
            //           Single-precision format)
            /* only supported on ISA 3.0 and newer */
            IRTemp result = newTemp( Ity_V128 );
            IRTemp src  = newTemp( Ity_I64 );

            if (!allow_isa_3_0) return False;

            DIP("xvcvhpsp v%d,v%d\n", XT,XB);
            /* The instruction does not set the C or FPCC fields.  The
             * instruction takes four 16-bit values stored in a 128-bit value
             * as follows:   x V | x V | x V | x V  where V is a 16-bit
             * value and x is an unused 16-bit value. To use Iop_F16toF32x4
             * the four 16-bit values will be gathered into a single 64 bit
             * value.  The backend will scatter the four 16-bit values back
             * into a 128-bit operand before issuing the instruction.
             */
            /* Gather 16-bit float values from V128 source into new 64-bit
             * source value for the Iop.
             */
            assign( src,
                    unop( Iop_V128to64,
                          binop( Iop_Perm8x16,
                                 mkexpr( vB ),
                                 binop ( Iop_64HLtoV128,
                                         mkU64( 0 ),
                                         mkU64( 0x020306070A0B0E0F) ) ) ) );

            assign( result, unop( Iop_F16toF32x4, mkexpr( src ) ) );

            putVSReg( XT, mkexpr( result ) );

         } else if (inst_select == 25) {
            // xvcvsphp, (VSX Vector round and Convert single-precision
            // format to half-precision format)
            /* only supported on ISA 3.0 and newer */
            IRTemp result = newTemp( Ity_V128 );
            IRTemp tmp64  = newTemp( Ity_I64 );

            if (!allow_isa_3_0) return False;
            DIP("xvcvsphp v%d,v%d\n", XT,XB);

            /* Iop_F32toF16x4 is V128 -> I64, scatter the 16-bit floats in the
             * I64 result to the V128 register to store.
             */
            assign( tmp64, unop( Iop_F32toF16x4_DEP, mkexpr( vB ) ) );

            /* Scatter 16-bit float values from returned 64-bit value
             * of V128 result.
             */
            if (host_endness == VexEndnessLE)
               /* Note location 0 may have a valid number in it.  Location
                * 15 should always be zero.  Use 0xF to put zeros in the
                * desired bytes.
                */
               assign( result,
                       binop( Iop_Perm8x16,
                              binop( Iop_64HLtoV128,
                                     mkexpr( tmp64 ),
                                     mkU64( 0 ) ),
                              binop ( Iop_64HLtoV128,
                                      mkU64( 0x0F0F00010F0F0203 ),
                                      mkU64( 0x0F0F04050F0F0607 ) ) ) );
            else
               assign( result,
                       binop( Iop_Perm8x16,
                              binop( Iop_64HLtoV128,
                                     mkexpr( tmp64 ),
                                     mkU64( 0 ) ),
                              binop ( Iop_64HLtoV128,
                                      mkU64( 0x0F0F06070F0F0405 ),
                                      mkU64( 0x0F0F02030F0F0001 ) ) ) );
            putVSReg( XT, mkexpr( result ) );

         } else if ( inst_select == 31 ) {
            int i;
            int shift_left = 8;
            int shift_right = 120;
            IRTemp sub_element[16];
            IRTemp new_xT[9];

            DIP("xxbrq v%u, v%u\n", (UInt) XT, (UInt) XB);

            new_xT[0] = newTemp( Ity_V128 );
            assign( new_xT[0], binop( Iop_64HLtoV128,
                                      mkU64( 0 ),
                                      mkU64( 0 ) ) );

            for ( i = 0; i < 8; i++ ) {
               new_xT[i+1] = newTemp( Ity_V128 );
               sub_element[i]   = newTemp( Ity_V128 );
               sub_element[i+8] = newTemp( Ity_V128 );

               assign( sub_element[i],
                       binop( Iop_ShrV128,
                              binop( Iop_AndV128,
                                     binop( Iop_64HLtoV128,
                                            mkU64( ( 0xFFULL << (7 - i) * 8 ) ),
                                            mkU64( 0x0ULL ) ),
                                     mkexpr( vB ) ),
                              mkU8( shift_right ) ) );
               shift_right = shift_right - 16;

               assign( sub_element[i+8],
                       binop( Iop_ShlV128,
                              binop( Iop_AndV128,
                                     binop( Iop_64HLtoV128,
                                            mkU64( 0x0ULL ),
                                            mkU64( ( 0xFFULL << (7 - i) * 8 ) ) ),
                                     mkexpr( vB ) ),
                              mkU8( shift_left ) ) );
               shift_left = shift_left + 16;

               assign( new_xT[i+1],
                       binop( Iop_OrV128,
                              mkexpr( new_xT[i] ),
                              binop( Iop_OrV128,
                                     mkexpr ( sub_element[i] ),
                                     mkexpr ( sub_element[i+8] ) ) ) );
               }

            putVSReg( XT, mkexpr( new_xT[8] ) );

         } else {
            vex_printf("dis_vxs_misc(ppc) Invalid instruction selection\n");
            return False;
         }
         break;
      }

      case 0x3D4: // xvtstdcdp (VSX Test Data Class Double-Precision)
      {
         UInt DX_mask = IFIELD( theInstr, 16, 5 );
         UInt DC_mask = IFIELD( theInstr, 6, 1 );
         UInt DM_mask = IFIELD( theInstr, 2, 1 );
         UInt DCMX_mask = (DC_mask << 6) | (DM_mask << 5) | DX_mask;

         IRTemp NaN[2], inf[2], zero[2], dnorm[2], pos[2], DCM[2];
         IRTemp match_value[2];
         IRTemp value[2];
         Int i;

         DIP("xvtstdcdp v%u,v%u,%u\n", (UInt)XT, (UInt)XB, DCMX_mask);

         for (i = 0; i < 2; i++) {
            NaN[i] = newTemp(Ity_I64);
            inf[i] = newTemp(Ity_I64);
            pos[i] = newTemp(Ity_I64);
            DCM[i] = newTemp(Ity_I64);
            zero[i] = newTemp(Ity_I64);
            dnorm[i] = newTemp(Ity_I64);

            value[i] = newTemp(Ity_I64);
            match_value[i] = newTemp(Ity_I64);

            assign( value[i],
                    unop( Iop_V128to64,
                          binop( Iop_AndV128,
                                 binop( Iop_ShrV128,
                                        mkexpr( vB ),
                                        mkU8( (1-i)*64 ) ),
                                 binop( Iop_64HLtoV128,
                                        mkU64( 0x0 ),
                                        mkU64( 0xFFFFFFFFFFFFFFFF ) ) ) ) );

            assign( pos[i], unop( Iop_1Uto64,
                                 binop( Iop_CmpEQ64,
                                        binop( Iop_Shr64,
                                               mkexpr( value[i] ),
                                               mkU8( 63 ) ),
                                        mkU64( 0 ) ) ) );

            assign( NaN[i], unop( Iop_1Uto64, is_NaN( Ity_I64, value[i] ) ) );
            assign( inf[i], unop( Iop_1Uto64, is_Inf( Ity_I64, value[i] ) ) );
            assign( zero[i], unop( Iop_1Uto64, is_Zero( Ity_I64, value[i] ) ) );
            assign( dnorm[i], unop( Iop_1Uto64, is_Denorm( Ity_I64,
                                                           value[i] ) ) );

            assign( DCM[i], create_DCM( Ity_I64, NaN[i], inf[i], zero[i],
                                        dnorm[i], pos[i] ) );

            assign( match_value[i],
                    unop( Iop_1Sto64,
                          binop( Iop_CmpNE64,
                                 binop( Iop_And64,
                                        mkU64( DCMX_mask ),
                                        mkexpr( DCM[i] ) ),
                                 mkU64( 0 ) ) ) );
         }
         putVSReg( XT, binop( Iop_64HLtoV128,
                              mkexpr( match_value[0] ),
                              mkexpr( match_value[1] ) ) );
      }
      break;

      case 0x3E0:  // xviexpdp (VSX Vector Insert Exponent Double-Precision)
      {
            Int i;
            IRTemp new_XT[3];
            IRTemp A_value[2];
            IRTemp B_value[2];
            IRExpr *sign[2], *expr[2], *fract[2];

            DIP("xviexpdp v%d,v%d\n", XT, XB);
            new_XT[0] = newTemp(Ity_V128);
            assign( new_XT[0], binop( Iop_64HLtoV128,
                                      mkU64( 0x0 ),
                                      mkU64( 0x0 ) ) );

            for (i = 0; i < 2; i++) {
               A_value[i] = newTemp(Ity_I64);
               B_value[i] = newTemp(Ity_I64);

               assign( A_value[i],
                       unop( Iop_V128to64,
                             binop( Iop_AndV128,
                                    binop( Iop_ShrV128,
                                           mkexpr( vA ),
                                           mkU8( (1-i)*64 ) ),
                                    binop( Iop_64HLtoV128,
                                           mkU64( 0x0 ),
                                           mkU64( 0xFFFFFFFFFFFFFFFF ) ) ) ) );
               assign( B_value[i],
                       unop( Iop_V128to64,
                             binop( Iop_AndV128,
                                    binop( Iop_ShrV128,
                                           mkexpr( vB ),
                                           mkU8( (1-i)*64 ) ),
                                    binop( Iop_64HLtoV128,
                                           mkU64( 0x0 ),
                                           mkU64( 0xFFFFFFFFFFFFFFFF ) ) ) ) );

               sign[i] = binop( Iop_And64, mkexpr( A_value[i] ),
                                mkU64( 0x8000000000000000 ) );
               expr[i] = binop( Iop_Shl64,
                                binop( Iop_And64, mkexpr( B_value[i] ),
                                       mkU64( 0x7FF ) ),
                                mkU8( 52 ) );
               fract[i] = binop( Iop_And64, mkexpr( A_value[i] ),
                                 mkU64( 0x000FFFFFFFFFFFFF ) );

               new_XT[i+1] = newTemp(Ity_V128);
               assign( new_XT[i+1],
                       binop( Iop_OrV128,
                              binop( Iop_ShlV128,
                                     binop( Iop_64HLtoV128,
                                            mkU64( 0 ),
                                            binop( Iop_Or64,
                                                   binop( Iop_Or64,
                                                          sign[i],
                                                          expr[i] ),
                                                   fract[i] ) ),
                                     mkU8( (1-i)*64 ) ),
                              mkexpr( new_XT[i] ) ) );
            }
            putVSReg( XT, mkexpr( new_XT[2] ) );
      }
      break;

      default:
         vex_printf( "dis_vxs_misc(ppc)(opc2)\n" );
         return False;
   }
   return True;
}

/*
 * VSX vector miscellaneous instructions
 */

static Bool
dis_vx_misc ( UInt prefix, UInt theInstr, UInt opc2 )
{
   /* XX3-Form */
   UChar XT = ifieldRegXT ( theInstr );
   UChar XA = ifieldRegXA ( theInstr );
   UChar XB = ifieldRegXB ( theInstr );
   IRTemp vA = newTemp( Ity_V128 );
   IRTemp vB = newTemp( Ity_V128 );
   IRTemp src1 = newTemp(Ity_I64);
   IRTemp src2 = newTemp(Ity_I64);
   IRTemp result_mask = newTemp(Ity_I64);
   IRTemp cmp_mask = newTemp(Ity_I64);
   IRTemp nan_mask = newTemp(Ity_I64);
   IRTemp snan_mask = newTemp(Ity_I64);
   IRTemp word_result = newTemp(Ity_I64);
   IRTemp check_result = newTemp(Ity_I64);
   IRTemp xT = newTemp( Ity_V128 );
   IRTemp nan_cmp_value = newTemp(Ity_I64);
   UInt trap_enabled = 0;  /* 0 - trap enabled is False */

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   assign( vA, getVSReg( XA ) );
   assign( vB, getVSReg( XB ) );
   assign( xT, getVSReg( XT ) );

   assign(src1, unop( Iop_V128HIto64, mkexpr( vA ) ) );
   assign(src2, unop( Iop_V128HIto64, mkexpr( vB ) ) );

   assign( nan_mask,
           binop( Iop_Or64,
                  unop( Iop_1Sto64, is_NaN( Ity_I64, src1 ) ),
                  unop( Iop_1Sto64, is_NaN( Ity_I64, src2 ) ) ) );

   if ( trap_enabled == 0 )
      /* Traps on invalid operation are assumed not enabled, assign
         result of comparison to xT.
      */
      assign( snan_mask, mkU64( 0 ) );

   else
      assign( snan_mask,
              binop( Iop_Or64,
                     unop( Iop_1Sto64, is_sNaN( Ity_I64, src1 ) ),
                     unop( Iop_1Sto64, is_sNaN( Ity_I64, src2 ) ) ) );

   assign (result_mask, binop( Iop_Or64,
                               mkexpr( snan_mask ),
                               mkexpr( nan_mask ) ) );

   switch (opc2) {
   case 0xC: //xscmpeqdp
      {
         DIP("xscmpeqdp v%d,v%d,v%d\n", XT, XA, XB);
         /* extract double-precision floating point source values from
            double word 0 */

         /* result of Iop_CmpF64 is 0x40 if operands are equal,
            mask is all 1's if equal. */

         assign( cmp_mask,
                 unop( Iop_1Sto64,
                       unop(Iop_32to1,
                            binop(Iop_Shr32,
                                  binop( Iop_CmpF64,
                                         unop( Iop_ReinterpI64asF64,
                                               mkexpr( src1 ) ),
                                         unop( Iop_ReinterpI64asF64,
                                               mkexpr( src2 ) ) ),
                                  mkU8( 6 ) ) ) ) );

         assign( word_result,
                 binop( Iop_Or64,
                        binop( Iop_And64, mkexpr( cmp_mask ),
                               mkU64( 0xFFFFFFFFFFFFFFFF ) ),
                        binop( Iop_And64,
                               unop( Iop_Not64, mkexpr( cmp_mask ) ),
                               mkU64( 0x0 ) ) ) );
         assign( nan_cmp_value, mkU64( 0 ) );
         break;
      }

   case 0x2C: //xscmpgtdp
      {
         DIP("xscmpgtdp v%d,v%d,v%d\n", XT, XA, XB);
         /* Test for src1 > src2 */

         /* Result of Iop_CmpF64 is 0x1 if op1 < op2, set mask to all 1's. */
         assign( cmp_mask,
                 unop( Iop_1Sto64,
                       unop(Iop_32to1,
                            binop(Iop_CmpF64,
                                  unop( Iop_ReinterpI64asF64,
                                        mkexpr( src2 ) ),
                                  unop( Iop_ReinterpI64asF64,
                                       mkexpr( src1 ) ) ) ) ) );
         assign( word_result,
                 binop( Iop_Or64,
                        binop( Iop_And64, mkexpr( cmp_mask ),
                               mkU64( 0xFFFFFFFFFFFFFFFF ) ),
                        binop( Iop_And64,
                               unop( Iop_Not64, mkexpr( cmp_mask ) ),
                               mkU64( 0x0 ) ) ) );
         assign( nan_cmp_value, mkU64( 0 ) );
         break;
      }

   case 0x4C: //xscmpgedp
      {
         DIP("xscmpgedp v%u,v%u,v%u\n", XT, XA, XB);
         /* compare src 1 >= src 2 */
         /* result of Iop_CmpF64 is 0x40 if operands are equal,
            mask is all 1's if equal. */
         assign( cmp_mask,
                 unop( Iop_1Sto64,
                       unop(Iop_32to1,
                            binop( Iop_Or32,
                                   binop( Iop_Shr32,
                                          binop(Iop_CmpF64,   /* EQ test */
                                                unop( Iop_ReinterpI64asF64,
                                                      mkexpr( src1 ) ),
                                                unop( Iop_ReinterpI64asF64,
                                                      mkexpr( src2 ) ) ),
                                          mkU8( 6 ) ),
                                   binop(Iop_CmpF64,  /* src2 < src 1 test */
                                         unop( Iop_ReinterpI64asF64,
                                               mkexpr( src2 ) ),
                                         unop( Iop_ReinterpI64asF64,
                                               mkexpr( src1 ) ) ) ) ) ) );
         assign( word_result,
                 binop( Iop_Or64,
                        binop( Iop_And64, mkexpr( cmp_mask ),
                               mkU64( 0xFFFFFFFFFFFFFFFF ) ),
                        binop( Iop_And64,
                               unop( Iop_Not64, mkexpr( cmp_mask ) ),
                               mkU64( 0x0 ) ) ) );
         assign( nan_cmp_value, mkU64( 0 ) );
         break;
      }

   case 0x200: //xsmaxcdp
      {
         DIP("xsmaxcdp v%d,v%d,v%d\n", XT, XA, XB);
         /* extract double-precision floating point source values from
            double word 0 */

         /* result of Iop_CmpF64 is 0x1 if arg1 LT then arg2, */
         assign( cmp_mask,
                 unop( Iop_1Sto64,
                       unop( Iop_32to1,
                             binop(Iop_CmpF64,
                                   unop( Iop_ReinterpI64asF64,
                                         mkexpr( src2 ) ),
                                   unop( Iop_ReinterpI64asF64,
                                         mkexpr( src1 ) ) ) ) ) );
         assign( word_result,
                 binop( Iop_Or64,
                        binop( Iop_And64, mkexpr( cmp_mask ), mkexpr( src1 ) ),
                        binop( Iop_And64,
                               unop( Iop_Not64, mkexpr( cmp_mask ) ),
                               mkexpr( src2 ) ) ) );
         assign( nan_cmp_value, mkexpr( src2 ) );
         break;
      }

   case 0x220: //xsmincdp
      {
         DIP("xsmincdp v%d,v%d,v%d\n", XT, XA, XB);
         /* extract double-precision floating point source values from
            double word 0 */

         /* result of Iop_CmpF64 is 0x1 if arg1 less then arg2, */
         assign( cmp_mask,
                 unop( Iop_1Sto64,
                       unop( Iop_32to1,
                             binop(Iop_CmpF64,
                                   unop( Iop_ReinterpI64asF64,
                                         mkexpr( src1 ) ),
                                   unop( Iop_ReinterpI64asF64,
                                         mkexpr( src2 ) ) ) ) ) );
         assign( word_result,
                 binop( Iop_Or64,
                        binop( Iop_And64, mkexpr( cmp_mask ), mkexpr( src1 ) ),
                        binop( Iop_And64,
                               unop( Iop_Not64, mkexpr( cmp_mask ) ),
                               mkexpr( src2 ) ) ) );
         assign( nan_cmp_value, mkexpr( src2 ) );
         break;
      }

   default:
      vex_printf( "dis_vx_misc(ppc)(opc2)\n" );
      return False;
   }

   /* If either argument is NaN, result is src2.  If either argument is
      SNaN, we are supposed to generate invalid operation exception.
      Currently don't support generating exceptions.  In case of an
      trap enabled invalid operation (SNaN) XT is not changed. The
      snan_mask is setup appropriately for trap enabled or not.
   */
   assign( check_result,
           binop( Iop_Or64,
                  binop( Iop_And64, mkexpr( snan_mask ),
                         unop( Iop_V128HIto64, mkexpr( xT ) ) ),
                  binop( Iop_And64, unop( Iop_Not64,
                                          mkexpr( snan_mask ) ),
                         binop( Iop_Or64,
                                binop( Iop_And64, mkexpr( nan_mask ),
                                       mkexpr( nan_cmp_value ) ),
                                binop( Iop_And64,
                                       unop( Iop_Not64,
                                             mkexpr( nan_mask ) ),
                                       mkU64( 0 ) ) ) ) ) );

   /* If SNaN is true, then the result is unchanged if a trap-enabled
      Invalid Operation occurs.  Result mask already setup for trap-enabled
      case.
   */
   putVSReg( XT,
             binop( Iop_64HLtoV128,
                    binop( Iop_Or64,
                           binop( Iop_And64,
                                  unop( Iop_Not64, mkexpr( result_mask ) ),
                                  mkexpr( word_result ) ),
                           binop( Iop_And64,
                                  mkexpr( result_mask ),
                                  mkexpr( check_result ) ) ),
                    mkU64( 0 ) ) );
   return True;
}

/*
 * VSX Logical Instructions
 */
static Bool
dis_vx_logic ( UInt prefix, UInt theInstr, UInt opc2 )
{
   /* XX3-Form */
   UChar opc1 = ifieldOPC( theInstr );
   UChar XT = ifieldRegXT ( theInstr );
   UChar XA = ifieldRegXA ( theInstr );
   UChar XB = ifieldRegXB ( theInstr );
   IRTemp vA = newTemp( Ity_V128 );
   IRTemp vB = newTemp( Ity_V128 );

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   if (opc1 != 0x3C) {
      vex_printf( "dis_vx_logic(ppc)(instr)\n" );
      return False;
   }

   assign( vA, getVSReg( XA ) );
   assign( vB, getVSReg( XB ) );

   switch (opc2) {
      case 0x268: // xxlxor
         DIP("xxlxor v%d,v%d,v%d\n", XT, XA, XB);
         putVSReg( XT, binop( Iop_XorV128, mkexpr( vA ), mkexpr( vB ) ) );
         break;
      case 0x248: // xxlor
         DIP("xxlor v%d,v%d,v%d\n", XT, XA, XB);
         putVSReg( XT, binop( Iop_OrV128, mkexpr( vA ), mkexpr( vB ) ) );
         break;
      case 0x288: // xxlnor
         DIP("xxlnor v%d,v%d,v%d\n", XT, XA, XB);
         putVSReg( XT, unop( Iop_NotV128, binop( Iop_OrV128, mkexpr( vA ),
                                                 mkexpr( vB ) ) ) );
         break;
      case 0x208: // xxland
         DIP("xxland v%d,v%d,v%d\n", XT, XA, XB);
         putVSReg( XT, binop( Iop_AndV128, mkexpr( vA ), mkexpr( vB ) ) );
         break;
      case 0x228: //xxlandc
         DIP("xxlandc v%d,v%d,v%d\n", XT, XA, XB);
         putVSReg( XT, binop( Iop_AndV128, mkexpr( vA ), unop( Iop_NotV128,
                                                               mkexpr( vB ) ) ) );
         break;
      case 0x2A8: // xxlorc (VSX Logical OR with complement)
         DIP("xxlorc v%d,v%d,v%d\n", XT, XA, XB);
         putVSReg( XT, binop( Iop_OrV128,
                              mkexpr( vA ),
                              unop( Iop_NotV128, mkexpr( vB ) ) ) );
         break;
      case 0x2C8: // xxlnand (VSX Logical NAND)
         DIP("xxlnand v%d,v%d,v%d\n", XT, XA, XB);
         putVSReg( XT, unop( Iop_NotV128,
                             binop( Iop_AndV128, mkexpr( vA ),
                                    mkexpr( vB ) ) ) );
         break;
      case 0x2E8: // xxleqv (VSX Logical Equivalence)
         DIP("xxleqv v%d,v%d,v%d\n", XT, XA, XB);
         putVSReg( XT, unop( Iop_NotV128,
                             binop( Iop_XorV128,
                             mkexpr( vA ), mkexpr( vB ) ) ) );
         break;
      default:
         vex_printf( "dis_vx_logic(ppc)(opc2)\n" );
         return False;
   }
   return True;
}

/*
 * VSX Load Instructions
 * NOTE: VSX supports word-aligned storage access.
 */
static Bool
dis_vx_load ( UInt prefix, UInt theInstr )
{
   /* XX1-Form */
   UChar opc1 = ifieldOPC( theInstr );
   UChar XT = ifieldRegXT ( theInstr );
   UChar rA_addr = ifieldRegA( theInstr );
   UChar rB_addr = ifieldRegB( theInstr );
   UInt opc2 = ifieldOPClo10( theInstr );

   IRType ty = mode64 ? Ity_I64 : Ity_I32;
   IRTemp EA = newTemp( ty );

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   if (opc1 != 0x1F) {
      vex_printf( "dis_vx_load(ppc)(instr)\n" );
      return False;
   }

   assign( EA, ea_rAor0_idxd( rA_addr, rB_addr ) );

   switch (opc2) {
   case 0x00C: // lxsiwzx (Load VSX Scalar as Integer Word and Zero Indexed)
   {
      IRExpr * exp;
      DIP("lxsiwzx %d,r%u,r%u\n", XT, rA_addr, rB_addr);

      if (host_endness == VexEndnessLE)
         exp = unop( Iop_64to32, load( Ity_I64, mkexpr( EA ) ) );
      else
         exp = unop( Iop_64HIto32, load( Ity_I64, mkexpr( EA ) ) );

      putVSReg( XT, binop( Iop_64HLtoV128,
                           unop( Iop_32Uto64, exp),
                           mkU64(0) ) );
      break;
   }

   case 0x00D: // lxvrbx
   {
      IRExpr * exp;
      DIP("lxvrbx v%u,r%u,r%u\n", XT, rA_addr, rB_addr);
      exp = load( Ity_I64, mkexpr( EA ) );

      if (host_endness == VexEndnessLE)
         putVSReg( XT, binop( Iop_64HLtoV128,
                              mkU64( 0x0 ),
                              binop( Iop_And64, mkU64( 0xFF ), exp ) ) );
      else
         putVSReg( XT,
                   binop( Iop_ShrV128,
                          binop( Iop_64HLtoV128,
                                 mkU64( 0x0 ),
                                 binop( Iop_And64, mkU64( 0xFF ), exp ) ),
                          mkU8( 15*8 ) ) );   // data is left most byte
      break;
   }

   case 0x02D: // lxvrhx
   {
      IRExpr * exp;

      DIP("lxvrhx v%u,r%u,r%u\n", XT, rA_addr, rB_addr);

      exp = load( Ity_I64, mkexpr( EA ) );

      if (host_endness == VexEndnessLE)
         putVSReg( XT, binop( Iop_64HLtoV128,
                              mkU64( 0x0 ),
                              binop( Iop_And64, mkU64( 0xFFFF ), exp ) ) );
      else
         putVSReg( XT,
                   binop( Iop_ShrV128,
                          binop( Iop_64HLtoV128,
                                 mkU64( 0x0 ),
                                 binop( Iop_And64, mkU64( 0xFFFF ), exp ) ),
                          mkU8( 7*16 ) ) );   // data is left most half-word
      break;
   }

   case 0x04D: // lxvrwx
   {
      IRExpr * exp;

      DIP("lxvrwx v%u,r%u,r%u\n", XT, rA_addr, rB_addr);

      exp = load( Ity_I64, mkexpr( EA ) );

      if (host_endness == VexEndnessLE)
         putVSReg( XT, binop( Iop_64HLtoV128,
                              mkU64( 0x0 ),
                              binop( Iop_And64, mkU64( 0xFFFFFFFF ), exp ) ) );
      else
         putVSReg( XT,
                   binop( Iop_ShrV128,
                          binop( Iop_64HLtoV128,
                                 mkU64( 0x0 ),
                                 binop( Iop_And64,
                                        mkU64( 0xFFFFFFFF ), exp ) ),
                          mkU8( 3*32 ) ) );   // data is left most word
      break;
   }

   case 0x06D: // lxvrdx
   {
      IRExpr * exp;

      DIP("lxvrdx v%u,r%u,r%u\n", XT, rA_addr, rB_addr);

      exp = load( Ity_I64, mkexpr( EA ) );

      if (host_endness == VexEndnessLE)
         putVSReg( XT, binop( Iop_64HLtoV128,
                              mkU64( 0x0 ),
                              binop( Iop_And64,
                                     mkU64( 0xFFFFFFFFFFFFFFFFULL), exp ) ) );
      else
         putVSReg( XT,
                   binop( Iop_ShrV128,
                          binop( Iop_64HLtoV128,
                                 mkU64( 0x0 ),
                                 binop( Iop_And64,
                                        mkU64( 0xFFFFFFFFFFFFFFFFULL), exp ) ),
                          mkU8( 1*64 ) ) );   // data is left most double word
      break;
   }

   case 0x08D: // stxvrbx
   {
      IRExpr * fetched_exp;
      IRExpr * store_exp;
      IRTemp vS = newTemp( Ity_V128 );

      DIP("stxvrbx v%u,r%u,r%u\n", XT, rA_addr, rB_addr);

      fetched_exp = load( Ity_I64, mkexpr( EA ) );
      assign( vS, getVSReg( XT ) );

      /* Fetch 64 bits, merge byte element 15 into the fetched value and
       * store.  */
      if (host_endness == VexEndnessLE) {
         store_exp = binop( Iop_Or64,
                            binop( Iop_And64,
                                   mkU64( 0x00000000000000FF ),
                                   unop( Iop_V128to64, mkexpr( vS ) ) ),
                            binop( Iop_And64,
                                   mkU64( 0xFFFFFFFFFFFFFF00 ),
                                   fetched_exp ) );
         store(  mkexpr( EA ),  store_exp );
      } else {
         store_exp = binop( Iop_Or64,
                            binop( Iop_And64,
                                   mkU64( 0xFF00000000000000 ),
                                   unop( Iop_V128HIto64, mkexpr( vS ) ) ),
                            binop( Iop_And64,
                                   mkU64( 0x00FFFFFFFFFFFFFF ),
                                   fetched_exp ) );
         store(  mkexpr( EA ),  store_exp );
      }
      break;
   }

   case 0x0AD: // stxvrhx
   {
      IRExpr * fetched_exp;
      IRExpr * store_exp;
      IRTemp vS = newTemp( Ity_V128 );

      DIP("stxvrhx v%u,r%u,r%u\n", XT, rA_addr, rB_addr);

      fetched_exp = load( Ity_I64, mkexpr( EA ) );
      assign( vS, getVSReg( XT ) );

      /* Fetch 64 bits, merge half-word element 7 into the fetched value and
       * store.  */
      if (host_endness == VexEndnessLE) {
         store_exp = binop( Iop_Or64,
                            binop( Iop_And64,
                                   mkU64( 0x000000000000FFFF ),
                                   unop( Iop_V128to64, mkexpr( vS ) ) ),
                            binop( Iop_And64,
                                   mkU64( 0xFFFFFFFFFFFF0000 ),
                                   fetched_exp ) );
         store(  mkexpr( EA ),  store_exp );
      } else {
         store_exp = binop( Iop_Or64,
                            binop( Iop_And64,
                                   mkU64( 0xFFFF000000000000 ),
                                   unop( Iop_V128HIto64, mkexpr( vS ) ) ),
                            binop( Iop_And64,
                                   mkU64( 0x0000FFFFFFFFFFFF ),
                                   fetched_exp ) );
         store(  mkexpr( EA ),  store_exp );
      }
      break;
   }

   case 0x0CD: // stxvrwx
   {
      IRExpr * fetched_exp;
      IRExpr * store_exp;
      IRTemp vS = newTemp( Ity_V128 );

      DIP("stxvrwx v%u,r%u,r%u\n", XT, rA_addr, rB_addr);

      fetched_exp = load( Ity_I64, mkexpr( EA ) );
      assign( vS, getVSReg( XT ) );

      /* Fetch 64 bits, merge word element 3 into the fetched value and
       * store.  */
      if (host_endness == VexEndnessLE) {
         store_exp = binop( Iop_Or64,
                            binop( Iop_And64,
                                   mkU64( 0x00000000FFFFFFFF ),
                                   unop( Iop_V128to64, mkexpr( vS ) ) ),
                            binop( Iop_And64,
                                   mkU64( 0xFFFFFFFF00000000 ),
                                   fetched_exp ) );
         store(  mkexpr( EA ),  store_exp );
      } else {
         store_exp = binop( Iop_Or64,
                            binop( Iop_And64,
                                   mkU64( 0xFFFFFFFF00000000 ),
                                   unop( Iop_V128HIto64, mkexpr( vS ) ) ),
                            binop( Iop_And64,
                                   mkU64( 0x00000000FFFFFFFF ),
                                   fetched_exp ) );
         store(  mkexpr( EA ),  store_exp );
      }
      break;
   }

   case 0x0ED: // stxvrdx
   {
      IRExpr * store_exp;
      IRTemp vS = newTemp( Ity_V128 );

      DIP("stxvrdx v%u,r%u,r%u\n", XT, rA_addr, rB_addr);

      assign( vS, getVSReg( XT ) );

      /* Fetch 64 bits, merge double word element 1 into the fetched value and
       * store.  Well, this is just store vS bits[63:0] at EA. */
      if (host_endness == VexEndnessLE) {
         store_exp = binop( Iop_And64,
                            mkU64( 0xFFFFFFFFFFFFFFFF ),
                            unop( Iop_V128to64, mkexpr( vS ) ) );
         store(  mkexpr( EA ),  store_exp );
      } else {
         store_exp = binop( Iop_And64,
                            mkU64( 0xFFFFFFFFFFFFFFFF ),
                            unop( Iop_V128HIto64, mkexpr( vS ) ) );
         store(  mkexpr( EA ),  store_exp );
      }
      break;
   }

   case 0x04C: // lxsiwax (Load VSX Scalar as Integer Word Algebraic Indexed)
   {
      IRExpr * exp;
      DIP("lxsiwax %d,r%u,r%u\n", XT, rA_addr, rB_addr);

      if (host_endness == VexEndnessLE)
         exp = unop( Iop_64to32, load( Ity_I64, mkexpr( EA ) ) );
      else
         exp = unop( Iop_64HIto32, load( Ity_I64, mkexpr( EA ) ) );

      putVSReg( XT, binop( Iop_64HLtoV128,
                           unop( Iop_32Sto64, exp),
                           mkU64(0) ) );
      break;
   }
   case 0x10C: // lxvx
   {
      UInt ea_off = 0;
      IRExpr* irx_addr;
      IRTemp word[4];
      int i;

      DIP("lxvx %u,r%u,r%u\n", (UInt)XT, rA_addr, rB_addr);

      if ( host_endness == VexEndnessBE ) {
         for ( i = 3; i>= 0; i-- ) {
            word[i] = newTemp( Ity_I64 );

            irx_addr =
               binop( mkSzOp( ty, Iop_Add8 ), mkexpr( EA ),
                      ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );

            assign( word[i], unop( Iop_32Uto64,
                                   load( Ity_I32, irx_addr ) ) );
            ea_off += 4;
         }

         putVSReg( XT, binop( Iop_64HLtoV128,
                              binop( Iop_Or64,
                                     mkexpr( word[2] ),
                                     binop( Iop_Shl64,
                                            mkexpr( word[3] ),
                                            mkU8( 32 ) ) ),
                              binop( Iop_Or64,
                                     mkexpr( word[0] ),
                                     binop( Iop_Shl64,
                                            mkexpr( word[1] ),
                                            mkU8( 32 ) ) ) ) );
      } else {
         for ( i = 0; i< 4; i++ ) {
            word[i] = newTemp( Ity_I64 );

            irx_addr =
               binop( mkSzOp( ty, Iop_Add8 ), mkexpr( EA ),
                      ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );

            assign( word[i], unop( Iop_32Uto64,
                                   load( Ity_I32, irx_addr ) ) );
            ea_off += 4;
         }

         putVSReg( XT, binop( Iop_64HLtoV128,
                              binop( Iop_Or64,
                                     mkexpr( word[2] ),
                                     binop( Iop_Shl64,
                                            mkexpr( word[3] ),
                                            mkU8( 32 ) ) ),
                              binop( Iop_Or64,
                                     mkexpr( word[0] ),
                                     binop( Iop_Shl64,
                                            mkexpr( word[1] ),
                                            mkU8( 32 ) ) ) ) );
      }
      break;
   }

   case 0x10D: // lxvl
   {
      DIP("lxvl %u,r%u,r%u\n", (UInt)XT, rA_addr, rB_addr);

      IRTemp byte[16];
      UInt i;
      UInt ea_off = 0;
      IRExpr* irx_addr;
      IRTemp tmp_low[9];
      IRTemp tmp_hi[9];
      IRTemp shift = newTemp( Ity_I8 );
      IRTemp nb_gt16 = newTemp( Ity_I8 );
      IRTemp ld_result = newTemp( Ity_V128 );
      IRTemp nb_not_zero = newTemp( Ity_I64 );

      IRTemp base_addr = newTemp( ty );

      tmp_low[0] = newTemp( Ity_I64 );
      tmp_hi[0] = newTemp( Ity_I64 );

      assign( base_addr, ea_rAor0( rA_addr ) );
      assign( tmp_low[0], mkU64( 0 ) );
      assign( tmp_hi[0], mkU64( 0 ) );

      /* shift is 15 - nb, where nb = rB[0:7], used to zero out upper bytes */
      assign( nb_not_zero, unop( Iop_1Sto64,
                                  binop( Iop_CmpNE64,
                                         mkU64( 0 ),
                                         binop( Iop_Shr64,
                                                getIReg( rB_addr ),
                                                mkU8( 56 ) ) ) ) );

      assign( nb_gt16, unop( Iop_1Sto8,
                             binop( Iop_CmpLT64U,
                                    binop( Iop_Shr64,
                                           getIReg( rB_addr ),
                                           mkU8( 60 ) ),
                                    mkU64( 1 ) ) ) );

      /* Set the shift to 0, by ANDing with nb_gt16.  nb_gt16 will be all
       * zeros if nb > 16.  This will result in quad word load being stored.
       */
      assign( shift,
              binop( Iop_And8,
                     unop( Iop_64to8,
                           binop( Iop_Mul64,
                                  binop( Iop_Sub64,
                                         mkU64 ( 16 ),
                                         binop( Iop_Shr64,
                                                getIReg( rB_addr ),
                                                mkU8( 56 ) ) ),
                                  mkU64( 8 ) ) ),
                     mkexpr( nb_gt16 ) ) );

      /* fetch all 16 bytes, we will remove what we don't want later */
      if ( host_endness == VexEndnessBE ) {
         for ( i = 0; i < 8; i++ ) {
            byte[i] = newTemp( Ity_I64 );
            tmp_hi[i+1] = newTemp( Ity_I64 );

            irx_addr =
               binop( mkSzOp( ty, Iop_Add8 ), mkexpr( base_addr ),
                      ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );
            ea_off += 1;

            assign( byte[i], binop( Iop_Shl64,
                                      unop( Iop_8Uto64,
                                            load( Ity_I8, irx_addr ) ),
                                      mkU8( 8 * ( 7 - i ) ) ) );

            assign( tmp_hi[i+1], binop( Iop_Or64,
                                        mkexpr( byte[i] ),
                                        mkexpr( tmp_hi[i] ) ) );
         }

         for ( i = 0; i < 8; i++ ) {
            byte[i+8] = newTemp( Ity_I64 );
            tmp_low[i+1] = newTemp( Ity_I64 );

            irx_addr =
               binop( mkSzOp( ty, Iop_Add8 ), mkexpr( base_addr ),
                      ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );
            ea_off += 1;

            assign( byte[i+8], binop( Iop_Shl64,
                                    unop( Iop_8Uto64,
                                          load( Ity_I8, irx_addr ) ),
                                    mkU8( 8 * ( 7 - i ) ) ) );

            assign( tmp_low[i+1], binop( Iop_Or64,
                                         mkexpr( byte[i+8] ),
                                         mkexpr( tmp_low[i] ) ) );
         }
         assign( ld_result, binop( Iop_ShlV128,
                                   binop( Iop_ShrV128,
                                          binop( Iop_64HLtoV128,
                                                 mkexpr( tmp_hi[8] ),
                                                 mkexpr( tmp_low[8] ) ),
                                          mkexpr( shift ) ),
                                   mkexpr( shift ) ) );
      } else {
         for ( i = 0; i < 8; i++ ) {
            byte[i] = newTemp( Ity_I64 );
            tmp_low[i+1] = newTemp( Ity_I64 );

            irx_addr =
               binop( mkSzOp( ty, Iop_Add8 ), mkexpr( base_addr ),
                      ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );
            ea_off += 1;

            assign( byte[i], binop( Iop_Shl64,
                                    unop( Iop_8Uto64,
                                          load( Ity_I8, irx_addr ) ),
                                    mkU8( 8 * i ) ) );

            assign( tmp_low[i+1],
                    binop( Iop_Or64,
                           mkexpr( byte[i] ), mkexpr( tmp_low[i] ) ) );
         }

         for ( i = 0; i < 8; i++ ) {
            byte[i + 8] = newTemp( Ity_I64 );
            tmp_hi[i+1] = newTemp( Ity_I64 );

            irx_addr =
               binop( mkSzOp( ty, Iop_Add8 ), mkexpr( base_addr ),
                      ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );
            ea_off += 1;

            assign( byte[i+8], binop( Iop_Shl64,
                                      unop( Iop_8Uto64,
                                            load( Ity_I8, irx_addr ) ),
                                      mkU8( 8 * i ) ) );

            assign( tmp_hi[i+1], binop( Iop_Or64,
                                        mkexpr( byte[i+8] ),
                                        mkexpr( tmp_hi[i] ) ) );
         }
         assign( ld_result, binop( Iop_ShrV128,
                                   binop( Iop_ShlV128,
                                          binop( Iop_64HLtoV128,
                                                 mkexpr( tmp_hi[8] ),
                                                 mkexpr( tmp_low[8] ) ),
                                          mkexpr( shift ) ),
                                   mkexpr( shift ) ) );
      }


      /* If nb = 0, mask out the calculated load result so the stored
       * value is zero.
       */

      putVSReg( XT, binop( Iop_AndV128,
                           mkexpr( ld_result ),
                           binop( Iop_64HLtoV128,
                                  mkexpr( nb_not_zero ),
                                  mkexpr( nb_not_zero ) ) ) );
      break;
   }

   case 0x12D: // lxvll (Load VSX Vector Left-Justified with Length XX1 form)
   {
      IRTemp byte[16];
      IRTemp tmp_low[9];
      IRTemp tmp_hi[9];
      IRTemp mask = newTemp(Ity_V128);
      IRTemp rB = newTemp( Ity_I64 );
      IRTemp nb = newTemp( Ity_I64 );
      IRTemp nb_zero = newTemp(Ity_V128);
      IRTemp mask_shift = newTemp(Ity_I64);
      Int i;
      UInt ea_off = 0;
      IRExpr* irx_addr;
      IRTemp base_addr = newTemp( ty );
      IRTemp nb_compare_zero = newTemp( Ity_I64 );

      DIP("lxvll %u,r%u,r%u\n", (UInt)XT, rA_addr, rB_addr);

      tmp_low[0] = newTemp(Ity_I64);
      tmp_hi[0] = newTemp(Ity_I64);

      assign( rB, getIReg(rB_addr));
      assign( base_addr, ea_rAor0( rA_addr ) );
      assign( tmp_low[0], mkU64( 0 ) );
      assign( tmp_hi[0], mkU64( 0 ) );

      /* mask_shift is number of 16 bytes minus (nb times 8-bits per byte) */
      assign( nb, binop( Iop_Shr64, mkexpr( rB ), mkU8( 56 )  ) );

      assign( nb_compare_zero, unop( Iop_1Sto64,
                                     binop( Iop_CmpEQ64,
                                            mkexpr( nb ),
                                            mkU64( 0 ) ) ) );

      /* nb_zero is 0xFF..FF if the nb_field = 0 */
      assign( nb_zero, binop( Iop_64HLtoV128,
                              mkexpr( nb_compare_zero ),
                              mkexpr( nb_compare_zero ) ) );

      assign( mask_shift, binop( Iop_Sub64,
                                 mkU64( 16*8 ),
                                 binop( Iop_Mul64,
                                        mkexpr( nb ),
                                        mkU64( 8 ) ) ) );

      /* fetch all 16 bytes, we will remove what we don't want later */
      for (i = 0; i < 8; i++) {
         byte[i] = newTemp(Ity_I64);
         tmp_hi[i+1] = newTemp(Ity_I64);

         irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( base_addr ),
                           ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );
         ea_off += 1;

         /* Instruction always loads in Big Endian format */
         assign( byte[i], binop( Iop_Shl64,
                                 unop( Iop_8Uto64,
                                       load( Ity_I8, irx_addr ) ),
                                 mkU8( 8 * (7 - i) ) ) );
         assign( tmp_hi[i+1],
                 binop( Iop_Or64,
                        mkexpr( byte[i] ), mkexpr( tmp_hi[i] ) ) );
      }

      for (i = 0; i < 8; i++) {
         byte[i + 8]  = newTemp(Ity_I64);
         tmp_low[i+1] = newTemp(Ity_I64);

         irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( base_addr ),
                           ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );
         ea_off += 1;

         /* Instruction always loads in Big Endian format */
        assign( byte[i+8], binop( Iop_Shl64,
                                   unop( Iop_8Uto64,
                                         load( Ity_I8, irx_addr ) ),
                                   mkU8( 8 * (7 - i) ) ) );
         assign( tmp_low[i+1], binop( Iop_Or64,
                                      mkexpr( byte[i+8] ),
                                      mkexpr( tmp_low[i] ) ) );
      }

      /* Create mask to clear the right most 16 - nb bytes, set to zero
       * if nb= 0.
       */
      assign( mask, binop( Iop_AndV128,
                          binop( Iop_ShlV128,
                                  binop( Iop_ShrV128,
                                         mkV128( 0xFFFF ),
                                         unop( Iop_64to8, mkexpr( mask_shift ) ) ),
                                  unop( Iop_64to8, mkexpr( mask_shift ) ) ),
                           unop( Iop_NotV128, mkexpr( nb_zero ) ) ) );

      putVSReg( XT, binop( Iop_AndV128,
                           mkexpr( mask ),
                           binop( Iop_64HLtoV128,
                                  mkexpr( tmp_hi[8] ),
                                  mkexpr( tmp_low[8] ) ) ) );
      break;
   }

   case 0x16C: // lxvwsx
   {
      IRTemp data = newTemp( Ity_I64 );

      DIP("lxvwsx %u,r%u,r%u\n", (UInt)XT, rA_addr, rB_addr);

      /* The load is a 64-bit fetch that is Endian aware, just want
       * the lower 32 bits. */
      if ( host_endness == VexEndnessBE ) {
         UInt ea_off = 4;
         IRExpr* irx_addr;

         irx_addr =
            binop( mkSzOp( ty, Iop_Sub8 ), mkexpr( EA ),
                   ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );

         assign( data, binop( Iop_And64,
                              load( Ity_I64, irx_addr ),
                              mkU64( 0xFFFFFFFF ) ) );

      } else {
         assign( data, binop( Iop_And64,
                              load( Ity_I64, mkexpr( EA ) ),
                              mkU64( 0xFFFFFFFF ) ) );
      }

      /* Take lower 32-bits and spat across the four word positions */
      putVSReg( XT,
                binop( Iop_64HLtoV128,
                       binop( Iop_Or64,
                              mkexpr( data ),
                              binop( Iop_Shl64,
                                     mkexpr( data ),
                                     mkU8( 32 ) ) ),
                       binop( Iop_Or64,
                              mkexpr( data ),
                              binop( Iop_Shl64,
                                     mkexpr( data ),
                                     mkU8( 32 ) ) ) ) );
      break;
   }

   case 0x20C: // lxsspx (Load VSX Scalar Single-Precision Indexed)
   {
      IRExpr * exp;
      DIP("lxsspx %d,r%u,r%u\n", XT, rA_addr, rB_addr);
      /* Take 32-bit floating point value in the upper half of the fetched
       * 64-bit value, convert to 64-bit floating point value and load into
       * top word of V128.
       */
      exp = unop( Iop_ReinterpF64asI64,
                  unop( Iop_F32toF64,
                        unop( Iop_ReinterpI32asF32,
                              load( Ity_I32, mkexpr( EA ) ) ) ) );

      putVSReg( XT, binop( Iop_64HLtoV128, exp, mkU64( 0 ) ) );
      break;
   }
   case 0x24C: // lxsdx
   {
      IRExpr * exp;
      DIP("lxsdx %d,r%u,r%u\n", XT, rA_addr, rB_addr);
      exp = load( Ity_I64, mkexpr( EA ) );
      // We need to pass an expression of type Ity_V128 with putVSReg, but the load
      // we just performed is only a DW.  But since the contents of VSR[XT] element 1
      // are undefined after this operation, we can just do a splat op.
      putVSReg( XT, binop( Iop_64HLtoV128, exp, exp ) );
      break;
   }

   case 0x30D: // lxsibzx
   {
      IRExpr *byte;
      IRExpr* irx_addr;

      DIP("lxsibzx %u,r%u,r%u\n", (UInt)XT, rA_addr, rB_addr);

      if ( host_endness == VexEndnessBE )
         irx_addr = binop( Iop_Sub64, mkexpr( EA ), mkU64( 7 ) );

      else
         irx_addr = mkexpr( EA );
      /* byte load */
      byte = load( Ity_I8, irx_addr );
      putVSReg( XT, binop( Iop_64HLtoV128,
                           unop( Iop_8Uto64, byte ),
                           mkU64( 0 ) ) );
      break;
   }

   case 0x32D: // lxsihzx
   {
      IRExpr *hword;
      IRExpr* irx_addr;

      DIP("lxsihzx %u,r%u,r%u\n", (UInt)XT, rA_addr, rB_addr);

      if ( host_endness == VexEndnessBE )
         irx_addr = binop( Iop_Sub64, mkexpr( EA ), mkU64( 6 ) );

      else
         irx_addr = mkexpr( EA );

      hword = load( Ity_I16, irx_addr );
      putVSReg( XT, binop( Iop_64HLtoV128,
                            unop( Iop_16Uto64,
                                  hword ),
                           mkU64( 0 ) ) );
      break;
   }
   case 0x34C: // lxvd2x
   {
      IRExpr *t128;
      DIP("lxvd2x %d,r%u,r%u\n", XT, rA_addr, rB_addr);
      t128 = load( Ity_V128, mkexpr( EA ) );

      /* The data in the vec register should be in big endian order.
         So if we just did a little endian load then swap around the
         high and low double words. */
      if (host_endness == VexEndnessLE) {
         IRTemp high = newTemp(Ity_I64);
         IRTemp low = newTemp(Ity_I64);
         assign( high, unop(Iop_V128HIto64, t128) );
         assign( low, unop(Iop_V128to64, t128) );
         t128 = binop( Iop_64HLtoV128, mkexpr (low), mkexpr (high) );
      }

      putVSReg( XT, t128 );
      break;
   }
   case 0x14C: // lxvdsx
   {
      IRTemp data = newTemp(Ity_I64);
      DIP("lxvdsx %d,r%u,r%u\n", XT, rA_addr, rB_addr);
      assign( data, load( Ity_I64, mkexpr( EA ) ) );
      putVSReg( XT, binop( Iop_64HLtoV128, mkexpr( data ), mkexpr( data ) ) );
      break;
   }
   case 0x30C:
   {
      IRExpr *t0;

      DIP("lxvw4x %u,r%u,r%u\n", XT, rA_addr, rB_addr);

      /* The load will result in the data being in BE order. */
      if (host_endness == VexEndnessLE) {
         IRExpr *t0_BE;
         IRTemp perm_LE = newTemp(Ity_V128);

         t0_BE = load( Ity_V128, mkexpr( EA ) );

         /*  Permute the data to LE format */
         assign( perm_LE, binop( Iop_64HLtoV128, mkU64(0x0c0d0e0f08090a0bULL),
                                 mkU64(0x0405060700010203ULL)));

         t0 = binop( Iop_Perm8x16, t0_BE, mkexpr(perm_LE) );
      } else {
         t0 = load( Ity_V128, mkexpr( EA ) );
      }

      putVSReg( XT, t0 );
      break;
   }

   case 0x32C: // lxvh8x
   {
      DIP("lxvh8x %u,r%u,r%u\n", (UInt)XT, rA_addr, rB_addr);

      IRTemp h_word[8];
      int i;
      UInt ea_off = 0;
      IRExpr* irx_addr;
      IRTemp tmp_low[5];
      IRTemp tmp_hi[5];

      tmp_low[0] = newTemp( Ity_I64 );
      tmp_hi[0] = newTemp( Ity_I64 );
      assign( tmp_low[0], mkU64( 0 ) );
      assign( tmp_hi[0], mkU64( 0 ) );

      for ( i = 0; i < 4; i++ ) {
         h_word[i]    = newTemp(Ity_I64);
         tmp_low[i+1] = newTemp(Ity_I64);

         irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( EA ),
                           ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );
         ea_off += 2;

         assign( h_word[i], binop( Iop_Shl64,
                                   unop( Iop_16Uto64,
                                         load( Ity_I16, irx_addr ) ),
                                   mkU8( 16 * ( 3 - i ) ) ) );

         assign( tmp_low[i+1],
                 binop( Iop_Or64,
                        mkexpr( h_word[i] ), mkexpr( tmp_low[i] ) ) );
      }

      for ( i = 0; i < 4; i++ ) {
         h_word[i+4]   = newTemp( Ity_I64 );
         tmp_hi[i+1] = newTemp( Ity_I64 );

         irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( EA ),
                           ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );
         ea_off += 2;

         assign( h_word[i+4], binop( Iop_Shl64,
                                     unop( Iop_16Uto64,
                                           load( Ity_I16, irx_addr ) ),
                                     mkU8( 16 * ( 3 - i ) ) ) );

         assign( tmp_hi[i+1], binop( Iop_Or64,
                                     mkexpr( h_word[i+4] ),
                                     mkexpr( tmp_hi[i] ) ) );
      }
      putVSReg( XT, binop( Iop_64HLtoV128,
                           mkexpr( tmp_low[4] ), mkexpr( tmp_hi[4] ) ) );
      break;
   }

   case 0x36C: // lxvb16x
   {
      DIP("lxvb16x %u,r%u,r%u\n", (UInt)XT, rA_addr, rB_addr);

      /* The result of lxvb16x should be the same on big and little
         endian systems. We do a host load, then reverse the bytes in
         the double words. If the host load was little endian we swap
         them around again. */

      IRTemp high = newTemp(Ity_I64);
      IRTemp high_rev = newTemp(Ity_I64);
      IRTemp low = newTemp(Ity_I64);
      IRTemp low_rev = newTemp(Ity_I64);

      IRExpr *t128 = load( Ity_V128, mkexpr( EA ) );

      assign( high, unop(Iop_V128HIto64, t128) );
      assign( high_rev, unop(Iop_Reverse8sIn64_x1, mkexpr(high)) );
      assign( low, unop(Iop_V128to64, t128) );
      assign( low_rev, unop(Iop_Reverse8sIn64_x1, mkexpr(low)) );

      if (host_endness == VexEndnessLE)
         t128 = binop( Iop_64HLtoV128, mkexpr (low_rev), mkexpr (high_rev) );
      else
         t128 = binop( Iop_64HLtoV128, mkexpr (high_rev), mkexpr (low_rev) );

      putVSReg( XT, t128 );
      break;
   }

   default:
      vex_printf( "dis_vx_load(ppc)(opc2)\n" );
      return False;
   }
   return True;
}

/*
 * VSX Move Instructions
 */
static Bool
dis_vx_move ( UInt prefix, UInt theInstr )
{
   /* XX1-Form */
   UChar opc1 = ifieldOPC( theInstr );
   UChar XS = ifieldRegXS( theInstr );
   UChar rA_addr = ifieldRegA( theInstr );
   UChar rB_addr = ifieldRegB( theInstr );
   IRTemp vS = newTemp( Ity_V128 );
   UInt opc2 = ifieldOPClo10( theInstr );
   IRType ty = Ity_I64;

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   if ( opc1 != 0x1F ) {
      vex_printf( "dis_vx_move(ppc)(instr)\n" );
      return False;
   }

   switch (opc2) {
   case 0x133: // mfvsrld RA,XS   Move From VSR Lower Doubleword
      DIP("mfvsrld %u,r%u\n", (UInt)XS, rA_addr);

      assign( vS, getVSReg( XS ) );
      putIReg( rA_addr, unop(Iop_V128to64, mkexpr( vS) ) );

      break;

   case 0x193: // mfvsrdd  XT,RA,RB  Move to VSR Double Doubleword
   {
      IRTemp tmp = newTemp( Ity_I32 );

      DIP("mfvsrdd %u,r%u\n", (UInt)XS, rA_addr);

      assign( tmp, unop( Iop_64to32, getIReg(rA_addr) ) );
      assign( vS, binop( Iop_64HLtoV128,
                         binop( Iop_32HLto64,
                                mkexpr( tmp ),
                                mkexpr( tmp ) ),
                         binop( Iop_32HLto64,
                                mkexpr( tmp ),
                                mkexpr( tmp ) ) ) );
      putVSReg( XS, mkexpr( vS ) );
   }
   break;

   case 0x1B3: // mtvsrws  XT,RA  Move to VSR word & Splat
   {
      IRTemp rA = newTemp( ty );
      IRTemp rB = newTemp( ty );

      DIP("mfvsrws %u,r%u\n", (UInt)XS, rA_addr);

      if ( rA_addr == 0 )
         assign( rA, mkU64 ( 0 ) );
      else
         assign( rA, getIReg(rA_addr) );

      assign( rB, getIReg(rB_addr) );
      assign( vS, binop( Iop_64HLtoV128, mkexpr( rA ), mkexpr( rB ) ) );
      putVSReg( XS, mkexpr( vS ) );
   }
   break;

   default:
      vex_printf( "dis_vx_move(ppc)(opc2)\n" );
      return False;
   }
   return True;
}

/*
 * VSX Store Instructions
 * NOTE: VSX supports word-aligned storage access.
 */
static Bool
dis_vsx_vector_paired_load_store ( UInt prefix, UInt theInstr )
{
   /* X-Form/DS-Form */
   UInt   opc2    = ifieldOPClo9(theInstr);
   UChar  rA_addr = ifieldRegA(theInstr);
   UChar  rB_addr = ifieldRegB(theInstr);
   IRType ty      = mode64 ? Ity_I64 : Ity_I32;
   IRTemp EA      = newTemp(ty);
   IRTemp EA_16        = newTemp(ty);
   UChar XTp      = ifieldRegXTp(theInstr);

   assign( EA, ea_rAor0_idxd( rA_addr, rB_addr ) );

   // address of next 128bits
   assign( EA_16, binop( Iop_Add64, mkU64( 16), mkexpr( EA ) ) );

   switch (opc2) {
   case 0x14D:  // lxvpx
      DIP( "lxvpx %u,%d(%u)\n", XTp, rA_addr, rB_addr );
      if ( host_endness == VexEndnessBE ) {
         putVSReg( XTp,   load( Ity_V128, mkexpr( EA ) ) );
         putVSReg( XTp+1, load( Ity_V128, mkexpr( EA_16 ) ) );
      } else {
         putVSReg( XTp+1, load( Ity_V128, mkexpr( EA ) ) );
         putVSReg( XTp,   load( Ity_V128, mkexpr( EA_16 ) ) );
      }
      break;

   case 0x1CD: { // stxvpx
      IRTemp EA_8  = newTemp(ty);
      IRTemp EA_24 = newTemp(ty);

      DIP( "stxvpx %u,%d(%u)\n", XTp, rA_addr, rB_addr );

      assign( EA_8, binop( Iop_Add64, mkU64( 8 ), mkexpr( EA ) ) );
      assign( EA_24, binop( Iop_Add64, mkU64( 24 ), mkexpr( EA ) ) );

      if ( host_endness == VexEndnessBE ) {
         store( mkexpr( EA ), unop( Iop_V128to64, getVSReg( XTp ) ) );
         store( mkexpr( EA_8 ), unop( Iop_V128HIto64, getVSReg( XTp ) ) );
         store( mkexpr( EA_16 ), unop( Iop_V128to64, getVSReg( XTp+1 ) ) );
         store( mkexpr( EA_24 ), unop( Iop_V128HIto64, getVSReg( XTp+1 ) ) );

      } else {
         store( mkexpr( EA ), unop( Iop_V128to64, getVSReg( XTp+1 ) ) );
         store( mkexpr( EA_8 ), unop( Iop_V128HIto64, getVSReg( XTp+1 ) ) );
         store( mkexpr( EA_16 ), unop( Iop_V128to64, getVSReg( XTp ) ) );
         store( mkexpr( EA_24 ), unop( Iop_V128HIto64, getVSReg( XTp ) ) );
      }
      break;
   }

   default:
      vex_printf("dis_vsx_vector_paired_load_store\n");
      return False;
   }

   return True;
}

static Bool
dis_vx_store ( UInt prefix, UInt theInstr )
{
   /* XX1-Form */
   UChar opc1 = ifieldOPC( theInstr );
   UChar XS = ifieldRegXS( theInstr );
   UChar rA_addr = ifieldRegA( theInstr );
   UChar rB_addr = ifieldRegB( theInstr );
   IRTemp vS = newTemp( Ity_V128 );
   UInt opc2 = ifieldOPClo10( theInstr );

   IRType ty = mode64 ? Ity_I64 : Ity_I32;
   IRTemp EA = newTemp( ty );

   /* There is no prefixed version of these instructions.  */
   PREFIX_CHECK

   if (opc1 != 0x1F) {
      vex_printf( "dis_vx_store(ppc)(instr)\n" );
      return False;
   }

   assign( EA, ea_rAor0_idxd( rA_addr, rB_addr ) );
   assign( vS, getVSReg( XS ) );

   switch (opc2) {
   case 0x08C:
   {
     /* Need the next to the most significant 32-bit word from
      * the 128-bit vector.
      */
      IRExpr * high64, * low32;
      DIP("stxsiwx %d,r%u,r%u\n", XS, rA_addr, rB_addr);
      high64 = unop( Iop_V128HIto64, mkexpr( vS ) );
      low32  = unop( Iop_64to32, high64 );
      store( mkexpr( EA ), low32 );
      break;
   }

   case 0x18C: // stxvx Store VSX Vector Indexed
   {
      UInt ea_off = 0;
      IRExpr* irx_addr;
      IRTemp word0 = newTemp( Ity_I64 );
      IRTemp word1 = newTemp( Ity_I64 );
      IRTemp word2 = newTemp( Ity_I64 );
      IRTemp word3 = newTemp( Ity_I64 );
      DIP("stxvx %u,r%u,r%u\n", (UInt)XS, rA_addr, rB_addr);

      assign( word0,  binop( Iop_Shr64,
                             unop( Iop_V128HIto64, mkexpr( vS ) ),
                             mkU8( 32 ) ) );

      assign( word1, binop( Iop_And64,
                            unop( Iop_V128HIto64, mkexpr( vS ) ),
                            mkU64( 0xFFFFFFFF ) ) );

      assign( word2, binop( Iop_Shr64,
                            unop( Iop_V128to64, mkexpr( vS ) ),
                            mkU8( 32 ) ) );

      assign( word3, binop( Iop_And64,
                            unop( Iop_V128to64, mkexpr( vS ) ),
                            mkU64( 0xFFFFFFFF ) ) );

      if (host_endness == VexEndnessBE) {
         store( mkexpr( EA ), unop( Iop_64to32, mkexpr( word0 ) ) );

         ea_off += 4;
         irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( EA ),
                           ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );

         store( irx_addr, unop( Iop_64to32, mkexpr( word1 ) ) );

         ea_off += 4;
         irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( EA ),
                           ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );

         store( irx_addr, unop( Iop_64to32, mkexpr( word2 ) ) );
         ea_off += 4;
         irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( EA ),
                           ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );

         store( irx_addr, unop( Iop_64to32, mkexpr( word3 ) ) );
      } else {
         store( mkexpr( EA ), unop( Iop_64to32, mkexpr( word3 ) ) );

         ea_off += 4;
         irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( EA ),
                           ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );

         store( irx_addr, unop( Iop_64to32, mkexpr( word2 ) ) );

         ea_off += 4;
         irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( EA ),
                           ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );

         store( irx_addr, unop( Iop_64to32, mkexpr( word1 ) ) );
         ea_off += 4;
         irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( EA ),
                           ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );

         store( irx_addr, unop( Iop_64to32, mkexpr( word0 ) ) );
      }
      break;
   }

   case 0x18D: // stxvl Store VSX Vector Indexed
   {
      UInt ea_off = 0;
      IRExpr* irx_addr;
      IRTemp word0 = newTemp( Ity_I64 );
      IRTemp word1 = newTemp( Ity_I64 );
      IRTemp word2 = newTemp( Ity_I64 );
      IRTemp word3 = newTemp( Ity_I64 );
      IRTemp shift = newTemp( Ity_I8 );
      IRTemp nb_gt16 = newTemp( Ity_I8 );
      IRTemp nb_zero = newTemp( Ity_V128 );
      IRTemp nb = newTemp( Ity_I8 );
      IRTemp nb_field = newTemp( Ity_I64 );
      IRTemp n_bytes = newTemp( Ity_I8 );
      IRTemp base_addr = newTemp( ty );
      IRTemp current_mem = newTemp( Ity_V128 );
      IRTemp store_val = newTemp( Ity_V128 );
      IRTemp nb_mask = newTemp( Ity_V128 );

      DIP("stxvl %u,r%u,r%u\n", (UInt)XS, rA_addr, rB_addr);

      assign( nb_field, binop( Iop_Shr64,
                               getIReg(rB_addr),
                               mkU8( 56 ) ) );

      assign( nb, unop( Iop_64to8, mkexpr( nb_field ) ) );

      /* nb_gt16 will be all zeros if nb > 16 */
      assign( nb_gt16, unop( Iop_1Sto8,
                             binop( Iop_CmpLT64U,
                                    binop( Iop_Shr64,
                                           mkexpr( nb_field ),
                                           mkU8( 4 ) ),
                                    mkU64( 1 ) ) ) );

      /* nb_zero is 0xFF..FF if the nb_field = 0 */
      assign( nb_zero, binop( Iop_64HLtoV128,
                              unop( Iop_1Sto64,
                                    binop( Iop_CmpEQ64,
                                           mkexpr( nb_field ),
                                           mkU64( 0 ) ) ),
                              unop( Iop_1Sto64,
                                    binop( Iop_CmpEQ64,
                                           mkexpr( nb_field ),
                                           mkU64( 0 ) ) ) ) );

      /* set n_bytes to 0 if nb >= 16.  Otherwise, set to nb. */
      assign( n_bytes, binop( Iop_And8, mkexpr( nb ), mkexpr( nb_gt16 ) ) );
      assign( shift, unop( Iop_64to8,
                           binop( Iop_Mul64,
                                  binop( Iop_Sub64,
                                         mkU64( 16 ),
                                         unop( Iop_8Uto64,
                                               mkexpr( n_bytes ) ) ),
                                  mkU64( 8 ) ) ) );

      /* We only have a 32-bit store function. So, need to fetch the
       * contents of memory merge with the store value and do two
       * 32-byte stores so we preserve the contents of memory not
       * addressed by nb.
       */
      assign( base_addr, ea_rAor0( rA_addr ) );

      assign( current_mem,
              binop( Iop_64HLtoV128,
                     load( Ity_I64,
                           binop( mkSzOp( ty, Iop_Add8 ),
                                  mkexpr( base_addr ),
                                  ty == Ity_I64 ? mkU64( 8 ) : mkU32( 8 )
                                  ) ),
		     load( Ity_I64, mkexpr( base_addr ) ) ) );

      /* Set the nb_mask to all zeros if nb = 0 so the current contents
       * of memory get written back without modifications.
       *
       * The store_val is a combination of the current memory value
       * and the bytes you want to store.  The nb_mask selects the
       * bytes you want stored from Vs.
       */
       assign( nb_mask,
               binop( Iop_OrV128,
                      binop( Iop_AndV128,
                             mkexpr( nb_zero ),
                             mkV128( 0 ) ),
                      binop( Iop_AndV128,
			     binop( Iop_ShrV128,
				    mkV128( 0xFFFF ),
				    mkexpr( shift ) ),
                             unop( Iop_NotV128, mkexpr( nb_zero ) ) ) ) );

       assign( store_val,
               binop( Iop_OrV128,
                      binop( Iop_AndV128,
			     mkexpr( vS ),
                             mkexpr( nb_mask ) ),
                      binop( Iop_AndV128,
                             unop( Iop_NotV128, mkexpr( nb_mask ) ),
                             mkexpr( current_mem) ) ) );

      /* Store the value in 32-byte chunks */
      assign( word0,  binop( Iop_Shr64,
                             unop( Iop_V128HIto64, mkexpr( store_val ) ),
                             mkU8( 32 ) ) );

      assign( word1, binop( Iop_And64,
                            unop( Iop_V128HIto64, mkexpr( store_val ) ),
                            mkU64( 0xFFFFFFFF ) ) );

      assign( word2, binop( Iop_Shr64,
                            unop( Iop_V128to64, mkexpr( store_val ) ),
                            mkU8( 32 ) ) );

      assign( word3, binop( Iop_And64,
                            unop( Iop_V128to64, mkexpr( store_val ) ),
                            mkU64( 0xFFFFFFFF ) ) );

      ea_off = 0;
      irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( base_addr ),
                           ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );

      store( irx_addr, unop( Iop_64to32, mkexpr( word3 ) ) );

      ea_off += 4;
      irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( base_addr ),
                        ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );

      store( irx_addr, unop( Iop_64to32, mkexpr( word2 ) ) );

      ea_off += 4;
      irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( base_addr ),
                        ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );

      store( irx_addr, unop( Iop_64to32, mkexpr( word1 ) ) );

      ea_off += 4;
      irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( base_addr ),
                        ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );

      store( irx_addr, unop( Iop_64to32, mkexpr( word0 ) ) );
      break;
   }

   case 0x1AD: // stxvll (Store VSX Vector Left-justified with length XX1-form)
   {
      UInt ea_off = 0;
      IRExpr* irx_addr;
      IRTemp word0[5];
      IRTemp word1[5];
      IRTemp word2[5];
      IRTemp word3[5];
      IRTemp shift = newTemp(Ity_I8);
      IRTemp nb_gt16 = newTemp(Ity_I8);
      IRTemp nb_zero = newTemp(Ity_V128);
      IRTemp nb = newTemp(Ity_I8);
      IRTemp nb_field = newTemp(Ity_I64);
      IRTemp n_bytes = newTemp(Ity_I8);
      IRTemp base_addr = newTemp( ty );
      IRTemp current_mem = newTemp(Ity_V128);
      IRTemp store_val = newTemp(Ity_V128);
      IRTemp nb_mask = newTemp(Ity_V128);
      IRTemp mask = newTemp( Ity_I64 );
      IRTemp byte[16];
      IRTemp tmp_low[9];
      IRTemp tmp_hi[9];
      IRTemp nb_field_compare_zero = newTemp( Ity_I64 );
      Int i;

      DIP("stxvll %u,r%u,r%u\n", (UInt)XS, rA_addr, rB_addr);

      assign( nb_field, binop( Iop_Shr64,
                               getIReg(rB_addr),
                               mkU8( 56 ) ) );
      assign( nb, unop( Iop_64to8, mkexpr( nb_field ) ) );
      assign( mask, mkU64( 0xFFFFFFFFFFFFFFFFULL ) );

      /* nb_gt16 will be all zeros if nb > 16 */
      assign( nb_gt16, unop( Iop_1Sto8,
                             binop( Iop_CmpLT64U,
                                    binop( Iop_Shr64,
                                           mkexpr( nb_field ),
                                           mkU8( 4 ) ),
                                    mkU64( 1 ) ) ) );

      assign( nb_field_compare_zero, unop( Iop_1Sto64,
                                           binop( Iop_CmpEQ64,
                                                  mkexpr( nb_field ),
                                                  mkU64( 0 ) ) ) );

      /* nb_zero is 0xFF..FF if the nb_field = 0 */
      assign( nb_zero, binop( Iop_64HLtoV128,
                              mkexpr( nb_field_compare_zero ),
                              mkexpr( nb_field_compare_zero ) ) );


      /* set n_bytes to 0 if nb >= 16.  Otherwise, set to nb. */
      assign( n_bytes, binop( Iop_And8, mkexpr( nb ), mkexpr( nb_gt16 ) ) );
      assign( shift,
              unop( Iop_64to8,
                    binop( Iop_Mul64,
                           binop( Iop_Sub64,
                                  mkU64( 16 ),
                                  unop( Iop_8Uto64, mkexpr( n_bytes ) )),
                           mkU64( 8 ) ) ) );

      /* We only have a 32-bit store function. So, need to fetch the
       * contents of memory merge with the store value and do two
       * 32-byte stores so we preserve the contents of memory not
       * addressed by nb.
       */
      assign( base_addr, ea_rAor0( rA_addr ) );
      /* fetch all 16 bytes and store in Big Endian format */
      word0[0] = newTemp(Ity_I64);
      assign( word0[0], mkU64( 0 ) );

      word1[0] = newTemp(Ity_I64);
      assign( word1[0], mkU64( 0 ) );

      word2[0] = newTemp(Ity_I64);
      assign( word2[0], mkU64( 0 ) );

      word3[0] = newTemp(Ity_I64);
      assign( word3[0], mkU64( 0 ) );

      for (i = 0; i < 4; i++) {
         word0[i+1] = newTemp(Ity_I64);

         irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( base_addr ),
                           ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );
         ea_off += 1;

         /* Instruction always loads in Big Endian format */
         assign( word0[i+1],
                 binop( Iop_Or64,
                        binop( Iop_Shl64,
                               unop( Iop_8Uto64,
                                     load( Ity_I8,
                                           irx_addr ) ),
                               mkU8( (3-i)*8 ) ),
                        mkexpr( word0[i] ) ) );
      }

      for (i = 0; i < 4; i++) {
         word1[i+1] = newTemp(Ity_I64);

         irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( base_addr ),
                           ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );
         ea_off += 1;

         /* Instruction always loads in Big Endian format */
         assign( word1[i+1],
                 binop( Iop_Or64,
                        binop( Iop_Shl64,
                               unop( Iop_8Uto64,
                                     load( Ity_I8,
                                           irx_addr ) ),
                               mkU8( (3-i)*8 ) ),
                        mkexpr( word1[i] ) ) );
      }
      for (i = 0; i < 4; i++) {
         word2[i+1] = newTemp(Ity_I64);

         irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( base_addr ),
                           ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );
         ea_off += 1;

         /* Instruction always loads in Big Endian format */
         assign( word2[i+1],
                 binop( Iop_Or64,
                        binop( Iop_Shl64,
                               unop( Iop_8Uto64,
                                     load( Ity_I8,
                                           irx_addr ) ),
                               mkU8( (3-i)*8 ) ),
                        mkexpr( word2[i] ) ) );
      }
      for (i = 0; i < 4; i++) {
         word3[i+1] = newTemp(Ity_I64);

         irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( base_addr ),
                           ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );
         ea_off += 1;

         /* Instruction always loads in Big Endian format */
         assign( word3[i+1],
                 binop( Iop_Or64,
                        binop( Iop_Shl64,
                               unop( Iop_8Uto64,
                                     load( Ity_I8,
                                           irx_addr ) ),
                               mkU8( (3-i)*8 ) ),
                        mkexpr( word3[i] ) ) );
      }


      assign( current_mem,
              binop( Iop_64HLtoV128,
                     binop( Iop_Or64,
                            binop( Iop_Shl64,
                                   mkexpr( word0[4] ),
                                   mkU8( 32 ) ),
                            mkexpr( word1[4] ) ),
                     binop( Iop_Or64,
                            binop( Iop_Shl64,
                                   mkexpr( word2[4] ),
                                   mkU8( 32 ) ),
                            mkexpr( word3[4] ) ) ) );

      /* Set the nb_mask to all zeros if nb = 0 so the current contents
       * of memory get written back without modifications.
       *
       * The store_val is a combination of the current memory value
       * and the bytes you want to store.  The nb_mask selects the
       * bytes you want stored from Vs.
       */
      /* The instruction always uses Big Endian order */
      assign( nb_mask,
              binop( Iop_OrV128,
                     binop( Iop_AndV128,
                            binop( Iop_ShlV128,
                                   binop( Iop_ShrV128,
                                          binop( Iop_64HLtoV128,
                                                 mkexpr( mask ),
                                                 mkexpr( mask ) ),
                                          mkexpr( shift ) ),
                                   mkexpr( shift ) ),
                            unop( Iop_NotV128, mkexpr( nb_zero ) ) ),
                     binop( Iop_AndV128,
                            mkexpr( nb_zero ),
                            binop( Iop_64HLtoV128,
                                   mkU64( 0x0 ),
                                   mkU64( 0x0 ) ) ) ) );

      assign( store_val,
              binop( Iop_OrV128,
                     binop( Iop_AndV128,
                            mkexpr( vS ),
                            mkexpr( nb_mask ) ),
                     binop( Iop_AndV128,
                            unop( Iop_NotV128, mkexpr( nb_mask ) ),
                            mkexpr( current_mem) ) ) );

      /* store the merged value in Big Endian format */
      tmp_low[0] = newTemp(Ity_I64);
      tmp_hi[0] = newTemp(Ity_I64);
      assign( tmp_low[0], mkU64( 0 ) );
      assign( tmp_hi[0], mkU64( 0 ) );

      for (i = 0; i < 8; i++) {
         byte[i] = newTemp(Ity_I64);
         byte[i+8] = newTemp(Ity_I64);
         tmp_low[i+1] = newTemp(Ity_I64);
         tmp_hi[i+1]  = newTemp(Ity_I64);

         assign( byte[i], binop( Iop_And64,
                                 binop( Iop_Shr64,
                                        unop( Iop_V128HIto64,
                                              mkexpr( store_val ) ),
                                        mkU8( (7-i)*8 ) ),
                                 mkU64( 0xFF ) ) );
         assign( byte[i+8], binop( Iop_And64,
                                   binop( Iop_Shr64,
                                          unop( Iop_V128to64,
                                                mkexpr( store_val ) ),
                                          mkU8( (7-i)*8 ) ),
                                   mkU64( 0xFF ) ) );

         assign( tmp_low[i+1],
                 binop( Iop_Or64,
                        mkexpr( tmp_low[i] ),
                        binop( Iop_Shl64, mkexpr( byte[i] ), mkU8( i*8 ) ) ) );
         assign( tmp_hi[i+1],
                 binop( Iop_Or64,
                        mkexpr( tmp_hi[i] ),
                        binop( Iop_Shl64, mkexpr( byte[i+8] ),
                               mkU8( i*8 ) ) ) );
      }

      /* Store the value in 32-byte chunks */
      ea_off = 0;
      irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( base_addr ),
                        ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );

      store( irx_addr, unop( Iop_64to32, mkexpr( tmp_low[8] ) ) );

      ea_off += 4;
      irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( base_addr ),
                        ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );

      store( irx_addr, unop( Iop_64HIto32, mkexpr( tmp_low[8] ) ) );

      ea_off += 4;
      irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( base_addr ),
                        ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );

      store( irx_addr, unop( Iop_64to32, mkexpr( tmp_hi[8] ) ) );

      ea_off += 4;
      irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( base_addr ),
                        ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );

      store( irx_addr, unop( Iop_64HIto32, mkexpr( tmp_hi[8] ) ) );

      break;
      }

   case 0x28C:
   {
      IRTemp high64 = newTemp(Ity_F64);
      IRTemp val32  = newTemp(Ity_I32);
      DIP("stxsspx %u,r%u,r%u\n", (UInt)XS, rA_addr, rB_addr);
      assign(high64, unop( Iop_ReinterpI64asF64,
                           unop( Iop_V128HIto64, mkexpr( vS ) ) ) );
      assign(val32, unop( Iop_ReinterpF32asI32,
                          unop( Iop_TruncF64asF32,
                                mkexpr(high64) ) ) );
      store( mkexpr( EA ), mkexpr( val32 ) );
      break;
   }
   case 0x2CC:
   {
      IRExpr * high64;
      DIP("stxsdx %u,r%u,r%u\n", (UInt)XS, rA_addr, rB_addr);
      high64 = unop( Iop_V128HIto64, mkexpr( vS ) );
      store( mkexpr( EA ), high64 );
      break;
   }

   case 0x38D: // stxsibx
   {
      IRTemp byte_to_store = newTemp( Ity_I8 );

      DIP("stxsibx %u,r%u,r%u\n", (UInt)XS, rA_addr, rB_addr);

      assign( byte_to_store, unop( Iop_64to8,
                                    unop( Iop_V128HIto64,
                                          mkexpr( vS ) ) ) );

      store( mkexpr( EA ), mkexpr( byte_to_store ) );
      break;
   }

   case 0x3AD: // stxsihx
   {
      IRTemp hword_to_store = newTemp( Ity_I16 );

      DIP("stxsihx %u,r%u,r%u\n", (UInt)XS, rA_addr, rB_addr);

      assign( hword_to_store, unop( Iop_64to16,
                                    unop( Iop_V128HIto64,
                                          mkexpr( vS ) ) ) );

      store( mkexpr( EA ), mkexpr( hword_to_store ) );
      break;
   }

   case 0x3CC:
   {
      IRExpr * high64, *low64;
      DIP("stxvd2x %u,r%u,r%u\n", (UInt)XS, rA_addr, rB_addr);
      high64 = unop( Iop_V128HIto64, mkexpr( vS ) );
      low64 = unop( Iop_V128to64, mkexpr( vS ) );
      store( mkexpr( EA ), high64 );
      store( binop( mkSzOp( ty, Iop_Add8 ), mkexpr( EA ),
                    ty == Ity_I64 ? mkU64( 8 ) : mkU32( 8 ) ), low64 );
      break;
   }
   case 0x38C:
   {
      UInt ea_off = 0;
      IRExpr* irx_addr;
      IRTemp hi64 = newTemp( Ity_I64 );
      IRTemp lo64 = newTemp( Ity_I64 );

      DIP("stxvw4x %u,r%u,r%u\n", (UInt)XS, rA_addr, rB_addr);

      // This instruction supports word-aligned stores, so EA may not be
      // quad-word aligned.  Therefore, do 4 individual word-size stores.
      assign( hi64, unop( Iop_V128HIto64, mkexpr( vS ) ) );
      assign( lo64, unop( Iop_V128to64, mkexpr( vS ) ) );
      store( mkexpr( EA ), unop( Iop_64HIto32, mkexpr( hi64 ) ) );
      ea_off += 4;
      irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( EA ),
                        ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );
      store( irx_addr, unop( Iop_64to32, mkexpr( hi64 ) ) );
      ea_off += 4;
      irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( EA ),
                        ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );
      store( irx_addr, unop( Iop_64HIto32, mkexpr( lo64 ) ) );
      ea_off += 4;
      irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( EA ),
                        ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );
      store( irx_addr, unop( Iop_64to32, mkexpr( lo64 ) ) );

      break;
   }
   case 0x3AC: // stxvh8x Store VSX Vector Halfword*8 Indexed
   {
      UInt ea_off = 0;
      IRExpr* irx_addr;
      IRTemp half_word0 = newTemp( Ity_I64 );
      IRTemp half_word1 = newTemp( Ity_I64 );
      IRTemp half_word2 = newTemp( Ity_I64 );
      IRTemp half_word3 = newTemp( Ity_I64 );
      IRTemp half_word4 = newTemp( Ity_I64 );
      IRTemp half_word5 = newTemp( Ity_I64 );
      IRTemp half_word6 = newTemp( Ity_I64 );
      IRTemp half_word7 = newTemp( Ity_I64 );

      DIP("stxvb8x %u,r%u,r%u\n", (UInt)XS, rA_addr, rB_addr);

      assign( half_word0, binop( Iop_Shr64,
                                 unop( Iop_V128HIto64, mkexpr( vS ) ),
                                 mkU8( 48 ) ) );

      assign( half_word1, binop( Iop_And64,
                                 binop( Iop_Shr64,
                                        unop( Iop_V128HIto64, mkexpr( vS ) ),
                                        mkU8( 32 ) ),
                                 mkU64( 0xFFFF ) ) );

      assign( half_word2, binop( Iop_And64,
                                 binop( Iop_Shr64,
                                        unop( Iop_V128HIto64, mkexpr( vS ) ),
                                        mkU8( 16 ) ),
                                 mkU64( 0xFFFF ) ) );

      assign( half_word3, binop( Iop_And64,
                                 unop( Iop_V128HIto64, mkexpr( vS ) ),
                                 mkU64( 0xFFFF ) ) );

      assign( half_word4, binop( Iop_Shr64,
                                 unop( Iop_V128to64, mkexpr( vS ) ),
                                 mkU8( 48 ) ) );

      assign( half_word5, binop( Iop_And64,
                                 binop( Iop_Shr64,
                                        unop( Iop_V128to64, mkexpr( vS ) ),
                                        mkU8( 32 ) ),
                                 mkU64( 0xFFFF ) ) );

      assign( half_word6, binop( Iop_And64,
                                 binop( Iop_Shr64,
                                        unop( Iop_V128to64, mkexpr( vS ) ),
                                        mkU8( 16 ) ),
                                 mkU64( 0xFFFF ) ) );

      assign( half_word7, binop( Iop_And64,
                                 unop( Iop_V128to64, mkexpr( vS ) ),
                                 mkU64( 0xFFFF ) ) );

      /* Do the 32-bit stores.  The store() does an Endian aware store. */
      if ( host_endness == VexEndnessBE ) {
         store( mkexpr( EA ), unop( Iop_64to32,
                                    binop( Iop_Or64,
                                           mkexpr( half_word1 ),
                                           binop( Iop_Shl64,
                                                  mkexpr( half_word0 ),
                                                  mkU8( 16 ) ) ) ) );

         ea_off += 4;
         irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( EA ),
                           ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );


         store( irx_addr, unop( Iop_64to32,
                                binop( Iop_Or64,
                                       mkexpr( half_word3 ),
                                       binop( Iop_Shl64,
                                              mkexpr( half_word2 ),
                                              mkU8( 16 ) ) ) ) );

         ea_off += 4;
         irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( EA ),
                           ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );

         store( irx_addr, unop( Iop_64to32,
                                binop( Iop_Or64,
                                       mkexpr( half_word5 ),
                                       binop( Iop_Shl64,
                                              mkexpr( half_word4 ),
                                              mkU8( 16 ) ) ) ) );
         ea_off += 4;
         irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( EA ),
                           ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );

         store( irx_addr, unop( Iop_64to32,
                                binop( Iop_Or64,
                                       mkexpr( half_word7 ),
                                       binop( Iop_Shl64,
                                              mkexpr( half_word6 ),
                                              mkU8( 16 ) ) ) ) );

      } else {
         store( mkexpr( EA ), unop( Iop_64to32,
                                    binop( Iop_Or64,
                                           mkexpr( half_word0 ),
                                           binop( Iop_Shl64,
                                                  mkexpr( half_word1 ),
                                                  mkU8( 16 ) ) ) ) );

         ea_off += 4;
         irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( EA ),
                           ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );

         store( irx_addr, unop( Iop_64to32,
                                binop( Iop_Or64,
                                       mkexpr( half_word2 ),
                                       binop( Iop_Shl64,
                                              mkexpr( half_word3 ),
                                              mkU8( 16 ) ) ) ) );
         ea_off += 4;
         irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( EA ),
                           ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );

         store( irx_addr, unop( Iop_64to32,
                                binop( Iop_Or64,
                                       mkexpr( half_word4 ),
                                       binop( Iop_Shl64,
                                              mkexpr( half_word5 ),
                                              mkU8( 16 ) ) ) ) );
         ea_off += 4;
         irx_addr = binop( mkSzOp( ty, Iop_Add8 ), mkexpr( EA ),
                           ty == Ity_I64 ? mkU64( ea_off ) : mkU32( ea_off ) );

         store( irx_addr, unop( Iop_64to32,
                                binop( Iop_Or64,
                                       mke