/*
 * LaTeD Version 1.1
 * (c) Gene Ressler 1993, 94, 97
 *   de8827@trotter.usma.edu
 *
 * LaTeD is a graphical editor for drawings in the LaTeX "picture" 
 * environment.  It runs under MSDOS or in a Windows DOS box.  The
 * distribution includes full sources, including LaTeX source for 
 * its documentation.
 *
 * No warranty of this software is expressed or implied by the author.
 *
 * Copy and use this program freely for any purpose except for sale
 * of the program (including the source code) itself.  That is, 
 * no one can copy this program for the purpose of providing it to 
 * another person in exchange for money or other compensation, even 
 * if this program is only part of the exchange.
 *
 * All copies of computer source code in this distribution, whether
 * copies in whole or in part, must have this notice attached.
 */

/* ENV.C --- LWW environment for the editor TED. */

#include <stdio.h>
#include <stdlib.h>
#include <dir.h>
#include <graphics.h>
#include <alloc.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#include "window.h"
#include "settings.h"
#include "canvas.h"
#include "env.h"
#include "resource.h"
#include "key.h"

/* ----- Canvas ----------------------------------------------------- */

void handle_pick_change(DISPLAY_LIST dl, ENV env);

DefCanvas(edit_canvas, 
	0, 0,           /* Dummy width and height. */
	23000,          /* Size (chosen so diagonal is < 32767) */
	32*(588-1)+1,   /* Viewport fully zoomed out. */
	0, 0,           /* Initially drawing in low left corner. */
	10,             /* em size in pts */
	4.3,            /* ex size in pts */
	1,              /* One canvas element in mm's (also unitlength) */
	cBLUE,          /* Background color */
	cLIGHTCYAN,     /* Thin line color */
	cWHITE,         /* Thick object color */
	cLIGHTRED,      /* Pick color */
	cCYAN,          /* Ruler, origin, edit pick color */
	DisplayListAction(handle_pick_change, NullEnv),
	BottomScrollbar|RightScrollbar)

/* ----- Pulldown menus. -------------------------------------------- */

static void handle_show_about_message(void);
static void handle_show_info(void);
static void handle_redraw(void);
static void handle_redraw_all(void);

BeginDefMenu(about_menu)
  DefMenuEntry("About...",   0,  NoHotKey, Enabled, ActionClosure(handle_show_about_message, NullEnv))
  DefMenuEntry("Info",       0,  ALT_I,    Enabled, ActionClosure(handle_show_info, NullEnv))
  DefMenuEntry("Redraw",     0,  ALT_R,    Enabled, ActionClosure(handle_redraw, NullEnv))
  DefMenuEntry("Refresh all",2,  NoHotKey, Enabled, ActionClosure(handle_redraw_all, NullEnv))
EndDefMenu(about_menu, 18)

static void handle_open(void);
static void handle_merge(void);
static void handle_new(void);
static void handle_save(void);
static void handle_save_as(void);
static void handle_stop(void);
BeginDefMenu(file_menu)
  DefMenuEntry("New",     0, NoHotKey, Enabled, ActionClosure(handle_new, NullEnv))
  DefMenuEntry("Open",    0, ctrl('O'),Enabled, ActionClosure(handle_open, NullEnv))
  DefMenuEntry("Merge",   0, NoHotKey, Enabled, ActionClosure(handle_merge, NullEnv))
  DefMenuEntry("Save",    0, ctrl('S'),Enabled, ActionClosure(handle_save, NullEnv))
  DefMenuEntry("Save as", 1, NoHotKey, Enabled|Ruled, ActionClosure(handle_save_as, NullEnv))
  DefMenuEntry("Quit",    0, ALT_X,    Enabled, ActionClosure(handle_stop, NullEnv))
EndDefMenu(file_menu, 16)

static void handle_meta_pick(ENV env);
static void handle_displacement(ENV env);

BeginDefMenu(pick_menu)
  DefMenuEntry("Pick all",   0, ALT_P, NoFlags, Closure(handle_meta_pick, val_one))
  DefMenuEntry("Clear pick", 0, ALT_C, NoFlags, Closure(handle_meta_pick, val_zero))
EndDefMenu(pick_menu, 18)

BeginDefMenu(edit_menu)
  DefSubmenuEntry("Pick", 0, pick_menu, Enabled)
  DefMenuEntry("Zoom out", 0, CTRL_HOME,  Ruled, ActionClosure(reset_zoom, NullEnv))
  DefMenuEntry("Repeat Copy", 0, ALT_2, NoFlags, ActionClosure(handle_repeat_copy, edit_canvas))
  DefMenuEntry("Displacement", 0, ALT_D, Enabled, ActionClosure(handle_displacement, NullEnv))
EndDefMenu(edit_menu, 20)

BeginDefPulldownBar(pulldown_bar)
  DefPulldownEntry("\xf", HotSpace, about_menu)
  DefPulldownEntry("File", 0, file_menu)
  DefPulldownEntry("Edit", 0, edit_menu)
  DefPulldownEntry("Options", 0, options_menu)
EndDefPulldownBar(pulldown_bar)

/* ----- Tool box --------------------------------------------------- */

#include "bitmap\pick.c"
#include "bitmap\unpick.c"
#include "bitmap\cyclpick.c"
#include "bitmap\copy.c"
#include "bitmap\move.c"
#include "bitmap\del.c"
#include "bitmap\modify.c"
#include "bitmap\text.c"
#include "bitmap\polyline.c"
#include "bitmap\vector.c"
#include "bitmap\rect.c"
#include "bitmap\dashrect.c"
#include "bitmap\fillrect.c"
#include "bitmap\oval.c"
#include "bitmap\oval2.c"
#include "bitmap\oval4.c"
#include "bitmap\circ.c"
#include "bitmap\fillcirc.c"
#include "bitmap\origin.c"
#include "bitmap\zoom.c"
#include "bitmap\undo.c"
#include "bitmap\chgthick.c"

static void handle_select(HANDLER_CODE handler);

#define TS_SHADOW_WIDTH 2
BeginDefSelector(tool_selector, 1)
  DefBitmapSelectorButton(tool_selector, pick,     F1,       TS_SHADOW_WIDTH, Enabled|Toggle|Selector, ActionClosure(handle_select, handle_pick))
  DefBitmapSelectorButton(tool_selector, unpick,   F2,       TS_SHADOW_WIDTH, Enabled|Toggle|Selector, ActionClosure(handle_select, handle_unpick)) 
  DefBitmapSelectorButton(tool_selector, cyclpick, F3,       TS_SHADOW_WIDTH, Repeat,                  ActionClosure(handle_cycle_pick, edit_canvas))
  DefBitmapSelectorButton(tool_selector, copy,     F4,       TS_SHADOW_WIDTH, Enabled|Toggle|Selector, ActionClosure(handle_select, handle_copy))
  DefBitmapSelectorButton(tool_selector, move,     F5,       TS_SHADOW_WIDTH, Enabled|Toggle|Selector, ActionClosure(handle_select, handle_move))
  DefBitmapSelectorButton(tool_selector, del,      F6,       TS_SHADOW_WIDTH, Enabled|Toggle|Selector, ActionClosure(handle_select, handle_delete))
  DefBitmapSelectorButton(tool_selector, modify,   F7,       TS_SHADOW_WIDTH, Enabled|Toggle|Selector, ActionClosure(handle_select, handle_modify))
  DefBitmapSelectorButton(tool_selector, text,     F8,       TS_SHADOW_WIDTH, Enabled|Toggle|Selector, ActionClosure(handle_select, handle_text))
  DefBitmapSelectorButton(tool_selector, polyline, F9,       TS_SHADOW_WIDTH, Enabled|Toggle|Selector, ActionClosure(handle_select, handle_line))
  DefBitmapSelectorButton(tool_selector, vector,   F10,      TS_SHADOW_WIDTH, Enabled|Toggle|Selector, ActionClosure(handle_select, handle_vector))
  DefBitmapSelectorButton(tool_selector, rect,     SHIFT_F1, TS_SHADOW_WIDTH, Enabled|Toggle|Selector, ActionClosure(handle_select, handle_rect))
  DefBitmapSelectorButton(tool_selector, dashrect, SHIFT_F2, TS_SHADOW_WIDTH, Enabled|Toggle|Selector, ActionClosure(handle_select, handle_dashrect))
  DefBitmapSelectorButton(tool_selector, fillrect, SHIFT_F3, TS_SHADOW_WIDTH, Enabled|Toggle|Selector, ActionClosure(handle_select, handle_fillrect))
  DefBitmapSelectorButton(tool_selector, oval,     SHIFT_F4, TS_SHADOW_WIDTH, Enabled|Toggle|Selector, ActionClosure(handle_select, handle_oval))
  DefBitmapSelectorButton(tool_selector, oval2,    SHIFT_F5, TS_SHADOW_WIDTH, Enabled|Toggle|Selector, ActionClosure(handle_select, handle_oval2))
  DefBitmapSelectorButton(tool_selector, oval4,    SHIFT_F6, TS_SHADOW_WIDTH, Enabled|Toggle|Selector, ActionClosure(handle_select, handle_oval4))
  DefBitmapSelectorButton(tool_selector, circ,     SHIFT_F7, TS_SHADOW_WIDTH, Enabled|Toggle|Selector, ActionClosure(handle_select, handle_circle))
  DefBitmapSelectorButton(tool_selector, fillcirc, SHIFT_F8, TS_SHADOW_WIDTH, Enabled|Toggle|Selector, ActionClosure(handle_select, handle_fillcircle))
  DefBitmapSelectorButton(tool_selector, origin,   SHIFT_F9, TS_SHADOW_WIDTH, Enabled|Toggle|Selector, ActionClosure(handle_select, handle_origin))
  DefBitmapSelectorButton(tool_selector, zoom,     SHIFT_F10,TS_SHADOW_WIDTH, Enabled|Toggle|Selector, ActionClosure(handle_select, handle_zoom))
  DefBitmapSelectorButton(tool_selector, undo,     ESC,      TS_SHADOW_WIDTH, 0,                       ActionClosure(undo_last_command, NullEnv))
  DefBitmapSelectorButton(tool_selector, chgthick, ALT_F8,   TS_SHADOW_WIDTH, Enabled|Toggle|Selector, ActionClosure(handle_select, handle_change_thickness))
EndDefSelector(tool_selector, Vertical|Enabled)

#include "bitmap\thick.c"
#include "bitmap\thin.c"

static LINE_THICKNESS lt_thick[1] = { ltTHICK }, lt_thin[1] = { ltTHIN };
BeginDefSelector(line_selector, 1)
  DefBitmapSelectorButton(line_selector, thick, ALT_F9,   1, Enabled|Toggle|Selector, ActionClosure(handle_thickness, lt_thick))
  DefBitmapSelectorButton(line_selector, thin,  ALT_F10,  1, On|Enabled|Toggle|Selector, ActionClosure(handle_thickness, lt_thin))
EndDefSelector(line_selector, Enabled)

/* ----- Text edit box ---------------------------------------------- */

#include "bitmap\ljust.c"
#include "bitmap\rjust.c"
#include "bitmap\hcjust.c"
#include "bitmap\tjust.c"
#include "bitmap\bjust.c"
#include "bitmap\vcjust.c"

static void validate_text_edit_edit(char *edit_str, ENV env);
static void handle_text_edit_button(int *ok_p);
static void handle_text_edit_key_close(char *str, ENV env);
static void set_hjust(int *just);
static void set_vjust(int *just);

BeginDefEditBox(text_edit_box)
  DefValidator(NameActionClosure(validate_text_edit_edit, val_zero))
  DefKeyAction('\r', NameActionClosure(handle_text_edit_key_close, NullEnv))
EndDefEditBox(text_edit_box, NoTitle, NoHotChar, 40, MAX_TEXT, Enabled)

static int just[] = { LEFT_TEXT, CENTER_TEXT, RIGHT_TEXT };
		   /* BOTTOM_TEXT, CENTER_TEXT, TOP_TEXT */
BeginDefSelector(hjust_selector, 1)
  DefBitmapSelectorButton(hjust_selector, ljust, CTRL_LEFT_ARROW,  1, On|Enabled|Toggle|Selector, ActionClosure(set_hjust, &just[LEFT_TEXT]))
  DefBitmapSelectorButton(hjust_selector,hcjust, CTRL_PGUP,        1,    Enabled|Toggle|Selector, ActionClosure(set_hjust, &just[CENTER_TEXT]))
  DefBitmapSelectorButton(hjust_selector, rjust, CTRL_RIGHT_ARROW, 1,    Enabled|Toggle|Selector, ActionClosure(set_hjust, &just[RIGHT_TEXT]))
EndDefSelector(hjust_selector, Vertical|Enabled)

BeginDefSelector(vjust_selector, 1)
  DefBitmapSelectorButton(vjust_selector, bjust, CTRL_END,  1, On|Enabled|Toggle|Selector, ActionClosure(set_vjust, &just[BOTTOM_TEXT]))
  DefBitmapSelectorButton(vjust_selector,vcjust, CTRL_PGDN, 1,    Enabled|Toggle|Selector, ActionClosure(set_vjust, &just[CENTER_TEXT]))
  DefBitmapSelectorButton(vjust_selector, tjust, CTRL_HOME, 1,    Enabled|Toggle|Selector, ActionClosure(set_vjust, &just[TOP_TEXT]))
EndDefSelector(vjust_selector, Enabled)

DefBitmapButton(edit_ok,     ok,     'k', 2, 2, Enabled, ActionClosure(handle_text_edit_button, val_one))
DefBitmapButton(edit_cancel, cancel, ESC, 2, 2, Enabled, ActionClosure(handle_text_edit_button, val_zero))

enum { TEXT_EDIT_BOX, TEXT_HJUST, TEXT_VJUST, TEXT_OK, TEXT_CANCEL };
BeginDefDialog(text_edit_dialog)
  DefDialogItem(EditBoxItem, text_edit_box, 36, 8)
  DefDialogItem(SelectorItem, hjust_selector, 8, 8)
  DefDialogItem(SelectorItem, vjust_selector, 34, 38)
  DefDialogItem(ButtonItem, edit_ok, 356, 32)
  DefDialogItem(ButtonItem, edit_cancel, 306, 32)
EndDefDialog(text_edit_dialog, "Edit Text", Enabled)

/* ----- Set displacement edit box ---------------------------------- */
static void handle_displacement_button(int *ok_p);
static void handle_displacement_key_close(char* str, ENV env);
static void validate_displacement_edit(char *str, EDIT_BOX edit_box);
static void handle_clear_button(ENV env);

DefTextButton(displacement_clear, "Clear", 0, NoHotKey, 0, 0, 2, 1, Enabled, ActionClosure(handle_clear_button, NullEnv))

BeginDefEditBox(displacement_x_edit_box)
  DefValidator(NameActionClosure(validate_displacement_edit, displacement_x_edit_box))
  DefKeyAction('\r', NameActionClosure(surrender_focus_action, displacement_x_edit_box))
EndDefEditBox(displacement_x_edit_box, "X", 0, 10, 10, Enabled)

BeginDefEditBox(displacement_y_edit_box)
  DefValidator(NameActionClosure(validate_displacement_edit, displacement_y_edit_box))
  DefKeyAction('\r', NameActionClosure(handle_displacement_key_close, displacement_y_edit_box))
EndDefEditBox(displacement_y_edit_box, "Y", 0, 10, 10, Enabled)

DefBitmapButton(displacement_ok,     ok,     'k', 2, 2, Enabled, ActionClosure(handle_displacement_button, val_one))
DefBitmapButton(displacement_cancel, cancel, ESC, 2, 2, Enabled, ActionClosure(handle_displacement_button, val_zero))

enum { DISPLACEMENT_X_EDIT_BOX, DISPLACEMENT_Y_EDIT_BOX, EDIT_OK, EDIT_CANCEL };
BeginDefDialog(displacement_dialog)
  DefDialogItem(EditBoxItem, displacement_x_edit_box, 8, 16)
  DefDialogItem(EditBoxItem, displacement_y_edit_box, 128, 16)
  DefDialogItem(ButtonItem, displacement_clear, 8, 42)
  DefDialogItem(ButtonItem, displacement_ok, 217, 40)
  DefDialogItem(ButtonItem, displacement_cancel, 167, 40)
EndDefDialog(displacement_dialog, "Set Copy Displacement", Enabled)

/* ----- Top level windows of the environment ----------------------- */

void move_tools(int right_p)
{
  int y_pb_bottom, x0, x1;

  y_pb_bottom = pulldown_bar->window.height + pulldown_bar->window.border_width;

  if (right_p) {
    x0 = root_window->width;
    x1 = 0;
    edit_canvas->status &= notbit(vRIGHT_SCROLLBAR);
  }
  else {
    x0 = 0;
    x1 = root_window->width;
    edit_canvas->status |= bit(vRIGHT_SCROLLBAR);
  }
  protect_focus();
  unmap_window(&edit_canvas->window);
  unmap_window(&tool_selector->window);
  unmap_window(&line_selector->window);
  locate_canvas_scrollbars(edit_canvas);
  locate_window(&tool_selector->window, x0, y_pb_bottom, 1);
  locate_window(&line_selector->window, x0, 
		window_y(&tool_selector->window) + 
		tool_selector->window.height + 
		tool_selector->window.border_width, 1);
  locate_window(&edit_canvas->window, x1, y_pb_bottom, 0);
  map_window(&line_selector->window);
  map_window(&edit_canvas->window);
  map_window(&tool_selector->window);
  unprotect_focus();
}

/* ===== Handlers =================================================== */

int edited_p = 0;

/* ----- Pulldown menus. -------------------------------------------- */

LOCAL(void) save_edit_file(void)
{
  char buf[MAXPATH], ext[MAXEXT];

  fnsplit(settings->edit_path, NULL, NULL, NULL, ext);
  if (settings->status & bit(sBACKUP_FILE) && stricmp(ext, ".bak") != 0) {
    unlink(change_suffix(settings->edit_path, ".bak", buf));
    rename(settings->edit_path, buf);
  }

  /* Write using the current global settings. */
  set_wait_cursor(1);
  write_objects(edit_canvas, settings->edit_path, settings, 0);
  set_wait_cursor(0);
  edited_p = 0;
}

void do_emergency_save(char* filename)
{
  if (!filename)
    filename = settings->edit_path;
  write_objects(edit_canvas, filename, settings, 1);
}

LOCAL(int) do_save_query(void)
{
  MESSAGE_BUTTON b;

  if (edited_p) {
    b = message(bit(mYES) | bit(mNO) | bit(mCANCEL), root_window, 
		   "File `%s'\nis not saved.\nSave now?", settings->edit_path);
    if (b == mYES)
      save_edit_file();
    else if (b == mCANCEL)
      return 0;
    edited_p = 0;
  }
  return 1;
}

LOCAL(int) set_edit_file(char *path)
{
  if (!do_save_query())
    return 0;
  clear_undo();
  clear_pick_buf();
  clear_canvas(edit_canvas);
  reset_zoom();
  set_edit_path(settings, path);
  return 1;
}

LOCAL(int) read_file(char *path)
{
  if (read_objects(edit_canvas, path, env_oval_rad, settings)) {
    settings2output_dialog(settings);
    return 1;
  }
  message(bit(mOK), root_window, "Error parsing\n`%s'", path);
  return 0;
}

static void handle_merge(void)
{
  char path[MAXPATH] = "*" DEFAULT_EXT;
  int r = get_path("Merge File", path, DEFAULT_EXT);

  if (r == 2)
    return;
  if (edit_canvas->dl.n_picked > 0) {
    MESSAGE_BUTTON b;

    b = message(bit(mYES) | bit(mNO) | bit(mCANCEL), root_window, 
		"Clear all picks before merge?\n");
    if (b == mYES)
      pick_all(edit_canvas, 0);
    else if (b == mCANCEL)
      return;
  }
  clear_undo();
  clear_pick_buf();
  reset_zoom();
  read_objects(edit_canvas, path, env_oval_rad, 0);
}

static void handle_open(void)
{
  char path[MAXPATH] = "*" DEFAULT_EXT;
  int r = get_path("Load File", path, DEFAULT_EXT);

  /* If user cancels load or set on open path, quit. */
  if (r == 2 || !set_edit_file(path))
    return;

  /* User selected existing file.  Try to parse it. */
  if (r == 0) {
    set_wait_cursor(1);
    if (!read_file(path)) 
      clear_canvas(edit_canvas);
    set_wait_cursor(0);
  }
}

static void handle_new(void)
{
  set_edit_file(NONAME);
}

LOCAL(int) has_suffix_p(char *str, char *sfx)
{
  return strstr(str, sfx) == str + (strlen(str) - strlen(sfx));
}

static void handle_save(void)
{
  if (has_suffix_p(settings->edit_path, NONAME))
    handle_save_as();
  else
    save_edit_file();
}

static void handle_save_as(void)
{
  unsigned code;
  char path[MAXPATH];

  path[0] = '\0';
  code = get_path("Save File as", path, DEFAULT_EXT);
  if (code == 0 && strcmp(path, settings->edit_path) != 0) {
    if (message(bit(mOK)|bit(mCANCEL), root_window,
		"File `%s' exists.\nOverwrite?", path)
	== mCANCEL)
      code = 2;
  }
  if (code != 2) {
    strcpy(settings->edit_path, path);
    save_edit_file();
  }
}

LOCAL(int) gcd(int x, int y) 
{
  int tmp;
  int a = abs(x);
  int b = abs(y);

  if (b > a) {
    tmp = a; 
    a = b; 
    b = tmp;
  }
  for(;;) {
    if (b == 0)
      return a;
    else if (b == 1)
      return b;
    else
    {
      tmp = b;
      b = a % b;
      a = tmp;
    }
  }
}

static void handle_show_info(void)
{
  CC vpdx = edit_canvas->vp_x_size - 1;
  CC dwdx = edit_canvas->draw.width - 1;
  int d = gcd(vpdx, dwdx);
  OBJECT op = dereference(edit_canvas, edit_canvas->dl.origin);

  message(bit(mOK), root_window,
	  TAB_ESC "<"
	  "Edit info:\n"
	  " File: %.40s %s\n"
	  " Picks: %d\n"
	  " Zoom: %u/%u\n"
	  " Memory remaining: %luK\n"
	  " \n"
	  "Picture info:\n"
	  " Objects: %d\n"
	  " Bounding box: %s\n"
	  "  origin: %s\n"
	  "  size:   %s\n"
	  " ",

	  has_suffix_p(settings->edit_path, NONAME) ?      /* filename */
	    "[none]" : settings->edit_path,
	  edited_p ? "(unsaved)" : "",                     /* edit flag */
	  edit_canvas->dl.n_picked,                        /* pick count */
	  vpdx/d, dwdx/d,                                  /* zoom frac */
	  (unsigned long)coreleft() >> 10,                 /* memory */

	  edit_canvas->dl.n_obj,                           /* objects count */
	  op ? "" : "[none]",                              /* origin flag */
	  op ? cp2str(op->ref) : "--",                     /* bb origin */
	  op ? cp2str(op->u.origin.sz) : "--"              /* bb size */
	  );
}

static void handle_show_about_message(void)
{
  extern char version[];
  message(bit(mOK), root_window,
	  " \n"
	  "LaTeD\n"
	  "The LaTeX Picture Editor\n"
	  " \n"
	  "Version %s\n" 
	  " \n"
	  "Copyright 1992, 1993, 1997\n"
	  "by Gene Ressler\n"
	  "de8827@trotter.usma.edu\n"
	  " \n", version);
}

static void handle_redraw(void)
{
  clear_window(&edit_canvas->draw, 1);
}

static void handle_redraw_all(void)
{
  unmap_window(&pulldown_bar->window);
  unmap_window(&tool_selector->window);
  unmap_window(&edit_canvas->window);
  unmap_window(&line_selector->window);
  clear_window(root_window, 0);
  map_window(&pulldown_bar->window);
  map_window(&tool_selector->window);
  map_window(&edit_canvas->window);
  map_window(&line_selector->window);
}

static void handle_stop(void)
{
  if (do_save_query())
    stop(0);
}

static void handle_meta_pick(ENV env)
{
  pick_all(edit_canvas, *(int*)env);
}

#pragma argsused

static void handle_pick_change(DISPLAY_LIST dl, ENV env)
{
  if (dl->n_picked == 0) 
    disable_menu_entry(clear_pick_menu_entry);
  else 
    enable_menu_entry(clear_pick_menu_entry);
  if (dl->n_picked == 0) {
    set_button_toggle(&tool_selector_buttons[tsDEL], bit(bON)|bit(bSELECTOR)|bit(bTOGGLE));
    set_button_action(&tool_selector_buttons[tsDEL], handle_select, handle_delete);
    set_button_toggle(&tool_selector_buttons[tsCHG_LT], bit(bON)|bit(bSELECTOR)|bit(bTOGGLE));
    set_button_action(&tool_selector_buttons[tsCHG_LT], handle_select, handle_change_thickness);
  }
  else {
    if (b_status_p(&tool_selector_buttons[tsDEL], bON) ||
	b_status_p(&tool_selector_buttons[tsCHG_LT], bON))
      set_cursor_handlers(&edit_canvas->cursor, null_handler_code, null_handler_code);
    set_button_toggle(&tool_selector_buttons[tsDEL], bit(bSELECTOR)|bit(bTOGGLE));
    set_button_action(&tool_selector_buttons[tsDEL], handle_delete_button, edit_canvas);
    set_button_toggle(&tool_selector_buttons[tsCHG_LT], bit(bSELECTOR)|bit(bTOGGLE));
    set_button_action(&tool_selector_buttons[tsCHG_LT], handle_change_thickness_button, edit_canvas);
  }
  if (dl->n_picked < dl->n_obj)
    enable_menu_entry(pick_all_menu_entry);
  else
    disable_menu_entry(pick_all_menu_entry);
  if (dl->n_obj < settings->auto_redraw)
    enable_auto_redraw(edit_canvas);
  else
    disable_auto_redraw(edit_canvas);
  if (dl->n_picked > 0 && copy_displacement_valid_p(&edit_canvas->copy_displacement))
    enable_menu_entry(repeat_copy_menu_entry);
  else 
    disable_menu_entry(repeat_copy_menu_entry);
}

/* ----- Tool box --------------------------------------------------- */

static void handle_select(HANDLER_CODE handler)
{
  int i;
  unsigned cursor_mask;
  HANDLER_CODE mouse_cursor_actions[] = {
	handle_zoom,
	handle_pick,
	handle_unpick,
	handle_delete,
	handle_change_thickness,
	handle_modify };

  cursor_mask = env_graphic_ptr;
  for (i = 0; i < array_size(mouse_cursor_actions); ++i)
    if (handler == mouse_cursor_actions[i]) {
      cursor_mask = bit(cMOUSE);
      break;
    }
  env_set_cursor(cursor_mask);
  set_cursor_handlers(&edit_canvas->cursor, handler, handler);
}

/* ----- Text edit box ---------------------------------------------- */

int hjust = LEFT_TEXT, vjust = BOTTOM_TEXT;

LOCAL(void) init_text(void)
{
  if (!static_resource_opened_p(text_edit_dialog))
    open_dialog(text_edit_dialog, CENTER_WINDOW, CENTER_WINDOW, root_window);
}

char *get_text(char *init_str)
{
  init_text();
  if (init_str != NULL)
    set_edit_text(text_edit_box, init_str);
  ungrab_mouse();
  if (run_modal_dialog(text_edit_dialog))
    return strdup(get_edit_text(text_edit_box));
  return NULL;
}

void set_text(OBJECT box)
{
  init_text();
  set_edit_text(text_edit_box, box->u.frame_box.str);
  feign_button_press(&vjust_selector_buttons[box->u.frame_box.vjust]);
  feign_button_press(&hjust_selector_buttons[box->u.frame_box.hjust]);
}

LOCAL(int) check_text_edit_edit(char *text, unsigned msg_mask)
{
  int reply, rtn;
  static int already_checking_p = 0;

  /* Have to prevent a recursive check when message box
     causes focus to be withdrawn from edit. Another approach
     would be to put a legal string in the edit before
     invoking the message box. */
  if (already_checking_p)
    return 1;
  already_checking_p = 1;

  if (text_is_latext(text)) {
    rtn = 1;
  }
  else {
    reply = message(msg_mask, &text_edit_dialog->window, 
		    "This is questionable LaTeX.  It\n"
		    "will cause File Open or Merge\n"
		    "to fail later.\n%s",

		    (msg_mask & bit(mYES)) ? "Want to accept anyway?" : "");

    if (reply == mYES) 
      rtn = 1;
    else {
      set_focus(&text_edit_box->window);
      rtn = 0;
    }
  }
  already_checking_p = 0;
  return rtn;
}

#pragma argsused

static void validate_text_edit_edit(char *edit_str, ENV env)
{ 
  check_text_edit_edit(edit_str, bit(mOK));
}

static void handle_text_edit_button(int *ok_p) 
{ 
  if (!*ok_p || check_text_edit_edit(get_edit_text(text_edit_box), bit(mYES) | bit(mNO)))
    stop_modal_dialog(text_edit_dialog, *ok_p); 
}

#pragma argsused

static void handle_text_edit_key_close(char *str, ENV env)
{
  handle_text_edit_button(val_one);
}

static void set_hjust(int *just) { hjust = *just; }
static void set_vjust(int *just) { vjust = *just; }

/* ----- Environment focus ------------------------------------------ */

LOCAL(void) redirect_focus(EVENT e)
{
  static WINDOW fw = &edit_canvas->window;

  if (e->focus.other_window == &edit_canvas->window)
    fw = &pulldown_bar->window;
  if (e->focus.other_window == &pulldown_bar->window)
    fw = &edit_canvas->window;
  set_focus(fw);
}

BeginDefDispatch(root)
  Dispatch(eGAIN_FOCUS, redirect_focus)
EndDefDispatch(root)

/* ----- Set displacement dialog ------------------------------------ */

LOCAL(void) init_displacement(void)
{
  if (!static_resource_opened_p(displacement_dialog))
    open_dialog(displacement_dialog, CENTER_WINDOW, CENTER_WINDOW, root_window);
}

/* Convert decimal number to int*100. Return non-0 on success. */
LOCAL(int) str2cc(char *s, CC *cc)
{
  int sign, whole, whole_size, frac_size, frac;
  
  if (!s)
    return 0;

  while (isspace(*s))  
    s++;
  
  if (*s == '-') {
    s++;
    sign = -1;
  }
  else if (*s == '+') {
    s++;
    sign = 1;
  }
  else 
    sign = 1;
  
  whole_size = 0;
  whole = 0;
  while (isdigit(*s)) {
    whole = whole * 10 + (*s - '0');
    if (whole > 230) /* 230.00 is max coordinate */
      return 0;
    whole_size++;
    s++;
  }

  frac_size = 0;
  frac = 0;
  if (*s == '.') {
    s++;
    while (isdigit(*s)) {
      if (frac_size < 3) {
	frac = frac * 10 + (*s - '0');
	frac_size++;
      }
      s++;
    }
  }

  if ((whole_size == 0 && frac_size == 0) || *s != '\0')
    return 0;

  /* Make sure fraction is in 1000th's */
  while (frac_size < 3) {
    frac *= 10;
    frac_size++;
  }
  *cc = sign * (100 * whole + frac/10 + ((frac % 10) >= 5));

  return 1;
}

LOCAL(int) check_displacement_edit(EDIT_BOX edit_box)
{
  int rtn;
  CC dummy;
  char *text;
  static int already_checking_p = 0;

  if (already_checking_p)
    return 1;
  already_checking_p = 1;

  text = get_edit_text(edit_box);
  if (text[0] == '\0' || str2cc(text, &dummy))
    rtn = 1;
  else {
    message(bit(mOK), &displacement_dialog->window,
	    "`%s' is not a displacement.\n"
	    "Example: -123.45\n",
	    text);
    set_focus(&edit_box->window);
    rtn = 0;
  }
  already_checking_p = 0;
  return rtn;
}

LOCAL(int) check_empty(void)
{
  char *x_text = get_edit_text(displacement_x_edit_box);
  char *y_text = get_edit_text(displacement_y_edit_box);
  int x_empty = (x_text[0] == '\0');
  int y_empty = (y_text[0] == '\0');
  if (x_empty == y_empty)
    return 1;
  message(bit(mOK), &displacement_dialog->window,
	  "Missing %c coordinate.\n",
	  x_empty ? 'X' : 'Y');
  set_focus(x_empty ? &displacement_x_edit_box->window :
		      &displacement_y_edit_box->window);
  return 0;
}

static void handle_displacement_button(int *ok_p) 
{
  if (!*ok_p || 
      (check_displacement_edit(displacement_x_edit_box) &&
       check_displacement_edit(displacement_y_edit_box) &&
       check_empty()))
    stop_modal_dialog(displacement_dialog, *ok_p);
}

#pragma argsused

static void handle_displacement_key_close(char* str, ENV env)
{
  handle_displacement_button(val_one);
}

#pragma argsused

static void handle_clear_button(ENV env)
{
  set_edit_text(displacement_x_edit_box, "");
  set_edit_text(displacement_y_edit_box, "");
}

#pragma argsused

static void validate_displacement_edit(char *str, EDIT_BOX edit_box)
{ 
  check_displacement_edit(edit_box);
}

#pragma argsused

static void handle_displacement(ENV env)
{
  char *x_text, *y_text;

  init_displacement();
  if(copy_displacement_valid_p(&edit_canvas->copy_displacement)) {
    set_edit_text(displacement_x_edit_box, cc2str(edit_canvas->copy_displacement.x));
    set_edit_text(displacement_y_edit_box, cc2str(edit_canvas->copy_displacement.y));
  }
  else {
    set_edit_text(displacement_x_edit_box, "");
    set_edit_text(displacement_y_edit_box, "");
  }
  if (!run_modal_dialog(displacement_dialog))
    return;

  x_text = get_edit_text(dialog_item(displacement_dialog, DISPLACEMENT_X_EDIT_BOX, EDIT_BOX));
  y_text = get_edit_text(dialog_item(displacement_dialog, DISPLACEMENT_Y_EDIT_BOX, EDIT_BOX));
  if (x_text[0] == '\0' && y_text[0] == '\0')
    set_copy_displacement_invalid(&edit_canvas->copy_displacement);
  else {
    /* validity was checked above */
    str2cc(x_text, &edit_canvas->copy_displacement.x);
    str2cc(y_text, &edit_canvas->copy_displacement.y);
  }    
  force_canvas_change(edit_canvas);
}

/* ----- Initialization --------------------------------------------- */

void init_env(void)
{
  root_window->event_mask |= bit(eGAIN_FOCUS);
  SetDispatch(root_window, root);
  open_pulldown_bar(pulldown_bar, root_window);
  open_selector(tool_selector, 0, 0, root_window);
  open_selector(line_selector, 0, 0, root_window);
  edit_canvas->window.width = root_window->width - tool_selector->window.width - 2*tool_selector->window.border_width - 1;
  edit_canvas->window.height = root_window->height - pulldown_bar->window.height - pulldown_bar->window.border_width;
  open_canvas(edit_canvas, 0, pulldown_bar->window.height, root_window);
  set_canvas_colors(edit_canvas, settings->color);
  move_tools(settings->status & bit(sTOOLS_RIGHT));
  map_window(&pulldown_bar->window);
  env_set_cursor(bit(cMOUSE));
  set_focus(&edit_canvas->window);

  /* Execute settings that affect environment. */
  (settings->status & bit(sRULER_ON) ? enable_canvas_ruler : disable_canvas_ruler)(edit_canvas);
  enable_flashing_pick(edit_canvas);
  if (!(settings->status & bit(sFLASHING_PICK)))
    disable_flashing_pick();
  set_mouse_speed(settings->mouse_speed);
  set_double_click_delta(settings->double_click_delta);
  if ((settings->status & bit(sAUTO_LOAD)) && 
       !has_suffix_p(settings->edit_path, NONAME)) {
    set_wait_cursor(1);
    read_file(settings->edit_path);
    set_wait_cursor(0);
  }
}

