require(rstan)

# Stan generative model
sim_stan <- "
functions {
  int zibb_rng(int n, real mu, real phi, real kappa) {
    if (bernoulli_rng(kappa) == 1) {
      return (0);
    } else {
      return (beta_binomial_rng(n, mu * phi, (1 - mu) * phi));
    }
  }
}

data {
  int<lower=0> N_rep;
  int<lower=0> N_individual;
  int<lower=0> N_gene;
  int<lower=0> N;
  real sigma;
  real sigma_rep;
  real phi;
  real kappa;
}

generated quantities {
  array [N_gene, N_individual, N_rep] int Y;
  real mu_rep;
  real mu;
  real a;
  
  for(g in 1:N_gene) {
    a = normal_rng(-3, 1);
    for(s in 1:N_individual) {
      mu = normal_rng(a, sigma);
      for(r in 1:N_rep) {
        mu_rep = normal_rng(mu, sigma_rep);
        Y[g,s,r] = zibb_rng(N, 1/(1+exp(-(mu_rep))), phi, kappa);
      }
    }
  }
}
" 

# compile model
m <- rstan::stan_model(model_code = sim_stan)

# generate data based on following fixed parameters
set.seed(123456)
N_individual <- 5
N_rep <- 3
N_gene <- 15
N <- 10^3
sigma <- 0.5
sigma_rep <- 0.2
phi <- 200
kappa <- 0.03

l <- list(N_individual = N_individual, 
          N_gene = N_gene, 
          N_rep = N_rep,
          N = N,
          sigma = sigma,
          sigma_rep = sigma_rep,
          phi = phi,
          kappa = kappa)

# simulate
sim <- rstan::sampling(object = m,
                       data = l, 
                       iter = 1, 
                       chains = 1, 
                       algorithm = "Fixed_param",
                       seed = 123456)

# extract simulation and convert to be used as input of IgGeneUsage
ysim <- rstan::extract(object = sim, par = "Y")$Y
ysim <- ysim[1,,,]

ysim_df <- reshape2::melt(ysim)
colnames(ysim_df) <- c("gene_name", "individual_id", "replicate", "gene_usage_count")
ysim_df$condition <- "C_1"
ysim_df <- ysim_df[, c("individual_id", "condition", "gene_name", "replicate", "gene_usage_count")]
ysim_df$individual_id <- paste0("I_", as.character(ysim_df$individual_id))
ysim_df$gene_name <- paste0("G_", as.character(ysim_df$gene_name))
ysim_df$replicate <- paste0("R_", as.character(ysim_df$replicate))
d_zibb_2 <- ysim_df

# save
save(d_zibb_2, file = "data/d_zibb_2.RData", compress = TRUE)
