## ----summary, echo=FALSE------------------------------------------------------
table_data <- data.frame(
  Protocol = c("Visium", "Visium HD", "Slide-seq/Curio-seeker", "Stereo-seq", "Curio-seeker"),
  Resolution_micron = c(50, 2, 10, 10, 10),
  Spot_distance_micron = c(100, "0 (no gaps)", 10, 0.5, 10),
  Typical_array_size = c("0.65cm x 0.65cm / 1.1cm x 1.1cm", 
                         "0.65cm x 0.65cm / 1.1cm x 1.1cm", 
                         "/", 
                         "1cm x 1cm; (max: 13.2cm x 13.2cm)", 
                         "0.3cm x 0.3cm / 1cm x 1cm")
)

knitr::kable(table_data, format = "markdown", col.names = c("Protocol", "Resolution (micron)", "Spot distance (micron)", "Typical array size"))

## ----setup, message=FALSE-----------------------------------------------------
library(stPipe)
config <- system.file("config", "config_stPipe.yml", package = "stPipe")
yaml_content <- readLines(config)
cat(yaml_content, sep = "\n")

## ----input--------------------------------------------------------------------
# FASTQ file path:
data_directory <- file.path(tempdir(), "stPipe_demo_data")
if (!dir.exists(data_directory)) {
  dir.create(data_directory, recursive = TRUE)
}
options(timeout = 600)
R1_url <- "https://zenodo.org/records/14920583/files/stPipe_demo_R1.fq.gz?download=1"
R2_url <- "https://zenodo.org/records/14920583/files/stPipe_demo_R2.fq.gz?download=1"
R1_path <- file.path(data_directory, "stPipe_demo_R1.fq.gz")
R2_path <- file.path(data_directory, "stPipe_demo_R2.fq.gz")
download.file(R1_url, R1_path)
download.file(R2_url, R2_path)
config <- system.file("config", "config_stPipe.yml", package = "stPipe")
config <- yaml::read_yaml(config)
config$data_directory <- data_directory

## ----yaml, eval=TRUE----------------------------------------------------------
output_directory <- file.path(tempdir(), "stPipe_output")
if (!dir.exists(output_directory)) {
  dir.create(output_directory, recursive = TRUE)
}
config$output_directory <- output_directory
demo_config_path <- tempfile(fileext = ".yml")
yaml::write_yaml(config, demo_config_path)

## ----Run_ST,eval=TRUE---------------------------------------------------------
Run_ST(config = demo_config_path, show.config = FALSE)

## ----Run_Loc_Match, eval=TRUE-------------------------------------------------
st.result <- Run_Loc_Match(config = demo_config_path, pixel = FALSE, show.config = FALSE)
head(st.result$matched_data)

## ----qc, eval=TRUE------------------------------------------------------------
# adjust some parameter and update the config file
config$qc_per <- c("0.4_0.6")
yaml::write_yaml(config, demo_config_path)

## ----Run_QC, eval=TRUE, warning=FALSE-----------------------------------------
qc.results <- Run_QC(config = demo_config_path, matched.data = st.result$matched_data, gene.matrix = st.result$gene_count, show.config = FALSE)
dim(qc.results$filtered_spatial_coords)[1]/dim(st.result$matched_data)[1]

## ----Run_Clustering, eval=TRUE------------------------------------------------
output <- capture.output(
  Cluster.results <- Run_Clustering(
    gene.count = st.result$gene_count,
    matched.data = st.result$matched_data,
    num_clusters = 5
  )
)
cat(output[seq_along(1:15)], sep = "\n")

## ----Run_Interactive, eval=TRUE-----------------------------------------------
 Run_Interactive(
   matched_data = st.result$matched_data,
   clustering_result = Cluster.results$tsne_df,
   background_img = NULL,
   reduction_method = "tsne",
   point_size = 1
)

## ----Run_Visualization, eval=TRUE---------------------------------------------
Vis.results <- Run_Visualization(
  matched.data = st.result$matched_data,
  config = demo_config_path,
  Vis.spatial = TRUE,
  Vis.read = TRUE,
  show.config = FALSE
)

## ----demultiplex_plot, eval=TRUE, fig.height=7, fig.width=7-------------------
Vis.results$demultiplex_plot

## ----mapping_plot, eval=TRUE, fig.height=7, fig.width=7-----------------------
Vis.results$mapping_plot

## ----UMI_plot, echo=FALSE, fig.show='hold', fig.height=5, fig.width=5---------
Vis.results$raw_UMI_plot
Vis.results$log_UMI_plot

## ----seu_obj------------------------------------------------------------------
library(Seurat)
library(SeuratObject)
seu <- Run_Create_Obj(
   gene.matrix = st.result$gene_count,
   matched.data = st.result$matched_data,
   obj.type = 'Seurat',
   tech = 'Visium'
 )
seu

## ----sessionInfo--------------------------------------------------------------
sessionInfo()

