/*
 * Copyright (c) 2018-2019, NVIDIA CORPORATION.  All rights reserved.
 *
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include <stdint.h>

struct TimestampConverter
{
    int64_t dstAtSyncPoint;
    int64_t srcAtSyncPoint;
    double scale;

    int64_t operator()(int64_t srcTimestamp) const
    {
        auto srcDelta = srcTimestamp - srcAtSyncPoint;
        auto dstDelta = static_cast<int64_t>(scale * static_cast<double>(srcDelta));
        return dstDelta + dstAtSyncPoint;
    }
};

inline TimestampConverter CreateTimestampConverter(
    int64_t srcStart,
    int64_t srcEnd,
    int64_t dstStart,
    int64_t dstEnd)
{
    auto dstDelta = dstEnd - dstStart;
    auto srcDelta = srcEnd - srcStart;
    double scale = (srcDelta == 0.0) ? 0.0 :
        (static_cast<double>(dstDelta) / static_cast<double>(srcDelta));

    // Any sync point can be used for conversions.  Since we are subtracting the
    // sync point from each timestamp before scaling it, the 53-bit mantissa of
    // double makes our scaling precision about 1 nanosecond per week of distance
    // from the sync point (assuming 1 GHz clocks).  So, accuracy is best near
    // the sync point, samples one week later could be off by 1ns, samples two
    // weeks later by 2ns, etc.  For short captures the choice of sync point is
    // irrelevant, but in a snapshot-based tool where the region of interest is
    // nearer to the end, we should prioritize accuracy at the end highest, so we
    // select the end of capture as our sync point.

    return {dstEnd, srcEnd, scale};
}

// Helper functions for common timestamp-conversion cases:
// - Converting CPU ticks to nanoseconds
// - Creating a new converter that is the reverse of an existing one
// - Creating a new converter that is the composition two existing ones

inline TimestampConverter CpuToNsTimestampConverter(
    int64_t cpuTimestampTicksPerSecond,
    int64_t cpuTimestampStart)
{
    // Source is CPU timestamps, destination is nanoseconds since cpuTimestampStart
    double nsPerCpuTimestampTick =
        static_cast<double>(1'000'000'000) / static_cast<double>(cpuTimestampTicksPerSecond);
    return {0, cpuTimestampStart, nsPerCpuTimestampTick};
}

inline TimestampConverter ReverseConverter(
    TimestampConverter const& in)
{
    // Swap dst & src offsets, and invert scale
    return {in.srcAtSyncPoint, in.dstAtSyncPoint, 1.0 / in.scale};
}

inline TimestampConverter ComposeConverters(
    TimestampConverter const& ab,
    TimestampConverter const& bc)
{
    // ConvAtoB(a) = (a - abSrc) * abScale + abDst
    // ConvBtoC(b) = (b - bcSrc) * bcScale + bcDst
    // So, replacing b in ConvBtoC with ConvAtoB(a), we define ConvAtoC(a):
    //     = (((a - abSrc) * abScale + abDst) - bcSrc) * bcScale + bcDst
    //     = ((a - abSrc) * abScale + (abDst - bcSrc)) * bcScale + bcDst
    //     = (a - abSrc) * abScale * bcScale + (abDst - bcSrc) * bcScale + bcDst
    // ...which fits the form (a - acSrc) * acScale + acDst, using:
    //     acScale = abScale * bcScale
    //     acSrc = abSrc
    //     acDst = bcDst + (abDst - bcSrc) * bcScale
    int64_t acSrc = ab.srcAtSyncPoint;
    int64_t acDst = bc.dstAtSyncPoint +
        static_cast<int64_t>(static_cast<double>(ab.dstAtSyncPoint - bc.srcAtSyncPoint) * bc.scale);
    return {acDst, acSrc, ab.scale * bc.scale};
}