/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.coordinator.transaction;

import java.util.Collection;
import java.util.HashSet;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Supplier;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.coordinator.transaction.TransactionState;
import org.apache.kafka.coordinator.transaction.TxnTransitMetadata;
import org.apache.kafka.server.common.TransactionVersion;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MarkerFactory;

public class TransactionMetadata {
    private static final Logger LOGGER = LoggerFactory.getLogger(TransactionMetadata.class);
    private final String transactionalId;
    private long producerId;
    private long prevProducerId;
    private long nextProducerId;
    private short producerEpoch;
    private short lastProducerEpoch;
    private int txnTimeoutMs;
    private TransactionState state;
    private HashSet<TopicPartition> topicPartitions;
    private volatile long txnStartTimestamp;
    private volatile long txnLastUpdateTimestamp;
    private TransactionVersion clientTransactionVersion;
    private Optional<TransactionState> pendingState;
    private boolean hasFailedEpochFence;
    private final ReentrantLock lock;

    public static boolean isEpochExhausted(short producerEpoch) {
        return producerEpoch >= 32766;
    }

    public TransactionMetadata(String transactionalId, long producerId, long prevProducerId, long nextProducerId, short producerEpoch, short lastProducerEpoch, int txnTimeoutMs, TransactionState state, Set<TopicPartition> topicPartitions, long txnStartTimestamp, long txnLastUpdateTimestamp, TransactionVersion clientTransactionVersion) {
        this.transactionalId = transactionalId;
        this.producerId = producerId;
        this.prevProducerId = prevProducerId;
        this.nextProducerId = nextProducerId;
        this.producerEpoch = producerEpoch;
        this.lastProducerEpoch = lastProducerEpoch;
        this.txnTimeoutMs = txnTimeoutMs;
        this.state = state;
        this.topicPartitions = new HashSet<TopicPartition>(topicPartitions);
        this.txnStartTimestamp = txnStartTimestamp;
        this.txnLastUpdateTimestamp = txnLastUpdateTimestamp;
        this.clientTransactionVersion = clientTransactionVersion;
        this.pendingState = Optional.empty();
        this.hasFailedEpochFence = false;
        this.lock = new ReentrantLock();
    }

    public <T> T inLock(Supplier<T> function) {
        this.lock.lock();
        try {
            T t = function.get();
            return t;
        }
        finally {
            this.lock.unlock();
        }
    }

    public void addPartitions(Collection<TopicPartition> partitions) {
        this.topicPartitions.addAll(partitions);
    }

    public void removePartition(TopicPartition topicPartition) {
        if (this.state != TransactionState.PREPARE_COMMIT && this.state != TransactionState.PREPARE_ABORT) {
            throw new IllegalStateException("Transaction metadata's current state is " + String.valueOf((Object)this.state) + ", and its pending state is " + String.valueOf(this.pendingState) + " while trying to remove partitions whose txn marker has been sent, this is not expected");
        }
        this.topicPartitions.remove(topicPartition);
    }

    public TxnTransitMetadata prepareNoTransit() {
        return new TxnTransitMetadata(this.producerId, this.prevProducerId, this.nextProducerId, this.producerEpoch, this.lastProducerEpoch, this.txnTimeoutMs, this.state, new HashSet<TopicPartition>(this.topicPartitions), this.txnStartTimestamp, this.txnLastUpdateTimestamp, this.clientTransactionVersion);
    }

    public TxnTransitMetadata prepareFenceProducerEpoch() {
        if (this.producerEpoch == Short.MAX_VALUE) {
            throw new IllegalStateException("Cannot fence producer with epoch equal to Short.MaxValue since this would overflow");
        }
        short bumpedEpoch = this.hasFailedEpochFence ? this.producerEpoch : (short)(this.producerEpoch + 1);
        TransitionData data = new TransitionData(TransactionState.PREPARE_EPOCH_FENCE);
        data.producerEpoch = bumpedEpoch;
        return this.prepareTransitionTo(data);
    }

    public TxnTransitMetadata prepareIncrementProducerEpoch(int newTxnTimeoutMs, Optional<Short> expectedProducerEpoch, long updateTimestamp) {
        if (this.isProducerEpochExhausted()) {
            throw new IllegalStateException("Cannot allocate any more producer epochs for producerId " + this.producerId);
        }
        TransitionData data = new TransitionData(TransactionState.EMPTY);
        short bumpedEpoch = (short)(this.producerEpoch + 1);
        if (expectedProducerEpoch.isEmpty()) {
            data.producerEpoch = bumpedEpoch;
            data.lastProducerEpoch = (short)-1;
        } else if (this.producerEpoch == -1 || expectedProducerEpoch.get() == this.producerEpoch) {
            data.producerEpoch = bumpedEpoch;
            data.lastProducerEpoch = this.producerEpoch;
        } else if (expectedProducerEpoch.get() == this.lastProducerEpoch) {
            data.producerEpoch = this.producerEpoch;
            data.lastProducerEpoch = this.lastProducerEpoch;
        } else {
            LOGGER.info("Expected producer epoch {} does not match current producer epoch {} or previous producer epoch {}", new Object[]{expectedProducerEpoch.get(), this.producerEpoch, this.lastProducerEpoch});
            throw Errors.PRODUCER_FENCED.exception();
        }
        data.txnTimeoutMs = newTxnTimeoutMs;
        data.topicPartitions = new HashSet();
        data.txnStartTimestamp = -1L;
        data.txnLastUpdateTimestamp = updateTimestamp;
        return this.prepareTransitionTo(data);
    }

    public TxnTransitMetadata prepareProducerIdRotation(long newProducerId, int newTxnTimeoutMs, long updateTimestamp, boolean recordLastEpoch) {
        if (this.hasPendingTransaction()) {
            throw new IllegalStateException("Cannot rotate producer ids while a transaction is still pending");
        }
        TransitionData data = new TransitionData(TransactionState.EMPTY);
        data.producerId = newProducerId;
        data.producerEpoch = 0;
        data.lastProducerEpoch = (short)(recordLastEpoch ? (int)this.producerEpoch : -1);
        data.txnTimeoutMs = newTxnTimeoutMs;
        data.topicPartitions = new HashSet();
        data.txnStartTimestamp = -1L;
        data.txnLastUpdateTimestamp = updateTimestamp;
        return this.prepareTransitionTo(data);
    }

    public TxnTransitMetadata prepareAddPartitions(Set<TopicPartition> addedTopicPartitions, long updateTimestamp, TransactionVersion clientTransactionVersion) {
        long newTxnStartTimestamp = this.state == TransactionState.EMPTY || this.state == TransactionState.COMPLETE_ABORT || this.state == TransactionState.COMPLETE_COMMIT ? updateTimestamp : this.txnStartTimestamp;
        HashSet<TopicPartition> newTopicPartitions = new HashSet<TopicPartition>(this.topicPartitions);
        newTopicPartitions.addAll(addedTopicPartitions);
        TransitionData data = new TransitionData(TransactionState.ONGOING);
        data.topicPartitions = newTopicPartitions;
        data.txnStartTimestamp = newTxnStartTimestamp;
        data.txnLastUpdateTimestamp = updateTimestamp;
        data.clientTransactionVersion = clientTransactionVersion;
        return this.prepareTransitionTo(data);
    }

    public TxnTransitMetadata prepareAbortOrCommit(TransactionState newState, TransactionVersion clientTransactionVersion, long nextProducerId, long updateTimestamp, boolean noPartitionAdded) {
        TransitionData data = new TransitionData(newState);
        if (clientTransactionVersion.supportsEpochBump()) {
            data.producerEpoch = (short)(this.producerEpoch + 1);
            data.lastProducerEpoch = this.producerEpoch;
        } else {
            data.producerEpoch = this.producerEpoch;
            data.lastProducerEpoch = this.lastProducerEpoch;
        }
        data.txnStartTimestamp = noPartitionAdded ? updateTimestamp : this.txnStartTimestamp;
        data.nextProducerId = nextProducerId;
        data.txnLastUpdateTimestamp = updateTimestamp;
        data.clientTransactionVersion = clientTransactionVersion;
        return this.prepareTransitionTo(data);
    }

    public TxnTransitMetadata prepareComplete(long updateTimestamp) {
        this.hasFailedEpochFence = false;
        TransitionData data = new TransitionData(this.state == TransactionState.PREPARE_COMMIT ? TransactionState.COMPLETE_COMMIT : TransactionState.COMPLETE_ABORT);
        if (this.clientTransactionVersion.supportsEpochBump() && this.nextProducerId != -1L) {
            data.producerId = this.nextProducerId;
            data.producerEpoch = 0;
        } else {
            data.producerId = this.producerId;
            data.producerEpoch = this.producerEpoch;
        }
        data.nextProducerId = -1L;
        data.topicPartitions = new HashSet();
        data.txnLastUpdateTimestamp = updateTimestamp;
        return this.prepareTransitionTo(data);
    }

    public TxnTransitMetadata prepareDead() {
        TransitionData data = new TransitionData(TransactionState.DEAD);
        data.topicPartitions = new HashSet();
        return this.prepareTransitionTo(data);
    }

    public boolean isProducerEpochExhausted() {
        return TransactionMetadata.isEpochExhausted(this.producerEpoch);
    }

    public boolean isDistributedTwoPhaseCommitTxn() {
        return this.txnTimeoutMs == Integer.MAX_VALUE;
    }

    private boolean hasPendingTransaction() {
        return this.state == TransactionState.ONGOING || this.state == TransactionState.PREPARE_ABORT || this.state == TransactionState.PREPARE_COMMIT;
    }

    private TxnTransitMetadata prepareTransitionTo(TransitionData data) {
        if (this.pendingState.isPresent()) {
            throw new IllegalStateException("Preparing transaction state transition to " + String.valueOf((Object)this.state) + " while it already a pending state " + String.valueOf((Object)this.pendingState.get()));
        }
        if (data.producerId < 0L) {
            throw new IllegalArgumentException("Illegal new producer id " + data.producerId);
        }
        if (data.state != TransactionState.DEAD && data.producerEpoch < 0) {
            throw new IllegalArgumentException("Illegal new producer epoch " + data.producerEpoch);
        }
        if (data.state.validPreviousStates().contains((Object)this.state)) {
            TxnTransitMetadata transitMetadata = new TxnTransitMetadata(data.producerId, this.producerId, data.nextProducerId, data.producerEpoch, data.lastProducerEpoch, data.txnTimeoutMs, data.state, data.topicPartitions, data.txnStartTimestamp, data.txnLastUpdateTimestamp, data.clientTransactionVersion);
            LOGGER.debug("TransactionalId {} prepare transition from {} to {}", new Object[]{this.transactionalId, this.state, data.state});
            this.pendingState = Optional.of(data.state);
            return transitMetadata;
        }
        throw new IllegalStateException("Preparing transaction state transition to " + String.valueOf((Object)data.state) + " failed since the target state " + String.valueOf((Object)data.state) + " is not a valid previous state of the current state " + String.valueOf((Object)this.state));
    }

    public void completeTransitionTo(TxnTransitMetadata transitMetadata) {
        TransactionState toState = this.pendingState.orElseThrow(() -> {
            LOGGER.error(MarkerFactory.getMarker((String)"FATAL"), "{}'s transition to {} failed since pendingState is not defined: this should not happen", (Object)this, (Object)transitMetadata);
            return new IllegalStateException("TransactionalId " + this.transactionalId + " completing transaction state transition while it does not have a pending state");
        });
        if (!toState.equals((Object)transitMetadata.txnState())) {
            this.throwStateTransitionFailure(transitMetadata);
        }
        switch (toState) {
            case EMPTY: {
                if ((this.producerEpoch == transitMetadata.producerEpoch() || this.validProducerEpochBump(transitMetadata)) && transitMetadata.topicPartitions().isEmpty() && transitMetadata.txnStartTimestamp() == -1L) break;
                this.throwStateTransitionFailure(transitMetadata);
                break;
            }
            case ONGOING: {
                if (this.validProducerEpoch(transitMetadata) && transitMetadata.topicPartitions().containsAll(this.topicPartitions) && this.txnTimeoutMs == transitMetadata.txnTimeoutMs()) break;
                this.throwStateTransitionFailure(transitMetadata);
                break;
            }
            case PREPARE_ABORT: 
            case PREPARE_COMMIT: {
                boolean validTimestamp;
                boolean allowedEmptyAbort = toState == TransactionState.PREPARE_ABORT && transitMetadata.clientTransactionVersion().supportsEpochBump() && (this.state == TransactionState.EMPTY || this.state == TransactionState.COMPLETE_COMMIT || this.state == TransactionState.COMPLETE_ABORT);
                boolean bl = validTimestamp = this.txnStartTimestamp == transitMetadata.txnStartTimestamp() || allowedEmptyAbort;
                if (this.validProducerEpoch(transitMetadata) && this.topicPartitions.equals(transitMetadata.topicPartitions()) && this.txnTimeoutMs == transitMetadata.txnTimeoutMs() && validTimestamp) break;
                this.throwStateTransitionFailure(transitMetadata);
                break;
            }
            case COMPLETE_ABORT: 
            case COMPLETE_COMMIT: {
                if (this.validProducerEpoch(transitMetadata) && this.txnTimeoutMs == transitMetadata.txnTimeoutMs() && transitMetadata.txnStartTimestamp() != -1L) break;
                this.throwStateTransitionFailure(transitMetadata);
                break;
            }
            case PREPARE_EPOCH_FENCE: {
                this.throwStateTransitionFailure(transitMetadata);
                break;
            }
            case DEAD: {
                throw new IllegalStateException("TransactionalId " + this.transactionalId + " is trying to complete a transition to " + String.valueOf((Object)toState) + ". This means that the transactionalId was being expired, and the only acceptable completion of this operation is to remove the transaction metadata from the cache, not to persist the " + String.valueOf((Object)toState) + " in the log.");
            }
        }
        LOGGER.debug("TransactionalId {} complete transition from {} to {}", new Object[]{this.transactionalId, this.state, transitMetadata});
        this.producerId = transitMetadata.producerId();
        this.prevProducerId = transitMetadata.prevProducerId();
        this.nextProducerId = transitMetadata.nextProducerId();
        this.producerEpoch = transitMetadata.producerEpoch();
        this.lastProducerEpoch = transitMetadata.lastProducerEpoch();
        this.txnTimeoutMs = transitMetadata.txnTimeoutMs();
        this.topicPartitions = transitMetadata.topicPartitions();
        this.txnStartTimestamp = transitMetadata.txnStartTimestamp();
        this.txnLastUpdateTimestamp = transitMetadata.txnLastUpdateTimestamp();
        this.clientTransactionVersion = transitMetadata.clientTransactionVersion();
        this.pendingState = Optional.empty();
        this.state = toState;
    }

    private boolean validProducerEpoch(TxnTransitMetadata transitMetadata) {
        boolean isAtLeastTransactionsV2 = transitMetadata.clientTransactionVersion().supportsEpochBump();
        TransactionState txnState = transitMetadata.txnState();
        short transitProducerEpoch = transitMetadata.producerEpoch();
        long transitProducerId = transitMetadata.producerId();
        short transitLastProducerEpoch = transitMetadata.lastProducerEpoch();
        if (isAtLeastTransactionsV2 && (txnState == TransactionState.COMPLETE_COMMIT || txnState == TransactionState.COMPLETE_ABORT) && transitProducerEpoch == 0) {
            return transitLastProducerEpoch == this.lastProducerEpoch && transitMetadata.prevProducerId() == this.producerId;
        }
        if (isAtLeastTransactionsV2 && (txnState == TransactionState.PREPARE_COMMIT || txnState == TransactionState.PREPARE_ABORT)) {
            return transitLastProducerEpoch == this.producerEpoch && transitProducerId == this.producerId;
        }
        return transitProducerEpoch == this.producerEpoch && transitProducerId == this.producerId;
    }

    private boolean validProducerEpochBump(TxnTransitMetadata transitMetadata) {
        short transitEpoch = transitMetadata.producerEpoch();
        long transitProducerId = transitMetadata.producerId();
        return transitEpoch == (short)(this.producerEpoch + 1) || transitEpoch == 0 && transitProducerId != this.producerId;
    }

    private void throwStateTransitionFailure(TxnTransitMetadata txnTransitMetadata) {
        LOGGER.error(MarkerFactory.getMarker((String)"FATAL"), "{}'s transition to {} failed: this should not happen", (Object)this, (Object)txnTransitMetadata);
        throw new IllegalStateException("TransactionalId " + this.transactionalId + " failed transition to state " + String.valueOf(txnTransitMetadata) + " due to unexpected metadata");
    }

    public boolean pendingTransitionInProgress() {
        return this.pendingState.isPresent();
    }

    public String transactionalId() {
        return this.transactionalId;
    }

    public void setProducerId(long producerId) {
        this.producerId = producerId;
    }

    public long producerId() {
        return this.producerId;
    }

    public void setPrevProducerId(long prevProducerId) {
        this.prevProducerId = prevProducerId;
    }

    public long prevProducerId() {
        return this.prevProducerId;
    }

    public void setProducerEpoch(short producerEpoch) {
        this.producerEpoch = producerEpoch;
    }

    public short producerEpoch() {
        return this.producerEpoch;
    }

    public void setLastProducerEpoch(short lastProducerEpoch) {
        this.lastProducerEpoch = lastProducerEpoch;
    }

    public short lastProducerEpoch() {
        return this.lastProducerEpoch;
    }

    public int txnTimeoutMs() {
        return this.txnTimeoutMs;
    }

    public void state(TransactionState state) {
        this.state = state;
    }

    public TransactionState state() {
        return this.state;
    }

    public Set<TopicPartition> topicPartitions() {
        return this.topicPartitions;
    }

    public long txnStartTimestamp() {
        return this.txnStartTimestamp;
    }

    public void txnLastUpdateTimestamp(long txnLastUpdateTimestamp) {
        this.txnLastUpdateTimestamp = txnLastUpdateTimestamp;
    }

    public long txnLastUpdateTimestamp() {
        return this.txnLastUpdateTimestamp;
    }

    public void clientTransactionVersion(TransactionVersion clientTransactionVersion) {
        this.clientTransactionVersion = clientTransactionVersion;
    }

    public TransactionVersion clientTransactionVersion() {
        return this.clientTransactionVersion;
    }

    public void pendingState(Optional<TransactionState> pendingState) {
        this.pendingState = pendingState;
    }

    public Optional<TransactionState> pendingState() {
        return this.pendingState;
    }

    public void hasFailedEpochFence(boolean hasFailedEpochFence) {
        this.hasFailedEpochFence = hasFailedEpochFence;
    }

    public boolean hasFailedEpochFence() {
        return this.hasFailedEpochFence;
    }

    public String toString() {
        return "TransactionMetadata(transactionalId=" + this.transactionalId + ", producerId=" + this.producerId + ", prevProducerId=" + this.prevProducerId + ", nextProducerId=" + this.nextProducerId + ", producerEpoch=" + this.producerEpoch + ", lastProducerEpoch=" + this.lastProducerEpoch + ", txnTimeoutMs=" + this.txnTimeoutMs + ", state=" + String.valueOf((Object)this.state) + ", pendingState=" + String.valueOf(this.pendingState) + ", topicPartitions=" + String.valueOf(this.topicPartitions) + ", txnStartTimestamp=" + this.txnStartTimestamp + ", txnLastUpdateTimestamp=" + this.txnLastUpdateTimestamp + ", clientTransactionVersion=" + String.valueOf(this.clientTransactionVersion) + ")";
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null || this.getClass() != obj.getClass()) {
            return false;
        }
        TransactionMetadata other = (TransactionMetadata)obj;
        return this.transactionalId.equals(other.transactionalId) && this.producerId == other.producerId && this.prevProducerId == other.prevProducerId && this.nextProducerId == other.nextProducerId && this.producerEpoch == other.producerEpoch && this.lastProducerEpoch == other.lastProducerEpoch && this.txnTimeoutMs == other.txnTimeoutMs && this.state.equals((Object)other.state) && this.topicPartitions.equals(other.topicPartitions) && this.txnStartTimestamp == other.txnStartTimestamp && this.txnLastUpdateTimestamp == other.txnLastUpdateTimestamp && this.clientTransactionVersion.equals((Object)other.clientTransactionVersion);
    }

    public int hashCode() {
        return Objects.hash(new Object[]{this.transactionalId, this.producerId, this.prevProducerId, this.nextProducerId, this.producerEpoch, this.lastProducerEpoch, this.txnTimeoutMs, this.state, this.topicPartitions, this.txnStartTimestamp, this.txnLastUpdateTimestamp, this.clientTransactionVersion});
    }

    private class TransitionData {
        final TransactionState state;
        long producerId;
        long nextProducerId;
        short producerEpoch;
        short lastProducerEpoch;
        int txnTimeoutMs;
        HashSet<TopicPartition> topicPartitions;
        long txnStartTimestamp;
        long txnLastUpdateTimestamp;
        TransactionVersion clientTransactionVersion;

        private TransitionData(TransactionState state) {
            this.producerId = TransactionMetadata.this.producerId;
            this.nextProducerId = TransactionMetadata.this.nextProducerId;
            this.producerEpoch = TransactionMetadata.this.producerEpoch;
            this.lastProducerEpoch = TransactionMetadata.this.lastProducerEpoch;
            this.txnTimeoutMs = TransactionMetadata.this.txnTimeoutMs;
            this.topicPartitions = TransactionMetadata.this.topicPartitions;
            this.txnStartTimestamp = TransactionMetadata.this.txnStartTimestamp;
            this.txnLastUpdateTimestamp = TransactionMetadata.this.txnLastUpdateTimestamp;
            this.clientTransactionVersion = TransactionMetadata.this.clientTransactionVersion;
            this.state = state;
        }
    }
}

