"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WritableDocument = exports.ReadableDocument = void 0;
const utils_1 = require("./utils");
/**
 * Represents a textual document.
 */
class ReadableDocument {
    constructor() {
        /**
         * Current version of the document.
         */
        this.version = 0;
    }
    /**
     * Get the length of the document's content
     */
    getTextLength() {
        return this.getText().length;
    }
    /**
     * Get the line and character based on the offset
     * @param offset The index of the position
     */
    positionAt(offset) {
        return (0, utils_1.positionAt)(offset, this.getText(), this.getLineOffsets());
    }
    /**
     * Get the index of the line and character position
     * @param position Line and character position
     */
    offsetAt(position) {
        return (0, utils_1.offsetAt)(position, this.getText(), this.getLineOffsets());
    }
    getLineOffsets() {
        if (!this.lineOffsets) {
            this.lineOffsets = (0, utils_1.getLineOffsets)(this.getText());
        }
        return this.lineOffsets;
    }
    /**
     * Implements TextDocument
     */
    get uri() {
        return this.getURL();
    }
    get lineCount() {
        return this.getText().split(/\r?\n/).length;
    }
}
exports.ReadableDocument = ReadableDocument;
/**
 * Represents a textual document that can be manipulated.
 */
class WritableDocument extends ReadableDocument {
    /**
     * Batch update the document with the LSP change events.
     */
    update(changes) {
        const pendingChanges = [];
        for (const change of changes) {
            let start = 0;
            let end = 0;
            if ('range' in change) {
                start = this.offsetAt(change.range.start);
                end = this.offsetAt(change.range.end);
            }
            else {
                end = this.getTextLength();
            }
            pendingChanges.push({ text: change.text, start, end });
        }
        // VSCode already sends the change in this order,
        // But it's not written in the spec, so sorting it just to be sure.
        const sortedDescending = pendingChanges.sort((a, b) => b.end - a.end);
        const content = this.getText();
        let newText = content;
        for (const change of sortedDescending) {
            newText = newText.slice(0, change.start) + change.text + newText.slice(change.end);
        }
        this.lineOffsets = undefined;
        this.setText(newText);
    }
}
exports.WritableDocument = WritableDocument;
//# sourceMappingURL=DocumentBase.js.map