<?php

namespace {

	/**
	 * <p>Represents ZooKeeper session.</p>
	 * @link https://php.net/manual/en/class.zookeeper.php
	 * @since PECL zookeeper >= 0.1.0
	 */
	class Zookeeper {

		/**
		 * @var int <p>Can read nodes value and list its children</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int PERM_READ = 1;

		/**
		 * @var int <p>Can set the nodes value</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int PERM_WRITE = 2;

		/**
		 * @var int <p>Can create children</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int PERM_CREATE = 4;

		/**
		 * @var int <p>Can delete children</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int PERM_DELETE = 8;

		/**
		 * @var int <p>Can execute set_acl()</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int PERM_ADMIN = 16;

		/**
		 * @var int <p>All of the above flags ORd together</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int PERM_ALL = 31;

		/**
		 * @var int <p>If Zookeeper::EPHEMERAL flag is set, the node will automatically get removed if the client session goes away.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int EPHEMERAL = 1;

		/**
		 * @var int <p>If the Zookeeper::SEQUENCE flag is set, a unique monotonically increasing sequence number is appended to the path name. The sequence number is always fixed length of 10 digits, 0 padded.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int SEQUENCE = 2;

		/**
		 * @var int <p>Outputs only error mesages</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int LOG_LEVEL_ERROR = 1;

		/**
		 * @var int <p>Outputs errors/warnings</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int LOG_LEVEL_WARN = 2;

		/**
		 * @var int <p>Outputs big action messages besides errors/warnings</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int LOG_LEVEL_INFO = 3;

		/**
		 * @var int <p>Outputs all</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int LOG_LEVEL_DEBUG = 4;

		/**
		 * @var int <p>Connected but session expired</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int EXPIRED_SESSION_STATE = -112;

		/**
		 * @var int <p>Connected but auth failed</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int AUTH_FAILED_STATE = -113;

		/**
		 * @var int <p>Connecting</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int CONNECTING_STATE = 1;

		/**
		 * @var int <p>Associating</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int ASSOCIATING_STATE = 2;

		/**
		 * @var int <p>Connected</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int CONNECTED_STATE = 3;

		/**
		 * @var int <p>TODO: help us improve this extension.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int READONLY_STATE = 5;

		/**
		 * @var int <p>Connection failed</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int NOTCONNECTED_STATE = 999;

		/**
		 * @var int <p>A node has been created</p> <p>This is only generated by watches on non-existent nodes. These watches are set using Zookeeper::exists.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int CREATED_EVENT = 1;

		/**
		 * @var int <p>A node has been deleted</p> <p>This is only generated by watches on nodes. These watches are set using Zookeeper::exists and Zookeeper::get.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int DELETED_EVENT = 2;

		/**
		 * @var int <p>A node has changed</p> <p>This is only generated by watches on nodes. These watches are set using Zookeeper::exists and Zookeeper::get.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int CHANGED_EVENT = 3;

		/**
		 * @var int <p>A change as occurred in the list of children</p> <p>This is only generated by watches on the child list of a node. These watches are set using Zookeeper::getChildren.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int CHILD_EVENT = 4;

		/**
		 * @var int <p>A session has been lost</p> <p>This is generated when a client loses contact or reconnects with a server.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int SESSION_EVENT = -1;

		/**
		 * @var int <p>A watch has been removed</p> <p>This is generated when the server for some reason, probably a resource constraint, will no longer watch a node for a client.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int NOTWATCHING_EVENT = -2;

		/**
		 * @var int <p>This is never thrown by the server, it shouldn't be used other than to indicate a range. Specifically error codes greater than this value, but lesser than Zookeeper::APIERROR, are system errors.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int SYSTEMERROR = -1;

		/**
		 * @var int <p>A runtime inconsistency was found.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int RUNTIMEINCONSISTENCY = -2;

		/**
		 * @var int <p>A data inconsistency was found.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int DATAINCONSISTENCY = -3;

		/**
		 * @var int <p>Connection to the server has been lost.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int CONNECTIONLOSS = -4;

		/**
		 * @var int <p>Error while marshalling or unmarshalling data.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int MARSHALLINGERROR = -5;

		/**
		 * @var int <p>Operation is unimplemented.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int UNIMPLEMENTED = -6;

		/**
		 * @var int <p>Operation timeout.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int OPERATIONTIMEOUT = -7;

		/**
		 * @var int <p>Invalid arguments.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int BADARGUMENTS = -8;

		/**
		 * @var int <p>Invalid zhandle state.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int INVALIDSTATE = -9;

		/**
		 * @var int <p>No quorum of new config is connected and up-to-date with the leader of last committed config - try invoking reconfiguration after new servers are connected and synced.</p> <p>Available as of ZooKeeper 3.5.0</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int NEWCONFIGNOQUORUM = -13;

		/**
		 * @var int <p>Reconfiguration requested while another reconfiguration is currently in progress. This is currently not supported. Please retry.</p> <p>Available as of ZooKeeper 3.5.0</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int RECONFIGINPROGRESS = -14;

		/**
		 * @var int <p>Everything is OK.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int OK = 0;

		/**
		 * @var int <p>This is never thrown by the server, it shouldn't be used other than to indicate a range. Specifically error codes greater than this value are API errors (while values less than this indicate a Zookeeper::SYSTEMERROR).</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int APIERROR = -100;

		/**
		 * @var int <p>Node does not exist.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int NONODE = -101;

		/**
		 * @var int <p>Not authenticated.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int NOAUTH = -102;

		/**
		 * @var int <p>Version conflict.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int BADVERSION = -103;

		/**
		 * @var int <p>Ephemeral nodes may not have children.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int NOCHILDRENFOREPHEMERALS = -108;

		/**
		 * @var int <p>The node already exists.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int NODEEXISTS = -110;

		/**
		 * @var int <p>The node has children.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int NOTEMPTY = -111;

		/**
		 * @var int <p>The session has been expired by the server.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int SESSIONEXPIRED = -112;

		/**
		 * @var int <p>Invalid callback specified.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int INVALIDCALLBACK = -113;

		/**
		 * @var int <p>Invalid ACL specified.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int INVALIDACL = -114;

		/**
		 * @var int <p>Client authentication failed.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int AUTHFAILED = -115;

		/**
		 * @var int <p>ZooKeeper is closing.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int CLOSING = -116;

		/**
		 * @var int <p>(not error) No server responses to process.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int NOTHING = -117;

		/**
		 * @var int <p>Session moved to another server, so operation is ignored.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int SESSIONMOVED = -118;

		/**
		 * @var int <p>State-changing request is passed to read-only server.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int NOTREADONLY = -119;

		/**
		 * @var int <p>Attempt to create ephemeral node on a local session.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int EPHEMERALONLOCALSESSION = -120;

		/**
		 * @var int <p>The watcher couldn't be found.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int NOWATCHER = -121;

		/**
		 * @var int <p>Attempts to perform a reconfiguration operation when reconfiguration feature is disabled.</p>
		 * @link https://php.net/manual/en/class.zookeeper.php
		 */
		const int RECONFIGDISABLED = -122;

		/**
		 * Create a handle to used communicate with zookeeper
		 * <p>This method creates a new handle and a zookeeper session that corresponds to that handle. Session establishment is asynchronous, meaning that the session should not be considered established until (and unless) an event of state ZOO_CONNECTED_STATE is received.</p>
		 * @param string $host <p>comma separated host:port pairs, each corresponding to a zk server. e.g. "127.0.0.1:3000,127.0.0.1:3001,127.0.0.1:3002"</p>
		 * @param callable $watcher_cb <p>the global watcher callback function. When notifications are triggered this function will be invoked.</p>
		 * @param int $recv_timeout <p>the timeout for this session, only valid if the connections is currently connected (ie. last watcher state is ZOO_CONNECTED_STATE).</p>
		 * @return self
		 * @link https://php.net/manual/en/zookeeper.construct.php
		 * @since PECL zookeeper >= 0.1.0
		 */
		public function __construct(string $host = '', callable $watcher_cb = null, int $recv_timeout = 10000) {}

		/**
		 * Specify application credentials
		 * <p>The application calls this function to specify its credentials for purposes of authentication. The server will use the security provider specified by the scheme parameter to authenticate the client connection. If the authentication request has failed: - the server connection is dropped. - the watcher is called with the ZOO_AUTH_FAILED_STATE value as the state parameter.</p>
		 * @param string $scheme <p>The id of authentication scheme. Natively supported: "digest" password-based authentication</p>
		 * @param string $cert <p>Application credentials. The actual value depends on the scheme.</p>
		 * @param callable $completion_cb <p>The routine to invoke when the request completes. One of the following result codes may be passed into the completion callback: - ZOK operation completed successfully - ZAUTHFAILED authentication failed</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/zookeeper.addauth.php
		 * @since PECL zookeeper >= 0.1.0
		 */
		public function addAuth(string $scheme, string $cert, callable $completion_cb = null): bool {}

		/**
		 * Close the zookeeper handle and free up any resources
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/zookeeper.close.php
		 * @since PECL zookeeper >= 0.5.0
		 */
		public function close(): void {}

		/**
		 * Create a handle to used communicate with zookeeper
		 * <p>This method creates a new handle and a zookeeper session that corresponds to that handle. Session establishment is asynchronous, meaning that the session should not be considered established until (and unless) an event of state ZOO_CONNECTED_STATE is received.</p>
		 * @param string $host <p>Comma separated host:port pairs, each corresponding to a zk server. e.g. "127.0.0.1:3000,127.0.0.1:3001,127.0.0.1:3002"</p>
		 * @param callable $watcher_cb <p>The global watcher callback function. When notifications are triggered this function will be invoked.</p>
		 * @param int $recv_timeout <p>The timeout for this session, only valid if the connections is currently connected (ie. last watcher state is ZOO_CONNECTED_STATE).</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/zookeeper.connect.php
		 * @since PECL zookeeper >= 0.2.0
		 */
		public function connect(string $host, callable $watcher_cb = null, int $recv_timeout = 10000): void {}

		/**
		 * Create a node synchronously
		 * <p>This method will create a node in ZooKeeper. A node can only be created if it does not already exists. The Create Flags affect the creation of nodes. If ZOO_EPHEMERAL flag is set, the node will automatically get removed if the client session goes away. If the ZOO_SEQUENCE flag is set, a unique monotonically increasing sequence number is appended to the path name.</p>
		 * @param string $path <p>The name of the node. Expressed as a file name with slashes separating ancestors of the node.</p>
		 * @param string $value <p>The data to be stored in the node.</p>
		 * @param array $acls <p>The initial ACL of the node. The ACL must not be null or empty.</p>
		 * @param int $flags <p>this parameter can be set to 0 for normal create or an OR of the Create Flags</p>
		 * @return string <p>Returns the path of the new node (this might be different than the supplied path because of the ZOO_SEQUENCE flag) on success, and false on failure.</p>
		 * @link https://php.net/manual/en/zookeeper.create.php
		 * @since PECL zookeeper >= 0.1.0
		 */
		public function create(string $path, string $value, array $acls, int $flags = null): string {}

		/**
		 * Delete a node in zookeeper synchronously
		 * @param string $path <p>The name of the node. Expressed as a file name with slashes separating ancestors of the node.</p>
		 * @param int $version <p>The expected version of the node. The function will fail if the actual version of the node does not match the expected version. If -1 is used the version check will not take place.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/zookeeper.delete.php
		 * @since PECL zookeeper >= 0.2.0
		 */
		public function delete(string $path, int $version = -1): bool {}

		/**
		 * Checks the existence of a node in zookeeper synchronously
		 * @param string $path <p>The name of the node. Expressed as a file name with slashes separating ancestors of the node.</p>
		 * @param callable $watcher_cb <p>if nonzero, a watch will be set at the server to notify the client if the node changes. The watch will be set even if the node does not</p>
		 * @return array <p>Returns the value of stat for the path if the given node exists, otherwise false.</p>
		 * @link https://php.net/manual/en/zookeeper.exists.php
		 * @since PECL zookeeper >= 0.1.0
		 */
		public function exists(string $path, callable $watcher_cb = null): array {}

		/**
		 * Gets the data associated with a node synchronously
		 * @param string $path <p>The name of the node. Expressed as a file name with slashes separating ancestors of the node.</p>
		 * @param callable $watcher_cb <p>If nonzero, a watch will be set at the server to notify the client if the node changes.</p>
		 * @param array $stat <p>If not NULL, will hold the value of stat for the path on return.</p>
		 * @param int $max_size <p>Max size of the data. If 0 is used, this method will return the whole data.</p>
		 * @return string <p>Returns the data on success, and false on failure.</p>
		 * @link https://php.net/manual/en/zookeeper.get.php
		 * @since PECL zookeeper >= 0.1.0
		 */
		public function get(string $path, callable $watcher_cb = null, array &$stat = null, int $max_size = 0): string {}

		/**
		 * Gets the acl associated with a node synchronously
		 * @param string $path <p>The name of the node. Expressed as a file name with slashes separating ancestors of the node.</p>
		 * @return array <p>Return acl array on success and false on failure.</p>
		 * @link https://php.net/manual/en/zookeeper.getacl.php
		 * @since PECL zookeeper >= 0.1.0
		 */
		public function getAcl(string $path): array {}

		/**
		 * Lists the children of a node synchronously
		 * @param string $path <p>The name of the node. Expressed as a file name with slashes separating ancestors of the node.</p>
		 * @param callable $watcher_cb <p>If nonzero, a watch will be set at the server to notify the client if the node changes.</p>
		 * @return array <p>Returns an array with children paths on success, and false on failure.</p>
		 * @link https://php.net/manual/en/zookeeper.getchildren.php
		 * @since PECL zookeeper >= 0.1.0
		 */
		public function getChildren(string $path, callable $watcher_cb = null): array {}

		/**
		 * Return the client session id, only valid if the connections is currently connected (ie. last watcher state is ZOO_CONNECTED_STATE)
		 * @return int <p>Returns the client session id on success, and false on failure.</p>
		 * @link https://php.net/manual/en/zookeeper.getclientid.php
		 * @since PECL zookeeper >= 0.1.0
		 */
		public function getClientId(): int {}

		/**
		 * Get instance of ZookeeperConfig
		 * @return ZookeeperConfig <p>Returns instance of <code>ZookeeperConfig</code>.</p>
		 * @link https://php.net/manual/en/zookeeper.getconfig.php
		 * @since PECL zookeeper >= 0.6.0, ZooKeeper >= 3.5.0
		 */
		public function getConfig(): \ZookeeperConfig {}

		/**
		 * Return the timeout for this session, only valid if the connections is currently connected (ie. last watcher state is ZOO_CONNECTED_STATE). This value may change after a server re-connect
		 * @return int <p>Returns the timeout for this session on success, and false on failure.</p>
		 * @link https://php.net/manual/en/zookeeper.getrecvtimeout.php
		 * @since PECL zookeeper >= 0.1.0
		 */
		public function getRecvTimeout(): int {}

		/**
		 * Get the state of the zookeeper connection
		 * @return int <p>Returns the state of zookeeper connection on success, and false on failure.</p>
		 * @link https://php.net/manual/en/zookeeper.getstate.php
		 * @since PECL zookeeper >= 0.1.0
		 */
		public function getState(): int {}

		/**
		 * Checks if the current zookeeper connection state can be recovered
		 * <p>The application must close the handle and try to reconnect.</p>
		 * @return bool <p>Returns true/false on success, and false on failure.</p>
		 * @link https://php.net/manual/en/zookeeper.isrecoverable.php
		 * @since PECL zookeeper >= 0.1.0
		 */
		public function isRecoverable(): bool {}

		/**
		 * Sets the data associated with a node
		 * @param string $path <p>The name of the node. Expressed as a file name with slashes separating ancestors of the node.</p>
		 * @param string $value <p>The data to be stored in the node.</p>
		 * @param int $version <p>The expected version of the node. The function will fail if the actual version of the node does not match the expected version. If -1 is used the version check will not take place.</p>
		 * @param array $stat <p>If not NULL, will hold the value of stat for the path on return.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/zookeeper.set.php
		 * @since PECL zookeeper >= 0.1.0
		 */
		public function set(string $path, string $value, int $version = -1, array &$stat = null): bool {}

		/**
		 * Sets the acl associated with a node synchronously
		 * @param string $path <p>The name of the node. Expressed as a file name with slashes separating ancestors of the node.</p>
		 * @param int $version <p>The expected version of the path.</p>
		 * @param array $acl <p>The acl to be set on the path.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/zookeeper.setacl.php
		 * @since PECL zookeeper >= 0.1.0
		 */
		public function setAcl(string $path, int $version, array $acl): bool {}

		/**
		 * Sets the debugging level for the library
		 * @param int $logLevel <p>ZooKeeper log level constants.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/zookeeper.setdebuglevel.php
		 * @since PECL zookeeper >= 0.1.0
		 */
		public static function setDebugLevel(int $logLevel): bool {}

		/**
		 * Enable/disable quorum endpoint order randomization
		 * <p>If passed a true value, will make the client connect to quorum peers in the order as specified in the zookeeper_init() call. A false value causes zookeeper_init() to permute the peer endpoints which is good for more even client connection distribution among the quorum peers. ZooKeeper C Client uses false by default.</p>
		 * @param bool $yesOrNo <p>Disable/enable quorum endpoint order randomization.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/zookeeper.setdeterministicconnorder.php
		 * @since PECL zookeeper >= 0.1.0
		 */
		public static function setDeterministicConnOrder(bool $yesOrNo): bool {}

		/**
		 * Sets the stream to be used by the library for logging
		 * <p>The zookeeper library uses stderr as its default log stream. Application must make sure the stream is writable. Passing in NULL resets the stream to its default value (stderr).</p>
		 * @param resource $stream <p>The stream to be used by the library for logging.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/zookeeper.setlogstream.php
		 * @since PECL zookeeper >= 0.1.0
		 */
		public function setLogStream($stream): bool {}

		/**
		 * Set a watcher function
		 * @param callable $watcher_cb <p>A watch will be set at the server to notify the client if the node changes.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/zookeeper.setwatcher.php
		 * @since PECL zookeeper >= 0.1.0
		 */
		public function setWatcher(callable $watcher_cb): bool {}
	}

	/**
	 * <p>The ZooKeeper authentication exception handling class.</p>
	 * @link https://php.net/manual/en/class.zookeeperauthenticationexception.php
	 * @since PECL zookeeper >= 0.3.0
	 */
	class ZookeeperAuthenticationException extends \ZookeeperException {

		/**
		 * @var string <p>The exception message</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.message
		 */
		protected string $message = "";

		/**
		 * @var string <p>The string representation of the stack trace</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.string
		 */
		private string $string = "";

		/**
		 * @var int <p>The exception code</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.code
		 */
		protected int $code;

		/**
		 * @var string <p>The filename where the exception was created</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.file
		 */
		protected string $file = "";

		/**
		 * @var int <p>The line where the exception was created</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.line
		 */
		protected int $line;

		/**
		 * @var array <p>The stack trace as an array</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.trace
		 */
		private array $trace = [];

		/**
		 * @var ?Throwable <p>The previously thrown exception</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.previous
		 */
		private ?\Throwable $previous = null;

		/**
		 * Clone the exception
		 * <p><code>Exception</code>s cannot be cloned, and attempting to do so will throw an <code>Error</code>.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/exception.clone.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		private function __clone() {}

		/**
		 * String representation of the exception
		 * <p>Returns the <code>string</code> representation of the exception.</p>
		 * @return string <p>Returns the <code>string</code> representation of the exception.</p>
		 * @link https://php.net/manual/en/exception.tostring.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function __toString(): string {}

		/**
		 * Gets the Exception code
		 * <p>Returns the Exception code.</p>
		 * @return int <p>Returns the exception code as <code>int</code> in <code>Exception</code> but possibly as other type in <code>Exception</code> descendants (for example as <code>string</code> in <code>PDOException</code>).</p>
		 * @link https://php.net/manual/en/exception.getcode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getCode(): int {}

		/**
		 * Gets the file in which the exception was created
		 * <p>Get the name of the file in which the exception was created.</p>
		 * @return string <p>Returns the filename in which the exception was created.</p>
		 * @link https://php.net/manual/en/exception.getfile.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getFile(): string {}

		/**
		 * Gets the line in which the exception was created
		 * <p>Get line number where the exception was created.</p>
		 * @return int <p>Returns the line number where the exception was created.</p>
		 * @link https://php.net/manual/en/exception.getline.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getLine(): int {}

		/**
		 * Gets the Exception message
		 * <p>Returns the Exception message.</p>
		 * @return string <p>Returns the Exception message as a string.</p>
		 * @link https://php.net/manual/en/exception.getmessage.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getMessage(): string {}

		/**
		 * Returns previous Throwable
		 * <p>Returns previous <code>Throwable</code> (which had been passed as the third parameter of <code>Exception::__construct()</code>).</p>
		 * @return ?Throwable <p>Returns the previous <code>Throwable</code> if available or <b><code>null</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/exception.getprevious.php
		 * @since PHP 5 >= 5.3.0, PHP 7, PHP 8
		 */
		final public function getPrevious(): ?\Throwable {}

		/**
		 * Gets the stack trace
		 * <p>Returns the Exception stack trace.</p>
		 * @return array <p>Returns the Exception stack trace as an <code>array</code>.</p>
		 * @link https://php.net/manual/en/exception.gettrace.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getTrace(): array {}

		/**
		 * Gets the stack trace as a string
		 * <p>Returns the Exception stack trace as a string.</p>
		 * @return string <p>Returns the Exception stack trace as a string.</p>
		 * @link https://php.net/manual/en/exception.gettraceasstring.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getTraceAsString(): string {}
	}

	/**
	 * <p>The ZooKeeper Config handling class.</p>
	 * @link https://php.net/manual/en/class.zookeeperconfig.php
	 * @since PECL zookeeper >= 0.6.0, ZooKeeper >= 3.5.0
	 */
	class ZookeeperConfig {

		/**
		 * Add servers to the ensemble
		 * @param string $members <p>Comma separated list of servers to be added to the ensemble. Each has a configuration line for a server to be added (as would appear in a configuration file), only for maj. quorums.</p>
		 * @param int $version <p>The expected version of the node. The function will fail if the actual version of the node does not match the expected version. If -1 is used the version check will not take place.</p>
		 * @param array $stat <p>If not NULL, will hold the value of stat for the path on return.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/zookeeperconfig.add.php
		 * @since PECL zookeeper >= 0.6.0, ZooKeeper >= 3.5.0
		 */
		public function add(string $members, int $version = -1, array &$stat = null): void {}

		/**
		 * Gets the last committed configuration of the ZooKeeper cluster as it is known to the server to which the client is connected, synchronously
		 * @param callable $watcher_cb <p>If nonzero, a watch will be set at the server to notify the client if the node changes.</p>
		 * @param array $stat <p>If not NULL, will hold the value of stat for the path on return.</p>
		 * @return string <p>Returns the configuration string on success, and false on failure.</p>
		 * @link https://php.net/manual/en/zookeeperconfig.get.php
		 * @since PECL zookeeper >= 0.6.0, ZooKeeper >= 3.5.0
		 */
		public function get(callable $watcher_cb = null, array &$stat = null): string {}

		/**
		 * Remove servers from the ensemble
		 * @param string $id_list <p>Comma separated list of server IDs to be removed from the ensemble. Each has an id of a server to be removed, only for maj. quorums.</p>
		 * @param int $version <p>The expected version of the node. The function will fail if the actual version of the node does not match the expected version. If -1 is used the version check will not take place.</p>
		 * @param array $stat <p>If not NULL, will hold the value of stat for the path on return.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/zookeeperconfig.remove.php
		 * @since PECL zookeeper >= 0.6.0, ZooKeeper >= 3.5.0
		 */
		public function remove(string $id_list, int $version = -1, array &$stat = null): void {}

		/**
		 * Change ZK cluster ensemble membership and roles of ensemble peers
		 * @param string $members <p>Comma separated list of new membership (e.g., contents of a membership configuration file) - for use only with a non-incremental reconfiguration.</p>
		 * @param int $version <p>The expected version of the node. The function will fail if the actual version of the node does not match the expected version. If -1 is used the version check will not take place.</p>
		 * @param array $stat <p>If not NULL, will hold the value of stat for the path on return.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/zookeeperconfig.set.php
		 * @since PECL zookeeper >= 0.6.0, ZooKeeper >= 3.5.0
		 */
		public function set(string $members, int $version = -1, array &$stat = null): void {}
	}

	/**
	 * <p>The ZooKeeper connection exception handling class.</p>
	 * @link https://php.net/manual/en/class.zookeeperconnectionexception.php
	 * @since PECL zookeeper >= 0.3.0
	 */
	class ZookeeperConnectionException extends \ZookeeperException {

		/**
		 * @var string <p>The exception message</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.message
		 */
		protected string $message = "";

		/**
		 * @var string <p>The string representation of the stack trace</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.string
		 */
		private string $string = "";

		/**
		 * @var int <p>The exception code</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.code
		 */
		protected int $code;

		/**
		 * @var string <p>The filename where the exception was created</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.file
		 */
		protected string $file = "";

		/**
		 * @var int <p>The line where the exception was created</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.line
		 */
		protected int $line;

		/**
		 * @var array <p>The stack trace as an array</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.trace
		 */
		private array $trace = [];

		/**
		 * @var ?Throwable <p>The previously thrown exception</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.previous
		 */
		private ?\Throwable $previous = null;

		/**
		 * Clone the exception
		 * <p><code>Exception</code>s cannot be cloned, and attempting to do so will throw an <code>Error</code>.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/exception.clone.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		private function __clone() {}

		/**
		 * String representation of the exception
		 * <p>Returns the <code>string</code> representation of the exception.</p>
		 * @return string <p>Returns the <code>string</code> representation of the exception.</p>
		 * @link https://php.net/manual/en/exception.tostring.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function __toString(): string {}

		/**
		 * Gets the Exception code
		 * <p>Returns the Exception code.</p>
		 * @return int <p>Returns the exception code as <code>int</code> in <code>Exception</code> but possibly as other type in <code>Exception</code> descendants (for example as <code>string</code> in <code>PDOException</code>).</p>
		 * @link https://php.net/manual/en/exception.getcode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getCode(): int {}

		/**
		 * Gets the file in which the exception was created
		 * <p>Get the name of the file in which the exception was created.</p>
		 * @return string <p>Returns the filename in which the exception was created.</p>
		 * @link https://php.net/manual/en/exception.getfile.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getFile(): string {}

		/**
		 * Gets the line in which the exception was created
		 * <p>Get line number where the exception was created.</p>
		 * @return int <p>Returns the line number where the exception was created.</p>
		 * @link https://php.net/manual/en/exception.getline.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getLine(): int {}

		/**
		 * Gets the Exception message
		 * <p>Returns the Exception message.</p>
		 * @return string <p>Returns the Exception message as a string.</p>
		 * @link https://php.net/manual/en/exception.getmessage.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getMessage(): string {}

		/**
		 * Returns previous Throwable
		 * <p>Returns previous <code>Throwable</code> (which had been passed as the third parameter of <code>Exception::__construct()</code>).</p>
		 * @return ?Throwable <p>Returns the previous <code>Throwable</code> if available or <b><code>null</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/exception.getprevious.php
		 * @since PHP 5 >= 5.3.0, PHP 7, PHP 8
		 */
		final public function getPrevious(): ?\Throwable {}

		/**
		 * Gets the stack trace
		 * <p>Returns the Exception stack trace.</p>
		 * @return array <p>Returns the Exception stack trace as an <code>array</code>.</p>
		 * @link https://php.net/manual/en/exception.gettrace.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getTrace(): array {}

		/**
		 * Gets the stack trace as a string
		 * <p>Returns the Exception stack trace as a string.</p>
		 * @return string <p>Returns the Exception stack trace as a string.</p>
		 * @link https://php.net/manual/en/exception.gettraceasstring.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getTraceAsString(): string {}
	}

	/**
	 * <p>The ZooKeeper exception handling class.</p>
	 * @link https://php.net/manual/en/class.zookeeperexception.php
	 * @since PECL zookeeper >= 0.3.0
	 */
	class ZookeeperException extends \Exception {

		/**
		 * @var string <p>The exception message</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.message
		 */
		protected string $message = "";

		/**
		 * @var string <p>The string representation of the stack trace</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.string
		 */
		private string $string = "";

		/**
		 * @var int <p>The exception code</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.code
		 */
		protected int $code;

		/**
		 * @var string <p>The filename where the exception was created</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.file
		 */
		protected string $file = "";

		/**
		 * @var int <p>The line where the exception was created</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.line
		 */
		protected int $line;

		/**
		 * @var array <p>The stack trace as an array</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.trace
		 */
		private array $trace = [];

		/**
		 * @var ?Throwable <p>The previously thrown exception</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.previous
		 */
		private ?\Throwable $previous = null;

		/**
		 * Clone the exception
		 * <p><code>Exception</code>s cannot be cloned, and attempting to do so will throw an <code>Error</code>.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/exception.clone.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		private function __clone() {}

		/**
		 * String representation of the exception
		 * <p>Returns the <code>string</code> representation of the exception.</p>
		 * @return string <p>Returns the <code>string</code> representation of the exception.</p>
		 * @link https://php.net/manual/en/exception.tostring.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function __toString(): string {}

		/**
		 * Gets the Exception code
		 * <p>Returns the Exception code.</p>
		 * @return int <p>Returns the exception code as <code>int</code> in <code>Exception</code> but possibly as other type in <code>Exception</code> descendants (for example as <code>string</code> in <code>PDOException</code>).</p>
		 * @link https://php.net/manual/en/exception.getcode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getCode(): int {}

		/**
		 * Gets the file in which the exception was created
		 * <p>Get the name of the file in which the exception was created.</p>
		 * @return string <p>Returns the filename in which the exception was created.</p>
		 * @link https://php.net/manual/en/exception.getfile.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getFile(): string {}

		/**
		 * Gets the line in which the exception was created
		 * <p>Get line number where the exception was created.</p>
		 * @return int <p>Returns the line number where the exception was created.</p>
		 * @link https://php.net/manual/en/exception.getline.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getLine(): int {}

		/**
		 * Gets the Exception message
		 * <p>Returns the Exception message.</p>
		 * @return string <p>Returns the Exception message as a string.</p>
		 * @link https://php.net/manual/en/exception.getmessage.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getMessage(): string {}

		/**
		 * Returns previous Throwable
		 * <p>Returns previous <code>Throwable</code> (which had been passed as the third parameter of <code>Exception::__construct()</code>).</p>
		 * @return ?Throwable <p>Returns the previous <code>Throwable</code> if available or <b><code>null</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/exception.getprevious.php
		 * @since PHP 5 >= 5.3.0, PHP 7, PHP 8
		 */
		final public function getPrevious(): ?\Throwable {}

		/**
		 * Gets the stack trace
		 * <p>Returns the Exception stack trace.</p>
		 * @return array <p>Returns the Exception stack trace as an <code>array</code>.</p>
		 * @link https://php.net/manual/en/exception.gettrace.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getTrace(): array {}

		/**
		 * Gets the stack trace as a string
		 * <p>Returns the Exception stack trace as a string.</p>
		 * @return string <p>Returns the Exception stack trace as a string.</p>
		 * @link https://php.net/manual/en/exception.gettraceasstring.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getTraceAsString(): string {}
	}

	/**
	 * <p>The ZooKeeper exception (while marshalling or unmarshalling data) handling class.</p>
	 * @link https://php.net/manual/en/class.zookeepermarshallingexception.php
	 * @since PECL zookeeper >= 0.3.0
	 */
	class ZookeeperMarshallingException extends \ZookeeperException {

		/**
		 * @var string <p>The exception message</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.message
		 */
		protected string $message = "";

		/**
		 * @var string <p>The string representation of the stack trace</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.string
		 */
		private string $string = "";

		/**
		 * @var int <p>The exception code</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.code
		 */
		protected int $code;

		/**
		 * @var string <p>The filename where the exception was created</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.file
		 */
		protected string $file = "";

		/**
		 * @var int <p>The line where the exception was created</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.line
		 */
		protected int $line;

		/**
		 * @var array <p>The stack trace as an array</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.trace
		 */
		private array $trace = [];

		/**
		 * @var ?Throwable <p>The previously thrown exception</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.previous
		 */
		private ?\Throwable $previous = null;

		/**
		 * Clone the exception
		 * <p><code>Exception</code>s cannot be cloned, and attempting to do so will throw an <code>Error</code>.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/exception.clone.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		private function __clone() {}

		/**
		 * String representation of the exception
		 * <p>Returns the <code>string</code> representation of the exception.</p>
		 * @return string <p>Returns the <code>string</code> representation of the exception.</p>
		 * @link https://php.net/manual/en/exception.tostring.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function __toString(): string {}

		/**
		 * Gets the Exception code
		 * <p>Returns the Exception code.</p>
		 * @return int <p>Returns the exception code as <code>int</code> in <code>Exception</code> but possibly as other type in <code>Exception</code> descendants (for example as <code>string</code> in <code>PDOException</code>).</p>
		 * @link https://php.net/manual/en/exception.getcode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getCode(): int {}

		/**
		 * Gets the file in which the exception was created
		 * <p>Get the name of the file in which the exception was created.</p>
		 * @return string <p>Returns the filename in which the exception was created.</p>
		 * @link https://php.net/manual/en/exception.getfile.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getFile(): string {}

		/**
		 * Gets the line in which the exception was created
		 * <p>Get line number where the exception was created.</p>
		 * @return int <p>Returns the line number where the exception was created.</p>
		 * @link https://php.net/manual/en/exception.getline.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getLine(): int {}

		/**
		 * Gets the Exception message
		 * <p>Returns the Exception message.</p>
		 * @return string <p>Returns the Exception message as a string.</p>
		 * @link https://php.net/manual/en/exception.getmessage.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getMessage(): string {}

		/**
		 * Returns previous Throwable
		 * <p>Returns previous <code>Throwable</code> (which had been passed as the third parameter of <code>Exception::__construct()</code>).</p>
		 * @return ?Throwable <p>Returns the previous <code>Throwable</code> if available or <b><code>null</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/exception.getprevious.php
		 * @since PHP 5 >= 5.3.0, PHP 7, PHP 8
		 */
		final public function getPrevious(): ?\Throwable {}

		/**
		 * Gets the stack trace
		 * <p>Returns the Exception stack trace.</p>
		 * @return array <p>Returns the Exception stack trace as an <code>array</code>.</p>
		 * @link https://php.net/manual/en/exception.gettrace.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getTrace(): array {}

		/**
		 * Gets the stack trace as a string
		 * <p>Returns the Exception stack trace as a string.</p>
		 * @return string <p>Returns the Exception stack trace as a string.</p>
		 * @link https://php.net/manual/en/exception.gettraceasstring.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getTraceAsString(): string {}
	}

	/**
	 * <p>The ZooKeeper exception (when node does not exist) handling class.</p>
	 * @link https://php.net/manual/en/class.zookeepernonodeexception.php
	 * @since PECL zookeeper >= 0.3.0
	 */
	class ZookeeperNoNodeException extends \ZookeeperException {

		/**
		 * @var string <p>The exception message</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.message
		 */
		protected string $message = "";

		/**
		 * @var string <p>The string representation of the stack trace</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.string
		 */
		private string $string = "";

		/**
		 * @var int <p>The exception code</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.code
		 */
		protected int $code;

		/**
		 * @var string <p>The filename where the exception was created</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.file
		 */
		protected string $file = "";

		/**
		 * @var int <p>The line where the exception was created</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.line
		 */
		protected int $line;

		/**
		 * @var array <p>The stack trace as an array</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.trace
		 */
		private array $trace = [];

		/**
		 * @var ?Throwable <p>The previously thrown exception</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.previous
		 */
		private ?\Throwable $previous = null;

		/**
		 * Clone the exception
		 * <p><code>Exception</code>s cannot be cloned, and attempting to do so will throw an <code>Error</code>.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/exception.clone.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		private function __clone() {}

		/**
		 * String representation of the exception
		 * <p>Returns the <code>string</code> representation of the exception.</p>
		 * @return string <p>Returns the <code>string</code> representation of the exception.</p>
		 * @link https://php.net/manual/en/exception.tostring.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function __toString(): string {}

		/**
		 * Gets the Exception code
		 * <p>Returns the Exception code.</p>
		 * @return int <p>Returns the exception code as <code>int</code> in <code>Exception</code> but possibly as other type in <code>Exception</code> descendants (for example as <code>string</code> in <code>PDOException</code>).</p>
		 * @link https://php.net/manual/en/exception.getcode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getCode(): int {}

		/**
		 * Gets the file in which the exception was created
		 * <p>Get the name of the file in which the exception was created.</p>
		 * @return string <p>Returns the filename in which the exception was created.</p>
		 * @link https://php.net/manual/en/exception.getfile.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getFile(): string {}

		/**
		 * Gets the line in which the exception was created
		 * <p>Get line number where the exception was created.</p>
		 * @return int <p>Returns the line number where the exception was created.</p>
		 * @link https://php.net/manual/en/exception.getline.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getLine(): int {}

		/**
		 * Gets the Exception message
		 * <p>Returns the Exception message.</p>
		 * @return string <p>Returns the Exception message as a string.</p>
		 * @link https://php.net/manual/en/exception.getmessage.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getMessage(): string {}

		/**
		 * Returns previous Throwable
		 * <p>Returns previous <code>Throwable</code> (which had been passed as the third parameter of <code>Exception::__construct()</code>).</p>
		 * @return ?Throwable <p>Returns the previous <code>Throwable</code> if available or <b><code>null</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/exception.getprevious.php
		 * @since PHP 5 >= 5.3.0, PHP 7, PHP 8
		 */
		final public function getPrevious(): ?\Throwable {}

		/**
		 * Gets the stack trace
		 * <p>Returns the Exception stack trace.</p>
		 * @return array <p>Returns the Exception stack trace as an <code>array</code>.</p>
		 * @link https://php.net/manual/en/exception.gettrace.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getTrace(): array {}

		/**
		 * Gets the stack trace as a string
		 * <p>Returns the Exception stack trace as a string.</p>
		 * @return string <p>Returns the Exception stack trace as a string.</p>
		 * @link https://php.net/manual/en/exception.gettraceasstring.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getTraceAsString(): string {}
	}

	/**
	 * <p>The ZooKeeper operation timeout exception handling class.</p>
	 * @link https://php.net/manual/en/class.zookeeperoperationtimeoutexception.php
	 * @since PECL zookeeper >= 0.3.0
	 */
	class ZookeeperOperationTimeoutException extends \ZookeeperException {

		/**
		 * @var string <p>The exception message</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.message
		 */
		protected string $message = "";

		/**
		 * @var string <p>The string representation of the stack trace</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.string
		 */
		private string $string = "";

		/**
		 * @var int <p>The exception code</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.code
		 */
		protected int $code;

		/**
		 * @var string <p>The filename where the exception was created</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.file
		 */
		protected string $file = "";

		/**
		 * @var int <p>The line where the exception was created</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.line
		 */
		protected int $line;

		/**
		 * @var array <p>The stack trace as an array</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.trace
		 */
		private array $trace = [];

		/**
		 * @var ?Throwable <p>The previously thrown exception</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.previous
		 */
		private ?\Throwable $previous = null;

		/**
		 * Clone the exception
		 * <p><code>Exception</code>s cannot be cloned, and attempting to do so will throw an <code>Error</code>.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/exception.clone.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		private function __clone() {}

		/**
		 * String representation of the exception
		 * <p>Returns the <code>string</code> representation of the exception.</p>
		 * @return string <p>Returns the <code>string</code> representation of the exception.</p>
		 * @link https://php.net/manual/en/exception.tostring.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function __toString(): string {}

		/**
		 * Gets the Exception code
		 * <p>Returns the Exception code.</p>
		 * @return int <p>Returns the exception code as <code>int</code> in <code>Exception</code> but possibly as other type in <code>Exception</code> descendants (for example as <code>string</code> in <code>PDOException</code>).</p>
		 * @link https://php.net/manual/en/exception.getcode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getCode(): int {}

		/**
		 * Gets the file in which the exception was created
		 * <p>Get the name of the file in which the exception was created.</p>
		 * @return string <p>Returns the filename in which the exception was created.</p>
		 * @link https://php.net/manual/en/exception.getfile.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getFile(): string {}

		/**
		 * Gets the line in which the exception was created
		 * <p>Get line number where the exception was created.</p>
		 * @return int <p>Returns the line number where the exception was created.</p>
		 * @link https://php.net/manual/en/exception.getline.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getLine(): int {}

		/**
		 * Gets the Exception message
		 * <p>Returns the Exception message.</p>
		 * @return string <p>Returns the Exception message as a string.</p>
		 * @link https://php.net/manual/en/exception.getmessage.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getMessage(): string {}

		/**
		 * Returns previous Throwable
		 * <p>Returns previous <code>Throwable</code> (which had been passed as the third parameter of <code>Exception::__construct()</code>).</p>
		 * @return ?Throwable <p>Returns the previous <code>Throwable</code> if available or <b><code>null</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/exception.getprevious.php
		 * @since PHP 5 >= 5.3.0, PHP 7, PHP 8
		 */
		final public function getPrevious(): ?\Throwable {}

		/**
		 * Gets the stack trace
		 * <p>Returns the Exception stack trace.</p>
		 * @return array <p>Returns the Exception stack trace as an <code>array</code>.</p>
		 * @link https://php.net/manual/en/exception.gettrace.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getTrace(): array {}

		/**
		 * Gets the stack trace as a string
		 * <p>Returns the Exception stack trace as a string.</p>
		 * @return string <p>Returns the Exception stack trace as a string.</p>
		 * @link https://php.net/manual/en/exception.gettraceasstring.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getTraceAsString(): string {}
	}

	/**
	 * <p>The ZooKeeper session exception handling class.</p>
	 * @link https://php.net/manual/en/class.zookeepersessionexception.php
	 * @since PECL zookeeper >= 0.3.0
	 */
	class ZookeeperSessionException extends \ZookeeperException {

		/**
		 * @var string <p>The exception message</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.message
		 */
		protected string $message = "";

		/**
		 * @var string <p>The string representation of the stack trace</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.string
		 */
		private string $string = "";

		/**
		 * @var int <p>The exception code</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.code
		 */
		protected int $code;

		/**
		 * @var string <p>The filename where the exception was created</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.file
		 */
		protected string $file = "";

		/**
		 * @var int <p>The line where the exception was created</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.line
		 */
		protected int $line;

		/**
		 * @var array <p>The stack trace as an array</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.trace
		 */
		private array $trace = [];

		/**
		 * @var ?Throwable <p>The previously thrown exception</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.previous
		 */
		private ?\Throwable $previous = null;

		/**
		 * Clone the exception
		 * <p><code>Exception</code>s cannot be cloned, and attempting to do so will throw an <code>Error</code>.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/exception.clone.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		private function __clone() {}

		/**
		 * String representation of the exception
		 * <p>Returns the <code>string</code> representation of the exception.</p>
		 * @return string <p>Returns the <code>string</code> representation of the exception.</p>
		 * @link https://php.net/manual/en/exception.tostring.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function __toString(): string {}

		/**
		 * Gets the Exception code
		 * <p>Returns the Exception code.</p>
		 * @return int <p>Returns the exception code as <code>int</code> in <code>Exception</code> but possibly as other type in <code>Exception</code> descendants (for example as <code>string</code> in <code>PDOException</code>).</p>
		 * @link https://php.net/manual/en/exception.getcode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getCode(): int {}

		/**
		 * Gets the file in which the exception was created
		 * <p>Get the name of the file in which the exception was created.</p>
		 * @return string <p>Returns the filename in which the exception was created.</p>
		 * @link https://php.net/manual/en/exception.getfile.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getFile(): string {}

		/**
		 * Gets the line in which the exception was created
		 * <p>Get line number where the exception was created.</p>
		 * @return int <p>Returns the line number where the exception was created.</p>
		 * @link https://php.net/manual/en/exception.getline.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getLine(): int {}

		/**
		 * Gets the Exception message
		 * <p>Returns the Exception message.</p>
		 * @return string <p>Returns the Exception message as a string.</p>
		 * @link https://php.net/manual/en/exception.getmessage.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getMessage(): string {}

		/**
		 * Returns previous Throwable
		 * <p>Returns previous <code>Throwable</code> (which had been passed as the third parameter of <code>Exception::__construct()</code>).</p>
		 * @return ?Throwable <p>Returns the previous <code>Throwable</code> if available or <b><code>null</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/exception.getprevious.php
		 * @since PHP 5 >= 5.3.0, PHP 7, PHP 8
		 */
		final public function getPrevious(): ?\Throwable {}

		/**
		 * Gets the stack trace
		 * <p>Returns the Exception stack trace.</p>
		 * @return array <p>Returns the Exception stack trace as an <code>array</code>.</p>
		 * @link https://php.net/manual/en/exception.gettrace.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getTrace(): array {}

		/**
		 * Gets the stack trace as a string
		 * <p>Returns the Exception stack trace as a string.</p>
		 * @return string <p>Returns the Exception stack trace as a string.</p>
		 * @link https://php.net/manual/en/exception.gettraceasstring.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getTraceAsString(): string {}
	}

	/**
	 * Calls callbacks for pending operations
	 * <p>The <b>zookeeper_dispatch()</b> function calls the callbacks passed by operations like <code>Zookeeper::get()</code> or <code>Zookeeper::exists()</code>.</p><p>Since version 0.4.0, this function must be called manually to achieve asynchronous operations. If you want that to be done automatically, you also can declare ticks at the beginning of your program.</p><p>After PHP 7.1, you can ignore this function. This extension uses EG(vm_interrupt) to implement async dispatch.</p>
	 * @return void <p>No value is returned.</p>
	 * @link https://php.net/manual/en/function.zookeeper-dispatch.php
	 * @since PECL zookeeper >= 0.4.0
	 */
	function zookeeper_dispatch(): void {}

}
