﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/kinesisvideo/KinesisVideoRequest.h>
#include <aws/kinesisvideo/KinesisVideo_EXPORTS.h>
#include <aws/kinesisvideo/model/StreamStorageConfiguration.h>

#include <utility>

namespace Aws {
namespace KinesisVideo {
namespace Model {

/**
 */
class CreateStreamRequest : public KinesisVideoRequest {
 public:
  AWS_KINESISVIDEO_API CreateStreamRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateStream"; }

  AWS_KINESISVIDEO_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the device that is writing to the stream. </p>  <p>In the
   * current implementation, Kinesis Video Streams doesn't use this name.</p>
   */
  inline const Aws::String& GetDeviceName() const { return m_deviceName; }
  inline bool DeviceNameHasBeenSet() const { return m_deviceNameHasBeenSet; }
  template <typename DeviceNameT = Aws::String>
  void SetDeviceName(DeviceNameT&& value) {
    m_deviceNameHasBeenSet = true;
    m_deviceName = std::forward<DeviceNameT>(value);
  }
  template <typename DeviceNameT = Aws::String>
  CreateStreamRequest& WithDeviceName(DeviceNameT&& value) {
    SetDeviceName(std::forward<DeviceNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A name for the stream that you are creating.</p> <p>The stream name is an
   * identifier for the stream, and must be unique for each account and region.</p>
   */
  inline const Aws::String& GetStreamName() const { return m_streamName; }
  inline bool StreamNameHasBeenSet() const { return m_streamNameHasBeenSet; }
  template <typename StreamNameT = Aws::String>
  void SetStreamName(StreamNameT&& value) {
    m_streamNameHasBeenSet = true;
    m_streamName = std::forward<StreamNameT>(value);
  }
  template <typename StreamNameT = Aws::String>
  CreateStreamRequest& WithStreamName(StreamNameT&& value) {
    SetStreamName(std::forward<StreamNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The media type of the stream. Consumers of the stream can use this
   * information when processing the stream. For more information about media types,
   * see <a
   * href="http://www.iana.org/assignments/media-types/media-types.xhtml">Media
   * Types</a>. If you choose to specify the <code>MediaType</code>, see <a
   * href="https://tools.ietf.org/html/rfc6838#section-4.2">Naming Requirements</a>
   * for guidelines.</p> <p>Example valid values include "video/h264" and
   * "video/h264,audio/aac".</p> <p>This parameter is optional; the default value is
   * <code>null</code> (or empty in JSON).</p>
   */
  inline const Aws::String& GetMediaType() const { return m_mediaType; }
  inline bool MediaTypeHasBeenSet() const { return m_mediaTypeHasBeenSet; }
  template <typename MediaTypeT = Aws::String>
  void SetMediaType(MediaTypeT&& value) {
    m_mediaTypeHasBeenSet = true;
    m_mediaType = std::forward<MediaTypeT>(value);
  }
  template <typename MediaTypeT = Aws::String>
  CreateStreamRequest& WithMediaType(MediaTypeT&& value) {
    SetMediaType(std::forward<MediaTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the Key Management Service (KMS) key that you want Kinesis Video
   * Streams to use to encrypt stream data.</p> <p>If no key ID is specified, the
   * default, Kinesis Video-managed key (<code>aws/kinesisvideo</code>) is used.</p>
   * <p> For more information, see <a
   * href="https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html#API_DescribeKey_RequestParameters">DescribeKey</a>.
   * </p>
   */
  inline const Aws::String& GetKmsKeyId() const { return m_kmsKeyId; }
  inline bool KmsKeyIdHasBeenSet() const { return m_kmsKeyIdHasBeenSet; }
  template <typename KmsKeyIdT = Aws::String>
  void SetKmsKeyId(KmsKeyIdT&& value) {
    m_kmsKeyIdHasBeenSet = true;
    m_kmsKeyId = std::forward<KmsKeyIdT>(value);
  }
  template <typename KmsKeyIdT = Aws::String>
  CreateStreamRequest& WithKmsKeyId(KmsKeyIdT&& value) {
    SetKmsKeyId(std::forward<KmsKeyIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of hours that you want to retain the data in the stream. Kinesis
   * Video Streams retains the data in a data store that is associated with the
   * stream.</p> <p>The default value is 0, indicating that the stream does not
   * persist data. The minimum is 1 hour.</p> <p>When the
   * <code>DataRetentionInHours</code> value is 0, consumers can still consume the
   * fragments that remain in the service host buffer, which has a retention time
   * limit of 5 minutes and a retention memory limit of 200 MB. Fragments are removed
   * from the buffer when either limit is reached.</p>
   */
  inline int GetDataRetentionInHours() const { return m_dataRetentionInHours; }
  inline bool DataRetentionInHoursHasBeenSet() const { return m_dataRetentionInHoursHasBeenSet; }
  inline void SetDataRetentionInHours(int value) {
    m_dataRetentionInHoursHasBeenSet = true;
    m_dataRetentionInHours = value;
  }
  inline CreateStreamRequest& WithDataRetentionInHours(int value) {
    SetDataRetentionInHours(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of tags to associate with the specified stream. Each tag is a
   * key-value pair (the value is optional).</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateStreamRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateStreamRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The configuration for the stream's storage, including the default storage
   * tier for stream data. This configuration determines how stream data is stored
   * and accessed, with different tiers offering varying levels of performance and
   * cost optimization.</p> <p>If not specified, the stream will use the default
   * storage configuration with HOT tier for optimal performance.</p>
   */
  inline const StreamStorageConfiguration& GetStreamStorageConfiguration() const { return m_streamStorageConfiguration; }
  inline bool StreamStorageConfigurationHasBeenSet() const { return m_streamStorageConfigurationHasBeenSet; }
  template <typename StreamStorageConfigurationT = StreamStorageConfiguration>
  void SetStreamStorageConfiguration(StreamStorageConfigurationT&& value) {
    m_streamStorageConfigurationHasBeenSet = true;
    m_streamStorageConfiguration = std::forward<StreamStorageConfigurationT>(value);
  }
  template <typename StreamStorageConfigurationT = StreamStorageConfiguration>
  CreateStreamRequest& WithStreamStorageConfiguration(StreamStorageConfigurationT&& value) {
    SetStreamStorageConfiguration(std::forward<StreamStorageConfigurationT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_deviceName;

  Aws::String m_streamName;

  Aws::String m_mediaType;

  Aws::String m_kmsKeyId;

  int m_dataRetentionInHours{0};

  Aws::Map<Aws::String, Aws::String> m_tags;

  StreamStorageConfiguration m_streamStorageConfiguration;
  bool m_deviceNameHasBeenSet = false;
  bool m_streamNameHasBeenSet = false;
  bool m_mediaTypeHasBeenSet = false;
  bool m_kmsKeyIdHasBeenSet = false;
  bool m_dataRetentionInHoursHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_streamStorageConfigurationHasBeenSet = false;
};

}  // namespace Model
}  // namespace KinesisVideo
}  // namespace Aws
