﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/pinpoint/Pinpoint_EXPORTS.h>
#include <aws/pinpoint/model/MessageHeader.h>
#include <aws/pinpoint/model/TemplateType.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Pinpoint {
namespace Model {

/**
 * <p>Provides information about the content and settings for a message template
 * that can be used in messages that are sent through the email
 * channel.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/pinpoint-2016-12-01/EmailTemplateResponse">AWS
 * API Reference</a></p>
 */
class EmailTemplateResponse {
 public:
  AWS_PINPOINT_API EmailTemplateResponse() = default;
  AWS_PINPOINT_API EmailTemplateResponse(Aws::Utils::Json::JsonView jsonValue);
  AWS_PINPOINT_API EmailTemplateResponse& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_PINPOINT_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the message template.</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  inline bool ArnHasBeenSet() const { return m_arnHasBeenSet; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  EmailTemplateResponse& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date, in ISO 8601 format, when the message template was created.</p>
   */
  inline const Aws::String& GetCreationDate() const { return m_creationDate; }
  inline bool CreationDateHasBeenSet() const { return m_creationDateHasBeenSet; }
  template <typename CreationDateT = Aws::String>
  void SetCreationDate(CreationDateT&& value) {
    m_creationDateHasBeenSet = true;
    m_creationDate = std::forward<CreationDateT>(value);
  }
  template <typename CreationDateT = Aws::String>
  EmailTemplateResponse& WithCreationDate(CreationDateT&& value) {
    SetCreationDate(std::forward<CreationDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The JSON object that specifies the default values that are used for message
   * variables in the message template. This object is a set of key-value pairs. Each
   * key defines a message variable in the template. The corresponding value defines
   * the default value for that variable.</p>
   */
  inline const Aws::String& GetDefaultSubstitutions() const { return m_defaultSubstitutions; }
  inline bool DefaultSubstitutionsHasBeenSet() const { return m_defaultSubstitutionsHasBeenSet; }
  template <typename DefaultSubstitutionsT = Aws::String>
  void SetDefaultSubstitutions(DefaultSubstitutionsT&& value) {
    m_defaultSubstitutionsHasBeenSet = true;
    m_defaultSubstitutions = std::forward<DefaultSubstitutionsT>(value);
  }
  template <typename DefaultSubstitutionsT = Aws::String>
  EmailTemplateResponse& WithDefaultSubstitutions(DefaultSubstitutionsT&& value) {
    SetDefaultSubstitutions(std::forward<DefaultSubstitutionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The message body, in HTML format, that's used in email messages that are
   * based on the message template.</p>
   */
  inline const Aws::String& GetHtmlPart() const { return m_htmlPart; }
  inline bool HtmlPartHasBeenSet() const { return m_htmlPartHasBeenSet; }
  template <typename HtmlPartT = Aws::String>
  void SetHtmlPart(HtmlPartT&& value) {
    m_htmlPartHasBeenSet = true;
    m_htmlPart = std::forward<HtmlPartT>(value);
  }
  template <typename HtmlPartT = Aws::String>
  EmailTemplateResponse& WithHtmlPart(HtmlPartT&& value) {
    SetHtmlPart(std::forward<HtmlPartT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date, in ISO 8601 format, when the message template was last
   * modified.</p>
   */
  inline const Aws::String& GetLastModifiedDate() const { return m_lastModifiedDate; }
  inline bool LastModifiedDateHasBeenSet() const { return m_lastModifiedDateHasBeenSet; }
  template <typename LastModifiedDateT = Aws::String>
  void SetLastModifiedDate(LastModifiedDateT&& value) {
    m_lastModifiedDateHasBeenSet = true;
    m_lastModifiedDate = std::forward<LastModifiedDateT>(value);
  }
  template <typename LastModifiedDateT = Aws::String>
  EmailTemplateResponse& WithLastModifiedDate(LastModifiedDateT&& value) {
    SetLastModifiedDate(std::forward<LastModifiedDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier for the recommender model that's used by the message
   * template.</p>
   */
  inline const Aws::String& GetRecommenderId() const { return m_recommenderId; }
  inline bool RecommenderIdHasBeenSet() const { return m_recommenderIdHasBeenSet; }
  template <typename RecommenderIdT = Aws::String>
  void SetRecommenderId(RecommenderIdT&& value) {
    m_recommenderIdHasBeenSet = true;
    m_recommenderId = std::forward<RecommenderIdT>(value);
  }
  template <typename RecommenderIdT = Aws::String>
  EmailTemplateResponse& WithRecommenderId(RecommenderIdT&& value) {
    SetRecommenderId(std::forward<RecommenderIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The subject line, or title, that's used in email messages that are based on
   * the message template.</p>
   */
  inline const Aws::String& GetSubject() const { return m_subject; }
  inline bool SubjectHasBeenSet() const { return m_subjectHasBeenSet; }
  template <typename SubjectT = Aws::String>
  void SetSubject(SubjectT&& value) {
    m_subjectHasBeenSet = true;
    m_subject = std::forward<SubjectT>(value);
  }
  template <typename SubjectT = Aws::String>
  EmailTemplateResponse& WithSubject(SubjectT&& value) {
    SetSubject(std::forward<SubjectT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The list of <a
   * href="https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-email.html#templates-template-name-email-model-messageheader">MessageHeaders</a>
   * for the email. You can have up to 15 Headers.</p>
   */
  inline const Aws::Vector<MessageHeader>& GetHeaders() const { return m_headers; }
  inline bool HeadersHasBeenSet() const { return m_headersHasBeenSet; }
  template <typename HeadersT = Aws::Vector<MessageHeader>>
  void SetHeaders(HeadersT&& value) {
    m_headersHasBeenSet = true;
    m_headers = std::forward<HeadersT>(value);
  }
  template <typename HeadersT = Aws::Vector<MessageHeader>>
  EmailTemplateResponse& WithHeaders(HeadersT&& value) {
    SetHeaders(std::forward<HeadersT>(value));
    return *this;
  }
  template <typename HeadersT = MessageHeader>
  EmailTemplateResponse& AddHeaders(HeadersT&& value) {
    m_headersHasBeenSet = true;
    m_headers.emplace_back(std::forward<HeadersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A string-to-string map of key-value pairs that identifies the tags that are
   * associated with the message template. Each tag consists of a required tag key
   * and an associated tag value.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  EmailTemplateResponse& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  EmailTemplateResponse& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The custom description of the message template.</p>
   */
  inline const Aws::String& GetTemplateDescription() const { return m_templateDescription; }
  inline bool TemplateDescriptionHasBeenSet() const { return m_templateDescriptionHasBeenSet; }
  template <typename TemplateDescriptionT = Aws::String>
  void SetTemplateDescription(TemplateDescriptionT&& value) {
    m_templateDescriptionHasBeenSet = true;
    m_templateDescription = std::forward<TemplateDescriptionT>(value);
  }
  template <typename TemplateDescriptionT = Aws::String>
  EmailTemplateResponse& WithTemplateDescription(TemplateDescriptionT&& value) {
    SetTemplateDescription(std::forward<TemplateDescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the message template.</p>
   */
  inline const Aws::String& GetTemplateName() const { return m_templateName; }
  inline bool TemplateNameHasBeenSet() const { return m_templateNameHasBeenSet; }
  template <typename TemplateNameT = Aws::String>
  void SetTemplateName(TemplateNameT&& value) {
    m_templateNameHasBeenSet = true;
    m_templateName = std::forward<TemplateNameT>(value);
  }
  template <typename TemplateNameT = Aws::String>
  EmailTemplateResponse& WithTemplateName(TemplateNameT&& value) {
    SetTemplateName(std::forward<TemplateNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of channel that the message template is designed for. For an email
   * template, this value is EMAIL.</p>
   */
  inline TemplateType GetTemplateType() const { return m_templateType; }
  inline bool TemplateTypeHasBeenSet() const { return m_templateTypeHasBeenSet; }
  inline void SetTemplateType(TemplateType value) {
    m_templateTypeHasBeenSet = true;
    m_templateType = value;
  }
  inline EmailTemplateResponse& WithTemplateType(TemplateType value) {
    SetTemplateType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The message body, in plain text format, that's used in email messages that
   * are based on the message template.</p>
   */
  inline const Aws::String& GetTextPart() const { return m_textPart; }
  inline bool TextPartHasBeenSet() const { return m_textPartHasBeenSet; }
  template <typename TextPartT = Aws::String>
  void SetTextPart(TextPartT&& value) {
    m_textPartHasBeenSet = true;
    m_textPart = std::forward<TextPartT>(value);
  }
  template <typename TextPartT = Aws::String>
  EmailTemplateResponse& WithTextPart(TextPartT&& value) {
    SetTextPart(std::forward<TextPartT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier, as an integer, for the active version of the message
   * template, or the version of the template that you specified by using the version
   * parameter in your request.</p>
   */
  inline const Aws::String& GetVersion() const { return m_version; }
  inline bool VersionHasBeenSet() const { return m_versionHasBeenSet; }
  template <typename VersionT = Aws::String>
  void SetVersion(VersionT&& value) {
    m_versionHasBeenSet = true;
    m_version = std::forward<VersionT>(value);
  }
  template <typename VersionT = Aws::String>
  EmailTemplateResponse& WithVersion(VersionT&& value) {
    SetVersion(std::forward<VersionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_arn;

  Aws::String m_creationDate;

  Aws::String m_defaultSubstitutions;

  Aws::String m_htmlPart;

  Aws::String m_lastModifiedDate;

  Aws::String m_recommenderId;

  Aws::String m_subject;

  Aws::Vector<MessageHeader> m_headers;

  Aws::Map<Aws::String, Aws::String> m_tags;

  Aws::String m_templateDescription;

  Aws::String m_templateName;

  TemplateType m_templateType{TemplateType::NOT_SET};

  Aws::String m_textPart;

  Aws::String m_version;
  bool m_arnHasBeenSet = false;
  bool m_creationDateHasBeenSet = false;
  bool m_defaultSubstitutionsHasBeenSet = false;
  bool m_htmlPartHasBeenSet = false;
  bool m_lastModifiedDateHasBeenSet = false;
  bool m_recommenderIdHasBeenSet = false;
  bool m_subjectHasBeenSet = false;
  bool m_headersHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_templateDescriptionHasBeenSet = false;
  bool m_templateNameHasBeenSet = false;
  bool m_templateTypeHasBeenSet = false;
  bool m_textPartHasBeenSet = false;
  bool m_versionHasBeenSet = false;
};

}  // namespace Model
}  // namespace Pinpoint
}  // namespace Aws
