/*
 * MultiHostHandler.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 1999-2002 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: cstevens.
 * Portions created by cstevens are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): cstevens, suhler.
 *
 * Version:  2.1
 * Created by cstevens on 99/11/03
 * Last modified by suhler on 02/10/01 16:36:36
 */

package sunlabs.brazil.handler;

import java.io.IOException;
import sunlabs.brazil.server.FileHandler;
import sunlabs.brazil.server.Handler;
import sunlabs.brazil.server.Request;
import sunlabs.brazil.server.Server;
import sunlabs.brazil.util.Glob;
import java.util.Vector;
import java.util.Properties;
import java.util.StringTokenizer;

/**
 * The <code>MultiHostHandler</code> allows the user to handle a set
 * of host names that are all running on the same IP address.  This
 * handler looks at the "Host:" and redispatches the request to the
 * appropriate sub-server.
 * <p>
 * Only the main server is actually listening to the port on the specified IP
 * address.  The sub-servers are not running in separate threads.  Indeed,
 * they are not "running" at all.  They exist merely as a convenient bag to
 * hold each of the server-specific configuration parameters.
 * <p>
 * The <code>respond</code> method of the main handler for the appropriate
 * sub-server is called directly from the <code>respond</code> method of
 * this handler.  
 * <p>
 * This handler uses the following configuration parameters: <dl class=props>
 *
 * <dt> servers
 * <dd> The list of prefixes for the other servers.  Each server will be
 *	initialized from the main <code>server.props</code> with the
 *	specified prefix.  In this way, the configuration parameters for
 *	all the sub-servers can be stored in the same <code>Properties</code>
 *	object.
 *
 * <dt> <i>prefix</i>.host
 * <dd> Each server is started with a given <i>prefix</i>.  The property
 *	<i>prefix</i>.host specifies a Glob pattern for a virtual hostname
 *      the server
 *	will be expected to handle.  If this property is not specified,
 *	the server's virtual hostname will just be <i>prefix</i>.
 *	If multiple host patterns could match a given "Host:" header, 
 *	the first match in the "servers" list matches first.
 *
 * <dt> <i>prefix</i>.handler
 * <dd> The main handler for the server with the given <i>prefix</i>.  If
 *	this property is not specified, it defaults to the
 *	<code>FileHandler</code>.
 *
 * <dt> <i>prefix</i>.log
 * <dd> The log level for the server with the given <i>prefix</i>.  If this
 *	property is not specified, it defaults to the log level of the
 *	parent server.
 * </dl>
 *
 * A sample set of configuration parameters illustrating how to use this
 * handler follows:
 * <pre>
 * handler=host
 * port=8081
 * log=5
 * 
 * host.class=sunlabs.brazil.server.MultiHostHandler
 * host.servers=mars jupiter saturn
 * 
 * mars.host=www.mars.com
 * mars.log=2
 * mars.handler=mars.file
 * mars.file.class=sunlabs.brazil.server.FileHandler
 * mars.file.root=public_html/mars
 * 
 * jupiter.host=jupiter.planet.org
 * jupiter.handler=sunlabs.brazil.server.FileHandler
 * jupiter.root=public_html/jupiter
 * 
 * saturn.host=*.saturn.planet.org
 * saturn.handler=sunlabs.brazil.server.FileHandler
 * saturn.root=public_html/saturn
 * </pre>
 * These parameters set up a normal <code>Server</code> on port 8081,
 * running a <code>MultiHostHandler</code>.  The <code>MultiHostHandler</code>
 * will create three additional servers that respond to the virtual hosts
 * "www.mars.com", "jupiter.planet.org", and "<anything>.saturn.planet.org". The
 * "mars" server will have a <code>Server.prefix</code> of "mars",
 * so that all other configuration parameters that the "mars" server
 * examines can begin with "mars" and be kept distinct from the "jupiter"
 * and "saturn" parameters.
 * <p>
 * The main server and the three sub-servers will all share the
 * same properties object, but can use their own individual prefixes to
 * keep their data separate.
 * <p>
 *
 * @author	Colin Stevens (colin.stevens@sun.com)
 * @version	2.1, 02/10/01
 */
public class MultiHostHandler
    implements Handler
{
    private Vector servers = new Vector();

    /**
     * Initializes the servers for the virtual hosts.  After creating
     * and initializing each sub-server, the <code>init</code> method of the
     * main handler for each sub-server is called.
     *
     * @param	server
     *		The HTTP server that created this handler.
     *
     * @param	prefix
     *		A prefix to prepend to all of the keys that this
     *		handler uses to extract configuration information.
     *
     * @return	<code>true</code> if at least one sub-server was found and
     *		could be initialized, <code>false</code> otherwise.
     *		Diagnostic messages are logged for each sub-server started.
     */
    public boolean
    init(Server server, String prefix)
    {
	Properties props = server.props;
	
	String list = props.getProperty(prefix + "servers", "");
	StringTokenizer st = new StringTokenizer(list);
	while (st.hasMoreTokens()) {
	    String name = st.nextToken();

	    server.log(Server.LOG_DIAGNOSTIC, prefix,
		    "creating subserver " + name);

	    String handler = props.getProperty(name + ".handler",
		    FileHandler.class.getName());

	    Server sub = new Server(server.listen, handler, props);
	    sub.hostName = props.getProperty(name + ".host", name);
	    sub.prefix = name;
	    try {
		String str = props.getProperty(name + ".log");
		sub.logLevel = Integer.decode(str).intValue();
	    } catch (Exception e) {
		sub.logLevel = server.logLevel;
	    }

	    if (sub.init()) {
	        servers.addElement(new Host(sub.hostName, sub));
	    }
	}
	return (servers.size() > 0);
    }

    /**
     * Responds to an HTTP request by examining the "Host:" request header
     * and dispatching to the main handler of the server that handles
     * that virtual host.  If the "Host:" request header was not specified,
     * or named a virtual host that was not initialized in <code>init</code>
     * from the list of virtual hosts, this method returns without
     * handling the request.
     *
     * @param	request
     *		The HTTP request to be forwarded to one of the sub-servers.
     *
     * @return	<code>true</code> if the sub-server handled the message,
     *		<code>false</code> if it did not.  <code>false</code> is
     *		also returned if the "Host:" was unspecified or unknown.
     */
    public boolean
    respond(Request request)
	throws IOException
    {
	String host = request.getRequestHeader("Host");
	if (host == null) {
	    return false;
	}
	for(int i=0;i<servers.size();i++) {
	    Server server=((Host)(servers.elementAt(i))).getServer(host);
	    if (server != null) {
	        request.server = server;
	        return server.handler.respond(request);
	    }
	}
	return false;
    }

    /**
     * Structure to keep track of servers and their glob patterns.
     */

    static class Host {
	Server server;		// the host name
	String pattern;		// the glob pattern
    
	Host(String pattern, Server server) {
	    this.server = server;
	    this.pattern = pattern;
	}

	Server
	getServer(String host) {
	    if (Glob.match(pattern, host)) {
		return server;
	    } else {
		return null;
	    }
	}
    }
}
