/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.test.infra.openai.mock;

import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;

import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.RegisterExtension;
import org.testcontainers.shaded.com.fasterxml.jackson.databind.JsonNode;
import org.testcontainers.shaded.com.fasterxml.jackson.databind.ObjectMapper;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertTrue;

public class OpenAIMockEmbeddingTest {

    @RegisterExtension
    public OpenAIMock openAIMock = new OpenAIMock().builder()
            .whenEmbedding("What is Apache Camel?")
            .replyWithEmbedding(new float[] { 0.1f, 0.2f, 0.3f, 0.4f })
            .end()
            .whenEmbedding("Hello world")
            .replyWithEmbedding(1536)
            .end()
            .whenEmbedding("Test input")
            .replyWithEmbedding(3)
            .end()
            .whenEmbedding("First text")
            .replyWithEmbedding(new float[] { 0.1f, 0.2f })
            .end()
            .whenEmbedding("Second text")
            .replyWithEmbedding(new float[] { 0.3f, 0.4f })
            .end()
            .build();

    @Test
    public void testExplicitEmbeddingVector() throws Exception {
        HttpClient client = HttpClient.newHttpClient();
        HttpRequest request = HttpRequest.newBuilder()
                .uri(URI.create(openAIMock.getBaseUrl() + "/v1/embeddings"))
                .header("Content-Type", "application/json")
                .POST(HttpRequest.BodyPublishers
                        .ofString("{\"input\": \"What is Apache Camel?\", \"model\": \"text-embedding-ada-002\"}"))
                .build();

        HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());
        String responseBody = response.body();

        ObjectMapper objectMapper = new ObjectMapper();
        JsonNode responseJson = objectMapper.readTree(responseBody);

        assertEquals("list", responseJson.path("object").asText());
        assertEquals("camel-embedding", responseJson.path("model").asText());

        JsonNode dataArray = responseJson.path("data");
        assertEquals(1, dataArray.size());

        JsonNode embeddingData = dataArray.get(0);
        assertEquals("embedding", embeddingData.path("object").asText());
        assertEquals(0, embeddingData.path("index").asInt());

        JsonNode embedding = embeddingData.path("embedding");
        assertEquals(4, embedding.size());
        assertEquals(0.1, embedding.get(0).asDouble(), 0.001);
        assertEquals(0.2, embedding.get(1).asDouble(), 0.001);
        assertEquals(0.3, embedding.get(2).asDouble(), 0.001);
        assertEquals(0.4, embedding.get(3).asDouble(), 0.001);

        // Check usage
        JsonNode usage = responseJson.path("usage");
        assertTrue(usage.has("prompt_tokens"));
        assertTrue(usage.has("total_tokens"));
    }

    @Test
    public void testAutoGeneratedEmbedding() throws Exception {
        HttpClient client = HttpClient.newHttpClient();
        HttpRequest request = HttpRequest.newBuilder()
                .uri(URI.create(openAIMock.getBaseUrl() + "/v1/embeddings"))
                .header("Content-Type", "application/json")
                .POST(HttpRequest.BodyPublishers
                        .ofString("{\"input\": \"Hello world\", \"model\": \"text-embedding-ada-002\"}"))
                .build();

        HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());
        String responseBody = response.body();

        ObjectMapper objectMapper = new ObjectMapper();
        JsonNode responseJson = objectMapper.readTree(responseBody);

        assertEquals("list", responseJson.path("object").asText());

        JsonNode embedding = responseJson.path("data").get(0).path("embedding");
        assertEquals(1536, embedding.size());

        // All values should be between -1 and 1
        for (int i = 0; i < embedding.size(); i++) {
            double value = embedding.get(i).asDouble();
            assertTrue(value >= -1.0 && value <= 1.0,
                    "Embedding value at index " + i + " should be between -1 and 1, but was " + value);
        }
    }

    @Test
    public void testArrayInputFormat() throws Exception {
        HttpClient client = HttpClient.newHttpClient();
        // OpenAI API also accepts array format for input
        HttpRequest request = HttpRequest.newBuilder()
                .uri(URI.create(openAIMock.getBaseUrl() + "/v1/embeddings"))
                .header("Content-Type", "application/json")
                .POST(HttpRequest.BodyPublishers
                        .ofString("{\"input\": [\"What is Apache Camel?\"], \"model\": \"text-embedding-ada-002\"}"))
                .build();

        HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());
        String responseBody = response.body();

        ObjectMapper objectMapper = new ObjectMapper();
        JsonNode responseJson = objectMapper.readTree(responseBody);

        assertEquals("list", responseJson.path("object").asText());

        JsonNode embedding = responseJson.path("data").get(0).path("embedding");
        assertEquals(4, embedding.size());
        assertEquals(0.1, embedding.get(0).asDouble(), 0.001);
    }

    @Test
    public void testDeterministicOutput() throws Exception {
        HttpClient client = HttpClient.newHttpClient();

        // First request
        HttpRequest request1 = HttpRequest.newBuilder()
                .uri(URI.create(openAIMock.getBaseUrl() + "/v1/embeddings"))
                .header("Content-Type", "application/json")
                .POST(HttpRequest.BodyPublishers
                        .ofString("{\"input\": \"Test input\", \"model\": \"text-embedding-ada-002\"}"))
                .build();

        HttpResponse<String> response1 = client.send(request1, HttpResponse.BodyHandlers.ofString());

        // Second request with same input
        HttpRequest request2 = HttpRequest.newBuilder()
                .uri(URI.create(openAIMock.getBaseUrl() + "/v1/embeddings"))
                .header("Content-Type", "application/json")
                .POST(HttpRequest.BodyPublishers
                        .ofString("{\"input\": \"Test input\", \"model\": \"text-embedding-ada-002\"}"))
                .build();

        HttpResponse<String> response2 = client.send(request2, HttpResponse.BodyHandlers.ofString());

        ObjectMapper objectMapper = new ObjectMapper();
        JsonNode json1 = objectMapper.readTree(response1.body());
        JsonNode json2 = objectMapper.readTree(response2.body());

        JsonNode embedding1 = json1.path("data").get(0).path("embedding");
        JsonNode embedding2 = json2.path("data").get(0).path("embedding");

        // Same input should produce same embedding
        assertEquals(embedding1.size(), embedding2.size());
        for (int i = 0; i < embedding1.size(); i++) {
            assertEquals(embedding1.get(i).asDouble(), embedding2.get(i).asDouble(), 0.0001,
                    "Embedding values should be identical for the same input");
        }
    }

    @Test
    public void testBatchEmbeddings() throws Exception {
        HttpClient client = HttpClient.newHttpClient();
        // Send multiple texts in a single request
        HttpRequest request = HttpRequest.newBuilder()
                .uri(URI.create(openAIMock.getBaseUrl() + "/v1/embeddings"))
                .header("Content-Type", "application/json")
                .POST(HttpRequest.BodyPublishers
                        .ofString("{\"input\": [\"First text\", \"Second text\"], \"model\": \"text-embedding-ada-002\"}"))
                .build();

        HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());
        String responseBody = response.body();

        ObjectMapper objectMapper = new ObjectMapper();
        JsonNode responseJson = objectMapper.readTree(responseBody);

        assertEquals("list", responseJson.path("object").asText());

        JsonNode dataArray = responseJson.path("data");
        assertEquals(2, dataArray.size(), "Should have 2 embeddings in the response");

        // First embedding
        JsonNode embedding1 = dataArray.get(0);
        assertEquals("embedding", embedding1.path("object").asText());
        assertEquals(0, embedding1.path("index").asInt());
        assertEquals(2, embedding1.path("embedding").size());
        assertEquals(0.1, embedding1.path("embedding").get(0).asDouble(), 0.001);
        assertEquals(0.2, embedding1.path("embedding").get(1).asDouble(), 0.001);

        // Second embedding
        JsonNode embedding2 = dataArray.get(1);
        assertEquals("embedding", embedding2.path("object").asText());
        assertEquals(1, embedding2.path("index").asInt());
        assertEquals(2, embedding2.path("embedding").size());
        assertEquals(0.3, embedding2.path("embedding").get(0).asDouble(), 0.001);
        assertEquals(0.4, embedding2.path("embedding").get(1).asDouble(), 0.001);
    }
}
