<?php
/**
 * IMP_Quota:: provides an API for retrieving Quota details from a mail
 * server.
 *
 * $Horde: imp/lib/Quota.php,v 1.16 2003/04/28 21:25:21 slusarz Exp $
 *
 * Copyright 2002-2003 Mike Cochrane <mike@graftonhall.co.nz>
 *
 * See the enclosed file COPYING for license information (GPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/gpl.html.
 *
 * @author  Mike Cochrane <mike@graftonhall.co.nz>
 * @version $Revision: 1.16 $
 * @since   IMP 4.0
 * @package imp.quota
 */
class IMP_Quota {

    /**
     * Hash containing connection parameters.
     *
     * @var array $_params
     */
    var $_params = array();

    /**
     * Attempts to return a concrete Quota instance based on $driver.
     *
     * @access public
     *
     * @param string $driver          The type of concrete Quota subclass
     *                                to return.  The is based on the quota
     *                                source ($driver).  The code is
     *                                dynamically included.
     *
     * @param optional array $params  A hash containing any additional
     *                                configuration or connection parameters
     *                                a subclass might need.
     *
     * @return mixed  The newly created concrete Qouta instance, or false
     *                on error.
     */
    function &factory($driver, $params = array())
    {
        $driver = basename($driver);
        require_once dirname(__FILE__) . '/Quota/' . $driver . '.php';
        $class = 'IMP_Quota_' . $driver;
        if (class_exists($class)) {
            return new $class($params);
        } else {
            return false;
        }
    }

    /**
     * Attempts to return a reference to a concrete Quota instance
     * based on $driver.  It will only create a new instance if no
     * Quota instance with the same parameters currently exists.
     *
     * This should be used if multiple quota sources are required.
     *
     * This method must be invoked as: $var = &Quota::singleton()
     *
     * @access public
     *
     * @param string $driver          The type of concrete Quota subclass
     *                                to return.  The is based on the quota
     *                                source ($driver).  The code is
     *                                dynamically included.
     *
     * @param optional array $params  A hash containing any additional
     *                                configuration or connection parameters
     *                                a subclass might need.
     *
     * @return mixed  The created concrete Quota instance, or false on error.
     */
    function &singleton($driver, $params = array())
    {
        static $instances;

        if (!isset($instances)) {
            $instances = array();
        }

        $signature = serialize(array($driver, $params));
        if (!array_key_exists($signature, $instances)) {
            $instances[$signature] = &IMP_Quota::factory($driver, $params);
        }

        return $instances[$signature];
    }

    /**
     * Constructor.
     *
     * @access public
     *
     * @param optional array $params  Hash containing connection parameters.
     */
    function IMP_Quota($params = array())
    {
        $this->_params = $params;
    }

    /**
     * Get quota HTML to push to user.
     *
     * @access public
     *
     * @return string  The HTML representation of the quota data.
     */
    function quotaHTML()
    {
        return '';
    }

    /**
     * Created the html used to display the quota to the user.
     *
     * @access private
     *
     * @param integer $used   Number of bytes used in the mailbox.
     * @param integer $total  Total quota allowance for mailbox (in bytes).
     *
     * @return string  HTML code to push to user.
     */
    function _quotaHtml($used, $total)
    {
        if ($total != 0) {
            $used = $used / (1024 * 1024.0);
            $total = $total / (1024 * 1024.0);
            $percent = ($used * 100) / $total;
            if ($percent >= 90) {
                $class = 'quotaalert';
            } elseif ($percent >= 75) {
                $class = 'quotawarn';
            } else {
                $class = 'control';
            }
            $message = sprintf(_("Quota status: %.2fMB / %.2fMB  (%.2f%%)"), $used, $total, $percent);
        } else {
            $class = 'control';
            $message = sprintf(_("Quota status: NO LIMIT"));
        }

        require IMP_TEMPLATES . '/quota/quota.inc';
    }

    /**
     * Display Error Message.
     *
     * @access private
     *
     * @param string $message  The error message to display.
     */
    function _quotaError($message)
    {
        $class = 'quotawarn';
        require IMP_TEMPLATES . '/quota/quota.inc';
    }

}
