<?php
/**
 * The IMP_MIME_Viewer_plain class renders out text/plain MIME parts
 * with URLs made into hyperlinks.
 *
 * $Horde: imp/lib/MIME/Viewer/plain.php,v 1.27 2003/08/20 20:03:59 jan Exp $
 *
 * Copyright 1999-2003 Anil Madhavapeddy <anil@recoil.org>
 *
 * See the enclosed file COPYING for license information (GPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/gpl.html.
 *
 * @author  Anil Madhavapeddy <anil@recoil.org>
 * @version $Revision: 1.27 $
 * @since   IMP 4.0
 * @package horde.mime.viewer
 */
class IMP_MIME_Viewer_plain extends MIME_Viewer {

    /**
     * Render out the currently set contents.
     *
     * @access public
     *
     * @param array $params  An array with a reference to a MIME_Contents
     *                       object.
     *
     * @return string  The rendered text in HTML.
     */
    function render($params)
    {
        $contents = &$params[0];

        global $conf, $prefs;

        $text = $this->mime_part->getContents();

        /* If calling as an attachment from view.php, we do not want
           to alter the text in any way. */
        if ($contents->viewAsAttachment()) {
            return $text;
        }

        if ($text === false) {
            return $contents->formatStatusMsg(_("There was an error displaying this message part"));
        }

        if (trim($text) == '') {
            return $contents->formatStatusMsg(_("There was no text in this message part"));
        }

        /* If requested, scan the message for PGP data. */
        if ($conf['utils']['gnupg'] && $prefs->getValue('pgp_scan_body')) {
            require_once IMP_BASE . '/lib/PGP.php';
            $imp_pgp = &new IMP_PGP();
            if (($out = $imp_pgp->parseMessageOutput($text, $contents, $this->mime_part))) {
                return $out;
            }
        }

        /* If requested, scan the message for UUencoded data. */
        if ($this->getConfigParam('uuencode')) {
            require_once 'Mail/mimeDecode.php';
            $files = &Mail_mimeDecode::uudecode($text);
        }

        require_once HORDE_BASE . '/lib/Text.php';

        // Do in-lining of http://xxx.xxx to link, xxx@xxx.xxx to
        // email.
        $charset = String::lower($this->getCharset());
        $text = Text::toHTML($text, TEXT_HTML_MICRO, $charset);

        // Highlight quoted parts of an email.
        if ($prefs->getValue('highlight_text')) {
            $text = Text::highlightQuotes($text);
        }

        // Highlight simple markup of an email.
        if ($prefs->getValue('highlight_simple_markup')) {
            $text = Text::simpleMarkup($text);
        }

        // Dim signatures.
        if ($prefs->getValue('dim_signature')) {
            $text = Text::dimSignature($text);
        }

        // Filter bad language.
        $text = IMP::filterText($text);

        // Expand tabs with respect to tab stops
        $text = Text::smartExpandTabs($text);

        // Wordwrap.
        $text = str_replace('  ', ' &nbsp;', $text);
        $text = str_replace("\n ", "\n&nbsp;", $text);
        if ($text{0} == ' ') {
            $text = '&nbsp;' . substr($text, 1);
        }
        $text = '<p class="fixed">' . $text . '</p>';

        /* If the text is in a different character set, and we're not
           already in a new window, give the user a link to open it in
           a new window with the right character set. */
        if ($charset != 'us-ascii') {
            $default_charset = NLS::getCharset();
            if (($default_charset != 'UTF-8') &&
                ($charset != String::lower($default_charset))) {
                $status = array(
                    sprintf(_("This message was written in a character set (%s) other than your own."), htmlspecialchars($charset)),
                    sprintf(_("If it is not displayed correctly, %s to open it in a new window."), $contents->linkViewJS($this->mime_part, 'view_attach', _("click here")))
                );
                $text = $contents->formatStatusMsg($status, null, false) . '<p />' . $text;
            }
        }

        /* Replace UUencoded data with links now. */
        if ($this->getConfigParam('uuencode') && !empty($files)) {
            foreach ($files as $file) {
                $uupart = &new MIME_Part();
                $uupart->setContents($file['filedata']);
                $uupart->setName(strip_tags($file['filename']));

                $uumessage = &MIME_Message::convertMIMEPart($uupart);
                $mc = &new MIME_Contents($uumessage, array('download' => 'download_attach', 'view' => 'view_attach'), array(&$contents));
                $mc->buildMessage();

                $text = preg_replace("/begin ([0-7]{3}) (.+)\r?\n(.+)\r?\nend/Us", '<table>' . $mc->getMessage(true) . '</table>', $text, 1);
            }
        }

        return $text;
    }

    /**
     * Return the content-type
     *
     * @access public
     *
     * @return string  The content-type of the output.
     */
    function getType()
    {
        if (MIME_Contents::viewAsAttachment()) {
            return 'text/plain';
        } else {
            return 'text/html';
        }
    }

    /**
     * Return the character set for this part.
     *
     * @access public
     *
     * @return string  The character set of this part.
     */
    function getCharset()
    {
        if (($charset = $this->mime_part->getCharset())) {
            return $charset;
        } else {
            return parent::getCharset();
        }
    }

}
