/* -*- c++ -*-
 *
 * mlconfig.cpp
 *
 * Copyright (C) 2003 Petter Stokke <ummo@hellokitty.com>
 * Copyright (C) 2003,2004,2007 Sebastian Sauer <mail@dipe.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#include <QCheckBox>
#include <QVBoxLayout>
#include <QList>

#include <kdebug.h>
#include <klocale.h>

#include "donkeyprotocol.h"
#include "network.h"
#include "mlconfig.h"
#include "kmldonkey.h"

NetworkPage::NetworkPage(QWidget* parent, const char* name)
    : QWidget(parent)
{
    setObjectName(name);
    QVBoxLayout* layout = new QVBoxLayout(this);

    foreach (Network *net, KMLDonkey::App->donkey->availableNetworks()) {
        QCheckBox* c = new QCheckBox(net->networkName(), this);
        c->setChecked(net->networkEnabled());
        layout->addWidget(c);
        connect(c, SIGNAL(toggled(bool)), SLOT(becomeDirty(bool)));
        nwBox.insert(net->networkNo(), c);
    }
    layout->addStretch(1);
}

void NetworkPage::becomeDirty(bool)
{
    emit pageDirty();
}

void NetworkPage::applyOptions(DonkeyProtocol* target)
{
    Q3IntDictIterator<QCheckBox> it(nwBox);
    for (; it.current(); ++it) {
        Network* net = target->findNetworkNo(it.currentKey());
        if (!net || net->networkEnabled() == it.current()->isChecked())
            continue;
        target->enableNetwork(it.currentKey(), it.current()->isChecked());
    }
}

MLDonkeyConfigDialog::MLDonkeyConfigDialog(QWidget* parent, const char* name)
    : KPageDialog(parent)
{
    setCaption(i18n("MLDonkey Configuration"));
    setObjectName(name ? name : "MLDonkeyConfigDialog");
    setFaceType(KPageDialog::Tree);
    setButtons(KDialog::Ok|KDialog::Apply|KDialog::Cancel);
    enableButton(Apply, false);

    // Add regular option sections
    OptionEditorWidget* page;
    foreach (const DonkeyOption &opt, KMLDonkey::App->donkey->sectionOptions()) {
        if (!KMLDonkey::App->advancedConfig && opt.optionAdvanced())
            continue;
        const QString section = opt.optionSection();
        page = pages.value(section);
        if (!page) {
            page = new OptionEditorWidget(this);
            addPage(page, section);
            connect(page, SIGNAL(listIsDirty()), SLOT(setDirtyDialog()));
            pages.insert(section, page);
        }
        page->addOption(opt);
    }
    // Add the root network page
    QString networksName = i18nc("mldonkey config: network options page", "Networks");

    QWidget* w = new QWidget(this);
    QVBoxLayout* l = new QVBoxLayout(w);
    l->setSpacing(0);
    l->setMargin(0);

    KPageWidgetItem* networkItem = addPage(w, networksName);
    m_networkPage = new NetworkPage(w);
    l->addWidget(m_networkPage);
    connect(m_networkPage, SIGNAL(pageDirty()), SLOT(setDirtyDialog()));

    // Add the network sub-pages
    foreach (const DonkeyOption &opt, KMLDonkey::App->donkey->pluginOptions()) {
        if (!KMLDonkey::App->advancedConfig && opt.optionAdvanced())
            continue;
        const QString section = opt.optionSection();
        page = pages.value(section);
        if (!page) {
            page = new OptionEditorWidget(this);
            addSubPage(networkItem, page, section);
            connect(page, SIGNAL(listIsDirty()), SLOT(setDirtyDialog()));
            pages.insert(section, page);
        }
        page->addOption(opt);
    }

    resize(QSize(580, 420).expandedTo( minimumSizeHint() ));
    //setInitialSize(configDialogSize("MLConfigDialog"));
    foreach (OptionEditorWidget *w, pages) {
        w->sortByColumn(-1, Qt::AscendingOrder);
        w->setSortingEnabled(true);
    }

    connect(this, SIGNAL(applyClicked()), this, SLOT(applyOptions()));
    connect(this, SIGNAL(accepted()), this, SLOT(applyOptions()));
    connect(this, SIGNAL(finished()), this, SLOT(closeDialog()));
}

void MLDonkeyConfigDialog::setDirtyDialog()
{
    enableButton(Apply, true);
}

void MLDonkeyConfigDialog::applyOptions()
{
    foreach (OptionEditorWidget *w, pages) {
        w->applyChangedOptions(KMLDonkey::App->donkey);
    }
    m_networkPage->applyOptions(KMLDonkey::App->donkey);
    enableButton(Apply, false);
}

void MLDonkeyConfigDialog::closeDialog()
{
    //saveDialogSize("MLConfigDialog");
    delayedDestruct();
}

#include "mlconfig.moc"
