/*******************************************************************************
 * Copyright (c) 2009, 2019 Xored Software Inc and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-v20.html
 *
 * Contributors:
 *     Xored Software Inc - initial API and implementation and/or initial documentation
 *******************************************************************************/
package org.eclipse.rcptt.reporting.core;

import java.io.OutputStream;
import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.rcptt.sherlock.core.model.sherlock.report.Report;

/**
 * Entry point for report generation from Q7.
 * 
 * Receive file factory and list of per test reports to generate for.
 * 
 * Same report renderer are used for report generation in Q7 RCP and in Q7
 * runner.
 * 
 * Report renderer receive Q7ReportIterator to iterate over all reports in
 * execution.
 * 
 */
public interface IReportRenderer {
	/**
	 * Renderer should implement interface to recieve progress monitor then
	 * report renderer are executed from Q7 report wizard.
	 */
	interface ProgressMonitorSupport {
		void setProgressMonitor(IProgressMonitor monitor);
	}

	public interface IContentFactory {
		/**
		 * Create new file output stream in user specified location. User
		 * specify location in Q7 RCP report wizard, or in Q7 runner.
		 * 
		 * @param fname
		 *            - file name to create.
		 * @return - Buffered file output stream.
		 */
		OutputStream createFileStream(String fname) throws CoreException;

		/**
		 * Creates new folder in user specified location.User specify location
		 * in Q7 RCP report wizard, or in Q7 runner.
		 * 
		 * @param name
		 *            - folder name
		 * @return - new content factory based on created folder.
		 */
		IContentFactory createFolder(String name);

		void removeFileOrFolder(String name) throws CoreException;

		List<String> matchFiles(String fname);

		boolean isFileExist(String fileName);
	}

	/**
	 * Receive a file factory to generate report in. And a list of test
	 * execution reports by Q7ReportIterrator..
	 * 
	 * @param factory
	 *            - file factory
	 * @param reportName
	 *            - report name to generate. User specified in UI.
	 * @param report
	 *            - report iterator.
	 * @return - Eclipse platform status object. Display error in UI if failed
	 *         to render report.
	 */
	IStatus generateReport(IContentFactory factory, String reportName,
			Iterable<Report> report);

	/**
	 * Return a list of files will be generated by report, to show user
	 * confirmations in UI.
	 * 
	 * @param reportName
	 *            - user specified report name.
	 * @return - Array of file names. Names then will be checked using
	 *         isFileExist() for otherwrite confirmation to display.
	 */
	String[] getGeneratedFileNames(String reportName);
}
